/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       asn.c
**     SYSTEM   NAME:       ASN1 Basic Encoding
**     ORIGINAL AUTHOR(S):  Dirk Wisse
**     VERSION  NUMBER:     1
**     CREATION DATE:       1990/11/22
**
** DESCRIPTION: ASN1 Basic Encoding Rules.
**              Encoding takes place from end to begin. A normal
**              procedure for definite encoding is:
**
**              AsnOpn (Asn, Buf, sizeof (Buf), ASN_ENC);
**              AsnEocEnc (Asn, &EndOfSeq);
**              AsnIntEnc (Asn, &EndOfInt, 3);
**              AsnHdrEnc (Asn, EndOfInt, ASN_UNI, ASN_PRI, ASN_INT);
**              AsnOtsEnc (Asn, &EndOfOts, "String", 6);
**              AsnHdrEnc (Asn, EndOfOts, ASN_UNI, ASN_PRI, ASN_OTS);
**              AsnHdrEnc (Asn, EndOfSeq, ASN_UNI, ASN_CON, ASN_SEQ);
**              AsnCls (Asn, &BufBeg, &BufLen);
**
**              To decode this we must do:
**
**              AsnOpn (Asn, BufBeg, BufLen, ASN_DEC);
**              AsnHdrDec (Asn, &EndOfSeq, Cls, Con, Tag);
**              AsnHdrDec (Asn, &EndOfOts, Cls, Con, Tag);
**              AsnOtsDec (Asn, EndOfOts, String, sizeof (String), Length);
**              AsnHdrDec (Asn, &EndOfInt, Cls, Con, Tag);
**              AsnIntDec (Asn, EndOfInt, &Integer);
**              AsnEocDec (Asn, EndOfSeq);
**              AsnCls (Asn, &BufBeg, &BufLen);
**              
**              For indefinite encoding EndOfSeq and &EndOfSeq in the
**              example above should be replaced by NULL.
**              For indefinite decoding nothing has to be changed.
**              This can be very useful if you want to decode both
**              definite and indefinite encodings.
**
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header$";
#endif
#include "asn1.h"

/**************************************************************
** NAME:        AsnOpn                                    [API]
** SYNOPSIS:    int
**                  AsnOpn
**                  (
**                      asn_sck     *Sck,
**                      char        *Buf,
**                      unsigned    Len,
**                      unsigned    Mde
**                  )
** DESCRIPTION: Opens an ASN1 socket.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Buf: Character buffer for encoding.
**              Len: Length of character buffer.
**              Mde: Encoding, Decoding (ASN_ENC, ASN_DEC).
**              Encoding starts at the end of the buffer, and
**              proceeds to the beginning.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
void
    AsnOpn
    (
        asn_sck     *Asn,
        char        *Buf,
        unsigned    Len,
        unsigned    Mde
    )
{
    Asn->SckBeg = Buf;
    Asn->SckEnd = Buf + Len;
    Asn->SckPtr = (Mde == ASN_ENC) ? Buf + Len : Buf;
}

/**************************************************************
** NAME:        AsnCls                                    [API]
** SYNOPSIS:    int
**                  AsnCls
**                  (
**                      asn_sck     *Sck,
**                      char        **Buf,
**                      unsigned    *Len
**                  )
** DESCRIPTION: Closes an ASN1 socket.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Buf: Pointer to beginning of encoding.
**              Len: Length of encoding.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
void
    AsnCls
    (
        asn_sck     *Asn,
        char        **Buf,
        unsigned    *Len
    )
{
    *Buf = Asn->SckPtr;
    *Len = Asn->SckEnd - Asn->SckPtr;
}



/**************************************************************
** NAME:        AsnOctEnc
** SYNOPSIS:    int
**                  AsnOctEnc
**                  (
**                      asn_sck         *Asn,
**                      unsigned char   Chr
**                  )
** DESCRIPTION: Encodes an octet.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnOctEnc
    (
        asn_sck         *Asn,
        unsigned char   Chr
    )
{
    if (Asn->SckPtr <= Asn->SckBeg)
        return (-1);
    *--(Asn->SckPtr) = Chr;
    return (0);
}

/**************************************************************
** NAME:        AsnOctDec
** SYNOPSIS:    int
**                  AsnOctDec
**                  (
**                      asn_sck         *Asn,
**                      unsigned char   *Chr
**                  )
** DESCRIPTION: Decodes an octet.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnOctDec
    (
        asn_sck         *Asn,
        unsigned char   *Chr
    )
{
    if (Asn->SckPtr >= Asn->SckEnd)
        return (-1);
    *Chr = *(Asn->SckPtr)++;
    return (0);
}

/**************************************************************
** NAME:        AsnTagEnc
** SYNOPSIS:    int
**                  AsnTagEnc
**                  (
**                      asn_sck         *Asn,
**                      unsigned        Tag
**                  )
** DESCRIPTION: Encodes a tag.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnTagEnc
    (
        asn_sck     *Asn,
        unsigned    Tag
    )
{
    unsigned char
        Chr;

    Chr = (unsigned char) (Tag & 0x7F);
    Tag >>= 7;
    if (AsnOctEnc (Asn, Chr) < 0)
        return (-1);
    while (Tag > 0)
    {
        Chr = (unsigned char) (Tag | 0x80);
        Tag >>= 7;
        if (AsnOctEnc (Asn, Chr) < 0)
            return (-1);
    }
    return (0);
}

/**************************************************************
** NAME:        AsnTagDec
** SYNOPSIS:    int
**                  AsnTagDec
**                  (
**                      asn_sck         *Asn,
**                      unsigned        *Tag
**                  )
** DESCRIPTION: Decodes a tag.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnTagDec
    (
        asn_sck     *Asn,
        unsigned    *Tag
    )
{
    unsigned char
        Chr;

    *Tag = 0;
    do
    {
        if (AsnOctDec (Asn, &Chr) < 0)
            return (-1);
        *Tag <<= 7;
        *Tag |= Chr & 0x7F;
    }
    while ((Chr & 0x80) == 0x80);
    return (0);
}

/**************************************************************
** NAME:        AsnIdrEnc
** SYNOPSIS:    int
**                  AsnIdrEnc
**                  (
**                      asn_sck     *Asn,
**                      unsigned    Cls,
**                      unsigned    Con,
**                      unsigned    Tag
**                  )
** DESCRIPTION: Encodes an identifier.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIdrEnc
    (
        asn_sck     *Asn,
        unsigned    Cls,
        unsigned    Con,
        unsigned    Tag
    )
{
    unsigned char
        Chr;

    if (Tag >= 0x1F)
    {
        if (AsnTagEnc (Asn, Tag) < 0)
            return (-1);
        Tag = 0x1F;
    }
    Chr = (unsigned char) ((Cls << 6) | (Con << 5) | (Tag));
    if (AsnOctEnc (Asn, Chr) < 0)
        return (-1);
    return (0);
}

/**************************************************************
** NAME:        AsnIdrDec
** SYNOPSIS:    int
**                  AsnIdrDec
**                  (
**                      asn_sck     *Asn,
**                      unsigned    *Cls,
**                      unsigned    *Con,
**                      unsigned    *Tag
**                  )
** DESCRIPTION: Decodes an identifier.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIdrDec
    (
        asn_sck     *Asn,
        unsigned    *Cls,
        unsigned    *Con,
        unsigned    *Tag
    )
{
    unsigned char
        Chr;

    if (AsnOctDec (Asn, &Chr) < 0)
        return (-1);
    *Cls = (Chr & 0xC0) >> 6;
    *Con = (Chr & 0x20) >> 5;
    *Tag = (Chr & 0x1F);
    if (*Tag == 0x1F)
    {
        if (AsnTagDec (Asn, Tag) < 0)
            return (-1);
    }
    return (0);
}

/**************************************************************
** NAME:        AsnLenEnc
** SYNOPSIS:    int
**                  AsnLenEnc
**                  (
**                      asn_sck     *Asn,
**                      unsigned    Def,
**                      unsigned    Len
**                  )
** DESCRIPTION: Encodes a definite or indefinite length.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnLenEnc
    (
        asn_sck     *Asn,
        unsigned    Def,
        unsigned    Len
    )
{
    unsigned char
        Chr,
        Cnt;

    if (!Def)
        Chr = 0x80;
    else
    {
        if (Len < 0x80)
            Chr = (unsigned char) Len;
        else
        {
            Cnt = 0;
            while (Len > 0)
            {
                Chr = (unsigned char) Len;
                Len >>= 8;
                if (AsnOctEnc (Asn, Chr) < 0)
                    return (-1);
                Cnt++;
            }
            Chr = (unsigned char) (Cnt | 0x80);
        }
    }
    if (AsnOctEnc (Asn, Chr) < 0)
        return (-1);
    return (0);
}

/**************************************************************
** NAME:        AsnLenDec
** SYNOPSIS:    int
**                  AsnLenDec
**                  (
**                      asn_sck     *Asn,
**                      unsigned    Def,
**                      unsigned    Len
**                  )
** DESCRIPTION: Decodes a definite or indefinite length.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnLenDec
    (
        asn_sck     *Asn,
        unsigned    *Def,
        unsigned    *Len
    )
{
    unsigned char
        Chr,
        Cnt;

    if (AsnOctDec (Asn, &Chr) < 0)
        return (-1);
    if (Chr == 0x80)
        *Def = 0;
    else
    {
        *Def = 1;
        if (Chr < 0x80)
            *Len = Chr;
        else
        {
            Cnt = (unsigned char) (Chr & 0x7F);
            *Len = 0;
            while (Cnt > 0)
            {
                if (AsnOctDec (Asn, &Chr) < 0)
                    return (-1);
                *Len <<= 8;
                *Len |= Chr;
                Cnt--;
            }
        }
    }
    return (0);
}

/**************************************************************
** NAME:        AsnHdrEnc                                 [API]
** SYNOPSIS:    int
**                  AsnHdrEnc
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc,
**                      unsigned    Cls,
**                      unsigned    Con,
**                      unsigned    Tag
**                  )
** DESCRIPTION: Encodes an ASN1 header.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Cls: Class (see asn1.h)
**              Con: Primitive, Constructed (ASN_PRI, ASN_CON)
**              Tag: Tag (see asn1.h)
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnHdrEnc
    (
        asn_sck     *Asn,
        char        *Eoc,
        unsigned    Cls,
        unsigned    Con,
        unsigned    Tag
    )
{
    unsigned
        Def,
        Len;

    if (Eoc == NULL)
    {
        Def = 0;
        Len = 0;
    }
    else
    {
        Def = 1;
        Len = Eoc - Asn->SckPtr;
    }
    if (AsnLenEnc (Asn, Def, Len) < 0)
        return (-1);
    if (AsnIdrEnc (Asn, Cls, Con, Tag) < 0)
        return (-1);
    return (0);
}

/**************************************************************
** NAME:        AsnHdrDec                                 [API]
** SYNOPSIS:    int
**                  AsnHdrDec
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc,
**                      unsigned    *Cls,
**                      unsigned    *Con,
**                      unsigned    *Tag
**                  )
** DESCRIPTION: Decodes an ASN1 header.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Cls: Class (see asn1.h)
**              Con: Primitive, Constructed (ASN_PRI, ASN_CON)
**              Tag: Tag (see asn1.h)
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnHdrDec
    (
        asn_sck     *Asn,
        char        **Eoc,
        unsigned    *Cls,
        unsigned    *Con,
        unsigned    *Tag
    )
{
    unsigned
        Def,
        Len;

    if (AsnIdrDec (Asn, Cls, Con, Tag) < 0)
        return (-1);
    if (AsnLenDec (Asn, &Def, &Len) < 0)
        return (-1);
    if (Def)
        *Eoc = Asn->SckPtr + Len;
    else
        *Eoc = NULL;
    return (0);
}


/**************************************************************
** NAME:        AsnEoc                                    [API]
** SYNOPSIS:    int
**                  AsnEoc
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc
**                  )
** DESCRIPTION: Checks if decoding is at End Of Contents.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
** RETURNS:     0 -->   not End Of Contents
**              else    End Of Contents
**************************************************************/
int
    AsnEoc
    (
        asn_sck *Asn,
        char    *Eoc
    )
{
    if (Eoc == NULL)
        return (Asn->SckPtr [0] == 0x00 && Asn->SckPtr [1] == 0x00);
    else
        return (Asn->SckPtr >= Eoc);
}

/**************************************************************
** NAME:        AsnEocEnc                                 [API]
** SYNOPSIS:    int
**                  AsnEocEnc
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc
**                  )
** DESCRIPTION: Encodes End Of Contents.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              If Eoc is NULL it encodes an ASN1 End Of
**              Contents (0x00 0x00), so it produces an
**              indefinite length encoding. If Eoc points to
**              a character pointer, Eoc is filled with the
**              pointer to the last encoded octet. This pointer
**              can be used in the next AsnHdrEnc to determine
**              the length of the encoding. This produces a
**              definite length encoding.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnEocEnc
    (
        asn_sck *Asn,
        char    **Eoc
    )
{
    if (Eoc == NULL)
    {
        if (AsnOctEnc (Asn, 0x00) < 0)
            return (-1);
        if (AsnOctEnc (Asn, 0x00) < 0)
            return (-1);
        return (0);
    }
    else
    {
        *Eoc = Asn->SckPtr;
        return (0);
    }
}

/**************************************************************
** NAME:        AsnEocDec                                 [API]
** SYNOPSIS:    int
**                  AsnEocDec
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc
**                  )
** DESCRIPTION: Decodes End Of Contents.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              If Eoc is NULL it decodes an ASN1 End Of
**              Contents (0x00 0x00), so it has to be an
**              indefinite length encoding. If Eoc is a
**              character pointer, it probably was filled by
**              AsnHdrDec, and should point to the octet
**              after the last of the encoding. It is checked
**              if this pointer points to the octet to be
**              decoded. This only takes place in decoding a
**              definite length encoding.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnEocDec
    (
        asn_sck *Asn,
        char    *Eoc
    )
{
    unsigned char
        Chr;

    if (Eoc == NULL)
    {
        if (AsnOctDec (Asn, &Chr) < 0)
            return (-1);
        if (Chr != 0x00)
            return (-1);
        if (AsnOctDec (Asn, &Chr) < 0)
            return (-1);
        if (Chr != 0x00)
            return (-1);
        return (0);
    }
    else
    {
        if (Asn->SckPtr != Eoc)
            return (-1);
        return (0);
    }
}

/**************************************************************
** NAME:        AsnNulEnc                                 [API]
** SYNOPSIS:    int
**                  AsnNulEnc
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc
**                  )
** DESCRIPTION: Encodes Null.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding.
**              Encodes nothing but can be used to fill Eoc.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnNulEnc
    (
        asn_sck *Asn,
        char    **Eoc
    )
{
    *Eoc = Asn->SckPtr;
    return (0);
}

/**************************************************************
** NAME:        AsnNulDec                                 [API]
** SYNOPSIS:    int
**                  AsnNulDec
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc
**                  )
** DESCRIPTION: Decodes Null.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Decodes anything up to Eoc.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnNulDec
    (
        asn_sck *Asn,
        char    *Eoc
    )
{
    Asn->SckPtr = Eoc;
    return (0);
}

/**************************************************************
** NAME:        AsnBolEnc                                 [API]
** SYNOPSIS:    int
**                  AsnBolEnc
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc,
**                      int         Bol
**                  )
** DESCRIPTION: Encodes Boolean.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding.
**              Bol: False, True (0, !0).
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnBolEnc
    (
        asn_sck *Asn,
        char    **Eoc,
        int     Bol
    )
{
    unsigned char
        Chr;

    *Eoc = Asn->SckPtr;
    Chr = (unsigned char) (Bol ? 0xFF : 0x00);
    if (AsnOctEnc (Asn, Chr) < 0)
        return (-1);
    return (0);
}

/**************************************************************
** NAME:        AsnBolDec                                 [API]
** SYNOPSIS:    int
**                  AsnBolDec
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc
**                      int         *Bol
**                  )
** DESCRIPTION: Decodes Boolean.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Bol: False, True (0, !0).
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnBolDec
    (
        asn_sck *Asn,
        char    *Eoc,
        int     *Bol
    )
{
    unsigned char
        Chr;

    if (AsnOctDec (Asn, &Chr) < 0)
        return (-1);
    *Bol = Chr ? 1 : 0;
    if (Asn->SckPtr != Eoc)
        return (-1);
    return (0);
}

/**************************************************************
** NAME:        AsnIntEnc                                 [API]
** SYNOPSIS:    int
**                  AsnIntEnc
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc,
**                      int         Int
**                  )
** DESCRIPTION: Encodes Integer.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding.
**              Int: Integer.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIntEnc
    (
        asn_sck *Asn,
        char    **Eoc,
        int     Int
    )
{
    unsigned char
        Chr,
        Sgn;
    int
        Lim;

    *Eoc = Asn->SckPtr;
    if (Int < 0)
    {
        Lim = -1;
        Sgn = 0x80;
    }
    else
    {
        Lim = 0;
        Sgn = 0x00;
    }
    do
    {
        Chr = (unsigned char) Int;
        Int >>= 8;
        if (AsnOctEnc (Asn, Chr) < 0)
            return (-1);
    }
    while ((Int != Lim) || (unsigned char) (Chr & 0x80) != Sgn);
    return (0);
}

/**************************************************************
** NAME:        AsnIntDec                                 [API]
** SYNOPSIS:    int
**                  AsnIntDec
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc
**                      int         *Int
**                  )
** DESCRIPTION: Decodes Integer.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Int: Integer.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIntDec
    (
        asn_sck *Asn,
        char    *Eoc,
        int     *Int
    )
{
    unsigned char
        Chr;
    unsigned
        Len;

    if (AsnOctDec (Asn, &Chr) < 0)
        return (-1);
    *Int = (signed) Chr;
    Len = 1;
    while (Asn->SckPtr < Eoc)
    {
        if (++Len > sizeof (int))
            return (-1);
        if (AsnOctDec (Asn, &Chr) < 0)
            return (-1);
        *Int <<= 8;
        *Int |= Chr;
    }
    return (0);
}

/**************************************************************
** NAME:        AsnIntEncLng                              [API]
** SYNOPSIS:    int
**                  AsnIntEncLng
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc,
**                      long        Int
**                  )
** DESCRIPTION: Encodes Long Integer.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding.
**              Int: Integer.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIntEncLng
    (
        asn_sck *Asn,
        char    **Eoc,
        long    Int
    )
{
    unsigned char
        Chr,
        Sgn;
    long
        Lim;

    *Eoc = Asn->SckPtr;
    if (Int < 0)
    {
        Lim = -1;
        Sgn = 0x80;
    }
    else
    {
        Lim = 0;
        Sgn = 0x00;
    }
    do
    {
        Chr = (unsigned char) Int;
        Int >>= 8;
        if (AsnOctEnc (Asn, Chr) < 0)
            return (-1);
    }
    while ((Int != Lim) || (unsigned char) (Chr & 0x80) != Sgn);
    return (0);
}

/**************************************************************
** NAME:        AsnIntDecLng                              [API]
** SYNOPSIS:    int
**                  AsnIntDecLng
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc,
**                      long        *Int
**                  )
** DESCRIPTION: Decodes Long Integer.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Int: Integer.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIntDecLng
    (
        asn_sck *Asn,
        char    *Eoc,
        long    *Int
    )
{
    unsigned char
        Chr;
    unsigned
        Len;

    if (AsnOctDec (Asn, &Chr) < 0)
        return (-1);
    *Int = (signed) Chr;
    Len = 1;
    while (Asn->SckPtr < Eoc)
    {
        if (++Len > sizeof (long))
            return (-1);
        if (AsnOctDec (Asn, &Chr) < 0)
            return (-1);
        *Int <<= 8;
        *Int |= Chr;
    }
    return (0);
}

/**************************************************************
** NAME:        AsnIntEncUns                              [API]
** SYNOPSIS:    int
**                  AsnIntEncUns
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc,
**                      unsigned    Int
**                  )
** DESCRIPTION: Encodes Unsigned Integer.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding.
**              Int: Integer.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIntEncUns
    (
        asn_sck     *Asn,
        char        **Eoc,
        unsigned    Int
    )
{
    unsigned char
        Chr;

    *Eoc = Asn->SckPtr;
    do
    {
        Chr = (unsigned char) Int;
        Int >>= 8;
        if (AsnOctEnc (Asn, Chr) < 0)
            return (-1);
    }
    while ((Int != 0) || (Chr & 0x80) != 0x00);
    return (0);
}

/**************************************************************
** NAME:        AsnIntDecUns                              [API]
** SYNOPSIS:    int
**                  AsnIntDecUns
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc,
**                      unsigned    *Int
**                  )
** DESCRIPTION: Decodes Unsigned Integer.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Int: Integer.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIntDecUns
    (
        asn_sck     *Asn,
        char        *Eoc,
        unsigned    *Int
    )
{
    unsigned char
        Chr;
    unsigned
        Len;

    if (AsnOctDec (Asn, &Chr) < 0)
        return (-1);
    *Int = Chr;
    if (Chr == 0)
        Len = 0;
    else
        Len = 1;
    while (Asn->SckPtr < Eoc)
    {
        if (++Len > sizeof (int))
            return (-1);
        if (AsnOctDec (Asn, &Chr) < 0)
            return (-1);
        *Int <<= 8;
        *Int |= Chr;
    }
    return (0);
}

/**************************************************************
** NAME:        AsnIntEncLngUns                           [API]
** SYNOPSIS:    int
**                  AsnIntEncLngUns
**                  (
**                      asn_sck         *Sck,
**                      char            **Eoc,
**                      long unsigned   Int
**                  )
** DESCRIPTION: Encodes Long Unsigned Integer.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding.
**              Int: Integer.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIntEncLngUns
    (
        asn_sck         *Asn,
        char            **Eoc,
        unsigned long   Int
    )
{
    unsigned char
        Chr;

    *Eoc = Asn->SckPtr;
    do
    {
        Chr = (unsigned char) Int;
        Int >>= 8;
        if (AsnOctEnc (Asn, Chr) < 0)
            return (-1);
    }
    while ((Int != 0) || (Chr & 0x80) != 0x00);
    return (0);
}

/**************************************************************
** NAME:        AsnIntDecLngUns                           [API]
** SYNOPSIS:    int
**                  AsnIntDecLngUns
**                  (
**                      asn_sck         *Sck,
**                      char            *Eoc,
**                      long unsigned   *Int
**                  )
** DESCRIPTION: Decodes Long Unsigned Integer.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Int: Integer.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnIntDecLngUns
    (
        asn_sck         *Asn,
        char            *Eoc,
        unsigned long   *Int
    )
{
    unsigned char
        Chr;
    unsigned
        Len;

    if (AsnOctDec (Asn, &Chr) < 0)
        return (-1);
    *Int = Chr;
    if (Chr == 0)
        Len = 0;
    else
        Len = 1;
    while (Asn->SckPtr < Eoc)
    {
        if (++Len > sizeof (long))
            return (-1);
        if (AsnOctDec (Asn, &Chr) < 0)
            return (-1);
        *Int <<= 8;
        *Int |= Chr;
    }
    return (0);
}

/**************************************************************
** NAME:        AsnBtsEnc                                 [API]
** SYNOPSIS:    int
**                  AsnBtsEnc
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc,
**                      char        *Bts,
**                      unsigned    BtsLen,
**                      char        BtsUnu
**                  )
** DESCRIPTION: Encodes Bit String.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding.
**              Bts: Pointer to begin of Bit String.
**              BtsLen: Length of Bit String in characters.
**              BtsUnu: Number of unused bits in last character.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnBtsEnc
    (
        asn_sck     	*Asn,
        char        	**Eoc,
        char        	*Bts,
        unsigned    	BtsLen,
        unsigned char   BtsUnu
    )
{
    *Eoc = Asn->SckPtr;
    Bts += BtsLen;
    while (BtsLen-- > 0)
    {
        if (AsnOctEnc (Asn, *--Bts) < 0)
            return (-1);
    }
    if (AsnOctEnc (Asn, BtsUnu) < 0)
        return (-1);
    return (0);
}

/**************************************************************
** NAME:        AsnBtsDec                                 [API]
** SYNOPSIS:    int
**                  AsnBtsDec
**                  (
**                      asn_sck     	*Sck,
**                      char        	*Eoc,
**                      char        	*Bts,
**                      unsigned    	BtsSze,
**                      unsigned    	*BtsLen,
**                      unsigned char   *BtsUnu
**                  )
** DESCRIPTION: Decodes Bit String.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Bts: Pointer to begin of Bit String.
**              BtsSze: Size of Bit String in characters.
**              BtsLen: Length of Bit String in characters.
**              BtsUnu: Number of unused bits in last character.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int                                                       
    AsnBtsDec
    (
        asn_sck     	*Asn,
        char        	*Eoc,
        char		*Bts,
        unsigned    	BtsSze,
        unsigned    	*BtsLen,
        unsigned char  	*BtsUnu
    )
{
    if (AsnOctDec (Asn, BtsUnu) < 0)
        return (-1);
    *BtsLen = 0;
    while (Asn->SckPtr < Eoc)
    {
        if (++(*BtsLen) > BtsSze)
            return (-1);
        if (AsnOctDec (Asn, (unsigned char *)Bts++) < 0)
            return (-1);
    }
    return (0);
}

/**************************************************************
** NAME:        AsnOtsEnc                                 [API]
** SYNOPSIS:    int
**                  AsnOtsEnc
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc,
**                      char        *Ots,
**                      unsigned    OtsLen
**                  )
** DESCRIPTION: Encodes Octet String.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding.
**              Ots: Pointer to begin of Octet String.
**              OtsLen: Length of Octet String in characters.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnOtsEnc
    (
        asn_sck     *Asn,
        char        **Eoc,
        char        *Ots,
        unsigned    OtsLen
    )
{
    *Eoc = Asn->SckPtr;
    Ots += OtsLen;
    while (OtsLen-- > 0)
    {
        if (AsnOctEnc (Asn, *--Ots) < 0)
            return (-1);
    }
    return (0);
}

/**************************************************************
** NAME:        AsnOtsDec                                 [API]
** SYNOPSIS:    int
**                  AsnOtsDec
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc,
**                      char        *Ots,
**                      unsigned    OtsSze,
**                      unsigned    *OtsLen
**                  )
** DESCRIPTION: Decodes Octet String.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Ots: Pointer to begin of Octet String.
**              OtsSze: Size of Octet String in characters.
**              OtsLen: Length of Octet String in characters.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnOtsDec
    (
        asn_sck     *Asn,
        char        *Eoc,
        char        *Ots,
        unsigned    OtsSze,
        unsigned    *OtsLen
    )
{
    *OtsLen = 0;
    while (Asn->SckPtr < Eoc)
    {
        if (++(*OtsLen) > OtsSze)
            return (-1);
        if (AsnOctDec (Asn, (unsigned char *)Ots++) < 0)
            return (-1);
    }
    return (0);
}

int
    AsnSbiEnc
    (
        asn_sck     *Asn,
        unsigned    Sbi
    )
{
    unsigned char
        Chr;

    Chr = (unsigned char) (Sbi & 0x7F);
    Sbi >>= 7;
    if (AsnOctEnc (Asn, Chr) < 0)
        return (-1);
    while (Sbi > 0)
    {
        Chr = (unsigned char) (Sbi | 0x80);
        Sbi >>= 7;
        if (AsnOctEnc (Asn, Chr) < 0)
            return (-1);
    }
    return (0);
}

int
    AsnSbiDec
    (
        asn_sck     *Asn,
        unsigned    *Sbi
    )
{
    unsigned char
        Chr;

    *Sbi = 0;
    do
    {
        if (AsnOctDec (Asn, &Chr) < 0)
            return (-1);
        *Sbi <<= 7;
        *Sbi |= Chr & 0x7F;
    }
    while ((Chr & 0x80) == 0x80);
    return (0);
}

/**************************************************************
** NAME:        AsnOjiEnc                                 [API]
** SYNOPSIS:    int
**                  AsnOjiEnc
**                  (
**                      asn_sck     *Sck,
**                      char        **Eoc,
**                      unsigned    *Oji,
**                      unsigned    OjiLen
**                  )
** DESCRIPTION: Encodes Obect Identifier.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding.
**              Oji: Pointer to begin of Object Identifier.
**              OjiLen: Length of Object Identifier in unsigneds.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnOjiEnc
    (
        asn_sck     *Asn,
        char        **Eoc,
        unsigned    *Oji,
        unsigned    OjiLen
    )
{
    unsigned
        Sbi;

    *Eoc = Asn->SckPtr;
    if (OjiLen < 2)
        return (-1);
    Sbi = Oji [1] + Oji [0] * 40;
    Oji += OjiLen;
    while (OjiLen-- > 2)
    {
        if (AsnSbiEnc (Asn, *--Oji) < 0)
            return (-1);
    }
    if (AsnSbiEnc (Asn, Sbi) < 0)
        return (-1);
    return (0);
}

/**************************************************************
** NAME:        AsnOjiDec                                 [API]
** SYNOPSIS:    int
**                  AsnOjiDec
**                  (
**                      asn_sck     *Sck,
**                      char        *Eoc,
**                      unsigned    *Oji,
**                      unsigned    OjiSze,
**                      unsigned    *OjiLen
**                  )
** DESCRIPTION: Decodes Obect Identifier.
**              Parameters:
**              Sck: Pointer to ASN1 socket.
**              Eoc: Pointer to end of encoding or NULL if
**                   indefinite.
**              Oji: Pointer to begin of Object Identifier.
**              OjiSze: Size of Obect Identifier in unsigneds
**              OjiLen: Length of Object Identifier in unsigneds.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AsnOjiDec
    (
        asn_sck     *Asn,
        char        *Eoc,
        unsigned    *Oji,
        unsigned    OjiSze,
        unsigned    *OjiLen
    )
{
    unsigned
        Sbi;

    if (OjiSze < 2)
        return (-1);
    if (AsnSbiDec (Asn, &Sbi) < 0)
        return (-1);
    if (Sbi < 40)
    {
        Oji [0] = 0;
        Oji [1] = Sbi;
    }
    else if (Sbi < 80)
    {
        Oji [0] = 1;
        Oji [1] = Sbi - 40;
    }
    else
    {
        Oji [0] = 2;
        Oji [1] = Sbi - 80;
    }
    *OjiLen = 2;
    Oji += 2;
    while (Asn->SckPtr < Eoc)
    {
        if (++(*OjiLen) > OjiSze)
            return (-1);
        if (AsnSbiDec (Asn, Oji++) < 0)
            return (-1);
    }
    return (0);
}

