/**********************************************************************
** MODULE INFORMATION*
**********************
**      FILE     NAME:       SAGE.C
**      SYSTEM   NAME:       Sage
**      ORIGINAL AUTHOR(S):  Jan van Oorschot
**      VERSION  NUMBER:     1.0
**      CREATION DATE:       89/10/21
**                                                                  
** DESCRIPTION: The Sage functions for DSCHEME
***********************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** CHANGER:     $Author$
** WORKFILE:    $Workfile$
** LOGFILE:     $Logfile$
** LOGINFO:     $Log$
**********************************************************************/
#define DEBUG

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <sys/time.h>

#include "schinc.h"
#include "schdef.h"
#include "snmp.h"

STATIC pkt_msg *SchemeToSNMP __((pkt_msg *Msg, CELP pair));
STATIC CELP SNMPToScheme __((pkt_msg *Msg));

STATIC CELP CDECL Sg_open     __((int n,CELP arg));
STATIC CELP CDECL Sg_close    __((CELP arg));
STATIC CELP CDECL Sg_get      __((int n,CELP arg));
STATIC CELP CDECL Sg_getnext  __((int n,CELP arg));
STATIC CELP CDECL Sg_set      __((int n,CELP arg));
STATIC CELP CDECL Sg_sleep    __((CELP arg));
STATIC CELP CDECL Sg_getenv   __((CELP arg));

/* connection structure */

#define   SAGE_MAX_CONNECT  4

typedef struct _sage_connect
{
    mgt_sck *Sck;    /* NULL iff not used */
    pkt_msg Msg;
} SAGE_CONNECT;

SAGE_CONNECT  sCon[SAGE_MAX_CONNECT];

static EXTDEF extensions[]=
    { 
        {"SNMP-OPEN",(EXTPROC)Sg_open,      -1},
        {"SNMP-CLOSE",(EXTPROC)Sg_close,    1, TYPE_INT},
        {"SNMP-GET",(EXTPROC)Sg_get,        -1},
        {"SNMP-GETNEXT",(EXTPROC)Sg_getnext,-1},
        {"SNMP-SET",(EXTPROC)Sg_set,        -1},
        {"SLEEP", (EXTPROC)Sg_sleep,        1,TYPE_INT},
        {"GETENV", (EXTPROC)Sg_getenv,        1,TYPE_STR},
        ENDOFLIST
    };


/***************************************************************
** NAME:        DSsage                                     [API]
** SYNOPSIS:    int DSsage(glo)
**              GLOBAL *glo;
** DESCRIPTION: Initializes and link Sage extensions to the
**              DScheme system.
** RETURNS:     S_ERROR, if error occured.
**              S_OKAY otherwise.
***************************************************************/
int PASCAL DSsage(glo)
GLOBAL *glo;
{
    int iCnt;
    
    /* initialise the connection array */
    for(iCnt=0;iCnt<SAGE_MAX_CONNECT;iCnt++)
    {
	sCon[iCnt].Sck = NULL;
	sCon[iCnt].Msg.MsgPdu.PduRor.RorRid = 0;
    }
    return(DSmultidef(glo,extensions));        /* link extended functions */
}

/***************************************************************
** NAME:        Sg_open
** SYNOPSIS:    STATIC CELP Sg_open(arg)
**              CELP arg;
** DESCRIPTION: Opens an SNMP connection to  a host.
**              <arg> should contain the following arguments:
**                  - hostname        (string)
**                  - port-number     (integer)
** RETURNS:     int, connection number
***************************************************************/
STATIC
CELP Sg_open(n,arg)
int n;
CELP arg;
{
    int iCnt,iFound;
    CELP arg1,arg2;
    
    
    /* check if arguments correct */
    if (n!=2) DSERROR(ERRARC,arg);
    arg1 = CARpart(arg);
    arg2 = CDRpart(arg);

    
    TYPCHECK(arg2,TYPE_INT);
    TYPCHECK(arg1,TYPE_STR);
    

    /* find a free Sage connection */
    for( iCnt=0,iFound=0;iCnt<SAGE_MAX_CONNECT && !iFound; iCnt++)
      if(sCon[iCnt].Sck==NULL)
	iFound=1;
    if(iFound==0)
      return(NIL);
    iCnt--;
    
    /* open the SNMP connection */
    sCon[iCnt].Sck = MgtOpn(
		      htons(INTpart(arg2)),
		      MgtAdr(STRPpart(arg1))
		      );
    
    if(sCon[iCnt].Sck == NULL)
      return(NIL);
    else
      return(DSINTCEL(iCnt));
}
/***************************************************************
** NAME:        Sg_close
** SYNOPSIS:    STATIC CELP Sg_close(arg)
**              CELP arg;
** DESCRIPTION: Closes an SNMP connection to  a host.
**              <arg> should contain the following arguments:
**                  - connection number (int)
** RETURNS:     int, connection number
***************************************************************/
STATIC
CELP Sg_close(arg)
CELP arg;
{
    MgtCls(sCon[INTpart(arg)].Sck);
    sCon[INTpart(arg)].Sck = NULL;
    return Q_invis;
}
/***************************************************************
** NAME:        Sg_get
** SYNOPSIS:    STATIC CELP Sg_get(arg,lst)
**              CELP arg;
**              CELP lst;
** DESCRIPTION: Get a series of SNMP variables,
**              <arg> should contain the following arguments:
**                  - connection number (int)
**                  - community
**                  - timeout
**                  - retries
**                  - list of requests, each containing
**                       - SNMP variable number (string)
**                       - SNMP type (int)
**                       - SNMP value (int,string,array)
**              For the GET request, the <type> and <value>
**              fields of each request are not used.
** RETURNS:     resulting list of requests.
***************************************************************/
STATIC
CELP Sg_get(n,arg)
CELP arg;
int n;

{
    CELP arg1,arg2,arg3,arg4,arg5;
    int iCon;
    int iRetries;
    long lTimeOut;
    
    /* check if arguments correct */
    if (n!=5) DSERROR(ERRARC,arg);
    arg1 = CARpart(arg); arg  = CDRpart(arg);
    arg2 = CARpart(arg); arg  = CDRpart(arg);
    arg3 = CARpart(arg); arg  = CDRpart(arg);
    arg4 = CARpart(arg); 
    arg5 = CDRpart(arg);
    
    TYPCHECK(arg1,TYPE_INT);
    TYPCHECK(arg2,TYPE_STR);
    TYPCHECK(arg3,TYPE_INT);
    TYPCHECK(arg4,TYPE_INT);
    TYPCHECK(arg5,TYPE_PAIR);

    /* fill the request unit */
    iCon = INTpart(arg1);
    lTimeOut = INTpart(arg3);
    iRetries = (int) INTpart(arg4);
        
    /* community */
    strncpy( sCon[iCon].Msg.MsgCom,STRPpart(arg2),STRLpart(arg2)); 
    sCon[iCon].Msg.MsgComLen = STRLpart(arg2);
    
    /* request type */
    sCon[iCon].Msg.MsgPdu.PduRor.RorTyp = PKT_GETRQS;
    sCon[iCon].Msg.MsgPdu.PduRor.RorRid++;
    sCon[iCon].Msg.MsgPdu.PduRor.RorErrSts = PKT_NOERROR;
    

    /* requests */
    SchemeToSNMP(&(sCon[iCon].Msg) , arg5);

    /* Do the call */
    if( MgtRqs(sCon[iCon].Sck,&(sCon[iCon].Msg),lTimeOut,iRetries) < 0)
    {
	return(NIL);
    }
    else
    {
	return( SNMPToScheme(&(sCon[iCon].Msg)));
    }
}
 

/***************************************************************
** NAME:        Sg_getnext
** SYNOPSIS:    STATIC CELP Sg_getnext(arg)
**              CELP arg;
** DESCRIPTION: Do a GetNext of a series of SNMP variables,
**              <arg> should contain the following arguments:
**                  - connection number (int)
**                  - list of requests, each containing
**                       - SNMP variable number (string)
**                       - SNMP type (int)
**                       - SNMP value (int,string,array)
**              For the GETNEXT request, the <type> and <value>
**              fields of each request are not used.
** RETURNS:     resulting list of requests.
***************************************************************/
STATIC
CELP Sg_getnext(n,arg)
CELP arg;
int n;

{
    CELP arg1,arg2,arg3,arg4,arg5;
    int iCon;
    int iRetries;
    long lTimeOut;
    
    /* check if arguments correct */
    if (n!=5) DSERROR(ERRARC,arg);
    arg1 = CARpart(arg); arg  = CDRpart(arg);
    arg2 = CARpart(arg); arg  = CDRpart(arg);
    arg3 = CARpart(arg); arg  = CDRpart(arg);
    arg4 = CARpart(arg); 
    arg5 = CDRpart(arg);
    
    TYPCHECK(arg1,TYPE_INT);
    TYPCHECK(arg2,TYPE_STR);
    TYPCHECK(arg3,TYPE_INT);
    TYPCHECK(arg4,TYPE_INT);
    TYPCHECK(arg5,TYPE_PAIR);

    /* fill the request unit */
    iCon = INTpart(arg1);
    lTimeOut = INTpart(arg3);
    iRetries = (int) INTpart(arg4);
        
    /* community */
    strncpy( sCon[iCon].Msg.MsgCom,STRPpart(arg2),STRLpart(arg2)); 
    sCon[iCon].Msg.MsgComLen = STRLpart(arg2);
    
    /* request type */
    sCon[iCon].Msg.MsgPdu.PduRor.RorTyp = PKT_NXTRQS;
    sCon[iCon].Msg.MsgPdu.PduRor.RorRid++;
    sCon[iCon].Msg.MsgPdu.PduRor.RorErrSts = PKT_NOERROR;
    

    /* requests */
    SchemeToSNMP(&(sCon[iCon].Msg) , arg5);

    /* Do the call */
    if( MgtRqs(sCon[iCon].Sck,&(sCon[iCon].Msg),lTimeOut,iRetries) < 0)
    {
	return(NIL);
    }
    else
    {
	return( SNMPToScheme(&(sCon[iCon].Msg)));
    }
}

/***************************************************************
** NAME:        Sg_set
** SYNOPSIS:    STATIC CELP Sg_set(arg)
**              CELP arg;
** DESCRIPTION: Sets a series of SNMP variables,
**              <arg> should contain the following arguments:
**                  - connection number (int)
**                  - list of requests, each containing
**                       - SNMP variable number (string)
**                       - SNMP type (int)
**                       - SNMP value (int,string,array)
** RETURNS:     resulting list of requests.
***************************************************************/
STATIC
CELP Sg_set(n,arg)
CELP arg;
int n;

{
    CELP arg1,arg2,arg3,arg4,arg5;
    int iCon;
    int iRetries;
    long lTimeOut;
    
    /* check if arguments correct */
    if (n!=5) DSERROR(ERRARC,arg);
    arg1 = CARpart(arg); arg  = CDRpart(arg);
    arg2 = CARpart(arg); arg  = CDRpart(arg);
    arg3 = CARpart(arg); arg  = CDRpart(arg);
    arg4 = CARpart(arg); 
    arg5 = CDRpart(arg);
    
    TYPCHECK(arg1,TYPE_INT);
    TYPCHECK(arg2,TYPE_STR);
    TYPCHECK(arg3,TYPE_INT);
    TYPCHECK(arg4,TYPE_INT);
    TYPCHECK(arg5,TYPE_PAIR);

    /* fill the request unit */
    iCon = INTpart(arg1);
    lTimeOut = INTpart(arg3);
    iRetries = (int) INTpart(arg4);
        
    /* community */
    strncpy( sCon[iCon].Msg.MsgCom,STRPpart(arg2),STRLpart(arg2)); 
    sCon[iCon].Msg.MsgComLen = STRLpart(arg2);
    
    /* request type */
    sCon[iCon].Msg.MsgPdu.PduRor.RorTyp = PKT_SETRQS;
    sCon[iCon].Msg.MsgPdu.PduRor.RorRid++;
    sCon[iCon].Msg.MsgPdu.PduRor.RorErrSts = PKT_NOERROR;
    

    /* requests */
    SchemeToSNMP(&(sCon[iCon].Msg) , arg5);

    /* Do the call */
    if( MgtRqs(sCon[iCon].Sck,&(sCon[iCon].Msg),lTimeOut,iRetries) < 0)
    {
	return(NIL);
    }
    else
    {
	/* return the result after translating it to scheme format */
	return( SNMPToScheme(&(sCon[iCon].Msg)));
    }
}
/***************************************************************
** NAME:        SchemeToSNMP
** SYNOPSIS:    SchemeToSNMP(Msg,pair)
**              MSG  Msg;    pointer to SNMP request block
**              CELP arg;     scheme list 
** DESCRIPTION: 
**              Converts a list of variable request
**              from Scheme format to SNMP request block
** RETURNS:     resulting SNMP request block pointer
***************************************************************/
pkt_msg *SchemeToSNMP( Msg, pair)
pkt_msg *Msg;
CELP    pair;
{
    CELP elem;
    CELP p;
    CELP arg1,arg2,arg3;
    char *pVar;
    int  iVarLen;
    int  iVarType;
    int iCnt;
        
    for(elem=pair,iCnt=0;ISTRUE(elem) && iCnt<PKT_SZERORLST;elem = CDRpart(elem),iCnt++)
    {
	/* get arguments */
	p = DsCar(elem);
	arg1 = DsCar(p);
	p = CDRpart(p);
	arg2 = DsCar(p);
	p = CDRpart(p);
	arg3 = DsCar(p);
	TYPCHECK(arg1,TYPE_STR);
	TYPCHECK(arg2,TYPE_INT);
	pVar = STRPpart(arg1);
	iVarLen = STRLpart(arg1);
	iVarType = INTpart(arg2);
	
	/* fill SNMP request block */
	Msg->MsgPdu.PduRor.RorLst[iCnt].ObjTag = iVarType;
	if ( StrDecOji(pVar,&(Msg->MsgPdu.PduRor.RorLst[iCnt])) < 0)
	  return(NIL);
	
 
	switch(iVarType)
	{
	  case PKT_NULL:
	    Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngInt = 0;
	    break;
	  case PKT_INTEGER:
	    TYPCHECK(arg3,TYPE_INT);
	    Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngInt = INTpart(arg3);
	    break;
	  case PKT_OCTETSTRING:
	    TYPCHECK(arg3,TYPE_STR);
            memcpy(
		   Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynBufChr,
		   STRPpart(arg3),
		   STRLpart(arg3));
	    Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSynLen = STRLpart(arg3);
	    break;
	  case PKT_OBJECTIDENTIFIER:
	    TYPCHECK(arg3,TYPE_STR);
	    StrDecSyn(
		      STRPpart(arg3),
		      &(Msg->MsgPdu.PduRor.RorLst[iCnt])
		      );
	    break;
	  case PKT_IPADDRESS:
	    TYPCHECK(arg3,TYPE_STR);
 	    Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngUns = 
	      MgtAdr(STRPpart(arg3));
	    break;
	  case PKT_COUNTER:
	    TYPCHECK(arg3,TYPE_INT);
 	    Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngUns = INTpart(arg3);
	    break;
	  case PKT_GAUGE:
	    TYPCHECK(arg3,TYPE_INT);
	    Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngUns = INTpart(arg3);
	    break;
	  case PKT_TIMETICKS:
	    TYPCHECK(arg3,TYPE_INT);
	    Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngUns = INTpart(arg3);
	    break;
	  case PKT_OPAQUE:
	    TYPCHECK(arg3,TYPE_STR);
            memcpy(
		   Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynBufChr,
		   STRPpart(arg3),
		   STRLpart(arg3));
	    Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSynLen = STRLpart(arg3);
	    break;
	  default:
	    break;
	}
    }
    Msg->MsgPdu.PduRor.RorLstLen = iCnt;
    return Msg;
}
/***************************************************************
** NAME:        SNMPToScheme
** SYNOPSIS:    SNMPToScheme(Msg)
**              MSG  Msg;     pointer to SNMP request block 
** DESCRIPTION: 
**              Converts a list of variable request
**              from SNMP request block to a Scheme list
** RETURNS:     Scheme list
***************************************************************/
STATIC
CELP SNMPToScheme(Msg)
  pkt_msg *Msg;
{
    int iCnt;
    CELP lst,elem,arg1,arg2,arg3;
    int iType,iLen;
    char cVar[254];
    char cVal[254];

    /* check if the Msg struct contains error */
    if(Msg->MsgPdu.PduRor.RorErrSts != PKT_NOERROR)
      return(NIL);
    lst = NIL;
    
    for(iCnt=Msg->MsgPdu.PduRor.RorLstLen-1 ;iCnt>=0; iCnt--)
    {
	/* decode argument <iCnt> */
	StrEncOji(cVar,&(Msg->MsgPdu.PduRor.RorLst[iCnt]));
	iType = Msg->MsgPdu.PduRor.RorLst[iCnt].ObjTag;
	switch(iType)
	{
	  case PKT_NULL:
	    arg3 = DSINTCEL(0);
	    break;
	  case PKT_INTEGER:
	    arg3 = DSINTCEL(Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngInt);
	    break;
	  case PKT_OCTETSTRING:
	    iLen = Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSynLen;
	    arg3 = DsGetStr(iLen);
            memcpy(
		   STRPpart(arg3),
		   Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynBufChr,
		   iLen
		   );
	    break;
	  case PKT_OBJECTIDENTIFIER:
	    StrEncSyn(
		      cVal,
		      &(Msg->MsgPdu.PduRor.RorLst[iCnt])
		      );
	    arg3 =  DsStrCell(cVal);
	    break;
	  case PKT_IPADDRESS:
	    StrEncSyn(
		      cVal,
		      &(Msg->MsgPdu.PduRor.RorLst[iCnt])
		      );
	    arg3 =  DsStrCell(cVal);
	    break;
	  case PKT_COUNTER:
	    arg3 = DSINTCEL(Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngUns);
	    break;
	  case PKT_GAUGE:
	    arg3 = DSINTCEL(Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngUns);
	    break;
	  case PKT_TIMETICKS:
	    arg3 = DSINTCEL(Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynLngUns);
	    break;
	  case PKT_OPAQUE:
	    iLen = Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSynLen;
	    arg3 = DsGetStr(iLen);
            memcpy(
		   STRPpart(arg3),
		   Msg->MsgPdu.PduRor.RorLst[iCnt].ObjSyn.SynBufChr,
		   iLen
		   );
	    break;
	  default:
	    DSERROR(1001,DsStrCell("Sage internal Inconsistency"));

	    break;
	}
	/* create list element */
	arg1 = DsStrCell(cVar);
	arg2 = DSINTCEL(iType);
	elem = DsCons(arg1,DsCons(arg2,DsCons(arg3,NIL)));
	lst = DsCons(elem,lst);
    }
    return lst;
}
/***************************************************************
** NAME:        Sg_sleep
** SYNOPSIS:    STATIC CELP Sg_sleep(arg)
**              CELP arg;
** DESCRIPTION: Sleeps for the given number of milliseconds.
**              <arg> should contain the following arguments:
**                  - # millisecs
** RETURNS:     none
***************************************************************/
STATIC
CELP Sg_sleep(arg)
CELP arg;
{
    struct timeval timeout;

    timeout.tv_sec = INTpart(arg) / 1000000L;
    timeout.tv_usec = (INTpart(arg) % 1000000L);

    (void) select(1, 0, 0, 0, &timeout);
    return Q_invis;
}
/***************************************************************
** NAME:        Sg_getenv
** SYNOPSIS:    STATIC CELP Sg_getenv(arg)
**              CELP arg;
** DESCRIPTION: Get the value of the given environment variable
**              <arg> should contain the following arguments:
**                  - # name env var
** RETURNS:     none
***************************************************************/
STATIC
CELP Sg_getenv(arg)
CELP arg;
{
    char *p;
    p= getenv(STRPpart(arg));
    if (p==NULL)
      p="";
    return(DsStrCell(p));
}

	
	
      
    
	


