/* 1235, Tue 28 Jun 94

   NMC_PARS.C:  Scanner/parser for nmc config and rules files

   Copyright (C) 1992-1994 by Nevil Brownlee,
   Computer Centre,  The University of Auckland */

#include <stdio.h>
#include <ctype.h>
#include <errno.h>

#include <string.h>
#include <malloc.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>

#include "ausnmp.h"

#include "snmp.h"
#include "snmpimpl.h"
#include "asn1.h"
#include "snmpclnt.h"
#include "snmpapi.h"
#include "mib.h"

#include "nmc.h"

void mswait(unsigned int ms)  /* Wait for ms milliseconds */
{
   struct timeval timeout;
   timeout.tv_sec = ms/1000;  timeout.tv_usec = (long)(ms%1000)*1000L;
   select(FD_SETSIZE, 0, 0, 0, &timeout);
   }

FILE *wfopen(fn)  /* Open next file in sequence for write */
char *fn;
{
   char lfn[NAME_LN];
   int n;
   FILE *f;
   for (n = 1; ; ++n) {
      sprintf(lfn,"%s.%03d",fn,n);
      if ((f = fopen(lfn,"r")) == NULL) {
         if ((f = fopen(lfn,"w")) != NULL) return f;
         printf("Failed to open %s\n",lfn);
         exit(0);
         }
      fclose(f);
      }
   }

char *gnbr(unsigned int *n, char *s)
   /* Get nbr from 's', return updated 's' */
{
   unsigned int v = 0, b = 10, d;
   while (*s == ' ') ++s;
   if (*s == 'x' || *s == 'X') {
      ++s;  b = 16;
      }
   else if (*s == '0') {
      ++s;  b = 8;
      }
   for (;;) {
      d = *s;
      if (b == 16) {
         if (!isxdigit(d)) break;
         }
      else {
         if (!isdigit(d)) break;
         if (b == 8 && d > '7') break;
         }
      if (d <= '9') d -= '0';
      else if (d <= 'F') d -= ('A'-10);
      else d -= ('a'-10);
      v = v*b + d;  ++s;
      }
   *n = v;  return s;
   }

char *gcstring(char *s, int *len)  /* Get string from s */
{
   static char escin[] = {
      'b', 'f', 'n', 'r', 't', 'v','\\','\'','\"','\?', 0 };
   static char escout[] ={
      '\b','\f','\n','\r','\t','\v','\\','\'','\"','\?' };
   unsigned int c,j;
   char sbuf[80], *t = sbuf, *rp;
   while (*s) {
      if (*s == '\\') {
	 ++s;  c = *s;
	 if (c == '0' || c == 'x' || c == 'X') {  /* Octal or hex nbr */
	    s = gnbr(&c,s);  *t++ = c;
	    }
	 else if isdigit(c) {  /* Octal number */
	    --s;  *s = '0';  /* Make gnbr use base 8 */
	    s = gnbr(&c,s);  *t++ = c;
	    }
	 else {
	    j = 0;  do {
	       if (c == escin[j]) break;
	       } while (escin[++j] != 0);
	    if (escin[j] != 0) {
	       *t++ = escout[j];  ++s;
	       }
	    }
	 }
      else *t++ = *s++;
      }
   *t = '\0';
   rp = malloc(*len = t-sbuf);  strcpy(rp,sbuf);
   return rp;
   }

static char ftb[32];

char *fmt_time(time_t *t)
{
   char *ts = ctime(t);
   strncpy(&ftb[0],  &ts[11], 9);  /* 17:31:42_ */
   strncpy(&ftb[9],  &ts[0],  4);  /* Thu_ */
   strncpy(&ftb[13], &ts[8],  3);  /* 23_  */
   strncpy(&ftb[16], &ts[4],  4);  /* Sep_ */
   strncpy(&ftb[20], &ts[22], 2);  /* 93   */
   ftb[22] = '\0';
   return ftb;
   }   

char *uptime_string(unsigned long timeticks, char *buf)
{
   int	seconds, minutes, hours, days;

   timeticks /= 100;
   days = timeticks / (60 * 60 * 24);
   timeticks %= (60 * 60 * 24);

   hours = timeticks / (60 * 60);
   timeticks %= (60 * 60);

   minutes = timeticks / 60;
   seconds = timeticks % 60;

   if (days == 0){
      sprintf(buf, "%d:%02d:%02d", hours, minutes, seconds);
      }
   else if (days == 1) {
      sprintf(buf, "%d day, %d:%02d:%02d",
	 days, hours, minutes, seconds);
      }
   else {
      sprintf(buf, "%d days, %d:%02d:%02d",
	 days, hours, minutes, seconds);
      }
   return buf;
   }

void printaddress(FILE *f, unsigned char *a, unsigned char addrsz)
{
   int j;
   switch (addrsz) {
   case MAC_ADDR_LEN:
   case NSAP_ADDR_LEN:
      fprintf(f,"%02X", a[0]);
      for (j = 1; j != addrsz; ++j) fprintf(f,"-%02X", a[j]);
      break;
   case DETAIL_ADDR_LEN:
      fprintf(f,"%u", a[0]*256 + a[1]);
      break;
   default:
      fprintf(f,"%u", a[0]);
      for (j = 1; j != addrsz; ++j) fprintf(f,".%u", a[j]);
      break;
      }
   }

void printruleaddress(FILE *f, unsigned char *a)
{
   int j, k;
   fprintf(f,"%u", a[0]);
   for (k = RULE_ADDR_LEN-1; k != 0; --k) if (a[k] != 0) break;
   for (j = 1; j <= k; ++j) fprintf(f,".%u", a[j]);
   }


int parse_open(char *fn)  /* Open file and initialise parser */
{
   if ((rfp = fopen(fn, "r")) == NULL) return 0;
   rferrors = rule_line = 0;  ic = '\n';
   return 1;
   }

void getarg(char *arg)
{
   for (;;) {
      if (ic == EOF) {
         *arg = NULL;  return;
         }
      if (!isspace(ic)) break;
      nextchar();
      }
   do {
      *arg++ = ic;
      nextchar();
      } while (ic != EOF && !isspace(ic));
   *arg = NULL;  return;
   }

int nextchar()
{
   lic = ic;
   for (;;) {
      if (lic == '\n') {
	 if (fgets(inbuf, sizeof(inbuf), rfp) == NULL) return ic = EOF;
	 iblisted = 0;  ++rule_line;
	 ibp = inbuf;
	 }
      ic = *ibp++;
      if (ic == '#') lic = '\n';  /* Ignore comments */
      else return ic;
      }
   }

int wordis(p,w)
char *p, *w;
{

   if (strlen(p) != strlen(w)) return 0;  /* Lengths differ */
   return strcmp(p,w) == 0;
   }

void p_error(char *msg)
{
   if (!iblisted) {
      printf("RULEFILE line %d: %s\n", rule_line,inbuf);
      iblisted = 1;
      }
   printf("%s !!!\n", msg);
   ++rferrors;
   }

int getword()
{
   char wbuf[30], *wp = wbuf;
   int j;
   for (;;) {
      *wp++ = tolower(ic);
      nextchar();
      if (ic == EOF) return EOF;
      if (!isalnum(ic) && ic != '_') break;
      }
   *wp = NULL;

   if (wordis(wbuf,"ip")) return AT_IP;
   if (wordis(wbuf,"novell")) return AT_NOVELL;
   if (wordis(wbuf,"decnet")) return AT_DECNET;
   if (wordis(wbuf,"ethertalk")) return AT_ETHERTALK;
   if (wordis(wbuf,"clns")) return AT_CLNS;
   if (wordis(wbuf,"other")) return AT_OTHER;

   if (wordis(wbuf,"count")) return RA_COUNT;
   if (wordis(wbuf,"tally")) return RA_TALLY;
   if (wordis(wbuf,"aggregate")) return RA_AGGREGATE;
   if (wordis(wbuf,"succeed")) return RA_SUCCEED;
   if (wordis(wbuf,"fail")) return RA_FAIL;
   if (wordis(wbuf,"pushto")) return RA_PUSHTO;
   if (wordis(wbuf,"popto")) return RA_POPTO;
   if (wordis(wbuf,"goto")) return RA_GOTO;
   if (wordis(wbuf,"gosub")) return RA_GOSUB;
   if (wordis(wbuf,"return")) return RA_RETURN;
   if (wordis(wbuf,"assign")) return RA_ASSIGN;

   if (wordis(wbuf,"set")) return RF_SET;
   if (wordis(wbuf,"rules")) return RF_RULES;
   if (wordis(wbuf,"actions")) return RF_ACTIONS;
   if (wordis(wbuf,"format")) return RF_FORMAT;
   if (wordis(wbuf,"statistics")) return RF_STATS;

   if (wordis(wbuf,"icmp")) return PT_ICMP;
   if (wordis(wbuf,"tcp")) return PT_TCP;
   if (wordis(wbuf,"udp")) return PT_UDP;

   if (wordis(wbuf,"ftpdata")) return WNP_FTPDATA;
   if (wordis(wbuf,"ftp")) return WNP_FTP;
   if (wordis(wbuf,"telnet")) return WNP_TELNET;
   if (wordis(wbuf,"smtp")) return WNP_SMTP;
   if (wordis(wbuf,"domain")) return WNP_DOMAIN;
   if (wordis(wbuf,"nntp")) return WNP_NNTP;
   if (wordis(wbuf,"ntp")) return WNP_NTP;
   if (wordis(wbuf,"snmp")) return WNP_SNMP;

   if (wordis(wbuf,"forward")) return FTFORWARD;
   if (wordis(wbuf,"v1")) return FTV1;
   if (wordis(wbuf,"v2")) return FTV2;
   if (wordis(wbuf,"v3")) return FTV3;
   if (wordis(wbuf,"v4")) return FTV4;
   if (wordis(wbuf,"v5")) return FTV5;

   for (j = 0; j != SZ_ATTRIBS; ++j)
      if (wordis(wbuf,attribs[j].name)) return attribs[j].index;

   for (j = 0; j != st_top; ++j) {
      if (wordis(wbuf,symbol_table[j].id)) {
         st_index = j;  return RF_SYMBOL;
         }
      }
   ++st_top;  /* Add new symbol */
   strcpy(symbol_table[st_index = j].id,wbuf);
   symbol_table[j].kind = 0;  /* Unknown */
   return RF_SYMBOL;
   }

int getnbr(int maxnbr)  /* NB, 22 Oct 93 */
{
   int v = 0;
   for (;;) {
      if (ic == EOF) return EOF;
      if (isdigit(ic)) break;
      else if (isalpha(ic)) return getword();
      else nextchar();
      }
   for (;;) {
      v = v*10 + ic-'0';
      if (nextchar() == EOF) return EOF;
      if (!isdigit(ic)) break;
      }
   if (v > maxnbr) {  /* NB, 22 Oct 93 */
      sprintf(err_msg, "Number > %d",maxnbr);  /* NB, 22 Oct 93 */
      p_error(err_msg);
      }
   return v;
   }

unsigned int getint(unsigned int *base)
{
   char wbuf[30], *wp;
   unsigned long v;
   unsigned char c, word;
   int sic;  char *sibp;
   for (;;) {
      if (ic == EOF) return EOF;
      if (isalnum(ic)) break;
      else nextchar();
      }
   sic = ic;  sibp = ibp;  /* Save scanner state */
   for (word = 0, wp = wbuf; ; ) {
      if (isalpha(ic) && !isxdigit(ic)) word = 1;
      *wp++ = tolower(ic);
      nextchar();
      if (ic == EOF) break;
      if (!isalnum(ic)) break;
      }
   if (word) {
      if (*base == 0) *base = 10;
      ic = sic;  ibp = sibp;  /* Back up scanner */
      return getword();
      }
   *wp = NULL;
   if (*base == 0)
      *base = ic == '-' ? 16 : 10;  /* Hex address bytes separated by - */
   for (wp = wbuf, v = 0; *wp != NULL; ) {
      c = *wp++;
      if (isdigit(c)) c -= '0';
      else c = c-'a' + 10;
      v = v*(*base) + c;
      }
   return v;
   }

void getaddress(unsigned char *a,unsigned char len, unsigned char addrsz)
{
   unsigned int j, base, v;
   for (base = j = 0; j != addrsz; ++j) {
      v = getint(&base);
      if (len != 1 &&  /* Two or more bytes in address */
            ic != '.' && ic != '-' &&  /* No byte separator character */
            j == 0) {  /* Allow 16-bit decimal in first two bytes */
         a[j++] = (v>>8) & 0xFF;  v &= 0x00FF;
         }
      a[j] = v;
      if (ic != '.' && ic != '-') {  /* End of token; pad with zeroes */
         for (++j; j != addrsz; ++j) a[j] = 0;
         return;
         }
      }
   }

int getattribute(unsigned char col, struct flow_info *fp)
{
   switch(col) {
   case FTLOWINTERFACE:
      fp->LowInterface = getnbr(255);
      break;
   case FTLOWADJACENTTYPE:
      fp->LowAdjType = getnbr(255);
      break;
   case FTLOWADJACENTADDRESS:
      getaddress(fp->LowAdjAddress,MAC_ADDR_LEN, MAC_ADDR_LEN);
      break;
   case FTLOWADJACENTMASK:
      getaddress(fp->LowAdjMask,MAC_ADDR_LEN, MAC_ADDR_LEN);
      break;
   case FTLOWPEERTYPE:
      fp->LowPeerType = getnbr(255);
      break;
   case FTLOWPEERTYPEMASK:
      fp->LowPeerTypeMask = getnbr(255);
      break;
   case FTLOWPEERADDRESS:
      getaddress(fp->LowPeerAddress,PEER_ADDR_LEN, PEER_ADDR_LEN);
      break;
   case FTLOWPEERMASK:
      getaddress(fp->LowPeerMask,PEER_ADDR_LEN, PEER_ADDR_LEN);
      break;
   case FTLOWDETAILTYPE:
      fp->LowDetailType = getnbr(255);
      break;
   case FTLOWDETAILTYPEMASK:
      fp->LowDetailTypeMask = getnbr(255);
      break;
   case FTLOWDETAILADDRESS:
      getaddress(fp->LowDetailAddress,DETAIL_ADDR_LEN, DETAIL_ADDR_LEN);
      break;
   case FTLOWDETAILMASK:
      getaddress(fp->LowDetailMask,DETAIL_ADDR_LEN, DETAIL_ADDR_LEN);
      break;
   case FTHIINTERFACE:
      fp->HighInterface = getnbr(255);
      break;
   case FTHIADJACENTTYPE:
      fp->HighAdjType = getnbr(255);
      break;
   case FTHIADJACENTADDRESS:
      getaddress(fp->HighAdjAddress,MAC_ADDR_LEN, MAC_ADDR_LEN);
      break;
   case FTHIADJACENTMASK:
      getaddress(fp->HighAdjMask,MAC_ADDR_LEN, MAC_ADDR_LEN);
      break;
   case FTHIPEERTYPE:
      fp->HighPeerType = getnbr(255);
      break;
   case FTHIPEERTYPEMASK:
      fp->HighPeerTypeMask = getnbr(255);
      break;
   case FTHIPEERADDRESS:
      getaddress(fp->HighPeerAddress,PEER_ADDR_LEN, PEER_ADDR_LEN);
      break;
   case FTHIPEERMASK:
      getaddress(fp->HighPeerMask,PEER_ADDR_LEN, PEER_ADDR_LEN);
      break;
   case FTHIDETAILTYPE:
      fp->HighDetailType = getnbr(255);
      break;
   case FTHIDETAILTYPEMASK:
      fp->HighDetailTypeMask = getnbr(255);
      break;
   case FTHIDETAILADDRESS:
      getaddress(fp->HighDetailAddress,DETAIL_ADDR_LEN, DETAIL_ADDR_LEN);
      break;
   case FTHIDETAILMASK:
      getaddress(fp->HighDetailMask,DETAIL_ADDR_LEN, DETAIL_ADDR_LEN);
      break;
   default:
      sprintf(err_msg,"Attribute %d not allowed in action",col);
      p_error(err_msg);
      }
   }

int scan_rulefile(ms,doset,list)
struct meter_status *ms;
int doset,list;
{
   struct rule_info ri;
   int rule_set, nrules, nactions, n, kind;
   struct flow_info ai;
   unsigned char a, b, actions[1+NATTRIBS];
   char sbuf[32], *sp;
   int len;

   if (!parse_open(ms->rulefile)) {
      printf("   Couldn't open rule file %s !!!\n", ms->rulefile);
      return 0;  /* Fail */
      }

   rule_set = 0;
   nrules = nactions = 0;  kind = RF_RULES;
   for (;;) {
      do {  /* First char of a line */
	 nextchar();
	 if (ic == EOF) break;
	 } while (lic != '\n');
      if (ic == EOF) break;
      n = getnbr(255);  /* What kind of line is it? */
      if (ic == EOF) break;
      if (n == RF_SET) {
         ri.RuleSet = rule_set = getnbr(255);
         kind = RF_RULES;  continue;
         }
      if (n == RF_RULES) {
         kind = RF_RULES;  continue;
         }
      if (n == RF_ACTIONS) {
         kind = RF_ACTIONS;   continue;
         }
      if (n == RF_FORMAT) {
         for (a = 0; a != NATTRIBS+1; ++a)
            ms->format[a] = ms->required[a] = 0;
         for (a = 0; ; ) {
            n = getnbr(255);
            if (n < NULL || n > FTLASTTIME) {
               sprintf(err_msg,"Attribute %d not allowed in format",n);
               p_error(err_msg);
	       }
            else {
               ms->format[a] = n;  ms->required[n] = 1;
               }
            if (ic == ';' || ic == EOF) break;
            for (;;) {
               if (ic == EOF) return EOF;
               if (ic == '\"' || isalnum(ic)) break;
               else nextchar();
               }
            if (ic == '\"') {  /* Separator string */
               nextchar();
               for (sp = sbuf; ; ) {
                  if (ic == EOF) {
                     p_error("Missing \"");
                     break;
	             }
                  else if (ic == '\"') {
                     *sp = '\0';
                     ms->separator[a] = gcstring(sbuf, &len);
                     break;
                     }
                  else *sp++ = ic;
                  nextchar();
                  }
               }
            else ms->separator[a] = " ";
            ++a;
            if (ic == ';' || ic == EOF) break;
            }
         if (list) {
	    printf("Format: ");
            for (n = ms->format[a = 0]; ; ) {
               for (b = 1; attribs[b].index != n; ++b) ;
               if (n != NULL) printf(attribs[b].name);
               if ((n = ms->format[a+1]) == NULL) break;
               printf(ms->separator[a++]);
               }
            printf("\n");
            }
         continue;
         }
      if (n == RF_STATS) {
         ms->statsreqd = 1;  /* Collect statistics */
         continue;
         }

      while (n == RF_SYMBOL) {  /* Allow multiple labels */
         if (pass == 1) {
            if (symbol_table[st_index].kind == 0) {
               symbol_table[st_index].kind = kind;
               symbol_table[st_index].value = 
                  (kind == RF_RULES ? nrules : nactions) + 1;
               if (verbose) printf("line %d: %s is %s %d\n", rule_line,
                  symbol_table[st_index].id,
                  kind == RF_RULES ? "rule" : "attribute",
                  symbol_table[st_index].value);
               }
            else {
               sprintf(err_msg,"Duplicate label %s", 
                  symbol_table[st_index].id);
               p_error(err_msg);
	       }
            }
         nextchar();
         n = getnbr(255);
         }

      if (kind == RF_RULES) {
         ri.RuleSelector = n;  /* Attribute nbr */
         getaddress(ri.RuleMask,attribs[n].len, RULE_ADDR_LEN);
         if (ic == EOF) break;
         getaddress(ri.RuleMatchedValue,attribs[n].len, RULE_ADDR_LEN);
         if (ic == EOF) break;
         ri.RuleAction = getnbr(255);
         if (ic == EOF) break;
         ri.RuleJumpIndex = getnbr(32767);  /* NB, 22 Oct 93 */
         if (ri.RuleJumpIndex == RF_SYMBOL) {
            ri.RuleJumpIndex = symbol_table[st_index].value;
            if (pass == 2) {
               if (ri.RuleJumpIndex == 0) {
                  sprintf(err_msg,"Undeclared label %s", 
                     symbol_table[st_index].id);
                  p_error(err_msg);
		  }
               else switch (ri.RuleAction) {
               case RA_COUNT:
               case RA_TALLY:
               case RA_AGGREGATE:
                  if (symbol_table[st_index].kind != RF_ACTIONS)
                     p_error("Action index required");
                  break;
               case RA_FAIL:
                  if (ri.RuleJumpIndex != 0)
                     p_error("Zero index required");
                  break;
               case RA_SUCCEED:
               case RA_PUSHTO:
               case RA_POPTO:
               case RA_GOTO:
               case RA_GOSUB:
                  if (symbol_table[st_index].kind != RF_RULES)
                     p_error("Rule index required");
                  break;
               case RA_RETURN:
               case RA_ASSIGN:
                  p_error("Constant index required");
                  break;
                  }
               }
            }
         else if (pass == 2) {  /* RuleJumpIndex not a symbol */
            n = ri.RuleJumpIndex;
            switch (ri.RuleAction) {
            case RA_COUNT:
            case RA_TALLY:
            case RA_AGGREGATE:
               if (n < 1 || n > ms->nactions)
                  p_error("Action index required");
               break;
            case RA_SUCCEED:
               if (n < 0 || n > ms->nrules)  /* 0 stops the rule match */
                  p_error("Rule index required");
               break;
            case RA_FAIL:
               if (n != 0)
                  p_error("Zero index required");
               break;
            case RA_PUSHTO:
            case RA_POPTO:
            case RA_GOTO:
            case RA_GOSUB:
               if (n < 1 || n > ms->nrules)
                  p_error("Rule index required");
               break;
            case RA_RETURN:
               if (n < 1 || n > NATTRIBS)
                  p_error("Attribute required");
               break;
            case RA_ASSIGN:
               if (n < 1 || n > ms->nrules/3)
                  p_error("Gosub offset required");
               break;
               }
	    }
         if (ic == EOF) break;
         ri.RuleNbr = ++nrules;
         if (list) {
	    printf("Rule %d,%d:  %d & ", rule_set,nrules,ri.RuleSelector);
	    printruleaddress(stdout, ri.RuleMask);
            printf(" = ");
	    printruleaddress(stdout, ri.RuleMatchedValue);
	    printf(": %d, %d\n", ri.RuleAction,ri.RuleJumpIndex);
	    }
         if (doset) add_rule(ms,&ri);  /* Add rule to meter's rule table */
	 }

      if (kind == RF_ACTIONS) {
         for (a = 0; a != NATTRIBS+1; ++a) actions[a] = 0;
         for (;;) {
            getattribute(n,&ai);
            if (n > NULL || n <= NATTRIBS) {
               actions[n] = 1;  /* Remember which attribs are set */
               }
            if (ic == ';' || ic == EOF) break;
            n = getnbr(255);
            }
         ++nactions;
         if (list) {
	    printf("Action %d,%d: ", rule_set,nactions);
            for (a = 1;  a != NATTRIBS+1; ++a) {
               if (actions[a] != 0) {
                  for (b = 1; attribs[b].index != a; ++b) ;
                  printf(" %s=",attribs[b].name);
                  write_attrib(stdout,&ai,a);
                  }
               }
            printf("\n");
            }
         if (doset) add_action(ms, &ai, rule_set,nactions, actions);
	 }
      }

   fclose(rfp);
   if (rferrors == 0) {
      ms->ruleset = rule_set;  ms->nrules = nrules;  ms->nactions = nactions;
      return 1;  /* Succeed */
      }
   return 0;  /* Fail */
   }

void parse_rulefile(struct meter_status *ms,
   int list, int syntax)
{
   int NewRuleSet;
   st_top = 0;  pass = 1;
   if (scan_rulefile(ms,0,0)) {  /* No errors so far */
      pass = 2;
      if (scan_rulefile(ms,0,list)) {  /* Still no errors */
         if (syntax || ms->ruleset == 0)  /* No SET statement */
            return;
         if (ms->nrules != 0) {  /* Rules were specified */
            if (ms->CurrentRuleSet == (NewRuleSet = ms->ruleset))  {
               ms->ruleset = 1;  /* Ask meter to use default rule set */
               set_rule_info(ms,1);
               ms->ruleset = NewRuleSet;
               }
            if (set_rule_info(ms,0)) {  /* Set rule+action table sizes */
               pass = 3;
               scan_rulefile(ms,1,0);  /* Download the rules and actions */
               }  
            }
         if (ms->CurrentRuleSet != ms->ruleset)
            set_rule_info(ms,1);  /* Start using specified rule set */
         }
      }
   }
