

/*
 * Allocator routines.
 *
 * These include the temporary allocator, which allocates only for the
 * duration of a single article.  The temporary pool is reset to read
 * in each header.   It consists of a static block, followed by large
 * mallocated permanent blocks that are grabbed when there is not enough
 * room for temporary allocs
 *
 * The temporary allocator is very fast, and of course there is no overhead
 * in freeing it.
 */

 /*
  * Newsclip(TM) Library Source Code.
  * Copyright 1989 Looking Glass Software Limited.  All Rights Reserved.
  * Unless otherwise licenced, the only authorized use of this source
  * code is compilation into a binary of the newsclip library for the
  * use of licenced Newsclip customers.  Minor source code modifications
  * are allowed.
  * Use of this code for a short term evaluation of the product, as defined
  * in the associated file, 'Licence', is permitted.
  */

#include "nl.h"

/*
 * The structure for the blocks
 */

typedef struct temp_block {
	struct temp_block *next_block;		/* next block in chain */
	unsigned int block_size;		/* size of this block */
	unsigned int current_pos;		/* allocation limit this block*/
	} *tbptr;

tbptr first_alloc = 0;			/* first perm_alloced temp block */
tbptr cur_block;			/* current block to alloc from*/
					/* also the end block in chain*/

init_tempalloc()
{
	extern char hdbuf[];
	extern tbptr alloc_block AC((unsigned int));

	first_alloc = alloc_block( 0 );
	reset_tempalloc();
}

/*
 * Reset temporary allocator.
 * This frees all tempoarily allocated memory.
 */



reset_tempalloc( )
{
	tbptr chain;		/* chain through temp mem to reset it */
	extern tbptr alloc_block AC((unsigned int));

	/* loop through blocks resetting them */
	for( chain = first_alloc; chain; chain = chain->next_block )
		chain->current_pos = sizeof(struct temp_block);
	cur_block = first_alloc;
}
		

/* Allocate a new block with at least the amount of memory given */

tbptr
alloc_block( alsize )
unsigned int alsize;			/* we need at least this much ram */
{
	unsigned int trysize;
	tbptr newblock;
	extern char *malloc AC((unsigned int));

	/* adjust the size to include room for the header */
	alsize += sizeof( struct temp_block );

	alsize = allign_size( alsize );

	/* normally get TBLOCK_SIZE bytes, unless asking for more */
	trysize = max( TBLOCK_SIZE, alsize );

	/* try to allocate blocks.  If it fails, keep cutting the block
	   size in half, as long as we're bigger than the block the user
	   wants and bigger than our minimum size */

	while( trysize >= alsize && trysize >= MIN_TBLOCK ) {
		newblock = (tbptr) malloc( trysize );
		if( newblock ) {
			newblock->block_size = trysize;
			newblock->current_pos = sizeof(struct temp_block);
			newblock->next_block = (tbptr)0;
			return newblock;
			}
		}
	/* we could not get it */
	error( "Abort: Out of Memory\n" );
	/*NOTREACHED*/
}

/* The actual temporary allocate call.  Normally it returns very quickly */



char *
temp_alloc( size )
unsigned int size;			/* size of region to allocate */
{
	char *ret;			/* returned memory */
	tbptr scan;			/* scan through blocks */
	tbptr newblock;			/* possible new block */

	size = allign_size( size );	/* adjust for allignment */

	/* check if there is room in the end block */
	if( cur_block->current_pos + size <= cur_block->block_size ) {
		ret = (char *)cur_block + cur_block->current_pos;
		cur_block->current_pos += size;
		return ret;
		}
	/* didn't fit in this block.  Let's scan the blocks for a fit */

	for( scan = first_alloc; scan; scan = scan->next_block ) 
		if( scan->current_pos + size <= scan->block_size ) {
			ret = (char *)scan + scan->current_pos;
			scan->current_pos += size;
			return ret;
			}
	/* did not find any memory anywhere.  Make a new block */

	newblock = alloc_block( size );
	if( first_alloc == 0 )
		first_alloc = newblock;
	/* chain it on the end */
	cur_block->next_block = newblock;
	cur_block = newblock;
	/* allocate the ram in the new block */
	ret = (char *)cur_block + cur_block->current_pos;
	cur_block->current_pos += size;
	return ret;
}

/* reallocate temporary memory.   This routine is particularly efficient
   if the block to be reallocated is the last block that was allocated. */

char *
temp_realloc( buf, oldsize, newsize )
char *buf;				/* buffer containing original memory */
unsigned int oldsize;			/* old size of buffer */
unsigned int newsize;			/* new size desired */
{
	char *newbuf;

	oldsize = allign_size( oldsize );
	newsize = allign_size( newsize );
	
	/* check to see if this is the last guy we allocated */
	if( buf + oldsize == ((char *)cur_block) + cur_block->current_pos ) {
		/* it is the same guy */
		if( newsize <= oldsize ) {	/* shrink block */
			cur_block->current_pos -= oldsize - newsize;
			return buf;
			}
		 else if( cur_block->current_pos - oldsize + newsize <=
						cur_block->block_size ) {
			/* expand the block */
			cur_block->current_pos += newsize - oldsize;
			return buf;
			}
		 else	/* free the block anyway */
			cur_block->current_pos -= oldsize;
		}
	/* ah well, easy attempts at realloc failed */
	/* this temp alloc can't go on top of our original buffer */
	newbuf = temp_alloc( newsize );
	/* block move the memory over */
	{
	register int i;
	register char *p1, *p2;
	i = oldsize;
	p1 = buf;
	p2 = newbuf;
	while( i-- )
		*p2++ = *p1++;
	}
	return newbuf;
}

/* Permanent allocation routines -- just an interface to malloc */

char *
perm_alloc( size )
unsigned int size;		/* size of block */
{
	char *p;
	extern char *malloc AC((unsigned int));

	p = malloc( size );
	if( !p )
		error( "Out of memory\n" );
	 else
		return p;
	/* NOTREACHED */
}

perm_free( block )
char *block;			/* block allocated by perm_alloc */
{
	free( block );
}
