
/*
 * Prepare a new newsrc file based on a subscription request and last article
 * seen counts from a previous newsrc file.
 * Copyright 1990 by Looking Glass Software Limited.
 * See the file READ.ME for licence details.
 */

#include <stdio.h>
#include "db.h"
#include "regexp.h"
#include "feed.h"



char *rcfile = NULL;

main( argc, argv )
int argc;	/* arg count */
char **argv;	/* arg vector */
{
	int argnum;
	char *strchr();
	FILE *the_rc;
	FILE *source;

	init_bases();
	read_active( ACTFILE );

	for( argnum = 1; argnum < argc; argnum++ ) {
		char *argline;
		char *argstr;		/* argument string */
		int argval;
		int isplus;		/* boolean tells +arg vs -arg */
		argline = argv[argnum];

		if (argstr = strchr(argline, '=')) {
			argstr++;
			argval = atoi(argstr);
			switch( argline[0] ) {
				case 'n':
					/* handle newsrc= option */
					rcfile = argstr;
					break;
				case 'a':
					/* additional active file */
					read_active( argstr );
					break;
				default:
					error( "Bad Option %s\n", argline );
				}
			}
		else if( (isplus = argline[0] == '+') || argline[0] == '-' ) {
			switch( argline[1] ) {
				default:
					error( "Bad Option %s\n", argline );
				}
			}
		else {
			/* code for untagged option */
			source = fopen( argline, "r" );
			if( !source )
				error( "Could not open %s\n", argline );
			read_newlist( source );
			fclose( source );
			}
		}
	/* body of program */

	if( rcfile ) {
		the_rc = fopen( rcfile, "r" );
		if( the_rc ) {
			read_oldrc( the_rc );
			fclose( the_rc );
			}
		 else
			fprintf( stderr, "Could not open .newsrc file %s\n", rcfile );
		}
	 /* no name, assume no previous rc */

	delete_exclusions();

	/* make sure no other program is using the .newsrc! */

	if( rcfile ) {
		the_rc = fopen( rcfile, "w" );
		if( !the_rc )
			error( "Could not open .newsrc file %s for writing\n", rcfile );
		}
	 else
		the_rc = stdout;
	write_newrc( the_rc );
	fclose( the_rc );

	
}

dbptr newsgroups;
dbptr distributions;
dbptr deletions;

/* prepare the databases */

init_bases()
{
	newsgroups = init_db( 1200, sizeof( group ) );
	distributions = init_db( 20, sizeof( group ) );
	deletions = init_db( 30, sizeof( group ) );
}
	

/* feeder's list of patterns to exclude & include
   recipient's list of patterns to include & exclude
   recipient's list of specific groups desired
 */


error( form, a, b, c, d, e, f, g, h )
char *form;
{
	fprintf( stderr, form, a,b,c,d,e,f,g,h );
	exit(1);
}


/* Read in the whole active file, record lowest groups
   Read in the old .newsrc and record numbers
   Read in the local exclusion file for this site, and local pattern file
   Tag groups that match local patterns
   Read in the new input file
   	Record remote exclusions
	If a group is found, tag it.
	If a pattern is found, tag all groups in active that match it.

   Delete all tagged groups that match exclusion pattern.

   Write out tagged groups:
	If new group, with 1-lowest article, subject to maximum
	If old group, with old number
	if old unsubscribed group and old number is <= lowest article,
	  do not write out.
 */


/*
 * Read the active file and remember what's in it.
 */

read_active(fname)
char *fname;
{
	FILE *active;
	char actbuf[MAX_GNAME+40];
	char *p;
	group *grp;

	active = fopen( fname, "r" );
	if( !active )
		error( "Could not open %s\n", fname );
	
	while( fgets( actbuf, sizeof(actbuf), active ) ) {
		p = strtok( actbuf, " \t" );
		if( p == NULL )
			break;
		grp = (group *)add_rec( newsgroups, p, AR_NEWONLY );
		if( !grp )
			continue;
		grp->gflags = 0;
		p = strtok( NULL, " \t" );
		if( p ) {
			grp->active_highest = atol( p );
			p = strtok( NULL, " \t" );
			if( p )
				grp->active_lowest = atol( p );
			}
		grp->newsrc_highest = 0;
		}
	fclose( active );
		
}

read_oldrc( desc )
FILE *desc;
{
	char rclbuf[MAX_GNAME+40];	/* no long lines, please! */
	char *p;			/* pointer into rc line */
	int flag;
	group *grp;


	flag = 0;

	while( fgets( rclbuf, sizeof(rclbuf), desc ) ) {
		p = strchr( rclbuf, ':' );
		if( p == NULL ) {
			p = strchr( rclbuf, '!' );
			flag |= GF_UNSUB;
			}
		if( p != NULL ) {
			*p = 0;
			grp = (group *)get_rec( newsgroups, rclbuf );
			if( !grp )
				continue;	/* unknown group? */
			/* is there a number?  If so it must be 1- */
			if( atoi( p + 1 ) > 0 ) {
				p = strchr( p + 1, '-' );
				if( p )
					grp->newsrc_highest = atol( p + 1 );
				}
			/* isolate group name */
			grp->gflags |= flag | GF_OLDRC;
			}
		}
}

/* read the list of new desired groups */

read_newlist( desc )
FILE *desc;
{
	char newline[MAX_LLEN];
	char *com, *arg;
	group *grp;

	while( fgets( newline, sizeof(newline), desc ) ) {

		com = strtok( newline, " \t\n" );
		if( !com )
			continue;
		arg = strtok( NULL, " \t\n" );
		
		switch( *com ) {
			case '#':		/* comment */
			case 'P':		/* allow password line? */
				continue;
			case ':':		/* subscribed group */
				if( !arg )
					continue;
				grp = (group *)add_rec( newsgroups, arg,
							AR_CREATE );
				grp->gflags |= GF_SELECTED;
				break;
			case 'D':		/* distribution */
				/* no commas to fake out delete rules */
				if( arg && strchr( arg, ',') == NULL )
					add_rec( distributions, arg, AR_CREATE);
				break;
			case '-':
			case '!':
				if( arg )
					add_rec( deletions, arg, AR_CREATE );
				break;
			case '+' :		/* pattern subscription */
				if( arg )
					subscribe_pattern( arg, 0 );
				break;
			case 'N':		/* new group pattern */
				if( arg ) {
					char *arg2;
					artnum limit;

					arg2 = strtok( NULL, " \t\n" );
					limit = arg2 ? atol(arg2) : NEW_LIMIT;
					subscribe_pattern( arg, limit );
					}
				break;
			default:
				fprintf( stderr, "Warning: Invalid input line starting with %s\n", com );
				break;
			}
		}

			
}

/* subscribe all the groups from the active file that match the pattern.
 * If the limit is non-zero, only subscribe groups with fewer than that
 * many articles (ie. new groups) or which still have article 10 around.
 */

subscribe_pattern(pat, limit)
char *pat;
artnum limit;		/* limit of articles for a 'new' group 0 == unlimited */
{
	group *grp;
	regexp *patcomp;

	if( pat == NULL )
		return;


	/* a group is 'new' if it has less than 'limit' (default 200) articles
	   or if the 10th article has yet to expire.  It thus gets subscribed to
	   with the 'new' patterns */


	patcomp = regcomp( pat );

	for( grp = (group *)first_rec(newsgroups); grp;
					grp = next_rec(newsgroups,grp)) 
		if( regexec( patcomp, grp->name ) &&
				(!limit || grp->active_highest < limit ) )
			grp->gflags |= GF_SELECTED;
	free( patcomp );
}

/*
 * As a last pass, delete any groups or distributions
 */

delete_exclusions()
{
	group *grp;
	group *distnext;		/* next in distribution chain */
	group *pattern;			/* pattern to delete */
	regexp *patcomp;		/* compiled regular expression */


	for( pattern = (group *)first_rec(deletions); pattern;
				pattern = next_rec( deletions, pattern ) ) {
		patcomp = regcomp( pattern->name );
		for( grp = (group *)first_rec(newsgroups); grp;
						grp = next_rec(newsgroups,grp)) 
			if( regexec( patcomp, grp->name ) )
				grp->gflags &= ~GF_SELECTED;
#ifdef DEL_DISTRS
		/* scan the distributions, too */
		for( grp = (group *)first_rec(distributions); grp;
							grp = distnext ) {
			distnext = next_rec( distributions, grp );
			if( regexec( patcomp, grp->name ) )
				del_rec( distributions, grp->name );
			}
#endif
		free( patcomp );		/* compiled expression freed */
		}

}

write_newrc( desc )
FILE *desc;
{
	group *grp;
	long readcount;

	/* print out the desired distributions */
	for( grp = (group *)first_rec(distributions); grp;
					grp = next_rec(distributions,grp)) {

		fprintf( desc, "options -D %s\n", grp->name );
		}


	for( grp = (group *)first_rec(newsgroups); grp;
					grp = next_rec(newsgroups,grp)) {

		/* output selected groups */
		if( grp->gflags & GF_SELECTED ) {
			/* make sure group is all read up to first avail
								article */
			if( grp->newsrc_highest < grp->active_lowest )
				grp->newsrc_highest = grp->active_lowest-1;
			/* set possible max read count if needed */
			if( grp->active_highest - grp->newsrc_highest >
								MAX_OLD_ARTS )
				readcount = grp->active_highest - MAX_OLD_ARTS;
			 else
				readcount = grp->newsrc_highest;
				
			/* if we have a number for it, use it */
			if( grp->gflags & GF_UNSUB ) {
				groupout( desc, grp->name, readcount );
				}
			else if( grp->gflags & GF_OLDRC )
				groupout( desc, grp->name,
						(long)grp->newsrc_highest);
			 else 
				groupout( desc, grp->name, readcount );
			}
		/* remember unsubscribed and de-subscribed grps, unless all
			sent articles have expired */
		else if( grp->gflags & GF_OLDRC &&
				grp->newsrc_highest >= grp->active_lowest) {
			fprintf( desc, "%s! 1-%ld\n", grp->name,
						(long)grp->newsrc_highest );
			
			}
		/* other groups just vanish */
		}
}

groupout( desc, name, num )
FILE *desc;
char *name;
long num;
{
	if( num > 0 )
		fprintf( desc, "%s: 1-%ld\n", name, num );
	 else
		fprintf( desc, "%s:\n", name );
}

void
regerror( str )
char *str;
{
	error( "regexp: %s\n", str );
}

zero( mem, bytes )
char *mem;
unsigned int bytes;
{
	memset( mem, 0, bytes );
}
