/*
 Copyright (c) 2002 RIPE

 All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of the author not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
 AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
 DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

------------------------------------------------------------------------------
 Module Header
 Filename          : supconf.c
 Purpose           : Configuration reader
 Author            : Can Bican <can@ripe.net>
 Language Version  : C
 $Id: supconf.c,v 1.11 2002/12/04 11:45:20 can Exp $
------------------------------------------------------------------------------
*/

#include "supconf.h"

void readConfFile(char *fileName, char *appLogDomain)
{
  FILE *f;
  gchar *buf, *bufv, *key = NULL, *val = NULL, *currentRule = NULL;
  gchar *ruleOp, *ruleRhs, *regexStr, *errStr;
  gboolean *queueDefinedList=NULL,inQueueDef = FALSE, inRuleDef = FALSE;
  queueDefinition *newDef;
  gint i,errStrSize, regResult, currentQueue = 0;

  queueDefinitions = NULL;
  QUEUECOUNT = 0;
  bufv = g_new(gchar, 1024);
  f = fopen(fileName, "r");
  if (f == NULL) {
    fprintf(stderr, "Can't open %s - %s", fileName, strerror(errno));
    exit(1);
  }
  while (fgets(bufv, 1000, f)) {
    buf = g_strdup(bufv);
    g_strchug(buf);
    g_strchomp(buf);
    if ((buf[0] != '#') && (buf[0] != 0)) {
      splitConfStr(buf, key, val);
      if (inQueueDef) {
	if (strcmp(key, "ENDQUEUE") == 0) {
	  inQueueDef = FALSE;
	} else if (inRuleDef) {
	  if (strcmp(key, "ENDRULE") == 0) {
	    inRuleDef = FALSE;
	    g_free(currentRule);
	  } else if ((buf[0] != '#') && (buf[0] != 0)) {
	    if (currentRule == NULL) {
	      fprintf(stderr, "No rule name\n");
	      exit(1);
	    }
	    if (currentQueue == 0) {
	      fprintf(stderr, "No queue number\n");
	      exit(1);
	    }
	    newDef = g_new(queueDefinition, 1);
	    newDef->queueNumber = currentQueue;
	    newDef->ruleName = g_strdup(currentRule);
	    if (key[0] == '~') {
	      regexStr = g_strdup(key + 1);
	      regexStr[strlen(regexStr) - 1] = 0;
	      newDef->isRegex = TRUE;
	      newDef->op = val[0];
	      val++;
	      g_strchug(val);
	      newDef->opRhs = atol(val);
	      regResult =
		  regcomp(&newDef->opLhsRegex, regexStr,
			  REG_EXTENDED | REG_ICASE | REG_NOSUB);
	      if (regResult != 0) {
		errStrSize =
		    regerror(regResult, &newDef->opLhsRegex, (char *) NULL,
			     (size_t) 0);
		errStr = g_new(gchar, errStrSize);
		regerror(regResult, &newDef->opLhsRegex, errStr,
			 errStrSize);
		fprintf(stderr, "error in regex \"%s\": %s\n", regexStr,
			errStr);
		g_free(errStr);
		exit(1);
	      }
	      g_free(regexStr);
	    } else {
	      newDef->isRegex = FALSE;
	      g_strchug(val);
	      g_strchomp(val);
	      splitConfStr(val, ruleOp, ruleRhs);
	      newDef->opLhs = g_strdup(key);
	      newDef->op = ruleOp[0];
	      newDef->opRhs = atol(ruleRhs);
	    }
	    if (!(isValidOp(newDef->op))) {
	      fprintf(stderr,"Invalid operation %c\n",newDef->op);
	      exit(1);
	    }
	    queueDefinitions=g_slist_insert(queueDefinitions,newDef,-1);
	  }
	} else if (strcmp(key, "RULE") == 0) {
	  inRuleDef = TRUE;
	  currentRule = g_strdup(val);
	} else {
	  fprintf(stderr, "Unknown keyword %s\n", key);
	  exit(1);
	}
      } else if (strcmp(key, "QUEUE") == 0) {
	inQueueDef = TRUE;
	currentQueue = atoi(val);
	queueDefinedList[currentQueue-1]=TRUE;
	if ((currentQueue < 1) || (currentQueue > QUEUECOUNT)) {
	  fprintf(stderr, "Invalid queue %s\n", val);
	  exit(1);
	}
      } else if (strcmp(key, "QUEUECOUNT") == 0) {
	QUEUECOUNT = atoi(val);
	queueDefinedList=g_new(gboolean,QUEUECOUNT);
	for (i=0;i<QUEUECOUNT;i++)
	  queueDefinedList[i]=FALSE;
      } else if (strcmp(key, "SERVERADDR") == 0) {
	SERVERADDR = g_strdup(val);
      } else if (strcmp(key, "FIRSTPORT") == 0) {
	FIRSTPORT = atoi(val);
      } else if (strcmp(key, "MAXDATALEN") == 0) {
	MAXDATALEN = atol(val);
      } else if (strcmp(key, "DBUPDATE") == 0) {
	DBUPDATE = g_strdup(val);
      } else if (strcmp(key, "LOGFILE") == 0) {
	LOGFILE = g_strdup(val);
      } else if (strcmp(key, "PIDFILE") == 0) {
	PIDFILE = g_strdup(val);
      } else if (strcmp(key, "ACLFILE") == 0) {
	ACLFILE = g_strdup(val);
      } else {
	fprintf(stderr, "Unknown keyword %s\n", key);
	exit(1);
      }
    }
    g_free(buf);
  }
  g_free(bufv);
  fclose(f);
  for(i=0;i<QUEUECOUNT;i++)
    if(!queueDefinedList[i]) {
      fprintf(stderr,"No rules for queue #%d\n",i+1);
      exit(1);
    }
  g_free(queueDefinedList);
}
