#include "ep_internal.h"

/*
  Check if a serial is revoked

  serial - serial
  issuer - Certificate issuer

  return - EP_GOOD_SIG if not revoked
           EP_BAD_SIG  if revoked
           EP_VERIFY_FATAL for file error
 */
int ep_crl_check(gchar* serial, gchar* issuer)
{
  FILE* crl;
#define LINE_LENGTH 400
  gchar string[LINE_LENGTH];
  gchar ifile[LINE_LENGTH];
  gchar dirfile[LINE_LENGTH];
  int i;

  LG_log(ep_ctx, LG_DEBUG, ">ep_crl_check: entered with serial [%s] issuer [%s]", serial, issuer);

  /* dont forget dir */
  for (i=0;i<strlen(issuer);i++)
  {
    if (issuer[i]=='/')
    {
      ifile[i] = '-';
    }
    else
    {
      ifile[i] = issuer[i];
    }
  }
  ifile[i] = 0;
  sprintf(dirfile, "%s/%s", ep_crl_dir, ifile);
  crl = fopen(dirfile, "r");
  if (crl == NULL)
  { 
    /* there is no CRL or can't open it */
    LG_log(ep_ctx, LG_DEBUG, "ep_crl_check: could not open crl file: [%s]", dirfile);
    LG_log(ep_ctx, LG_DEBUG, "<ep_crl_check: exiting with value EP_VERIFY_FATAL");
    return EP_VERIFY_FATAL;
  }
  while (fscanf(crl, "%s", string) != EOF)
  {
    if ( ! strcmp(string, serial) )
    {
      fclose(crl);
      LG_log(ep_ctx, LG_DEBUG, "ep_crl_check: serial found in crllist");
      LG_log(ep_ctx, LG_DEBUG, "<ep_crl_check: exiting with value EP_BAD_SIG");
      return EP_BAD_SIG;
    }
  }
  fclose(crl);
  LG_log(ep_ctx, LG_DEBUG, "<ep_crl_check: exiting with value EP_GOOD_SIG");
  return EP_GOOD_SIG;
}


int ep_verify_mail_X509_cert(gchar *mail_file, char *smime, int *serial_chk, char *dname)
{
#define LINE_LENGTH 400

  int retval = EP_NOT_SMIME;
  char *cmd = NULL;
  char *smime_cmd = "smime -verify -nochain";
  char *verify_cmd = "verify";
  FILE* general;
  GString* x509_line;
  gchar cert_file[LINE_LENGTH];
  gchar status_file[LINE_LENGTH];
  gchar tmp[LINE_LENGTH];
  gchar txt[LINE_LENGTH];
  gchar issuer[LINE_LENGTH];
  gchar serial[LINE_LENGTH];
 
  LG_log(ep_ctx, LG_DEBUG, ">ep_verify_mail_X509_cert: entered with smime [%s] file [%s]", 
                                 smime, mail_file);

  if  (ep_temporary_directory)
  {
    strcpy(tmp, ep_temporary_directory);
  }
  else
  {
    strcpy(tmp, "/tmp");
  }
  sprintf(cert_file,"%s/x509_cert_%d", tmp, getpid());
  sprintf(status_file,"%s/x509_status_%d", tmp, getpid());
  
  /* Check if the signature is valid */
  x509_line = g_string_new("openssl ");

  if ( strstr(smime, "smime") == smime )
  {
    g_string_append(x509_line, "smime -verify -nochain");
    g_string_append(x509_line, " -in ");
    g_string_append(x509_line, mail_file);
    g_string_append(x509_line, " -CAfile ");
    g_string_append(x509_line, ep_ca_file);
    g_string_append(x509_line, " -signer ");
    g_string_append(x509_line, cert_file);
    g_string_append(x509_line, " -out /tmp/irrelevant_text ");
    g_string_append(x509_line, " > ");
    g_string_append(x509_line, status_file);
    g_string_append(x509_line, " 2> ");
    g_string_append(x509_line, status_file);
  }
  else
  {
    g_string_append(x509_line, "verify");
    g_string_append(x509_line, " -CAfile ");
    g_string_append(x509_line, ep_ca_file);
    g_string_append(x509_line, " -out /tmp/irrelevant_text ");
    g_string_append(x509_line, " < ");
    g_string_append(x509_line, mail_file);
    g_string_append(x509_line, " > ");
    g_string_append(x509_line, status_file);
    g_string_append(x509_line, " 2> ");
    g_string_append(x509_line, status_file);
  }

  LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: openssl cmd:\n[%s]", x509_line->str);
  system(x509_line->str);
  g_string_free(x509_line, TRUE);

  /* Parsing openssl output */
  if (( general = fopen(status_file, "r")) == NULL)
  {
    LG_log(ep_ctx, LG_FATAL, "ep_verify_mail_X509_cert: Can't open status file [%s]", status_file);
    LG_log(ep_ctx, LG_DEBUG, "<ep_verify_mail_X509_cert: exiting with value EP_VERIFY_FATAL");
    return EP_VERIFY_FATAL;
  }

  while (fgets (txt, LINE_LENGTH - 1, general) != NULL)
  {
    LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: openssl return: [%s]", txt);
    if ((strstr(txt, "Verification successful")))
    {
      retval = EP_GOOD_SIG;
      break;
    }
    if ((strstr(txt, "Verification failure")))
    {
      retval = EP_BAD_SIG;
      break;
    }
  }
  fclose(general);
  unlink(status_file);

  if (retval == EP_NOT_SMIME)
  {
    LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: input stream NOT s/mime");
    unlink(cert_file);
    if ( strstr(smime, "smime") )
    {
      /* we are processing smime input but it is NOT smime, error */
      LG_log(ep_ctx, LG_DEBUG, "<ep_verify_mail_X509_cert: exiting with value %s", EP_verify_ret2str(retval));
      return retval;
    }
    else
    {
      /* we are NOT processing smime input so this error should not occur */
      LG_log(ep_ctx, LG_DEBUG, "<ep_verify_mail_X509_cert: exiting with value EP_VERIFY_FATAL");
      return EP_VERIFY_FATAL;
    }
  }
  else if (retval == EP_BAD_SIG)
  {
    unlink(cert_file);
    LG_log(ep_ctx, LG_DEBUG, "<ep_verify_mail_X509_cert: exiting with value %s", EP_verify_ret2str(retval));
    return retval;
  }

  /* Extract DN */
  x509_line = g_string_new("openssl ");
  g_string_append(x509_line, " x509 -text ");
  g_string_append(x509_line, " < ");
  g_string_append(x509_line, cert_file);
  g_string_append(x509_line, " > ");
  g_string_append(x509_line, status_file);
  g_string_append(x509_line, " 2> ");
  g_string_append(x509_line, status_file);
  LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: openssl cmd:\n[%s]", x509_line->str);
  system(x509_line->str);
  g_string_free(x509_line, TRUE);

  dname[0] = 0;
  issuer[0] = 0;
  serial[0] = 0;
  if (( general = fopen(status_file, "r")) == NULL)
  {
    LG_log(ep_ctx, LG_FATAL, "ep_verify_mail_X509_cert: Can't open status file [%s]", status_file);
    LG_log(ep_ctx, LG_DEBUG, "<ep_verify_mail_X509_cert: exiting with value EP_VERIFY_FATAL");
    return EP_VERIFY_FATAL;
  }
  
  /***********************************************************
    change this later to use -noout -subject, etc instead of counting spaces
    
    not so sure about these string copies either ???
  ************************************************************/
  
  while (fgets (txt, LINE_LENGTH - 1, general) != NULL)
  {
//    LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: openssl line: [%s]", txt);
    if (strncmp(txt, "        Subject:", 16) == 0)
    {
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: openssl line: [%s]", txt);
      strcpy(dname, txt+17);
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: dname [%s]", dname);
    }
    if (strncmp(txt, "        Issuer:", 15) == 0)
    {
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: openssl line: [%s]", txt);
      strcpy(issuer, txt+16);
      issuer[strlen(issuer)-1] = 0;
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: issuer [%s]", issuer);
    }
    if (strncmp(txt, "        Serial Number:", 22) == 0)
    {
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: openssl line: [%s]", txt);
      strcpy(serial, txt+29);
      serial[strlen(serial)-2] = 0;
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: serial [%s]", serial);
    }
  }
  fclose(general);
  unlink(status_file);
  unlink(cert_file);

  if (dname[0]  == 0)
  {
    LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: openssl: DN is empty");
    return EP_NO_DNAME;
  }
  LG_log(ep_ctx, LG_DEBUG, "ep_verify_mail_X509_cert: openssl: dname %s\n", dname);

  *serial_chk = ep_crl_check(serial, issuer);
  if ( *serial_chk == EP_VERIFY_FATAL )
  {
    retval = EP_VERIFY_FATAL;
  }

  LG_log(ep_ctx, LG_DEBUG, "<ep_verify_mail_X509_cert: exiting with value %s", EP_verify_ret2str(retval));
  return retval;
}


int ep_verify_sync_X509_cert(gchar *mail_file, char *smime, int *serial_chk, char *dname)
{
#define LINE_LENGTH 400

  int retval = EP_BAD_SIG;
  char *cmd = NULL;
  char *smime_cmd = "smime -verify -nochain";
  char *verify_cmd = "verify";
  FILE* general;
  GString* x509_line;
  gchar cert_file[LINE_LENGTH];
  gchar status_file[LINE_LENGTH];
  gchar tmp[LINE_LENGTH];
  gchar txt[LINE_LENGTH];
  gchar issuer[LINE_LENGTH];
  gchar serial[LINE_LENGTH];
 
  LG_log(ep_ctx, LG_DEBUG, ">ep_verify_sync_X509_cert: entered with smime [%s] file [%s]", 
                                 smime, mail_file);

  if  (ep_temporary_directory)
  {
    strcpy(tmp, ep_temporary_directory);
  }
  else
  {
    strcpy(tmp, "/tmp");
  }
  sprintf(cert_file,"%s/x509_cert_%d", tmp, getpid());
  sprintf(status_file,"%s/x509_status_%d", tmp, getpid());
  
  /* Check if the signature is valid */
  x509_line = g_string_new("openssl ");

  g_string_append(x509_line, "verify");
  g_string_append(x509_line, " -CAfile ");
  g_string_append(x509_line, ep_ca_file);
  g_string_append(x509_line, " < ");
  g_string_append(x509_line, mail_file);
  g_string_append(x509_line, " > ");
  g_string_append(x509_line, status_file);
  g_string_append(x509_line, " 2> ");
  g_string_append(x509_line, status_file);

  LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: openssl cmd:\n[%s]", x509_line->str);
  system(x509_line->str);
  g_string_free(x509_line, TRUE);

  /* Parsing openssl output */
  if (( general = fopen(status_file, "r")) == NULL)
  {
    LG_log(ep_ctx, LG_FATAL, "ep_verify_sync_X509_cert: Can't open status file [%s]", status_file);
    LG_log(ep_ctx, LG_DEBUG, "<ep_verify_sync_X509_cert: exiting with value EP_VERIFY_FATAL");
    return EP_VERIFY_FATAL;
  }

  while (fgets (txt, LINE_LENGTH - 1, general) != NULL)
  {
    LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: openssl return: [%s]", txt);
    if ((strstr(txt, "OK")))
    {
      retval = EP_GOOD_SIG;
      break;
    }
  }
  fclose(general);
  unlink(status_file);

  if (retval == EP_BAD_SIG)
  {
    LG_log(ep_ctx, LG_DEBUG, "<ep_verify_sync_X509_cert: exiting with value %s", EP_verify_ret2str(retval));
    return retval;
  }

  /* Extract DN */
  x509_line = g_string_new("openssl ");
  g_string_append(x509_line, " x509 -text ");
  g_string_append(x509_line, " < ");
  g_string_append(x509_line, mail_file);
  g_string_append(x509_line, " > ");
  g_string_append(x509_line, status_file);
  g_string_append(x509_line, " 2> ");
  g_string_append(x509_line, status_file);
  LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: openssl cmd:\n[%s]", x509_line->str);
  system(x509_line->str);
  g_string_free(x509_line, TRUE);

  dname[0] = 0;
  issuer[0] = 0;
  serial[0] = 0;
  if (( general = fopen(status_file, "r")) == NULL)
  {
    LG_log(ep_ctx, LG_FATAL, "ep_verify_sync_X509_cert: Can't open status file [%s]", status_file);
    LG_log(ep_ctx, LG_DEBUG, "<ep_verify_sync_X509_cert: exiting with value EP_VERIFY_FATAL");
    return EP_VERIFY_FATAL;
  }
  
  /***********************************************************
    change this later to use -noout -subject, etc instead of counting spaces
    
    not so sure about these string copies either ???
  ************************************************************/
  
  while (fgets (txt, LINE_LENGTH - 1, general) != NULL)
  {
//    LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: openssl line: [%s]", txt);
    if (strncmp(txt, "        Subject:", 16) == 0)
    {
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: openssl line: [%s]", txt);
      strcpy(dname, txt+17);
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: dname [%s]", dname);
    }
    if (strncmp(txt, "        Issuer:", 15) == 0)
    {
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: openssl line: [%s]", txt);
      strcpy(issuer, txt+16);
      issuer[strlen(issuer)-1] = 0;
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: issuer [%s]", issuer);
    }
    if (strncmp(txt, "        Serial Number:", 22) == 0)
    {
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: openssl line: [%s]", txt);
      strcpy(serial, txt+28);
      serial[strlen(serial)-2] = 0;
      LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: serial [%s]", serial);
    }
  }
  fclose(general);
  unlink(status_file);

  if (dname[0]  == 0)
  {
    LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: openssl: DN is empty");
    return EP_NO_DNAME;
  }
  LG_log(ep_ctx, LG_DEBUG, "ep_verify_sync_X509_cert: openssl: dname %s\n", dname);

  *serial_chk = ep_crl_check(serial, issuer);
  if ( *serial_chk == EP_VERIFY_FATAL )
  {
    retval = EP_VERIFY_FATAL;
  }

  LG_log(ep_ctx, LG_DEBUG, "<ep_verify_sync_X509_cert: exiting with value %s", EP_verify_ret2str(retval));
  return retval;
}


/*
  Processes an e-mail, trying to find a X509 credential reporting on it

  stream - e-mail

  return - A credential (can be non-valid) or NULL
 */
CR_credential_t* x509_process(gchar* stream)
{
#define LINE_LENGTH 400

  int retval;
  int serial_chk;
  FILE* mail_file_stream;
  GString* x509_line;
  gchar mail_file[LINE_LENGTH];
  gchar tmp[LINE_LENGTH];
  gchar dname[LINE_LENGTH];
  
  LG_log(ep_ctx, LG_DEBUG, ">x509_process: entered ");

  if  (ep_temporary_directory)
  {
    strcpy(tmp, ep_temporary_directory);
  }
  else
  {
    strcpy(tmp, "/tmp");
  }
  sprintf(mail_file,"%s/x509_mail_%d", tmp, getpid());
  if (( mail_file_stream = fopen(mail_file, "w")) == NULL)
  {
    LG_log(ep_ctx, LG_FATAL, "x509_process: Can't open mail file [%s]", mail_file);
    LG_log(ep_ctx, LG_DEBUG, "<x509_process: exiting with value NULL");
    return NULL;
  }
  fprintf(mail_file_stream, "%s", stream);
  fclose(mail_file_stream);

  dname[0] = 0;
  retval = ep_verify_mail_X509_cert(mail_file, "smime", &serial_chk, dname);
  unlink(mail_file);
  LG_log(ep_ctx, LG_DEBUG, "x509_process: serial_chk [%s] dname [%s]",
                                (serial_chk == EP_GOOD_SIG) ? "VALID" : "REVOKED", dname);

  if (retval == EP_NOT_SMIME)
  {
    LG_log(ep_ctx, LG_DEBUG, "<x509_process: exiting with value NULL");
    return NULL;
  }
  else if (retval == EP_BAD_SIG)
  {
    LG_log(ep_ctx, LG_DEBUG, "<x509_process: exiting with FALSE credential");
    return CR_credential_new(CR_X509, "unk DN", FALSE);
  }
  else if (retval == EP_NO_DNAME)
  {
    LG_log(ep_ctx, LG_DEBUG, "<x509_process: exiting with value NULL");
    return NULL;
  }
  else if (retval == EP_VERIFY_FATAL)
  {
    LG_log(ep_ctx, LG_DEBUG, "<x509_process: exiting with value NULL");
    return NULL;
  }

  LG_log(ep_ctx, LG_DEBUG, "<x509_process: exiting with %s credential",
                                (serial_chk == EP_GOOD_SIG) ? "GOOD" : "BAD");
  return CR_credential_new(CR_X509, dname, (serial_chk == EP_GOOD_SIG) ? TRUE : FALSE);
}


int ep_check_X509_cert( char *X509cert_file, gchar **ep_DN )
{
#define LINE_LENGTH 400

  int retval = EP_OK;
  int serial_chk;
  gchar dname[LINE_LENGTH];

  LG_log(ep_ctx, LG_DEBUG, ">ep_check_X509_cert: entered");

  dname[0] = 0;
  retval = ep_verify_sync_X509_cert(X509cert_file, "", &serial_chk, dname);
  LG_log(ep_ctx, LG_DEBUG, "ep_check_X509_cert: serial_chk [%s] dname [%s]",
                                (serial_chk == EP_GOOD_SIG) ? "VALID" : "REVOKED", dname);
  
  if (retval == EP_GOOD_SIG)
  {
    retval = EP_OK;
  }
  else if (retval == EP_VERIFY_FATAL)
  {
    retval = EP_FATAL;
  }
  else
  {
    retval = EP_FAIL;
  }
  
  if ( serial_chk == EP_BAD_SIG )
  {
    /* this cert has been revoked */
    retval = EP_FAIL;
  }
  else if ( serial_chk == EP_VERIFY_FATAL )
  {
    /* crllist read error */
    retval = EP_FATAL;
  }
  
  if ( retval == EP_OK )
  {
    /* set ep_DN */
    LG_log(ep_ctx, LG_DEBUG, "ep_check_X509_cert: set *ep_DN [%s]", dname);
    *ep_DN = g_strdup(dname);
  }

  LG_log(ep_ctx, LG_DEBUG, "<ep_check_X509_cert exiting with value %s", EP_ret2str(retval));
  return retval;
}
