/*
 *  Copyright 2004 the mime4j project
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
using System;
using System.IO;
using System.Collections;
using System.Text;

using NI.Email.Mime;
using NUnit.Framework;

namespace NI.Tests.Email.Mime {

	/**
	* 
	*
	* @author Niklas Therning
	* @version $Id: MimeStreamParserTest.java,v 1.6 2005/02/11 10:12:02 ntherning Exp $
	*/
	[TestFixture]
	public class MimeStreamParserTest {

			/*
		public void testRootAndBodyStreamsAreSynched() throws IOException {
			File dir = new File("testmsgs");
			File[] files = dir.listFiles();
	        
			for (int i = 0; i < files.length; i++) {
				File f = files[i];
	            
				if (f.getName().toLowerCase().endsWith(".msg")) {
					final RandomAccessFile file = new RandomAccessFile(f, "r");
					final EOLTrackingInputStream rootStream = 
						new EOLTrackingInputStream(
						new RandomAccessFileInputStream(file, 0, file.length()));
	                
					ContentHandler handler = new AbstractContentHandler() {
						public void body(BodyDescriptor bd, InputStream expected) throws IOException {
						int pos = rootStream.getMark();
						if (expected instanceof RootInputStream) {
							pos++;
						}
						InputStream actual = 
							new EOLConvertingInputStream(
									new RandomAccessFileInputStream(file, pos, file.length()));
	                       
						StringBuilder sb1 = new StringBuilder();
						StringBuilder sb2 = new StringBuilder();
						int b = 0;
						while ((b = expected.read()) != -1) {
							sb1.Append((char) (b & 0xff));
							sb2.Append((char) (actual.read() & 0xff));
						}
						assertEquals(sb1.ToString(), sb2.ToString());
						}
					};
	                
					System.out.println("Testing synch of " + f.getName());
	                
					MimeStreamParser parser = new MimeStreamParser();
					parser.setContentHandler(handler);
					parser.parse(rootStream);
				}
			}
		}*/
	    [Test]
		public void test_BoundaryInEpilogue() {
			StringBuilder sb = new StringBuilder();
			sb.Append("From: foo@bar.com\r\n");
			sb.Append("To: someone@else.com\r\n");
			sb.Append("Content-type: multipart/something; boundary=myboundary\r\n");
			sb.Append("\r\n");
			sb.Append("This is the preamble.\r\n");
			sb.Append("--myboundary\r\n");
			sb.Append("Content-type: text/plain\r\n");
			sb.Append("\r\n");
			sb.Append("This is the first body.\r\n");
			sb.Append("It's completely meaningless.\r\n");
			sb.Append("After this line the body ends.\r\n");
			sb.Append("\r\n");
			sb.Append("--myboundary--\r\n");
	        
			StringBuilder epilogue = new StringBuilder();
			epilogue.Append("Content-type: text/plain\r\n");
			epilogue.Append("\r\n");
			epilogue.Append("This is actually the epilogue but it looks like a second body.\r\n");
			epilogue.Append("Yada yada yada.\r\n");
			epilogue.Append("\r\n");
			epilogue.Append("--myboundary--\r\n");
			epilogue.Append("This is still the epilogue.\r\n");
	        
			sb.Append(epilogue.ToString());
	        
			MemoryStream bais = new MemoryStream( Encoding.ASCII.GetBytes( sb.ToString() ) );
			StringBuilder actual = new StringBuilder();
	        
			MimeStreamParser parser = new MimeStreamParser();
			parser.SetContentHandler( new BoundaryInEpilogue_ContentHandler(actual) );
			parser.Parse(bais);
	        
			Assert.AreEqual(epilogue.ToString(), actual.ToString());
		}
		
		public class BoundaryInEpilogue_ContentHandler : AbstractContentHandler {
			StringBuilder actual;
			
			internal BoundaryInEpilogue_ContentHandler(StringBuilder actual) {
				this.actual = actual;
			}
			
			public override void Epilogue(Stream inputStream) {
				int b;
				while ((b = inputStream.ReadByte()) != -1) {
					actual.Append((char) b);
				}
			}
			
		}
		
	    [Test]
		public void test_ParseOneLineFields() {
			StringBuilder sb = new StringBuilder();
			ArrayList expected = new ArrayList();
			expected.Add("From: foo@abr.com");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n");
			expected.Add("Subject: A subject");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n");
	        
			MimeStreamParser parser = new MimeStreamParser();
			parser.SetContentHandler(new ParseOneLineFields_ContentHandler(expected) );
	        
			parser.Parse(new MemoryStream( Encoding.ASCII.GetBytes(sb.ToString() ) ) );
	        
			Assert.AreEqual(0, expected.Count);
		}
		
		public class ParseOneLineFields_ContentHandler : AbstractContentHandler {
			ArrayList expected;
			
			internal ParseOneLineFields_ContentHandler(ArrayList expected) {
				this.expected = expected;
			}

			public override void Field(String fieldData) {
				Assert.AreEqual((String) expected[0], fieldData);
				expected.RemoveAt(0);
			}			
			
		}
				
	    [Test]
		public void testCRWithoutLFInHeader()  {
			/*
			* Test added because \r:s not followed by \n:s in the header would
			* cause an infinite loop. 
			*/
			StringBuilder sb = new StringBuilder();
			ArrayList expected = new ArrayList();
			
			expected.Add("The-field: This field\r\rcontains CR:s\r\r"
							+ "not\r\n\tfollowed by LF");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n");
	        
			MimeStreamParser parser = new MimeStreamParser();
			parser.SetContentHandler( new ParseOneLineFields_ContentHandler(expected) );
	        
			parser.Parse(new MemoryStream( Encoding.ASCII.GetBytes(sb.ToString() ) ));
	        
			Assert.AreEqual(0, expected.Count);
		}

	    [Test]
		public void test_ParseMultiLineFields() {
			StringBuilder sb = new StringBuilder();
			ArrayList expected = new ArrayList();
			
			expected.Add("Received: by netmbx.netmbx.de (/\\==/\\ Smail3.1.28.1)\r\n"
					+ "\tfrom mail.cs.tu-berlin.de with smtp\r\n"
					+ "\tid &lt;m0uWPrO-0004wpC&gt;;"
							+ " Wed, 19 Jun 96 18:12 MES");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n");
			expected.Add("Subject: A folded subject\r\n Line 2\r\n\tLine 3");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n");
	        
			MimeStreamParser parser = new MimeStreamParser();
			parser.SetContentHandler(new ParseOneLineFields_ContentHandler(expected));
	        
			parser.Parse(new MemoryStream( Encoding.ASCII.GetBytes(sb.ToString() ) ));
	        
			Assert.AreEqual(0, expected.Count);
		}
	    
		public class Stop_ContentHandler : TestHandler {
			MimeStreamParser parser;
			
			public Stop_ContentHandler(MimeStreamParser parser) {
				this.parser = parser;
			}
			
			public override void EndHeader() {
				base.EndHeader();
				parser.Stop();
			}
		}
			
	    [Test]
		public void test_Stop() {
			MimeStreamParser parser = new MimeStreamParser();
			TestHandler handler = new Stop_ContentHandler(parser);
			parser.SetContentHandler( handler );

			String msg = "Subject: Yada yada\r\n"
					+ "From: foo@bar.com\r\n"
					+ "\r\n"
					+ "Line 1\r\n"
					+ "Line 2\r\n";
			String expected = "<message>\r\n"
							+ "<header>\r\n"
							+ "<field>\r\n"
							+ "Subject: Yada yada"
							+ "</field>\r\n"
							+ "<field>\r\n"
							+ "From: foo@bar.com"
							+ "</field>\r\n"
							+ "</header>\r\n"
							+ "<body>\r\n"
							+ "</body>\r\n"
							+ "</message>\r\n";
	        
			parser.Parse(new MemoryStream( Encoding.ASCII.GetBytes(msg) ));
			String result = handler.sb.ToString();
	        
			Assert.AreEqual(expected, result);
		}
	    
		/*
		* Tests that invalid fields are ignored.
		*/
		[Test]
		public void test_InvalidFields() {
			StringBuilder sb = new StringBuilder();
			ArrayList expected = new ArrayList();
			sb.Append("From - foo@abr.com\r\n");
			expected.Add("From: some@one.com");
			sb.Append( expected[expected.Count-1].ToString() + "\r\n");
			expected.Add("Subject: A subject");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n");
			sb.Append("A line which should be ignored\r\n");
	        
			MimeStreamParser parser = new MimeStreamParser();
			parser.SetContentHandler( new ParseOneLineFields_ContentHandler(expected) );
			
			parser.Parse(new MemoryStream( Encoding.ASCII.GetBytes( sb.ToString() ) ) );
	        
			Assert.AreEqual(0, expected.Count);
		}

		public class EmptyStream_ContentHandler : AbstractContentHandler {
			ArrayList expected;
			
			public EmptyStream_ContentHandler(ArrayList expected) {
				this.expected = expected;
			}
				
			public override void Body(BodyDescriptor bd, Stream inputStream) {
				Assert.AreEqual((String) expected[0], "body");
				expected.RemoveAt(0);
			}
	        
			public override void EndMultipart() {
				throw new Exception("endMultipart shouldn't be called for empty stream");
			}

			public override void EndBodyPart() {
				throw new Exception("endBodyPart shouldn't be called for empty stream");
			}

			public override void EndHeader() {
				Assert.AreEqual( (String) expected[0], "endHeader");
				expected.RemoveAt(0);
			}

			public override void EndMessage() {
				Assert.AreEqual( (String) expected[0], "endMessage");
				expected.RemoveAt(0);
			}

			public override void Field(String fieldData) {
				throw new Exception("field shouldn't be called for empty stream");
			}

			public override void StartMultipart(BodyDescriptor bd) {
				throw new Exception("startMultipart shouldn't be called for empty stream");
			}

			public override void StartBodyPart() {
				throw new Exception("startBodyPart shouldn't be called for empty stream");
			}

			public override void StartHeader() {
				Assert.AreEqual((String) expected[0], "startHeader");
				expected.RemoveAt(0);
			}

			public override void StartMessage() {
				Assert.AreEqual((String) expected[0], "startMessage");
				expected.RemoveAt(0);
			}
			
		}

		/*
		* Tests that empty streams still generate the expected series of events.
		*/
		[Test]
		public void test_EmptyStream() {
			ArrayList expected = new ArrayList();
			expected.Add("startMessage");
			expected.Add("startHeader");
			expected.Add("endHeader");
			expected.Add("body");
			expected.Add("endMessage");
	        
			MimeStreamParser parser = new MimeStreamParser();
			parser.SetContentHandler(new EmptyStream_ContentHandler(expected) );
			parser.Parse(new MemoryStream(new byte[0]));
	        
			Assert.AreEqual(0, expected.Count);
		}
	    
		public class EmptyHeader_ContentHandler : AbstractContentHandler {
			StringBuilder body;
			
			internal EmptyHeader_ContentHandler(StringBuilder body) {
				this.body = body;
			}

			public override void Field(String fieldData) {
				throw new Exception("No fields should be reported");
			}
			public override void Body(BodyDescriptor bd, Stream inputStream) {
				int b;
				while ((b = inputStream.ReadByte()) != -1) {
					body.Append((char) b);
				}
			}			
		}	    
	    
		/*
		* Tests parsing of empty headers.
		*/
		[Test]
		public void test_EmptyHeader() {
			StringBuilder sb = new StringBuilder();
			sb.Append("\r\n");
			sb.Append("The body is right here\r\n");
	        
			StringBuilder body = new StringBuilder();
	        
			MimeStreamParser parser = new MimeStreamParser();
			parser.SetContentHandler(new EmptyHeader_ContentHandler(body) );
	        
			parser.Parse(new MemoryStream( Encoding.ASCII.GetBytes( sb.ToString() ) ));
	        
			Assert.AreEqual("The body is right here\r\n", body.ToString());
		}
	    
		public class EmptyBody_ContentHandler : AbstractContentHandler {
			ArrayList expected;
			
			internal EmptyBody_ContentHandler(ArrayList expected) {
				this.expected = expected;
			}

			public override void Field(String fieldData) {
				Assert.AreEqual((String)expected[0], fieldData);
				expected.RemoveAt(0);
			}
			public override void Body(BodyDescriptor bd, Stream inputStream) {
				Assert.AreEqual(-1, inputStream.ReadByte());
			}			
		}	    
	    
		/*
		* Tests parsing of empty body.
		*/
		[Test]
		public void test_EmptyBody() {
			StringBuilder sb = new StringBuilder();
			ArrayList expected = new ArrayList();
			expected.Add("From: some@one.com");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n");
			expected.Add("Subject: A subject");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n\r\n");
	        
			MimeStreamParser parser = new MimeStreamParser();
			parser.SetContentHandler(new EmptyBody_ContentHandler(expected) );
	        
			parser.Parse(new MemoryStream( Encoding.ASCII.GetBytes(sb.ToString() ) ));
	        
			Assert.AreEqual(0, expected.Count);
		}
	    
		/*
		* Tests that invalid fields are ignored.
		*/
		[Test]
		public void test_PrematureEOFAfterFields() {
			StringBuilder sb = new StringBuilder();
			ArrayList expected = new ArrayList();
			expected.Add("From: some@one.com");
			sb.Append( expected[expected.Count-1].ToString() + "\r\n");
			expected.Add("Subject: A subject");
			sb.Append(expected[expected.Count-1].ToString() );
	        
			MimeStreamParser parser = new MimeStreamParser();
			parser.SetContentHandler(new ParseOneLineFields_ContentHandler(expected) );
	        
			parser.Parse(new MemoryStream( Encoding.ASCII.GetBytes( sb.ToString() )) );
			Assert.AreEqual(0, expected.Count);
	        
			sb = new StringBuilder();
			expected.Clear();
			
			expected.Add("From: some@one.com");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n");
			expected.Add("Subject: A subject");
			sb.Append(expected[expected.Count-1].ToString() + "\r\n");
	        
			parser = new MimeStreamParser();
			parser.SetContentHandler(new ParseOneLineFields_ContentHandler(expected));
			parser.Parse(new MemoryStream(Encoding.ASCII.GetBytes( sb.ToString() )) );
	        
			Assert.AreEqual(0, expected.Count);
		}
	    
		[Test]
		public void test_Parse() {
			string[] fileNames = Directory.GetFiles("../../testmsgs", "*.msg");
			
			for (int i = 0; i < fileNames.Length; i++) {
	            
				MimeStreamParser parser = null;
				TestHandler handler = null;
				parser = new MimeStreamParser();
				handler = new TestHandler();
	            
				Console.WriteLine("Parsing " + fileNames[i]);
				parser.SetContentHandler(handler);
				parser.Parse( new FileStream(fileNames[i], FileMode.Open, FileAccess.Read) );
	            
				String result = handler.sb.ToString();
				String xmlFilePath = 
					Path.Combine(
						Path.GetDirectoryName(fileNames[i]),
						Path.GetFileNameWithoutExtension(fileNames[i]) + ".xml" );
				String expected = new StreamReader(xmlFilePath, Encoding.GetEncoding("ISO8859-1") ).ReadToEnd();
	            
				Assert.AreEqual(result, expected, "Error parsing " + fileNames[i]);
			}
		}

		public class TestHandler : IContentHandler {
			public StringBuilder sb = new StringBuilder();

			private String Escape(char c) {
				if (c == '&') {
					return "&amp;";
				}
				if (c == '>') {
					return "&gt;";
				}
				if (c == '<') {
					return "&lt;";
				}
				return "" + c;
			}
	        
			private String Escape(String s) {
				s = s.Replace("&", "&amp;");
				s = s.Replace(">", "&gt;");
				s = s.Replace("<", "&lt;");
				return s;
			}
	        
			public void Epilogue(Stream inputStream) {
				sb.Append("<epilogue>\r\n");
				int b = 0;
				while ((b = inputStream.ReadByte()) != -1) {
					sb.Append( Escape((char) b));
				}
				sb.Append("</epilogue>\r\n");
			}
			
			public void Preamble(Stream inputStream) {
				sb.Append("<preamble>\r\n");
				int b = 0;
				while ((b = inputStream.ReadByte()) != -1) {
					sb.Append( Escape((char) b));
				}
				sb.Append("</preamble>\r\n");
			}
			
			public void StartMultipart(BodyDescriptor bd) {
				sb.Append("<multipart>\r\n");
			}
			public void Body(BodyDescriptor bd, Stream inputStream) {
				sb.Append("<body>\r\n");
				int b = 0;
				while ((b = inputStream.ReadByte()) != -1) {
					sb.Append(Escape((char) b));
				}
				sb.Append("</body>\r\n");
			}
			
			public void EndMultipart() {
				sb.Append("</multipart>\r\n");
			}
			public void StartBodyPart() {
				sb.Append("<body-part>\r\n");
			}
			public void EndBodyPart() {
				sb.Append("</body-part>\r\n");
			}
			public void StartHeader() {
				sb.Append("<header>\r\n");
			}
			public void Field(String fieldData) {
				sb.Append("<field>\r\n" + Escape(fieldData) + "</field>\r\n");
			}
			public virtual void EndHeader() {
				sb.Append("</header>\r\n");
			}
			public void StartMessage() {
				sb.Append("<message>\r\n");
			}
			public void EndMessage() {
				sb.Append("</message>\r\n");
			}

			public void Raw(Stream inputStream) {
				throw new Exception("raw should never be called");
			}
		}
	}


}