/*
 *  Copyright 2004 the mime4j project
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
using System;
using System.IO;
using System.Text;

using NI.Email.Mime;
using NUnit.Framework;

namespace NI.Tests.Email.Mime {

	/**
	* 
	*
	* @author Niklas Therning
	* @version $Id: BodyDescriptorTest.java,v 1.2 2004/10/02 12:41:11 ntherning Exp $
	*/
	[TestFixture]
	public class BodyDescriptorTest {

		[Test]
		public void test_GetParameters() {
			BodyDescriptor bd = null;
	        
			bd = new BodyDescriptor();
			
			bd.AddField("Content-Type ", "text/plain; charset=ISO-8859-1; "
					+ "boundary=foo; param1=value1; param2=value2; param3=value3");
			
			Assert.AreEqual(3, bd.Parameters.Count);
			Assert.AreEqual( (String) bd.Parameters["param1"], "value1" );
			Assert.AreEqual( (String) bd.Parameters["param2"], "value2" );
			Assert.AreEqual( (String) bd.Parameters["param3"], "value3" );
	        
			bd = new BodyDescriptor();
			bd.AddField("Content-Type ", "text/plain; param1=value1; param2=value2;"
						+ " param3=value3");
			
			Assert.AreEqual(3, bd.Parameters.Count);
			Assert.AreEqual( (String) bd.Parameters["param1"], "value1" );
			Assert.AreEqual( (String) bd.Parameters["param2"], "value2" );
			Assert.AreEqual( (String) bd.Parameters["param3"], "value3" );
	        
			bd = new BodyDescriptor();
			bd.AddField("Content-Type ", "text/plain; "
					+ "param1= \" value with\tspaces \" ; "
					+ "param2=\"\\\"value4 with escaped \\\" \\\"\";");
			Assert.AreEqual(bd.Parameters.Count, 2);
			Assert.AreEqual((String) bd.Parameters["param1"], " value with\tspaces ");
			Assert.AreEqual((String) bd.Parameters["param2"], "\"value4 with escaped \" \"");
	        
			/*
			* Make sure escaped characters (except ") are still escaped.
			* The parameter value should be \n\"
			*/
			bd = new BodyDescriptor();
			bd.AddField("Content-Type ", "text/plain; param=\"\\n\\\\\\\"\"");
			Assert.AreEqual(bd.Parameters.Count, 1);
			Assert.AreEqual((String) bd.Parameters["param"], "\\n\\\"");
		}
	    
		[Test]
		public void test_AddField() {
			BodyDescriptor bd = null;
	        
			/*
			* Make sure that only the first Content-Type header added is used.
			*/
			bd = new BodyDescriptor();
			bd.AddField("Content-Type ", "text/plain; charset=ISO-8859-1");
			Assert.AreEqual(bd.MimeType, "text/plain");
			Assert.AreEqual(bd.Charset, "iso-8859-1");
			
			bd.AddField("Content-Type ", "text/html; charset=us-ascii");
			Assert.AreEqual(bd.MimeType, "text/plain");
			Assert.AreEqual(bd.Charset, "iso-8859-1");
		}
	    
		[Test]
		public void test_MimeType() {
			BodyDescriptor bd = null;
	        
			bd = new BodyDescriptor();
			bd.AddField("Content-Type ", "text/PLAIN");
			Assert.AreEqual("text/plain", bd.MimeType);
	        
			bd = new BodyDescriptor();
			bd.AddField("Content-Type ", "text/PLAIN;");
			Assert.AreEqual("text/plain", bd.MimeType);
	        
			bd = new BodyDescriptor();
			bd.AddField("content-type", "   TeXt / html   ");
			Assert.AreEqual("text/html", bd.MimeType);
	        
			bd = new BodyDescriptor();
			bd.AddField("CONTENT-TYPE", "   x-app/yada ;  param = yada");
			Assert.AreEqual("x-app/yada", bd.MimeType);
	        
			bd = new BodyDescriptor();
			bd.AddField("CONTENT-TYPE", "   yada");
			Assert.AreEqual("text/plain", bd.MimeType);
	        
			/*
			* Make sure that only the first Content-Type header added is used.
			*/
			bd = new BodyDescriptor();
			bd.AddField("Content-Type ", "text/plain");
			Assert.AreEqual("text/plain", bd.MimeType);
			bd.AddField("Content-Type ", "text/html");
			Assert.AreEqual("text/plain", bd.MimeType);
	        
			/*
			* Implicit mime types.
			*/
			BodyDescriptor child = null;
			BodyDescriptor parent = null;
	        
			parent = new BodyDescriptor();
			parent.AddField("Content-Type", "mutlipart/alternative; boundary=foo");
	        
			child = new BodyDescriptor(parent);
			Assert.AreEqual("text/plain", child.MimeType);
			child.AddField("Content-Type", " child/type");
			Assert.AreEqual("child/type", child.MimeType);
	        
			parent = new BodyDescriptor();
			parent.AddField("Content-Type", "multipart/digest; boundary=foo");
	        
			child = new BodyDescriptor(parent);
			Assert.AreEqual("message/rfc822", child.MimeType);
			child.AddField("Content-Type", " child/type");
			Assert.AreEqual("child/type", child.MimeType);
	        
		}
	    
		[Test]
		public void test_Parameters() {
			BodyDescriptor bd = null;

			/*
			* Test charset.
			*/
			bd = new BodyDescriptor();
			Assert.AreEqual("us-ascii", bd.Charset);
			bd.AddField("Content-Type ", "some/type; charset=ISO-8859-1");
			Assert.AreEqual("iso-8859-1", bd.Charset);
	        
			bd = new BodyDescriptor();
			Assert.AreEqual("us-ascii", bd.Charset);
			bd.AddField("Content-Type ", "some/type");
			Assert.AreEqual("us-ascii", bd.Charset);
	        
			/*
			* Test boundary.
			*/
			bd = new BodyDescriptor();
			bd.AddField("Content-Type", "text/html; boundary=yada yada");
			Assert.AreEqual(bd.Boundary, null);

			bd = new BodyDescriptor();
			bd.AddField("Content-Type", "multipart/yada; boundary=yada");
			Assert.AreEqual("yada", bd.Boundary);

			/*
			* Test some weird parameters.
			*/
			bd = new BodyDescriptor();
			bd.AddField("Content-Type", "multipart/yada; boundary=yada yada");
			Assert.AreEqual("yada", bd.Boundary);
	        
			bd = new BodyDescriptor();
			bd.AddField("Content-Type", "multipart/yada; boUNdarY= ya:*da; \tcharset\t =  big5");
			Assert.AreEqual("ya:*da", bd.Boundary);
			Assert.AreEqual("big5", bd.Charset);
	        
			bd = new BodyDescriptor();
			bd.AddField("Content-Type", "multipart/yada; boUNdarY= \"ya \\\"\\\"\tda \\\"\"; "
								+ "\tcharset\t =  \"\\\"hepp\\\"  =us\t-ascii\"");
			Assert.AreEqual("ya \"\"\tda \"", bd.Boundary);
			Assert.AreEqual("\"hepp\"  =us\t-ascii", bd.Charset);
	        
		}

	}


}