/*

  Copyright (C) 2004-2005 Smardec. All rights reserved.

  http://www.smardec.com

*/

package com.smardec.jndi.mirror;

import javax.naming.*;
import javax.naming.spi.NamingManager;
import java.beans.XMLEncoder;
import java.io.ByteArrayOutputStream;
import java.io.Serializable;
import java.util.Enumeration;
import java.util.Hashtable;


/**
 * <code>MirrorCtx</code> is a <code>Context</code> implementation. The class can be saved in XML or
 * serialized.
 */
public class MirrorCtx implements Context, Cloneable, Serializable {
    /**
     * <code>NameParser</code> instance.
     */
    private final static NameParser nameParser = new MirrorNameParser();
    /**
     * Environment properties.
     */
    private Hashtable environment = null;
    /**
     * Context's bindings.
     */
    private Hashtable bindings = new Hashtable();
    /**
     * Parent context.
     */
    private MirrorCtx parent = null;
    /**
     * Context name.
     */
    private String name = null;

    /**
     * Creates <code>MirrorCtx</code> object using the specified environment.
     *
     * @param environment Environment for the context
     */
    MirrorCtx(Hashtable environment) {
        if (environment != null)
            this.environment = (Hashtable) (environment.clone());
    }

    /**
     * Creates context with specified parent context, name and environment.
     *
     * @param parent      Parent context
     * @param name        Name of the created context
     * @param environment Environment for the context
     */
    private MirrorCtx(MirrorCtx parent, String name, Hashtable environment) {
        this(environment);
        this.parent = parent;
        this.name = name;
    }

    /**
     * Returns cloned object.
     *
     * @return Cloned object.
     * @throws CloneNotSupportedException If the context cannot be cloned.
     */
    public Object clone() throws CloneNotSupportedException {
        MirrorCtx clone = new MirrorCtx(parent, name, environment);
        clone.bindings = (Hashtable) bindings.clone();
        return clone;
    }

    /**
     * Utility method for processing composite/compound name.
     *
     * @param name The non-<code>null</code> composite or compound name to process
     * @return The non-<code>null</code> string name in this namespace to be processed.
     */
    private Name getParsedName(Name name) throws NamingException {
        if (name instanceof CompositeName) {
            if (name.size() > 1)
                throw new InvalidNameException("Cannot handle all components");
            return nameParser.parse(name.get(0));
        } else
            return name;
    }

    //---Context Implementation Start---//
    /**
     * Adds a new environment property to the environment of this
     * context.  If the property already exists, its value is overwritten.
     *
     * @param propName the name of the environment property to add; may not be <code>null</code>
     * @param propVal  the value of the property to add; may not be <code>null</code>
     * @return the previous value of the property, or <code>null</code> if the property was
     *         not in the environment before
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #getEnvironment()
     * @see #removeFromEnvironment(String)
     */
    public Object addToEnvironment(String propName, Object propVal) throws NamingException {
        if (environment == null)
            environment = new Hashtable();
        return environment.put(propName, propVal);
    }

    /**
     * Binds a name to an object.
     * All intermediate contexts and the target context (that named by all
     * but terminal atomic component of the name) must already exist.
     *
     * @param name the name to bind; may not be empty
     * @param obj  the object to bind; possibly <code>null</code>
     * @throws javax.naming.NameAlreadyBoundException
     *                                      if name is already bound
     * @throws javax.naming.directory.InvalidAttributesException
     *                                      if object did not supply all mandatory attributes
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #bind(String, Object)
     * @see #rebind(javax.naming.Name, Object)
     */
    public void bind(Name name, Object obj) throws NamingException {
        if (name.isEmpty())
            throw new InvalidNameException("Name cannot be empty");
        Name parsedName = getParsedName(name);
        String firstName = parsedName.get(0);
        Object firstBinding = bindings.get(firstName);
        if (parsedName.size() == 1) {
            if (firstBinding != null)
                throw new NameAlreadyBoundException("Name already bound");
            bindings.put(firstName, NamingManager.getStateToBind(obj, new CompositeName(firstName), this, environment));
        } else {
            if (!(firstBinding instanceof Context))
                throw new NameNotFoundException(firstName + " is not a context name");
            ((Context) firstBinding).bind(parsedName.getSuffix(1), obj);
        }
    }

    /**
     * Binds a name to an object.
     * See {@link #bind(javax.naming.Name, Object)} for details.
     *
     * @param name the name to bind; may not be empty
     * @param obj  the object to bind; possibly <code>null</code>
     * @throws javax.naming.NameAlreadyBoundException
     *                                      if name is already bound
     * @throws javax.naming.directory.InvalidAttributesException
     *                                      if object did not supply all mandatory attributes
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public void bind(String name, Object obj) throws NamingException {
        bind(new CompositeName(name), obj);
    }

    /**
     * Closes this context.
     * This method releases this context's resources immediately, instead of
     * waiting for them to be released automatically by the garbage collector.
     * <p/>
     * <p> This method is idempotent:  invoking it on a context that has
     * already been closed has no effect.  Invoking any other method
     * on a closed context is not allowed, and results in undefined behaviour.
     *
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public void close() throws NamingException {
        bindings.clear();
        environment = null;
    }

    /**
     * Composes the name of this context with a name relative to
     * this context.
     * Given a name (<code>name</code>) relative to this context, and
     * the name (<code>prefix</code>) of this context relative to one
     * of its ancestors, this method returns the composition of the
     * two names using the syntax appropriate for the naming
     * system(s) involved.  That is, if <code>name</code> names an
     * object relative to this context, the result is the name of the
     * same object, but relative to the ancestor context.  None of the
     * names may be <code>null</code>.
     * <p/>
     * For example, if this context is named "wiz.com" relative
     * to the initial context, then
     * <pre>
     *  composeName("east", "wiz.com")  </pre>
     * might return <code>"east.wiz.com"</code>.
     * If instead this context is named "org/research", then
     * <pre>
     *  composeName("user/jane", "org/research")    </pre>
     * might return <code>"org/research/user/jane"</code> while
     * <pre>
     *  composeName("user/jane", "research")    </pre>
     * returns <code>"research/user/jane"</code>.
     *
     * @param name   a name relative to this context
     * @param prefix the name of this context relative to one of its ancestors
     * @return the composition of <code>prefix</code> and <code>name</code>
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #composeName(String, String)
     */
    public Name composeName(Name name, Name prefix) throws NamingException {
        Name result = ((Name) (prefix.clone())).addAll(name);
        return new CompositeName(result.toString());
    }

    /**
     * Composes the name of this context with a name relative to
     * this context.
     * See {@link #composeName(javax.naming.Name, javax.naming.Name)} for details.
     *
     * @param name   a name relative to this context
     * @param prefix the name of this context relative to one of its ancestors
     * @return the composition of <code>prefix</code> and <code>name</code>
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public String composeName(String name, String prefix) throws NamingException {
        return composeName(new CompositeName(name), new CompositeName(prefix)).toString();
    }

    /**
     * Creates and binds a new context.
     * Creates a new context with the given name and binds it in
     * the target context (that named by all but terminal atomic
     * component of the name).  All intermediate contexts and the
     * target context must already exist.
     *
     * @param name the name of the context to create; may not be empty
     * @return the newly created context
     * @throws javax.naming.NameAlreadyBoundException
     *                                      if name is already bound
     * @throws javax.naming.directory.InvalidAttributesException
     *                                      if creation of the subcontext requires specification of
     *                                      mandatory attributes
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #createSubcontext(String)
     */
    public Context createSubcontext(Name name) throws NamingException {
        if (name.isEmpty())
            throw new InvalidNameException("Name cannot be empty");
        Name parsedName = getParsedName(name);
        String firstName = parsedName.get(0);
        Object firstBinding = bindings.get(firstName);
        if (parsedName.size() == 1) {
            if (firstBinding != null)
                throw new NameAlreadyBoundException("Name already bound");
            Context subcontext = new MirrorCtx(this, firstName, environment);
            bindings.put(firstName, subcontext);
            return subcontext;
        } else {
            if (!(firstBinding instanceof Context))
                throw new NameNotFoundException(firstName + " is not a context name");
            return ((Context) firstBinding).createSubcontext(parsedName.getSuffix(1));
        }
    }

    /**
     * Creates and binds a new context.
     * See {@link #createSubcontext(javax.naming.Name)} for details.
     *
     * @param name the name of the context to create; may not be empty
     * @return the newly created context
     * @throws javax.naming.NameAlreadyBoundException
     *                                      if name is already bound
     * @throws javax.naming.directory.InvalidAttributesException
     *                                      if creation of the subcontext requires specification of
     *                                      mandatory attributes
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public Context createSubcontext(String name) throws NamingException {
        return createSubcontext(new CompositeName(name));
    }

    /**
     * Destroys the named context and removes it from the namespace.
     * Any attributes associated with the name are also removed.
     * Intermediate contexts are not destroyed.
     * <p/>
     * <p> This method is idempotent.
     * It succeeds even if the terminal atomic name
     * is not bound in the target context, but throws
     * <tt>NameNotFoundException</tt>
     * if any of the intermediate contexts do not exist.
     * <p/>
     * <p> In a federated naming system, a context from one naming system
     * may be bound to a name in another.  One can subsequently
     * look up and perform operations on the foreign context using a
     * composite name.  However, an attempt destroy the context using
     * this composite name will fail with
     * <tt>NotContextException</tt>, because the foreign context is not
     * a "subcontext" of the context in which it is bound.
     * Instead, use <tt>unbind()</tt> to remove the
     * binding of the foreign context.  Destroying the foreign context
     * requires that the <tt>destroySubcontext()</tt> be performed
     * on a context from the foreign context's "native" naming system.
     *
     * @param name the name of the context to be destroyed; may not be empty
     * @throws javax.naming.NameNotFoundException
     *                                      if an intermediate context does not exist
     * @throws javax.naming.NotContextException
     *                                      if the name is bound but does not name a
     *                                      context, or does not name a context of the appropriate type
     * @throws javax.naming.ContextNotEmptyException
     *                                      if the named context is not empty
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #destroySubcontext(String)
     */
    public void destroySubcontext(Name name) throws NamingException {
        if (name.isEmpty())
            throw new InvalidNameException("Name cannot be empty");
        Name parsedName = getParsedName(name);
        String firstName = parsedName.get(0);
        Object firstBinding = bindings.get(firstName);
        if (parsedName.size() == 1) {
            if (firstBinding == null)
                return;
            if (!(firstBinding instanceof Context))
                throw new NotContextException(firstName + " is not a context name");
            if (((Context) firstBinding).list("").hasMore())
                throw new ContextNotEmptyException(firstName + " should be empty");
            bindings.remove(firstName);
        } else {
            if (!(firstBinding instanceof Context))
                throw new NameNotFoundException(firstName + " is not a context name");
            ((Context) firstBinding).destroySubcontext(parsedName.getSuffix(1));
        }
    }

    /**
     * Destroys the named context and removes it from the namespace.
     * See {@link #destroySubcontext(javax.naming.Name)} for details.
     *
     * @param name the name of the context to be destroyed; may not be empty
     * @throws javax.naming.NameNotFoundException
     *                                      if an intermediate context does not exist
     * @throws javax.naming.NotContextException
     *                                      if the name is bound but does not name a
     *                                      context, or does not name a context of the appropriate type
     * @throws javax.naming.ContextNotEmptyException
     *                                      if the named context is not empty
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public void destroySubcontext(String name) throws NamingException {
        destroySubcontext(new CompositeName(name));
    }

    /**
     * Retrieves the environment in effect for this context.
     * See class description for more details on environment properties.
     * <p/>
     * <p> The caller should not make any changes to the object returned:
     * their effect on the context is undefined.
     * The environment of this context may be changed using
     * <tt>addToEnvironment()</tt> and <tt>removeFromEnvironment()</tt>.
     *
     * @return the environment of this context; never <code>null</code>
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #addToEnvironment(String, Object)
     * @see #removeFromEnvironment(String)
     */
    public Hashtable getEnvironment() throws NamingException {
        if (environment == null)
            return new Hashtable();
        else
            return (Hashtable) environment.clone();
    }

    /**
     * Retrieves the full name of this context within its own namespace.
     * <p/>
     * <p> Many naming services have a notion of a "full name" for objects
     * in their respective namespaces.  For example, an LDAP entry has
     * a distinguished name, and a DNS record has a fully qualified name.
     * This method allows the client application to retrieve this name.
     * The string returned by this method is not a JNDI composite name
     * and should not be passed directly to context methods.
     * In naming systems for which the notion of full name does not
     * make sense, <tt>OperationNotSupportedException</tt> is thrown.
     *
     * @return this context's name in its own namespace; never <code>null</code>
     * @throws javax.naming.OperationNotSupportedException
     *                                      if the naming system does
     *                                      not have the notion of a full name
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @since 1.3
     */
    public String getNameInNamespace() throws NamingException {
        if (parent == null)
            return "";
        Name fullName = nameParser.parse("");
        MirrorCtx parentIter = this;
        while (parentIter != null && parentIter.name != null) {
            fullName.add(0, parentIter.name);
            parentIter = parentIter.parent;
        }
        return fullName.toString();
    }

    /**
     * Retrieves the parser associated with the named context.
     * In a federation of namespaces, different naming systems will
     * parse names differently.  This method allows an application
     * to get a parser for parsing names into their atomic components
     * using the naming convention of a particular naming system.
     * Within any single naming system, <tt>NameParser</tt> objects
     * returned by this method must be equal (using the <tt>equals()</tt>
     * test).
     *
     * @param name the name of the context from which to get the parser
     * @return a name parser that can parse compound names into their atomic
     *         components
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #getNameParser(String)
     * @see javax.naming.CompoundName
     */
    public NameParser getNameParser(Name name) throws NamingException {
        return nameParser;
    }

    /**
     * Retrieves the parser associated with the named context.
     * See {@link #getNameParser(javax.naming.Name)} for details.
     *
     * @param name the name of the context from which to get the parser
     * @return a name parser that can parse compound names into their atomic
     *         components
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public NameParser getNameParser(String name) throws NamingException {
        return getNameParser(new CompositeName(name));
    }

    /**
     * Enumerates the names bound in the named context, along with the
     * class names of objects bound to them.
     * The contents of any subcontexts are not included.
     * <p/>
     * <p> If a binding is added to or removed from this context,
     * its effect on an enumeration previously returned is undefined.
     *
     * @param name the name of the context to list
     * @return an enumeration of the names and class names of the
     *         bindings in this context.  Each element of the
     *         enumeration is of type <tt>NameClassPair</tt>.
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #list(String)
     * @see #listBindings(javax.naming.Name)
     */
    public NamingEnumeration list(Name name) throws NamingException {
        if (name.isEmpty())
            return new NamesEnumeration(bindings.keys());
        Object binding = lookup(name);
        if (binding instanceof Context)
            return ((Context) binding).list("");
        throw new NotContextException(name + " is not a context and cannot be listed");
    }

    /**
     * Enumerates the names bound in the named context, along with the
     * class names of objects bound to them.
     * See {@link #list(javax.naming.Name)} for details.
     *
     * @param name the name of the context to list
     * @return an enumeration of the names and class names of the
     *         bindings in this context.  Each element of the
     *         enumeration is of type <tt>NameClassPair</tt>.
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public NamingEnumeration list(String name) throws NamingException {
        return list(new CompositeName(name));
    }

    /**
     * Enumerates the names bound in the named context, along with the
     * objects bound to them.
     * The contents of any subcontexts are not included.
     * <p/>
     * <p> If a binding is added to or removed from this context,
     * its effect on an enumeration previously returned is undefined.
     *
     * @param name the name of the context to list
     * @return an enumeration of the bindings in this context.
     *         Each element of the enumeration is of type
     *         <tt>Binding</tt>.
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #listBindings(String)
     * @see #list(javax.naming.Name)
     */
    public NamingEnumeration listBindings(Name name) throws NamingException {
        if (name.isEmpty())
            return new BindingsEnumeration(bindings.keys());
        Object binding = lookup(name);
        if (binding instanceof Context)
            return ((Context) binding).listBindings("");
        throw new NotContextException(name + " is not a context and cannot be listed");
    }

    /**
     * Enumerates the names bound in the named context, along with the
     * objects bound to them.
     * See {@link #listBindings(javax.naming.Name)} for details.
     *
     * @param name the name of the context to list
     * @return an enumeration of the bindings in this context.
     *         Each element of the enumeration is of type
     *         <tt>Binding</tt>.
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public NamingEnumeration listBindings(String name) throws NamingException {
        return listBindings(new CompositeName(name));
    }

    /**
     * Retrieves the named object.
     * If <tt>name</tt> is empty, returns a new instance of this context
     * (which represents the same naming context as this context, but its
     * environment may be modified independently and it may be accessed
     * concurrently).
     *
     * @param name the name of the object to look up
     * @return the object bound to <tt>name</tt>
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #lookup(String)
     * @see #lookupLink(javax.naming.Name)
     */
    public Object lookup(Name name) throws NamingException {
        if (name.isEmpty()) {
            try {
                return clone();
            } catch (CloneNotSupportedException e) {
                e.printStackTrace();
                return null;
            }
        }
        Name parsedName = getParsedName(name);
        String firstName = parsedName.get(0);
        Object firstBinding = bindings.get(firstName);
        if (parsedName.size() == 1) {
            if (firstBinding == null)
                throw new NameNotFoundException(name + " not found");
            try {
                return NamingManager.getObjectInstance(firstBinding, new CompositeName(firstName), this, environment);
            } catch (Exception e) {
                NamingException namingException = new NamingException("NamingManager.getObjectInstance returned error");
                namingException.setRootCause(e);
                throw namingException;
            }
        } else {
            if (!(firstBinding instanceof Context))
                throw new NameNotFoundException(firstName + " is not a context name");
            return ((Context) firstBinding).lookup(parsedName.getSuffix(1));
        }
    }

    /**
     * Retrieves the named object.
     * See {@link #lookup(javax.naming.Name)} for details.
     *
     * @param name the name of the object to look up
     * @return the object bound to <tt>name</tt>
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public Object lookup(String name) throws NamingException {
        return lookup(new CompositeName(name));
    }

    /**
     * Retrieves the named object, following links except
     * for the terminal atomic component of the name.
     * If the object bound to <tt>name</tt> is not a link,
     * returns the object itself.
     *
     * @param name the name of the object to look up
     * @return the object bound to <tt>name</tt>, not following the
     *         terminal link (if any).
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #lookupLink(String)
     */
    public Object lookupLink(Name name) throws NamingException {
        return lookup(name);
    }

    /**
     * Retrieves the named object, following links except
     * for the terminal atomic component of the name.
     * See {@link #lookupLink(javax.naming.Name)} for details.
     *
     * @param name the name of the object to look up
     * @return the object bound to <tt>name</tt>, not following the
     *         terminal link (if any)
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public Object lookupLink(String name) throws NamingException {
        return lookupLink(new CompositeName(name));
    }

    /**
     * Binds a name to an object, overwriting any existing binding.
     * All intermediate contexts and the target context (that named by all
     * but terminal atomic component of the name) must already exist.
     *
     * @param name the name to bind; may not be empty
     * @param obj  the object to bind; possibly <code>null</code>
     * @throws javax.naming.directory.InvalidAttributesException
     *                                      if object did not supply all mandatory attributes
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #rebind(String, Object)
     * @see #bind(javax.naming.Name, Object)
     */
    public void rebind(Name name, Object obj) throws NamingException {
        if (name.isEmpty())
            throw new InvalidNameException("Name cannot be empty");
        Name parsedName = getParsedName(name);
        String firstName = parsedName.get(0);
        if (parsedName.size() == 1) {
            bindings.put(firstName, NamingManager.getStateToBind(obj, new CompositeName(firstName), this, environment));
        } else {
            Object firstBinding = bindings.get(firstName);
            if (!(firstBinding instanceof Context))
                throw new NameNotFoundException(firstName + " is not a context name");
            ((Context) firstBinding).rebind(parsedName.getSuffix(1), obj);
        }
    }

    /**
     * Binds a name to an object, overwriting any existing binding.
     * See {@link #rebind(javax.naming.Name, Object)} for details.
     *
     * @param name the name to bind; may not be empty
     * @param obj  the object to bind; possibly <code>null</code>
     * @throws javax.naming.directory.InvalidAttributesException
     *                                      if object did not supply all mandatory attributes
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public void rebind(String name, Object obj) throws NamingException {
        rebind(new CompositeName(name), obj);
    }

    /**
     * Removes an environment property from the environment of this
     * context.  See class description for more details on environment
     * properties.
     *
     * @param propName the name of the environment property to remove; may not be <code>null</code>
     * @return the previous value of the property, or <code>null</code> if the property was
     *         not in the environment
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #getEnvironment()
     * @see #addToEnvironment(String, Object)
     */
    public Object removeFromEnvironment(String propName) throws NamingException {
        if (environment == null)
            return null;
        return environment.remove(propName);
    }

    /**
     * Binds a new name to the object bound to an old name, and unbinds
     * the old name.  Both names are relative to this context.
     * Any attributes associated with the old name become associated
     * with the new name.
     * Intermediate contexts of the old name are not changed.
     *
     * @param oldName the name of the existing binding; may not be empty
     * @param newName the name of the new binding; may not be empty
     * @throws javax.naming.NameAlreadyBoundException
     *                                      if <tt>newName</tt> is already bound
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #rename(String, String)
     * @see #bind(javax.naming.Name, Object)
     * @see #rebind(javax.naming.Name, Object)
     */
    public void rename(Name oldName, Name newName) throws NamingException {
        if (oldName.isEmpty() || newName.isEmpty())
            throw new InvalidNameException("Name cannot be empty");
        Name oldParsedName = getParsedName(oldName);
        Name newParsedName = getParsedName(newName);
        if (oldParsedName.size() != newParsedName.size())
            throw new NamingException("Names should belong to the same context");
        String oldFirstName = oldParsedName.get(0);
        String newFirstName = newParsedName.get(0);
        if (oldParsedName.size() == 1) {
            if (bindings.get(newFirstName) != null)
                throw new NameAlreadyBoundException("New name already bound");
            Object oldBinding = bindings.remove(oldFirstName);
            if (oldBinding == null)
                throw new NameNotFoundException("Old name is not bound");
            bindings.put(newFirstName, oldBinding);
        } else {
            if (!oldFirstName.equals(newFirstName))
                throw new NamingException("Names should belong to the same context");
            Object oldFirstBinding = bindings.get(oldFirstName);
            if (!(oldFirstBinding instanceof Context))
                throw new NameNotFoundException(oldFirstName + " is not a context name");
            ((Context) oldFirstBinding).rename(oldParsedName.getSuffix(1), newParsedName.getSuffix(1));
        }
    }

    /**
     * Binds a new name to the object bound to an old name, and unbinds
     * the old name.
     * See {@link #rename(javax.naming.Name, javax.naming.Name)} for details.
     *
     * @param oldName the name of the existing binding; may not be empty
     * @param newName the name of the new binding; may not be empty
     * @throws javax.naming.NameAlreadyBoundException
     *                                      if <tt>newName</tt> is already bound
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public void rename(String oldName, String newName) throws NamingException {
        rename(new CompositeName(oldName), new CompositeName(newName));
    }

    /**
     * Unbinds the named object.
     * Removes the terminal atomic name in <code>name</code>
     * from the target context--that named by all but the terminal
     * atomic part of <code>name</code>.
     * <p/>
     * <p> This method is idempotent.
     * It succeeds even if the terminal atomic name
     * is not bound in the target context, but throws
     * <tt>NameNotFoundException</tt>
     * if any of the intermediate contexts do not exist.
     * <p/>
     * <p> Any attributes associated with the name are removed.
     * Intermediate contexts are not changed.
     *
     * @param name the name to unbind; may not be empty
     * @throws javax.naming.NameNotFoundException
     *                                      if an intermediate context does not exist
     * @throws javax.naming.NamingException if a naming exception is encountered
     * @see #unbind(String)
     */
    public void unbind(Name name) throws NamingException {
        if (name.isEmpty())
            throw new InvalidNameException("Name cannot be empty");
        Name parsedName = getParsedName(name);
        String firstName = parsedName.get(0);
        if (parsedName.size() == 1) {
            bindings.remove(firstName);
        } else {
            Object firstBinding = bindings.get(firstName);
            if (!(firstBinding instanceof Context))
                throw new NameNotFoundException(firstName + " is not a context name");
            ((Context) firstBinding).unbind(parsedName.getSuffix(1));
        }
    }

    /**
     * Unbinds the named object.
     * See {@link #unbind(javax.naming.Name)} for details.
     *
     * @param name the name to unbind; may not be empty
     * @throws javax.naming.NameNotFoundException
     *                                      if an intermediate context does not exist
     * @throws javax.naming.NamingException if a naming exception is encountered
     */
    public void unbind(String name) throws NamingException {
        unbind(new CompositeName(name));
    }
    //---Context Implementation End---//

    // Methods required for XML serialization.
    /**
     * Creates <code>MirrorCtx</code> object. Required for XML serialization.
     */
    public MirrorCtx() {}

    /**
     * Returns context bindings. Required for XML serialization.
     *
     * @return Context bindings.
     */
    public Hashtable getBindings() {
        return bindings;
    }

    /**
     * Sets context bindings. Required for XML serialization.
     *
     * @param bindings Context bindings
     */
    public void setBindings(Hashtable bindings) {
        this.bindings = bindings;
    }

    /**
     * Returns context name. Required for XML serialization.
     *
     * @return Context name.
     */
    public String getName() {
        return name;
    }

    /**
     * Sets context name. Required for XML serialization.
     *
     * @param name Context name
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Returns context parent. Required for XML serialization.
     *
     * @return Context parent.
     */
    public MirrorCtx getParent() {
        return parent;
    }

    /**
     * Sets context parent. Required for XML serialization.
     *
     * @param parent Context parent
     */
    public void setParent(MirrorCtx parent) {
        this.parent = parent;
    }

    /**
     * Writes itself as XML using <code>XMLEncoder</code>. Works on JDK1.4 and higher.
     *
     * @return resulting XML.
     */
    public String toString() {
        ByteArrayOutputStream output = new ByteArrayOutputStream(4096);
        XMLEncoder xmlEncoder = new XMLEncoder(output);
        xmlEncoder.writeObject(this);
        xmlEncoder.close();
        return new String(output.toByteArray());
    }

    /**
     * Class for enumerating name/class pairs.
     */
    class NamesEnumeration extends AbstractNamingEnumeration {
        /**
         * Creates <code>NamesEnumeration</code> object.
         *
         * @param enumeration original <code>Enumeration</code>
         */
        public NamesEnumeration(Enumeration enumeration) {
            this.originalEnumeration = enumeration;
        }

        /**
         * Retrieves the next <code>NameClassPair</code> element in the enumeration.
         *
         * @throws NamingException If a naming exception is encountered while attempting
         *                         to retrieve the next element. See <code>NamingException</code>
         *                         and its subclasses for the possible naming exceptions.
         * @throws java.util.NoSuchElementException
         *                         If attempting to get the next element when none is available.
         */
        public Object next() throws NamingException {
            String name = (String) this.originalEnumeration.nextElement();
            String className = MirrorCtx.this.bindings.get(name).getClass().getName();
            return new NameClassPair(name, className);
        }
    }

    /**
     * Class for enumerating bindings.
     */
    class BindingsEnumeration extends AbstractNamingEnumeration {
        /**
         * Creates <code>NamesEnumeration</code> object.
         *
         * @param enumeration original <code>Enumeration</code>
         */
        public BindingsEnumeration(Enumeration enumeration) {
            this.originalEnumeration = enumeration;
        }

        /**
         * Retrieves the next <code>Binding</code> element in the enumeration.
         *
         * @throws NamingException If a naming exception is encountered while attempting
         *                         to retrieve the next element. See <code>NamingException</code>
         *                         and its subclasses for the possible naming exceptions.
         * @throws java.util.NoSuchElementException
         *                         If attempting to get the next element when none is available.
         */
        public Object next() throws NamingException {
            String name = (String) this.originalEnumeration.nextElement();
            Object binding = MirrorCtx.this.bindings.get(name);
            try {
                binding = NamingManager.getObjectInstance(binding, new CompositeName(name), MirrorCtx.this, MirrorCtx.this.environment);
            } catch (Exception e) {
                NamingException namingException = new NamingException("getObjectInstance failed");
                namingException.setRootCause(e);
                throw namingException;
            }
            return new Binding(name, binding);
        }
    }
}
