/*

  Copyright (C) 2004-2005 Smardec. All rights reserved.

  http://www.smardec.com

*/

package com.smardec.jndi.mirror;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import java.util.Enumeration;
import java.util.NoSuchElementException;


/**
 * <code>AbstractNamingEnumeration</code> is basic implementation of <code>NamingEnumeration</code>.
 */
public abstract class AbstractNamingEnumeration implements NamingEnumeration {
    /**
     * Original <code>Enumeration</code>
     */
    protected Enumeration originalEnumeration;

    /**
     * Creates <code>AbstractNamingEnumeration</code> object.
     */
    protected AbstractNamingEnumeration() {}

    /**
     * Closes enumeration. The method does nothing.
     */
    public void close() {}

    /**
     * Determines whether there are any more elements in the enumeration.
     * This method allows naming exceptions encountered while
     * determining whether there are more elements to be caught and handled
     * by the application.
     *
     * @return <code>true</code> If there is more in the enumeration; <code>false</code> otherwise.
     * @throws NamingException If a naming exception is encountered while attempting
     *                         to determine whether there is another element
     *                         in the enumeration. See <code>NamingException</code>
     *                         and its subclasses for the possible naming exceptions.
     */
    public boolean hasMore() throws NamingException {
        return originalEnumeration.hasMoreElements();
    }

    /**
     * Tests if this enumeration contains more elements.
     *
     * @return <code>true</code> if and only if this enumeration object
     *         contains at least one more element to provide;
     *         <code>false</code> otherwise.
     */
    public boolean hasMoreElements() {
        try {
            return hasMore();
        } catch (NamingException e) {
            return false;
        }
    }

    /**
     * Retrieves the next element in the enumeration.
     * This method allows naming exceptions encountered while
     * retrieving the next element to be caught and handled
     * by the application.
     * <p/>
     * Note that <tt>next()</tt> can also throw the runtime exception
     * NoSuchElementException to indicate that the caller is
     * attempting to enumerate beyond the end of the enumeration.
     * This is different from a NamingException, which indicates
     * that there was a problem in obtaining the next element,
     * for example, due to a referral or server unavailability, etc.
     *
     * @return The possibly <code>null</code> element in the enumeration.
     *         <code>null</code> is only valid for enumerations that can return
     *         <code>null</code>.
     * @throws NamingException If a naming exception is encountered while attempting
     *                         to retrieve the next element. See <code>NamingException</code>
     *                         and its subclasses for the possible naming exceptions.
     * @throws java.util.NoSuchElementException
     *                         If attempting to get the next element when none is available.
     */
    public abstract Object next() throws NamingException;

    /**
     * Returns the next element of this enumeration if this enumeration
     * object has at least one more element to provide.
     *
     * @return The next element of this enumeration.
     * @throws NoSuchElementException If no more elements exist.
     */
    public Object nextElement() {
        try {
            return next();
        } catch (NamingException e) {
            throw new NoSuchElementException(e.getMessage());
        }
    }
}
