/*

  Copyright (C) 2004-2005 Smardec. All rights reserved.

  http://www.smardec.com

*/

package com.smardec.jndi.mirror.samples;

import com.smardec.jndi.mirror.CopyHelper;

import javax.naming.Context;
import javax.naming.InitialContext;
import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
import java.util.Hashtable;


/**
 * <code>CopySaveLoadSample</code> class shows how to make a full copy of the original context and serialize it
 * for further usage.
 */
public class CopySaveLoadSample {
    /**
     * Creates copy of the original context and saves it to the file.
     * If <code>filename</code> ends with '.xml' than the context will be XML serialized,
     * otherwise the context will be serialized in binary form.
     *
     * @param filename File name to save the context
     */
    public static void createCopyAndSave(String filename) {
        try {
            // create destination context
            Hashtable environment = new Hashtable();
            environment.put(Context.INITIAL_CONTEXT_FACTORY, "com.smardec.jndi.mirror.MirrorInitialCtxFactory");
            Context mirrorContext = new InitialContext(environment);

            // create source (original) context
            Hashtable anotherEnvironment = new Hashtable();
            anotherEnvironment.put(Context.INITIAL_CONTEXT_FACTORY, "InitialContextFactory class name");
            anotherEnvironment.put("property", "value");
            Context sourceContext = new InitialContext(anotherEnvironment);

            // make copy
            CopyHelper.copy(sourceContext, mirrorContext);

            // mirrorContext is an instance of InitialContext
            // (it was created as 'new InitialContext(environment)').
            // We need MirrorCtx instance for saving.
            // The operation below makes mirrorContext an instance of MirrorCtx.
            mirrorContext = (Context) mirrorContext.lookup("");

            if (filename.endsWith(".xml")) {
                // XML serialized form, JDK1.4 or higher is required
                FileOutputStream fileOutputStream = new FileOutputStream(filename);
                fileOutputStream.write(mirrorContext.toString().getBytes());
                fileOutputStream.close();
            } else {
                // binary serialized form, works on any JDK version
                ObjectOutputStream objectOutputStream = new ObjectOutputStream(new FileOutputStream(filename));
                objectOutputStream.writeObject(mirrorContext);
                objectOutputStream.close();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Loading previously saved context.
     *
     * @param filename File name with previously saved context
     * @return Loaded context.
     */
    public static Context loadContextFromFile(String filename) {
        try {
            Hashtable environment = new Hashtable();
            environment.put(Context.INITIAL_CONTEXT_FACTORY, "com.smardec.jndi.mirror.MirrorInitialCtxFactory");
            environment.put(Context.PROVIDER_URL, filename);
            return new InitialContext(environment);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }
}
