'
'
' ####################  Max Reason
' #####  PROLOG  #####  copyright 1988-2000
' ####################  XBLite sockets/network/internet function library
'
' subject to LGPL license - see COPYING_LIB
'
' v0.0026 - 2000/01/24 - Vincent Voois
' v0.0031 - 2008/04/11 - DTS
'  
'
'
PROGRAM	"xin"
VERSION	"0.0031"
CONSOLE
'
IMPORT	"xst"
IMPORT  "xsx"
IMPORT	"wsock32"
'
'
' This function library operates on a pseudo-non-blocking basis.
' Functions do not "hang" or "lock-up" indefinitely until complete.
' Many functions do have a "block" argument that takes an interval
' in microseconds to block, but values above 20000 aka 20ms are
' converted to 20000 before the select() function is called.
' Blocking times greater than 20 milliseconds thus never occur.
' To get the effect of blocking, programs must create a short
' loop that calls the function repeatedly until it succeeds.
' Short blocking times are supported so that programs do not
' waste cycles polling for results - the operating system will
' context switch to other tasks that have work to do.
'
' Note that network "servers" Open/Bind/Listen/Accept to allow
' network "clients" to connect to them, while network clients
' Open/Bind/Connect to a specific network server.  If a server
' wants to establish a "peer" connection to another server, it
' sets itself up as a server (so the peer can connect to it),
' then also connects to the "peer" server as a client would.
'
'
EXPORT
TYPE XSOCKET
	XLONG     .socket						' socket number of this socket
	XLONG     .remote						' remote socket connected to this socket
	XLONG     .syssocket				' socket number to operating system
	XLONG     .sysremote				' remote socket number to operating system
	GIANT     .address					' IP address this socket is bound to
	XLONG     .port							' port this socket is connected to
	XLONG     .hostnumber				' host identifier for XinGetHostData()
	XLONG     .status						' socket status - see $$STATUS constants
	XLONG     .socketType				' type of socket - $$SOCK_STREAM
	XLONG     .addressType			' address type - $$AF_INET
	XLONG     .addressBytes			' bytes in address - 4 for short IP
	XLONG     .addressFamily		' address family - $$AF_INET
	XLONG     .protocolFamily		' protocol family - $$PF_INET
	USHORT    .protocol					' active protocol - $$TCP
	USHORT    .service					' service - $$FTP
END TYPE
'
TYPE HOST
	STRING*32 .name							' "mhpcc.edu"
	STRING*32 .alias[2]					' "mhpcc.net", "mhpcc.org", "mhpcc.com"
	STRING*16 .system						' "Windows", "WindowsNT", "UNIX", "Linux"
	GIANT     .address					' 0x0104215C - can hold longer IP address
	GIANT     .addresses[7]			' host can support 8 more IP addresses
	XLONG     .hostnumber				' native host number
	XLONG     .addressBytes			' 4 for original IP addresses
	XLONG     .addressFamily		' $$AF_INET
	XLONG     .protocolFamily		' $$PF_INET
	XLONG     .protocol					' protocol # for "TCP"
	XLONG     .resv							'
	XLONG     .resw							'
	XLONG     .resx							'
	XLONG     .resy							'
	XLONG     .resz							'
END TYPE
'
DECLARE FUNCTION  Xin                      ( )
DECLARE FUNCTION  XinCleanup        			 ( )
DECLARE FUNCTION  XinInitialize            (@base, @hosts, @version, @sockets, @comments$, @notes$)
DECLARE FUNCTION  XinAddressNumberToString (addr$$, @addr$)
DECLARE FUNCTION  XinAddressStringToNumber (addr$, @addr$$)
DECLARE FUNCTION  XinHostNameToInfo        (host$, HOST @info)
DECLARE FUNCTION  XinHostNumberToInfo      (hostnum, HOST @info)
DECLARE FUNCTION  XinHostAddressToInfo     (hostaddr, HOST @info)
DECLARE FUNCTION  XinSetSocketOption       (socket, level, optname, optval, optlen)
DECLARE FUNCTION  XinSocketOpen            (@socket, addressFamily, socketType, flags)
DECLARE FUNCTION  XinSocketBind            (socket, block, address$$, port)
DECLARE FUNCTION  XinSocketListen          (socket, block, flags)
DECLARE FUNCTION  XinSocketAccept          (socket, block, @remote, flags)
DECLARE FUNCTION  XinSocketConnectRequest  (socket, block, address$$, port)
DECLARE FUNCTION  XinSocketConnectStatus   (socket, block, @connected)
DECLARE FUNCTION  XinSocketGetAddress      (socket, @port, @address$$, @remote, @port, @raddress$$)
DECLARE FUNCTION  XinSocketGetStatus       (socket, @remote, @syssocket, @syserror, @status, @socketType, @readbytes, @writebytes)
DECLARE FUNCTION  XinSocketRead            (socket, block, address, maxbytes, flags, @bytes)
DECLARE FUNCTION  XinSocketWrite           (socket, block, address, maxbytes, flags, @bytes)
DECLARE FUNCTION  XinSocketClose           (socket)
DECLARE FUNCTION  XinSetDebug              (state)
END EXPORT
'
' support functions  :::  should addr in AddHost() be addr$$ = GIANT ?????
'
INTERNAL FUNCTION  GetLastError            ( )
INTERNAL FUNCTION  AddHost                 (@host, addr, host$)
INTERNAL FUNCTION  SetSocketBlocking       (socket)
INTERNAL FUNCTION  SetSocketNonBlocking    (socket)
INTERNAL FUNCTION  SystemErrorToError      (errno, @error)
'
' emulate C macros for file descriptor bitmaps
'
INTERNAL FUNCTION  FDCLR                   (fildes, FD_SET @fd_set)
INTERNAL FUNCTION  FDSET                   (fildes, FD_SET @fd_set)
INTERNAL FUNCTION  FDISSET                 (fildes, FD_SET @fd_set)
INTERNAL FUNCTION  FDZERO                  (FD_SET @fd_set)
INTERNAL FUNCTION  FDCOUNT                 (FD_SET @fd_set)
'
'
' *****  constants  *****
'
EXPORT
	$$NETWORKVERSION         = 0x0202
'	$$NETWORKVERSION         = 0x0201
'	$$NETWORKVERSION         = 0x0200
'	$$NETWORKVERSION         = 0x0101
'
' SOCKET.status bit definitions
'
	$$SocketStatusOpenSuccess        = 0x00000001		' XinSocketOpen()
	$$SocketStatusBindSuccess        = 0x00000002		' XinSocketBind()
	$$SocketStatusListenSuccess      = 0x00000004		' XinSocketListen()
	$$SocketStatusAcceptSuccess      = 0x00000008		' XinSocketAccept()
	$$SocketStatusConnectRequest     = 0x00000010		' XinSocketConnect()
	$$SocketStatusConnectSuccess     = 0x00000020		' XinSocketConnecting()
	$$SocketStatusConnected          = 0x00000040		' clear on disconnect detect
	$$SocketStatusRemote             = 0x00000080		' remote socket accepted by XinSocketAccept()
'
	$$SocketStatusWaitingReadBuffer  = 0x00000100		' XinSocketRead()
	$$SocketStatusWaitingWriteBuffer = 0x00000200		' XinSocketWrite()
	$$SocketStatusUndefined1         = 0x00000400
	$$SocketStatusUndefined2         = 0x00000800
	$$SocketStatusUndefined3         = 0x00001000
	$$SocketStatusUndefined4         = 0x00002000
	$$SocketStatusUndefined5         = 0x00004000
	$$SocketStatusFailed             = 0x00008000		' network/socket failure
'
' flags argument in XinSocketRead()
'
	$$SocketReadPeekData             = 0x00000002		' leave data in socket
END EXPORT
'
'
' ####################
' #####  Xin ()  #####
' ####################
'
'	/*
'	[Xin]
' Description = The Xin function initializes the Xin library by calling XinInitialize(). Call Xin() before calling any other Xin functions.
' Function    = error = Xin ()
' ArgCount    = 0
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = 
'	See Also    = 
'	Examples    = 
'	*/
FUNCTION  Xin ()
	SHARED  initialized
	SOCKADDR_IN  sockaddrin
	SOCKADDR  sockaddr
	HOST  host

	IF initialized THEN RETURN
	initialized = $$TRUE

	error = XinInitialize (@local, @hosts, @version, @sockets, @comments$, @notes$)
	IF error THEN initialized = $$FALSE
	
	IF LIBRARY(0) THEN RETURN (error)
	
'	RETURN (error)
'
' print information to assist debug process
'
	#debug = $$TRUE
'
' *****  begin network tests  *****
'
	PRINT
	PRINT "###############################################"
	PRINT "#####  XBasic xin function library tests  #####"
	PRINT "###############################################"
	PRINT
	PRINT "#####  XinInitialize (@local, @hosts, @sockets, @version, @comments$, @notes$)  #####"
'
	XinInitialize (@local, @hosts, @version, @sockets, @comments$, @notes$)
'
' *****  print basic network information  *****
'
	PRINT
	PRINT "local                  = "; HEX$ (local,8)
	PRINT "hosts                  = "; HEX$ (hosts,8)
	PRINT "sockets                = "; HEX$ (sockets,8)
	PRINT "version                = "; HEX$ (version,8)
	PRINT "comments$              = "; comments$
	PRINT "notes$                 = "; notes$
'
'
' *****  print local host information  *****
'
'
	PRINT
	PRINT "#####  XinHostNumberToInfo (base, @info)  #####"
'
	XinHostNumberToInfo (0, @host)
	hostaddress = host.address
	address$$ = host.address
	host$ = host.name
'
	PRINT
	PRINT "host.name              = \""; host.name; "\""
	PRINT "host.alias[0]          = \""; host.alias[0]; "\""
	PRINT "host.alias[1]          = \""; host.alias[1]; "\""
	PRINT "host.alias[2]          = \""; host.alias[2]; "\""
	PRINT "host.system            = \""; host.system; "\""
	PRINT "host.hostnumber        = "; HEX$ (host.hostnumber,8)
	PRINT "host.address           = "; HEX$ (host.address,8); " = "; STRING$(host.address AND 0x000000FF) + "." + STRING$((host.address >> 8) AND 0x000000FF) + "." + STRING$((host.address >> 16) AND 0x000000FF) + "." + STRING$((host.address >> 24) AND 0x000000FF)
'
	FOR i = 0 TO 7
		PRINT "host.addresses[" + STRING$(i) + "]      = "; HEX$ (host.addresses[i],8); " = "; STRING$(host.addresses[i] AND 0x000000FF) + "." + STRING$((host.addresses[i] >> 8) AND 0x000000FF) + "." + STRING$((host.addresses[i] >> 16) AND 0x000000FF) + "." + STRING$((host.addresses[i] >> 24) AND 0x000000FF)
	NEXT i
'
	PRINT "host.addressBytes      = "; HEX$ (host.addressBytes,8)
	PRINT "host.addressFamily     = "; HEX$ (host.addressFamily,8)
	PRINT "host.protocolFamily    = "; HEX$ (host.protocolFamily,8)
	PRINT "host.protocol          = "; HEX$ (host.protocol,8)
'
' test ip address/string functions
'
	PRINT
	XinAddressNumberToString (@address$$, @address$)
	PRINT "<"; HEX$(address$$,16); "> <"; address$; "> <";
	XinAddressStringToNumber (@address$, @addr$$)
	PRINT address$; "> <"; HEX$(addr$$,16); ">"
'
'
' *****  Vince test code  *****
'
	GOSUB VinceTestCode
	RETURN									' execute to run ONLY the Vince test code
'
'
'
' *****  open a socket  *****
'
	PRINT
	PRINT "#####  error = XinSocketOpen (@socket, addressType, socketType, flags)  #####"
'
	flags = 0
	socketType = 0
	addressType = 0
	error = XinSocketOpen (@socket, @addressType, @socketType, flags)
'
	PRINT
	PRINT "error                  = "; HEX$ (error, 8)
	PRINT "socket                 = "; HEX$ (socket, 8)
	PRINT "addressType            = "; HEX$ (addressType, 8)
	PRINT "socketType             = "; HEX$ (socketType, 8)
	PRINT "flags                  = "; HEX$ (flags, 8)
'
'
' bind to an address and/or port
'
	PRINT
	PRINT "#####  error = XinSocketBind (socket, block, address$$, port)  #####"
'
	block = 0
	port = 0x2020
	error = XinSocketBind (socket, block, address$$, @port)
'
	PRINT
	PRINT "error                  = "; HEX$ (error, 8)
	PRINT "socket                 = "; HEX$ (socket, 8)
	PRINT "block                  = "; HEX$ (block, 8)
	PRINT "address$$              = "; HEX$ (address$$, 16)
	PRINT "port                   = "; HEX$ (port, 8)
'
'
' listen for a client to connect
'
	PRINT
	PRINT "#####  error = XinSocketListen (socket, block, flags)  #####"

	error = XinSocketListen (socket, block, flags)
'
	PRINT
	PRINT "error                  = "; HEX$ (error, 8)
	PRINT "socket                 = "; HEX$ (socket, 8)
	PRINT "block                  = "; HEX$ (block, 8)
	PRINT "flags                  = "; HEX$ (flags, 8)
'
'
' accept a connection
'
	PRINT
	PRINT "#####  error = XinSocketAccept (socket, block, @remote, flags)  #####"
'
	DO UNTIL error
		error = 0
		remote = 0
		block = 20000
		error = XinSocketAccept (socket, block, @remote, flags)
	LOOP UNTIL remote
'
	PRINT
	PRINT "error                  = "; HEX$ (error, 8)
	PRINT "socket                 = "; HEX$ (socket, 8)
	PRINT "block                  = "; HEX$ (block, 8)
	PRINT "remote                 = "; HEX$ (remote, 8)
	PRINT "flags                  = "; HEX$ (flags, 8)
'
' check status of socket
'
	PRINT
	PRINT "#####  error = XinSocketGetStatus (socket, @remote, @syssocket, @syserror, @status, @sockettype, @readbytes, @writebytes)  #####"
'
	error = XinSocketGetStatus (socket, @remote, @syssocket, @syserror, @status, @sockettype, @readbytes, @writebytes)
'
	PRINT
	PRINT "error                  = "; HEX$ (error,8)
	PRINT "socket                 = "; HEX$ (socket,8)
	PRINT "remote                 = "; HEX$ (remote,8)
	PRINT "syssocket              = "; HEX$ (syssocket,8)
	PRINT "sysremote              = "; HEX$ (sysremote,8)
	PRINT "syserror               = "; HEX$ (syserror,8)
	PRINT "status                 = "; HEX$ (status,8)
	PRINT "sockettype             = "; HEX$ (sockettype,8)
	PRINT "readbytes              = "; HEX$ (readbytes,8)
	PRINT "writebytes             = "; HEX$ (writebytes,8)
'
' check status of remote socket
'
	PRINT
	PRINT "#####  error = XinSocketGetStatus (rsocket, @rremote, @sysremote, @remerror, @rstatus, @rtype, @rreadbytes, @rwritebytes)  #####"
'
	rsocket = remote
	error = XinSocketGetStatus (rsocket, @rremote, @rsyssocket, @rsyserror, @rstatus, @rtype, @readbytes, @writebytes)
'
	PRINT
	PRINT "error                  = "; HEX$ (error,8)
	PRINT "rsocket                = "; HEX$ (rsocket,8)
	PRINT "rremote                = "; HEX$ (rremote,8)
	PRINT "rsyssocket             = "; HEX$ (rsyssocket,8)
	PRINT "rsyserror              = "; HEX$ (rsyserror,8)
	PRINT "rstatus                = "; HEX$ (rstatus,8)
	PRINT "rtype                  = "; HEX$ (rtype,8)
	PRINT "rreadbytes             = "; HEX$ (readbytes,8)
	PRINT "rwritebytes            = "; HEX$ (writebytes,8)
'
' get network addresses of this socket and remote socket
'
	PRINT
	PRINT "#####  error = XinSocketGetAddress (socket, @port, @address$$, @remote, @rport, @raddress$$)  #####"
'
	error = XinSocketGetAddress (socket, @port, @address$$, @remote, @rport, @raddress$$)
'
	PRINT
	PRINT "error                  = "; HEX$ (error,8)
	PRINT "port                   = "; HEX$ (port,8)
	PRINT "address$$              = "; HEX$ (address$$,16)
	PRINT "rport                  = "; HEX$ (rport,8)
	PRINT "raddress$$             = "; HEX$ (raddress$$,16)
'
'
'
' read and write four timestamp requests from client "aclient.x"
'
	FOR i = 0 TO 3
		GOSUB ReadTimeRequest
		GOSUB WriteTimeString
	NEXT i
'
' Close all sockets
'
	XinCleanup ()
	RETURN
'
'
'
'
' *****  ReadTimeRequest  *****  read bytes from "aclient.x"
'
SUB ReadTimeRequest
	buffer$ = NULL$ (4096)
	address = &buffer$
	maxbytes = 4096
'
	PRINT
	PRINT "#####  error = XinSocketRead (socket, block, address, maxbytes, flags, @bytes)  #####"
'
	DO UNTIL error
		bytes = 0
		error = 0
		read$ = ""
		block = 20000
		error = XinSocketRead (socket, block, address, maxbytes, flags, @bytes)
	LOOP UNTIL bytes
'
	IFZ error THEN read$ = LEFT$ (buffer$, bytes)
'
	PRINT
	PRINT "error                  = "; HEX$ (error, 8)
	PRINT "socket                 = "; HEX$ (socket, 8)
	PRINT "block                  = "; HEX$ (block, 8)
	PRINT "address                = "; HEX$ (address, 8)
	PRINT "maxbytes               = "; HEX$ (maxbytes, 8)
	PRINT "flags                  = "; HEX$ (flags, 8)
	PRINT "bytes                  = "; HEX$ (bytes, 8)
	PRINT "read                   = \""; read$; "\""
END SUB
'
'
' *****  WriteTimeString  *****  write bytes to "aclient.x"
'
SUB WriteTimeString
	GOSUB GetTime
	address = &time$
	maxbytes = SIZE (time$)
'
	PRINT
	PRINT "#####  error = XinSocketWrite (socket, block, address, maxbytes, flags, @bytes)  #####"
'
	bytes = 0
	error = 0
	block = 20000
	error = XinSocketWrite (socket, block, address, maxbytes, flags, @bytes)
'
	PRINT
	PRINT "error                  = "; HEX$ (error, 8)
	PRINT "socket                 = "; HEX$ (socket, 8)
	PRINT "block                  = "; HEX$ (block, 8)
	PRINT "address                = "; HEX$ (address, 8)
	PRINT "maxbytes               = "; HEX$ (maxbytes, 8)
	PRINT "flags                  = "; HEX$ (flags, 8)
	PRINT "bytes                  = "; HEX$ (bytes, 8)
	PRINT "write                  = \""; LEFT$ (time$, bytes)
END SUB
'
'
' *****  GetTime  *****
'
SUB GetTime
	time$ = ""
	XstGetDateAndTime (@year, @month, @day, @weekday, @hour, @minute, @second, @nanosecond)
	time$ = time$ + RIGHT$ ("0000" + STRING$(year), 4)
	time$ = time$ + RIGHT$ ("00" + STRING$(month), 2)
	time$ = time$ + RIGHT$ ("00" + STRING$(day), 2) + ":"
	time$ = time$ + RIGHT$ ("00" + STRING$(hour), 2)
	time$ = time$ + RIGHT$ ("00" + STRING$(minute), 2)
	time$ = time$ + RIGHT$ ("00" + STRING$(second), 2) + "."
	time$ = time$ + RIGHT$ ("000000000" + STRING$ (nanosecond), 9)
'
	length = LEN (time$)
	address = &time$
	sendlen = 0
'
	PRINT
	PRINT "time                   = \""; time$; "\""
END SUB
'
'
' *****  VinceTestCode  *****
'
SUB VinceTestCode
'
' translate "www.worldonline.nl" to host info and show it.
'
	XinHostNameToInfo ("www.worldonline.nl", @host)
	GOSUB ShowIt
'
' check local host, change it to own ip-address
'
	XinAddressStringToNumber ("76.7.4.3", @address$$)
	XinHostNumberToInfo (address$$, @host)
	GOSUB ShowIt
'
' check local DNS, change it to other if fail
'
	XinHostNameToInfo ("www.test.com", @host)
	GOSUB ShowIt
'
' retrieve host-info on "sidbase.webprovider.com" and then translate it's address to host-info
'
	XinHostNameToInfo ("sidbase.webprovider.com", @host)
	XinHostAddressToInfo (host.address, @host)
	GOSUB ShowIt
'
' retrieve host-info on "www.kpn.com" and then translate it's address to host-info
'
	XinHostNameToInfo ("www.kpn.com", @host)
	XinHostAddressToInfo (host.address, @host)
	GOSUB ShowIt
'
' retrieve host-info on "www.worldonline.nl" and then translate it's address to host-info
'
	XinHostNameToInfo("www.worldonline.nl", @host)
	XinHostAddressToInfo(host.address, @host)
	GOSUB ShowIt
END SUB
'
'
'  *****  ShowIt  *****
'
SUB ShowIt
	hostaddress = host.address
	address$$ = host.address
	host$ = host.name
'
	PRINT
	PRINT "host.name              = \""; host.name; "\""
	PRINT "host.alias[0]          = \""; host.alias[0]; "\""
	PRINT "host.alias[1]          = \""; host.alias[1]; "\""
	PRINT "host.alias[2]          = \""; host.alias[2]; "\""
	PRINT "host.system            = \""; host.system; "\""
	PRINT "host.hostnumber        = "; HEX$ (host.hostnumber,8)
	PRINT "host.address           = "; HEX$ (host.address,8); " = "; STRING$(host.address AND 0x000000FF) + "." + STRING$((host.address >> 8) AND 0x000000FF) + "." + STRING$((host.address >> 16) AND 0x000000FF) + "." + STRING$((host.address >> 24) AND 0x000000FF)
'
	FOR i = 0 TO 7
		PRINT "host.addresses[" + STRING$(i) + "]      = "; HEX$ (host.addresses[i],8); " = "; STRING$(host.addresses[i] AND 0x000000FF) + "." + STRING$((host.addresses[i] >> 8) AND 0x000000FF) + "." + STRING$((host.addresses[i] >> 16) AND 0x000000FF) + "." + STRING$((host.addresses[i] >> 24) AND 0x000000FF)
	NEXT i
'
	PRINT "host.addressBytes      = "; HEX$ (host.addressBytes,8)
	PRINT "host.addressFamily     = "; HEX$ (host.addressFamily,8)
	PRINT "host.protocolFamily    = "; HEX$ (host.protocolFamily,8)
	PRINT "host.protocol          = "; HEX$ (host.protocol,8)
'
' test ip address/string functions
'
	PRINT
	XinAddressNumberToString (@address$$, @address$)
	PRINT "<"; HEX$(address$$,16); "> <"; address$; "> <";
	XinAddressStringToNumber (@address$, @addr$$)
	PRINT address$; "> <"; HEX$(addr$$,16); ">"
END SUB
END FUNCTION
'
'
' ##############################
' #####  XinInitialize ()  #####
' ##############################
'
'	/*
'	[XinInitialize]
' Description = The XinInitialize function calls WSAStartup to intialize a socket session.
' Function    = error = XinInitialize (@base, @hosts, @version, @sockets, @comment$, @note$)
' ArgCount    = 6
' Arg1        = base : Internal HOST host[] array base. Returns 0.
' Arg2        = hosts : Returns the number of local hosts.
' Arg3				= version : Returns the version of the Windows Sockets specification that the Windows Sockets DLL expects the caller to use.
' Arg4				= sockets : Ignore.
' Arg5				= comment$ : Returns a description of the Windows Sockets implementation.
' Arg6				= note$ : Returns startup status or configuration information.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = Call Xin() to intialize the Xin library. Xin() will then call XinInitialize().
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinInitialize (@base, @hosts, @version, @sockets, @comment$, @note$)
	SHARED  initialized
	SHARED  HOST  host[]
	SHARED  WSADATA  wsadata
'
' Note:
' - initialized is used to signal if XinInitialize() has been called.

' *****  initialize arguments  *****
'
	base = 0
	hosts = 0
	sockets = 0
	version = 0
	system$ = ""
	notes$ = ""
'
'
' *****  get basic network information  *****
'
	error = WSAStartup ($$NETWORKVERSION, &wsadata)
'
	IF error THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "WSAStartup() : error : "; errno
		SystemErrorToError (errno, @error)
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
'
' *****  get local host name  *****
'
	buffer$ = NULL$ (511)
'
	error = gethostname (&buffer$, 511)
'
	IF error THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "gethostname() : error : "; errno
		error = ($$ErrorObjectNetwork << 8) + $$ErrorNatureUnavailable
		XstSetSystemError (errno)
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
'
' *****  get default local host name  *****
'
	host$ = TRIM$(CSTRING$(&buffer$))
	IFZ host$ THEN RETURN ($$TRUE)
'
'
' *****  add the default host to the host[] array  *****
'
	addr = 0
	name$ = host$
	AddHost (@host, @addr, @name$)
'
	IF (host < 0) THEN RETURN ($$TRUE)
'	IF (addr <= 0) THEN RETURN ($$TRUE)
	IFZ host[] THEN RETURN ($$TRUE)
	IFZ name$ THEN RETURN ($$TRUE)
'
'
' *****  set up return arguments  *****
'
	base 				= 0
	hosts 			= UBOUND (host[])+1
	version 		= wsadata.wVersion
	sockets 		= wsadata.iMaxSockets
	comment$ 		= CSTRING$ (&wsadata.szDescription)
	note$ 			= CSTRING$ (&wsadata.szSystemStatus)
	initialized = $$TRUE
	RETURN ($$FALSE)
END FUNCTION
'
'
' #########################################
' #####  XinAddressNumberToString ()  #####
' #########################################
'
'	/*
'	[XinAddressNumberToString]
' Description = The XinAddressNumberToString function converts a network address into a string in dotted format.
' Function    = error = XinAddressNumberToString (address$$, @address$)
' ArgCount    = 2
' Arg1        = address$$ : A network address (eg, host.address).
' Arg2        = address$ : Returns the network address in a dotted format.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = This function takes an Internet address structure specified by the address$$ parameter. It returns an ASCII string representing the address in . notation as a.b.c.d.
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinAddressNumberToString (address$$, @address$)
	SHARED  initialized
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinAddressNumberToString() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	address$ = ""
	IFZ address$$ THEN RETURN ($$TRUE)
'
	address = address$$
'
	addr = inet_ntoa (address)
'
	address$ = CSTRING$ (addr)
	IFZ address$ THEN RETURN ($$TRUE)
	RETURN ($$FALSE)
END FUNCTION
'
'
' #########################################
' #####  XinAddressStringToNumber ()  #####
' #########################################
'
'	/*
'	[XinAddressStringToNumber]
' Description = The XinAddressStringToNumber function converts a string containing an Internet Protocol dotted address into network address.
' Function    = error = XinAddressStringToNumber (address$, @address$$)
' ArgCount    = 2
' Arg1        = address$ : A string representing a number expressed in the internet standard . notation, eg, a.b.c.d..
' Arg2        = address$$ : Returns the network address.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = All Internet addresses are returned in IP's network order (bytes ordered from left to right).<br>Internet Addresses<br>Values specified using the . notation take one of the following forms:<br>a.b.c.d    a.b.c    a.b    a<br><br>When four parts are specified, each is interpreted as a byte of data and assigned, from left to right, to the four bytes of an Internet address. Note that when an Internet address is viewed as a 32-bit integer quantity on the Intel architecture, the bytes referred to above appear as d.c.b.a. That is, the bytes on an Intel processor are ordered from right to left.
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinAddressStringToNumber (address$, address$$)
	SHARED  initialized
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinAddressStringToNumber() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	address$$ = 0
	IFZ address$ THEN RETURN ($$TRUE)
'
	address$$ = inet_addr (&address$)
'
	IFZ address THEN RETURN ($$TRUE)
	RETURN ($$FALSE)
END FUNCTION
'
'
' ##################################
' #####  XinHostNameToInfo ()  #####
' ##################################
'
'	/*
'	[XinHostNameToInfo]
' Description = The XinHostNameToInfo function gets host information corresponding to a hostname.
' Function    = error = XinHostNameToInfo (name$, HOST @info)
' ArgCount    = 2
' Arg1        = name$ : A string representing a hostname.
' Arg2        = HOST info : Returns a filled HOST info.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = 
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinHostNameToInfo (name$, HOST data)
	SHARED  HOST  host[]
	SHARED  initialized
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinHostNameToInfo() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	addr = 0
	host = -1
'
	error = AddHost (@host, @addr, @name$)
'
	IF error THEN
		IF #debug THEN PRINT "XinHostNameToInfo() : error : AddHost() call returned error = " ; error
		RETURN ($$TRUE)
	END IF
'
	IF (host < 0) THEN RETURN ($$TRUE)
'	IF (addr <= 0) THEN RETURN ($$TRUE)
	IFZ host[] THEN RETURN ($$TRUE)
	IFZ name$ THEN RETURN ($$TRUE)
'
	upper = UBOUND (host[])
'
	FOR i = 0 TO upper
		IF (name$ == host[i].name) THEN
			data = host[i]
			RETURN
		END IF
	NEXT i
'
	RETURN ($$TRUE)
END FUNCTION
'
'
' ####################################
' #####  XinHostNumberToInfo ()  #####
' ####################################
'
'	/*
'	[XinHostNumberToInfo]
' Description = The XinHostNumberToInfo function gets host information corresponding to an index to the HOST host[] array.
' Function    = error = XinHostNumberToInfo (id, HOST @info)
' ArgCount    = 2
' Arg1        = id : A host index.
' Arg2        = HOST info : Returns a filled HOST info.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = 
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinHostNumberToInfo (id, HOST data)
	SHARED  HOST  host[]
	SHARED  initialized
'
	IFZ initialized THEN
		IF #debug THEN PRINT "XinHostNumberToInfo() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	upper = UBOUND (host[])
	IF (id < 0) THEN RETURN ($$TRUE)
	IF (id > upper) THEN RETURN ($$TRUE)
'
	data = host[id]
	RETURN ($$FALSE)
END FUNCTION
'
'
' #####################################
' #####  XinHostAddressToInfo ()  #####
' #####################################
'
'	/*
'	[XinHostAddressToInfo]
' Description = The XinHostAddressToInfo function gets host information corresponding to a host address.
' Function    = error = XinHostAddressToInfo (address, HOST @info)
' ArgCount    = 2
' Arg1        = address : A host address.
' Arg2        = HOST info : Returns a filled HOST info.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = 
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinHostAddressToInfo (address, HOST data)
	SHARED  HOST  host[]
	SHARED  initialized
'
	IFZ initialized THEN
		IF #debug THEN PRINT "XinHostAddressToInfo() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ address THEN RETURN ($$TRUE)
	upper = UBOUND (host[])
	name$ = ""
	host = -1
'
	error = AddHost (@host, @address, @name$)
'
	IF error THEN
		IF #debug THEN PRINT "XinHostAddressToInfo() : error : AddHost() call returned error = " ; error
		error = ($$ErrorObjectComputer << 8) OR $$ErrorNatureUnknown
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ host THEN RETURN ($$TRUE)
'
	FOR i = 0 TO UBOUND (host[])
		IF (address == host[i].address) THEN
			data = host[i]
			RETURN $$FALSE
		END IF
	NEXT i
'
	RETURN ($$TRUE)
END FUNCTION
'
'
' ##############################
' #####  XinSocketOpen ()  #####
' ##############################
'
'	/*
'	[XinSocketOpen]
' Description = The XinSocketOpen function creates a socket which is bound to a specific service provider.
' Function    = error = XinSocketOpen (@socket, @addressFamily, @socketType, flags)
' ArgCount    = 4
' Arg1        = socket : Returned new socket identifier.
' Arg2        = addressFamily : An address family specification, eg, $$AF_INET. If not specified, then a default address family specification is returned.
' Arg3				= socketType : A type specification for the new socket, eg, $$SOCK_STREAM. If not specified, then a default socket type is returned.
' Arg4				= flags : A particular protocol to be used with the socket which is specific to the indicated address family, eg, $$IPPROTO_TCP.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = 
'	See Also    = For more details on using this function, see socket in the Win32 Programmer's Reference.
'	Examples    = 
'	*/
'
FUNCTION  XinSocketOpen (socket, addressFamily, socketType, flags)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	SHARED  HOST  host[]
	XSOCKET  zero
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketOpen() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
' *****  if zero, fill addressFamily and socketType with defaults  *****
'
	IF (addressFamily <= 0) THEN addressFamily = host[0].addressFamily
	IF (socketType <= 0) THEN socketType = $$SOCK_STREAM
'
' *****  open a socket  *****
'
	syssocket = socket (addressFamily, socketType, flags)
'
	IF (syssocket = -1) THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSocketOpen() : error : "; errno
		SystemErrorToError (errno, @error)
		XstSetSystemError (errno)
		old = ERROR (error)
		socket = $$FALSE
		RETURN ($$TRUE)
	END IF
'
	IFZ socket[] THEN
		DIM socket[1]
	END IF
'
	upper = UBOUND (socket[])
'
	slot = -1
	FOR i = 1 TO upper
		IFZ socket[i].socket THEN
			slot = i
			EXIT FOR
		END IF
	NEXT i
'
	IF (slot < 0) THEN
		slot = upper + 1
		upper = upper + 1
		REDIM socket[upper]
	END IF
'
	socket = slot
	socket[socket] = zero
	socket[socket].socket = socket
	socket[socket].syssocket = syssocket
	socket[socket].socketType = socketType
	socket[socket].addressFamily = addressFamily
	socket[socket].protocol = host[0].protocolFamily
	socket[socket].status = $$SocketStatusOpenSuccess
END FUNCTION
'
'
' ##############################
' #####  XinSocketBind ()  #####
' ##############################
'
'	/*
'	[XinSocketBind]
' Description = The XinSocketBind function associates a local address with a socket.
' Function    = error = XinSocketBind (socket, block, address$$, port)
' ArgCount    = 4
' Arg1        = socket : An unbound socket identifier.
' Arg2        = block : The blocking timeout in microseconds.
' Arg3				= address$$ : The local network address.
' Arg4				= port : An integer number between 1 and 65535. Port numbers smaller than 1024 are considered well-known -- for example, telnet uses port 23, http uses 80, ftp uses 21, and so on, while tcp uses port numbers >= 1024.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = This routine is used on an unconnected connectionless or connection-oriented socket, before subsequent connects or listens. When a socket is created with socket, it exists in a name space (address family), but it has no name assigned. bind establishes the local association of the socket by assigning a local name to an unnamed socket.
'	See Also    = For more details on using this function, see bind in the Win32 Programmer's Reference.
'	Examples    = 
'	*/
'
FUNCTION  XinSocketBind (socket, block, address$$, port)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	SOCKADDR_IN  sockaddrin
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketBind() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketBind() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	status = socket[socket].status
	syssocket = socket[socket].syssocket
'
	IF (status AND $$SocketStatusBindSuccess) THEN
		IF #debug THEN PRINT "XinSocketBind() : error : socket bind already complete : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (port <= 0) THEN port = 0
	addressFamily = socket[socket].addressFamily
	IF (address$$ <= 0) THEN address$$ = $$INADDR_ANY
'
' the Windows 2nd argument is sockaddrin
' the Linux 2nd argument is sockaddr - but they overlay and are byte for byte equal - ???
'
	sockaddrin.sin_family = socket[socket].addressFamily
	sockaddrin.sin_port = htons (port)
	sockaddrin.sin_addr = address$$
'
	length = LEN (sockaddrin)
	error = bind (syssocket, &sockaddrin, length)
'
	IF (error = -1) THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSocketBind() : error : "; errno
		SystemErrorToError (errno, @error)
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	status = status OR $$SocketStatusBindSuccess
	socket[socket].status = status
	address$$ = sockaddrin.sin_addr
	port = sockaddrin.sin_port

END FUNCTION
'
'
' ################################
' #####  XinSocketListen ()  #####
' ################################
'
'	/*
'	[XinSocketListen]
' Description = The XinSocketListen function establishes a socket to listen for an incoming connection.
' Function    = error = XinSocketListen (socket, block, flags)
' ArgCount    = 3
' Arg1        = socket : A socket identifier.
' Arg2        = block : Blocking timeout in microseconds.
' Arg3				= flags : Not used.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = To accept connections, a socket is first created with XinSocketOpen, a backlog for incoming connections is specified with XinSocketListen, and then the connections are accepted with XinSocketAccept. XinSocketListen applies only to sockets that are connection oriented, for example, those of type $$SOCK_STREAM.
'	See Also    = For more details on using this function, see listen in the Win32 Programmer's Reference.
'	Examples    = 
'	*/
'
FUNCTION  XinSocketListen (socket, block, flags)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketListen() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketListen() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ socket[socket].socket THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketListen() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	status = socket[socket].status
	syssocket = socket[socket].syssocket
'
	IFZ syssocket THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketListen() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ (status AND $$SocketStatusBindSuccess) THEN
		IF #debug THEN PRINT "XinSocketListen() : error : need socket bind first : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (status AND $$SocketStatusListenSuccess) THEN
		IF #debug THEN PRINT "XinSocketListen() : error : socket listen already successful : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	error = listen (syssocket, 1)
'
	IF (error = -1) THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSocketListen() : error : "; errno
		SystemErrorToError (errno, @error)
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	status = status OR $$SocketStatusListenSuccess
	socket[socket].status = status
END FUNCTION
'
'
' ################################
' #####  XinSocketAccept ()  #####
' ################################
'
'	/*
'	[XinSocketAccept]
' Description = The XinSocketAccept function accepts a connection on a socket.
' Function    = error = XinSocketAccept (socket, block, @remote, flags)
' ArgCount    = 4
' Arg1        = socket : A socket identifier which is listening for connections after XinSocketListen.
' Arg2        = block : The blocking timeout in microseconds.
' Arg3				= remote : A returned newly created and accepted socket. The accepted socket cannot be used to accept more connections. The original socket remains open. 
' Arg4				= flags : Not used.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = For more details on using this function, see accept in the Win32 Programmer's Reference.
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinSocketAccept (socket, block, @remote, flags)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	SOCKADDR_IN  sockaddrin
	FD_SET  readset
	FD_SET  writeset
	FD_SET  errorset
	TIMEVAL  timeval
	AUTOX  length
	XSOCKET  zero
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketAccept() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	remote = $$FALSE
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketAccept() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ socket[socket].socket THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketAccept() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	status = socket[socket].status
	syssocket = socket[socket].syssocket
'
	IFZ syssocket THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketAccept() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ (status AND $$SocketStatusListenSuccess) THEN
		IF #debug THEN PRINT "XinSocketAccept() : error : need socket listen first : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (status AND $$SocketStatusAcceptSuccess) THEN
		IF #debug THEN PRINT "XinSocketAccept() : error : socket accept already successful : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (status AND $$SocketStatusConnectRequest) THEN
		IF #debug THEN PRINT "XinSocketAccept() : error : socket connect request already pending : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (status AND $$SocketStatusConnectSuccess) THEN
		IF #debug THEN PRINT "XinSocketAccept() : error : socket connect request already successful : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	SELECT CASE TRUE
		CASE (block < 1)			: block = 0				' do not block
		CASE (block < 10)			: block = 10			' min block = 10us
		CASE (block > 20000)	: block = 20000		' max block = 20ms
	END SELECT
'
	IF (block <= 0) THEN
		timeval.tv_sec = 0				' non-blocking
		timeval.tv_usec = 0				'
	ELSE
		timeval.tv_sec = 0				' blocking up to 20ms maximum
		timeval.tv_usec = block		'
	END IF
'
	FDZERO (@readset)
	FDZERO (@errorset)
	FDSET (syssocket, @readset)
	FDSET (syssocket, @errorset)
'
	count = select (syssocket+1, &readset, 0, &errorset, &timeval)
'
' (count = 0) means no connection attempt yet
'
	IF (count = 0) THEN RETURN ($$FALSE)
'
' (count < 0) means an error occured
'
	IF (count < 0) THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSocketAccept() : error : select() reported error : "; socket; errno
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
		socket[socket].status = status OR $$SocketStatusFailed
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
' (count > 0) means something was detected
'
' see if the socket or network died prior to accept success
'
	hit = FDISSET (syssocket, @errorset)
	IF hit THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSocketAccept() : error : select() reported error : "; socket; errno
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
		socket[socket].status = status OR $$SocketStatusFailed
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
' select() indicates success of an accept by saying socket is readable
'
	remote = $$FALSE
	sysremote = $$FALSE
'
	hit = FDISSET (syssocket, @readset)
	IF hit THEN
		length = SIZE (sockaddrin)
'
		sysremote = accept (syssocket, &sockaddrin, &length)
'
		IF (sysremote <= 0) THEN
			errno = GetLastError ()
			IF #debug THEN PRINT "XinSocketAccept() : error : accept() reported error : "; socket; errno
			error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
			socket[socket].status = status OR $$SocketStatusFailed
			old = ERROR (error)
			remote = $$FALSE
			RETURN ($$TRUE)
		END IF
'
		upper = UBOUND (socket[])
'
		slot = -1
		FOR i = 1 TO upper
			IFZ socket[i].socket THEN
				slot = i
				EXIT FOR
			END IF
		NEXT i
'
		IF (slot < 0) THEN
			slot = upper + 1
			upper = upper + 1
			REDIM socket[upper]
		END IF
'
		remote = slot
		socket[remote] = zero
		socket[remote].hostnumber = -1
		socket[remote].socket = remote
		socket[remote].remote = socket
		socket[remote].syssocket = sysremote
		socket[remote].sysremote = syssocket
		socket[remote].port = sockaddrin.sin_port
		socket[remote].address = sockaddrin.sin_addr
		socket[remote].protocol = socket[socket].protocol
		socket[remote].socketType = socket[socket].socketType
		socket[remote].addressType = socket[socket].addressType
		socket[remote].addressBytes = socket[socket].addressBytes
		socket[remote].addressFamily = sockaddrin.sin_family
		socket[remote].status = $$SocketStatusRemote OR $$SocketStatusConnected
'
		socket[socket].status = status OR $$SocketStatusAcceptSuccess OR $$SocketStatusConnected
		socket[socket].sysremote = sysremote
		socket[socket].remote = remote
		RETURN ($$FALSE)
	END IF
END FUNCTION
'
'
' ########################################
' #####  XinSocketConnectRequest ()  #####
' ########################################
'
'	/*
'	[XinSocketConnectRequest]
' Description = The XinSocketConnectRequest function establishes a connection to a peer.
' Function    = error = XinSocketConnectRequest (socket, block, address$$, port)
' ArgCount    = 4
' Arg1        = socket : A socket identifier.
' Arg2        = block : The blocking timeout in microseconds.
' Arg3				= address$$ : The destination address to which the socket is to be connected.
' Arg4				= port : An integer number between 1 and 65535. Port numbers smaller than 1024 are considered well-known -- for example, telnet uses port 23, http uses 80, ftp uses 21, and so on, while tcp uses port numbers >= 1024.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = This function is used to create a connection to the specified destination. If the socket, socket, is unbound, unique values are assigned to the local association by the system, and the socket is marked as bound.<br><br>For connection-oriented sockets (for example, type $$SOCK_STREAM), an active connection is initiated to the foreign host using name (an address in the name space of the socket; for a detailed description, see bind). When the socket call completes successfully, the socket is ready to send/receive data.
'	See Also    = For more details on using this function, see connect in the Win32 Programmer's Reference.
'	Examples    = 
'	*/
'
FUNCTION  XinSocketConnectRequest (socket, block, address$$, port)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	SOCKADDR_IN  sockaddrin
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketConnectRequest() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketConnectRequest() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	status = socket[socket].status
	syssocket = socket[socket].syssocket
'
	IFZ (status AND $$SocketStatusBindSuccess) THEN
		IF #debug THEN PRINT "XinSocketConnectRequest() : error : need socket bind first : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (status AND $$SocketStatusConnectRequest) THEN
		IF #debug THEN PRINT "XinSocketConnectRequest() : error : socket connect request already made : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (status AND $$SocketStatusConnectSuccess) THEN
		IF #debug THEN PRINT "XinSocketConnectRequest() : error : socket connect success already detected : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (status AND $$SocketStatusConnected) THEN
		IF #debug THEN PRINT "XinSocketConnectRequest() : error : socket connect request already connected : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
' set socket non-blocking so connect() doesn't block
'
	error = SetSocketNonBlocking (socket)
	IF error THEN RETURN ($$TRUE)
'
' send connect request - connect probably will not happen now
'
	sockaddrin.sin_family = $$AF_INET
	sockaddrin.sin_addr = address$$
	' Note: the user specifies the port-number in 'host order'; but sin_port
	' must be in 'network order' (big endian).
	sockaddrin.sin_port = htons(port)

	length = LEN (sockaddrin)
	connected = $$FALSE
	errno = $$FALSE
'
	error = connect (syssocket, &sockaddrin, length)
'
	SELECT CASE TRUE
		CASE (error = 0)	: connected = $$TRUE
												SetSocketNonBlocking (socket)
												status = status OR $$SocketStatusConnectRequest
												status = status OR $$SocketStatusConnectSuccess
		CASE (error < 0)	: errno = GetLastError ()
												SELECT CASE errno
													CASE $$WSAEWOULDBLOCK:
														errno = $$FALSE
														status = status OR $$SocketStatusConnectRequest
														status = status AND NOT $$SocketStatusConnected
													CASE ELSE:
														IF #debug THEN PRINT "XinSocketConnectRequest() : error : "; socket; errno
														status = status AND NOT $$SocketStatusConnectRequest
														status = status AND NOT $$SocketStatusConnected
														status = status OR $$SocketStatusFailed
												END SELECT
		CASE (error > 0)	: errno = GetLastError ()
												SELECT CASE errno
													CASE $$WSAEWOULDBLOCK	: errno = $$FALSE
													CASE ELSE							: IF #debug THEN PRINT "XinSocketConnectRequest() : error : "; socket; errno
												END SELECT
												status = status AND NOT $$SocketStatusConnectRequest
												status = status AND NOT $$SocketStatusConnected
												status = status OR $$SocketStatusFailed
	END SELECT
'
	socket[socket].status = status
	IF errno THEN RETURN (errno)
'
' if not connected, see if a short block was requested
'
	IFZ connected THEN
		SELECT CASE TRUE
			CASE (block < 1)			: block = 0				' do not block
			CASE (block < 10)			: block = 10			' min block = 10us
			CASE (block > 20000)	: block = 20000		' max block = 20ms
		END SELECT
'
		IF block THEN
			error = XinSocketConnectStatus (socket, block, @connected)
			IF error THEN
				status = status AND NOT $$SocketStatusConnectRequest
				status = status AND NOT $$SocketStatusConnected
				status = status OR $$SocketStatusFailed
				socket[socket].status = status
				RETURN ($$TRUE)
			END IF
		END IF
	END IF
'
' if connected, mark status as connected
'
	IF connected THEN
		status = status OR $$SocketStatusConnectRequest
		status = status OR $$SocketStatusConnectSuccess
	END IF
'
	socket[socket].status = status
'
	IF connected THEN
		error = SetSocketBlocking (socket)
		IF error THEN RETURN ($$TRUE)
	END IF
	RETURN ($$FALSE)
END FUNCTION
'
'
' #######################################
' #####  XinSocketConnectStatus ()  #####
' #######################################
'
'	/*
'	[XinSocketConnectStatus]
' Description = The XinSocketConnectStatus function determines the status of a socket, waiting if necessary.
' Function    = error = XinSocketConnectStatus (socket, block, @connected)
' ArgCount    = 3
' Arg1        = socket : A socket identifier.
' Arg2        = block : The blocking timeout in microseconds.
' Arg3				= connected : Returned connection status. If $$TRUE, then the socket is connected and ready to read or write.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = 
'	See Also    = For more details on using this function, see select in the Win32 Programmer's Reference.
'	Examples    = 
'	*/
'
FUNCTION  XinSocketConnectStatus (socket, block, @connected)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	FD_SET  readset
	FD_SET  writeset
	FD_SET  errorset
	TIMEVAL  timeval
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketConnectStatus() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	connected = $$FALSE
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketConnectStatus() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	status = socket[socket].status
	syssocket = socket[socket].syssocket
'
	IF (socket[socket].socket <= 0) THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketConnectStatus() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ syssocket THEN
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		IF #debug THEN PRINT "XinSocketConnectStatus() : error : undefined socket #"
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ (status AND $$SocketStatusConnectRequest) THEN
		IFZ (status AND $$SocketStatusAcceptSuccess) THEN
			IF #debug THEN PRINT "XinSocketConnectStatus() : error : need socket accept or connect request first : "; socket
			error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
			old = ERROR (error)
			RETURN ($$TRUE)
		END IF
	END IF
'
	IF (status AND $$SocketStatusConnected) THEN		' already connected
		connected = $$TRUE
		RETURN ($$FALSE)
	END IF
'
' figure blocking time
'
	SELECT CASE TRUE
		CASE (block < 1)			: block = 0				' do not block
		CASE (block < 10)			: block = 10			' min block = 10us
		CASE (block > 20000)	: block = 20000		' max block = 20ms
	END SELECT
'
	IF (block <= 0) THEN
		timeval.tv_sec = 0				' non-blocking
		timeval.tv_usec = 0				'
	ELSE
		timeval.tv_sec = 0				' blocking up to 20ms maximum
		timeval.tv_usec = block		'
	END IF
'
	connected = $$FALSE
'
	FDZERO (@writeset)
	FDZERO (@errorset)
	FDSET (syssocket, @writeset)
	FDSET (syssocket, @errorset)
'
	count = select (syssocket+1, 0, &writeset, &errorset, &timeval)
'
' (count = 0) means no connection attempt yet
' (count < 0) means an error occured
' (count > 0) means connection success - probably
'
	SELECT CASE TRUE
		CASE (count = 0)		: RETURN ($$FALSE)		' not yet connected
		CASE (count < 0)		: errno = GetLastError ()
													IF #debug THEN PRINT "XinSocketConnectStatus() : error : select() reported error : "; socket; errno
													error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
													status = status AND NOT $$SocketStatusConnectSuccess
													status = status AND NOT $$SocketStatusConnected
													status = status OR $$SocketStatusFailed
													socket[socket].status = status
													connected = $$FALSE
													old = ERROR (error)
													RETURN ($$TRUE)
		CASE (count > 0)		: hit = FDISSET (syssocket, @errorset)
													IF hit THEN
														errno = GetLastError ()
														IF #debug THEN PRINT "XinSocketConnectStatus() : error : select() reported error : "; socket; errno
														error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
														status = status AND NOT $$SocketStatusConnectSuccess
														status = status AND NOT $$SocketStatusConnected
														status = status OR $$SocketStatusFailed
														socket[socket].status = status
														connected = $$FALSE
														old = ERROR (error)
														RETURN ($$TRUE)
													END IF
													hit = FDISSET (syssocket, @writeset)
													IF hit THEN
														status = status OR $$SocketStatusConnectSuccess
														status = status OR $$SocketStatusConnected
														socket[socket].status = status
														SetSocketBlocking (socket)
														connected = $$TRUE
														RETURN ($$FALSE)
													END IF
	END SELECT
END FUNCTION
'
'
' ####################################
' #####  XinSocketGetAddress ()  #####
' ####################################
'
'	/*
'	[XinSocketGetAddress]
' Description = The XinSocketGetAddress function returns client/server port and addresss of the peer to which a socket is connected. It can be used only on a connected socket.
' Function    = error = XinSocketGetAddress (socket, @port, @address$$, @remote, @rport, @raddress$$)
' ArgCount    = 6
' Arg1        = socket : A socket identifier.
' Arg2        = port : Returned port for a connected client socket. 
' Arg3				= address$$ : Returned port for a connected client socket.
' Arg4				= remote : Returned remote socket for a connected server socket.
' Arg5				= rport : Returned remote port for a connected server socket.
' Arg6				= raddress$$ : Returned remote address for a connected server socket.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = For more details on using this function, see getpeername in the Win32 Programmer's Reference.
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinSocketGetAddress (socket, port, address$$, remote, rport, raddress$$)
'	EXTERNAL  errno
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	SOCKADDR_IN  sockaddrin
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketGetAddress() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	port = 0
	rport = 0
	remote = 0
	address$$ = 0
	raddress$$ = 0
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		IF #debug THEN PRINT "XinSocketGetAddress() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ socket[socket].socket THEN
		IF #debug THEN PRINT "XinSocketGetAddress() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	syssocket = socket[socket].syssocket
'
	IF (syssocket <= 0) THEN
		IF #debug THEN PRINT "XinSocketGetAddress() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
' getting address information is handled differently for server and client sockets
'
	remote = socket[socket].remote
	sysremote = socket[remote].syssocket
	IF remote THEN GOSUB Server ELSE GOSUB Client
	RETURN ($$FALSE)
'
'
' *****  Server  *****
'
' server sockets have remote socket numbers for the remote sockets connected to them
' to get the address of a remote client socket, servers getpeername() the remote socket
'
SUB Server
	length = SIZE (sockaddrin)
'
	error = getsockname (sysremote, &sockaddrin, &length)
	errno = GetLastError ()
'
	address$$ = sockaddrin.sin_addr
	port = sockaddrin.sin_port
'
	length = SIZE (sockaddrin)
'
	error = getpeername (sysremote, &sockaddrin, &length)
'
	raddress$$ = sockaddrin.sin_addr
	rport = sockaddrin.sin_port
END SUB
'
'
' *****  Client  *****
'
' client sockets do not have an associated "remote" socket like servers do
' to get the address of the remote server socket, clients getpeername() of their own socket
'
SUB Client
	length = SIZE (sockaddrin)
'
	error = getsockname (syssocket, &sockaddrin, &length)
'
	address$$ = sockaddrin.sin_addr
	port = sockaddrin.sin_port
'
	length = SIZE (sockaddrin)
'
	error = getpeername (syssocket, &sockaddrin, &length)
'
	raddress$$ = sockaddrin.sin_addr
	rport = sockaddrin.sin_port
	remote = 0
END SUB
END FUNCTION
'
'
' ###################################
' #####  XinSocketGetStatus ()  #####
' ###################################
'
'	/*
'	[XinSocketGetStatus]
' Description = The XinSocketGetStatus function returns current socket parameters.
' Function    = error = XinSocketGetStatus (socket, @remote, @syssocket, @syserror, @status, @socketType, @readbytes, @writebytes)
' ArgCount    = 8
' Arg1        = socket : A socket identifier.
' Arg2        = remote : Returned remote socket identifier.
' Arg3				= syssocket : Returned system socket identifier.
' Arg4				= syserror : Returned pending socket error.
' Arg5				= status : Returned current socket status, eg, $$SocketStatusOpenSuccess.
' Arg6				= socketType : Returned socket type, eg, $$SOCKET_STREAM.
' Arg7				= readbytes : Returned size of socket read buffer.
' Arg8				= writebytes : Returned size of socket write buffer.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = For more details on using this function, see getsockopt in the Win32 Programmer's Reference.
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinSocketGetStatus (socket, @remote, @syssocket, @syserror, @status, @socketType, @readbytes, @writebytes)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	AUTOX  rbytes
	AUTOX  wbytes
	AUTOX  serror
	AUTOX  stype
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketGetStatus() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	stype = 0
	serror = 0
	rbytes = 0
	wbytes = 0
	remote = 0
	status = 0
	syssocket = 0
	readbytes = 0
	writebytes = 0
'
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		IF #debug THEN PRINT "XinSocketGetStatus() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ socket[socket].socket THEN
		IF #debug THEN PRINT "XinSocketGetStatus() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	syssocket = socket[socket].syssocket
'
	IF (syssocket <= 0) THEN
		IF #debug THEN PRINT "XinSocketGetStatus() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	remote = socket[socket].remote
	status = socket[socket].status
'
' get size of socket read buffer
'
	level = $$SOL_SOCKET
	optname = $$SO_RCVBUF
	length = LEN (rbytes)
'
	error = getsockopt (syssocket, level, optname, &rbytes, &length)
'
	IFZ error THEN readbytes = rbytes
'
' get size of socket read buffer
'
	level = $$SOL_SOCKET
	optname = $$SO_SNDBUF
	length = LEN (wbytes)
'
	error = getsockopt (syssocket, level, optname, &wbytes, &length)
'
	IFZ error THEN writebytes = wbytes
'
' get pending socket error if any
'
	level = $$SOL_SOCKET
	optname = $$SO_ERROR
	length = LEN (serror)
'
	error = getsockopt (syssocket, level, optname, &serror, &length)
'
	IFZ error THEN syserror = serror
'
' get socket type - as in $$SOCKET_STREAM, the only supported type
'
	level = $$SOL_SOCKET
	optname = $$SO_TYPE
	length = LEN (stype)
'
	error = getsockopt (syssocket, level, optname, &stype, &length)
'
	IFZ error THEN socketType = stype
	RETURN ($$FALSE)
END FUNCTION
'
'
' ##############################
' #####  XinSocketRead ()  #####
' ##############################
'
'	/*
'	[XinSocketRead]
' Description = The XinSocketRead function receives data from a socket.
' Function    = error = XinSocketRead (socket, block, address, maxbytes, flags, @bytes)
' ArgCount    = 6
' Arg1        = socket : Socket identifier.
' Arg2        = block : Blocking timeout in microseconds. If block = 0, then no blocking. If block is < 10, then block = 10. If block is > 20000 then block is set to a maximum of 20000 (20 msec).
' Arg3				= address : Address of buffer for incoming data. 
' Arg4				= maxbytes : Length of buffer.
' Arg5				= flags : Specifies the way in which the call is made, eg, $$MSG_PEEK.
' Arg6				= bytes : Returned number of bytes received.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = This function is used on connected sockets and is used to read incoming data. For byte stream style socket, eg, type $$SOCK_STREAM, as much information as is currently available up to the size of the buffer supplied is returned. 
'	See Also    = For more details on using this function, see recv in the Win32 Programmer's Reference.
'	Examples    = response$ = NULL$(256)<br>DO UNTIL error<br>	error = XinSocketRead (socket, 1000, &response$, 256, 0, @bytes)<br>	IF error THEN<br>		? "XinSocketRead error"; error<br>		RETURN ($$TRUE)<br>	END IF<br>LOOP UNTIL bytes<br>
'	*/
'
FUNCTION  XinSocketRead (socket, block, address, maxbytes, flags, bytes)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	FD_SET  readset
	FD_SET  writeset
	FD_SET  errorset
	TIMEVAL  timeval
	AUTOX  length
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketRead() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	bytes = 0
	error = $$FALSE
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		IF #debug THEN PRINT "XinSocketRead() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (socket != socket[socket].socket) THEN
		IF #debug THEN PRINT "XinSocketRead() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	status = socket[socket].status
	remote = socket[socket].remote
	IFZ remote THEN remote = socket
	syssocket = socket[socket].syssocket
	sysremote = socket[socket].sysremote
'
	IF sysremote THEN
		IF (status AND $$SocketStatusAcceptSuccess) THEN syssocket = sysremote
	END IF
'
	IFZ syssocket THEN
		IF #debug THEN PRINT "XinSocketRead() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ address THEN
		IF #debug THEN PRINT "XinSocketRead() : error : (address = 0) : "; socket
		error = ($$ErrorObjectFunction << 8) OR $$ErrorNatureInvalidArgument
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (maxbytes <= 0) THEN
		IF #debug THEN PRINT "XinSocketRead() : error : (maxbytes <= 0) : "; socket
		error = ($$ErrorObjectFunction << 8) OR $$ErrorNatureInvalidArgument
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ syssocket THEN
		IF #debug THEN PRINT "XinSocketRead() : error : no remote socket : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ (status AND $$SocketStatusConnected) THEN
		IF #debug THEN PRINT "XinSocketRead() : error : socket not connected : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	SELECT CASE TRUE
		CASE (block < 1)			: block = 0				' do not block
		CASE (block < 10)			: block = 10			' min block = 10us
		CASE (block > 20000)	: block = 20000		' max block = 20ms
	END SELECT
'
	IF (block <= 0) THEN
		timeval.tv_sec = 0				' non-blocking
		timeval.tv_usec = 0				'
	ELSE
		timeval.tv_sec = 0				' blocking up to 20ms maximum
		timeval.tv_usec = block		'
	END IF
'
	FDZERO (@readset)
	FDZERO (@errorset)
	FDSET (syssocket, @readset)
	FDSET (syssocket, @errorset)
'
	count = select (syssocket+1, &readset, 0, &errorset, &timeval)
'
' (count = 0) means no data ready to read
'
	IF (count = 0) THEN RETURN ($$FALSE)
'
' (count < 0) means an error occured
'
	IF (count < 0) THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSocketRead() : error : select() reported error : "; socket; errno
		error = ($$ErrorObjectSystemFunction << 8) + $$ErrorNatureFailed
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
' (count > 0) means something was detected
'
' see if the socket or network died prior to select()
'
	hit = FDISSET (syssocket, @errorset)
	IF hit THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSocketRead() : error : select() reported error : "; socket; errno
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
		status = status AND NOT $$SocketStatusConnected
		status = status OR $$SocketStatusFailed
		socket[socket].status = status
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
' see if select() indicates bytes are available to read
'
	hit = FDISSET (syssocket, @readset)
	IF hit THEN
		bytes = recv (syssocket, address, maxbytes, flags)
'
		SELECT CASE TRUE
			CASE (bytes > 0)	: address = address + bytes
			CASE (bytes = 0)	: error = ($$ErrorObjectSocket << 8) + $$ErrorNatureDisconnected
													status = status AND NOT $$SocketStatusConnectSuccess
													status = status AND NOT $$SocketStatusConnectRequest
													status = status AND NOT $$SocketStatusAcceptSuccess
													status = status AND NOT $$SocketStatusConnected
													socket[socket].status = status
													IF remote THEN
														status = socket[remote].status
														status = status AND NOT $$SocketStatusConnectSuccess
														status = status AND NOT $$SocketStatusConnectRequest
														status = status AND NOT $$SocketStatusAcceptSuccess
														status = status AND NOT $$SocketStatusConnected
														socket[remote].status = status
													END IF
													old = ERROR (error)
													RETURN ($$TRUE)
			CASE (bytes < 0)	: errno = GetLastError ()
													connected = $$TRUE
													error = $$TRUE
													bytes = 0
													SELECT CASE errno
														CASE $$WSAEWOULDBLOCK
														CASE $$WSAECONNABORTED	: connected = $$FALSE
														CASE $$WSAECONNRESET		: connected = $$FALSE
														CASE $$WSAESHUTDOWN			: connected = $$FALSE
														CASE $$WSAENOTCONN			: connected = $$FALSE
														CASE $$WSAENETDOWN			: connected = $$FALSE
														CASE ELSE								: connected = $$FALSE
													END SELECT
													IF #debug THEN PRINT "XinSocketRead() : error : recv() reported error : "; socket; errno
													IFZ connected THEN
														error = ($$ErrorObjectSocket << 8) + $$ErrorNatureDisconnected
														status = status AND NOT $$SocketStatusConnectSuccess
														status = status AND NOT $$SocketStatusConnectRequest
														status = status AND NOT $$SocketStatusAcceptSuccess
														status = status AND NOT $$SocketStatusConnected
														socket[socket].status = status
														IF remote THEN
															status = socket[remote].status
															status = status AND NOT $$SocketStatusConnectSuccess
															status = status AND NOT $$SocketStatusConnectRequest
															status = status AND NOT $$SocketStatusAcceptSuccess
															status = status AND NOT $$SocketStatusConnected
															socket[remote].status = status
														END IF
													END IF
													old = ERROR (error)
													RETURN ($$TRUE)
		END SELECT
	END IF
END FUNCTION
'
'
' ###############################
' #####  XinSocketWrite ()  #####
' ###############################
'
'	/*
'	[XinSocketWrite]
' Description = The XinSocketWrite function sends data on a connected socket.
' Function    = error = XinSocketWrite (socket, block, address, maxbytes, flags, @bytes)
' ArgCount    = 6
' Arg1        = socket : Socket identifier.
' Arg2        = block : Blocking timeout in microseconds. If block = 0, then no blocking. If block is < 10, then block = 10. If block is > 20000 then block is set to a maximum of 20000 (20 msec).
' Arg3				= address : Address of a buffer containing data to be transmitted. 
' Arg4				= maxbytes : Length of buffer.
' Arg5				= flags : Specifies the way in which the call is made, eg, $$MSG_DONTROUTE.
' Arg6				= bytes : Returned number of bytes sent.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = 
'	See Also    = For more details on using this function, see send in the Win32 Programmer's Reference.
'	Examples    = 
'	*/
'
FUNCTION  XinSocketWrite (socket, block, address, maxbytes, flags, bytes)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	FD_SET  readset
	FD_SET  writeset
	FD_SET  errorset
	TIMEVAL  timeval
	AUTOX  length
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketWrite() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	bytes = 0
	error = $$FALSE
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		IF #debug THEN PRINT "XinSocketWrite() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ address THEN
		IF #debug THEN PRINT "XinSocketWrite() : error : (address = 0) : "; socket
		error = ($$ErrorObjectFunction << 8) OR $$ErrorNatureInvalidArgument
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IF (maxbytes <= 0) THEN
		IF #debug THEN PRINT "XinSocketWrite() : error : (maxbytes <= 0) : "; socket
		error = ($$ErrorObjectFunction << 8) OR $$ErrorNatureInvalidArgument
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	status = socket[socket].status
	remote = socket[socket].remote
	IFZ remote THEN remote = socket
	syssocket = socket[socket].syssocket
	sysremote = socket[socket].sysremote
	IF (status AND $$SocketStatusAcceptSuccess) THEN syssocket = sysremote
'
	IFZ syssocket THEN
		IF #debug THEN PRINT "XinSocketWrite() : error : no remote socket : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ (status AND $$SocketStatusConnected) THEN
		IF #debug THEN PRINT "XinSocketWrite() : error : socket not connected : "; socket
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureInvalidRequest
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	SELECT CASE TRUE
		CASE (block < 1)			: block = 0				' do not block
		CASE (block < 10)			: block = 10			' min block = 10us
		CASE (block > 20000)	: block = 20000		' max block = 20ms
	END SELECT
'
	IF (block <= 0) THEN
		timeval.tv_sec = 0				' non-blocking
		timeval.tv_usec = 0				'
	ELSE
		timeval.tv_sec = 0				' blocking up to 20ms maximum
		timeval.tv_usec = block		'
	END IF
'
	FDZERO (@writeset)
	FDZERO (@errorset)
	FDSET (syssocket, @writeset)
	FDSET (syssocket, @errorset)
'
	count = select (syssocket+1, 0, &writeset, &errorset, &timeval)
'
' (count = 0) means no connection attempt yet
'
	IF (count = 0) THEN RETURN ($$FALSE)
'
' (count < 0) means an error occured
'
	IF (count < 0) THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSocketWrite() : error : select() reported error : "; socket; errno
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
		socket[socket].status = status OR $$SocketStatusFailed
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
' (count > 0) means something was detected
'
' see if the socket or network died prior to accept success
'
	hit = FDISSET (syssocket, @errorset)
	IF hit THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSocketWrite() : error : select() reported error : "; socket; errno
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
		socket[socket].status = status OR $$SocketStatusFailed
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
' see if select() indicates bytes can be written
'
	hit = FDISSET (syssocket, @writeset)
	IF hit THEN
		bytes = send (syssocket, address, maxbytes, flags)
'
		SELECT CASE TRUE
			CASE (bytes > 0)	: address = address + bytes
			CASE (bytes = 0)	: error = ($$ErrorObjectSocket << 8) + $$ErrorNatureDisconnected
													status = status AND NOT $$SocketStatusConnectSuccess
													status = status AND NOT $$SocketStatusConnectRequest
													status = status AND NOT $$SocketStatusAcceptSuccess
													status = status AND NOT $$SocketStatusConnected
													socket[socket].status = status
													IF remote THEN
														status = socket[remote].status
														status = status AND NOT $$SocketStatusConnectSuccess
														status = status AND NOT $$SocketStatusConnectRequest
														status = status AND NOT $$SocketStatusAcceptSuccess
														status = status AND NOT $$SocketStatusConnected
														socket[remote].status = status
													END IF
													old = ERROR (error)
													RETURN ($$TRUE)
			CASE (bytes < 0)	: errno = GetLastError ()
													connected = $$TRUE
													error = $$TRUE
													bytes = 0
													SELECT CASE errno
														CASE $$WSAEWOULDBLOCK
														CASE $$WSAECONNABORTED	: connected = $$FALSE
														CASE $$WSAECONNRESET		: connected = $$FALSE
														CASE $$WSAESHUTDOWN			: connected = $$FALSE
														CASE $$WSAENOTCONN			: connected = $$FALSE
														CASE $$WSAENETDOWN			: connected = $$FALSE
														CASE ELSE								: connected = $$FALSE
													END SELECT
													IF #debug THEN PRINT "XinSocketWrite() : error : recv() reported error : "; socket; errno
													IFZ connected THEN
														error = ($$ErrorObjectSocket << 8) + $$ErrorNatureDisconnected
														status = status AND NOT $$SocketStatusConnectSuccess
														status = status AND NOT $$SocketStatusConnectRequest
														status = status AND NOT $$SocketStatusAcceptSuccess
														status = status AND NOT $$SocketStatusConnected
														socket[socket].status = status
														IF remote THEN
															status = socket[remote].status
															status = status AND NOT $$SocketStatusConnectSuccess
															status = status AND NOT $$SocketStatusConnectRequest
															status = status AND NOT $$SocketStatusAcceptSuccess
															status = status AND NOT $$SocketStatusConnected
															socket[remote].status = status
														END IF
													END IF
													old = ERROR (error)
													RETURN ($$TRUE)
		END SELECT
	END IF
END FUNCTION
'
'
' ###############################
' #####  XinSocketClose ()  #####
' ###############################
'
'	/*
'	[XinSocketClose]
' Description = The XinSocketClose function closes a socket identified by socket.
' Function    = error = XinSocketClose (socket)
' ArgCount    = 1
' Arg1        = socket : An index to a XSOCKET socket[] array.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = If socket value is -1, then XinSocketClose will close all open sockets.
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinSocketClose (socket)
	SHARED  initialized
	SHARED  XSOCKET  socket[]
	XSOCKET  zero
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinSocketClose() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	upper = UBOUND (socket[])
'
' socket = -1 means "close all my sockets"
'
	IF (socket = -1) THEN
		FOR i = 0 TO upper
			socket = socket[i].socket
			syssocket = socket[i].syssocket
			IF socket THEN
				IF syssocket THEN
					IF #debug THEN PRINT "XinSocketClose(-1) : "; socket; syssocket
					closesocket (syssocket)
					socket[i] = zero
				END IF
			END IF
		NEXT i
		socket = -1
		RETURN ($$FALSE)
	END IF
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		IF #debug THEN PRINT "XinSocketClose() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	syssocket = socket[socket].syssocket
'
	IF (syssocket <= 0) THEN
		IF #debug THEN PRINT "XinSocketClose() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	error = closesocket (syssocket)
'
	IF #debug THEN PRINT "XinSocketClose() : "; socket; syssocket
	socket[socket] = zero
'
	FOR i = 1 TO upper
		IF (socket = socket[i].remote) THEN
			socket[i].status = socket[i].status AND NOT $$SocketStatusConnected
			socket[i].remote = 0
		END IF
	NEXT i
END FUNCTION
'
'
' ############################
' #####  XinSetDebug ()  #####
' ############################
'
'	/*
'	[XinSetDebug]
' Description = The XinSetDebug function sets #debug variable. If state is $$TRUE then all debugging statements are displayed.
' Function    = XinSetDebug (state)
' ArgCount    = 1
' Arg1        = state : Debug state, set to $$TRUE or $$FALSE.
'	Return      = 
' Remarks     = 
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION  XinSetDebug (state)
'
	#debug = state
END FUNCTION
'
'
' ###########################
' #####  XinCleanup ()  #####
' ###########################
'
FUNCTION  XinCleanup ()
	SHARED  initialized
	SHARED  HOST  host[]
	SHARED  XSOCKET  socket[]
	XSOCKET  zero
'
	IFZ initialized THEN									' Xin() never called
		IF #debug THEN PRINT "XinCleanup() : error : Xin() library not initialized : call Xin() first"
		error = ($$ErrorObjectLibrary << 8) OR $$ErrorNatureNotInitialized
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF

	upper = UBOUND (socket[])
'
	IF #debug THEN PRINT "xin.x : XinCleanup() : upper : "; upper
'
	FOR i = 1 TO upper
		socket = socket[i].socket
		syssocket = socket[i].syssocket
		IF socket THEN
			IF syssocket THEN
				closesocket (syssocket)
				socket[i] = zero
				IF #debug THEN PRINT "#####  closesocket ("; STRING$(syssocket); ")  #####"
			END IF
		END IF
	NEXT i
'
	WSACleanup ()						' WSACleanup() is in Windows only
'
	initialized = $$FALSE
END FUNCTION
'
' #############################
' #####  GetLastError ()  #####
' #############################
'
FUNCTION  GetLastError ()
'
	errno = WSAGetLastError ()
'
	IF errno THEN XstSetSystemError (errno)
	RETURN (errno)
END FUNCTION
'
'
' ########################
' #####  AddHost ()  #####
' ########################
'
FUNCTION  AddHost (host, addr, host$)
	SHARED  HOST  host[]
	PROTOENT  protoent
	HOSTENT  hostent
	HOST  hhhh
'
	host = -1
	upper = UBOUND (host[])
'
' see if host name is recognized
'
	IF host$ THEN
		hostent = gethostbyname (&host$)
		IF hostent.h_length THEN addr = 0
	END IF
'
' see if host number is recognized
'
	IF addr THEN
		IFZ hostent.h_length THEN
			host$ = ""
			length = 4
			address = addr
			family = $$AF_INET
			hostent = gethostbyaddr (&address, length, family)
		END IF
	END IF
'
	IFZ hostent THEN host$ = "" : addr = 0 : RETURN ($$TRUE)
	host$ = CSTRING$ (hostent.h_name)
	aaaa = hostent.h_addr_list
'
' the following lines are suspicious	' ??????????????????????
'
	IF aaaa THEN
'		  Check if aaaa has a value else XLONGAT causes segment error when host does not exists
'		  The actual question is:does aaaa needs to be set twice here?(it is not referenced before the second reset)
'		  and why?
		aaaa = XLONGAT (aaaa)
		aaaa = XLONGAT (aaaa)
	END IF
'
'
' see if host is already in host[]
'
	FOR i = 0 TO upper
		name$ = host[i].name
		alias0$ = host[i].alias[0]
		alias1$ = host[i].alias[1]
		alias2$ = host[i].alias[2]
		address = host[i].address
		IF (host$ = name$) THEN host = i : EXIT FOR
		IF (host$ = alias0$) THEN host = i : EXIT FOR
		IF (host$ = alias1$) THEN host = i : EXIT FOR
		IF (host$ = alias2$) THEN host = i : EXIT FOR
		IF (addr = address) THEN host = i : EXIT FOR
	NEXT i
'
	IF (host >= 0) THEN RETURN ($$FALSE)		' host already in host[]
'
'
' *****  make new slot in host[] for this new host  *****
'
	INC upper
	host = upper
'
	REDIM host[host]
'
	hosts = upper + 1
	host$ = CSTRING$ (hostent.h_name)
'
'
' *****  add alias names to host[].alias[] array  *****
'
	alias = 0
	host[host].name = host$
'
	IF hostent.h_aliases THEN
		aaaa = XLONGAT (hostent.h_aliases)
		DO
			IFZ aaaa THEN EXIT DO
'			at = XLONGAT (aaaa)
'			name$ = CSTRING$ (at)
			name$ = CSTRING$ (aaaa)
			name$ = TRIM$ (name$)
			aaaa = aaaa + 4
			IF name$ THEN
				host[host].alias[alias] = name$
				IF (alias < 2) THEN INC alias ELSE EXIT DO
			END IF
		LOOP
	END IF
'
'
' *****  add additional IP addresses to host[].address[] array  *****
'
	n = 0
	aaaa = hostent.h_addr_list
	IF aaaa THEN item = XLONGAT (aaaa)
	IF item THEN ipaddr = XLONGAT (item)
	IF ipaddr THEN host[host].address = ipaddr : aaaa = aaaa + 4
'
	IF aaaa THEN
		DO
			item = XLONGAT (aaaa)
			IFZ item THEN EXIT DO
			data = XLONGAT (item)
			IF data THEN
				host[host].addresses[n] = data
				IF (n >= 7) THEN EXIT DO
				INC n
			END IF
			aaaa = aaaa + 4
		LOOP
	END IF
'
'
' *****  get protocol number for "TCP"  *****
'
	protoent.p_name = 0
	protoent.p_aliases = 0
	protoent.p_proto = 0
'
	protoent = getprotobyname (&"tcp")
'
	IFZ protoent THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "getprotobyname() : error : "; errno
		error = ($$ErrorObjectNetwork << 8) + $$ErrorNatureUnavailable
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	XstGetOSName (@system$)
	system$ = TRIM$ (system$)
	protocol$ = CSTRING$ (protoent.p_name)
'
	host[host].protocol = protoent.p_proto
	host[host].system = system$
	host[host].addressBytes = 4
	host[host].addressFamily = $$AF_INET
	host[host].protocolFamily = $$PF_INET
	RETURN ($$FALSE)
END FUNCTION
'
'
' ##################################
' #####  SetSocketBlocking ()  #####
' ##################################
'
FUNCTION  SetSocketBlocking (socket)
	SHARED  XSOCKET  socket[]
'
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		IF #debug THEN PRINT "SetSocketBlocking() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ socket[socket].socket THEN
		IF #debug THEN PRINT "SetSocketBlocking() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	syssocket = socket[socket].syssocket
'
	value = 0
	command = $$FIONBIO
'
	error = ioctlsocket (syssocket, command, &value)
'
	IF error THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "SetSocketBlocking() : error : attempt to set socket blocking failed : "; socket; errno
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
		connected = $$TRUE
'
		SELECT CASE errno
			CASE $$WSAEWOULDBLOCK
			CASE $$WSAECONNABORTED	: connected = $$FALSE
			CASE $$WSAECONNRESET		: connected = $$FALSE
			CASE $$WSAESHUTDOWN			: connected = $$FALSE
			CASE $$WSAENOTCONN			: connected = $$FALSE
			CASE $$WSAENETDOWN			: connected = $$FALSE
			CASE ELSE								:
		END SELECT
'
		IFZ connected THEN
			error = ($$ErrorObjectSocket << 8) + $$ErrorNatureDisconnected
			status = status AND NOT $$SocketStatusConnected
			status = status OR $$SocketStatusFailed
			socket[socket].status = status
			old = ERROR (error)
			RETURN (errno)
		ELSE
			error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
			socket[socket].status = status OR $$SocketStatusFailed
			old = ERROR (error)
			RETURN (errno)
		END IF
	END IF
'
	RETURN ($$FALSE)
END FUNCTION
'
'
' #####################################
' #####  SetSocketNonBlocking ()  #####
' #####################################
'
FUNCTION  SetSocketNonBlocking (socket)
	SHARED  XSOCKET  socket[]
'
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		IF #debug THEN PRINT "SetSocketNonBlocking() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ socket[socket].socket THEN
		IF #debug THEN PRINT "SetSocketNonBlocking() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	syssocket = socket[socket].syssocket
'
	IFZ syssocket THEN
		IF #debug THEN PRINT "SetSocketNonBlocking() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	value = 1
	command = $$FIONBIO
'
	error = ioctlsocket (syssocket, command, &value)
'
	IF error THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "SetSocketNonBlocking() : error : attempt to set socket non-blocking failed : "; socket; errno
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
		connected = $$TRUE
'
		SELECT CASE errno
			CASE $$WSAEWOULDBLOCK
			CASE $$WSAECONNABORTED	: connected = $$FALSE
			CASE $$WSAECONNRESET		: connected = $$FALSE
			CASE $$WSAESHUTDOWN			: connected = $$FALSE
			CASE $$WSAENOTCONN			: connected = $$FALSE
			CASE $$WSAENETDOWN			: connected = $$FALSE
			CASE ELSE								:
		END SELECT
'
		IFZ connected THEN
			error = ($$ErrorObjectSocket << 8) + $$ErrorNatureDisconnected
			status = status AND NOT $$SocketStatusConnected
			status = status OR $$SocketStatusFailed
			socket[socket].status = status
			old = ERROR (error)
			RETURN (errno)
		ELSE
			error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
			socket[socket].status = status OR $$SocketStatusFailed
			old = ERROR (error)
			RETURN (errno)
		END IF
	END IF
'
	RETURN ($$FALSE)
END FUNCTION
'
'
' ###################################
' #####  SystemErrorToError ()  #####
' ###################################
'
FUNCTION  SystemErrorToError (errno, error)
'
	SELECT CASE errno
		CASE $$WSAEINTR							:
		CASE $$WSAEBADF							:
		CASE $$WSAEACCES						:
		CASE $$WSAEFAULT						:
		CASE $$WSAEINVAL						: error = ($$ErrorObjectNetwork << 8) + $$ErrorNatureInvalidVersion
		CASE $$WSAEMFILE						:
		CASE $$WSAEWOULDBLOCK				:
		CASE $$WSAEINPROGRESS				:
		CASE $$WSAENOTSOCK					:
		CASE $$WSAEDESTADDRREQ			:
		CASE $$WSAEMSGSIZE					:
		CASE $$WSAEPROTOTYPE				:
		CASE $$WSAENOPROTOOPT				:
		CASE $$WSAEPROTONOSUPPORT		:
		CASE $$WSAESOCKTNOSUPPORT		:
		CASE $$WSAEOPNOTSUPP				:
		CASE $$WSAEPFNOSUPPORT			:
		CASE $$WSAEAFNOSUPPORT			:
		CASE $$WSAEADDRINUSE				:
		CASE $$WSAEADDRNOTAVAIL			:
		CASE $$WSAENETDOWN					:
		CASE $$WSAENETUNREACH				:
		CASE $$WSAENETRESET					:
		CASE $$WSAECONNABORTED			:
		CASE $$WSAECONNRESET				:
		CASE $$WSAENOBUFS						:
		CASE $$WSAEISCONN						:
		CASE $$WSAENOTCONN					:
		CASE $$WSAESHUTDOWN					:
		CASE $$WSAETOOMANYREFS			:
		CASE $$WSAETIMEDOUT					:
		CASE $$WSAECONNREFUSED			:
		CASE $$WSAELOOP							:
		CASE $$WSAENAMETOOLONG			:
		CASE $$WSAEHOSTDOWN					:
		CASE $$WSAEHOSTUNREACH			:
		CASE $$WSAENOTEMPTY					:
		CASE $$WSAEPROCLIM					:
		CASE $$WSAEUSERS						:
		CASE $$WSAEDQUOT						:
		CASE $$WSAESTALE						:
		CASE $$WSAEREMOTE						:
		CASE $$WSAESYSNOTREADY			: error = ($$ErrorObjectNetwork << 8) + $$ErrorNatureUnavailable
		CASE $$WSAEVERNOTSUPPORTED	: error = ($$ErrorObjectNetwork << 8) + $$ErrorNatureInvalidVersion
		CASE $$WSAENOTINITILISED		:
		CASE $$WSAENOTINITILIZED		:
		CASE $$WSAEDISCON						:
		CASE $$WSAHOST_NOT_FOUND		:
		CASE $$WSATRY_AGAIN					:
		CASE $$WSANO_RECOVERY				:
		CASE $$WSANO_DATA						:
		CASE $$WSANO_ADDRESS				:
		CASE ELSE										: error = ($$ErrorObjectNetwork << 8) + $$ErrorNatureUnknown
	END SELECT
END FUNCTION
'
'
' ######################
' #####  FDCLR ()  #####
' ######################
'
FUNCTION  FDCLR (fildes, FD_SET fd_set)

	count = 0
	error = $$TRUE
'
	FOR i = 0 TO 63
		now = fd_set.fd_array[i]
		IF now THEN
			IF (now = fildes) THEN
				fd_set.fd_array[i] = 0
				error = $$FALSE					' fildes was in fd_set
			ELSE
				INC count
			END IF
		END IF
	NEXT i
'
	fd_set.fd_count = count
	FDCOUNT (fd_set)
	RETURN (error)
END FUNCTION
'
'
' ######################
' #####  FDSET ()  #####
' ######################
'
FUNCTION  FDSET (fildes, FD_SET fd_set)

	slot = -1
	count = 0
	error = $$FALSE
'
	FOR i = 0 TO 63
		now = fd_set.fd_array[i]
		IFZ now THEN
			IF (slot < 0) THEN slot = i				' first open slot
		ELSE
			IF (now = fildes) THEN						' fildes already in fd_set
				DEC count												' because we count it again
				slot = i												' put it back in here again
			END IF
			INC count
		END IF
	NEXT i
'
	IF (slot >= 0) THEN
		fd_set.fd_array[slot] = fildes
		INC count
	ELSE
		error = $$TRUE											' fd_set.array[] is full
	END IF
'
	fd_set.fd_count = count
	FDCOUNT (fd_set)
	RETURN (error)
END FUNCTION
'
'
' ########################
' #####  FDISSET ()  #####
' ########################
'
FUNCTION  FDISSET (fildes, FD_SET fd_set)

	return = 0
	count = fd_set.fd_count
	IF (count <= 0) THEN RETURN ($$FALSE)		' nothing in fd_set
'
	check = 0
	FOR i = 0 TO 63
		now = fd_set.fd_array[i]
		IF now THEN
			INC check
			IF (now = fildes) THEN
				return = $$TRUE
				EXIT FOR
			END IF
			IF (check >= count) THEN
				return = $$FALSE
				EXIT FOR
			END IF
		END IF
	NEXT i
'
	RETURN (return)
END FUNCTION
'
'
' #######################
' #####  FDZERO ()  #####
' #######################
'
FUNCTION  FDZERO (FD_SET fd_set)
'
	upper = 63
'
	FOR i = 0 TO upper
		fd_set.fd_array[i] = 0
	NEXT i
	fd_set.fd_count = 0
END FUNCTION
'
'
' ########################
' #####  FDCOUNT ()  #####
' ########################
'
FUNCTION  FDCOUNT (FD_SET fd_set)
'
	count = 0
	upper = 63
'
	FOR i = 0 TO upper
		entry = fd_set.fd_array[i]
		IF entry THEN INC count
	NEXT i
	fd_set.fd_count = count
END FUNCTION
'
' ################################
' #####  XinSetSocketOption  #####
' ################################
'
'	/*
'	[XinSetSocketOption]
' Description = The XinSetSocketOption function sets a socket option.
' Function    = error = XinSetSocketOption (socket, level, optname, optval, optlen)
' ArgCount    = 5
' Arg1        = socket : Socket identifier.
' Arg2        = level : The level at which the option is defined; the supported levels include $$SOL_SOCKET and $$IPPROTO_TCP. 
' Arg3				= optname : The socket option for which the value is to be set. 
' Arg4				= optval : A pointer to the buffer in which the value for the requested option is supplied. 
' Arg5				= opelen : The size of the optval buffer.
'	Return      = If the function succeeds, the return value is 0. If the function fails, the return value is $$TRUE (-1). To get extended error information, call ERROR().
' Remarks     = For more details on using this function, see setsockopt in the Win32 Programmer's Reference.
'	See Also    = 
'	Examples    = 
'	*/
'
FUNCTION XinSetSocketOption (socket, level, optname, optval, optlen)

	SHARED  XSOCKET  socket[]
'
	upper = UBOUND (socket[])
'
	IF ((socket <= 0) OR (socket > upper)) THEN
		IF #debug THEN PRINT "XinSetSocketOption() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	IFZ socket[socket].socket THEN
		IF #debug THEN PRINT "XinSetSocketOption() : error : undefined socket # : "; socket
		error = ($$ErrorObjectSocket << 8) OR $$ErrorNatureUndefined
		old = ERROR (error)
		RETURN ($$TRUE)
	END IF
'
	syssocket = socket[socket].syssocket
	
	error = setsockopt (syssocket, level, optname, optval, optlen)
	
	IF error THEN
		errno = GetLastError ()
		IF #debug THEN PRINT "XinSetSocketOption() : error : attempt to set socket option failed : "; socket; errno
		error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
		connected = $$TRUE
'
		SELECT CASE errno
			CASE $$WSAEWOULDBLOCK
			CASE $$WSAECONNABORTED	: connected = $$FALSE
			CASE $$WSAECONNRESET		: connected = $$FALSE
			CASE $$WSAESHUTDOWN			: connected = $$FALSE
			CASE $$WSAENOTCONN			: connected = $$FALSE
			CASE $$WSAENETDOWN			: connected = $$FALSE
			CASE ELSE								:
		END SELECT
'
		IFZ connected THEN
			error = ($$ErrorObjectSocket << 8) + $$ErrorNatureDisconnected
			status = status AND NOT $$SocketStatusConnected
			status = status OR $$SocketStatusFailed
			socket[socket].status = status
			old = ERROR (error)
			RETURN (errno)
		ELSE
			error = ($$ErrorObjectSocket << 8) + $$ErrorNatureFailed
			socket[socket].status = status OR $$SocketStatusFailed
			old = ERROR (error)
			RETURN (errno)
		END IF
	END IF
'
	RETURN ($$FALSE)

END FUNCTION

END PROGRAM
