/*****************************************************************************
File    : PDFChart.h
Author  : Utility Warrior, (c) 2005
Project : PDF Chart Creator (PDFChart) Dynamic Link Library
Purpose : This file provides C prototypes for all DLL functions

Notes  : a) #include this header file at the top of any C code that references
            any of these DLL functions
		 b) Link with PDFChart.lib which provides the necessary linkage to
		    automatically generate the correct DLL call for the function, plus
			it will ensure that the PDFChart.DLL file is loaded when the
			calling application is run (NB there is no need to use LoadLibrary)
		 c) Ensure that PDFChart.DLL is either in the same directory as the
		    calling application or on the system path

Please refer to the accompanying documentation for full details.
******************************************************************************/


#ifndef _PDFChart_h
#define _PDFChart_h

/*****************************************************************************
   This allows the support of multiple versions of a DLL (if and when
   available) so that functions that might only be available in a later version
   can then only be called if the returned version currently being used is one 
   that supports those functions.
   
   Return value: integer value representing the DLL version, a value of 100 is
                 equivalent to version 1.00
******************************************************************************/
extern __declspec(dllimport) UINT PDFC_GetDLLVersion(void);


/*****************************************************************************
Required
******************************************************************************/
extern __declspec(dllimport) void PDFC_License(char *code);
/* This is required to remove the "Evaluation Mode" message that is stamped 
   over the top of every page in the output PDF document.   
   
   To purchase a license code please visit the Utility Warrior web site which 
   has the latest information and prices on the licenses that are available. 

   Return value: none 
*/

#define CHART_BAR_VERTICAL		1
#define CHART_BAR_HORIZONTAL	2
extern __declspec(dllimport) UINT PDFC_Chart(UINT type);
/* Specify which chart type is to be generated.

   Return value:
   0 - success
   1 - invalid chart type
*/

extern __declspec(dllimport) UINT PDFC_Page(UINT width, UINT height, double red, double green, double blue);
/* Specify page size and background color

   Width and height must be between 100 and 10000 inclusive

   Each color component (red, green, blue) value must be between 0.0 and 1.0 inclusive.

   Return value:
   0 - success
   1 - width invalid
   2 - height invalid
   3 - red invalid
   4 - green invalid
   5 - blue invalid
*/

extern __declspec(dllimport) UINT PDFC_DataSet(char *filename);
/* Load in chart data from an external file - either this and/or an appropriate chart
   type datum function must be used in order to define at least one item of data

   Up to 1000 data items can be added.

   Return value:
   0 - success
   1 - no chart type specified
   2 - could not open filename
   3 - maximum number of data items reached
   101...
     - line (101 - 100), ie line 1 contains invalid data
*/

extern __declspec(dllimport) UINT PDFC_DataSeparator(char *separator);
/* Set the data separator text that is used to delineate data fields when loading in data 
   from an external file, by default this is a comma

   A maximum of 10 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/


/*****************************************************************************
Bar Chart Specific
******************************************************************************/
extern __declspec(dllimport) UINT PDFC_BarDatum(char *label, double value);
/* Add an item of bar data.

   Up to 1000 data items can be added.

   Return value:
   0 - success
   1 - bar chart type not specified
   2 - maximum number of data items already reached
   3 - label invalid
*/

extern __declspec(dllimport) UINT PDFC_BarColor(double fromRed, double fromGreen, double fromBlue, double toRed, double toGreen, double toBlue);
/* Sets a color (range) for a bar

   Up to 50 bar colors can be defined

   Each color component (red, green, blue) value must be between 0.0 and 1.0 inclusive.

   Return value:
   0 - success
   1 - maximum number of bar colors already specified
   2 - from red invalid
   3 - from green invalid
   4 - from blue invalid
   5 - to red invalid
   6 - to green invalid
   7 - to blue invalid
*/

extern __declspec(dllimport) UINT PDFC_BarSpacing(UINT spacing);
/* Sets the amount of spacing between bars (bar size automatically calculated from this)

   Return value:
   0 - success
   1 - invalid spacing, must be between 0 and 500 inclusive
*/

extern __declspec(dllimport) UINT PDFC_BarSize(UINT size);
/* Sets the bar size (bar spacing automatically calculated from this)

   Return value:
   0 - success
   1 - invalid size, must be between 1 and 500 inclusive
*/

#define LABEL_AT_START						1
#define LABEL_AT_END						2
#define LABEL_AT_CENTER						3
#define LABEL_AT_AXES						4
#define LABEL_AT_AXES_AND_START				5
#define LABEL_AT_AXES_AND_END				6
#define LABEL_AT_AXES_AND_CENTER			7
extern __declspec(dllimport) UINT PDFC_BarLabel(UINT labelPosition, char *format);
/* Set the position and label style

   Return value:
   0 - success
   1 - label position not valid
   2 - invalid format
*/

extern __declspec(dllimport) UINT PDFC_BarBorder(double width, double red, double green, double blue);
/* Set the bar border width and color

   width must be > 0.0 and <= 100.0

   Each color component (red, green, blue) value must be between 0.0 and 1.0 inclusive.

   0 - success
   1 - width invalid
   2 - red invalid
   3 - green invalid
   4 - blue invalid
*/


/*****************************************************************************
Axes and Labels
******************************************************************************/
extern __declspec(dllimport) UINT PDFC_ChartBackground(double fromRed, double fromGreen, double fromBlue, double toRed, double toGreen, double toBlue);
/* Sets the background color (range) for the chart

   Each color component (red, green, blue) value must be between 0.0 and 1.0 inclusive.

   Return value:
   0 - success
   1 - from red invalid
   2 - from green invalid
   3 - from blue invalid
   4 - to red invalid
   5 - to green invalid
   6 - to blue invalid
*/

#define VERTICAL_ALIGNMENT_TOP				1
#define VERTICAL_ALIGNMENT_MIDDLE			2
#define VERTICAL_ALIGNMENT_BOTTOM			3

#define HORIZONTAL_ALIGNMENT_LEFT			1
#define HORIZONTAL_ALIGNMENT_CENTER			2
#define HORIZONTAL_ALIGNMENT_RIGHT			3
extern __declspec(dllimport) UINT PDFC_ChartHeading(UINT verticalAlignment, UINT horizontalAlignment);
/* Sets the chart heading position

   vertical alignment must be either top or bottom (NOT middle)

   Return value:
   0 - success
   1 - vertical alignment invalid
   2 - horizontal alignment invalid
*/

#define FONT_ID_COURIER						1
#define FONT_ID_COURIER_BOLD				2
#define FONT_ID_COURIER_BOLDOBLIQUE			3
#define FONT_ID_COURIER_OBLIQUE				4
#define FONT_ID_HELVETICA					5
#define FONT_ID_HELVETICA_BOLD				6
#define FONT_ID_HELVETICA_BOLDOBLIQUE		7
#define FONT_ID_HELVETICA_OBLIQUE			8
#define FONT_ID_TIMES_ROMAN					9
#define FONT_ID_TIMES_BOLD					10
#define FONT_ID_TIMES_ITALIC				11
#define FONT_ID_TIMES_BOLDITALIC			12
#define FONT_ID_SYMBOL						13
#define FONT_ID_ZAPFDINGBATS				14

#define TEXT_STYLE_NORMAL					1
#define TEXT_STYLE_STROKE					2
extern __declspec(dllimport) UINT PDFC_HeadingTitle(   UINT size, UINT fontID, double fillRed, double fillGreen, double fillBlue, UINT style, double otherRed, double otherGreen, double otherBlue);
extern __declspec(dllimport) UINT PDFC_HeadingSubTitle(UINT size, UINT fontID, double fillRed, double fillGreen, double fillBlue, UINT style, double otherRed, double otherGreen, double otherBlue);
extern __declspec(dllimport) UINT PDFC_AxesLabelX(     UINT size, UINT fontID, double fillRed, double fillGreen, double fillBlue, UINT style, double otherRed, double otherGreen, double otherBlue);
extern __declspec(dllimport) UINT PDFC_AxesLabelY(     UINT size, UINT fontID, double fillRed, double fillGreen, double fillBlue, UINT style, double otherRed, double otherGreen, double otherBlue);
/* Sets the appropriate text style

   size must be between 1 and 300 inclusive

   Each color component (red, green, blue) value must be between 0.0 and 1.0 inclusive

   other color is only required if the style is stroke

   Return value:
   0 - success
   1 - invalid size
   2 - invalid font 
   3 - fill red invalid
   4 - fill green invalid
   5 - fill blue invalid
   6 - invalid style
   7 - other red invalid
   8 - other green invalid
   9 - other blue invalid
*/

extern __declspec(dllimport) UINT PDFC_AxesCaptionX(UINT size, UINT fontID, double fillRed, double fillGreen, double fillBlue, UINT style, double otherRed, double otherGreen, double otherBlue, UINT horizontalAlignment, char *label);
extern __declspec(dllimport) UINT PDFC_AxesCaptionY(UINT size, UINT fontID, double fillRed, double fillGreen, double fillBlue, UINT style, double otherRed, double otherGreen, double otherBlue, UINT verticalAlignment, char *label);
/* Sets the appropriate axes label style and text

   size must be between 1 and 300 inclusive

   Each color component (red, green, blue) value must be between 0.0 and 1.0 inclusive

   other color is only required if the style is stroke

   Return value:
   0 - success
   1 - invalid size
   2 - invalid font 
   3 - fill red invalid
   4 - fill green invalid
   5 - fill blue invalid
   6 - invalid style
   7 - other red invalid
   8 - other green invalid
   9 - other blue invalid
   10 - invalid horizontal/vertical alignment
   11 - label is invalid
*/

#define AXES_LINE_XY		1
#define AXES_LINE_FULL		2
extern __declspec(dllimport) UINT PDFC_AxesLine(double width, double red, double green, double blue, UINT format);
/* Sets the width, color and style for the axes boundary lines surrounding the chart area

   width must be between 0.1 and 100.0 inclusive

   Each color component (red, green, blue) value must be between 0.0 and 1.0 inclusive

   Return value:
   0 - success
   1 - invalid width
   2 - red invalid
   3 - green invalid
   4 - blue invalid
   5 - invalid line format
*/

#define AXES_SCALE_FROM_ZERO				1
#define AXES_SCALE_UNDER_LOWEST_VALUE		2
#define AXES_SCALE_RANGE					3
extern __declspec(dllimport) UINT PDFC_AxesScale(UINT scale, double value1, double value2);
/* Sets the width, color and style for the axes boundary lines surrounding the chart area

   if scale is AXES_SCALE_FROM_ZERO then value1 and value2 are not used
   if scale is AXES_SCALE_UNDER_LOWEST_VALUE then value1 is an integer percentage from 0 to 100%, value2 is not used
   if scale is AXES_SCALE_RANGE then the axes range is from value1 to value2

   Return value:
   0 - success
   1 - invalid scale type
   2 - invalid percentage
*/

extern __declspec(dllimport) UINT PDFC_AxesDivisions(UINT divisions);
/* Sets the number of divisions in the axes scale

   divisions must be between 1 and 100 inclusive

   Return value:
   0 - success
   1 - invalid divisions
*/

extern __declspec(dllimport) UINT PDFC_DecimalPoints(UINT points);
/* Sets the number of decimal points to be displayed with a real number

   points must be between 0 and 9 inclusive

   Return value:
   0 - success
   1 - invalid points
*/

extern __declspec(dllimport) UINT PDFC_AxesGridLineStyle(UINT dash, UINT gap, UINT offset);
/* Sets the dashed line style for the grid lines

   all parameters must be between 0 and 100 inclusive

   Return value:
   0 - success
   1 - invalid dash
   2 - invalid gap
   3 - invalid offset
*/

extern __declspec(dllimport) UINT PDFC_AxesGridLine(double width, double red, double green, double blue);
/* Sets the width and color of the axes grid lines that are drawn within the chart area

   width must be between 0.1 and 100.0 inclusive

   Each color component (red, green, blue) value must be between 0.0 and 1.0 inclusive

   Return value:
   0 - success
   1 - invalid width
   2 - red invalid
   3 - green invalid
   4 - blue invalid
*/


/*****************************************************************************
PDF Settings
******************************************************************************/
extern __declspec(dllimport) void PDFC_SetPermissionEdit(void);
/* Allows editing/changing of the PDF document but NOT adding or changing of text 
   notes and AcroForm fields.

   Return value: none 
*/

extern __declspec(dllimport) void PDFC_SetPermissionEditExtra(void);
/* Allows the adding and changing of text notes and AcroForm fields within a PDF document.

   Return value: none 
*/

extern __declspec(dllimport) void PDFC_SetPermissionCopy(void);
/* Allows the copying of text and graphics from the PDF document.

   Return value: none 
*/

extern __declspec(dllimport) void PDFC_SetPermissionPrint(void);
/* Allows printing of the PDF document.

   Return value: none 
*/

extern __declspec(dllimport) UINT PDFC_SetOwnerPassword(char *password);
/* The owner of a PDF document would enter this password in order to view it
   and make changes and perform actions that may be restricted to a normal user.

   A maximum of 32 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT PDFC_SetUserPassword(char *password);
/* In order that only the intended user can view the PDF document, they must type
   in this password, otherwise the user is not allowed to read the PDF document.

   A maximum of 32 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/


/*****************************************************************************
PDF Meta Information
******************************************************************************/
extern __declspec(dllimport) UINT PDFC_SetProducer(char *producer);
/* The name of the application that converted the document from its native format to PDF.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT PDFC_SetCreator(char *creator);
/* If the document was converted into a PDF document from another form, this is 
   usually the name of the application that created the original document.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT PDFC_SetAuthor(char *author);
/* The name of the person who created the PDF document.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT PDFC_SetTitle(char *title);
/* The PDF document's title.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT PDFC_SetSubject(char *subject);
/* The PDF document's subject.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) UINT PDFC_SetKeywords(char *keywords);
/* A list of keywords associated with the PDF document.

   A maximum of 256 characters can be used.

   Return value:
   0 - success
   1 - invalid parameter
*/

extern __declspec(dllimport) void PDFC_SetCreationDate(void);
/* The date that the PDF document was created.

   This is set using the current local time of the PC.

   Return value: none
*/


/*****************************************************************************
PDF Viewer Preferences
******************************************************************************/
extern __declspec(dllimport) void PDFC_SetViewerPreferenceCenterWindow(void);
/* The window displaying the PDF document should be displayed in the center of 
   the computer's screen.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetViewerPreferenceHideToolbar(void);
/* The PDF viewer's toolbar should be hidden when the PDF document is active.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetViewerPreferenceHideMenubar(void);
/* The PDF viewer's menubar should be hidden when the PDF document is active.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetViewerPreferenceHideWindowUI(void);
/* The user interface elements within the PDF document's window should be hidden.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetViewerPreferenceFitWindow(void);
/* The PDF viewer should resize the window displaying the PDF document to fit the 
   size of the first displayed page of the PDF document.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetPageModeOutlines(void);
/* The PDF viewer should show the outline (ie bookmark) navigation item when the 
   PDF document is open.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetPageModeNone(void);
/* The PDF viewer should not show the outline (ie bookmark) or thumbnail navigation
   items when the PDF document is open.

   This is the default if no other page mode is specified.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetPageModeUseThumbs(void);
/* The PDF viewer should show the thumbnail navigation item when the PDF document 
   is open.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetPageModeFullScreen(void);
/* The PDF viewer should open the PDF document in full-screen mode without showing 
   the menu bar, window controls or any other window.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetPageLayoutSinglePage(void);
/* Display the pages one page at a time.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetPageLayoutOneColumn(void);
/* Display the pages in one column.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetPageLayoutTwoColumnLeft(void);
/* Display the pages in two columns, with odd-numbered pages on the left.

   Return value: none
*/

extern __declspec(dllimport) void PDFC_SetPageLayoutTwoColumnRight(void);
/* Display the pages in two columns, with odd-numbered pages on the right.

   Return value: none
*/


/*****************************************************************************
Specifies the name and location of the produced PDF document, 
******************************************************************************/
extern __declspec(dllimport) UINT PDFC_MakePDF(char *output, BOOL openPDF, BOOL reset, char *errorText, UINT maxErrorTextSize);
/* Generate the output PDF according to the settings already selected through
   the Application Programming Interface functions above.

   To have the output PDF open in the default PDF viewer set the openPDF
   parameter to TRUE instead of FALSE.

   To have all settings applied reset after PDF has been generated set the reset
   parameter to TRUE instead of FALSE (does NOT reset the License code).

   Return value:
   0 - success
   1 - internal error - see errorText for details (unless NULL passed)
   2 - output pdf could not be opened
   3 - chart type not specified
   4 - chart style not specified
   5 - no data items to chart
   6 - no output pdf filename specified
   7 - no bar colors specified
*/

#define SORT_ASCENDING		1
#define SORT_DESCENDING		2
extern __declspec(dllimport) UINT PDFC_DataSort(UINT field, UINT order);
/* Specify that the data items are to be sorted before being displayed within the chart

   field must be either 1 or 2

   0 - success
   1 - field invalid
   2 - order invalid
*/

#endif
