unit fAddScaledFile;

//{$DEFINE SG_DEBUG}

{$IFDEF SG_DEBUG}
  {$DEFINE MEMCHK}
{$ENDIF}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, ExtDlgs, StdCtrls, Spin, ExtCtrls, Buttons, DXFImage,
  {$IFDEF MEMCHK}
  MemCheck,
  {$ENDIF}
  sgConsts;

type
  PInsertProp = ^TInsertProp;
  TInsertProp = record
    Position: TFPoint;
    Scale: TFPoint;
    Rotation: Double;
    FileName: string;
  end;

type
  TfmAddScaledFile = class(TForm)
    pnlAddScaled: TPanel;
    pnlUp: TPanel;
    pnlDown: TPanel;
    pnlAll: TPanel;
    lblWasAdded: TLabel;
    lblCountWasAdded: TLabel;
    lblRotation: TLabel;
    btnOpenFile: TBitBtn;
    btnApplyDefault: TBitBtn;
    cbAddedFiles: TComboBox;
    seRotation: TSpinEdit;
    gbPosition: TGroupBox;
    lblPosX: TLabel;
    lblPosY: TLabel;
    lblPosZ: TLabel;
    sePosY: TSpinEdit;
    sePosZ: TSpinEdit;
    sePosX: TSpinEdit;
    gbScale: TGroupBox;
    lblScaleZ: TLabel;
    Label3: TLabel;
    lblScaleX: TLabel;
    Label6: TLabel;
    lblScaleY: TLabel;
    Label7: TLabel;
    seScaleZ: TSpinEdit;
    seScaleX: TSpinEdit;
    seScaleY: TSpinEdit;
    OpenDialog1: TOpenPictureDialog;
    cbWithoutBorder: TCheckBox;
    btnOk: TBitBtn;
    btnCancel: TButton;
    procedure btnOpenFileClick(Sender: TObject);
    procedure btnApplyDefaultClick(Sender: TObject);
    procedure btnOkClick(Sender: TObject);
    procedure sePropsChange(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure cbAddedFilesCloseUp(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    { Private declarations }
    FIsInternalChanging: Boolean;
    FFilesCount: Integer;
    FCADFile: TsgDXFImage;
    procedure SetFilesCount(const Value: Integer);
    procedure ShowProp(Prop: PInsertProp);
    procedure PropToStruct(P: PInsertProp);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    { Public declarations }
    property CADFile: TsgDXFImage read FCADFile;
    property FilesCount: Integer read FFilesCount write SetFilesCount;
  end;

implementation

{$R *.dfm}

procedure TfmAddScaledFile.SetFilesCount(const Value: Integer);
begin
  FFilesCount := Value;
end;

procedure TfmAddScaledFile.ShowProp(Prop: PInsertProp);
begin
  FIsInternalChanging := True;
  sePosX.Value := Round(Prop^.Position.X);
  sePosY.Value := Round(Prop^.Position.Y);
  sePosZ.Value := Round(Prop^.Position.Z);
  seScaleX.Value := Round(Prop^.Scale.X * 100.0);
  seScaleY.Value := Round(Prop^.Scale.Y * 100.0);
  seScaleZ.Value := Round(Prop^.Scale.Z * 100.0);
  seRotation.Value := Round(Prop^.Rotation);
  FIsInternalChanging := False;
  {if (cbAddedFiles.Items.Count > 0)and(cbAddedFiles.ItemIndex <> -1) then
  begin
    S := IntToStr(cbAddedFiles.ItemIndex)+'. '+ExtractFileName(P^.FileName);
    cbAddedFiles.Items[cbAddedFiles.ItemIndex] := S;
  end;}
end;

procedure TfmAddScaledFile.PropToStruct(P: PInsertProp);
begin
  P^.Position.X := sePosX.Value*1.0;
  P^.Position.Y := sePosY.Value*1.0;
  P^.Position.Z := sePosZ.Value*1.0;
  P^.Scale.X := seScaleX.Value / 100.0;
  P^.Scale.Y := seScaleY.Value / 100.0;
  P^.Scale.Z := seScaleZ.Value / 100.0;
  P^.Rotation := seRotation.Value*1.0;
  if (cbAddedFiles.Items.Count > 0) and (cbAddedFiles.ItemIndex <> -1) then
    P^.FileName := PInsertProp(cbAddedFiles.Items.Objects[cbAddedFiles.ItemIndex])^.FileName;
end;

procedure TfmAddScaledFile.btnOpenFileClick(Sender: TObject);
var
  P: PInsertProp;
  S: string;
begin
  OpenDialog1.FileName := '';
  if not OpenDialog1.Execute then Exit;
  New(P);
  PropToStruct(P);
  P^.FileName := OpenDialog1.FileName;
  S := IntToStr(FFilesCount)+'. '+ExtractFileName(P^.FileName);
  cbAddedFiles.Items.Add(S);
  cbAddedFiles.Items.Objects[cbAddedFiles.Items.Count - 1] := Pointer(P);
  cbAddedFiles.ItemIndex := cbAddedFiles.Items.Count - 1;
  cbAddedFilesCloseUp(Sender);
  Inc(FFilesCount);
  lblCountWasAdded.Caption := IntToStr(FFilesCount);
  cbWithoutBorder.Enabled := True;
end;

procedure TfmAddScaledFile.btnApplyDefaultClick(Sender: TObject);
begin
  sePosX.Value := 0;
  sePosY.Value := 0;
  sePosZ.Value := 0;
  seScaleX.Value := 100;
  seScaleY.Value := 100;
  seScaleZ.Value := 100;
  seRotation.Value := 0;
end;

procedure TfmAddScaledFile.btnOkClick(Sender: TObject);
var
  I: Integer;
  P: PInsertProp;
  S: string;
  Picture: TPicture;
  vCADImade: TsgDXFImage;
begin
  if (cbAddedFiles.Items.Count > 0) then
  begin
    Picture := TPicture.Create;
    try
      FCADFile.IsWithoutBorder := cbWithoutBorder.Checked;
      for I := 0 to cbAddedFiles.Items.Count - 1 do
      begin
        P := PInsertProp(cbAddedFiles.Items.Objects[I]);
        Picture.LoadFromFile(P^.FileName);
        if Picture.Graphic is TsgDXFImage then
        begin
          vCADImade := TsgDXFImage.Create;
          vCADImade.Assign(Picture.Graphic);
          vCADImade.IsWithoutBorder := cbWithoutBorder.Checked;
          S := ChangeFileExt(ExtractFileName(P^.FileName), '');
          FCADFile.AddScaledDXF(vCADImade, S, P^.Position, P^.Scale, P^.Rotation);
        end;
      end;
    finally
      Picture.Free;
    end;
  end;
  FCADFile.GetExtents;
end;

constructor TfmAddScaledFile.Create(AOwner: TComponent);
begin
  inherited;
  FFilesCount := 0;
  FCADFile := TsgDXFImage.Create;
  FIsInternalChanging := False;
end;

destructor TfmAddScaledFile.Destroy;
begin
  FCADFile.Free;
  inherited;
end;

procedure TfmAddScaledFile.sePropsChange(Sender: TObject);
var
  P: PInsertProp;
begin
  if not FIsInternalChanging and (TSpinEdit(Sender).Text <> '') and (cbAddedFiles.Items.Count > 0) then
  begin
    P := PInsertProp(cbAddedFiles.Items.Objects[cbAddedFiles.ItemIndex]);
    PropToStruct(P);
  end;
end;

procedure TfmAddScaledFile.FormDestroy(Sender: TObject);
var
  I: Integer;
  P: PInsertProp;
begin
  for I := 0 to cbAddedFiles.Items.Count - 1 do
  begin
    P := PInsertProp(cbAddedFiles.Items.Objects[I]);
    if P <> nil then
      Dispose(P);
  end;
end;

procedure TfmAddScaledFile.cbAddedFilesCloseUp(Sender: TObject);
var
  P: PInsertProp;
begin
  if (cbAddedFiles.Items.Count > 0) then
  begin
    P := PInsertProp(cbAddedFiles.Items.Objects[cbAddedFiles.ItemIndex]);
    ShowProp(P);
  end;
end;

procedure TfmAddScaledFile.FormCreate(Sender: TObject);
begin
   OpenDialog1.Filter := 'CAD Formats (*.dwg;*.dxf;*.plt;*.hgl;*.hg;*.hpg;'+
     '*.plo;*.hp;*.hp1;*.hp2;*.hpgl;*.hpgl2;*.gl2;*.prn;*.spl;*.rtl;*.cgm;*.svg;*.svgz)'+
     '|*.dwg;*.dxf;*.plt;*.hgl;*.hg;*.hpg;'+
     '*.plo;*.hp;*.hp1;*.hp2;*.hpgl;*.hpgl2;*.gl2;*.prn;*.spl;*.rtl;*.cgm;*.svg;*.svgz';
end;

initialization
  {$IFDEF MEMCHK}
  MemChk
  {$ENDIF}

end.
