/*********************************************************************

SYBREX SYSTEMS INCORPORATED
Copyright (C) 2005 Sybrex Systems Incorporated
All rights reserved.

NOTICE: Sybrex permits you to use, modify, and distribute this file
in accordance with the terms of the Sybrex license agreement
accompanying it. If you have received this file from a source other
than Sybrex, then your use, modification, or distribution of it
requires the prior written permission of Sybrex.

----------------------------------------------------------------------
VSDocA.h
*********************************************************************/

#ifndef VSDocA_H
#define VSDocA_H /* PDF Document API Module */

#ifndef __GNUC__ /* suppress gcc warning */
#pragma once
#endif /* __GNUC__ */

#include "VSTypes.h"

#ifdef WINDOWS_PLATFORM
#include <Windows.h>
#include <WinGDI.h>
#endif

#ifdef __cplusplus
extern "C"{
#endif

#define PDFLastSupportedVersion 6       /* Last Supported PDF Document Version is 1.6 */

    /*  Password Type of Crypted PDF Document. Password Validity. */
    typedef enum  _t_TKeyValidType {
        kvtInvalid                      = 0, /* Invalid Password */
        kvtUser                         = 1, /* User Password */
        kvtOwner                        = 2  /* Owner Password */
    } TKeyValidType;

    /*  Protection Key-Length Type of Crypted PDF Document */
    typedef enum _t_TPDFProtectionType {
        pt40BitProtection                           = 0, /*  40 Bit protection key length */
        pt128BitProtection                          = 1  /* 128 Bit protection key length */
    } TPDFProtectionType;

    /*   Type of supported PDF Document Versions */
    typedef enum PDFVersion {
        pdfver10                    = 0,        /* PDF Document Version is 1.0 */
        pdfver11                    = 1,        /* PDF Document Version is 1.1 */
        pdfver12                    = 2,        /* PDF Document Version is 1.2 */
        pdfver13                    = 3,        /* PDF Document Version is 1.3 */
        pdfver14                    = 4,        /* PDF Document Version is 1.4 */
        pdfver15                    = 5,		/* PDF Document Version is 1.5 */
		pdfver16                    = 6 		/* PDF Document Version is 1.6 */
    } TPDFVersion;

    /*  Type of usual PDF Document's Page Sizes */
    typedef enum _t_TPDFPageSize {
        psLetter = 0,   /* Document's Page Size is 792 x 612 */
        psA4,           /* Document's Page Size is 842 x 595 */
        psA3,           /* Document's Page Size is 1190 x 842 */
        psLegal,        /* Document's Page Size is 1008 x 612 */
        psB5,           /* Document's Page Size is 728 x 516 */
        psC5,           /* Document's Page Size is 649 x 459 */
        ps8x11,         /* Document's Page Size is 792 x 595 */
        psB4,           /* Document's Page Size is 1031 x 728 */
        psA5,           /* Document's Page Size is 595 x 419 */
        psFolio,        /* Document's Page Size is 936 x 612 */
        psExecutive,    /* Document's Page Size is 756 x 522 */
        psEnvB4,        /* Document's Page Size is 1031 x 728 */
        psEnvB5,        /* Document's Page Size is 708 x 499 */
        psEnvC6,        /* Document's Page Size is 459 x 323 */
        psEnvDL,        /* Document's Page Size is 623 x 312 */
        psEnvMonarch,   /* Document's Page Size is 540 x 279 */
        psEnv9,         /* Document's Page Size is 639 x 279 */
        psEnv10,        /* Document's Page Size is 684 x 297 */
        psEnv11         /* Document's Page Size is 747 x 324 */
    } TPDFPageSize;

    /*  Page Orientation Type  */
    typedef enum _t_TPDFPageOrientation {
        poPagePortrait   = 0, /* Orientation of Page is Portrait */
        poPageLandScape       /* Orientation of Page is Landscape */
    } TPDFPageOrientation;

    /*  Document Description's Item Type  */
    typedef enum _t_TPDFInformation { /* */ 
        piCreator = 0,      /* Information about creator of the PDF Document */
        piAuthor,           /* Information about author of the PDF Document */
        piDate,             /* Information about date of the creation PDF Document */
        piProducer,         /* Information about producer of the PDF Document */
        piTitle,            /* Information about title of the PDF Document */
        piSubject,          /* Information about subject of the PDF Document */
        piKeyWords,         /* Information about keywords */
        piModificationData  /* Information about modification data */
    } TPDFInformation; /* */ 

    /*   Document Restriction's Flags   */
#define prPrint                 1   /* Allowed print the document */
#define prModifyContent         2   /* Allowed modify the content of the document */
#define prCopyInformation       4   /* Allowed copy text and graphics from the document */
#define prModifyAnnotation      8   /* Allowed add or modify annotations */
#define prFillAcroForm          16  /* Allowed fill in interactive form fields */
#define prExtractTextAndImage   32  /* Allowed extract text and graphics from the document */
#define prAssembleDocument      64  /* Allowed assemble the document */
#define prPrintHiResolution     128 /* Allowed print the document with high resolution */



    /************************************************************************
        Description :
            Create New PDF Document.
        Parameters :
            Lib                 - [in] Current loaded PDF Library.
        Returns :
            PDF Document Handle.
        See also :
            PDFDocClose.
    ************************************************************************/
    PDFDocHandle PDFDocCreate( PDFLibHandle Lib );
    /* Description
       Checking the PDF Document on Security.
       Parameters
       Doc :  [in] Checking PDF Document Handle.
       Returns
       Boolean value : true - if document is crypted, false is not
       crypted.
       See Also
       PDFDocCheckPassword, PDFDocSetSecurity.                     */
    ppBool PDFDocIsCrypted( PDFDocHandle Doc );
    /************************************************************************
        Description :
            Close PDF Document and free all structures.
        Parameters :
            Doc                 - [in] PDF Document Handle.
        Returns :
            None.
        See also :
            PDFDocCreate.
    ************************************************************************/
    void PDFDocClose( PDFDocHandle Doc );
    /************************************************************************
        Description :
            Checking the PDF Document on Validity of Password.
        Parameters :
            Doc                 - [in] PDF Document Handle.
            Password            - [in] Password text string.
        Returns :
            Password Validity Type. If kvtInvalid then Password is invalid.
        See also :
            PDFDocIsCrypted, PDFDocSetSecurity, TKeyValidType.
    ************************************************************************/
    TKeyValidType PDFDocCheckPassword( PDFDocHandle Doc, char *Password );
    /* Description
       Set PDF Document Security.
       Parameters
       Doc :                [in] PDF Document Handle.
       ProtectionEnabled :  [in] If true set document security. If false
                            remove security from PDF document. 
       Permission :         [in] Document Permission's Flags.
       KeyLength :          [in] 40\-bits or 128\-bits Security Key
                            Length.
       UserPassword :       [in] User Password text string.
       OwnerPassword :      [in] Owner Password text string.
       Returns
       None.
       See Also
       PDFDocIsCrypted, PDFDocCheckPassword, TPDFProtectionType.         */
    void PDFDocSetSecurity( PDFDocHandle Doc, ppBool ProtectionEnabled, ppInt32 Permission,
                           TPDFProtectionType KeyLength, char *UserPassword, char *OwnerPassword );
    /************************************************************************
        Description :
            Get Document Permission's Flags.
        Parameters :
            Doc                 - [in] Checking PDF Document Handle.
        Returns :
            Document Permission's Flags.
        See also :
            PDFDocSetSecurity.
    ************************************************************************/
    ppInt32 PDFDocGetPermission( PDFDocHandle Doc );
    /************************************************************************
        Description :
            Add Page in PDF Document.
        Parameters :
            Doc                 - [in] PDF Document Handle.
            Width               - [in] Width of Page.
            Height              - [in] Height of Page.
        Returns :
            Index of New Page in Document.
        See also :
            PDFDocAppendPage2.
    ************************************************************************/
    ppInt32 PDFDocAppendPage( PDFDocHandle Doc, ppReal Width, ppReal Height );
    /************************************************************************
        Description :
            Add Page in PDF Document.
        Parameters :
            Doc                 - [in] PDF Document Handle.
            Size                - [in] Type of Page Size, see TPDFPageSize.
            Orientation         - [in] Orientation of Page, see TPDFPageOrientation.
        Returns :
            Index of New Page in Document.
        See also :
            PDFDocAppendPage, TPDFPageSize, TPDFPageOrientation.
    ************************************************************************/
    ppInt32 PDFDocAppendPage2( PDFDocHandle Doc, TPDFPageSize Size, TPDFPageOrientation Orientation );
    /************************************************************************
        Description :
            Get Page Count of PDF Document.
        Parameters :
            Doc                 - [in] PDF Document Handle.
        Returns :
            Page Count of PDF Document.
    ************************************************************************/
    ppInt32 PDFDocGetPageCount( PDFDocHandle Doc );
    /* Description
       Save PDF Document in memory stream or file stream, see PDF
       Streams.
       Parameters
       Doc :     [in] PDF Document Handle.
       Stream :  [out] PDF Stream Handle.
       Returns
       None.
       See Also
       PDFDocLoadFromStream, PDFStreamHandle, PDFDocSaveToMemory,
       PDFDocSaveToFile.                                          */
    void PDFDocSaveToStream( PDFDocHandle Doc, PDFStreamHandle Stream );
    /* Description
       Save PDF Document in memory.
       Parameters
       Doc :   [in] PDF Document Handle.
       Size :  [out] Size of buffer.
       Returns
       Pointer on PDF document beginning.
       See Also
       PDFDocLoadFromBuffer.              */
    void *PDFDocSaveToMemory( PDFDocHandle Doc, ppInt32 *Size );
    /* Description
       Save PDF Document in file.
       Parameters
       Doc :       [in] PDF Document Handle.
       FileName :  [in] Name of output file, text string.
       Returns
       None.
       See Also
       PDFDocLoadFromFile.                                */
    void PDFDocSaveToFile( PDFDocHandle Doc, char *FileName );
    /* Description
       Get Information from Document Description.
       Parameters
       Doc :   [in] PDF Document Handle.
       Info :  [in] Type of Description Information.
       Returns
       Handle of Information Object.
       See Also
       PDFDocSetInfo, TPDFInformation.               */
    PDFCosHandle PDFDocGetInfo( PDFDocHandle Doc, TPDFInformation Info );
    /* Description
       Save information in Document description.
       Parameters
       Doc :    [in] PDF Document Handle.
       Info :   [in] Type of Description Information.
       Value :  [in] Handle of Information Object.
       Returns
       None.
       See Also
       PDFDocGetInfo, TPDFInformation.                */
    void PDFDocSetInfo( PDFDocHandle Doc, TPDFInformation Info, PDFCosHandle Value );
    /* Description
       Load PDF Document from memory stream or file stream, see PDF
       Streams.
       Parameters
       Lib :     [in] PDF Library Handle.
       Stream :  [in] PDF Stream Handle.
       Returns
       PDF Document Handle.
       See Also
       PDFDocSaveToStream, PDFStreamHandle, PDFDocLoadFromFile,
       PDFDocLoadFromBuffer.                                        */
    PDFDocHandle PDFDocLoadFromStream( PDFLibHandle Lib, PDFStreamHandle Stream );
    /* Description
       Load PDF Document from file.
       Parameters
       Lib :       [in] PDF Library Handle.
       FileName :  [in] Name of input file, text string.
       Returns
       PDF Document Handle.
       See Also
       PDFDocSaveToFile.                                 */
    PDFDocHandle PDFDocLoadFromFile( PDFLibHandle Lib, char *FileName );
    /* Description
       Load PDF Document from Memory Stream.
       Parameters
       Lib :     [in] PDF Library Handle.
       Buffer :  [in] Pointer on PDF document beginning.
       Length :  [in] Length of buffer in bytes.
       Returns
       PDF Document Handle.
       See Also
       PDFDocSaveToMemory.                               */
    PDFDocHandle PDFDocLoadFromBuffer( PDFLibHandle Lib, void *Buffer, ppInt32 Length );
    /* Description
       Set whether PDF document is stored as packed document or not.
       Parameters
       Doc :     [in] PDF Document Handle.
       Packed :  [in] if true use packed save method. If false don't
                 use it.
       Returns
       None.                                                         */
    void PDFDocSetPacked( PDFDocHandle Doc, ppBool Packed );
    /* Description
       Set whether PDF document is stored as linearized document or
       not.
       Parameters
       Doc :         [in] PDF Document Handle.
       Linearized :  [in] if true use linearized save method. If false
                     non linearized method.
       Returns
       None.                                                           */
    void PDFDocSetLinearized( PDFDocHandle Doc, ppBool Linearized );
    /* Description
       Remove all unused objects from PDF Document before save it.
       Parameters
       Doc :     [in] PDF Document Handle.
       Remove :  [in] If true remove unused ( unlink ) objects. If
                 false don't remove. 
       Returns
       None.                                                       */
    void PDFDocSetRemoveUnUsed( PDFDocHandle Doc, ppBool Remove );
    /* Description
       Set "Jpeg Images Quality" for jpeg images stored in PDF
       Document.
       Parameters
       Doc :      [in] PDF Document Handle.
       Quality :  [in] Value from 0 to 100\: 0 \- worst quality,
                  smallest size. 100 \- best quality, biggest size.
       Returns
       None.                                                        */
    void PDFDocSetJpegImageQuality ( PDFDocHandle Doc, ppInt32 Quality );

#ifdef WINDOWS_PLATFORM
    /* Description
       Set AutoLaunch option for PDF Document, it launches after
       saving (only Windows Platform).
       Parameters
       Doc :         [in] PDF Document Handle.
       AutoLaunch :  [in] If true launch PDF document, If false don't.
       Returns
       None.                                                           */
	void PDFDocSetAutoLaunch( PDFDocHandle Doc, ppBool AutoLaunch );
    /* Description
       Set hDC concerning which EMF images will be parsed.
       Parameters
       Doc :  [in] PDF Document Handle.
       DC :   [in] Handle of Device Context.
       Returns
       None.                                               */
	void PDFDocSetUsedDC( PDFDocHandle Doc, HDC DC ); 
    /* Description
       Set "Emf Images as jpeg" option for PDF Document.
       Parameters
       Doc :        [in] PDF Document Handle.
       EmfAsJpeg :  [in] If value has set to true, all images stored in the
                    EMF file will be saved as JPEG images.<p/>If value has
                    set to false, all images stored in the EMF file will be
                    saved with flate compression.
       Returns
       None.                                                                */
    void PDFDocSetEMFImagesAsJpeg ( PDFDocHandle Doc, ppBool EmfAsJpeg );
    /***********************************************************************/
#endif

#ifdef __cplusplus
}
#endif

#endif /* #ifndef VSDocA_H */
