/*********************************************************************

SYBREX SYSTEMS INCORPORATED
Copyright (C) 2005 Sybrex Systems Incorporated
All rights reserved.

NOTICE: Sybrex permits you to use, modify, and distribute this file
in accordance with the terms of the Sybrex license agreement
accompanying it. If you have received this file from a source other
than Sybrex, then your use, modification, or distribution of it
requires the prior written permission of Sybrex.

----------------------------------------------------------------------
VSCosA.h
*********************************************************************/

#ifndef VSCosA_H
#define VSCosA_H  /*  */
#ifndef __GNUC__ /* suppress gcc warning */
#pragma once
#endif /* __GNUC__ */
#include "VSTypes.h"

#ifdef __cplusplus
extern "C"{
#endif
        
#define ppRealNULL -32000 /* Cos real NULL */ 

	/* Cos objects */ 

    /* Available Cos objects */ 
    typedef enum {
        CosNull                                 = 2, /* NULL Cos object         */
        CosInteger                              = 4, /* Integer Cos Object      */
        CosReal                                 = 8, /* Real Cos Object         */
        CosBoolean                              = 16, /* Boolean Cos Object      */
        CosName                                 = 32, /* Name Cos Object         */
        CosString                               = 64, /* String Cos Object       */
        CosDict                                 = 128, /* Dictionary Cos Object   */
        CosArray                                = 256, /* Array Cos Object        */
        CosStream                               = 512  /* Stream Cos Object       */
    } CosType;

    /*  Null Object  */

    /********************************************************************
    Description: 
        Creates a new direct null object.
    Parameters:
        Doc         - [in]  The document in which the null object is used.
    Returns: 
        The newly-created null Cos object.
    See Also:
        CosFreeObj
        CosGetType
    *********************************************************************/
    PDFCosHandle CosNewNull( PDFDocHandle Doc );



    /*  Int Object  */

    /* Description
       Creates a new integer object and sets the specified value.
       Parameters
       Doc :         [in] The document in which the integer is used.
       IsIndirect :  [in] If true, creates the integer object as an
                     indirect object.
       Value :       [in] The value the new integer will have.
       Returns
       The newly-created integer Cos object.
       See Also
       CosFreeObj CosGetType CosGetIntValue CosSetIntValue           */
    PDFCosHandle CosNewInt(PDFDocHandle Doc, ppBool IsIndirect, ppInt32 Value);



    /********************************************************************
    Description: 
        Gets the value of the specified integer object.
    Parameters: 
        CosObject   - [in]  The integer Cos object whose value is obtained.
    Returns:
        Value of the integer Cos object.
    See Also:
        CosNewInt
        CosSetIntValue
    *********************************************************************/
    ppInt32 CosGetIntValue(PDFCosHandle CosObject);



    /********************************************************************
    Description: 
        Sets the value of the specified integer object.
    Parameters: 
        CosObject   - [in]  The integer Cos object whose value is assigned.
        Value       - [in]  New value of the integer Cos object.
    Returns: 
        None
    See Also:
        CosNewInt
        CosGetIntValue
    *********************************************************************/
    void CosSetIntValue(PDFCosHandle CosObject, ppInt32 Value);



    /*  Real Object  */

    /* Description
       Creates a new real object and sets the specified value.
       Parameters
       Doc :         [in] The document in which the real is used.
       IsIndirect :  [in] If true, creates the real object as an indirect
                     object.
       Value :       [in] The value the new real will have.
       Returns
       The newly-created real Cos object.
       See Also
       CosFreeObj CosGetType CosGetRealValue CosSetRealValue              */
    PDFCosHandle CosNewReal(PDFDocHandle Doc, ppBool IsIndirect,  ppReal Value);



    /********************************************************************
    Description: 
        Gets the value of the specified real object.
    Parameters: 
        CosObject   - [in]  The real Cos object whose value is obtained.
    Returns:
        Value of the real Cos object.
    See Also:
        CosNewReal
        CosSetRealValue
    *********************************************************************/
    ppReal CosGetRealValue(PDFCosHandle CosObject);



    /********************************************************************
    Description: 
        Sets the value of the specified real object.
    Parameters: 
        CosObject   - [in]  The real Cos object whose value is assigned.
        Value       - [in]  New value of the real Cos object.
    Returns: 
        None
    See Also:
        CosNewReal
        CosGetRealValue
    *********************************************************************/
    void CosSetRealValue(PDFCosHandle CosObject, ppReal Value);



    /*  Boolean Object  */

    /* Description
       Creates a new boolean object and sets the specified value.
       Parameters
       Doc :         [in] The document in which the boolean is used.
       IsIndirect :  [in] If true, creates the boolean object as an
                     indirect object.
       Value :       [in] The value which new boolean will have.
       Returns
       The newly-created boolean Cos object.
       See Also
       CosFreeObj CosGetType CosGetBoolValue CosSetBoolValue         */
    PDFCosHandle CosNewBool(PDFDocHandle Doc, ppBool IsIndirect, ppBool Value);



    /********************************************************************
    Description: 
        Gets the value of the specified boolean object.
    Parameters: 
        CosObject   - [in]  The boolean Cos object whose value is obtained.
    Returns:    
        Value of the boolean Cos object.
    See Also:
        CosNewBool
        CosSetBoolValue
    *********************************************************************/
    ppBool CosGetBoolValue(PDFCosHandle CosObject);



    /********************************************************************
    Description: 
        Sets the value of the specified boolean object.
    Parameters: 
        CosObject   - [in]  The boolean Cos object whose value is assigned.
        Value       - [in]  New value of the Cos boolean object.
    Returns: 
        None
    See Also:
        CosNewBool
        CosGetBoolValue
    *********************************************************************/
    void CosSetBoolValue(PDFCosHandle CosObject, ppBool Value);



    /*  String  Object */

    /* Description
       Creates a new string object and sets the specified value.
       Parameters
       Doc :         [in] The document in which the string is used.
       IsIndirect :  [in] If true, creates the string object as an indirect
                     object.
       String :      [in] The value that the new string will have. It is not
                     a C string, since Cos strings can contain NULL
                     characters. The data in String is copied, that is, if
                     String was dynamically allocated, it can be free after
                     this call.
       Length :      [in] The length of String.
       Returns
       The newly-created string Cos object.
       See Also
       CosFreeObj CosGetType CosGetStringValue CosSetStringValue             */
    PDFCosHandle CosNewString(PDFDocHandle Doc, ppBool IsIndirect, char *String, ppInt32 Length);



    /* Description
       Gets the value of string Cos object and the string's length.
       Parameters
       CosObject :  [in] The string Cos object whose value is
                    obtained.
       Length :     [out] Length of the value in bytes.
       Returns
       The value of string Cos object.
       See Also
       CosNewString CosSetStringValue                               */
    char *CosGetStringValue(PDFCosHandle CosObject, ppInt32 *Length);



    /* Description
       Sets the new value for string Cos object.
       Parameters
       CosObject :  [in] The string Cos object whose value is assigned.
       String :     [in] The new value that the string Cos object will have.
                    It is not a C string, since Cos strings can contain NULL
                    characters. The data in String is copied, that is, if
                    String was dynamically allocated, it can be free after
                    this call.
       Length :     [in] The new length of String.
       Returns
       None.
       See Also
       CosNewString CosGetStringValue                                        */
    void CosSetStringValue(PDFCosHandle CosObject, char *String, ppInt32 Length);



    /*  Name Object */

    /* Description
       Creates a new name object and sets the specified value.
       Parameters
       Doc :         [in] The document in which the name object is used.
       IsIndirect :  [in] If true, creates the name object as an indirect
                     object.
       Value :       [in] The value the new name will have.
       Returns
       The newly-created name Cos object.
       See Also
       CosFreeObj CosGetType CosGetNameValue CosSetNameValue              */
    PDFCosHandle CosNewName(PDFDocHandle Doc, ppBool IsIndirect, ppAtom Value);



    /********************************************************************
    Description: 
        Gets the value of the specified name object.
    Parameters: 
        CosObject   - [in]  The name Cos object whose value is obtained.
    Returns:    
        Value of the name Cos object.
    See Also:
        CosNewName
        CosSetNameValue
    *********************************************************************/
    ppAtom CosGetNameValue(PDFCosHandle CosObject);



    /********************************************************************
    Description: 
        Sets the value of the specified name object.
    Parameters: 
        CosObject   - [in]  The boolean Cos object whose value is assigned.
        Value       - [in]  New value of the name Cos object.
    Returns: 
        None
    See Also:
        CosNewName
        CosGetNameValue
    *********************************************************************/
    void CosSetNameValue(PDFCosHandle CosObject, ppAtom Value);



    /*  Array  */

    /********************************************************************
    Description: 
        Creates and returns a new array Cos object.
    Parameters: 
        Doc         - [in]  The document in which the array is used.
        IsIndirect  - [in]  If true, creates the array object 
                            as an indirect object.  
        Entries     - [in]  The number of elements that will be in the array. This value
                            only a hint; Cos arrays grow dynamically as needed.
    Returns:    
        The newly-created array Cos object.
    See Also:
        CosFreeObj
        CosGetType
        CosArrayCount
        CosArrayInsert
        CosArrayAppend
        CosArrayRemove
        CosArrayClear
    *********************************************************************/
    PDFCosHandle CosNewArray(PDFDocHandle Doc, ppBool IsIndirect, ppInt32 Entries);



    /* Description
       Gets the number of elements in array.
       Parameters
       CosObject :  [in] The array for which the number of elements are
                    determined.
       Returns
       The number of elements in array.
       See Also
       CosNewArray CosArrayInsert CosArrayAppend CosArrayRemove
       CosArrayClear                                                    */
    ppInt32 CosArrayCount(PDFCosHandle CosObject);



    /* Description
       Inserts an cos object into an array.
       Parameters
       CosObject :     [in] The array into which the object is inserted.
       NewCosObject :  [in] The object to insert.
       Position :      [in] The location in the array to insert the cos
                       object. The cos object is inserted before the specified
                       location. The first element in an array has a pos of
                       zero. If pos \>= CosArrayCount ( CosObject ), it
                       appends obj to array (increasing the array count by 1).
       Returns
       Position in which Cos object was inserted.
       See Also
       CosNewArray CosArrayCount CosArrayInsert CosArrayAppend
       CosArrayRemove CosArrayClear                                            */
    ppInt32 CosArrayInsert(PDFCosHandle CosObject, PDFCosHandle NewCosObject, ppInt32 pos);



    /* Description
       Appends an cos object into an array.
       Parameters
       CosObject :     [in] The array into which the object is
                       appended.
       NewCosObject :  [in] The object to append.
       Returns
       Position in which Cos object was inserted.
       See Also
       CosNewArray CosArrayCount CosArrayInsert CosArrayAppend
       CosArrayRemove CosArrayClear                            */
    ppInt32 CosArrayAppend(PDFCosHandle CosObject, PDFCosHandle NewCosObject);



    /* Summary
       Removes element from array.
       Description
       Checks whether the position is within the array bounds and
       then removes it from the array and moves each subsequent
       element to the slot with the next smaller Index and
       decrements the array's length by 1. Removed element will be
       free.
       Parameters
       CosObject :  [in] The array Cos object to remove the member from
                    it.
       Index :      [in] The Index for the array member to remove. Array
                    indices start at 0.
       Returns
       None
       See Also
       CosNewArray CosArrayCount CosArrayInsert CosArrayAppend
       CosArrayRemove CosArrayClear                                      */
    void CosArrayRemove(PDFCosHandle CosObject, ppInt32 Index);



    /* Description
       Gets the specified element from an array.
       Parameters
       CosObject :  [in] The array from which an element is obtained.
       Index :      [in] The Index for the array member to obtain. Array
                    indices start at 0.
       Returns
       The Cos object occupying the index element of array. Returns
       a null Cos object if Index is outside the array bounds. If
       specified element is referenced Cos object function returns
       Cos object with ID equal to value of referenced Cos object.
       See Also
       CosNewArray CosArrayCount CosArrayInsert CosArrayAppend
       CosArrayRemove CosArrayClear                                      */
    PDFCosHandle CosArrayItem(PDFCosHandle CosObject, ppInt32 Index);



    /* Description
       Clears and gets free all elements from an array.
       Parameters
       CosObject :  [in] The array from which an elements are removed.
       Returns
       None
       See Also
       CosNewArray CosArrayCount CosArrayInsert CosArrayAppend
       CosArrayRemove CosArrayClear                                    */
    void CosArrayClear(PDFCosHandle CosObject);



    /*  Dictionary  */

    /********************************************************************
    Description: 
        Creates a new dictionary.
    Parameters: 
        Doc         - [in]  The document in which the dictionary is used.
        IsIndirect  - [in]  If true, creates the dictionary object 
                            as an indirect object.  
        Entries     - [in]  Number of entries in the dictionary. This value is 
                            only a hint - Cos dictionaries grow dynamically as needed.
    Returns:
        The newly-created dictionary Cos object.
    See Also:
        CosGetType
        CosFreeObj
        CosDictCount
        CosDictGetPair
        CosDictAppend
        CosDictRemoveKey
        CosDictValueByName
        CosDictClear
    *********************************************************************/
    PDFCosHandle CosNewDict(PDFDocHandle Doc, ppBool IsIndirect, ppInt32 Entries);



    /* Description
       Gets the number of key-value pair in the dictionary. This
       method can also be used with a stream object. In that case,
       \returns number the key-value pair from the stream's
       attributes dictionary.
       Parameters
       CosObject :  [in] The dictionary or stream for which the number of
                    key\-value pair is determined.
       Returns
       The number of key-value pair in the dictionary.
       See Also
       CosNewDict CosDictGetPair CosDictAppend CosDictRemoveKey
       CosDictValueByName CosDictClear                                    */
    ppInt32 CosDictCount(PDFCosHandle CosObject);

    /* Description
       Gets the key-value pair in the dictionary. This method can
       also be used with a stream object. In that case, returns the
       key-value pair from the stream's attributes dictionary.
       Parameters
       CosObject :  [in] The dictionary or stream for which the
                    key\-value pair is determined.
       Index :      [in] Index of the pair for which is needed to obtain
                    key and value.
       Key :        [out] Key from pair.
       Value :      [out] Value from pair.
       Returns
       The number of key-value pair in the dictionary.
       See Also
       CosNewDict CosDictCount CosDictAppend CosDictRemoveKey
       CosDictValueByName CosDictClear                                   */

    void CosDictGetPair(PDFCosHandle CosObject, ppInt32 Index, ppAtom *Key, PDFCosHandle *Value);

    /* Description
       Sets the value of a dictionary key, adding the key to the
       dictionary. This method can also be used with a stream
       \object. In that case, the key-value pair is added to the
       stream's attributes dictionary.
       Parameters
       CosObject :  [in] The dictionary or stream in which a value is
                    set.
       Key :        [in] The key which value is set.
       Value :      [in] The value to set.
       Returns
       None
       See Also
       CosNewDict CosDictCount CosDictGetPair CosDictRemoveKey
       CosDictValueByName CosDictClear                                */
    void CosDictAppend(PDFCosHandle CosObject, ppAtom Key, PDFCosHandle KeyValue);



    /* Description
       Removes and gets free a key-value pair from a dictionary.
       This method can also be used with a stream object. In that
       case, the key-value pair is removed from the stream's
       attributes dictionary.
       Parameters
       CosObject :  [in] The dictionary from which the key\-value pair
                    is removed.
       Key :        [in] The key to remove.
       Returns
       None
       See Also
       CosNewDict CosDictCount CosDictGetPair CosDictAppend
       CosDictValueByName CosDictClear                                 */
    void CosDictRemoveKey(PDFCosHandle CosObject, ppAtom Key);



    /* Description
       Gets the value of the specified key in the specified
       dictionary. If it's called with a stream object instead of a
       dictionary object, this method gets the value of the
       specified key from the stream's attributes dictionary.
       Parameters
       CosObject :  [in] The dictionary or stream from which a value is
                    obtained.
       Key :        [in] The key whose value is obtained.
       Returns
       The object associated with the specified key. Returns a null
       Cos object if key is not present. If value is referenced Cos
       \object returns Cos object with ID equal to value of
       referenced Cos object.
       Example
       <code>
           PDFCosHandle dict, obj;
           \obj = CosDicValueByName ( dict, ULStringToAtom ( Lib, "Pages" ) );
       </code>
       See Also
       CosNewDict CosDictCount CosDictGetPair CosDictAppend
       CosDictRemoveKey CosDictClear                                           */
    PDFCosHandle CosDictValueByName(PDFCosHandle CosObject, ppAtom Key);



    /* Description
       Clears and gets free all keys and values from the dictionary
       \or stream.
       Parameters
       CosObject :  [in] The dictionary or stream from which elements
                    are removed.
       Returns
       None
       See Also
       CosNewDict CosDictCount CosDictGetPair CosDictAppend
       CosDictRemoveKey CosDictValueByName                            */
    void CosDictClear( PDFCosHandle CosObject);

    /*  Stream  */

    /********************************************************************
    Description: 
        Creates a new stream.
    Parameters: 
        Doc         - [in]  The document in which the dictionary is used.
        IsIndirect  - [in]  Must always be true, specifying that the Cos stream is
                            created as an indirect object.
        Entries     - [in]  Number of entries in the attribute dictionary. This value 
                            is only a hint - Cos dictionaries grow dynamically as needed.
    Returns:
        The newly-created stream Cos object.
    See Also:
        CosFreeObj
        CosGetType
        CosSteamGetAttr
        CosStreamGetValue
    *********************************************************************/
    PDFCosHandle CosNewStream(PDFDocHandle Doc, ppBool IsIndirect, ppInt32 Entries);



    /********************************************************************
    Description: 
        Gets a stream's attributes dictionary.
    Parameters: 
        Lib         - [in]  PDF Library Object.
        CosObject   - [in]  The stream whose attributes dictionary is obtained.
    Returns:
        The stream's attributes dictionary Cos object.
    See Also:
        CosNewStream
        CosStreamGetValue
    *********************************************************************/
    PDFCosHandle CosStreamGetAttr(PDFCosHandle CosObject);



    /********************************************************************
    Description: 
        Gets a stream's value.
    Parameters: 
        Lib         - [in]  PDF Library Object.
        CosObject   - [in]  The stream whose attributes value is obtained.
    Returns:    
        The value of the stream Cos object.
    See Also:
        CosNewStream
        CosStreamGetAttr
    *********************************************************************/
    PDFStreamHandle CosStreamGetValue(PDFCosHandle CosObject);



    /*   Other Functions for work with CosObject */


    /********************************************************************
    Description: 
        Gets an objects type.
    Parameters: 
        CosObject   - [in]  The object whose type is obtained.
    Returns:    
        The objects type.
    *********************************************************************/
    CosType CosGetType(PDFCosHandle CosObject);



    /* Description
       Gets free Cos object. If it's a composite object (array,
       dictionary or stream) :
         * all the direct objects in it will be automatically
           destroyed
         * the indirect objects in it will be not destroyed
       Parameters
       CosObject :  [in] The object to free.
       Returns
       None.                                                    */
    void CosFreeObj(PDFCosHandle CosObject);



    /********************************************************************
    Description: 
        Gets the value of the specified real or integer object.
    Parameters: 
        CosObject   - [in]  The real or integer Cos object whose value is obtained.
    Returns:
        Value of the real or integer Cos object.
    *********************************************************************/    
    ppReal CosGetNumberValue(PDFCosHandle CosObject);


    /* Description
       Creates new Cos object and copies all data from source Cos
       \object excluding indirect information.
       Parameters
       CosObject :  [in] The Cos object from which information will be
                    received for copy.
       Returns
       New Cos object which has all infomation from source Cos
       \object.                                                        */    
    PDFCosHandle CosCopyObj(PDFDocHandle Doc, PDFCosHandle CosObject);


    /* Description
       Gets the indirect Cos object from document.
       Parameters
       Doc :  [in] The document from which Cos object will be loaded.
       ID :   [in] The index of the indirect Cos object which is to be
              returned.
       Returns
       Either Cos object or the null object returns if there is no
       \object with this ID.                                           */ 
    PDFCosHandle CosGetFromDoc(PDFDocHandle Doc, ppInt32 ID);

    
    /* Description
       Tests object if it's indirect or direct.
       Parameters
       CosObject :  [in] The object to test.
       Returns
       true if Cos Object is indirect, false if Cos Object is
       direct.
       See Also
       CosObjGetID CosObjGetGeneration                        */
    ppBool CosObjIsIndirect(PDFCosHandle CosObject);



    /* Description
       Gets the index for an indirect object.
       Parameters
       CosObject :  [in] The indirect CosObj for which the ID is obtained. A
                    CosObj can be determined as indirect using
                    CosObjIsIndirect function.
       Returns
       The ID of CosObj.
       See Also
       CosObjIsIndirect CosObjGetGeneration                                  */
    ppInt32 CosObjGetID( PDFCosHandle CosObject);



    /* Description
       Gets the generation number of an indirect Cos object.
       Parameters
       CosObject :  [in] The indirect CosObj for which the generation number
                    is obtained. A CosObj can be determined as indirect
                    using CosObjIsIndirect function.
       Returns
       The generation number of CosObj.
       See Also
       CosObjIsIndirect CosObjGetID                                          */
    ppUns16 CosObjGetGeneration( PDFCosHandle CosObject);

#ifdef __cplusplus
}
#endif

#endif /* #ifndef VSCosA_H */
