/*********************************************************************

SYBREX SYSTEMS INCORPORATED
Copyright (C) 2005 Sybrex Systems Incorporated
All rights reserved.

NOTICE: Sybrex permits you to use, modify, and distribute this file
in accordance with the terms of the Sybrex license agreement
accompanying it. If you have received this file from a source other
than Sybrex, then your use, modification, or distribution of it
requires the prior written permission of Sybrex.

----------------------------------------------------------------------
VSBaseA.h
*********************************************************************/

#ifndef VSBaseA_H
#define VSBaseA_H /* Common Types Routines Module */

#ifndef __GNUC__ /* suppress gcc warning */
#pragma once
#endif /* __GNUC__ */

#include "VSTypes.h"

/* \File Open Mode Type */
typedef enum {
    ppFileReadMode = 0, /* Read File Mode */
    ppFileWriteMode     /* Write File Mode */
} ppFileOpenMode;

#ifdef __cplusplus
extern "C"{
#endif

/************************************************************************/
/* Atom section                                                         */
/************************************************************************/


    /* Description
       Gets atom key by String in PDF Library.
       Parameters
       Lib :     [in] PDF Library Handle.
       String :  [in] Text String Name.
       Returns
       Atom Key. Name Index in PDF Library
       See Also
       ULAtomToString                          */
    ppAtom ULStringToAtom( PDFLibHandle Lib, char *String );



    /* Description
       Gets Text String Name by Atom Key in PDF Library.
       Parameters
       Lib :   [in] PDF Library Handle.
       Atom :  [in] Atom Key.
       Returns
       Text String Name in PDF Library
       See Also
       ULStringToAtom                                    */
    char *ULAtomToString( PDFLibHandle Lib, ppAtom Atom );



    /* Description
       Gets atom count in PDF Library.
       Parameters
       Lib :  [in] PDF Library Handle.
       Returns
       Atom count in PDF Library.      */
    ppInt32 ULGetAtomCount( PDFLibHandle Lib );



    /* Description
       Clears atoms in PDF Library. Gets free namespace.
       Parameters
       Lib :  [in] PDF Library Handle.
       Returns
       None.                                             */
    void ULClearAtoms( PDFLibHandle Lib );



    /* Description
       Tests if atom exists in PDF Library for searching text
       string.
       Parameters
       Lib :     [in] PDF Library Handle.
       String :  [in] Text String Name.
       Returns
       Boolean : true - exists, false - name not found.
       See Also
       ULStringToAtom                                         */
    ppBool ULExistsAtomForString( PDFLibHandle Lib, char *String );




/************************************************************************/
/* File section                                                         */
/************************************************************************/



    /* Description
       Sets file cursor to position ( byte offset ).
       Parameters
       FileHandle :  [in] PDF File Handle.
       Position :    [in] File offset in bytes.
       Returns
       Position which is set.
       See Also
       ULGetFilePosition                             */
    ppInt32 ULSetFilePosition( PDFFileHandle FileHandle, ppInt32 Position );
    
    
    
    /* Description
       Reads data from file to buffer. Length is in bytes.
       Parameters
       FileHandle :  [in] PDF File Handle.
       Buffer :      [in] Destination buffer in memory for data.
       Length :      [in] Size of read block in bytes.
       Returns
       Size of real read block in bytes.
       See Also
       ULWriteFile                                               */
    ppInt32 ULReadFile( PDFFileHandle FileHandle, void *Buffer, ppInt32 Length );
    
    
    
    /* Description
       Writes data from buffer to file. Length is in bytes.
       Parameters
       FileHandle :  [in] PDF File Handle.
       Buffer :      [in] Source data buffer in memory.
       Length :      [in] Size of write block in bytes.
       Returns
       Size of real write block in bytes.
       See Also
       ULReadFile                                           */
    ppInt32 ULWriteFile( PDFFileHandle FileHandle, void *Buffer, ppInt32 Length );
    
    
    
    /* Description
       Gets file cursor position ( byte offset from beginning of the
       \file ).
       Parameters
       FileHandle :  [in] PDF File Handle.
       Returns
       \File offset in bytes.
       See Also
       ULSetFilePosition                                            */
    ppInt32 ULGetFilePosition( PDFFileHandle FileHandle );
    
    
    
    /* Description
       Gets file size in bytes.
       Parameters
       FileHandle :  [in] PDF File Handle.
       Returns
       \File size in bytes.                */
    ppInt32 ULGetFileSize( PDFFileHandle FileHandle );
    
    
    
    /* Description
       Gets one character from file.
       Parameters
       FileHandle :  [in] PDF File Handle.
       Returns
       One character form file. If it returns -1 than it is EOF (
       end of file ).                                             */
    ppInt32 ULGetFileChar( PDFFileHandle FileHandle );
    
    
    
    /* Description
       Gets one character from file. Same as ULGetFileChar, only
       \file cursor stays on that place.
       Parameters
       FileHandle :  [in] PDF File Handle.
       Returns
       One character form file. If it returns -1 than it is EOF (
       end of file ).                                             */
    ppInt32 ULLookFileChar( PDFFileHandle FileHandle );
    
    
    
    /* Description
       Opens file and returns PDF File Handle.
       Parameters
       Lib :       [in] PDF Library Handle.
       FileName :  [in] Filename, text string.
       OpenMode :  [in] Open Mode \: read or write.
       Returns
       PDF File Handle.
       See Also
       ULCloseFile, ppFileOpenMode                  */
    PDFFileHandle ULOpenFile( PDFLibHandle Lib, char *FileName, ppFileOpenMode OpenMode );
    
    
    
    /* Description
       Closes PDF File.
       Parameters
       FileHandle :  [in] PDF File Handle.
       Returns
       None.
       See Also
       ULOpenFile                          */
    void ULCloseFile( PDFFileHandle FileHandle );

/************************************************************************/
/* Stream section                                                       */
/************************************************************************/



    /* Description
       Creates new Memory PDF Stream with initializing size.
       Parameters
       Lib :   [in] PDF Library Handle.
       Size :  [in] Initializing size of memory Stream in bytes.
       Returns
       Memory PDF Stream Handle.
       See Also
       ULStrmClose                                               */
    PDFStreamHandle ULMemStrmNew( PDFLibHandle Lib, ppInt32 Size );
    
    
    
    /* Description
       Converts memory buffer to PDF Stream.
       Parameters
       Lib :     [in] PDF Library Handle.
       Buffer :  [in] Existed memory buffer.
       Length :  [in] Size of buffer in bytes.
       Returns
       Memory PDF Stream Handle.
       See Also
       ULStrmClose                             */
    PDFStreamHandle ULMemStrmRDOpen( PDFLibHandle Lib, void *Buffer, ppInt32 Length );
    
    
    
    /* Description
       Creates new file PDF Stream by filename and open mode.
       Parameters
       Lib :       [in] PDF Library Handle.
       FileName :  [in] PDF File Name, text string.
       OpenMode :  [in] Open Mode \: read or write.
       Returns
       \File PDF Stream Handle.
       See Also
       ULStrmClose                                            */
    PDFStreamHandle ULFileStrmNew( PDFLibHandle Lib, char *FileName, ppFileOpenMode OpenMode );
    
    
    
    /* Description
       Creates new file PDF Stream by PDF File Handle.
       Parameters
       Lib :         [in] PDF Library Handle.
       FileHandle :  [in] PDF File Handle.
       Returns
       \File PDF Stream Handle.
       See Also
       ULStrmClose                                     */
    PDFStreamHandle ULFileHandleStrmNew( PDFLibHandle Lib, PDFFileHandle FileHandle );
    
    
    
    /* Description
       Closes PDF Stream.
       Parameters
       Stream :  [in] PDF Stream Handle.
       Returns
       None.
       See Also
       ULMemStrmNew, ULFileStrmNew       */
    void ULStrmClose( PDFStreamHandle Stream );
    
    
    
    /* Description
       Sets new Stream position.
       Parameters
       Stream :       [in] PDF Stream Handle.
       NewPosition :  [in] New position in bytes ( offset from
                      beginning ).
       Returns
       Stream position after setting.
       See Also
       ULStrmGetPosition                                       */
    ppInt32 ULStrmSetPosition( PDFStreamHandle Stream, ppInt32 NewPosition );
    
    
    
    /* Description
       Gets Stream position ( offset from start of stream ).
       Parameters
       Stream :  [in] PDF Stream Handle.
       Returns
       Stream position.
       See Also
       ULStrmSetPosition                                     */
    ppInt32 ULStrmGetPosition( PDFStreamHandle Stream );
    
    
    
    /* Description
       Gets Stream size in bytes.
       Parameters
       Stream :  [in] PDF Stream Handle.
       Returns
       Stream size in bytes.
       See Also
       ULStrmSetSize                     */
    ppInt32 ULStrmGetSize( PDFStreamHandle Stream );
    
    
    
    /* Description
       Sets new Stream size in bytes. Enlarges stream capacity.
       Parameters
       Stream :  [in] PDF Stream Handle.
       Size :    [in] New size of PDF Stream.
       Returns
       Stream size in bytes.
       See Also
       ULStrmGetSize                                            */
    ppInt32 ULStrmSetSize( PDFStreamHandle Stream, ppInt32 Size );
    
    
    
    /* Description
       Reads from PDF Stream to memory buffer some count of the
       bytes.
       Parameters
       Stream :  [in] PDF Stream Handle.
       Buffer :  [out] Memory buffer for data.
       Count :   [in] Count of bytes which we want to read from
                 Stream.
       Returns
       Count of bytes which read from Stream.
       See Also
       ULStrmWriteBuffer                                        */
    ppInt32 ULStrmReadBuffer( PDFStreamHandle Stream, void *Buffer, ppInt32 Count );
    
    
    
    /* Description
       Writes from memory buffer to PDF Stream some count of the
       bytes.
       Parameters
       Stream :  [out] PDF Stream Handle.
       Buffer :  [in] Memory buffer with data.
       Count :   [in] Count of bytes which we want to write in
                 Stream.
       Returns
       Count of written bytes in Stream.
       See Also
       ULStrmReadBuffer                                          */
    ppInt32 ULStrmWriteBuffer( PDFStreamHandle Stream, void *Buffer, ppInt32 Count );
    
    
    
    /* Description
       Clears PDF Stream with initializing size ( maybe zero ).
       Parameters
       Stream :  [in] PDF Stream Handle.
       Size :    [in] Initializing size of PDF Stream in bytes.
       Returns
       None.                                                    */
    void ULStrmClear( PDFStreamHandle Stream, ppUns32 Size );
    
    
    
    /* Description
       Copies from one Stream in another Stream.
       Parameters
       FromStream :  [in] PDF Stream Handle.
       ToStream :    [in] PDF Stream Handle.
       Returns
       Size of bytes which is copied.            */
    ppInt32 ULStrmCopyToStrm( PDFStreamHandle FromStream, PDFStreamHandle ToStream );
    
    
    
    /* Description
       Reads one character from Stream.
       Parameters
       Stream :  [in] PDF Stream Handle.
       Returns
       One character form Stream. If it returns -1 than it is EOF (
       end of stream ).
       See Also
       ULStrmWriteChar                                              */
    ppInt32 ULStrmReadChar( PDFStreamHandle Stream );
    
    
    
    /* Description
       Reads one character from Stream. Same as ULStrmReadChar, only
       stream position stays on that place.
       Parameters
       Stream :  [in] PDF Stream Handle.
       Returns
       One character form Stream. If it returns -1 than it is EOF (
       end of stream ).
       See Also
       ULStrmWriteChar                                               */
    ppInt32 ULStrmLookChar( PDFStreamHandle Stream );
    
    
    
    /* Description
       Reads one line from Stream. Line is text string to character
       EOL ( end of line )
       Parameters
       Stream :  [in] PDF Stream Handle.
       Returns
       Line from Stream. Text string terminated by zero. Must be
       free after use.                                              */
    char *ULStrmReadLine( PDFStreamHandle Stream );
    
    
    
    /* Description
       Writes one character to Stream.
       Parameters
       Stream :     [in] PDF Stream Handle.
       Character :  [in] Writing data in size of one byte.
       Returns
       Count of written bytes in Stream.
       See Also
       ULStrmReadChar                                      */
    ppInt32 ULStrmWriteChar( PDFStreamHandle Stream, char Character );

#ifdef __cplusplus
}
#endif

#endif /* #ifndef VSBaseA_H */
