/*
 * program: psfilt
 * file: troff.c
 *
 * Copyright  1992 1993 Robert Joop
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: troff.c,v $
 * Revision 1.3  1994/07/09  16:44:21  rj
 * a lot of const's removed and added
 *
 * Revision 1.2  1994/01/09  23:46:21  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.1.1.1  1993/12/31  20:56:43  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: troff.c,v 1.3 1994/07/09 16:44:21 rj Exp $";

#include <stdarg.h>
#include "psfilt.h"
#include "avl.h"
#include "ucs2list.h"
#include "adobe.h"
#include "troff.h"

/*\[banner "troff characters"]----------------------------------------------------------------------------------------------------*/
typedef struct
{
  cstring	input_name;
#ifdef DEBUG
  cstring	character_name;
#endif
  t_ucs2	ucs2;
} t_troff_charmap;

#ifdef DEBUG
#  define CHAR(iname, charname, ucs2)	iname, charname, ucs2
#else
#  define CHAR(iname, charname, ucs2)	iname, ucs2
#endif

static int chrmapcmp (const t_troff_charmap **lhs, const t_troff_charmap **rhs)
{
  return strcmp ((*lhs)->input_name, (*rhs)->input_name);
}

static const t_troff_charmap troff_standard_chars[] =
{
  CHAR ("em", "3/4 Em dash", UCS2_em_dash),
  CHAR ("hy", "hyphen", UCS2_hyphen),
  CHAR ("bu", "bullet", UCS2_bullet),
  CHAR ("sq", "square", UCS2_white_square),
  CHAR ("ru", "rule", UCS2_low_line),
  CHAR ("14", "1/4", UCS2_fraction_one_quarter),
  CHAR ("12", "1/2", UCS2_fraction_one_half),
  CHAR ("34", "3/4", UCS2_fraction_three_quarters),
  CHAR ("fi", "fi", UCS2_latin_small_ligature_fi),
  CHAR ("fl", "fl", UCS2_latin_small_ligature_fl),
  CHAR ("ff", "ff", UCS2_latin_small_ligature_ff),
  CHAR ("Fi", "ffi", UCS2_latin_small_ligature_ffi),
  CHAR ("Fl", "ffl", UCS2_latin_small_ligature_ffl),
  CHAR ("de", "degree", UCS2_degree_sign),
  CHAR ("dg", "dagger", UCS2_dagger),
  CHAR ("fm", "foot mark", UCS2_prime), /* ??? */
  CHAR ("ct", "cent sign", UCS2_cent_sign),
  CHAR ("rg", "registered", UCS2_registered_sign),
  CHAR ("co", "copyright", UCS2_copyright_sign),
};

static const t_troff_charmap troff_special_chars[] =
{
  CHAR ("pl", "math plus", UCS2_plus_sign),
  CHAR ("mi", "math minus", UCS2_minus_sign),
  CHAR ("eq", "math equals", UCS2_equals_sign),
  CHAR ("**", "math star", UCS2_asterisk_operator), /* ??? */
  CHAR ("sc", "section", UCS2_section_sign),
  CHAR ("aa", "acute accent", UCS2_acute_accent),
  CHAR ("ga", "grave accent", UCS2_spacing_grave),
  CHAR ("ul", "underrule", UCS2_spacing_underscore),
  CHAR ("sl", "slash", UCS2_slash),

  CHAR ("*a", "alpha", UCS2_greek_small_letter_alpha),
  CHAR ("*b", "beta", UCS2_greek_small_letter_beta),
  CHAR ("*g", "gamma", UCS2_greek_small_letter_gamma),
  CHAR ("*d", "delta", UCS2_greek_small_letter_delta),
  CHAR ("*e", "epsilon", UCS2_greek_small_letter_epsilon),
  CHAR ("*z", "zeta", UCS2_greek_small_letter_zeta),
  CHAR ("*y", "eta", UCS2_greek_small_letter_eta),
  CHAR ("*h", "theta", UCS2_greek_small_letter_theta),
  CHAR ("*i", "iota", UCS2_greek_small_letter_iota),
  CHAR ("*k", "kappa", UCS2_greek_small_letter_kappa),
  CHAR ("*l", "lambda", UCS2_greek_small_letter_lambda),
  CHAR ("*m", "mu", UCS2_greek_small_letter_mu),
  CHAR ("*n", "nu", UCS2_greek_small_letter_nu),
  CHAR ("*c", "xi", UCS2_greek_small_letter_xi),
  CHAR ("*o", "omicron", UCS2_greek_small_letter_omicron),
  CHAR ("*p", "pi", UCS2_greek_small_letter_pi),
  CHAR ("*r", "rho", UCS2_greek_small_letter_rho),
  CHAR ("*s", "sigma", UCS2_greek_small_letter_sigma),
  CHAR ("ts", "terminal sigma", UCS2_greek_small_letter_final_sigma),
  CHAR ("*t", "tau", UCS2_greek_small_letter_tau),
  CHAR ("*u", "upsilon", UCS2_greek_small_letter_upsilon),
  CHAR ("*f", "phi", UCS2_greek_small_letter_phi),
  CHAR ("*x", "chi", UCS2_greek_small_letter_chi),
  CHAR ("*q", "psi", UCS2_greek_small_letter_psi),
  CHAR ("*w", "omega", UCS2_greek_small_letter_omega),

  CHAR ("*A", "Alpha", UCS2_greek_capital_letter_alpha),
  CHAR ("*B", "Beta", UCS2_greek_capital_letter_beta),
  CHAR ("*G", "Gamma", UCS2_greek_capital_letter_gamma),
  CHAR ("*D", "Delta", UCS2_greek_capital_letter_delta),
  CHAR ("*E", "Epsilon", UCS2_greek_capital_letter_epsilon),
  CHAR ("*Z", "Zeta", UCS2_greek_capital_letter_zeta),
  CHAR ("*Y", "Eta", UCS2_greek_capital_letter_eta),
  CHAR ("*H", "Theta", UCS2_greek_capital_letter_theta),
  CHAR ("*I", "Iota", UCS2_greek_capital_letter_iota),
  CHAR ("*K", "Kappa", UCS2_greek_capital_letter_kappa),
  CHAR ("*L", "Lambda", UCS2_greek_capital_letter_lambda),
  CHAR ("*M", "Mu", UCS2_greek_capital_letter_mu),
  CHAR ("*N", "Nu", UCS2_greek_capital_letter_nu),
  CHAR ("*C", "Xi", UCS2_greek_capital_letter_xi),
  CHAR ("*O", "Omicron", UCS2_greek_capital_letter_omicron),
  CHAR ("*P", "Pi", UCS2_greek_capital_letter_pi),
  CHAR ("*R", "Rho", UCS2_greek_capital_letter_rho),
  CHAR ("*S", "Sigma", UCS2_greek_capital_letter_sigma),
  CHAR ("*T", "Tau", UCS2_greek_capital_letter_tau),
  CHAR ("*U", "Upsilon", UCS2_greek_capital_letter_upsilon),
  CHAR ("*F", "Phi", UCS2_greek_capital_letter_phi),
  CHAR ("*X", "Chi", UCS2_greek_capital_letter_chi),
  CHAR ("*Q", "Psi", UCS2_greek_capital_letter_psi),
  CHAR ("*W", "Omega", UCS2_greek_capital_letter_omega),

  CHAR ("sr", "square root", UCS2_square_root),
  CHAR ("rn", "root en extender", UCS2_overline),
  CHAR (">=", ">=", UCS2_greater_than_or_equal_to),
  CHAR ("<=", "<=", UCS2_less_than_or_equal_to),
  CHAR ("==", "identically equal", UCS2_identical_to),
  CHAR ("~=", "approx =", UCS2_approximately_equal_to),
  CHAR ("ap", "approximates", UCS2_approximately_but_not_actually_equal_to), /* ??? */
  CHAR ("!=", "not equal", UCS2_not_equal_to),
  CHAR ("->", "right arrow", UCS2_rightwards_arrow),
  CHAR ("<-", "left arrow", UCS2_leftwards_arrow),
  CHAR ("ua", "up arrow", UCS2_upwards_arrow),
  CHAR ("da", "down arrow", UCS2_downwards_arrow),
  CHAR ("mu", "multiply", UCS2_multiplication_sign),
  CHAR ("di", "divide", UCS2_division_sign),
  CHAR ("+-", "plus-minus", UCS2_plus_or_minus_sign),
  CHAR ("cu", "cup (union)", UCS2_union),
  CHAR ("ca", "cap (intersection)", UCS2_intersection),
#if 0 /* *** ??? *** */
  CHAR ("sb", "subset of", UCS2_proper_inclusion_in_set),
  CHAR ("sp", "superset of", UCS2_properly_includes_in_set),
  CHAR ("ib", "improper subset", UCS2_contained_in_or_equals),
  CHAR ("ip", "improper superset", UCS2_contains_or_equals),
#endif /* *** ??? *** */
  CHAR ("if", "infinity", UCS2_infinity),
  CHAR ("pd", "partial derivative", UCS2_partial_differential), /* ???:  partial_derivative */
  CHAR ("gr", "gradient", UCS2_nabla),
  CHAR ("no", "not", UCS2_not_sign),
  CHAR ("is", "integral sign", UCS2_integral),
  CHAR ("pt", "proportional to", UCS2_proportional_to),
  CHAR ("es", "empty set", UCS2_empty_set),
  CHAR ("mo", "member of", UCS2_element_of),
  CHAR ("br", "box vertical rule", UCS2_vertical_bar),
  CHAR ("dd", "double dagger", UCS2_double_dagger),
  CHAR ("rh", "right hand", UCS2_rightwards_double_arrow),
  CHAR ("lh", "left hand", UCS2_leftwards_double_arrow),
  CHAR ("bs", "Bell System logo", UCS2_black_heart_suit),
  CHAR ("or", "or", UCS2_vertical_bar),
  CHAR ("ci", "circle", UCS2_large_circle),
#if 0 /* *** ??? *** */
  CHAR ("lt", "left top of big curly bracket", UCS2_top_portion_of_left_brace),
  CHAR ("lb", "left bottom", UCS2_bottom_portion_of_left_brace),
  CHAR ("rt", "right top", UCS2_top_portion_of_right_brace),
  CHAR ("rb", "right bot", UCS2_bottom_portion_of_right_brace),
  CHAR ("lk", "left center of big curly bracket", UCS2_center_portion_of_left_brace),
  CHAR ("rk", "right center of big curly bracket", UCS2_center_portion_of_right_brace),
  CHAR ("bv", "bold vertical", UCS2_extension_of_multiline_brace),
  CHAR ("lf", "left floor (left bottom of big square bracket)", UCS2_bottom_portion_of_left_bracket),
  CHAR ("rf", "right floor (right bottom)", UCS2_bottom_portion_of_right_bracket),
  CHAR ("lc", "left ceiling (left top)", UCS2_top_portion_of_left_bracket),
  CHAR ("rc", "right ceiling (right top)", UCS2_top_portion_of_right_bracket),
#endif /* *** ??? *** */
};

static const t_troff_charmap ditroff_special_chars[] =
{
  CHAR ("PL", "", UCS2_latin_capital_letter_l_slash),
  CHAR ("al", "", UCS2_infinity),
  CHAR ("l<", "", UCS2_bra),
  CHAR ("r>", "", UCS2_ket),
  CHAR ("<>", "", UCS2_left_right_double_arrow),
/*
  CHAR ("io", "", UCS2_double_double_arrow),
*/
  CHAR ("<:", "", UCS2_leftwards_double_arrow),
/*
  CHAR ("lh", "", UCS2_leftwards_double_arrow),
*/
  CHAR (":>", "", UCS2_rightwards_double_arrow),
  CHAR ("im", "", UCS2_rightwards_double_arrow),
/*
  CHAR ("rh", "", UCS2_rightwards_double_arrow),
*/
  CHAR ("be", "", UCS2_breve),
  CHAR ("hc", "", UCS2_caron), /* hachek */
  CHAR ("cr", "", UCS2_symbol_for_carriage_return),
  CHAR ("cd", "", UCS2_cedilla),
#if 0 /* *** ??? *** */
  CHAR ("ax", "", UCS2_abstract_multiply),
  CHAR ("a+", "", UCS2_abstract_plus),
#endif /* *** ??? *** */
  CHAR ("dm", "", UCS2_black_diamond_suit),
  CHAR ("..", "", UCS2_diaeresis),
  CHAR ("dt", "", UCS2_dot_above),
  CHAR ("ui", "", UCS2_latin_small_letter_dotless_i),
  CHAR ("m.", "", UCS2_n_ary_product), /* ???: scalar_product */
  CHAR ("cm", "", UCS2_element_of),
  CHAR ("en", "", UCS2_en_dash),
  CHAR ("!!", "", UCS2_inverted_exclamation_mark),
  CHAR ("te", "", UCS2_there_exists),
  CHAR ("$D", "", UCS2_latin_small_letter_script_f), /* florin */
  CHAR ("ss", "", UCS2_latin_small_letter_sharp_s),
/*
  CHAR ("gr", "", UCS2_nabla),
*/
  CHAR ("d<", "", UCS2_left_pointing_guillemet),
  CHAR ("d>", "", UCS2_right_pointing_guillemet),
  CHAR ("an", "", UCS2_logical_and),
  CHAR ("la", "", UCS2_logical_and),
  CHAR ("lo", "", UCS2_logical_or),
  CHAR ("Pl", "", UCS2_latin_small_letter_l_slash),
  CHAR ("mc", "", UCS2_macron),
  CHAR ("ma", "", UCS2_macron),
/*
  CHAR ("fm", "", UCS2_prime),
*/
  CHAR ("mt", "", UCS2_prime),
  CHAR ("!m", "", UCS2_not_an_element_of),
  /* ????
  CHAR ("!s", "", UCS2_is_not_contained_in),
  */
  CHAR ("og", "", UCS2_ogonek),
  CHAR ("pp", "", UCS2_paragraph_sign),
  CHAR ("bt", "", UCS2_up_tack), /* perpendicular */
  CHAR ("pm", "", UCS2_per_mille_sign),
  CHAR ("??", "", UCS2_inverted_question_mark),
  CHAR ("lq", "", UCS2_left_double_quotation_mark),
  CHAR ("rq", "", UCS2_right_double_quotation_mark),
  CHAR ("n'", "", UCS2_apostrophe),
  CHAR ("ri", "", UCS2_ring_above),
  CHAR ("sd", "", UCS2_double_prime),
  CHAR ("ps", "", UCS2_pound_sign),
  CHAR ("po", "", UCS2_pound_sign),
  CHAR ("tf", "", UCS2_therefore),
  CHAR ("tm", "", UCS2_trademark),
  CHAR ("fa", "", UCS2_for_all),
  CHAR ("yn", "", UCS2_yen_sign),
  CHAR ("$J", "", UCS2_yen_sign),

  CHAR ("ob", "", UCS2_white_bullet),
  CHAR ("bx", "", UCS2_black_square),
  CHAR ("vr", "vertical rule", UCS2_vertical_bar),

  CHAR ("13", "", UCS2_fraction_one_third),
  CHAR ("23", "", UCS2_fraction_two_thirds),
  CHAR ("18", "", UCS2_fraction_one_eighth),
  CHAR ("38", "", UCS2_fraction_three_eighths),
  CHAR ("58", "", UCS2_fraction_five_eighths),
  CHAR ("78", "", UCS2_fraction_seven_eighths),
};

static const t_troff_charmap groff_special_chars[] =
{
  CHAR ("sh", "", UCS2_number_sign),
  CHAR ("Do", "", UCS2_dollar_sign),
  CHAR ("at", "", UCS2_commercial_at),
  CHAR ("lB", "", UCS2_opening_square_bracket),
  CHAR ("rB", "", UCS2_closing_square_bracket),
  CHAR ("rs", "", UCS2_backslash),
  CHAR ("lC", "", UCS2_opening_curly_bracket),
  CHAR ("ba", "", UCS2_vertical_bar),
  CHAR ("rC", "", UCS2_closing_curly_bracket),
  CHAR ("Fn", "", UCS2_latin_small_letter_script_f), /* florin */

  CHAR ("a^", "circumflex", UCS2_spacing_circumflex),
  CHAR ("a~", "tilde", UCS2_tilde),
  CHAR ("a\"", "hungarumlaut", UCS2_double_acute_accent),
  CHAR ("a.", "dotaccent", UCS2_dot_above),
  CHAR ("ab", "breve", UCS2_breve),
  CHAR ("ah", "caron", UCS2_caron), /* hachek */
  CHAR ("ao", "ring", UCS2_ring_above),
  CHAR ("ho", "ogonek", UCS2_ogonek),
  CHAR ("ad", "dieresis", UCS2_diaeresis),
  CHAR ("a-", "macron", UCS2_macron),
  CHAR ("ac", "cedilla", UCS2_cedilla),
  CHAR (".i", "dotlessi", UCS2_latin_small_letter_dotless_i),

  CHAR ("oq", "", UCS2_left_single_quotation_mark),
  CHAR ("aq", "", UCS2_apostrophe),
  CHAR ("bq", "", UCS2_low_single_comma_quotation_mark),
  CHAR ("Bq", "", UCS2_low_double_comma_quotation_mark),
  CHAR ("Fo", "", UCS2_left_pointing_guillemet),
  CHAR ("Fc", "", UCS2_right_pointing_guillemet),
  CHAR ("fo", "", UCS2_left_pointing_single_guillemet),
  CHAR ("fc", "", UCS2_right_pointing_single_guillemet),

  CHAR ("f/", "", UCS2_fraction_slash),
  CHAR ("%0", "", UCS2_per_mille_sign),
  CHAR ("r!", "", UCS2_inverted_exclamation_mark),
  CHAR ("r?", "", UCS2_inverted_question_mark),
  CHAR ("Po", "", UCS2_pound_sign),
  CHAR ("Cs", "", UCS2_currency_sign),
  CHAR ("Ye", "", UCS2_yen_sign),
  CHAR ("bb", "", UCS2_broken_bar),
  CHAR ("Of", "ordfeminine", UCS2_feminine_ordinal_indicator),
  CHAR ("Om", "ordmasculine", UCS2_masculine_ordinal_indicator),
/* undefined:
  CHAR ("S1", "onesuperior", UCS2_superscript_digit_one),
*/
  CHAR ("S2", "twosuperior", UCS2_superscript_digit_two),
  CHAR ("S3", "threesuperior", UCS2_superscript_digit_three),
/* double (ditroff):
  CHAR ("ps", "", UCS2_paragraph_sign),
*/
  CHAR ("TP", "Thorn", UCS2_latin_capital_letter_thorn),
  CHAR ("Tp", "thorn", UCS2_latin_small_letter_thorn),
  CHAR ("-D", "Eth", UCS2_latin_capital_letter_eth),
  CHAR ("Sd", "eth", UCS2_latin_small_letter_eth),
};

/* private extensions: */
static const t_troff_charmap private_special_chars[] =
{
  CHAR ("tr", "", UCS2_telephone_recorder),
/*
  CHAR ("wc", "wheel chair", UCS2_handicapped),
*/
  CHAR ("yy", "", UCS2_yin_yang),
  CHAR ("sm", "smile", UCS2_smile),
  CHAR ("fr", "frown", UCS2_frown),

  CHAR ("15", "", UCS2_fraction_one_fifth),
  CHAR ("25", "", UCS2_fraction_two_fifths),
  CHAR ("35", "", UCS2_fraction_three_fifths),
  CHAR ("45", "", UCS2_fraction_four_fifths),
  CHAR ("16", "", UCS2_fraction_one_sixth),
  CHAR ("56", "", UCS2_fraction_five_sixths),
};

static t_avl	*chars = NULL;

static void init_chars()
{
  int			len;
  const t_troff_charmap	*table;

  assert (chars = avlcreate (sizeof (t_troff_charmap *), chrmapcmp, NULL));
  for (table=troff_standard_chars, len=ARRAYDIM (troff_standard_chars); len--; table++)
    assert (avlinsert (chars, &table));
  for (table=troff_special_chars, len=ARRAYDIM (troff_special_chars); len--; table++)
    assert (avlinsert (chars, &table));
  for (table=ditroff_special_chars, len=ARRAYDIM (ditroff_special_chars); len--; table++)
    assert (avlinsert (chars, &table));
  for (table=groff_special_chars, len=ARRAYDIM (groff_special_chars); len--; table++)
    assert (avlinsert (chars, &table));
  for (table=private_special_chars, len=ARRAYDIM (private_special_chars); len--; table++)
    assert (avlinsert (chars, &table));
/*
    if (!avlinsert (chars, &table))
      error ("can't insert `%s'", table->input_name);
*/
}

const t_xucs2 *troffchar2ucs2 (cstring name)
{
  t_troff_charmap	search, *sp = &search, **found;

  if (!chars)
    init_chars();

  search.input_name = name;
  if (found = avlsearch (chars, &sp))
  {
    static t_xucs2	xucs2;

    xucs2.count = 1;
    *xucs2.codes = (*found)->ucs2;
    return &xucs2;
  }
  else
    return troffdiacritic2ucs2 (name);
}

/*\[banner "troff strings"]-------------------------------------------------------------------------------------------------------*/
typedef struct
{
  cstring	name;
  t_ucs2	ucs2;
} t_troff_stringmap;

static int strmapcmp (const t_troff_stringmap **lhs, const t_troff_stringmap **rhs)
{
  return strcmp ((*lhs)->name, (*rhs)->name);
}

static void init_strings (t_avl **strings, ...)
{
  va_list		va;
  t_troff_stringmap	*table;
  int			len;

  assert (*strings = avlcreate (sizeof (t_troff_stringmap *), strmapcmp, NULL));

  va_start (va, strings);
  while (table = va_arg (va, t_troff_stringmap *))
    for (len = va_arg (va, int); len--; table++)
      assert (avlinsert (*strings, &table));
  va_end (va);
}

static const t_troff_stringmap troff_ms_1strings[] =
{
  "!", UCS2_inverted_exclamation_mark,
  "?", UCS2_inverted_question_mark,
  "8", UCS2_latin_small_letter_sharp_s,
  "3", UCS2_latin_small_letter_yogh,
#if 0
  "q", UCS2_,	/* "hooked o" */
#endif
};

static const t_troff_stringmap troff_german_1strings[] =
{
  "a", UCS2_latin_small_letter_a_diaeresis,
  "o", UCS2_latin_small_letter_o_diaeresis,
  "u", UCS2_latin_small_letter_u_diaeresis,
  "s", UCS2_latin_small_letter_sharp_s,
  "A", UCS2_latin_capital_letter_a_diaeresis,
  "O", UCS2_latin_capital_letter_o_diaeresis,
  "U", UCS2_latin_capital_letter_u_diaeresis,
};

static t_avl	*string1 = NULL;

const t_xucs2 *troff1str2ucs2 (cstring name)
{
  t_troff_stringmap		search;
  const t_troff_stringmap	*sp = &search, **found;

  if (!string1)
    init_strings (&string1, troff_ms_1strings, ARRAYDIM (troff_ms_1strings), troff_german_1strings, ARRAYDIM (troff_german_1strings), NULL);

  search.name = name;
  if (found = (const t_troff_stringmap **)avlsearch (string1, &sp))
  {
    static t_xucs2	xucs2;

    xucs2.count = 1;
    *xucs2.codes = (*found)->ucs2;
    return &xucs2;
  }
  else
    return NULL;
}

static const t_troff_stringmap troff_ms_2strings[] =
{
  "Th", UCS2_latin_capital_letter_thorn,
  "th", UCS2_latin_small_letter_thorn,
  "D-", UCS2_latin_capital_letter_eth,
  "d-", UCS2_latin_small_letter_eth,
  "ae", UCS2_latin_small_letter_a_e,
  "AE", UCS2_latin_capital_letter_a_e,
  "oe", UCS2_latin_small_ligature_oe,
  "OE", UCS2_latin_capital_ligature_oe,
};

static t_avl	*string2 = NULL;

const t_xucs2 *troff2str2ucs2 (cstring name)
{
  t_troff_stringmap		search;
  const t_troff_stringmap	*sp = &search, **found;

  if (!string2)
    init_strings (&string2, troff_ms_2strings, ARRAYDIM (troff_ms_2strings), NULL);

  search.name = name;
  if (found = (const t_troff_stringmap **)avlsearch (string2, &sp))
  {
    static t_xucs2	xucs2;

    xucs2.count = 1;
    *xucs2.codes = (*found)->ucs2;
    return &xucs2;
  }
  else
    return NULL;
}

/*\[banner "troff strings: diacritics"]-------------------------------------------------------------------------------------------*/
/* troff accent marks can simply be mapped into postscript names: */

typedef struct
{
  char		troffmark;
  cstring	psname;
} t_accent;

static const t_accent accents[] =
{
  '\'', "acute",
  '`', "grave",
  '^', "circumflex",
  ',', "cedilla",
  '~', "tilde",
  ':', "dieresis",
  'v', "caron", /* hachek */
  '_', "macron",
  '.', "underdot",
  '/', "slash",
  'o', "ring",
};

const t_xucs2 *troffdiacritic2ucs2 (cstring troffname)
{
  const t_accent	*accent;
  int			i;
  t_buf			psname;

  for (accent=accents, i=0; i<ARRAYDIM (accents); accent++, i++)
    if (accent->troffmark == troffname[0])
      break;
  if (i == ARRAYDIM (accents))
    return NULL;
  sprintf (psname, "%c%s", troffname[1], accent->psname);
  return ps2ucs2 (psname);
}

/*\[banner "troff fonts"]---------------------------------------------------------------------------------------------------------*/
typedef struct
{
  string	troffname,
		postscriptname;
} t_troff_fontmap;

static int fontmappcmp (t_troff_fontmap **lhs, t_troff_fontmap **rhs)
{
  return strcmp ((*lhs)->troffname, (*rhs)->troffname);
}

/* table from Chris Lewis' psroff, release 3.0, file: lib/ps.fonts	*/
static const t_troff_fontmap fontmap[] =
{
  "S", "Symbol",

  "R", "Times-Roman",
  "I", "Times-Italic",
  "B", "Times-Bold",
  "BI", "Times-BoldItalic",

  "AR", "AvantGarde-Book",
  "AI", "AvantGarde-BookOblique",
  "AB", "AvantGarde-Demi",
  "AX", "AvantGarde-DemiOblique",

  "BR", "Bookman-Light",
  "BO", "Bookman-LightItalic",
  "BB", "Bookman-Demi",
  "BX", "Bookman-DemiItalic",

  "C", "Courier",
  "CO", "Courier-Oblique",
  "CB", "Courier-Bold",
  "CX", "Courier-BoldOblique",

  "H", "Helvetica",
  "HO", "Helvetica-Oblique",
  "HB", "Helvetica-Bold",
  "HX", "Helvetica-BoldOblique",

  "Hn", "Helvetica-Narrow",
  "Ho", "Helvetica-Narrow-Oblique",
  "Hb", "Helvetica-Narrow-Bold",
  "Hx", "Helvetica-Narrow-BoldOblique",

  "NR", "NewCenturySchlbk-Roman",
  "NI", "NewCenturySchlbk-Italic",
  "NB", "NewCenturySchlbk-Bold",
  "NX", "NewCenturySchlbk-BoldItalic",

  "PR", "Palatino-Roman",
  "PI", "Palatino-Italic",
  "PB", "Palatino-Bold",
  "PX", "Palatino-BoldItalic",

  "ZC", "ZapfChancery-MediumItalic",
  "ZD", "ZapfDingbats",

  "HW", "HaeberliWriting",
  "TE", "Tymes-Elfin",

  "N",  "Helvetica-Narrow",
#if 0	/* NI, NB and NX collide with NewCenturySchlbk */
  "NB", "Helvetica-Narrow-Bold",
  "ND", "Helvetica-Narrow-BoldOblique",
  "NO", "Helvetica-Narrow-Oblique",
#endif

  "X",  "NewCenturySchlbk-Roman",
  "XB", "NewCenturySchlbk-Bold",
  "XD", "NewCenturySchlbk-BoldItalic",
  "XI", "NewCenturySchlbk-Italic",
};

static t_avl	*fonts = NULL;

static void init_fonts()
{
  int			len;
  const t_troff_fontmap	*table;

  assert (fonts = avlcreate (sizeof (t_troff_fontmap *), fontmappcmp, NULL));
  for (table=fontmap, len=ARRAYDIM (fontmap); len--; table++)
    assert (avlinsert (fonts, &table));
}

string troff2psfontbyname (string name)
{
  t_troff_fontmap	search, *sp = &search, **found;

  if (!fonts)
    init_fonts();

  search.troffname = name;
  if (found = avlsearch (fonts, &sp))
    return (*found)->postscriptname;
  else
    return NULL;
}

string troff2psfontbynumber (int number)
{
  static string	pos[] =
  {
    NULL, "R", "I", "B", "S",
  };
  return 0 < number && number < ARRAYDIM (pos) ? troff2psfontbyname (pos[number]) : NULL;
}
