/*
 * program: psfilt
 * file: psfilt.c
 *
 * Copyright  1992 1993 Robert Joop
 *
 * Main module
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: psfilt.c,v $
 * Revision 1.6  1994/08/18  12:13:19  rj
 * new input mode for Tcl scripts.
 *
 * Revision 1.5  1994/07/16  12:35:47  rj
 * provisions to use X11R6 PS type 1 fonts
 *
 * Revision 1.4  1994/07/09  16:44:13  rj
 * a lot of const's removed and added
 *
 * Revision 1.3  1994/03/18  12:07:40  rj
 * support for duplex mode added
 * lots of missing `#ifdef COLORED' inserted
 *
 * Revision 1.2  1994/01/09  23:46:17  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.1.1.1  1993/12/31  20:56:43  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: psfilt.c,v 1.6 1994/08/18 12:13:19 rj Exp $";

#include <pwd.h>

#include "psfilt.h"
#include "dirs.h"

#include "afm.h"
#include "ppd.h"
#include "job.h"
#include "troff.h"
#include "defaults.h"
#include "verbose.h"
#include "error.h"
#include "version.h"
#include "opt.h"

static cstring	cmdname;

static void usage (bool terse)
{
  FILE	*fp = terse ? stderr : stdout;
  fprintf (fp, "usage: %s [options] [files]\n", cmdname);
  if (terse)
    fprintf (fp, "for more help enter `%s -help`\n", cmdname);
  else
    fprintf (fp, "\
Options:
  -help
  -version		print version
  -s[ilent]		silent (suppress all output)
  -q[uiet]		quiet (suppress most output)
  -vn[ormal], -verbn[ormal]	default verbosity
  -v[erbose]		verbose
  -vv, -veryv[erbose]	very verbose
  -cache list		usage and behaviour of the afm cache
			list is a comma-separated list of
			{,no}{use,{check,ck}{,dir,files},update}
  -globalcachefile file	one global cache file
  -localcachefile file	one cache file in every directory
  -nofile[s]		don't process any input files (don't default to stdin)
  -o[utput] file	direct output to file
  -kern list		kerning switches
			list is a comma-separated list of
			{no,{,no}{pairwise,pairs,track}}
  -7			printer channel is not 8 bit clean
  -8			printer channel is 8 bit clean
  -bin[ary]		printer channel may transmit all 256 codes
  -(un)struct		produce (un)structured PostScript

  -printer[type] name	overrides $PRINTERTYPE
  -paper[type] name	overrides $PAPERTYPE; default: "DEFAULT_PAPERTYPE"
  -duplex mode		duplex (double sided) printing, e.g. DuplexTumble

  -afm[dir] dir		extend search path for .afm files
  -font[dir] dir	extend search path for font files (.pfb, .ps, .pfa, .gsf)
  -ppd[dir] dir		extend search path for .PPD files

  -sh[eet]hh height	sheet header height
  -filehh height	file header height
  -pagehh height	page header height
  -bannerh[eight] height
			banner height
  -userbannerh[eight] height
			user banner height
  -leftb[orderwidth] width,
  -lbw width		width of left border
  -rightb[orderwidth] width,
  -rbw width		width of right border
  -topb[orderwidth] width,
  -tbw width		width of top border
  -bottomb[orderwidth] width,
  -bbw width		width of bottom border
  -landscape		(paper orientation)
  -portrait		(paper orientation)
  -col[umns] columns	columns per sheet
"
#ifdef COLORED
"
  -lcolor		default color for literal mode
  -tcolor		default color for translated mode
  -color		default color for both modes
"
#endif
"
  -lff fontfamily	default font family for literal mode
  -tff fontfamily	default font family for translation mode
  -ff fontfamily	default font family for both modes
  -lfw fontweight	initial font weight for literal mode
  -tfw fontweight	initial font weight for translation mode
  -fw fontweight	initial font weight for both modes
  -lfi boolean		initial font slant for literal mode
  -tfi boolean		initial font slant for translation mode
  -fi boolean		initial font slant for both modes
  -lfsz fontsize	initial font size for literal mode
  -tfsz fontsize	initial font size for translation mode
  -fsz fontsize		initial font size for both modes
"/*
  -fs fontname		default symbol font
*/"
  -lnsp { f[ixed] | a[utomatic] } { height | *factor }
			height is absolute, *factor is relative
  -minlnsp height	minimum linespacing (absolute)
  -tabcolumns columns	input characters (after translation) to count for a tab stop
  -tabwidth { width | *factor }
			(output) width of a tab stop
			width is absolute, *factor is relative
  -[no]ignLF		(don't) ignore LF after FF
  -[no]igncharunavail	(don't) abort on characters unavailable

	height, width and fontsize may be specified in the following units:
 		p and pt (points, default), mm (millimeters),
		cm (centimeters), \" and in (inches).

  Input types:
    -text		translate all input
    -mail, -news	mail/news: headers are handled specially
    -man		recognize output of `nroff -man`
    -make[file]		translate in # ...
    -sh[ell]		translate in strings and # ...
    -c, -C		translate in /* ... */ and // ...
    -c++, -C++		translate in /* ... */ and // ...
    -perl		similar to shell, but different for \" and '
    -ps, -PS		translate in %% ...
    -lisp, -scm, -scheme
			translate in ; ...
    -tcl		translate in # ...

  Encodings:
    -[iso]646		assume ISO 646 encoding
    -ascii		assume ASCII encoding
    -[din]66003		assume DIN 66003 encoding
    -[iso]latin1, -8859-1 assume ISO Latin-1 encoding
    -cp437 -ibm		assume IBM Codepage 437 (a.k.a. \"IBM ASCII\") encoding

  Translation modes:
    -lit[eral]		don't recognize any escapes
    -tr[off]		recognize (some) `troff` escapes
    -mos		recognize (some) `troff -mos` escapes
    -ms			recognize (some) `troff -ms` escapes
    -tex, -TeX		recognize (some) TeX escapes

Files:
-		stands for standard input and ends the option list
*.[chly]	if no input type is specified, it defaults to -c
*.C *.cc	if no input type is specified, it defaults to -c++
[Mm]akefile	if no input type is specified, it defaults to -makefile
*.man		if no input type is specified, it defaults to -man
*.sh		if no input type is specified, it defaults to -sh
*.ps *.eps	if no input type is specified, it defaults to -ps
*.scm		if no input type is specified, it defaults to -scm
*.tcl		if no input type is specified, it defaults to -tcl
");
  exit (terse);
}

static FILE	*redirect = NULL;
static cstring	username = NULL;

t_cachespec	cache;
t_cachespecbools cachespec;

t_printerspec	printerspec;

t_input		input;
t_inputbools	inputspec;

t_inout		inout;
t_inoutbools	inoutspec;

t_goutput	goutput;
t_goutputbools	goutputspec;

t_foutput	foutput;
t_foutputbools	foutputspec;

static void filedefaults (t_input *input, t_inputbools *inputspec, t_inout *inout, t_inoutbools *inoutspec, t_foutput *foutput, t_foutputbools *foutputspec)
{
  t_iomode	iomode;

#define DEFAULT(member, value)	\
  {				\
    if (!inputspec->member)	\
    {				\
      input->member = value;	\
      inputspec->member = true;	\
    }				\
  }
  DEFAULT (type, IT_guess);
  switch (input->type)
  {
    case IT_MAN:
	DEFAULT (encoding, ENC_ISOlatin1)
	if (input->translation != TRLT_MAN)
	  say (IMP_warning, "-man allows no other input type!\n");
	input->translation = TRLT_MAN;	/* man input enforces man translation */
	break;
    case IT_Mail:
    case IT_News:
	DEFAULT (encoding, ENC_ISOlatin1)
	DEFAULT (translation, TRLT_guess)
	break;
    case IT_MAKEFILE:
	DEFAULT (translation, TRLT_none)
	break;
    default:
  }
  DEFAULT (encoding, DEFAULT_ENCODING)
  DEFAULT (translation, TRLT_none)
  DEFAULT (verbmode, VERB_MIXED)
  DEFAULT (tabcolumns, DEFAULT_TABCOLUMNS)
  DEFAULT (ignLFafterFF, true)
#undef DEFAULT

#define DEFAULT(member, value)	\
  {				\
    if (!inoutspec->member)	\
    {				\
      inout->member = value;	\
      inoutspec->member = true;	\
    }				\
  }
  switch (input->type)
  {
    case IT_MAN:
    case IT_TEXT:
      inout->mode = IOM_trlt;
      break;
    default:
      inout->mode = IOM_lit;
  }
  DEFAULT (tabmode, TMsimple)
#undef DEFAULT

#define DEFAULT(member, value)	\
  {				\
    if (!foutputspec->member)	\
    {				\
      foutput->member = value;	\
      foutputspec->member = true;\
    }				\
  }
  DEFAULT (headerheight.file, DEFAULT_FILEHEADERHEIGHT)
  DEFAULT (headerheight.page, DEFAULT_PAGEHEADERHEIGHT)
  DEFAULT (headerheight.banner, DEFAULT_BANNERHEADERHEIGHT)
  DEFAULT (headerheight.userbanner, DEFAULT_USERBANNERHEADERHEIGHT)
#ifdef COLORED
  {
    t_color nocolor;
    clearcolor (&nocolor);
    DEFAULT (color[IOM_lit], nocolor)
    DEFAULT (color[IOM_trlt], nocolor)
  }
#endif
  DEFAULT (font[IOM_lit].size, DEFAULT_FONTSIZE)
  DEFAULT (font[IOM_trlt].size, DEFAULT_FONTSIZE)
  switch (input->type)
  {
    case IT_MAN:
	DEFAULT (font[inout->mode].family, "Courier")
	if (!foutputspec->linespacing.height)
	{
	  foutput->linespacing.automatic = false;
	  foutput->linespacing.relative = false;
	  foutput->linespacing.height.factor = 1.2 * foutput->font[IOM_trlt].size;
	  foutputspec->linespacing.height = true;
	}
	break;
    default:
	DEFAULT (font[IOM_lit].family, "Times")
	DEFAULT (font[IOM_trlt].family, "Times")
	if (!foutputspec->linespacing.height)
	{
	  foutput->linespacing.automatic = true;
	  foutput->linespacing.relative = true;
	  foutput->linespacing.height.factor = 1.2;
	  foutputspec->linespacing.height = true;
	}
  }
  DEFAULT (linespacing.minheight, .8 * foutput->font[inout->mode].size);
  for (iomode=0; iomode<N_IOM; iomode++)
  {
    DEFAULT (font[iomode].rendition.weight, FONTWEIGHT_normal)
    DEFAULT (font[iomode].rendition.italic, iomode != inout->mode)
  }
  if (!foutputspec->tabs.width)
  {
    foutput->tabs.relative = true;
    if (streq (foutput->font[inout->mode].family, "Courier"))
      foutput->tabs.width.factor = 0.6;
    else
      foutput->tabs.width.factor = 0.5;
    foutputspec->tabs.width = true;
  }
  /* tablist */
#undef DEFAULT

  switch (inout->tabmode)
  {
    case TMsimple:
	say (IMP_entertain2, "tabcols=%d, fontsize=%g/%g", input->tabcolumns, foutput->font[IOM_lit].size, foutput->font[IOM_trlt].size);
	if (foutput->tabs.relative)
	  say (IMP_entertain2, "tabwidth=*%g\n", foutput->tabs.width.factor);
	else
	  say (IMP_entertain2, "tabwidth=%g\n", foutput->tabs.width.size);
	break;
    case TMlist:
	ierror ("tab lists are not yet implemented.\n");
	break;
  }
}

static void enqueue (t_list *jobs, string fn)
{
  t_job		job, *jp;
  struct
  {
    t_inputbools	input;
    t_inoutbools	inout;
    t_foutputbools	output;
  }		specs;

  say (IMP_debug, "enqueue (%s)\n", fn?fn:"<stdin>");
  if (!jobs)
    error ("illegal file %s after option -nofile.\n", fn?fn:"-");

  if (fn)
  {
    if (access (fn, R_OK))
      error ("can't access `%s'", fn);
    job.srctype = JOB_FILE;
    job.source.fn = fn;
  }
  else
    job.srctype = JOB_STDIN;

  job.input = input;
  specs.input = inputspec;
  job.inout = inout;
  specs.inout = inoutspec;
  job.output = foutput;
  specs.output = foutputspec;
  filedefaults (&job.input, &specs.input, &job.inout, &specs.inout, &job.output, &specs.output);

  if (job.input.type == IT_guess)
  {
    if (fn)
    {
      cstring bn = basename (fn);
      job.input.type = strmatch ("*.[chly]", bn) ? IT_C
			: strmatch ("*.cc", bn) || strmatch ("*.C", bn) ? IT_Cplusplus
			: strmatch ("*.ps", bn) || strmatch ("*.eps", bn)|| strmatch ("*.epsi", bn) || strmatch ("*.gsf", bn) || strmatch ("*.pfa", bn) ? IT_PS
			: strmatch ("*.sh", bn) ? IT_SHELL
			: strmatch ("*.pl", bn) || strmatch ("*.ph", bn) ? IT_PERL
			: strmatch ("*.man", bn) ? IT_MAN
			: strmatch ("*.scm", bn) ? IT_LISP
			: strmatch ("*.tcl", bn) ? IT_TCL
			: strmatch ("[Mm]akefile", bn) ? IT_MAKEFILE
			: IT_TEXT;
    }
    else
      job.input.type = IT_TEXT;
    say (IMP_entertain, "input type for `%s' defaults to `%s'\n", fn ? fn : "<stdin>", inputtypename (job.input.type));
  }

  jp = liappend (jobs, &job); assert (jp);
}

typedef struct
{
  cstring	name;
  int		value;
} t_optionbits;

static void optionbits (const t_optionbits *map, int maplen, cstring option, int *var)
{
  int	i;

  for (;; option += i+1)
  {
    bool	last, negate = false;
    int		flag;

    if (last = (i = chrind (option, ',')) == fault)
      i = strlen (option);
    if (i >= 2 && (negate = strneq (option, "no", 2)))
      option += 2, i -= 2;
    for (flag=0; flag<maplen; flag++)
      if (strlen (map[flag].name) == i && !memcmp (map[flag].name, option, i))
	break;
    if (flag == maplen)
      usage (true);
    if (negate)
      *var &=~ map[flag].value;
    else
      *var |= map[flag].value;
    if (last)
      return;
  }
}

static t_fontweight getweight (cstring arg)
{
  t_fontweight weight;

  if ((weight = fontweightbyname (arg)) == FONTWEIGHT_unknown)
    error ("%s: illegal weight `%s'", cmdname, arg);
  return weight;
}

static bool getbool (cstring arg)
{
  static cstring	names[] =
  {
    "no", "yes",
    "false", "true",
  };
  int	i;

  for (i=0; i<ARRAYDIM (names); i++)
    if (!strcasecmp (names[i], arg))
      return i&1;
  error ("%s: illegal boolean `%s'", cmdname, arg);
abort();/* standard C doesn't recognize volatile for non-returning functions. but gcc knows that abort() and exit() don't return... */
}

static t_point getsize (cstring arg)
{
  double		size;
  static cstring	units[] = { "p", "pt", "mm", "cm", "\"", "in" };
  t_buf			unit;

  switch (sscanf (arg, "%lg%s", &size, unit))
  {
    case 1:
	return size;					/* points */
    case 2:
	switch (strind (units, unit))
	{
	  case 0:
	  case 1:	return size;			/* points */
	  case 2:	return size / 25.4 * 72.;	/* millimeters */
	  case 3:	return size / 2.54 * 72.;	/* centimeters */
	  case 4:
	  case 5:	return size * 72.;		/* inches */
	}
	/* fall thru */
    default:
	error ("%s: argument `%s' is not a legal size", cmdname, arg);
	abort();/* standard C doesn't recognize volatile for non-returning functions. but gcc knows that abort() and exit() don't return... */
  }
}

#ifdef COLORED
static t_color getcolor (string arg)
{
  t_color	color;

  if (scancolor (arg, &color))
    error ("%s: illegal color `%s'", cmdname, arg);
  return color;
}
#else
static void colorwarning()
{
  say (IMP_warning, "color option ignored. (program hasn't been compiled with color support)");
}
#endif

static const t_optionbits cacheflagmap[] =
{
  { "use", CACHE_use },
  { "ck", CACHE_check },
  { "check", CACHE_check },
  { "ckdir", CACHE_checkdir },
  { "ckdirs", CACHE_checkdir },
  { "checkdir", CACHE_checkdir },
  { "checkdirs", CACHE_checkdir },
  { "ckfiles", CACHE_checkfiles },
  { "checkfiles", CACHE_checkfiles },
  { "upd", CACHE_update },
  { "update", CACHE_update },
};

static const t_optionbits kernflagmap[] =
{
  { "pairwise", KERN_pairwise },
  { "pairs", KERN_pairwise },
  { "track", KERN_track },
  { "", KERN_pairwise|KERN_track },
};

static void parse_args (int argc, string *argv, t_list **jobs)
{
  int	arg;
  bool	options = true;

#define CSPEC(member, value)	{ cache.member = value; cachespec.member = true; }
#define ISPEC(member, value)	{ input.member = value; inputspec.member = true; }
#define GOSPEC(member, value)	{ goutput.member = value; goutputspec.member = true; }
#define FOSPEC(member, value)	{ foutput.member = value; foutputspec.member = true; }

  for (arg=0; arg<argc; arg++)
  {
    if (argv[arg][0] == '-')
      if (!argv[arg][1])
      {
	enqueue (*jobs, NULL);
	continue;
      }
      elif (options)
      {
	t_names	*metaopt;

	if (argv[arg][1] == '-')
	  options = false;
	elif (metaopt = getmetaopt_lock (argv[arg]))
	{
	  parse_args (metaopt->count, metaopt->names, jobs);
	  metaopt_unlock (argv[arg]);
	}
	elif (streq ("-help", argv[arg]))
	  usage (false);
	elif (streq ("-version", argv[arg]))
	  printversion();
	elif (strnabbr ("-silent", argv[arg], 2))
	  verbosity = VERBOSITY_silent;
	elif (strnabbr ("-quiet", argv[arg], 2))
	  verbosity = VERBOSITY_quiet;
	elif (strnabbr ("-vnormal", argv[arg], 3) || strnabbr ("-verbnormal", argv[arg], 6))
	  verbosity = VERBOSITY_normal;
	elif (strnabbr ("-verbose", argv[arg], 2))
	  verbosity = VERBOSITY_verbose;
	elif (streq ("-vv", argv[arg]) || strnabbr ("-veryverbose", argv[arg], 6))
	  verbosity = VERBOSITY_veryverbose;
	elif (streq ("-globalcachefile", argv[arg]))
	{
	  if (arg+2 > argc)
	    usage (true);
	  cache.flags |= CACHE_global;
	  cachespec.flags = true;
	  CSPEC (filename, argv[++arg])
	}
	elif (streq ("-localcachefile", argv[arg]))
	{
	  if (arg+2 > argc)
	    usage (true);
	  cache.flags &= ~CACHE_global;
	  cachespec.flags = true;
	  CSPEC (filename, argv[++arg])
	}
	elif (streq ("-cache", argv[arg]))
	{
	  if (arg+2 > argc)
	    usage (true);
	  optionbits (cacheflagmap, ARRAYDIM (cacheflagmap), argv[++arg], &cache.flags);
	  cachespec.flags = true;
	}
	elif (streq ("-kern", argv[arg]))
	{
	  if (arg+2 > argc)
	    usage (true);
	  optionbits (kernflagmap, ARRAYDIM (kernflagmap), argv[++arg], &kernflags);
	}
	elif (strnabbr ("-nofiles", argv[arg], 7))
	{
	  if (*jobs && lilength (*jobs))
	    error ("illegal files (e.g. %s) before option -nofile.\n", jobname (lilast (*jobs)));
	  *jobs = lidestroy (*jobs);
	}
	elif (strnabbr ("-output", argv[arg], 2))
	{
	  if (arg+2 > argc)
	    usage (true);
	  elif (redirect)
	    error ("more than one -o option");
	  elif (!(redirect = fopen (argv[++arg], "w")))
	    error ("can't open `%s' for writing", argv[arg]);
	}
	elif (streq ("-7", argv[arg]))
	  dscdata = DSC_Clean7bit;
	elif (streq ("-8", argv[arg]))
	  dscdata = DSC_Clean8bit;
	elif (strnabbr ("-binary", argv[arg], 4))
	  dscdata = DSC_Binary;
	elif (streq ("-struct", argv[arg]))
	  structured = true;
	elif (streq ("-unstruct", argv[arg]))
	  structured = false;

	elif (strnabbr ("-afmdir", argv[arg], 4))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    afmdirs = appenddir (afmdirs, argv[++arg]);
	}
	elif (strnabbr ("-fontdir", argv[arg], 5))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    fontdirs = appenddir (fontdirs, argv[++arg]);
	}
	elif (strnabbr ("-ppddir", argv[arg], 4))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    ppddirs = appenddir (ppddirs, argv[++arg]);
	}

	elif (streq ("-sheethh", argv[arg]) || streq ("-shhh", argv[arg]))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    GOSPEC (headerheight.sheet, getsize (argv[++arg]))
	}
	elif (streq ("-filehh", argv[arg]))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    FOSPEC (headerheight.file, getsize (argv[++arg]))
	}
	elif (streq ("-pagehh", argv[arg]))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    FOSPEC (headerheight.page, getsize (argv[++arg]))
	}
	elif (strnabbr ("-bannerheight", argv[arg], 8))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    FOSPEC (headerheight.banner, getsize (argv[++arg]))
	}
	elif (strnabbr ("-userbannerheight", argv[arg], 12))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    FOSPEC (headerheight.userbanner, getsize (argv[++arg]))
	}

	elif (streq ("-lbw", argv[arg]) || strnabbr ("-leftborderwidth", argv[arg], 6))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    GOSPEC (borderwidth.left, getsize (argv[++arg]))
	}
	elif (streq ("-rbw", argv[arg]) || strnabbr ("-rightborderwidth", argv[arg], 7))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    GOSPEC (borderwidth.right, getsize (argv[++arg]))
	}
	elif (streq ("-tbw", argv[arg]) || strnabbr ("-topborderwidth", argv[arg], 5))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    GOSPEC (borderwidth.top, getsize (argv[++arg]))
	}
	elif (streq ("-bbw", argv[arg]) || strnabbr ("-bottomborderwidth", argv[arg], 8))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    GOSPEC (borderwidth.bottom, getsize (argv[++arg]))
	}

	elif (streq ("-landscape", argv[arg]))
	  GOSPEC (landscape, true)
	elif (streq ("-portrait", argv[arg]))
	  GOSPEC (landscape, false)

	elif (streq ("-text", argv[arg]))
	  ISPEC (type, IT_TEXT)
	elif (streq ("-man", argv[arg]))
	{
	  ISPEC (type, IT_MAN)
	  ISPEC (translation, TRLT_MAN)
	}
	elif (strnabbr ("-makefile", argv[arg], 5))
	  ISPEC (type, IT_MAKEFILE)
	elif (strnabbr ("-shell", argv[arg], 3))
	  ISPEC (type, IT_SHELL)
	elif (streq ("-C", argv[arg]) || streq ("-c", argv[arg]))
	  ISPEC (type, IT_C)
	elif (streq ("-C++", argv[arg]) || streq ("-c++", argv[arg]))
	  ISPEC (type, IT_Cplusplus)
	elif (streq ("-perl", argv[arg]))
	  ISPEC (type, IT_PERL)
	elif (streq ("-PS", argv[arg]) || streq ("-ps", argv[arg]))
	  ISPEC (type, IT_PS)
	elif (streq ("-lisp", argv[arg]) || streq ("-scm", argv[arg]) || streq ("-scheme", argv[arg]))
	  ISPEC (type, IT_LISP)
	elif (streq ("-tcl", argv[arg]))
	  ISPEC (type, IT_TCL)
	elif (streq ("-mail", argv[arg]))
	  ISPEC (type, IT_Mail)
	elif (streq ("-news", argv[arg]))
	  ISPEC (type, IT_News)

	elif (streq ("-iso646", argv[arg]) || streq ("-646", argv[arg]))
	  ISPEC (encoding, ENC_ISO646)
	elif (streq ("-ascii", argv[arg]))
	  ISPEC (encoding, ENC_ASCII)
	elif (streq ("-din66003", argv[arg]) || streq ("-66003", argv[arg]))
	  ISPEC (encoding, ENC_DIN66003)
	elif (streq ("-isolatin1", argv[arg]) || streq ("-latin1", argv[arg]) || streq ("-8859-1", argv[arg]))
	  ISPEC (encoding, ENC_ISOlatin1)
	elif (streq ("-cp437", argv[arg]) || streq ("-ibm", argv[arg]))
	  ISPEC (encoding, ENC_CP437)

	elif (strnabbr ("-literal", argv[arg], 4))
	  ISPEC (translation, TRLT_none)
	elif (strnabbr ("-troff", argv[arg], 3))
	  ISPEC (translation, TRLT_TROFF)
	elif (streq ("-mos", argv[arg]))
	  ISPEC (translation, TRLT_MOS)
	elif (streq ("-ms", argv[arg]))
	  ISPEC (translation, TRLT_MS)
	elif (streq ("-tex", argv[arg]) || streq ("-TeX", argv[arg]))
	  ISPEC (translation, TRLT_TEX)

	elif (streq (argv[arg], "-ff"))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (font[IOM_lit].family, argv[++arg])
	  FOSPEC (font[IOM_trlt].family, argv[arg])
	}
	elif (streq (argv[arg], "-fw"))
	{
	  t_fontweight weight;
	  if (arg+2 > argc)
	    usage (true);
	  weight = getweight (argv[++arg]);
	  FOSPEC (font[IOM_lit].rendition.weight, weight)
	  FOSPEC (font[IOM_trlt].rendition.weight, weight)
	}
	elif (streq (argv[arg], "-fi"))
	{
	  bool italic;
	  if (arg+2 > argc)
	    usage (true);
	  italic = getbool (argv[++arg]);
	  FOSPEC (font[IOM_lit].rendition.italic, italic)
	  FOSPEC (font[IOM_trlt].rendition.italic, italic)
	}
	elif (streq (argv[arg], "-fsz"))
	{
	  t_point size;
	  if (arg+2 > argc)
	    usage (true);
	  size = getsize (argv[++arg]);
	  FOSPEC (font[IOM_lit].size, size)
	  FOSPEC (font[IOM_trlt].size, size)
	}
	elif (streq (argv[arg], "-lcolor"))
	{
	  if (arg+2 > argc)
	    usage (true);
#ifdef COLORED
	  FOSPEC (color[IOM_lit], getcolor (argv[++arg]));
#else
          colorwarning();
#endif
	}
	elif (streq (argv[arg], "-tcolor"))
	{
	  if (arg+2 > argc)
	    usage (true);
#ifdef COLORED
	  FOSPEC (color[IOM_trlt], getcolor (argv[++arg]));
#else
          colorwarning();
#endif
	}
	elif (streq (argv[arg], "-color"))
	{
#ifdef COLORED
	  t_color color;
#endif
	  if (arg+2 > argc)
	    usage (true);
#ifdef COLORED
	  color = getcolor (argv[++arg]);
	  FOSPEC (color[IOM_lit], color);
	  FOSPEC (color[IOM_trlt], color);
#else
          colorwarning();
#endif
	}
	elif (streq (argv[arg], "-lff"))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (font[IOM_lit].family, argv[++arg])
	}
	elif (streq (argv[arg], "-lfw"))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (font[IOM_lit].rendition.weight, getweight (argv[++arg]))
	}
	elif (streq (argv[arg], "-lfi"))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (font[IOM_lit].rendition.italic, getbool (argv[++arg]))
	}
	elif (streq (argv[arg], "-lfsz"))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (font[IOM_lit].size, getsize (argv[++arg]))
	}
	elif (streq (argv[arg], "-tff"))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (font[IOM_trlt].family, argv[++arg])
	}
	elif (streq (argv[arg], "-tfw"))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (font[IOM_trlt].rendition.weight, getweight (argv[++arg]))
	}
	elif (streq (argv[arg], "-tfi"))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (font[IOM_trlt].rendition.italic, getbool (argv[++arg]))
	}
	elif (streq (argv[arg], "-tfsz"))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (font[IOM_trlt].size, getsize (argv[++arg]))
	}

	elif (streq ("-lnsp", argv[arg]))
	{
	  if (arg+3 > argc)
	    usage (true);
	  if (strabbr ("automatic", argv[++arg]))
	    foutput.linespacing.automatic = true;
	  elif (strabbr ("fixed", argv[arg]))
	    foutput.linespacing.automatic = false;
	  else
	    usage (true);
	  if (foutput.linespacing.relative = *argv[++arg] == '*')
	    foutput.linespacing.height.factor = atof (argv[arg]+1);
	  else
	    foutput.linespacing.height.size = getsize (argv[arg]);
	  foutputspec.linespacing.height = true;
	}
	elif (streq ("-minlnsp", argv[arg]))
	{
	  if (arg+2 > argc)
	    usage (true);
	  FOSPEC (linespacing.minheight, getsize (argv[++arg]))
	}
	elif (strnabbr ("-tabcolumns", argv[arg], 4))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    ISPEC (tabcolumns, atoi (argv[++arg]))
	}
	elif (strnabbr ("-tabwidth", argv[arg], 4))
	{
	  if (arg+2 > argc)
	    usage (true);
	  if (foutput.tabs.relative = *argv[++arg] == '*')
	    foutput.tabs.width.factor = atof (argv[arg]+1);
	  else
	    foutput.tabs.width.size = getsize (argv[arg]);
	  foutputspec.tabs.width = true;
	}
	elif (streq ("-ignLF", argv[arg]))
	  ISPEC (ignLFafterFF, true)
	elif (streq ("-noignLF", argv[arg]))
	  ISPEC (ignLFafterFF, false)
	elif (streq ("-igncharunavail", argv[arg]))
	  abortoncharunavail = false;
	elif (streq ("-noigncharunavail", argv[arg]))
	  abortoncharunavail = true;
	elif (strnabbr ("-columns", argv[arg], 4))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    GOSPEC (columns, atoi (argv[++arg]))
	}

	elif (strnabbr ("-printertype", argv[arg], 8))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    printerspec.printertype = argv[++arg];
	}
	elif (strnabbr ("-papertype", argv[arg], 6))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    printerspec.papertype = argv[++arg];
	}
	elif (streq ("-duplex", argv[arg]))
	{
	  if (arg+2 > argc)
	    usage (true);
	  else
	    printerspec.duplexmode = argv[++arg];
	}

	else
	{
	  say (IMP_warning, "unknown option `%s'.\n", argv[arg]);
	  usage (true);
	}
	continue;
      }
    enqueue (*jobs, argv[arg]);
  }
}

static cstring getusername()
{
  string	username;
  struct passwd	*pwd;
  int		len;
  t_buf		uid;

  if ((username = getenv ("NAME")) && strlen (strstrip (username)))
    return username;
  if (pwd = getpwuid (getuid()))
    if ((username = pwd->pw_gecos) && ((len = chrind (username, ',')) > 0 || (len = strlen (username)) > 0))
      return strnsave (username, len);
    else
      return strsave (pwd->pw_name);
  say (IMP_warning, "warning: can't get your passwd entry!\n");
  sprintf (uid, "\\fI:%ld:", (long)getuid());
  return strsave (uid);
}

static void globaldefaults (void)
{
  if (!cachespec.flags)
    cache.flags = DEFAULT_CACHEFLAGS;
  if (!cache.filename) /* unspecified by option */
    if (cache.filename = getenv ("PSFILTCACHEFILE"))
      cache.flags |= CACHE_global;
    else /* unspecified by user */
    {
      if (DEFAULT_CACHEGLOBAL)
	cache.flags |= CACHE_global;
      else
	cache.flags &= ~CACHE_global;
      assert (DEFAULT_CACHEFILE);
      cache.filename = DEFAULT_CACHEFILE;
    }

#define DEFAULT(member, value)	\
  {				\
    if (!goutputspec.member)	\
    {				\
      goutput.member = value;	\
      goutputspec.member = true;\
    }				\
  }
  DEFAULT (landscape, DEFAULT_ORIENTATION_LANDSCAPE);
  DEFAULT (borderwidth.left, DEFAULT_BORDERWIDTH)
  DEFAULT (borderwidth.right, DEFAULT_BORDERWIDTH)
  DEFAULT (borderwidth.top, DEFAULT_BORDERWIDTH)
  DEFAULT (borderwidth.bottom, DEFAULT_BORDERWIDTH)
  DEFAULT (headerheight.sheet, DEFAULT_SHEETHEADERHEIGHT)
  DEFAULT (columns, DEFAULT_COLUMNS)
  DEFAULT (columnsep, goutput.headerheight.sheet/2.)
#undef DEFAULT

  ppddirs = appendpath (ppddirs, getenv ("PPDPATH"));
  ppddirs = appendpath (ppddirs, DEFAULT_PPDPATH);
#ifdef DEBUG
  say (IMP_entertain2, ".PPD files will be searched in:\n");
  if (VERBp (IMP_entertain2))
    limap (ppddirs, printdir);
#endif

  fontdirs = appendpath (fontdirs, getenv ("FONTPATH"));
  fontdirs = appendpath (fontdirs, DEFAULT_FONTPATH);
#ifdef DEBUG
  say (IMP_entertain2, "font files will be searched in:\n");
  if (VERBp (IMP_entertain2))
    limap (fontdirs, printdir);
#endif

  afmdirs = appendpath (afmdirs, getenv ("AFMPATH"));
  afmdirs = appendpath (afmdirs, DEFAULT_AFMPATH);
#ifdef DEBUG
  say (IMP_entertain2, ".afm files will be searched in:\n");
  if (VERBp (IMP_entertain2))
    limap (afmdirs, printdir);
#endif

  username = getusername();
}

int main (int argc, string *argv)
{
  t_list	*jobs = NULL;

#ifdef _DEBUG_MALLOC_INC
  say (IMP_entertain, "this version is compiled with dbmalloc!\n");
  {
    union dbmalloptarg val;
    val.i = 2;
    dbmallopt (MALLOC_FILLAREA , &val);
  }
#endif

  cmdname = *argv++; --argc;
  jobs = licreate (sizeof (t_job), NULL, NULL); assert (jobs);
  init_metaoptions();
  parse_args (argc, argv, &jobs);
  globaldefaults();
  if (init_fonts (&cache))
    return 1;
  if (jobs)
  {
    if (!lilength (jobs))
    {
      say (IMP_entertain, "no files specified. reading stdin.\n");
      enqueue (jobs, NULL);
    }
    init_output (&printerspec, redirect ? redirect : stdout, username, input.type, &goutput, jobs);
    process (jobs);
  }

  return ok;
}
