/*
 * program: psfilt
 * file: ps.c
 *
 * Copyright  1992 1993 Robert Joop
 *
 * this module generates all P*stScr*pt output
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: ps.c,v $
 * Revision 1.7  1994/08/18  12:21:09  rj
 * unused variable removed.
 *
 * Revision 1.6  1994/08/18  12:13:32  rj
 * fix for postscript parser font caches that ignore reencoding changes.
 *
 * Revision 1.5  1994/07/13  18:16:58  rj
 * beta status removed
 * ascender and descender for psfilt's internal font weren't set.
 * some glyphs refined
 *
 * Revision 1.4  1994/07/09  16:44:11  rj
 * a lot of const's removed and added
 *
 * Revision 1.3  1994/03/18  12:24:08  rj
 * new glyph `envelope' added.
 *
 * Revision 1.2  1994/01/09  23:46:11  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.1.1.1  1993/12/31  20:56:43  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: ps.c,v 1.7 1994/08/18 12:21:09 rj Exp $";

#include <time.h>	/* ctime() */

#include "psfilt.h"
#include "version.h"
#include "patchlevel.h"
#include "iso646.h"
#include "ps.h"
#include "defaults.h"
#include "list.h"
#include "verbose.h"
#include "error.h"
#include "misc.h"	/* for fcopy() */

t_dscdata	dscdata = DEFAULT_DSC;
bool		structured = true;		/* produce output according to Document Structuring Conventions (DSC) */
t_langlevels	languagelevels = { 1, 1 };

static bool getlevel (int desired)
{
  bool	success;

  if (success = languagelevels.printer >= desired)
      if (languagelevels.output < desired)
	languagelevels.output = desired;
  return success;
}

cstring dscdata_name (t_dscdata dscdata)
{
  static cstring documentdata_strings[] = { "Clean7bit", "Clean8bit", "Binary" };

  return 0 <= dscdata && dscdata < ARRAYDIM (documentdata_strings) ? documentdata_strings[dscdata] : NULL;
}

int pscharpcmp (const t_pschar *l, const t_pschar *r)
{
  return strcmp (l->name, r->name);
}

static FILE	*printer,	/* Prolog and Script */
		  /* Prolog: Header and Procedure Definitions */
		    *header,
		    *procedures,
		  /* Script: Document Setup, Pages and Document Trailer */
		    *setup,
		    *pages,
		    *trailer;

static bool	proceduresused,
		setupused;

static t_list	*neededresources,
		*suppliedresources;

static FILE *componentfp (t_component component)
{
  switch (component)
  {
    case Header:
	return header;
    case Procedures:
	proceduresused = true;
	return procedures;
    case Setup:
	setupused = true;
	return setup;
    case Pages:
	return pages;
    case Trailer:
	return trailer;
    default:
	abort();
  }
}

static cstring componentname (t_component component)
{
  switch (component)
  {
    case Header:
	return "header";
    case Procedures:
	return "procedures";
    case Setup:
	return "setup";
    case Pages:
	return "pages";
    case Trailer:
	return "trailer";
    default:
	abort();
  }
}

static void werror (cstring component)
{
  error ("can't write %s", component);
}

static int psdata (t_component component, const uchar *s, size_t len, bool escape, int columns)
{
  FILE	*ostream = componentfp (component);

  for (; len--; s++)
  {
    if (columns > 250 && structured)
    {
      if (fprint (ostream, "\\\n") == EOF)
	werror (componentname (component));
      columns = 0;
    }
    if (*s < ISO646_SPACE && dscdata != DSC_Binary)			/* non printable characters		*/
    {
      switch (*s)
      {
	case ISO646_BS:
	    if (fprint (ostream, "\\b") == EOF)
	      werror (componentname (component));
	    columns += 2;
	    break;
	case ISO646_HT:
	    if (fprint (ostream, escape ? "\\t" : "\t") == EOF)
	      werror (componentname (component));
	    if (escape)
	      columns += 2;
	    else
	      columns = 8*(columns/8+1);
	    break;
	case ISO646_LF:
	    if (fprint (ostream, escape ? "\\n" : "\n") == EOF)
	      werror (componentname (component));
	    if (escape)
	      columns += 2;
	    else
	      columns = 0;
	    break;
	case ISO646_FF:
	    if (fprint (ostream, "\\f") == EOF)
	      werror (componentname (component));
	    columns += 2;
	    break;
	case ISO646_CR:
	    if (fprint (ostream, "\\r") == EOF)
	      werror (componentname (component));
	    columns += 2;
	    break;
	default:
	    if (fprintf (ostream, "\\%03o", *s) == EOF)
	      werror (componentname (component));
	    columns += 4;
      }
    }
    elif (*s >= ISO646_DEL && dscdata == DSC_Clean7bit)	/* non printable and non-ISO646 characters		*/
    {
      if (fprintf (ostream, "\\%03o", *s) == EOF)
	werror (componentname (component));
      columns += 4;
    }
    elif (escape && chrind ("(\\)", *s) != fault)	/* characters with special meaning in postscript	*/
    {
      if (fprintf (ostream, "\\%c", *s) == EOF)
	werror (componentname (component));
      columns += 2;
    }
    else
    {
      if (fputc (*s, ostream) == EOF)
	werror (componentname (component));
      if (*s == '\n')
	columns = 0;
      else
	columns++;
    }
  }
  return columns;
}

static void upload (t_component component, FILE *istream)
{
  uchar		buf[8192];
  size_t	len;
  int		columns = 0;

  while ((len = fread (buf, 1, sizeof buf, istream)) > 0)
    columns = psdata (component, buf, len, false, columns);
}

/*\[sep]-------------------------------------------------------------------------------------------------------------------------*/ 
typedef struct
{
  string	name;
  t_width	width;
  t_rectangle	bbox;
  string	procedure;

  bool		used;
} t_pschardef;

static int pschardefpcmp (const t_pschardef **lhs, const t_pschardef **rhs)
{
  return strcmp ((*lhs)->name, (*rhs)->name);
}

static t_pschardef symboltable[] =
{
  { ".notdef", 0., 0., 0., 0., 0., 0., "", true },

  /* eigenentwrfe: */
  { "space", 250., 0., 0., 0., 0., 0., "" },
  { "miss", 350., 0., 0., -200., 350., 800., "20 -200 moveto 310 0 rlineto 0 1000 rlineto -310 0 rlineto closepath fill" },
  { "telrec", 640., 0., -62., -82., 640., 620., "newpath 320 300 300 -135 225 arc\n-150 -150 rmoveto\n300 300 rlineto stroke" },
  { "peace", 640., 0., 20., 0., 640., 620., "newpath 320 300 300 90 450 arc\n320 20 moveto 320 600 lineto\n107.87 87.87 moveto 320 300 lineto 532.13 87.87 lineto\nstroke" },
  { "yinyang", 650., 0., -40., -40., 640., 640., "40 setlinewidth\nnewpath\n300 300 320 0 360 arc stroke\n450 300 40 0 360 arc fill\n300 300 300 180 360 arc\n450 300 150 0 180 arcn\n150 300 150 0 180 arc\n190 300 moveto\n150 300 40 360 0 arcn\nfill" },
  { "handicapped", 700., 0., -20., -20., 670., 680., "1 setlinecap\n1 setlinejoin\nnewpath\n200 200 200 0 360 arc stroke\n200 600  80 0 360 arc fill\n200 600 moveto\n200 200 lineto\n450 200 lineto\n550   0 lineto\n650   0 lineto\nstroke\n200 400 moveto\n450 400 lineto\nstroke" },
  { "stereo", 640., 0., 0., 0., 640., 440., "newpath\n220 220 200 0 360 arc stroke\n420 220 200 0 360 arc stroke" },
  { "married", 840., 0., 0., 0., 840., 440., "newpath\n220 220 200 0 360 arc stroke\n620 220 200 0 360 arc stroke" },
  { "divorced", 920., 0., 0., 0., 920., 440., "newpath\n220 220 200 0 360 arc stroke\n460 0 moveto 460 440 lineto stroke\n700 220 200 0 360 arc stroke" },
  { "female", 440., 0., 0., -150., 440., 670., "newpath\n220 450 200 0 360 arc\n220 250 moveto 220 -150 lineto\n20 50 moveto 420 50 lineto stroke" },
  { "male", 640., 0., 0., 50., 640., 870., "newpath\n220 250 200 45 405 arc\n620 650 lineto\n420 650 moveto 620 650 lineto 620 450 lineto stroke" },
  { "envelope", 800., 0., 50., 100., 750., 540., "newpath\n50 100 moveto\n750 100 lineto\n750 540 lineto\n50 540 lineto\nclosepath\n50 100 moveto\n300 315 lineto\n750 100 moveto\n500 315 lineto\n750 540 moveto\n400 230 lineto\n50 540 lineto\nstroke" },

  /* aus DIThacks: */
  { "ru", 500., 0., -260., -260., 900., 900., "500 0 rlineto stroke" },
  { "rn", 500., 0., -260., -260., 900., 900., "0 840 moveto 500 0 rlineto stroke" },
  { "vr",   0., 0., -260., -260., 900., 900., "0 800 moveto 0 -770 rlineto stroke" },
  { "bv", 416., 0., -260., -260., 900., 900., "0 800 moveto 0 -1000 rlineto stroke" },
  { "br",   0., 0., -260., -260., 900., 900., "0 750 moveto 0 -1000 rlineto stroke" },
  { "ul", 500., 0., -260., -260., 900., 900., "0 -140 moveto 500 0 rlineto stroke" },
  { "ob", 350., 0., -260., -260., 900., 900., "200 250 rmoveto currentpoint newpath 200 0 360 arc closepath stroke" },
  { "bu", 350., 0., -260., -260., 900., 900., "200 250 rmoveto currentpoint newpath 200 0 360 arc closepath fill" },
  { "sq", 750., 0., -260., -260., 900., 900., "80 0 rmoveto currentpoint transform round exch round exch itransform newpath moveto 640 0 rlineto 0 640 rlineto -640 0 rlineto closepath stroke" },
  { "bx", 750., 0., -260., -260., 900., 900., "80 0 rmoveto currentpoint transform round exch round exch itransform newpath moveto 640 0 rlineto 0 640 rlineto -640 0 rlineto closepath fill" },
  { "ci", 750., 0., -260., -260., 900., 900., "500 360 rmoveto currentpoint newpath 333 0 360 arc 50 setlinewidth stroke" },
  { "lt", 416., 0., -260., -260., 900., 900., "0 -200 moveto 0 550 rlineto currentpoint pop 800 2 copy exch 250 add exch 250 arcto pop pop pop pop stroke" },
  { "lb", 416., 0., -260., -260., 900., 900., "0 800 moveto 0 -550 rlineto currentpoint pop -200 2 copy exch 250 add exch 250 arcto pop pop pop pop stroke" },
  { "rt", 416., 0., -260., -260., 900., 900., "0 -200 moveto 0 550 rlineto currentpoint pop 800 2 copy exch 250 sub exch 250 arcto pop pop pop pop stroke" },
  { "rb", 416., 0., -260., -260., 900., 900., "0 800 moveto 0 -500 rlineto currentpoint pop -200 2 copy exch 250 sub exch 250 arcto pop pop pop pop stroke" },
  { "lk", 416., 0., -260., -260., 900., 900., "0 800 moveto 0 300 -300 300 250 arcto pop pop 1000 sub 0 300 4 2 roll 250 arcto pop pop pop pop 0 -200 lineto stroke" },
  { "rk", 416., 0., -260., -260., 900., 900., "0 800 moveto 0 300 500 300 250 arcto pop pop 1000 sub 0 300 4 2 roll 250 arcto pop pop pop pop 0 -200 lineto stroke" },
  { "lf", 416., 0., -260., -260., 900., 900., "0 800 moveto 0 -1000 rlineto 250 0 rlineto stroke" },
  { "rf", 416., 0., -260., -260., 900., 900., "0 800 moveto 0 -1000 rlineto -250 0 rlineto stroke" },
  { "lc", 416., 0., -260., -260., 900., 900., "0 -200 moveto 0 1000 rlineto 250 0 rlineto stroke" },
  { "rc", 416., 0., -260., -260., 900., 900., "0 -200 moveto 0 1000 rlineto -250 0 rlineto stroke" },
};

static t_rectangle	fontbbox;

#define SCALE	.001

static void fonthead (int charcount, FILE *fp)
{
  fprintf (fp, "\
/%s 8 dict dup begin
  /FontType 3 def
  /FontMatrix [%g 0 0 %g 0 0] def
  /FontBBox [%g %g %g %g] def

  /Encoding [ 256 { /.notdef } repeat ] def
  /CharProcs %d dict\
", INTERNALFONT_NAME, SCALE, SCALE, fontbbox.llx, fontbbox.lly, fontbbox.urx, fontbbox.ury, charcount);
  if (structured)
    fprint (fp, " dup begin\n");
}

static void fonttail (FILE *fp)
{
  bool	level2 = getlevel (2);

  if (structured)
    fprint (fp, "  end");
  fprintf (fp, "\
 def

  %s
  bind def
end definefont pop
", level2
?"/BuildGlyph
  {
    40 setlinewidth
    exch /CharProcs get exch
    2 copy known not { pop /.notdef } if
    get exec
  }"
:"/BuildChar
  {
    40 setlinewidth
    exch begin
      Encoding exch get
      CharProcs exch get
    end
    exec
  }"
);
}

static void psprintchardef (const t_pschardef *pschardef, int indent, FILE *fp)
{
  const char	*p;
  bool		beginofline;

  fprintf (fp, "%*s/%s {\n", indent, "", pschardef->name);
  fprintf (fp, "%*s  %g %g %g %g %g %g setcachedevice\n", indent, "", pschardef->width.x, pschardef->width.y, pschardef->bbox.llx, pschardef->bbox.lly, pschardef->bbox.urx, pschardef->bbox.ury);
  for (p=pschardef->procedure, beginofline=true; *p; p++)
  {
    if (*p == '\n')
      beginofline = true;
    else
    {
      if (beginofline)
	fprintf (fp, "%*s  ", indent, "");
      beginofline = false;
    }
    fputc (*p, fp);
  }
  if (!beginofline)
    fputc ('\n', fp);
  fprintf (fp, "%*s} bind def\n", indent, "");
}

static t_avl	*symbols;

static void initsymbols()
{
  int		len;
  t_pschardef	*table;

  assert (symbols = avlcreate (sizeof (t_pschardef *), pschardefpcmp, NULL));
  fontbbox.llx = fontbbox.lly = 1000.;	/* to be lowered */
  fontbbox.urx = fontbbox.ury = 0.;	/* to be raised */
  for (table=symboltable, len=ARRAYDIM (symboltable); len--; table++)
  {
    assert (avlinsert (symbols, &table));
    fontbbox.llx = MIN (fontbbox.llx, table->bbox.llx);
    fontbbox.lly = MIN (fontbbox.lly, table->bbox.lly);
    fontbbox.urx = MAX (fontbbox.urx, table->bbox.urx);
    fontbbox.ury = MAX (fontbbox.ury, table->bbox.ury);
  }
  if (!structured)
  {
    fonthead (ARRAYDIM (symboltable), procedures);
    fonttail (procedures);
    proceduresused = true;
  }
}

bool psisfontinternal (cstring fontname, t_version *fontversion)
{
  if (streq (fontname, INTERNALFONT_NAME) && !versioncmp (*fontversion, INTERNALFONT_VERSION))
  {
    *fontversion = INTERNALFONT_VERSION;
    return true;
  }
  return false;
}

void psinitafm (t_afmfile *afm)
{
  if (!symbols)
    initsymbols();

  afm->file.stat = FS_valid;
  afm->fontname = INTERNALFONT_NAME;
  afm->fullname = "Psfilt's own internal font";
  afm->familyname = INTERNALFONT_NAME;
  afm->weight = FONTWEIGHT_normal;
  afm->italicangle = 0.0;
  afm->version = INTERNALFONT_VERSION;
  afm->isfixedpitch = false;
  afm->ascender = SCALE * fontbbox.ury;
  afm->descender = SCALE * fontbbox.lly;
}

void pscompleteafm (t_afmfile *afm)
{
  t_buf		fontversion;
  int		len;
  t_pschardef	*table;
  t_charmetric	charmetric;

  say (IMP_entertain, "generating metrics for font /%s (%s)\n", afm->fontname, strversion (fontversion, afm->version));

  afm->bbox.llx = SCALE * fontbbox.llx;
  afm->bbox.lly = SCALE * fontbbox.lly;
  afm->bbox.urx = SCALE * fontbbox.urx;
  afm->bbox.ury = SCALE * fontbbox.ury;

  charmetric.ligatures = NULL; /* no ligaturs at all */
  for (table=symboltable, len=ARRAYDIM (symboltable); len--; table++)
  {
    charmetric.name = table->name;
    charmetric.code = -1;
    charmetric.width.x = SCALE * table->width.x;
    charmetric.width.y = SCALE * table->width.y;
    charmetric.bbox.llx = SCALE * table->bbox.llx;
    charmetric.bbox.lly = SCALE * table->bbox.lly;
    charmetric.bbox.urx = SCALE * table->bbox.urx;
    charmetric.bbox.ury = SCALE * table->bbox.ury;
    assert (avlinsert (afm->metrics, &charmetric));
  }
  afm->file.stat = FS_incore;
}

int psgetsymbol (string charname)
{
  t_pschardef	search, *sp = &search, **found;

  if (!symbols)
    initsymbols();

  search.name = charname;
  if (found = avlsearch (symbols, &sp))
  {
    if (!(*found)->used)
    {
      (*found)->used = true;
      if (!structured)
      {
	fprintf (procedures, "/%s findfont /CharProcs get\n", INTERNALFONT_NAME);
	psprintchardef (*found, 2, procedures);
	fprint (procedures, "put\n");
      }
    }
    return ok;
  }
  else
    return fault;
}

static int countsymbol (t_pschardef **symbol, int *count)
{
  if ((*symbol)->used)
    ++*count;
  return ok;
}

static int putsymbol (t_pschardef **symbol, FILE *fp)
{
  if ((*symbol)->used)
    psprintchardef (*symbol, 4, fp);
  return ok;
}

static void putsymbols()
{
  int	count = 0;
#if 0
  char	buf[1024*ARRAYDIM (symboltable)];
  FILE	*buffer;
#else
  FILE	*font;
#endif

  assert (structured);

  avlamap (symbols, countsymbol, &count);
  if (count > 1) /* 1 for /.notdef */
  {
#if 1
    cstring	tmpdir,
		fontfn;

    if (!(tmpdir = getenv ("TMPDIR")))
      tmpdir = "/tmp";

    if (!(font = fopen (fontfn = tempnam (tmpdir, "psfT"), "w+")))
      error ("can't open tmp file `%s' for font", fontfn);
    unlink (fontfn);

    fonthead (count, font);
    avlamap (symbols, putsymbol, font);
    fonttail (font);
    rewind (font);

    pssupplyresource (font, "font %s", INTERNALFONT_NAME);
    fclose (font);
#else
fprintf (stderr, "opening buffer ");
    buffer = fsopen (buf, sizeof buf, "w");
fprintf (stderr, "filling buffer ");
    fonthead (count, buffer);
    avlamap (symbols, putsymbol, buffer);
    fonttail (buffer);
fprintf (stderr, "closing buffer\n");
    fsclose (buffer);

fprintf (stderr, "reopening buffer ");
    buffer = fsopen (buf, sizeof buf, "r");
fprintf (stderr, "using buffer ");
    pssupplyresource (buffer, "font %s", INTERNALFONT_NAME);
fprintf (stderr, "closing buffer\n");
    fsclose (buffer);
#endif
  }
}

/*\[sep]-------------------------------------------------------------------------------------------------------------------------*/ 
void init_ps (FILE *ostream)
{
  long	now;

  printer = ostream;
  assert (printer);
  now = time (NULL);
  if (structured)
  {
    cstring	tmpdir,
		proceduresfn, setupfn, pagesfn, trailerfn;

    if (!(tmpdir = getenv ("TMPDIR")))
      tmpdir = "/tmp";

    header = printer;

    if (!(procedures = fopen (proceduresfn = tempnam (tmpdir, "psfP"), "w+")))
      error ("can't open tmp file `%s' for procedures", proceduresfn);
    unlink (proceduresfn);

    if (!(setup = fopen (setupfn = tempnam (tmpdir, "psfS"), "w+")))
      error ("can't open tmp file `%s' for setup", setupfn);
    unlink (setupfn);

    if (!(pages = fopen (pagesfn = tempnam (tmpdir, "psfP"), "w+")))
      error ("can't open tmp file `%s' for pages", pagesfn);
    unlink (pagesfn);

    if (!(trailer = fopen (trailerfn = tempnam (tmpdir, "psfT"), "w+")))
      error ("can't open tmp file `%s' for trailer", trailerfn);
    unlink (trailerfn);

    if (fprint (header, "%!PS-Adobe-3.0\n") == EOF)
      werror ("header");
    psstructure (Header, "Creator: psfilt %d.%d.%d", RELEASE, REVISION, PATCHLEVEL);
    psstructure (Header, "CreationDate: %.24s", ctime (&now));

    psstructure (Procedures, "BeginProlog");
    proceduresused = false;
    psstructure (Setup, "BeginSetup");
    setupused = false;
    psstructure (Trailer, "Trailer");
  }
  else 
  {
    header = procedures = setup = pages = trailer = printer;
    if (fprint (printer, "%!\n") == EOF)
      werror ("output");
    pscomment ("output created %.24s", ctime (&now));
  }
}

static int resources (cstring name, cstring *categ)
{
  cstring	key;

  if (*categ)
  {
    key = *categ;
    *categ = NULL;
  }
  else
    key = "+";

  if (structured)
    psstructure (Header, "%s %s", key, name);
  
  return ok; /* for liamap() */
}

void finish_ps()
{
  if (structured)
  {
    psstructure (Header, "DocumentData: %s", dscdata_name (dscdata));
    psstructure (Header, "LanguageLevel: \"%d\"", languagelevels.output);

    putsymbols();	/* may add to suppliedresources */

    if (neededresources)
    {
      static const char name[] = "DocumentNeededResources:", *first = name;
      liamap (neededresources, resources, &first);
      neededresources = lidestroy (neededresources);
    }
    if (suppliedresources)
    {
      static const char name[] = "DocumentSuppliedResources:", *first = name;
      liamap (suppliedresources, resources, &first);
      suppliedresources = lidestroy (suppliedresources);
    }

    psstructure (Header, "EndComments");
    header = NULL;

    if (proceduresused)	/* omit if empty */
    {
      psstructure (Procedures, "EndProlog");
      rewind (procedures);
      if (fcopy (procedures, printer) == fault)
	error ("can't copy procedures into output");
    }
    fclose (procedures); procedures = NULL;

    if (setupused)	/* omit if empty */
    {
      psstructure (Setup, "EndSetup");
      rewind (setup);
      if (fcopy (setup, printer) == fault)
	error ("can't copy setup into output");
    }
    fclose (setup); setup = NULL;

    rewind (pages);
    if (fcopy (pages, printer) == fault)
      error ("can't copy pages into output");
    fclose (pages); pages = NULL;

    psstructure (Trailer, "EOF");
    rewind (trailer);
    if (fcopy (trailer, printer) == fault)
      error ("can't copy trailer into output");
    fclose (trailer); trailer = NULL;
  }
  if (fflush (printer) == EOF)
    werror ("output");
}

void psneededresource (cstring fmt, ...)
{
  va_list	va;
  t_buf		resource;

  if (!structured)
    return;

  if (!neededresources)
    neededresources = licreate (0, NULL, NULL);
  assert (neededresources);

  va_start (va, fmt);
  vsprintf (resource, fmt, va);
  va_end (va);
  liappend (neededresources, resource, strlen (resource)+1);

  psstructure (Setup, "IncludeResource: %s", resource);
}

void pssupplyresource (FILE *istream, cstring fmt, ...)
{
  va_list	va;
  t_buf		resource;

  if (!suppliedresources)
    suppliedresources = licreate (0, NULL, NULL);
  assert (suppliedresources);

  va_start (va, fmt);
  vsprintf (resource, fmt, va);
  va_end (va);
  liappend (suppliedresources, resource, strlen (resource)+1);

  if (structured)
    psstructure (Setup, "BeginResource: %s", resource);
  upload (Setup, istream);
  if (structured)
    psstructure (Setup, "EndResource");
}

void psfeature (cstring name, cstring value, cstring invocation)
{
  if (structured)
    psstructure (Setup, "BeginFeature: *%s %s", name, value);
  if (*invocation == '\n')
    invocation++;
  if (fprint (componentfp (Setup), invocation) == EOF)
    werror (componentname (Setup));
  if (*invocation && invocation[strlen (invocation)-1] != '\n')
    fputc ('\n', componentfp (Setup));
  if (structured)
    psstructure (Setup, "EndFeature");
}

void psstructure (t_component component, cstring fmt, ...)
{
  FILE		*fp;
  va_list	va;

  assert (structured);
  fp = componentfp (component);
  if (fprint (fp, "%%") == EOF)
    werror (componentname (component));
  va_start (va, fmt);
  if (vfprintf (fp, fmt, va) == EOF)
    werror (componentname (component));
  va_end (va);
  if (fprint (fp, "\n") == EOF)
    werror (componentname (component));
}

void pscomment (cstring fmt, ...)
{
  va_list	va;

  assert (!structured);
  if (fprint (printer, "% ") == EOF)
    werror ("output");
  va_start (va, fmt);
  if (vfprintf (printer, fmt, va) == EOF)
    werror ("output");
  va_end (va);
  if (fprint (printer, "\n") == EOF)
    werror ("output");
}

/*\[sep]-------------------------------------------------------------------------------------------------------------------------*/ 
void psdupfont (cstring oldname, cstring newname)
{
  static bool	initialized = false;

  if (!initialized)
  {
    if (fprintf (procedures, "\
/dupfont
{
  findfont dup length dict exch
  {
    1 index /FID eq
    { pop pop }
    {
      1 index /Encoding eq { %s } if
      1 index /FontName eq { pop 2 index } if
      2 index 3 1 roll put
    }
    ifelse
  }
  forall
  definefont pop
}
bind def
", structured ?
#if 1
		"pop [ 256 { /.notdef } repeat ]"
#else
		"dup length array copy"
#endif
						: "256 array copy") == EOF)
      werror (componentname (Procedures));
    proceduresused = true;
    initialized = true;
  }
  if (fprintf (pages, "/%s /%s dupfont\n", newname, oldname) == EOF)
    werror (componentname (Setup));
}

void psfindfont (cstring fontname)		/* puts a fontdict on stack top	*/
{
  if (fprintf (pages, "/%s findfont ", fontname) == EOF)
    werror (componentname (Pages));
}

void psaddmap (t_softfont *font, const cstring *newencoding, int count)	/* assumes that a fontdict is on stack top	*/
{
  int		code;
  t_pschar	pschar, *pscharp;

  if (fprint (pages, "dup /Encoding get\n") == EOF)
    werror (componentname (Pages));
  for (code=0; code<256; code++)
  {
    if (newencoding[code])
    {
    assert (!font->encoding[code]);
      if (fprint (pages, --count ? "dup " : "    ") == EOF)
	werror (componentname (Pages));
      if (fprintf (pages, "8#%03o/%s put\n", code, newencoding[code]) == EOF)
	werror (componentname (Pages));
      /* neuen namen als codiert markieren: */
      pschar.name = newencoding[code];
      pscharp = avlinsert (font->charstrings, &pschar); assert (pscharp);
      pscharp->code = code;
      /* neuen namen eintragen:	*/
      font->encoding[code] = newencoding[code];
      --font->emptyslots;
    }
  }
  assert (!count);
}

void psscalesetfont (t_point factor)		/* assumes that a fontdict is on stack top	*/
{
  static bool	initialized = false;

  if (!initialized)
  {
    if (fprint (procedures, "/SSF { scalefont setfont } bind def\n") == EOF)
      werror (componentname (Procedures));
    proceduresused = true;
    initialized = true;
  }

  if (fprintf (pages, "%g SSF\n", factor) == EOF)
    werror (componentname (Pages));
}

void psfindscalesetfont (cstring fontname, t_point factor)
{
  static bool	initialized = false,
		level2;

  if (!initialized)
  {
    level2 = getlevel (2);
    if (fprintf (procedures, "/%s { %s } bind def\n", level2 ? "SF" : "FSSF", level2 ? "selectfont" : "findfont exch scalefont setfont") == EOF)
      werror (componentname (Procedures));
    proceduresused = true;
    initialized = true;
  }
  if ((level2 ? fprintf (pages, "/%s %g SF\n", fontname, factor) : fprintf (pages, "%g/%s FSSF\n", factor, fontname)) == EOF)
    werror (componentname (Pages));
}

void psgsave()
{
  if (fprint (pages, "gsave\n") == EOF)
    werror (componentname (Pages));
}

void psgrestore()
{
  if (fprint (pages, "grestore\n") == EOF)
    werror (componentname (Pages));
}

void pstranslate (t_vector point)
{
  if (fprintf (pages, "%g %g translate\n", point.x, point.y) == EOF)
    werror (componentname (Pages));
}

void psrotate (double angle)
{
  if (fprintf (pages, "%g rotate\n", angle) == EOF)
    werror (componentname (Pages));
}

void psmoveto (t_vector point)
{
  static bool	initialized = false;

  if (!initialized)
  {
    if (fprint (procedures, "/M { moveto } bind def\n") == EOF)
      werror (componentname (Procedures));
    proceduresused = true;
    initialized = true;
  }

  if (fprintf (pages, "%g %g M\n", point.x, point.y) == EOF)
    werror (componentname (Pages));
}

void psmovetoalign (t_vector point)
{
  static bool	initialized = false;

  if (!initialized)
  {
    if (fprint (procedures, "/Ma { transform round exch round exch itransform moveto } bind def\n") == EOF)
      werror (componentname (Procedures));
    proceduresused = true;
    initialized = true;
  }

  if (fprintf (pages, "%g %g Ma\n", point.x, point.y) == EOF)
    werror (componentname (Pages));
}

void pssetlinewidthandstyle (const t_line *line)
{
  t_buf		array;
  int		offset = 0;

  switch (line->style)
  {
    case LS_solid:
	*array = '\0';
	break;
    case LS_dashed:
	sprintf (array, "%g %g", 6*line->width, 4*line->width);
	break;
    case LS_dotted:
	sprintf (array, "%g %g", line->width, 4*line->width);
	break;
    case LS_dash_dotted:
	sprintf (array, "%g %g %g %g", 6*line->width, 3*line->width, line->width, 4*line->width);
	break;
    case LS_dash_dot_dotted:
	sprintf (array, "%g %g %g %g %g %g", 6*line->width, 3*line->width, line->width, 3*line->width, line->width, 4*line->width);
	break;
    default:
	ierror ("pssetline()");
  }
  if (fprintf (pages, "%g setlinewidth\n[%s] %d setdash\n", line->width, array, offset) == EOF)
    werror (componentname (Pages));
}

void psrlineto (t_vector point)
{
  if (fprintf (pages, "%g %g rlineto\n", point.x, point.y) == EOF)
    werror (componentname (Pages));
}

void psstroke()
{
  if (fprint (pages, "stroke\n") == EOF)
    werror (componentname (Pages));
}

void pssetgray (double gray)
{
  if (fprintf (pages, "%g setgray\n", gray) == EOF)
    werror (componentname (Pages));
}

#ifdef COLORED
void pssetcolor (const t_color *color)
{
  switch (color->model)
  {
    case CM_rgb:
      if (fprintf (pages, "%g %g %g setrgbcolor\n", color->u.rgb.red, color->u.rgb.green, color->u.rgb.blue) == EOF)
	werror (componentname (Pages));
      break;
    case CM_hsb:
      if (fprintf (pages, "%g %g %g sethsbcolor\n", color->u.hsb.hue, color->u.hsb.saturation, color->u.hsb.brightness) == EOF)
	werror (componentname (Pages));
      break;
    default:
      ierror ("pssetcolor: illegal color model\n");
  }
}
#endif

void psclosepath()
{
  if (fprint (pages, "closepath\n") == EOF)
    werror (componentname (Pages));
}

void psfill()
{
  if (fprint (pages, "fill\n") == EOF)
    werror (componentname (Pages));
}

void psgraybox (t_vector offset, t_dimension area, double shade)
{
  static bool	initialized = false,
		level2;

  if (!initialized)
  {
    level2 = getlevel (2);
    if (fprintf (procedures, "\
/GB {
  %% gray box
  %% usage: x y w h shade GB
  gsave
    setgray
   %s
  grestore
} bind def
", level2
? " rectfill"
: " 4 2 roll moveto
    1 index 0 rlineto
    0 exch rlineto
    neg 0 rlineto
    closepath
    fill"
) == EOF)
      werror (componentname (Procedures));
    proceduresused = true;
    initialized = true;
  }

  if (fprintf (pages, "%g %g %g %g %g GB\n", offset.x, offset.y, area.width, area.height, shade) == EOF)
    werror (componentname (Pages));
}

void psbannerbox (t_vector offset, t_dimension area)
{
  double	min, step;
  int		steps;
  char		*shades;

  min = MIN (area.width, area.height);
  if (min > 10.)
  {
    static char s[] = { 97, 92, 87, 82, 77 };
    steps = 5;
    step = min / 20.;
    shades = s;
  }
  else
  {
    static char s[] = { 97, 87, 77 };
    steps = 3;
    step = min / 10.;
    shades = s;
  }
  while (steps-->0)
  {
    psgraybox (offset, area, shades[steps] / 100.);
    offset.x += step; offset.y += step;
    area.width -= 2 * step; area.height -= 2 * step;
  }
}

static int psstring (const uchar *s, size_t len, int columns)
{
  if (fprint (pages, "(") == EOF)
    werror (componentname (Pages));
  columns = psdata (Pages, s, len, true, ++columns);
  if (columns > 240 && structured)
    if (fprint (pages, "\\\n") == EOF)
      werror (componentname (Pages));
  if (fprint (pages, ")") == EOF)
    werror (componentname (Pages));
  return columns;
}

void psshow (const uchar *s, size_t len)
{
  static bool	initialized = false;

  if (!initialized)
  {
    if (fprint (procedures, "/S { show } bind def\n") == EOF)
      werror (componentname (Procedures));
    proceduresused = true;
    initialized = true;
  }

  psstring (s, len, 0);
  if (fprint (pages, "S\n") == EOF)
    werror (componentname (Pages));
}

void pskxshow (const uchar *s, const t_point *kx, size_t len)
{
  static bool	initialized = false;
  int		kp, columns;

  if (!initialized)
  {
    if (fprint (procedures, "/KxS { { pop pop 0 rmoveto } exch kshow } bind def\n") == EOF)
      werror (componentname (Procedures));
    proceduresused = true;
    initialized = true;
  }

  columns = 0;
  for (kp=len-1; kp--; )
  {
    t_buf	buf;

    if (columns > 240 && structured)
    {
      if (fprint (pages, "\n") == EOF)
	werror (componentname (Pages));
      columns = 0;
    }
    sprintf (buf, "%g ", kx[kp]);
    if (fprint (pages, buf) == EOF)
      werror (componentname (Pages));
    columns += strlen (buf);
  }
  columns = psstring (s, len, columns);
  if (fprint (pages, "KxS\n") == EOF)
    werror (componentname (Pages));
}

void pskshow (const uchar *s, const t_width *k, size_t len)
{
  static bool	initialized = false;
  int		kp, columns;

  if (!initialized)
  {
    if (fprint (procedures, "/KS { { pop pop rmoveto } exch kshow } bind def\n") == EOF)
      werror (componentname (Procedures));
    proceduresused = true;
    initialized = true;
  }

  columns = 0;
  for (kp=len-1; kp--; )
  {
    t_buf	buf;

    if (columns > 230 && structured)
    {
      if (fprint (pages, "\n") == EOF)
	werror (componentname (Pages));
      columns = 0;
    }
    sprintf (buf, "%g %g ", k[kp].x, k[kp].y);
    if (fprint (pages, buf) == EOF)
      werror (componentname (Pages));
    columns += strlen (buf);
  }
  columns = psstring (s, len, columns);
  if (fprint (pages, "KS\n") == EOF)
    werror (componentname (Pages));
}

void psshowpage()
{
  if (fprint (pages, "showpage\n") == EOF)
    werror (componentname (Pages));
}
