/*
 * program: psfilt
 * file: printer.c
 *
 * Copyright  1992 1993 Robert Joop
 *
 * font and character management
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: printer.c,v $
 * Revision 1.6  1994/08/18  12:21:08  rj
 * unused variable removed.
 *
 * Revision 1.5  1994/08/18  12:13:31  rj
 * fix for postscript parser font caches that ignore reencoding changes.
 *
 * Revision 1.4  1994/07/09  16:44:09  rj
 * a lot of const's removed and added
 *
 * Revision 1.3  1994/03/18  12:07:36  rj
 * support for duplex mode added
 * lots of missing `#ifdef COLORED' inserted
 *
 * Revision 1.2  1994/01/09  23:46:08  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.1.1.1  1993/12/31  20:56:43  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: printer.c,v 1.6 1994/08/18 12:21:08 rj Exp $";

#include "psfilt.h"
#include "defaults.h"
#include "printer.h"
#include "verbose.h"
#include "str.h"	/* strunique() */
#include "mem.h"
#include "error.h"

typedef struct
{
  t_ppd		ppd;

  t_paperinfo	paperinfo;
  t_duplexinfo	duplexinfo;

  t_avl		*pfonts;	/* array of t_pfont, sorted by fontname */

#ifdef COLORED
  t_color	currentcolor;
#endif
  struct
  {
    t_pfont	*pfont;
    cstring	name;
    t_point	size;
  } currentfont;
} t_printerstate;		/* stellt den ist-zustand dar */

/* keep track of printer status */
static t_printerstate	pstate;		/* this program generates output for only one printer at a time, so there's only one variable */

static int pfontpcmp (const t_pfont *l, const t_pfont *r)
{
  int	cp;

  return (cp = strcmp (l->font->fontname, r->font->fontname)) ? cp : versioncmp (l->reqversion, r->reqversion);
}

void init_printer (t_printerspec *printerspec, FILE *ostream)
{
  init_ps (ostream);

  if (!printerspec->printertype && !(printerspec->printertype = getenv ("PRINTERTYPE")))
  {
    say (IMP_error, "can't proceed without knowledge of printer type.\n");
    if (lilength (ppddirs))
    {
      say (IMP_error, "please use one of the following printer descriptions:\n");
      listppdfiles();
    }
    exit (3);
  }
  if (readppd (printerspec->printertype, &pstate.ppd))
    error ("can't proceed without knowledge about printer type \"%s\".", printerspec->printertype);
  if (VERBp (IMP_debug2))
    print_ppd (&pstate.ppd, stderr);

  if (structured)
    psstructure (Header, "DocumentPrinterRequired: () %s", getproduct (&pstate.ppd));
  else
    pscomment ("output generated for a \"%s\"", getname (&pstate.ppd));

  languagelevels.printer = getlanguagelevel (&pstate.ppd);

  {
    string	papertypes[] =
    {
      printerspec->papertype,
      getenv ("PAPERTYPE"),
      DEFAULT_PAPERTYPE,
      "A4",
      "Letter",
    };
    int	i;

    for (i=0; i<ARRAYDIM (papertypes); i++)
      if (papertypes[i])
      {
	int j;

	for (j=i-1; j-->0; )
	  if (!str0cmp (papertypes[j], papertypes[i]))
	    break;
	if (j >= 0)
	  continue;
	if (!getpaperinfo (&pstate.ppd, papertypes[i], &pstate.paperinfo))
	  break;
	else
	  say (IMP_warning, "printer `%s' has no paper `%s'\n", getname (&pstate.ppd), papertypes[i]);
      }
    if (i == ARRAYDIM (papertypes))
      error ("can't find paper type. giving up.");
  }
  if (printerspec->duplexmode)
    if (getduplexinfo (&pstate.ppd, printerspec->duplexmode, &pstate.duplexinfo))
    {
      say (IMP_warning, "can't determine how to set printer into duplexmode `%s'\n", printerspec->duplexmode);
      listduplexmodes (&pstate.ppd);
      exit (4);
    }
  say (IMP_entertain2, "paper: %s, dimension = %g %g, imageable area = %g %g %g %g\n", pstate.paperinfo.name, pstate.paperinfo.size.width, pstate.paperinfo.size.height, pstate.paperinfo.imageablearea.llx, pstate.paperinfo.imageablearea.lly, pstate.paperinfo.imageablearea.urx, pstate.paperinfo.imageablearea.ury);
  if (structured)
    psstructure (Header, "DocumentMedia: %s %g %g 0 () ()", pstate.paperinfo.name, pstate.paperinfo.size.width, pstate.paperinfo.size.height);
  else
    pscomment ("paper: %s, dimension = %g %g, imageable area = %g %g %g %g", pstate.paperinfo.name, pstate.paperinfo.size.width, pstate.paperinfo.size.height, pstate.paperinfo.imageablearea.llx, pstate.paperinfo.imageablearea.lly, pstate.paperinfo.imageablearea.urx, pstate.paperinfo.imageablearea.ury);
  psfeature ("PageSize", pstate.paperinfo.name, pstate.paperinfo.invocation);
  if (pstate.duplexinfo.invocation)
    psfeature ("Duplex", pstate.duplexinfo.name, pstate.duplexinfo.invocation);

  pstate.pfonts = avlcreate (sizeof (t_pfont), pfontpcmp, NULL);
  assert (pstate.pfonts);
}

void get_imageablearea (t_rectangle *imageablearea)
{
  *imageablearea = pstate.paperinfo.imageablearea;
}

static int buildencoding (t_charmetric *charmetric, t_softfont *font)
{
  t_pschar	pschar, *pscharp;

  pschar.name = charmetric->name;
  pschar.code = charmetric->code;
  pscharp = avlinsert (font->charstrings, &pschar); assert (pscharp);
  if (charmetric->code != -1)
  {
    assert (!font->encoding[charmetric->code]);
    --font->emptyslots;
    font->encoding[charmetric->code] = charmetric->name;
  }
  return ok;
}

static bool matchinternal (t_pfont *pfont, t_version version, bool exact)
{
  t_afmfile	afmsearch;
  bool		partially = version != VERSION_unspecified && !exact;

  afmsearch.fontname = pfont->font->fontname;

  if ((pfont->u.fontversion = version, psisfontinternal (pfont->font->fontname, &pfont->u.fontversion)) || (partially && (pfont->u.fontversion = VERSION_unspecified, psisfontinternal (pfont->font->fontname, &pfont->u.fontversion))))
  {
    if ((afmsearch.version = exact ? pfont->u.fontversion : version, pfont->afm = avlsearch (pfont->font->afms, &afmsearch)) || (partially && (afmsearch.version = VERSION_unspecified, pfont->afm = avlsearch (pfont->font->afms, &afmsearch))))
    {
      pfont->flags |= FONT_INTERN;
assert ((pfont->flags & (FONT_BUILTIN|FONT_INTERN|FONT_EXTERN)) == FONT_INTERN);
      if (VERBp (IMP_entertain))
      {
	t_buf	version;

	fprintf (stderr, "Font /%s: ", pfont->font->fontname);
	if (versioncmp (pfont->reqversion, pfont->u.fontversion))
	  fprintf (stderr, "requested version %s; ", strversion (version, pfont->reqversion));
	fprintf (stderr, "internal, version %s", strversion (version, pfont->u.fontversion));
	if (versioncmp (pfont->reqversion, pfont->afm->version) || versioncmp (pfont->u.fontversion, pfont->afm->version))
	  fprintf (stderr, ", afm %s", strversion (version, pfont->afm->version));
	fprintf (stderr, ".\n");
      }
      return true;
    }
  }
  return false;
}

static bool matchbuiltin (t_pfont *pfont, t_version version, bool exact)
{
  t_afmfile	afmsearch;
  bool		partially = version != VERSION_unspecified && !exact;

  afmsearch.fontname = pfont->font->fontname;

  if ((pfont->u.fontversion = version, isfontbuiltin (&pstate.ppd, pfont->font->fontname, &pfont->u.fontversion)) || (partially && (pfont->u.fontversion = VERSION_unspecified, isfontbuiltin (&pstate.ppd, pfont->font->fontname, &pfont->u.fontversion))))
  {
    if ((afmsearch.version = exact ? pfont->u.fontversion : version, pfont->afm = avlsearch (pfont->font->afms, &afmsearch)) || (partially && (afmsearch.version = VERSION_unspecified, pfont->afm = avlsearch (pfont->font->afms, &afmsearch))))
    {
      pfont->flags |= FONT_BUILTIN;
assert ((pfont->flags & (FONT_BUILTIN|FONT_INTERN|FONT_EXTERN)) == FONT_BUILTIN);
      if (VERBp (IMP_entertain))
      {
	t_buf	version;

	fprintf (stderr, "Font /%s: ", pfont->font->fontname);
	if (versioncmp (pfont->reqversion, pfont->u.fontversion))
	  fprintf (stderr, "requested version %s; ", strversion (version, pfont->reqversion));
	fprintf (stderr, "built-in, version %s", strversion (version, pfont->u.fontversion));
	if (versioncmp (pfont->reqversion, pfont->afm->version) || versioncmp (pfont->u.fontversion, pfont->afm->version))
	  fprintf (stderr, ", afm %s", strversion (version, pfont->afm->version));
	fprintf (stderr, ".\n");
      }
      return true;
    }
  }
  return false;
}

static int findmatch (t_fontfile *fontfile, va_list va)
{
  t_version	fontversion = va_arg (va, t_version);

  if (!versioncmp (fontfile->version, fontversion))
  {
    t_pfont	*pfont = va_arg (va, t_pfont *);

    pfont->flags |= FONT_EXTERN;
    pfont->u.external = fontfile;
    return true;
  }
  return false;
}

static int matchexternal (t_afmfile *afm, va_list va)
{
  t_version	version = va_arg (va, t_version);

  if (!versioncmp (afm->version, version))
  {
    bool	exact = va_arg (va, bool);
    t_pfont	*pfont = va_arg (va, t_pfont *);

    version = exact ? afm->version : VERSION_unspecified;
    if (avlvamap (pfont->font->fonts, findmatch, version, pfont))
    {
assert ((pfont->flags & (FONT_BUILTIN|FONT_INTERN|FONT_EXTERN)) == FONT_EXTERN);
      pfont->afm = afm;
      if (VERBp (IMP_entertain))
      {
	t_buf	version;
	t_fnbuf	fn;

	fprintf (stderr, "Font /%s: ", pfont->font->fontname);
	if (versioncmp (pfont->reqversion, pfont->u.external->version))
	  fprintf (stderr, "requested version %s; ", strversion (version, pfont->reqversion));
	fprintf (stderr, "external, file %s, version %s", fi_filename (fn, &pfont->u.external->file), strversion (version, pfont->u.external->version));
	if (versioncmp (pfont->reqversion, pfont->afm->version) || versioncmp (pfont->u.external->version, pfont->afm->version))
	  fprintf (stderr, ", afm %s", strversion (version, pfont->afm->version));
	fprintf (stderr, ".\n");
      }
      return true;
    }
  }
  return false;
}

static int findfontwithafm (t_pfont *pfont)
{
  /* Auswahl mit folgender Priorisierung:
   * - built-in font mit passendem afm file
   * - bei fonts mit passendem afm file: hchste version
   * - built-in font mit sonstigem afm file
   * - font mit sonstigem afm file
   */

  /* -------- versuchen, version, fontversion und afmversion passend zu bekommen: */
  if (pfont->reqversion != VERSION_unspecified)	/* es wird explizit eine bestimmte version angefordert */
  {
    if (matchinternal (pfont, pfont->reqversion, true))
      return ok;
    if (matchbuiltin (pfont, pfont->reqversion, true))
      return ok;
    if (avlvarmap (pfont->font->afms, matchexternal, pfont->reqversion, true, pfont))
      return ok;
  }

  /* -------- versuchen, fontversion und afmversion passend zu bekommen: */
  if (matchinternal (pfont, VERSION_unspecified, true))
    return ok;
  if (matchbuiltin (pfont, VERSION_unspecified, true))
    return ok;
  if (avlvarmap (pfont->font->afms, matchexternal, VERSION_unspecified, true, pfont))
    return ok;

  /* -------- versuchen, fontversion oder afmversion passend zu version bekommen: */
  if (pfont->reqversion != VERSION_unspecified)
  {
    if (matchinternal (pfont, pfont->reqversion, false))
      return ok;
    if (matchbuiltin (pfont, pfont->reqversion, false))
      return ok;
    if (avlvarmap (pfont->font->afms, matchexternal, pfont->reqversion, false, pfont))
      return ok;
  }

  /* -------- letzter versuch: irgendwelche versionen */
  if (matchinternal (pfont, VERSION_unspecified, false))
    return ok;
  if (matchbuiltin (pfont, VERSION_unspecified, false))
    return ok;
  if (avlvarmap (pfont->font->afms, matchexternal, VERSION_unspecified, false, pfont))
    return ok;

  return fault;
}

t_pfont *getpfont (const t_cnames *fontnames, t_version fontversion)
{
  int		fni;
  t_pfont	pfsearch, *pfont;
  bool		insert;

  for (fni=0; fni<fontnames->count; fni++)
  {
    cstring fontname = fontnames->names[fni];
    if (!(pfsearch.font = getfontbyname (fontname)))
      continue;
    pfsearch.reqversion = fontversion;
    insert = true;
    pfont = avlinsrch (pstate.pfonts, &insert, &pfsearch); assert (pfont);
    if (insert)
    {
      pfont->flags = 0; /* initialize */
      pfont->misschars = 0;
      if (findfontwithafm (pfont) || completeafm (pfont->afm))
      {
	pfont->flags |= FONT_UNAVAILABLE;
	continue;
      }
      pfont->dup = NULL;
      if (VERBp (IMP_debug2))
	printafmfile (pfont->afm, true);
    }
    elif (pfont->flags & FONT_UNAVAILABLE)
      continue;
    return pfont;
  }
  return NULL;
}

static int resetpfont (t_pfont *pfont)
{
  if (pfont->flags & FONT_DUPLICATED && !(pfont->flags & FONT_INTERN))
  {
    pfont->dup->charstrings = avldestroy (pfont->dup->charstrings);
    for (pfont->dup->emptyslots=0; pfont->dup->emptyslots<256; pfont->dup->emptyslots++)
      pfont->dup->encoding[pfont->dup->emptyslots] = NULL;
    pfont->dup->fontname = strfree (pfont->dup->fontname);
    free (pfont->dup);
    pfont->dup = NULL;
    pfont->flags &= ~FONT_DUPLICATED;
  }
  return ok; /* for avlmap() */
}

void resetpfonts()
{
  avlmap (pstate.pfonts, resetpfont);
}

bool	abortoncharunavail = false;

t_glyph PSCHAR (t_pfont *pfont, t_point fontsize, const t_color *color, const t_cnames *charnames)
{
  int		ci;
  t_glyph	glyph;

#ifdef COLORED
  glyph.color = *color;
#endif
  glyph.pfont = NULL;
  for (ci=0; ci<charnames->count; ci++)
    if (glyph.metric = getcharmetric (pfont->afm, charnames->names[ci]))
    {
      glyph.pfont = pfont;
      break;
    }
  if (!glyph.pfont)
  {
    t_cnames	symbolfontnames[] =
		{
#define IF symbolfontnames /* internal font (see below) */
		  { 1, INTERNALFONT_NAME },
		  { 1, "Symbol" },
		  { 1, "ZapfDingbats" },
		};
    t_cnames	fallbackfontnames[] = /* just in case <pfont> is a funny font */
		{
		  { 1, "Times-Roman" },
		  { 1, "Courier" },
		};
    int		fi;
    t_pfont	*font;
    char	cbuf[1024] = "";

    for (fi=0; fi<ARRAYDIM (symbolfontnames); fi++)
      if (strcmp (pfont->font->fontname, *symbolfontnames[fi].names) && (font = getpfont (symbolfontnames+fi, VERSION_unspecified)))
	for (ci=0; ci<charnames->count; ci++)
	  if (glyph.metric = getcharmetric (font->afm, charnames->names[ci]))
	  {
	    glyph.pfont = font;
	    goto break2a;
	  }
  break2a:
    if (!glyph.pfont)
    {
      for (ci=0; ci<charnames->count; ci++)
	strcat (strcat (cbuf, "/"), charnames->names[ci]);
      for (fi=0; fi<ARRAYDIM (fallbackfontnames); fi++)
	if (strcmp (pfont->font->fontname, *fallbackfontnames[fi].names) && (font = getpfont (fallbackfontnames+fi, VERSION_unspecified)))
	  for (ci=0; ci<charnames->count; ci++)
	    if (glyph.metric = getcharmetric (font->afm, charnames->names[ci]))
	    {
	      if (VERBp (IMP_entertain))
	      {
		bool insert;
		int ninsert = 0;
		if (!pfont->misschars)
		  pfont->misschars = avlcreate (sizeof (cstring *), strpcmp, NULL);
		for (ci=0; ci<charnames->count; ci++)
		{
		  insert = true;
		  avlinsrch (pfont->misschars, &insert, charnames->names+ci);
		  if (insert)
		    ninsert++;
		}
		if (ninsert)
		  say (IMP_entertain, "character %s not available from /%s -- getting it from /%s.\n", cbuf, pfont->font->fontname, font->font->fontname);
	      }
	      glyph.pfont = font;
	      goto break2b;
	    }
    }
  break2b:
    if (!glyph.pfont)
    {
      char	sfbuf[1024];
      char	fbfbuf[1024];
      char	nfbuf[1024];
      char	ebuf[3072];

      *sfbuf = *fbfbuf = *nfbuf = '\0';
      for (fi=0; fi<ARRAYDIM (symbolfontnames); fi++)
      {
	char	*p = getpfont (symbolfontnames+fi, VERSION_unspecified) ? sfbuf : nfbuf;
	if (*p)
	  strcat (p, ", ");
	strcat (strcat (p, "/"), symbolfontnames[fi].names[0]);
      }
      for (fi=0; fi<ARRAYDIM (fallbackfontnames); fi++)
      {
	char	*p = getpfont (fallbackfontnames+fi, VERSION_unspecified) ? fbfbuf : nfbuf;
	if (*p)
	  strcat (p, ", ");
	strcat (strcat (p, "/"), fallbackfontnames[fi].names[0]);
      }
      sprintf (ebuf, "can't find any %s in /%s nor in any symbol font (%s) or any fallback font (%s)\n", cbuf, pfont->font->fontname, sfbuf, fbfbuf);
      if (*nfbuf)
	sprintf (ebuf+strlen (ebuf), "\nunavailable: %s", nfbuf);
      if (abortoncharunavail)
	error (ebuf);
      else
	say (IMP_warning, ebuf);

      glyph.pfont = getpfont (IF, VERSION_unspecified); assert (glyph.pfont);
      glyph.metric = getcharmetric (glyph.pfont->afm, "miss"); assert (glyph.metric);
#ifdef COLORED
      glyph.color = rgbcolor (1, 0, 0);
#endif
    }
#undef IF
  }
  if (glyph.pfont->flags & FONT_INTERN)
    psgetsymbol (glyph.metric->name);
  elif (!(glyph.pfont->flags & FONT_INSTALLED))
  {
    if (glyph.pfont->flags & FONT_BUILTIN)
      psneededresource ("font %s", glyph.pfont->font->fontname);
    elif (glyph.pfont->flags & FONT_EXTERN)
    {
      FILE	*fp;

      if (!(fp = pfreopen (&glyph.pfont->u.external->file, O_rdonly)))
      {
	t_fnbuf	fnbuf;
	error ("Can't open font file \"%s%s%s\".", fi_filename (fnbuf, &glyph.pfont->u.external->file));
      }
      pssupplyresource (fp, "font %s", glyph.pfont->font->fontname);
      pfclose (&glyph.pfont->u.external->file);
    }
    else
      ierror ("pschar()");
    glyph.pfont->flags |= FONT_INSTALLED;
  }
  glyph.fontsize = fontsize;
  return glyph;
}

void getglyphinfo (const t_glyph *glyph, t_point scalefactor, t_glyphinfo *glyphinfo)
{
  t_point	fontsize = glyph->fontsize * scalefactor;

  glyphinfo->charname = glyph->metric->name;
#if 1	/* ascender and descender from font's bounding box: */
  glyphinfo->ascender = glyph->pfont->afm->ascender * fontsize;
  glyphinfo->descender = glyph->pfont->afm->descender * fontsize;
#else	/* ascender and descender from glyph's bounding box: */
  glyphinfo->ascender = glyph->metric->bbox.ury * fontsize;
  glyphinfo->descender = glyph->metric->bbox.lly * fontsize;
#endif
  glyphinfo->width = scalewidth (glyph->metric->width, fontsize);
say (IMP_debug2, "getglyphinfo: /%s asc=%g dsc=%g wx=%g\n", glyphinfo->charname, glyphinfo->ascender, glyphinfo->descender, glyphinfo->width.x);
}

/*\[sep]*/
#ifdef COLORED
void setcolor (const t_color *color)
{
  static bool	mentioned = false;

  if (iscolordevice (&pstate.ppd))
  {
    if (colorpcmp (&pstate.currentcolor, color))
    {
      switch (color->model)
      {
	case CM_void:
	  pssetgray (0);
	  break;
	case CM_rgb:
	case CM_hsb:
	  pssetcolor (color);
	  break;
	default:
	  ierror ("setcolor: illegal color mode\n");
      }
      pstate.currentcolor = *color;
    }
  }
  elif (color->model == CM_rgb || color->model == CM_hsb)
    if (!mentioned)
    {
      say (IMP_warning, "printer `%s' is not a color device!\n", getname (&pstate.ppd));
      mentioned = true;
    }
}
#endif

void setline (const t_line *line)
{
#ifdef COLORED
  setcolor (&line->color);
#endif
  pssetlinewidthandstyle (line);
}

static cstring accenttable[] =
{
  "acute",
  "grave",
  "circumflex",
  "cedilla",
  "tilde",
  "dieresis",
  "caron", /* hachek */
  "macron",
  "underdot",
  "slash",
  "ring",
};

static t_avl *accents = NULL;

static void init_accents()
{
  int		len;
  cstring	*table;

  assert (accents = avlcreate (sizeof (cstring *), strpcmp, NULL));
  for (table=accenttable, len=ARRAYDIM (accenttable); len--; table++)
    assert (avlinsert (accents, table));
}

typedef struct
{
  cstring	name;		/* wurde encodefont() bergeben */
  int		code;		/* array index, an der der buchstabe im font /Encoding steht */
  int		mapindex;	/* wird von encodefont() geliefert */
} t_map;

static struct
{
  t_pfont	*pfont;
  t_point	fontsize;

  bool		useorig;		/* original font can only be used if only encoded characters are needed */
  t_avl		*charstringmap;		/* mapping for all characters used, built by getfont() */
  uchar		encodingmap[256];	/* lookup table for show(), built by setfont() */
  int		nmapped;		/* number of different characters used */
  int		addmap;			/* number of encoding changes (additions only) */ 
} state;

static int mappcmp (const t_map *l, const t_map *r)
{
  return strcmp (l->name, r->name);
}

void findfont (t_pfont *pfont, t_point fontsize)
{
  if (!accents)
    init_accents();

  if (state.charstringmap)
    state.charstringmap = avldestroy (state.charstringmap);

  state.pfont = pfont;
  state.fontsize = fontsize;
  state.useorig = structured ? true : state.pfont->flags & FONT_DUPLICATED ? false : true;
  state.charstringmap = avlcreate (sizeof (t_map), mappcmp, NULL); assert (state.charstringmap);
  state.nmapped = state.addmap = 0;
}

static void dupfont (t_pfont *pfont)
{
  int		code;

  pfont->dup = getmem (sizeof (t_softfont));
  if (pfont->flags & FONT_INTERN)
    pfont->dup->fontname = pfont->font->fontname;
  else
  {
    t_buf newname;

    sprintf (newname, "Dup-%s", pfont->font->fontname);
    pfont->dup->fontname = strsave (newname);
    psdupfont (pfont->font->fontname, pfont->dup->fontname);
  }
  pfont->dup->charstrings = avlcreate (sizeof (t_pschar), pscharpcmp, NULL);
  for (code=256; code--; )
    pfont->dup->encoding[code] = NULL;
  pfont->dup->emptyslots = 256;
  if (!structured)
    avlamap (pfont->afm->metrics, buildencoding, pfont->dup);
  pfont->flags |= FONT_DUPLICATED;
}

static int adaptmap (t_map *charmap)
{
  t_pschar	searchchar, *foundchar;

  assert (state.pfont->dup);
  searchchar.name = charmap->name;
  if (state.pfont->dup->encoding[charmap->code] && streq (state.pfont->dup->encoding[charmap->code], charmap->name))	/* an gleicher stelle codiert */
    ;
  elif (foundchar = avlsearch (state.pfont->dup->charstrings, &searchchar))			/* codiert, aber an anderer stelle */
    charmap->code = foundchar->code;
  else												/* noch nicht codiert */
  {
    charmap->code = -1;
    state.addmap++;
  }
  return ok;
}

/* um mglichst wenig buchstaben ummappen zu mssen, werden die codes ber einen zwischenschritt abgebildet.  */
int encodefont (cstring charname)		/* return code: fault: more than 256 encoded chars; 0-255: index into encodingmap[] */
{
  t_map	search, *charmap;
  bool	insert;

  {/* "In Level 1 implementations, if an encoding vector includes the name of
    * an accented character, it must also include the names of the components
    * of that character."
    * [PostScript Language Reference Manual, Second Edition, page 276]
    *
    * assumption: all diacritic characters have a name that is constructed of the base name with length 1 followed by the diacritical mark's name with length > 0.
    */
    cstring	accent = charname+1;
    char	base[2];

    if (languagelevels.printer == 1 && *accent && avlsearch (accents, &accent))
    {
      base[0] = *charname;
      base[1] = '\0';
      if (getcharmetric (state.pfont->afm, base) && getcharmetric (state.pfont->afm, accent))
	if (encodefont (strunique (base)) == fault || encodefont (accent) == fault) /* strunique (base), da sonst, falls base noch nicht encodiert wurde, der pointer auf den stack verwendet wird! */
	  return fault;
    }
  }
  if (state.useorig)
  {
    const t_charmetric	*charmetric = getcharmetric (state.pfont->afm, charname);
    int			charcode = charmetric->code;

    charname = charmetric->name;		/* both are equal, but not necessarily identical: the <charname> passed to this function may reside on the stack */
    if (charcode != -1)
    {
      insert = true;
      search.name = charname;
      charmap = avlinsrch (state.charstringmap, &insert, &search); assert (charmap);
      if (insert)
      {
	assert (state.nmapped < 256);
	charmap->code = charcode;		/* 1:1 abbildung */
	charmap->mapindex = state.nmapped++;
      }
      return charmap->mapindex;
    }
    else	/* der eingebaute font kann nicht weiter benutzt werden, weil sein /Encoding nicht verndert werden kann */
    {
      if (state.nmapped == 256)	/* won't fit anyway */
	return fault;

      state.useorig = false;

      if (!(state.pfont->flags & FONT_DUPLICATED))	/* der font wurde noch nicht dupliziert */
	dupfont (state.pfont);

      avlmap (state.charstringmap, adaptmap);
    }
  }

  search.name = charname;
  insert = state.nmapped < 256;						/* insert only if still empty slot in encoding vector */
  charmap = avlinsrch (state.charstringmap, &insert, &search);
  if (charmap)
  {
    if (insert)
    {
      const t_charmetric	*charmetric = getcharmetric (state.pfont->afm, charname);
      int			charcode = charmetric->code;
      t_pschar			searchchar, *foundchar;

      charname = charmetric->name;		/* both are equal, but not necessarily identical: the <charname> passed to this function may lie on the stack */
      searchchar.name = charname;
      if (charcode != -1 && state.pfont->dup->encoding[charcode] && streq (state.pfont->dup->encoding[charcode], charname))	/* an gleicher stelle codiert */
	charmap->code = charcode;
      elif (foundchar = avlsearch (state.pfont->dup->charstrings, &searchchar))			/* codiert, aber an anderer stelle */
      {
	if ((charmap->code = foundchar->code) == -1)
	  state.addmap++;
      }
      else											/* noch nicht codiert */
      {
	charmap->code = -1;
	state.addmap++;
      }
      charmap->mapindex = state.nmapped++;
    }
    return charmap->mapindex;
  }
  else
    return fault;
}

/* noch nicht codierte chars auf derzeit nicht gebrauchte positionen mappen */
static int addmap (t_map *charmap, va_list va)
{
  if (charmap->code == -1)
  {
    cstring	*newencoding = va_arg (va, cstring *);
    int		charcode = getcharmetric (state.pfont->afm, charmap->name)->code;

    /* since different methods for finding available slots are used, newencoding[] has to be checked */

    /* ist der buchstabe blicherweise gemapped, und sein slot nicht gebraucht: */
    if (charcode != -1 && !state.pfont->dup->encoding[charcode] && !newencoding[charcode])
      charmap->code = charcode;
    else	/* eine andere stelle suchen: */
    {
      int	*emptyslots = va_arg (va, int *);
      int	*code = va_arg (va, int *);

      assert (*emptyslots);
      while (state.pfont->dup->encoding[--*code] || newencoding[*code])
	if (*code <= 0)
	  abort();
      charmap->code = *code;
      if (!--*emptyslots)	/* reinitialize */
	*code = 256;
    }
    newencoding[charmap->code] = charmap->name;
  }
  return ok;
}

/* noch nicht codierte chars auf derzeit nicht gebrauchte positionen mappen */
static int encode (t_map *charmap, va_list va)
{
  assert (charmap->code != -1);
  state.encodingmap[charmap->mapindex] = charmap->code;
  return ok;
}

void setfont()
{
  cstring	fontname;

  fontname = state.useorig ? state.pfont->font->fontname : state.pfont->dup->fontname;
  if (!pstate.currentfont.pfont || pstate.currentfont.name != fontname /* strcmp() ain't necessary */ || pstate.currentfont.size != state.fontsize || state.addmap)
  {
    if (state.addmap)
    {
      int	emptyslots, code;		/* to speed up the search for unneeded characters */
      cstring	newencoding[256];

      emptyslots = state.pfont->dup->emptyslots;
      for (code=0; code<256; code++)
	newencoding[code] = NULL;
      avlvamap (state.charstringmap, addmap, newencoding, &emptyslots, &code);
      psfindfont (fontname);
      psaddmap (state.pfont->dup, newencoding, state.addmap);
      psscalesetfont (state.fontsize);
    }
    else
      psfindscalesetfont (fontname, state.fontsize);

    pstate.currentfont.pfont = state.pfont;
    pstate.currentfont.name = fontname;
    pstate.currentfont.size = state.fontsize;
  }
  avlvamap (state.charstringmap, encode);
}

void show (uchar *s, size_t len)
{
  size_t	i;

  for (i = len; i--; )
    s[i] = state.encodingmap[s[i]];
  psshow (s, len);
}

void kxshow (uchar *s, const t_point *kx, size_t len)
{
  size_t	i;

  for (i = len; i--; )
    s[i] = state.encodingmap[s[i]];
  pskxshow (s, kx, len);
}

void kshow (uchar *s, const t_width *k, size_t len)
{
  size_t	i;

  for (i = len; i--; )
    s[i] = state.encodingmap[s[i]];
  pskshow (s, k, len);
}
