/*
 * program: psfilt
 * file: ppd.c
 *
 * Copyright  1992 1993 Robert Joop
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: ppd.c,v $
 * Revision 1.4  1994/07/09  16:44:06  rj
 * a lot of const's removed and added
 *
 * Revision 1.3  1994/03/18  12:07:35  rj
 * support for duplex mode added
 * lots of missing `#ifdef COLORED' inserted
 *
 * Revision 1.2  1994/01/09  23:46:04  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.1.1.1  1993/12/31  20:56:42  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: ppd.c,v 1.4 1994/07/09 16:44:06 rj Exp $";

#include "psfilt.h"
#include "ppd.h"
#include "lex-ppd.h"
#include "error.h"
#include "str.h"

/*\[sep]*/
static int ppderror (cstring fmt, ...)
{
  t_buf	buf;
  va_list va;
  va_start (va, fmt);
  vsprintf (buf, fmt, va);
  va_end (va);
  ppdsay (IMP_error, "parse error: %s", buf);
  return fault;
}

/*\[sep]*/
typedef enum
{
  PPD_cap,
  PPD_cap_req,
  PPD_caplist,
  PPD_caplist_req,
  PPD_font,
  PPD_symbol,
} t_keytype;

typedef struct
{
  string	name;
  t_keytype	type;
} t_key;

static int keypcmp (const t_key *l, const t_key *r)
{
  return strcmp (l->name, r->name);
}

static t_avl	*keys;

void prepare_key (string name, t_keytype type)
{
  t_key	key;

  key.name = name;
  key.type = type;
  if (!avlinsert (keys, &key))
    ierror ("prepare_key (%s)", name);
}

static void init_ppd()
{
  static bool	initialized = false;

  if (initialized)
    return;

  keys = avlcreate (sizeof (t_key), keypcmp, NULL);

  prepare_key ("ModelName", PPD_cap); /* only required since version 4.0 */
  prepare_key ("NickName", PPD_cap_req);
  prepare_key ("Product", PPD_cap_req);
  prepare_key ("ColorDevice", PPD_cap);
  prepare_key ("LanguageLevel", PPD_cap);

  prepare_key ("Font", PPD_font);

  prepare_key ("PageSize", PPD_caplist_req);
  prepare_key ("PageRegion", PPD_caplist_req);
  prepare_key ("PaperDimension", PPD_caplist_req);
  prepare_key ("ImageableArea", PPD_caplist_req);

  prepare_key ("Duplex", PPD_caplist);

  prepare_key ("SymbolValue", PPD_symbol);

  initialized = true;
}

static const t_key *getkey (string name)
{
  t_key		srch;

  srch.name = name;
  return avlsearch (keys, &srch);
}

/*\[sep]*/
int ppdcappcmp (const t_ppdcap *l, const t_ppdcap *r)
{
  return strcmp (l->name, r->name);
}

int ppdcaplistpcmp (const t_ppdcaplist *l, const t_ppdcaplist *r)
{
  return strcmp (l->name, r->name);
}

int ppdfontpcmp (const t_ppdfont *l, const t_ppdfont *r)
{
  int	cp;

  return (cp = strcmp (l->fontname, r->fontname)) ? cp : versioncmp (l->version, r->version);
}

int ppdsymbolpcmp (const t_ppdsymbol *l, const t_ppdsymbol *r)
{
  return strcmp (l->name, r->name);
}

/*\[sep]*/
void clear_ppd (t_ppd *ppd)
{
  init_ppd();

  ppd->caps = avlcreate (sizeof (t_ppdcap), ppdcappcmp, NULL);
  ppd->caplists = avlcreate (sizeof (t_ppdcaplist), ppdcaplistpcmp, NULL);
  ppd->fonts._default = NULL;
  ppd->fonts.fonts = avlcreate (sizeof (t_ppdfont), ppdfontpcmp, NULL);
  ppd->symbols = avlcreate (sizeof (t_ppdsymbol), ppdsymbolpcmp, NULL);
}

/*\[sep]*/
int handle_key (t_ppd *ppd, string name)
{
  return ok;
}

int handle_cap (t_ppd *ppd, string name, t_list *values)
{
  const t_key	*key;

  if (key = getkey (name))
    switch (key->type)
    {
      case PPD_cap:
      case PPD_cap_req:
	{
	  t_ppdcap	srch;

	  if (lilength (values) < 1)
	    return ppderror ("too few args");
	  srch.name = strunique (name);
	  srch.value = *(t_ppdvalue *)ligoto (values, 0);
	  avlinsert (ppd->caps, &srch);
	}
	break;
      default:
	return ppderror ("wrong format for *%s capability", name);
    }
  return ok;
}

static int ppdscanversion (cstring name, t_version *version, cstring value)
{
  int	rc;

  if (rc = scanversion (version, *value == '(' ? value+1 : value))
    ppdsay (IMP_warning, "illegal %s version \"%s\"", name, value);
  return rc;
}

int handle_caplist (t_ppd *ppd, string name, const t_ppdoption *option, t_list *values)
{
  const t_key	*key;

  if (key = getkey (name))
    switch (key->type)
    {
      case PPD_caplist:
      case PPD_caplist_req:
	{
	  bool		insert = true;
	  t_ppdcaplist	srchlist, *list;
	  t_ppdcap	srch;

	  if (lilength (values) < 1)
	    return ppderror ("too few args for *%s capability", name);

	  srchlist.name = strunique (name);
	  list = avlinsrch (ppd->caplists, &insert, &srchlist);
	  if (insert)
	  {
	    list->_default = NULL;
	    list->list = avlcreate (sizeof (t_ppdcap), ppdcappcmp, NULL);
	  }
	  srch.name = option->value;
	  srch.value = *(t_ppdvalue *)ligoto (values, 0);
	  avlinsert (list->list, &srch);
	}
	break;
      case PPD_font:
	{
	  t_ppdvalue	*encoding, *version;
	  t_ppdfont	font;

	  if (option->type != PPD_StringOption)
	    return ppderror ("illegal fontname option");
	  if (lilength (values) < 2)
	    return ppderror ("too few args for *Font %s", option->value);
	  if ((encoding = (t_ppdvalue *)ligoto (values, 0))->type != PPD_StringValue)
	    return ppderror ("illegal font encoding for *Font %s", option->value);
	  if ((version = (t_ppdvalue *)ligoto (values, 1))->type != PPD_InvocationValue)
	    return ppderror ("illegal version for *Font %s (type %d)", option->value, version->type);

	  font.fontname = option->value;
#if BINARY_STRINGS
	  if (ppdscanversion ("font", &font.version, version->value.str))
#else
	  if (ppdscanversion ("font", &font.version, version->value))
#endif
	    return fault;
	  avlinsert (ppd->fonts.fonts, &font);
	}
	break;
      case PPD_symbol:
	{
	  t_ppdsymbol	symbol;

	  if (option->type != PPD_SymbolOption)
	    return ppderror ("illegal symbol option");
	  symbol.name = option->value;
	  avlinsert (ppd->symbols, &symbol);
	}
	break;
      default:
	return ppderror ("wrong format for *%s capability", name);
    }
  return ok;
}

int handle_default (t_ppd *ppd, string name, const t_ppdvalue *stringval)
{
  const t_key	*key;

  if (key = getkey (name))
    switch (key->type)
    {
      case PPD_caplist:
      case PPD_caplist_req:
      case PPD_font:
    }
  return ok;
}

int handle_query (t_ppd *ppd, string name, const t_ppdvalue *quotedval)
{
#if 0
  /* this is not a printer manager, we don't ask the printer */
  const t_key	*key;

  if (key = getkey (name))
    ;
#endif
  return ok;
}

/*\[sep]*/
static const t_ppdcap *getcap (const t_ppd *ppd, string name)
{
  t_ppdcap	srch;

  srch.name = name;
  return avlsearch (ppd->caps, &srch);
}

static const t_ppdcaplist *getcaplist (const t_ppd *ppd, string name)
{
  t_ppdcaplist	srch;

  srch.name = name;
  return avlsearch (ppd->caplists, &srch);
}

static const t_ppdcap *getlistcap (const t_ppd *ppd, string listname, string entryname)
{
  const t_ppdcaplist	*list;
  t_ppdcap		srch;

  srch.name = entryname;
  return (list = getcaplist (ppd, listname)) ? avlsearch (list->list, &srch) : NULL;
}

/*\[sep]*/
static int countfont (const t_ppdfont *font)
{
  return true;
}

static int checkkey (const t_key *key, va_list va)
{
  t_ppd	*ppd = va_arg (va, t_ppd *);

  switch (key->type)
  {
  /* optional: */
    case PPD_cap:
    case PPD_caplist:
    case PPD_symbol:
      break;
  /* required: */
    case PPD_cap_req:
      {
	const t_ppdcap *cap = getcap (ppd, key->name);
	if (!cap)
	  error ("required *%s entry missing on PPD file for %s", key->name, va_arg (va, cstring));
      }
      break;
    case PPD_caplist_req:
      {
	const t_ppdcaplist *caplist = getcaplist (ppd, key->name);
	if (!caplist || !caplist->list)
	  error ("required *%s entries missing on PPD file for %s", key->name, va_arg (va, cstring));
      }
      break;
    case PPD_font:
      {
	if (!avlmap (ppd->fonts.fonts, countfont))
	  error ("required *%s entries missing on PPD file for %s", key->name, va_arg (va, cstring));
      }
      break;
    default:
      ierror ("checkkey (%s, %d)", key->name, key->type);
  }
  return ok;
}

int finish_ppd (string printername, t_ppd *ppd)
{
  return avlvamap (keys, checkkey, ppd, printername);
}

/*\[sep]*/
int printcap (const t_ppdcap *cap, FILE *fp)
{
#if BINARY_STRINGS
  return fprintf (fp, "cap %s = %s\n", cap->name, cap->value.type == PPD_QuotedValue ? cap->value.value.bin.buf : cap->value.value.str) == EOF;
#else
  return fprintf (fp, "cap %s = %s\n", cap->name, cap->value.value) == EOF;
#endif
}

int printcaplist (const t_ppdcaplist *caplist, FILE *fp)
{
  return fprintf (fp, "-------\ncaplist %s:\n", caplist->name) == EOF || avlamap (caplist->list, printcap, fp);
}

int printppdfont (const t_ppdfont *ppdfont, FILE *fp)
{
  t_buf	version;
  return fprintf (fp, "font %s: %s\n", ppdfont->fontname, strversion (version, ppdfont->version)) == EOF;
}

int printsymbol (const t_ppdsymbol *symbol, FILE *fp)
{
  return fprintf (fp, "symbol %s = %s\n", symbol->name, symbol->value);
}

void print_ppd (const t_ppd *ppd, FILE *fp)
{
  avlamap (ppd->caps, printcap, fp);
  avlamap (ppd->caplists, printcaplist, fp);
  fprintf (fp, "-------\nfonts:\n");
  avlamap (ppd->fonts.fonts, printppdfont, fp);
  fprintf (fp, "default = %s\n", ppd->fonts._default);
  avlamap (ppd->symbols, printsymbol, fp);
}

/*\[banner "information functions"]-------------------------------------------*/
#if BINARY_STRINGS
const t_binstr *getquotedval (const t_ppd *ppd, cstring name)
{
  const t_ppdcap	*cap;

  return (cap = getcap (ppd, name)) && cap->value.type == PPD_QuotedValue ? &cap->value.value.bin : NULL;
}

cstring getstringval (const t_ppd *ppd, cstring name)
{
  const t_ppdcap	*cap;

  return (cap = getcap (ppd, name)) && cap->value.type == PPD_StringValue ? cap->value.value.str : NULL;
}

const t_binstr *getname (t_ppd *ppd)
{
  const t_binstr *val;
  return (val = getquotedval (ppd, "NickName")) || (val = getquotedval (ppd, "Product")) || (val = getquotedval (ppd, "ModelName")) ? val : NULL;
}

const t_binstr *getproduct (t_ppd *ppd)
{
  const t_binstr *val;

  return (val = getquotedval (ppd, "Product")) ? val : NULL;
}

bool iscolordevice (const t_ppd *ppd)
{
  cstring val;

  return (val = getstringval (ppd, "ColorDevice")) && streq (val, "True");
}
#else
string getname (t_ppd *ppd)
{
  const t_ppdcap *cap;
  return (cap = getcap (ppd, "NickName")) || (cap = getcap (ppd, "Product")) || (cap = getcap (ppd, "ModelName")) ? cap->value.value : NULL;
}

string getproduct (t_ppd *ppd)
{
  const t_ppdcap *cap;

  return (cap = getcap (ppd, "Product")) ? cap->value.value : NULL;
}

bool iscolordevice (const t_ppd *ppd)
{
  const t_ppdcap	*cap;

  return (cap = getcap (ppd, "ColorDevice")) && streq (cap->value.value, "True");
}
#endif

bool isfontbuiltin (const t_ppd *ppd, string fontname, t_version *fontversion)
{
  t_ppdfont	search, *fontinfo;

  search.fontname = fontname;
  search.version = *fontversion;
  if (fontinfo = avlsearch (ppd->fonts.fonts, &search))
  {
    *fontversion = fontinfo->version;
    return true;
  }
  return false;
}

int getlanguagelevel (const t_ppd *ppd)
{
  const t_ppdcap	*cap;
  int			level;

  return (cap = getcap (ppd, "LanguageLevel")) && sscanf (cap->value.value, "%d", &level) == 1 ? level : 1;
}

int getpaperinvocation (const t_ppd *ppd, string papertype, string *invocation)
{
  const t_ppdcap	*cap;

  if (!(cap = getlistcap (ppd, "PageSize", papertype)))
    return fault;
#if BINARY_STRINGS
#else
  *invocation = cap->value.value;
#endif
  return ok;
}

int getpaperdimension (const t_ppd *ppd, string papertype, t_dimension *size)
{
  const t_ppdcap	*cap;

  if (!(cap = getlistcap (ppd, "PaperDimension", papertype)))
    return fault;
#if BINARY_STRINGS
#else
  if (sscanf (cap->value.value, "%f%f", &size->width, &size->height) != 2)
#endif
    return fault;
  return ok;
}

int getimageablearea (const t_ppd *ppd, string papertype, t_rectangle *area)
{
  const t_ppdcap	*cap;

  if (!(cap = getlistcap (ppd, "ImageableArea", papertype)))
    return fault;
#if BINARY_STRINGS
#else
  if (sscanf (cap->value.value, "%f%f%f%f", &area->llx, &area->lly, &area->urx, &area->ury) != 4)
#endif
    return fault;
  return ok;
}

int getpaperinfo (const t_ppd *ppd, string papertype, t_paperinfo *paperinfo)
{
  if (getpaperinvocation (ppd, papertype, &paperinfo->invocation) || getpaperdimension (ppd, papertype, &paperinfo->size) || getimageablearea (ppd, papertype, &paperinfo->imageablearea))
    return fault;
  paperinfo->name = papertype;
  return ok;
}

int getduplexinfo (const t_ppd *ppd, string duplexmode, t_duplexinfo *duplexinfo)
{
  const t_ppdcap	*cap;

  if (!(cap = getlistcap (ppd, "Duplex", duplexmode)))
    return fault;
  duplexinfo->name = duplexmode;
#if BINARY_STRINGS
#else
  duplexinfo->invocation = cap->value.value;
#endif
  return ok;
}

static int printduplexmode (const t_ppdcap *mode)
{
  say (IMP_error, "\t%s\n", mode->name);
  return ok;
}

void listduplexmodes (const t_ppd *ppd)
{
  const t_ppdcaplist	*caplist;

  if (!(caplist = getcaplist (ppd, "Duplex")))
    say (IMP_error, "the printer hasn't got any duplex modes.\n");
  else
  {
    say (IMP_error, "the printer has got the following duplex modes:\n");
    avlmap (caplist->list, printduplexmode);
  }
}
