/*
 * program: psfilt
 * file: parse-input.y
 *
 * Copyright  1992 1993 Robert Joop
 *
 * grammatical parser for all types of input
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: parse-input.y,v $
 * Revision 1.12  1994/08/19  09:36:39  rj
 * IT_TEXTline mode for mail and news
 *
 * Revision 1.11  1994/08/18  12:04:35  rj
 * don't ignore text after line comments
 *
 * Revision 1.10  1994/07/16  13:36:16  rj
 * refrain from guessing troff translation when the line ends in a backslash
 *
 * Revision 1.9  1994/07/09  17:34:39  rj
 * input token buffers (especially in mail mode) are no longer in danger of overflowing
 *
 * Revision 1.8  1994/07/09  17:21:50  rj
 * stopping to read at NUL characters stopped
 *
 * Revision 1.7  1994/07/09  16:44:03  rj
 * a lot of const's removed and added
 *
 * Revision 1.6  1994/07/09  15:34:20  rj
 * stuff for configuration via GNU autoconf added
 *
 * Revision 1.5  1994/03/18  12:07:32  rj
 * support for duplex mode added
 * lots of missing `#ifdef COLORED' inserted
 *
 * Revision 1.4  1994/01/09  23:45:58  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.3  1994/01/03  13:22:52  rj
 * fix for flex 2.3.7 (unsigned char *yytext) vs. flex 2.4.5 (char *yytext).
 *
 * Revision 1.2  1993/12/31  22:24:53  rj
 * minor fix.
 *
 * Revision 1.1.1.1  1993/12/31  20:56:45  rj
 * erster cvs import.
 *
 */

%{

static const char RCSId[] = "$Id: parse-input.y,v 1.12 1994/08/19 09:36:39 rj Exp $";

/* AIX requires this to be the first thing in the file.  */
#ifdef __GNUC__
#define alloca __builtin_alloca
#else /* not __GNUC__ */
#if HAVE_ALLOCA_H
#include <alloca.h>
#else /* not HAVE_ALLOCA_H */
#ifdef _AIX
 #pragma alloca
#else /* not _AIX */
char *alloca ();
#endif /* not _AIX */
#endif /* not HAVE_ALLOCA_H */
#endif /* not __GNUC__ */

#if YYDEBUG_INPUT
#  define YYDEBUG	1
#endif
#if YYDEBUG
#  define YYERROR_VERBOSE
#  define YYPRINT(file, type, value)   yyprintinput (file, type, value)
  static void yyprintinput();
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>	/* atof() */

#include "psfilt.h"
#include "lex-input.h"
#include "job.h"
#include "troff.h"
#include "tex.h"
#include "ucs2list.h"
#include "adobe.h"
#include "mnemo.h"
#include "iso646.h"
#include "iso8859.h"
#include "cp437.h"
#include "str.h"
#include "strdeesc.h"
#include "mem.h"
#include "mail.h"
#include "verbose.h"
#include "error.h"

typedef struct t_parsejob
{
  const t_job		*job;
  FILE			*fp;
  struct t_parsejob	*parent;

  t_input		input;
  int			iline;		/* input line, for nice error messages */
  int			icolumn;	/* input column to compute tabstops */

  t_inout		inout;

  t_foutput		output;
#ifdef COLORED
  /* the following member corresponds to output.color[]: */
  t_list		*colorstack[N_IOM];	/* list of t_color */
#endif
  /* the following four members correspond to output.font[]: */
  t_list		*fontstack[N_IOM];	/* list of t_pfont */
  t_pfont		*pfont[N_IOM];
  t_list		*sizestack[N_IOM];	/* list of t_point */
  t_point		fontsize[N_IOM];

  enum
  {
    MAILvoid,
    MAILheader,
    MAILbody,
  }			mailstate;
  int			messageno;
} t_parsejob;

static void freeparsejob (t_parsejob *parsejob)
{
  t_iomode	iomode;

  for (iomode=0; iomode<N_IOM; iomode++)
  {
    lidestroy (parsejob->fontstack[iomode]);
    lidestroy (parsejob->sizestack[iomode]);
  }
}

static t_parsejob	*state;
static t_list		*statestack;

#define yyparse	parse_command
#define yylex	lex_command
#define yyerror	commanderror

static int		lex_command (/*YYSTYPE *lvalp*/);

static t_list		*ytoken;
static bool		ybreak;

static void		jobwarning (const t_parsejob *state, cstring fmt, ...);
#ifndef COLORED
static void		jobcolorwarning (const t_parsejob *state);
#endif
static volatile void	joberror (const t_parsejob *state, cstring fmt, ...);
static void		setfontm (t_parsejob *state, t_pfont *pfont, t_iomode iomode);
static void		pushfontm (t_parsejob *state, t_pfont *pfont, t_iomode iomode);
static void		popfontm (t_parsejob *state, t_iomode iomode);
static void		pushfont (t_parsejob *state, t_pfont *pfont);
static void		popfont (t_parsejob *state);
static void		setsizem (t_parsejob *state, t_point size, t_iomode iomode);
static void		pushsizem (t_parsejob *state, t_point size, t_iomode iomode);
static void		popsizem (t_parsejob *state, t_iomode iomode);
static void		pushsize (t_parsejob *state, t_point size);
static void		popsize (t_parsejob *state);
#ifdef COLORED
static void		setcolorm (t_parsejob *state, t_color color, t_iomode iomode);
static void		pushcolorm (t_parsejob *state, t_color color, t_iomode iomode);
static void		popcolorm (t_parsejob *state, t_iomode iomode);
#if 0 /* (unused) */
static void		pushcolor (t_parsejob *state, t_color color);
static void		popcolor (t_parsejob *state);
#endif /* (unused) */
#endif
static void		appendpsglyph (t_list *token, t_parsejob *state, string name);
static void		appendxucs2glyph (t_list *token, t_parsejob *state, const t_xucs2 *xucs2);
static void		appendstring (t_list *token, t_parsejob *state, cstring);
static void		appendstringssmall (t_list *token, t_parsejob *state, ...);
static const t_xucs2	*name2glyph (string name);
static t_list		*parse_line (string str, t_parsejob *state);

static void commanderror (cstring);

%}

%pure_parser

%union
{
  t_iomode	iomode;
  struct
  {
    string	name;
    t_version	version;
  }		font;
  t_version	version;
  t_pfont	*pfont;
  t_linestyle	linestyle;
  t_adjustment	adjustment;
  t_dimension	dimension;
  bool		boolean;
  double	factor;
  t_point	size;
#ifdef COLORED
  t_color	color;
#endif
  const t_xucs2	*xucs2;
  string	str;
  int		number;
  t_list	*tablist;
}

%token LITERAL TRANSLATED
%token SET PUSH POP
%token FONT SIZE COLOR
%token SEPARATOR
%token HEADER
%token TABULATOR
%token LEFT RIGHT UP DOWN
%token CENTER
%token NEED
%token HORIZONTAL VERTICAL
%token COLUMNS
%token WIDTH HEIGHT
%token SOLID DASHED DOTTED DASHDOTTED DASHDOTDOTTED
%token MISS

%token <xucs2>		XUCS2
%token <str>		STR SYMBOL
%token <boolean>	PLUSMINUS
%token <factor>		FACTOR
%token <number>		NUMBER
%token <size>		PTSIZE
%token <color>		COLORSPEC

%type <iomode>		iomode
%type <font>		font
%type <version>		fontversion
%type <pfont>		pfont
%type <size>		size
%type <color>		color opt_color
%type <linestyle>	linestyle
%type <str>		header_str
%type <adjustment>	opt_hor_adjust hor_adjust
%type <dimension>	dimension
%type <size>		width sep_width
%type <size>		height
%type <str>		str strtoken
%type <tablist>		tab_list
%type <str>		name

%start commands

%%

empty		:
		;

commands	:	command
		|	command ';' commands
		;

SETopt		:	empty
		|	SET
		;

command		:	iomode PUSH pfont
			{
			  pushfontm (state, $3, $1);
			}
		|	iomode SETopt pfont
			{
			  setfontm (state, $3, $1);
			}
		|	iomode POP FONT
			{
			  popfontm (state, $1);
			}
		|	iomode PUSH SIZE size
			{
			  pushsizem (state, $4, $1);
			}
		|	iomode SETopt SIZE size
			{
			  setsizem (state, $4, $1);
			}
		|	iomode PUSH SIZE PLUSMINUS size
			{
			  pushsizem (state, state->fontsize[$1] + ($4 ? $5 : -$5), $1);
			}
		|	iomode SETopt SIZE PLUSMINUS size
			{
			  setsizem (state, state->fontsize[$1] + ($4 ? $5 : -$5), $1);
			}
		|	iomode PUSH SIZE FACTOR
			{
			  pushsizem (state, $4 * state->fontsize[$1], $1);
			}
		|	iomode SETopt SIZE FACTOR
			{
			  setsizem (state, $4 * state->fontsize[$1], $1);
			}
		|	iomode POP SIZE
			{
			  popsizem (state, $1);
			}
		|	iomode PUSH color
			{
#ifdef COLORED
			  pushcolorm (state, $3, $1);
#endif
			}
		|	iomode SETopt color
			{
#ifdef COLORED
			  setcolorm (state, $3, $1);
#endif
			}
		|	iomode POP COLOR
			{
#ifdef COLORED
			  popcolorm (state, $1);
#endif
			}
		|	chars_and_strings
		|	TABULATOR NUMBER COLUMNS FACTOR
			{
			  state->inout.tabmode = TMsimple;
			  state->input.tabcolumns = $2;
			  state->output.tabs.relative = true;
			  state->output.tabs.width.factor = $4;
			}
		|	TABULATOR NUMBER COLUMNS width
			{
			  state->inout.tabmode = TMsimple;
			  state->input.tabcolumns = $2;
			  state->output.tabs.relative = false;
			  state->output.tabs.width.size = $4;
			}
		|	TABULATOR tab_list
			{
/*
			  liappend (ytoken, commandtoken (COMMAND_TabList, $2));
*/
			  ierror ("sorry, not implemented: tab list");
			}
		|	NEED dimension
			{
			  liappend (ytoken, commandtoken (COMMAND_need_space2, $2));
			}
		|	SEPARATOR opt_color linestyle sep_width
			{
			  t_line	line = {
#ifdef COLORED
						 $2,
#endif
						     $4, $3 };

			  liappend (ytoken, commandtoken (COMMAND_Separator, line, line.width+state->fontsize[state->inout.mode]));
			  /* don't break, so the rest of the line will be ignored in output.c: fillpage{elem,}() */
			}
		|	HEADER hor_adjust str
			{
			  liappend (ytoken, commandtoken (COMMAND_Banner, parse_line ($3, state), $2));
			  if ($3)
			    free ($3);
			  /* don't break, so the rest of the line will be ignored in output.c: fillpage{elem,}() */
			}
		|	header_str
			{
			  liappend (ytoken, commandtoken (COMMAND_Banner, parse_line ($1, state), ADJcenter));
			  if ($1)
			    free ($1);
			  /* don't break, so the rest of the line will be ignored in output.c: fillpage{elem,}() */
			}
		|	header_str str str
			{
			  liappend (ytoken, commandtoken (COMMAND_Banner3, parse_line ($1, state), parse_line ($2, state), parse_line ($3, state)));
			  if ($1)
			    free ($1);
			  if ($2)
			    free ($2);
			  if ($3)
			    free ($3);
			  /* don't break, so the rest of the line will be ignored in output.c: fillpage{elem,}() */
			}
		;

header_str	:	HEADER str	{ $$ = $2; }

iomode		:	empty		{ $$ = state->inout.mode; }
		|	LITERAL		{ $$ = IOM_lit; }
		|	TRANSLATED	{ $$ = IOM_trlt; }
		;

pfont		:	font
			{
			  t_cnames	fontname;

			  fontname.names[0] = $1.name;
			  fontname.count = 1;
			  if (!($$ = getpfont (&fontname, $1.version)))
			  {
			    t_buf	fontversion;

			    joberror (state, "can't get font %s %s", $1.name, strversion (fontversion, $1.version));
			  }
			}
		;

font		:	FONT SYMBOL fontversion
			{
			  $$.name = strunique ($2); free ($2);
			  $$.version = $3;
			}
		;

fontversion	:	empty
			{
			  $$ = VERSION_unspecified;
			}
		|	str
			{
			  if (scanversion (&$$, $1))
			    joberror (state, "illegal font version \"%s\".", $1);
			  free ($1);
			}
		;

opt_color	:	empty		{
#ifdef COLORED
					  clearcolor (&$$);
#endif
					}
		|	color
		;

color		:	COLORSPEC	{
#ifdef COLORED
					  $$ = $1;
#endif
					}
		|	COLOR COLORSPEC	{
#ifdef COLORED
					  $$ = $2;
#endif
					}
		|	COLOR name
			{
#ifdef COLORED
			  if (scancolor ($2, &$$))
			  {
			    jobwarning (state, "illegal color name \"%s\"", $2);
			    clearcolor (&$$);
			  }
#else
			  jobcolorwarning (state);
#endif
			  free ($2);
			}
		;

name		:	SYMBOL
		|	str
		;

chars_and_strings
		:	empty
		|	chars_and_strings MISS
			{
			  t_xucs2 miss = { 1, { UCS2_replacement_character } };
			  appendxucs2glyph (ytoken, state, &miss);
			}
		|	chars_and_strings XUCS2
			{
			  appendxucs2glyph (ytoken, state, $2);
			}
		|	chars_and_strings SYMBOL
			{
			  appendpsglyph (ytoken, state, strunique ($2));
			  free ($2);
			}
		|	chars_and_strings str
			{
			  if ($2)
			  {
			    appendstring (ytoken, state, $2);
			    free ($2);
			  }
			}
		;

str		:	'"' strtoken '"'	{ $$ = $2 ? strdeesc ($2) : NULL; }
		|	'(' strtoken ')'	{ $$ = $2 ? strdeesc ($2) : NULL; }
		;

strtoken	:	empty
			{
			  $$ = NULL;
			}
		|	strtoken STR
			{
			  if ($1)
			  {
			    int	len1, len2;

			    len1 = strlen ($1);
			    len2 = strlen ($2);
			    $$ = getmem (len1+len2+1);
			    memcpy ($$, $1, len1); free ($1);
			    memcpy ($$+len1, $2, len2+1); free ($2);
			  }
			  else
			    $$ = $2;
			}
		;

tab_list	:	empty
			{
			  $$ = licreate (sizeof (t_tab), NULL, NULL);
			}
		|	tab_list opt_hor_adjust width
			{
			  t_tab	tab = { $2, $3 };

			  if (lilength ($1) && ((t_tab *)lilast ($1))->width > $3)
			    joberror (state, "please specify tabs in ascending order!");
			  liappend ($$ = $1, &tab);
			}
		;

linestyle	:	empty		{ $$ = LS_solid; }
		|	SOLID		{ $$ = LS_solid; }
		|	DASHED		{ $$ = LS_dashed; }
		|	DOTTED		{ $$ = LS_dotted; }
		|	DASHDOTTED	{ $$ = LS_dash_dotted; }
		|	DASHDOTDOTTED	{ $$ = LS_dash_dot_dotted; }
		;

opt_hor_adjust	:	empty		{ $$ = ADJleft; }
		|	hor_adjust
		;

hor_adjust	:	LEFT		{ $$ = ADJleft; }
		|	CENTER		{ $$ = ADJcenter; }
		|	RIGHT		{ $$ = ADJright; }
		;

sep_width	:	empty		{ $$ = .025*state->fontsize[state->inout.mode]; }
		|	width
		;

dimension	:	empty
			{
			  $$.width = 0.;
			  $$.height = 0.;
			}
		|	dimension width
			{
			  $$.width = $2;
			  $$.height = $1.height;
			}
		|	dimension height
			{
			  $$.width = $1.width;
			  $$.height = $2;
			}
		;

width		:	WIDTH size	{ $$ = $2; }
		;

height		:	HEIGHT size	{ $$ = $2; }
		;

size		:	NUMBER	{ $$ = $1; }
		|	PTSIZE	{ $$ = $1; }
		;

%%

#if YYDEBUG
  static void yyprintinput (FILE *file, int type, YYSTYPE value)
  {
    t_buf	buf;

    switch (type)
    {
      case XUCS2:	fprintf (file, " %s", strucs2 (buf, value.xucs2->codes[0]));	break;
      case STR:		fprintf (file, " \"%s\"", value.str);	break;
      case SYMBOL:	fprintf (file, " /%s", value.str);	break;
      case SIZE:	fprintf (file, " %g", value.size);	break;
      case NUMBER:	fprintf (file, " %d", value.number);	break;
    }
  }
#endif

static void commanderror (cstring msg)
{
  joberror (state, "command parser: %s\n", msg);
}

/* ============================================================================================================================== */

static void vjobsay (const t_parsejob *state, cstring fmt, va_list va)
{
  const t_parsejob	*s;

  for (s=state; s; s=s->parent)
    say (IMP_error, "file %s, line %d%s ", jobname (s->job), s->iline, s->parent ? ", from" : ":");
  vsay (IMP_error, fmt, va);
  say (IMP_error, "\n");
}

static void jobwarning (const t_parsejob *state, cstring fmt, ...)
{
  va_list		va;

  va_start (va, fmt);
  vjobsay (state, fmt, va);
  va_end (va);
}

#ifndef COLORED
static void jobcolorwarning (const t_parsejob *state)
{
  jobwarning (state, "request for color is ignored. (program hasn't been compiled with color support)");
}
#endif

static volatile void joberror (const t_parsejob *state, cstring fmt, ...)
{
  va_list		va;

  va_start (va, fmt);
  vjobsay (state, fmt, va);
  va_end (va);
  exit (2);
}

static void pushjob (t_job *job, bool link)
{
  static int	stackdepthwarninglimit = 10;
  t_parsejob	*parent = state;

  assert (job);
  if (!statestack)
    statestack = licreate (sizeof (t_parsejob), NULL, freeparsejob);
  assert (statestack);
  if (lilength (statestack) > stackdepthwarninglimit)
  {
    say (IMP_warning, "warning: input parser stack deeper than %d\n", stackdepthwarninglimit);
    stackdepthwarninglimit += 10;
  }
  state = lipush (statestack, NULL);
  assert (state);

  memset (state, 0, sizeof *state);
  state->job = job;
  say (IMP_entertain2, "processing %s...\n", jobname (state->job));
if (link) assert (parent);
  state->parent = link ? parent : NULL;

  switch (state->job->srctype)
  {
    case JOB_STRING:
	state->fp = fsopen (state->job->source.str, strlen (state->job->source.str), "r");
	break;
    case JOB_STDIN:
	state->fp = stdin;
	break;
    case JOB_FILE:
	if (!(state->fp = fopen (state->job->source.fn, "r")))
	  error ("can't open \"%s\"", state->job->source.fn);
	break;
    default:
	abort();
  }
  assert (state->fp);

  state->iline = 1;
  state->icolumn = 0;

  if (state->parent)
  {
    input_handdown (&state->input, &state->parent->input, &state->job->input);
    state->inout = state->parent->inout;
    state->output = state->parent->output;
    memcpy (state->pfont, state->parent->pfont, sizeof state->pfont);
    memcpy (state->fontsize, state->parent->fontsize, sizeof state->fontsize);
assert (state->pfont[state->inout.mode]);
  }
  else
  {
    t_iomode	iomode;

    state->input = state->job->input;
    state->inout = state->job->inout;
    state->output = state->job->output;
    for (iomode=0; iomode<N_IOM; iomode++)
      if (state->output.font[iomode].family)
      {
	t_cnames *fontnames;

	if (!(fontnames = getfontsfromfamily (state->output.font[iomode].family, &state->output.font[iomode].rendition)))
	  joberror (state, "can't get font from family \"%s\".\n", state->output.font[iomode].family);
	if (!(state->pfont[iomode] = getpfont (fontnames, VERSION_unspecified)))
	  if (fontnames->count > 1)
	  {
	    char	*buf;
	    int		fni, len = 3;

	    for (fni=0; fni<fontnames->count; fni++)
	      len += strlen (fontnames->names[fni]);
	    buf = alloca (len);
	    *buf = '\0';
	    for (fni=0; fni<fontnames->count; fni++)
	    {
	      strcat (buf, fontnames->names[fni]);
	      strcat (buf, "/");
	    }
	    joberror (state, "can't get any font of %s", buf);
	  }
	  else
	    joberror (state, "can't get font %s", *fontnames->names);
	free (fontnames);
	state->fontsize[iomode] = state->output.font[iomode].size;
      }
  }
assert (state->inout.mode >= IOM_lit);
assert (state->inout.mode <= IOM_trlt);
assert (state->pfont[state->inout.mode]);

  pushlexjob (jobname (state->job), state->fp, &state->input, &state->inout, &state->iline, &state->icolumn);
}

static void popjob()
{
  if (statestack && lilength (statestack))
  {
    if (state->parent)
    {
      input_handup (&state->parent->input, &state->input);
      state->parent->inout = state->inout;
      state->parent->output = state->output;
      memcpy (state->parent->pfont, state->pfont, sizeof state->pfont);
      memcpy (state->parent->fontsize, state->fontsize, sizeof state->fontsize);
    }
    poplexjob();
assert (liposition (statestack) == 0);
memset (state, 0, sizeof *state);
    lidelete (statestack, 1);
    state = lifirst (statestack);
  }
  else
    ierror ("input parser: empty state stack");
}

/*\[sep]*/
static void setfontm (t_parsejob *state, t_pfont *pfont, t_iomode iomode)
{
  state->pfont[iomode] = pfont;
}

static void pushfontm (t_parsejob *state, t_pfont *pfont, t_iomode iomode)
{
  if (!pfont)
    return;

  if (!state->fontstack[iomode])
    state->fontstack[iomode] = licreate (sizeof (t_pfont *), NULL, NULL);
  assert (state->fontstack[iomode]);

  if (lilength (state->fontstack[iomode]) > 10)
    say (IMP_warning, "warning: fontstack deeper than 10\n");

  assert (lipush (state->fontstack[iomode], &state->pfont[iomode]));
  state->pfont[iomode] = pfont;
}

static void pushfont (t_parsejob *state, t_pfont *pfont)
{
  pushfontm (state, pfont, state->inout.mode);
}

static void popfontm (t_parsejob *state, t_iomode iomode)
{
  if (state->fontstack[iomode] && lilength (state->fontstack[iomode]))
  {
    state->pfont[iomode] = *(t_pfont **)lifirst (state->fontstack[iomode]);
    lidelete (state->fontstack[iomode], 1);
  }
  else
    say (IMP_warning, "warning: fontstack empty, keeping last font\n");
}

static void popfont (t_parsejob *state)
{
  popfontm (state, state->inout.mode);
}

/*\[sep]*/
static void setsizem (t_parsejob *state, t_point size, t_iomode iomode)
{
  state->fontsize[iomode] = size;
}

static void pushsizem (t_parsejob *state, t_point size, t_iomode iomode)
{
  if (size < 0)
  {
    jobwarning (state, "font size %g less than 0! (using 0)", size);
    size = 0;
  }

  if (!state->sizestack[iomode])
    state->sizestack[iomode] = licreate (sizeof (t_point), NULL, NULL);

  if (lilength (state->sizestack[iomode]) > 10)
    say (IMP_warning, "warning: sizestack deeper than 10\n");

  assert (lipush (state->sizestack[iomode], &state->fontsize[iomode]));
  state->fontsize[iomode] = size;
}

static void pushsize (t_parsejob *state, t_point size)
{
  pushsizem (state, size, state->inout.mode);
}

static void popsizem (t_parsejob *state, t_iomode iomode)
{
  if (state->sizestack[iomode] && lilength (state->sizestack[iomode]))
  {
    state->fontsize[iomode] = *(t_point *)liaddr (state->sizestack[iomode]);
    lidelete (state->sizestack[iomode], 1);
  }
  else
    say (IMP_warning, "warning: sizestack empty, keeping last size\n");
}

static void popsize (t_parsejob *state)
{
  popsizem (state, state->inout.mode);
}

/*\[sep]*/
#ifdef COLORED
static void setcolorm (t_parsejob *state, t_color color, t_iomode iomode)
{
  state->output.color[iomode] = color;
}

static void pushcolorm (t_parsejob *state, t_color color, t_iomode iomode)
{
  if (!state->colorstack[iomode])
    state->colorstack[iomode] = licreate (sizeof (t_color), NULL, NULL);

  if (lilength (state->colorstack[iomode]) > 10)
    say (IMP_warning, "warning: colorstack deeper than 10\n");

  assert (lipush (state->colorstack[iomode], &state->output.color[iomode]));
  state->output.color[iomode] = color;
}

#if 0 /* unused */
static void pushcolor (t_parsejob *state, t_color color)
{
  pushcolorm (state, color, state->inout.mode);
}
#endif

static void popcolorm (t_parsejob *state, t_iomode iomode)
{
  if (state->colorstack[iomode] && lilength (state->colorstack[iomode]))
  {
    state->output.color[iomode] = *(t_color *)liaddr (state->colorstack[iomode]);
    lidelete (state->colorstack[iomode], 1);
  }
  else
    say (IMP_warning, "warning: colorstack empty, keeping last color\n");
}

#if 0 /* unused */
static void popcolor (t_parsejob *state)
{
  popcolorm (state, state->inout.mode);
}
#endif
#endif /* COLORED */

/*\[sep]*/
static void controls (t_list *token, t_parsejob *state, t_ucs2 ucs2)
{
  /* for a maximum of eight names: */
  t_cnames	*names = alloca (sizeof (t_cnames) + 7*sizeof (cstring));

  switch (ucs2)
  {
    case UCS2_non_breaking_space:
	names->names[0] = "nbspace";	/* try it,	*/
	names->names[1] = "space";	/*	else default to normal */
	names->count = 2;
	break;
    case UCS2_non_breaking_hyphen:
	names->names[0] = "nbhyphen";	/* try it,	*/
	names->names[1] = "hyphen";	/*	else default to normal */
	names->count = 2;
	break;
    default:
	ierror ("controls()");
  }
  liappend (token, GLYPHTOKEN (state->pfont[state->inout.mode], .6*state->fontsize[state->inout.mode], &state->output.color[state->inout.mode], names));
}

static void fraction (t_list *token, t_parsejob *state, t_ucs2 ucs2)
{
  t_cnames	numerator, *fraction, denominator;
  t_token	numeratortoken, fractiontoken, denominatortoken;
  t_width	width;
  t_point	ascender, descender;
  t_glyphinfo	glyphinfo;

  switch (ucs2)
  {
    case UCS2_fraction_one_half:
    case UCS2_fraction_one_third:
    case UCS2_fraction_one_quarter:
    case UCS2_fraction_one_fifth:
    case UCS2_fraction_one_sixth:
    case UCS2_fraction_one_eighth:	numerator.names[0] = "one";	break;
    case UCS2_fraction_two_thirds:
    case UCS2_fraction_two_fifths:	numerator.names[0] = "two";	break;
    case UCS2_fraction_three_quarters:
    case UCS2_fraction_three_fifths:
    case UCS2_fraction_three_eighths:	numerator.names[0] = "three";	break;
    case UCS2_fraction_four_fifths:	numerator.names[0] = "four";	break;
    case UCS2_fraction_five_sixths:
    case UCS2_fraction_five_eighths:	numerator.names[0] = "five";	break;
    case UCS2_fraction_seven_eighths:	numerator.names[0] = "seven";	break;
    default:	ierror ("fraction, 1st switch");
  }
  numerator.count = 1;

  fraction = alloca (sizeof (t_names)+sizeof (cstring));
  fraction->names[0] = "fraction";
  fraction->names[1] = "slash";
  fraction->count = 2;

  switch (ucs2)
  {
    case UCS2_fraction_one_half:	denominator.names[0] = "two";	break;
    case UCS2_fraction_one_third:
    case UCS2_fraction_two_thirds:	denominator.names[0] = "three";	break;
    case UCS2_fraction_one_quarter:
    case UCS2_fraction_three_quarters:	denominator.names[0] = "four";	break;
    case UCS2_fraction_one_fifth:
    case UCS2_fraction_two_fifths:
    case UCS2_fraction_three_fifths:
    case UCS2_fraction_four_fifths:	denominator.names[0] = "five";	break;
    case UCS2_fraction_one_sixth:
    case UCS2_fraction_five_sixths:	denominator.names[0] = "six";	break;
    case UCS2_fraction_one_eighth:
    case UCS2_fraction_three_eighths:
    case UCS2_fraction_five_eighths:
    case UCS2_fraction_seven_eighths:	denominator.names[0] = "eight";	break;
    default:	ierror ("fraction, 2nd switch");
  }
  denominator.count = 1;

  numeratortoken = *GLYPHTOKEN (state->pfont[state->inout.mode], .6*state->fontsize[state->inout.mode], &state->output.color[state->inout.mode], &numerator);
  fractiontoken = *GLYPHTOKEN (state->pfont[state->inout.mode], state->fontsize[state->inout.mode], &state->output.color[state->inout.mode], fraction);
  denominatortoken = *GLYPHTOKEN (state->pfont[state->inout.mode], .6*state->fontsize[state->inout.mode], &state->output.color[state->inout.mode], &denominator);

  getglyphinfo (&numeratortoken.u.glyph, 1.0, &glyphinfo);
  width = glyphinfo.width;
  ascender = glyphinfo.ascender + .3*state->fontsize[state->inout.mode];
  descender = glyphinfo.descender + .3*state->fontsize[state->inout.mode];

  getglyphinfo (&fractiontoken.u.glyph, 1.0, &glyphinfo);
  width.x += glyphinfo.width.x;
  ascender = MAX (ascender, glyphinfo.ascender + width.y);
  descender = MIN (descender, glyphinfo.descender + width.y);
  width.y += glyphinfo.width.y;

  getglyphinfo (&denominatortoken.u.glyph, 1.0, &glyphinfo);
  width.x += glyphinfo.width.x;
  ascender = MAX (ascender, glyphinfo.ascender + width.y);
  descender = MIN (descender, glyphinfo.descender + width.y);

  /* damit der aus drei teilen bestehende glyph nicht auf mehrere zeilen verteilt wird, wird der platz reserviert. */
  liappend (token, commandtoken (COMMAND_need_space3, width.x, ascender, descender));
  liappend (token, commandtoken (COMMAND_move_vertical, .3*state->fontsize[state->inout.mode]));
  liappend (token, &numeratortoken);
  liappend (token, commandtoken (COMMAND_move_vertical, -.3*state->fontsize[state->inout.mode]));
  liappend (token, &fractiontoken);
  liappend (token, &denominatortoken);
}

typedef struct
{
  t_ucs2	ucs2;
  void		(*handler)(t_list *token, t_parsejob *state, t_ucs2 ucs2);
} t_special;

static t_special special_chars[] =
{
  UCS2_non_breaking_hyphen, controls,
  UCS2_non_breaking_space, controls,
  UCS2_fraction_one_half, fraction,
  UCS2_fraction_one_third, fraction,
  UCS2_fraction_two_thirds, fraction,
  UCS2_fraction_one_quarter, fraction,
  UCS2_fraction_three_quarters, fraction,
  UCS2_fraction_one_fifth, fraction,
  UCS2_fraction_two_fifths, fraction,
  UCS2_fraction_three_fifths, fraction,
  UCS2_fraction_four_fifths, fraction,
  UCS2_fraction_one_sixth, fraction,
  UCS2_fraction_five_sixths, fraction,
  UCS2_fraction_one_eighth, fraction,
  UCS2_fraction_three_eighths, fraction,
  UCS2_fraction_five_eighths, fraction,
  UCS2_fraction_seven_eighths, fraction,
};

static int specialpcmp (t_special **l, t_special **r)
{
  return ucs2cmp ((*l)->ucs2, (*r)->ucs2);
}

static t_avl *specials = NULL;

static void init_specials()
{
  int		len;
  t_special	*table;

  assert (specials = avlcreate (sizeof (t_special *), specialpcmp, NULL));
  for (table=special_chars, len=ARRAYDIM (special_chars); len--; table++)
    assert (avlinsert (specials, &table));
}

static void appendpsglyph (t_list *token, t_parsejob *state, string name)
{
  t_cnames	glyphname;

  glyphname.names[0] = name;
  glyphname.count = 1;
  assert (liappend (token, GLYPHTOKEN (state->pfont[state->inout.mode], state->fontsize[state->inout.mode], &state->output.color[state->inout.mode], &glyphname)));
  state->icolumn++;	/* hack, should happen in lex-input.l, but it is much simpler here... */
}

static void appendxucs2glyph (t_list *token, t_parsejob *state, const t_xucs2 *xucs2)
{
  int	n;

  if (!specials)
    init_specials();

  if (!xucs2)
    joberror (state, "unknown glyph encountered!");

  for (n=0; n<xucs2->count; n++)
  {
    t_ucs2	ucs2 = xucs2->codes[n];
    t_special	search, *sp = &search, **found;

    search.ucs2 = ucs2;
    if (found = avlsearch (specials, &sp))
    {
      (*(*found)->handler)(token, state, ucs2);
      return;
    }
    else
    {
      const t_cnames	*charnames;

      if (charnames = ucs22ps (ucs2))
      {
	assert (liappend (token, GLYPHTOKEN (state->pfont[state->inout.mode], state->fontsize[state->inout.mode], &state->output.color[state->inout.mode], charnames)));
	break;
      }
      else
      {
	t_buf	buf;

	jobwarning (state, "can't translate UCS2 %s into a PostScript name.", strucs2 (buf, ucs2));
      }
    }
  }
  if (n == xucs2->count)
  {
    t_cnames	miss = { 1, { "miss" } };

    jobwarning (state, "translation of UCS2 list into a PostScript name failed. inserting replacement character.");
    assert (liappend (token, GLYPHTOKEN (state->pfont[state->inout.mode], state->fontsize[state->inout.mode], &state->output.color[state->inout.mode], &miss)));
  }

  state->icolumn++;	/* hack, should happen in lex-input.l, but it is much simpler here... */
}

static void appendchar (t_list *token, t_parsejob *state, char c)
{
  t_xucs2	xucs2;

assert (state->inout.mode >= IOM_lit);
assert (state->inout.mode <= IOM_trlt);
assert (state->pfont[state->inout.mode]);
  xucs2.count = 1;
  switch (state->input.encoding)
  {
    /* 7 bit encodings: */
    default:
    case ENC_ISO646:		/* ISO 646 IRV encoding		*/
	xucs2.codes[0] = ISO646toUCS2 (c);
	break;
    case ENC_ASCII:		/* ASCII encoding		*/
	xucs2.codes[0] = ASCIItoUCS2 (c);
	break;
    case ENC_DIN66003:		/* DIN 66003 encoding		*/
	xucs2.codes[0] = DIN66003toUCS2 (c);
	break;

    /* 8 bit encodings: */
    case ENC_ISOlatin1:		/* ISO Latin 1 encoding		*/
	xucs2.codes[0] = ISOlatin1toUCS2 (c);
	break;
    case ENC_CP437:		/* codepage 437 encoding	*/
	xucs2.codes[0] = CP437toUCS2 (c);
	break;
  }
  if (xucs2.codes[0] == UCS2_VOID)
  {
    jobwarning (state, "unknown glyph encountered! (character code: 0%o, %d, 0x%02x)", (uchar)c, (uchar)c, (uchar)c);
    /* all our encodings are based on ISO646, so they can be translated into the same meta-notation: */
    pushsize (state, state->fontsize[state->inout.mode] * .7);
    appendstring (token, state, verbcontrol (state->input.verbmode, c));
    popsize (state);
  }
  else
    appendxucs2glyph (token, state, &xucs2);
}

static void appendstring (t_list *token, t_parsejob *state, cstring s)
{
  while (*s)
    appendchar (token, state, *s++);
}

static void appendlstring (t_list *token, t_parsejob *state, cstring s, size_t len)
{
  while (len--)
    appendchar (token, state, *s++);
}

static void appendstringssmall (t_list *token, t_parsejob *state, ...)
{
  va_list	va;
  cstring	s;

  va_start (va, state);
  pushsize (state, state->fontsize[state->inout.mode] * 0.7);
  while (s=va_arg (va, cstring))
    appendstring (token, state, s);
  popsize (state);
  va_end (va);
}

/*\[sep]-------------------------------------------------------------------------------------------------------------------------*/ 
static void guess_enc_tr (cstring str, t_encoding *encoding, t_translation *translation)
{
  cstring	p;

  if (*translation != TRLT_guess)
    return;

  *encoding = ENC_ISOlatin1;

  for (p=str; *p; p++)
    if ('\200' <= *p && *p < '\240')	/* the only encoding that uses this range? */
    {
      *encoding = ENC_CP437;
      goto try_tex; /* don't expect troff...? */
    }

  for (p=str; *p; p++)
    if (*p == ISO646_GS)
    {
      *encoding = ENC_ASCII;
      *translation = TRLT_sendmail;
      return;
    }

  for (p=str; *p; p++)
    if (*p == '\b')
    {
      *translation = TRLT_MAN;
      break;
    }

  if (p = strchr (str, '\\'))
  {
    do
      switch (*++p)
      {
	case '\0':
	    goto try_tex;
	case '\n':
	    *translation = TRLT_none;
	    return;
	case '\\':
	    continue;
	case 'f':
	{
	  char	fn[3] = { 0, 0, 0 };

	  switch (*++p)
	  {
	    case '(':
	      if (!(fn[0] = *++p) || !(fn[1] = *++p) || !troff2psfontbyname (fn))
		goto try_tex;
	      break;
	    default:
	      if (!(fn[0] = *p) || !troff2psfontbyname (fn))
		goto try_tex;
	  }
	  *translation = TRLT_TROFF;
	  break;
	}
	case 's':
	case '(':
#if 1 /* simple */
	case '*':
#else
	    *translation = TRLT_TROFF;
	    break;
	case '*':
	    if (*++p && strchr ("AOUaous*", *p))
#endif
	      *translation = TRLT_TROFF;
	      break;
	default:
	    goto try_tex;
      }
    while (p = strchr (p+1, '\\'));
    if (*translation == TRLT_TROFF)
      return;
  }

try_tex:
  if (p = strchr (str, '"'))
  {
    do
    {
      if (p[1] && strchr ("AOUaous", p[1]))
	continue;
      else
	goto try_other;
    }
    while (p = strchr (p+1, '"'));
    *translation = TRLT_TEX;
    return;
  }

try_other:
  *translation = TRLT_none;
}

static t_list *parse_line (string str, t_parsejob *state)
{
  const t_job	*job = state->job;
  t_job		subjob;
  t_list	*token;

  if (!str)
    return NULL;

  subjob = *job;
  guess_enc_tr (str, &subjob.input.encoding, &subjob.input.translation);
  subjob.srctype = JOB_STRING;
  subjob.source.str = str;

  subjob.input.type = IT_TEXTline;

  token = parse_input (&subjob, true);
  return token;
}

typedef struct
{
  cstring	name;
  t_ucs2	ucs2;
} t_glyphmap;

static int glyphpcmp (t_glyphmap **l, t_glyphmap **r)
{
  return strcmp ((*l)->name, (*r)->name);
}

/*
 * expression from lex-input.l:
 *	"+-"|[^-+ \t/"(;\r\n\f\]]+
 */
static t_glyphmap glyphtable[] =
{
  "+-", UCS2_plus_or_minus_sign,
  "...", UCS2_horizontal_ellipsis,
  "%%", UCS2_per_mille_sign,

  "<", UCS2_left_pointing_single_guillemet,
  ">", UCS2_right_pointing_single_guillemet,
  "<<", UCS2_left_pointing_guillemet,
  ">>", UCS2_right_pointing_guillemet,

  ",", UCS2_low_single_comma_quotation_mark,		/* german opening quote */
  "`", UCS2_single_reversed_comma_quotation_mark,	/* english opening quote, german closing quote */
  "'", UCS2_single_comma_quotation_mark,		/* english closing quote */
  ",,", UCS2_low_double_comma_quotation_mark,		/* german opening quote */
  "``", UCS2_double_reversed_comma_quotation_mark, 	/* english opening quote, german closing quote */
  "''", UCS2_double_comma_quotation_mark,		/* english closing quote */
};

static t_avl	*glyphs = NULL;

static void init_glyphs()
{
  int		len;
  t_glyphmap	*table;

  glyphs = avlcreate (sizeof (t_glyphmap *), glyphpcmp, NULL); assert (glyphs);
  for (table=glyphtable, len=ARRAYDIM (glyphtable); len--; table++)
{void *p =
    avlinsert (glyphs, &table);
assert (p);}
}

static const t_xucs2 *name2glyph (string name)
{
  t_glyphmap	search, *sp = &search, **found;
  const t_xucs2	*xucs2;

  if (!glyphs)
    init_glyphs();

  search.name = name;
  if (found = avlsearch (glyphs, &sp))
  {
    static t_xucs2	xucs2;

    xucs2.count = 1;
    *xucs2.codes = (*found)->ucs2;
    return &xucs2;
  }
  elif (xucs2 = ps2ucs2 (name))
    return xucs2;
  else
    return NULL;
}

typedef struct
{
  cstring	name;
  int		token;
} t_commandmap;

static int commandpcmp (t_commandmap **l, t_commandmap **r)
{
  return strcmp ((*l)->name, (*r)->name);
}

static t_commandmap commandtable[] =
{
  "lit", LITERAL, "literal", LITERAL,
  "trlt", TRANSLATED, "translated", TRANSLATED,
  "set", SET,
  "push", PUSH,
  "pop", POP,
  "font", FONT,
  "size", SIZE,
  "color", COLOR, "colour", COLOR,
  "sep", SEPARATOR, "separator", SEPARATOR,
  "header", HEADER, "banner", HEADER,
  "tabs", TABULATOR, "tab", TABULATOR,
  "left", LEFT, "lt", LEFT,
  "right", RIGHT, "rt", RIGHT,
  "up", UP,
  "down", DOWN, "dn", DOWN,
  "centre", CENTER, "center", CENTER,
  "need", NEED,
  "horizontal", HORIZONTAL, "horiz", HORIZONTAL, "hor", HORIZONTAL,
  "vertical", VERTICAL, "vert", VERTICAL, "ver", VERTICAL,
  "width", WIDTH, "w", WIDTH,
  "height", HEIGHT, "h", HEIGHT,
  "columns", COLUMNS, "cols", COLUMNS,
  "solid", SOLID, "sol", SOLID,
  "dashed", DASHED, "dash", DASHED,
  "dotted", DOTTED, "dot", DOTTED,
  "dashdotted", DASHDOTTED, "dashdot", DASHDOTTED,
  "dashdotdotted", DASHDOTDOTTED, "dashdotdot", DASHDOTDOTTED,
};

static t_avl	*commands = NULL;

static void init_commands()
{
  int		len;
  t_commandmap	*table;

  commands = avlcreate (sizeof (t_commandmap *), commandpcmp, NULL); assert (commands);
  for (table=commandtable, len=ARRAYDIM (commandtable); len--; table++)
{void *p =
    avlinsert (commands, &table);
assert (p);}
}

static int name2command (string name)
{
  t_commandmap	search, *sp = &search, **found;

  if (!commands)
    init_commands();

  search.name = name;
  if (found = avlsearch (commands, &sp))
    return (*found)->token;
  return fault;
}

static int lex_command (YYSTYPE *lvalp);
static int lex_command (YYSTYPE *lvalp)
{
  t_inputtoken	lval;
  int		c;

#if YYDEBUG
  c = lex_input (&lval);
  fprintf (stderr, "lex_command: c=%d\n", c);
  switch (c)
#else
  switch (c = lex_input (&lval))
#endif
  {
    case YY_NULL:
	joberror (state, "EOF in command");
    case END_COMMAND:
	return 0; /* signals end of input/command */
    case COMMAND_PLUSMINUS:
	lvalp->boolean = lval.boolean;
	return PLUSMINUS;
    case COMMAND_FACTOR:
	lvalp->factor = lval.factor;
	return FACTOR;
    case COMMAND_NUMBER:
	lvalp->number = lval.number;
	return NUMBER;
    case COMMAND_SIZE:
	lvalp->size = lval.size;
	return PTSIZE;
    case COMMAND_COLOR:
#ifdef COLORED
	lvalp->color = lval.color;
#else
	jobcolorwarning (state);
#endif
	return COLORSPEC;
    case COMMAND_SYMBOL:
	lvalp->str = lval.ltext.buf;
	return SYMBOL;
    case COMMAND_NAME:
      {
	int		command;

	if ((command = name2command (lval.ltext.buf)) != fault)
	{
	  free (lval.ltext.buf);
	  return command;
	}
	if (lvalp->xucs2 = name2glyph (lval.ltext.buf))
	{
	  free (lval.ltext.buf);
	  return XUCS2;
	}
	jobwarning (state, "illegal command/glyph name \"%s\". (replaced)", lval.ltext.buf);
	free (lval.ltext.buf);
	return MISS;
      }
    case COMMAND_MNEMO:
	lvalp->xucs2 = mnemo2glyph (lval.ltext.buf);
	free (lval.ltext.buf);
	if (lvalp->xucs2)
	  return XUCS2;
	jobwarning (state, "illegal mnemonic `%s'. (replaced)", lval.text);
	return MISS;
    case STRING:
	lvalp->str = strsave (lval.text);
	return STR;
    default:
	if (c < 256)
	  return c;
	else
	{
	  joberror (state, "syntax error in command");
	  abort();/* standard C doesn't recognize volatile for non-returning functions. but gcc knows that abort() and exit() don't return... */
	}
  }
}

/* newjob == NULL: continue old job; else start new job	*/
/* 	if starting new job: link -> share attributes with parent parse job */
t_list *parse_input (t_job *newjob, bool link)
{
  t_list	*token;
  t_inputtoken	lval;
  int		c;

  token = licreate (sizeof (t_token), NULL, NULL);

  if (newjob)
  {
    pushjob (newjob, link);
    switch (state->job->srctype)
    {
      case JOB_STRING:
	  break;
      case JOB_STDIN:
	  liappend (token, commandtoken (COMMAND_START, NULL));
	  break;
      case JOB_FILE:
	{
	  struct stat	statbuf;

	  fstat (fileno (state->fp), &statbuf);
	  liappend (token, commandtoken (COMMAND_START, state->job->source.fn, statbuf.st_mtime));
	}
	break;
      default:
	abort();
    }
  }

  while ((c = lex_input (&lval)) != YY_NULL)
#ifdef YYDEBUG
  {
    if (c < 128)
      fprintf (stderr, " c=%d/'%c' ", c, c);
    elif (c == NEWLINE)
      fprintf (stderr, " NL ");
    elif (c == STRING)
      fprintf (stderr, " STRING(%s) ", lval.text);
    else
      fprintf (stderr, " c=%d/%#x ", c, c);
#endif
    switch (c)
    {
      case ENTER_TRANSLATION:
	{
assert (state->inout.mode == IOM_lit);
	  if (state->output.font[IOM_trlt].family)
	    state->inout.mode = IOM_trlt;
	  appendstring (token, state, lval.text);
	  break;
	}
      case STRING:
	  appendstring (token, state, lval.text);
	  break;
      case lSTRING:
	  appendlstring (token, state, lval.ltext.buf, lval.ltext.len);
	  free (lval.ltext.buf);
	  break;
      case LEAVE_TRANSLATION:
      case LEAVE_TRANSLATION__NEWLINE:
	  appendstring (token, state, lval.text);
	  if (state->output.font[IOM_lit].family)
	    state->inout.mode = IOM_lit;
	  if (c == LEAVE_TRANSLATION__NEWLINE)
	    goto newline;
	  break;

      case TAB:
	{
	  int		tab;
	  t_point	width = state->output.tabs.relative
	 			 ? state->input.tabcolumns * state->output.tabs.width.factor * state->fontsize[state->inout.mode]
	 			 : state->output.tabs.width.size;

	  tab = state->icolumn / state->input.tabcolumns + 1;
	  state->icolumn = state->input.tabcolumns * tab;
	  liappend (token, commandtoken (COMMAND_TAB, tab*width));
	  break;
	}
      case INDENT:
	{
	  t_point	width = state->output.tabs.relative
	 			 ? state->input.tabcolumns * state->output.tabs.width.factor * state->fontsize[state->inout.mode]
	 			 : state->output.tabs.width.size;
	  state->icolumn += lval.number;
	  say (IMP_entertain2, "indent of %d to %g pt\n", state->icolumn, state->icolumn * width/state->input.tabcolumns);
	  liappend (token, commandtoken (COMMAND_TAB, state->icolumn * width/state->input.tabcolumns));
	  break;
	}
      case escapedNEWLINE:
	  appendstring (token, state, lval.text);
	  /* fall thru */
      case NEWLINE:
    newline:
	  liappend (token, commandtoken (COMMAND_NEWLINE));
	  if (state->job->srctype != JOB_STRING)
	    return token;
	  break;
      case NEWPAGE:
	  liappend (token, commandtoken (COMMAND_NEWPAGE));
	  if (state->job->srctype != JOB_STRING)
	    return token;
	  break;
      case ITALIC_CHAR:
	{
	  t_cnames	*italicfontnames;
	  t_pfont	*italicfont;

	  if (!(italicfont = getpfont (italicfontnames = getfontsfromfamily2 (state->output.font[state->inout.mode].family, state->output.font[state->inout.mode].rendition.weight, true), VERSION_unspecified)))
	    joberror (state, "can't get an italic font from family \"%s\"", state->output.font[state->inout.mode].family);
	  free (italicfontnames);
	  pushfont (state, italicfont);
	  appendchar (token, state, *lval.text);
	  popfont (state);
	  break;
	}
      case BOLD_CHAR:
	{
	  t_cnames	*boldfontnames;
	  t_pfont	*boldfont;

	  if (!(boldfont = getpfont (boldfontnames = getfontsfromfamily2 (state->output.font[state->inout.mode].family, FONTWEIGHT_bold, state->output.font[state->inout.mode].rendition.italic), VERSION_unspecified)))
	    joberror (state, "can't get a bold font from family \"%s\"", state->output.font[state->inout.mode].family);
	  free (boldfontnames);
	  pushfont (state, boldfont);
	  appendchar (token, state, *lval.text);
	  popfont (state);
	  break;
	}

      case TROFF_HYPHENATION:
	  /* just ignore it */
	  break;

      case TROFF_SPACE:
	{
	  t_spacetype	space;

	  switch (*lval.text)
	  {
	    case ' ':	space = SPACE_SPACE;		break;
	    case '0':	space = SPACE_DIGIT;		break;
	    case '|':	space = SPACE_NARROW;		break;
	    case '^':	space = SPACE_HALFNARROW;	break;
	    case '&':	space = SPACE_ZERO;		break;
	    default:	abort();
	  }
	  liappend (token, commandtoken (COMMAND_SPACE, space, state->pfont[state->inout.mode], state->fontsize[state->inout.mode]));
	  break;
	}
      case TROFF_CHAR2:		/* troff two letter Non-ASCII character	*/
	{
	  const t_xucs2 *xucs2;

	  if (xucs2 = troffchar2ucs2 (lval.text))
	    appendxucs2glyph (token, state, xucs2);
	  else
	  {
	    jobwarning (state, "unrecognized troff character `\\(%s'", lval.text);
	    appendstringssmall (token, state, "\\(", lval.text, NULL);
	  }
	  break;
	}
      case TROFF_FONTn:		/* troff font ref'd by number	*/
	  if (lval.number)
	  {
	    t_cnames	fontname;
	    t_pfont	*pfont;

	    if (!(fontname.names[0] = troff2psfontbynumber (lval.number)))
	      joberror (state, "unrecognized troff font #%d", lval.number);
	    fontname.count = 1;
	    if (!(pfont = getpfont (&fontname, VERSION_unspecified)))
	      joberror (state, "font `%s' is not available", fontname.names[0]);
	    pushfont (state, pfont);
	  }
	  else
	    popfont (state);
	  break;
      case TROFF_FONT1:		/* troff font with one letter name	*/
      case TROFF_FONT2:		/* troff font with two letter name	*/
	  if (strcmp (lval.text, "P"))
	  {
	    t_cnames	fontname;
	    t_pfont	*pfont;

	    if (!(fontname.names[0] = troff2psfontbyname (lval.text)))
	      joberror (state, "unrecognized troff font `%s'", lval.text);
	    fontname.count = 1;
	    if (!(pfont = getpfont (&fontname, VERSION_unspecified)))
	      joberror (state, "font `%s' is not available", fontname.names[0]);
	    pushfont (state, pfont);
	  }
	  else
	    popfont (state);
	  break;
      case TROFF_FONT:
	  {
	    t_cnames	fontname;
	    t_pfont	*pfont;

	    fontname.names[0] = lval.ltext.buf;
	    fontname.count = 1;
	    if (!(pfont = getpfont (&fontname, VERSION_unspecified)))
	      joberror (state, "font `%s' is not available", lval.text);
	    pushfont (state, pfont);
	    free (lval.ltext.buf);
	  }
	  break;
      case TROFF_SIZE:		/* troff font size change	*/
	{
	  t_point size;
	  size = atof (lval.ltext.buf + (strchr ("+-*/", *lval.ltext.buf) ? 1 : 0));
	  switch (*lval.ltext.buf)
	  {
	    case '+':
		pushsize (state, state->fontsize[state->inout.mode] + size);
		break;
	    case '-':
		pushsize (state, state->fontsize[state->inout.mode] - size);
		break;
	    case '*':
		pushsize (state, state->fontsize[state->inout.mode] * size);
		break;
	    case '/':
		pushsize (state, state->fontsize[state->inout.mode] / size);
		break;
	    default:
		if (size)
		  pushsize (state, size);
		else
		  popsize (state);
	  }
	  free (lval.ltext.buf);
	  break;
	}
      case TROFF_half_line_up:
	  liappend (token, commandtoken (COMMAND_move_vertical, .5 * state->fontsize[state->inout.mode]));
	  break;
      case TROFF_half_line_down:
	  liappend (token, commandtoken (COMMAND_move_vertical, -.5 * state->fontsize[state->inout.mode]));
	  break;
      case TROFF_STRING1:	/* troff string with one letter name	*/
	{
	  const t_xucs2 *xucs2;

	  if (xucs2 = troff1str2ucs2 (lval.text))
	    appendxucs2glyph (token, state, xucs2);
	  else
	  {
	    jobwarning (state, "unrecognized troff string `\\*%s'", lval.text);
	    appendstringssmall (token, state, "\\*", lval.text, NULL);
	  }
	  break;
	}
      case TROFF_STRING2:	/* troff string with two letter name	*/
	{
	  const t_xucs2 *xucs2;

	  if (xucs2 = troff2str2ucs2 (lval.text))
	    appendxucs2glyph (token, state, xucs2);
	  else
	  {
	    jobwarning (state, "unrecognized troff string `\\*(%s'", lval.text);
	    appendstringssmall (token, state, "\\*(", lval.text, NULL);
	  }
	  break;
	}
      case TROFF_DIACRITIC:	/* troff string indicating diacritical mark	*/
	{
	  const t_xucs2 *xucs2;

	  if (xucs2 = troffdiacritic2ucs2 (lval.text))
	    appendxucs2glyph (token, state, xucs2);
	  else
	  {
	    jobwarning (state, "unrecognized troff string diacritic `%c' on character `%c'", lval.text[0], lval.text[1]);
	    appendstringssmall (token, state, "<", lval.text, ">", NULL);
	  }
	  break;
	}
#if 0
      case SENDMAIL_DIACRITIC:	/* sendmail(1)'s mapping of diacritical marks	*/
	{
	  const t_xucs2 *xucs2;

	  if (xucs2 = sendmaildiacritic2ucs2 (lval.text))
	    appendxucs2glyph (token, state, xucs2);
	  else
	  {
	    jobwarning (state, "unrecognized sendmail diacritic `%s'", lval.text);
	    appendstringssmall (token, state, "<", lval.text, ">", NULL);
	  }
	  break;
	}
#endif
      
      case BEGIN_COMMAND:
	  ytoken = token;
	  ybreak = false;
#if YYDEBUG
	  yydebug = true;
#endif
	  parse_command();
	  if (ybreak)
	    return token;
	  break;

      case TEX_DIACRITIC:
	{
	  const t_xucs2 *xucs2;

	  if (xucs2 = texdiacritic2ucs2 (lval.text))
	    appendxucs2glyph (token, state, xucs2);
	  else
	    joberror (state, "unrecognized tex string diacritic `%s'", lval.text);
	  break;
	}

      case GLYPH:
	{
	  t_xucs2	xucs2 = { 1, { lval.ucs2 } };

	  appendxucs2glyph (token, state, &xucs2);
	  break;
	}

      case MailHeaderBeginField:
	{
	  t_point		sizefactor;
	  t_fontrendition	fontrendition;
	  t_cnames		*fontnames;
	  t_pfont		*pfont;

#if DEBUGMAIL
fprintf (stderr, "MailHeaderBeginField: from %d\n", state->mailstate);
#endif
	  mailrendition (lval.ltext.buf, &sizefactor, &fontrendition);
	  switch (state->mailstate)
	  {
	    case MAILvoid:
	    case MAILbody:
		liappend (token, commandtoken (state->input.type == IT_Mail ? COMMAND_MailMessage : COMMAND_NewsArticle, ++state->messageno));
		state->mailstate = MAILheader;
		break;
	    case MAILheader:
		popsize (state);
		popfont (state);
		break;
	    default:
		abort();
	  }
	  if (!(fontnames = getfontsfromfamily (state->output.font[state->inout.mode].family, &fontrendition))
	    || !(pfont = getpfont (fontnames, VERSION_unspecified)))
	  {
	    t_buf	fw;

#if !DEBUGMAIL
	    say (IMP_entertain2,
#else
	    prompt (
#endif
			"can't get font in mailrendition (weight=%s, italic=%d) from family `%s', keeping old one.\n", strfontweight (fw, fontrendition.weight), fontrendition.italic, state->output.font[state->inout.mode].family);
	    pfont = state->pfont[state->inout.mode];
	  }
	  if (fontnames)
	    free (fontnames);
	  pushfont (state, pfont);
	  pushsize (state, MAX (state->fontsize[state->inout.mode] * sizefactor, 3.));
	}
	/* FALL THRU */
      case MailHeaderContField:
#if DEBUGMAIL
if (c == MailHeaderContField) fprintf (stderr, "MailHeaderContField\n");
fprintf (stderr, ">> MailHeader\n");
#endif
	  token = liconcat (token, parse_line (lval.ltext.buf, state));
	  free (lval.ltext.buf);
#if DEBUGMAIL
liamap (token, printtoken, stderr);
fprintf (stderr, "<< MailHeader\n");
#endif
	  return token;
      case MailEndHeader:
	{
	  t_line	line = {
#ifdef COLORED
				{ CM_void },
#endif
				  .025*state->fontsize[state->inout.mode], LS_solid };

assert (state->mailstate == MAILheader);
	  popsize (state);
	  popfont (state);
	  liappend (token, commandtoken (COMMAND_Separator, line, state->fontsize[state->inout.mode]));
	  state->mailstate = MAILbody;
#if DEBUGMAIL
fprintf (stderr, "MailEndHeader\n");
#endif
	  return token;
	}
      case MailBody:
assert (state->mailstate == MAILbody);
#if DEBUGMAIL
fprintf (stderr, ">> MailBody\n");
#endif
	  token = liconcat (token, parse_line (lval.ltext.buf, state));
	  free (lval.ltext.buf);
#if DEBUGMAIL
liamap (token, printtoken, stderr);
fprintf (stderr, "<< MailBody\n");
#endif
	  return token;

      default:
	  if (c & 0xffffff00)
	    ierror ("parse_input(): unexpected default with c=%#x/%d", c, c);
	  appendchar (token, state, c);
    }
#ifdef YYDEBUG
  }
  fprintf (stderr, " c=YY_NULL");
#endif

  switch (state->job->srctype)
  {
    case JOB_STRING:
	fsclose (state->fp);
	break;

    case JOB_FILE:
	fclose (state->fp);
	/* FALL THRU */
    case JOB_STDIN:
	break;

    default:
	abort();
  }
  popjob();

  liappend (token, commandtoken (COMMAND_END));
  return token;
}

t_list *parse_string (string str, t_encoding encoding, t_translation translation, string fontfamily, t_point fontsize, bool link)
{
  t_job		job;

  guess_enc_tr (str, &encoding, &translation);
  job = defaultjob (encoding, translation, fontfamily, fontsize);
  job.srctype = JOB_STRING;
  job.source.str = str;
  return parse_input (&job, link);
}
