/*
 * program: psfilt
 * file: parse-afm.y
 *
 * Copyright  1992 1993 Robert Joop
 *
 * Parser for .afm files.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: parse-afm.y,v $
 * Revision 1.3  1994/07/09  16:44:00  rj
 * a lot of const's removed and added
 *
 * Revision 1.2  1994/01/09  23:45:57  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.1.1.1  1993/12/31  20:56:38  rj
 * erster cvs import.
 *
 */

%{

static const char RCSId[] = "$Id: parse-afm.y,v 1.3 1994/07/09 16:44:00 rj Exp $";

#if YYDEBUG_AFM
#  define YYDEBUG	1
#endif

#define yyparse		parse_afm
#define yylex		lex_afm
#define yyerror		afmerror

#include <sys/types.h>
#include <sys/stat.h>

#include "psfilt.h"
#include "afm.h"
#include "lex-afm.h"
#include "fonts.h"	/* for fontdirs, fonts, fontfamilies	*/
#include "ps.h"		/* for psinitafm(), pscompleteafm() */
#include "cache.h"
#include "str.h"
#include "mem.h"
#include "verbose.h"
#include "error.h"

#if YYDEBUG
#  define YYERROR_VERBOSE
#  define YYPRINT(file, type, value)   yyprintafm (file, type, value)
  static void yyprintafm();
#endif

static t_afmfile	*globalafm;	/* variable parse_afm() writes into	*/
static t_charmetric	globalcharmetric;

/* an afm file usually contains the global information first, followed by the individual character metrics.
 * reading the global information only is about 10 times faster than reading all information
 */
static bool		getglobalfontinfoonly = false;

static void parseafmsay (t_verblevel importance, cstring fmt, ...)
{
  va_list	va;

  say (importance, "afm parser: file %s, line %d: ", globalafm->file.fullname, yylineno_afm);
  va_start (va, fmt);
  vsay (importance, fmt, va);
  va_end (va);
  say (importance, "\n");
}

static void afmerror (cstring msg)
{
  parseafmsay (IMP_warning, "%s", msg);
}

static string sharename (string name)
{
  t_charmetric	cm, *cmp;

  cm.name = name;
  if (!(cmp = avlsearch (globalafm->metrics, &cm)))
    error ("illegal charname \"%s\" in line %d", name, yylineno_afm);
  assert (name != cmp->name);
  free (name);
  return cmp->name;
}

static int shareligature (t_ligature *lig)
{
  lig->first = sharename (lig->first);
  lig->second = sharename (lig->second);
  return ok;
}

static int shareligatures (t_charmetric *cm)
{
  return limap (cm->ligatures, shareligature);
}

%}

%pure_parser

%union
{
  bool		boolean;
  string	str;
  int		number;
  double	real;
  t_ligature	ligature;
}

%token STARTFONTMETRICS
%token ENDFONTMETRICS
%token FONTNAME
%token FULLNAME
%token FAMILYNAME
%token WEIGHT
%token ITALICANGLE
%token ISFIXEDPITCH
%token FONTBBOX
%token UNDERLINEPOSITION
%token UNDERLINETHICKNESS
%token VERSION_
%token NOTICE
%token ENCODINGSCHEME
%token CAPHEIGHT
%token XHEIGHT
%token ASCENDER
%token DESCENDER
%token STARTCHARMETRICS
%token ENDCHARMETRICS
%token C
%token WX
%token W
%token N
%token B
%token L
%token STARTKERNDATA
%token ENDKERNDATA
%token STARTTRACKKERN
%token ENDTRACKKERN
%token TRACKKERN
%token STARTKERNPAIRS
%token ENDKERNPAIRS
%token KP
%token KPX
%token STARTCOMPOSITES
%token ENDCOMPOSITES
%token CC
%token PCC
%token TRUE
%token FALSE

%token <number>		INTEGER
%token <real>		REAL
%token <str>		NAME
%token <str>		STRING

%type <boolean>		Bool
%type <real>		Units
%type <real>		Number
%type <real>		llx
%type <real>		lly
%type <real>		urx
%type <real>		ury
%type <ligature>	Ligature

%start AFMFile

%%

AFMFile		:	Start Structure End
		;

Start		:	STARTFONTMETRICS STRING
			{
			  parseafmsay (IMP_entertain2, "reading font metrics (format %s).", $2);
			  free ($2);
			}
		;

Structure	:	/* empty */
		|	Structure GlobalFontInformation
		|	Structure
			{
			  if (getglobalfontinfoonly)
			    YYACCEPT;
			}
			IndividualCharacterMetrics
			{
			  avlmap (globalafm->metrics, shareligatures);
			}
		|	Structure KerningData
		|	Structure CompositeCharacterData
		;

End		:	ENDFONTMETRICS
		;

GlobalFontInformation
		:	FontName
		|	FullName
		|	FamilyName
		|	Weight
		|	ItalicAngle
		|	IsFixedPitch
		|	FontBBox
		|	UnderlinePosition
		|	UnderlineThickness
		|	Version
		|	Notice
		|	EncodingScheme
		|	CapHeight
		|	XHeight
		|	Ascender
		|	Descender
		;

FontName	:	FONTNAME STRING
			{
			  globalafm->fontname = $2;
			}
		;

FullName	:	FULLNAME STRING
			{
			  globalafm->fullname = $2;
			}
		;

FamilyName	:	FAMILYNAME STRING
			{
			  globalafm->familyname = $2;
			}
		;

Weight		:	WEIGHT STRING
			{
			  if ((globalafm->weight = fontweightbyname (strcmpr ($2))) == FONTWEIGHT_unknown)
			    parseafmsay (IMP_warning, "can't translate font weight \"%s\"", $2);
			  free ($2);
			}
		;

ItalicAngle	:	ITALICANGLE Number
			{
			  globalafm->italicangle = $2;
			}
		;

IsFixedPitch	:	ISFIXEDPITCH Bool
			{
			  globalafm->isfixedpitch = $2;
			}
		;

FontBBox	:	FONTBBOX llx lly urx ury
			{
			  globalafm->bbox.llx = $2;
			  globalafm->bbox.lly = $3;
			  globalafm->bbox.urx = $4;
			  globalafm->bbox.ury = $5;
			}
		;

UnderlinePosition
		:	UNDERLINEPOSITION Units
			{
			  globalafm->underlineposition = $2;
			}
		;

UnderlineThickness
		:	UNDERLINETHICKNESS Units
			{
			  globalafm->underlinethickness = $2;
			}
		;


Version		:	VERSION_ STRING
			{
			  if (scanversion (&globalafm->version, $2))
			  {
			    parseafmsay (IMP_warning, "illegal font version \"%s\"", $2);
			    globalafm->version = VERSION_unknown;
			  }
			  free ($2);
			}
		;

Notice		:	NOTICE STRING
			{
			  globalafm->notice = $2;
			}
		;

EncodingScheme	:	ENCODINGSCHEME STRING
			{
			  globalafm->encodingscheme = $2;
			}
		;

CapHeight	:	CAPHEIGHT Units
			{
			  globalafm->capheight = $2;
			}
		;

XHeight		:	XHEIGHT Units
			{
			  globalafm->Xheight = $2;
			}
		;

Ascender	:	ASCENDER Units
			{
			  globalafm->ascender = $2;
			}
		;

Descender	:	DESCENDER Units
			{
			 globalafm->descender = $2;
			}
		;

IndividualCharacterMetrics
		:	STARTCHARMETRICS OptInteger
			IndCharMetr
			ENDCHARMETRICS
		;

OptInteger	:	/* empty */
		|	INTEGER		{}
		;

IndCharMetr	:	/* empty */
		|	IndCharMetr MetricLine
			{
#ifndef NDEBUG
			  void *p;
#endif
			  /*printcharmetric (&globalcharmetric);*/
			  assert (globalafm->metrics);
			  p = avlinsert (globalafm->metrics, &globalcharmetric); assert (p);
			  if (globalcharmetric.code != -1)
			    globalafm->encoding[globalcharmetric.code] = globalcharmetric.name;
			}
		;

MetricLine	:	{
			  clearcharmetric (&globalcharmetric);
			}
			Code MetricValues
		;

MetricValues	:	/* empty */
		|	MetricValues MetricValue
		;

MetricValue	:	XWidth 		/* one time */
		|	WidthVector 	/* one time */
		|	Name 		/* one time */
		|	BoundingBox 	/* one time */
		|	Ligature	/* n times */
			{
			  if (!globalcharmetric.ligatures)
			    globalcharmetric.ligatures = licreate (sizeof (t_ligature), NULL, NULL);
			  liappend (globalcharmetric.ligatures, &$1);
			}
		;

Code		:	C INTEGER ';'
			{
			  globalcharmetric.code = $2;
			}
		;

XWidth		:	WX Units ';'
			{
			  globalcharmetric.width.x = $2;
			  globalcharmetric.width.y = 0.;
			}
		;

WidthVector	:	W Units Units ';'
			{
			  globalcharmetric.width.x = $2;
			  globalcharmetric.width.y = $3;
			}
		;

Name		:	N NAME ';'
			{
			  globalcharmetric.name = $2;
			}
		;

BoundingBox	:	B llx lly urx ury ';'
			{
			  globalcharmetric.bbox.llx = $2;
			  globalcharmetric.bbox.lly = $3;
			  globalcharmetric.bbox.urx = $4;
			  globalcharmetric.bbox.ury = $5;
			}
		;

llx		:	Units
			{
			  $$ = $1;
			}
		;

lly		:	Units
			{
			  $$ = $1;
			}
		;

urx		:	Units
			{
			  $$ = $1;
			}
		;

ury		:	Units
			{
			  $$ = $1;
			}
		;

Ligature	:	L NAME NAME ';'
			{
			  $$.first = $2;
			  $$.second = $3;
			}
		;

KerningData	:	STARTKERNDATA
			KernData
			ENDKERNDATA
		;

KernData	:	/* empty */
		|	KernData TrackKerning
		|	KernData PairWiseKerning
		;

TrackKerning	:	STARTTRACKKERN INTEGER
			TrackKerns
			ENDTRACKKERN
			{
			  parseafmsay (IMP_entertain2, "the file contains %d track kerning entries.", $2);
			  parseafmsay (IMP_whocares, "track kerning is ignored.");
			}
		;

TrackKerns	:	/* empty */
		|	TrackKerns TrackKern
		;

TrackKern	:	TRACKKERN Degree MinPtSize MinKern MaxPtSize MaxKern
		;

Degree		:	INTEGER		{}
		;

MinPtSize	:	Units		{}
		;

MinKern		:	Units		{}
		;

MaxPtSize	:	Units		{}
		;

MaxKern		:	Units		{}
		;

PairWiseKerning	:	STARTKERNPAIRS INTEGER
			PairWiseKern
			ENDKERNPAIRS
			{
			  parseafmsay (IMP_entertain2, "the file contains %d kerning pairs.", $2);
			}
		;

PairWiseKern	:	/* empty */
		|	PairWiseKern PairWiseKernLine
		;

PairWiseKernLine
		:	KPX NAME NAME Units
			{
			  t_kernpair	kern;

			  kern.left = sharename ($2);
			  kern.right = sharename ($3);
			  kern.width.x = $4;
			  kern.width.y = 0.;
			  avlinsert (globalafm->kernpairs, &kern);
			}
		|	KP NAME NAME Units Units
			{
			  t_kernpair	kern;

			  kern.left = sharename ($2);
			  kern.right = sharename ($3);
			  kern.width.x = $4;
			  kern.width.y = $5;
			  avlinsert (globalafm->kernpairs, &kern);
			}
		;

CompositeCharacterData
		:	STARTCOMPOSITES INTEGER
			CompositeCharData
			ENDCOMPOSITES
			{
			  parseafmsay (IMP_entertain2, "the file contains data for %d composite characters.", $2);
			  parseafmsay (IMP_whocares, "composite characters are ignored.");
			}
		;

CompositeCharData
		:	/* empty */
		|	CompositeCharData CompCharDataLine
		;

CompCharDataLine:	CompositeCharacter CompositeCharacterValues
		;

CompositeCharacter
		:	CC NAME INTEGER ';'
			{
			  free ($2);
			}
		;

CompositeCharacterValues
		:	/* empty */
		|	CompositeCharacterValues CompositeCharacterValue
		;

CompositeCharacterValue
		:	PCC NAME Number Number ';'
			{
			  free ($2);
			}
		;

Units		:	Number		{ $$ = $1 / 1000.0; }
		;

Number		:	INTEGER		{ $$ = $1; }
		|	REAL		{ $$ = $1; }
		;

Bool		:	TRUE		{ $$ = true; }
		|	FALSE		{ $$ = false; }
		;

%%

#if YYDEBUG
  static void yyprintafm (FILE *file, int type, YYSTYPE value)
  {
    switch (type)
    {
      case INTEGER:	fprintf (file, " %d", value.number);	break;
      case REAL:	fprintf (file, " %g", value.real);	break;
      case NAME:	fprintf (file, " /%s", value.str);	break;
      case STRING:	fprintf (file, " \"%s\"", value.str);	break;
    }
  }
#endif

/*\[sep]-------------------------------------------------------------------------------------------------------------------------*/ 
int scanafm (FILE *fp, t_afmfile *afm, bool globalonly)
{
  int rc;

  init_afmlexer (fp);
  getglobalfontinfoonly = globalonly;
  globalafm = afm;
  afm->file.stat = (rc = parse_afm()) ? FS_void : globalonly ? FS_valid : FS_incore;
  return rc;
}

static int checkmatch (string fn, cstring name, string *old, string *new)
{
  int rc;

  if (!*old)
    rc = ok;
  elif (rc = str0cmp (*old, *new))
    say (IMP_error, "rescanafm: file %s: mismatch in %s: %s vs. %s\n", fn, name, *old, *new);
  else
  {
    string tmp = *old;
    *old = *new;
    *new = tmp;
  }
  return rc;
}

int rescanafm (FILE *fp, t_afmfile *afm, bool globalonly)
{
  t_afmfile	new;
  int		rc;

  clearafmfile (&new);
  new.file = afm->file; clearfileinfo (&afm->file);
  rc = scanafm (fp, &new, false);

  /* pointers may already be in use: exchange them */
#define CHECK(n, m)	\
  if (checkmatch (new.file.fullname, n, &afm->m, &new.m))	\
    return fault;
  CHECK ("FontName", fontname)
  CHECK ("FullName", fullname)
  CHECK ("FamilyName", familyname)
  CHECK ("Notice", notice)
  CHECK ("EncodingScheme", encodingscheme)
#undef CHECK

  freeafmfile (afm);
  *afm = new;
  return rc;
}

t_list		*afmdirs = NULL;

int completeafm (t_afmfile *afm)
{
  if (afm->file.stat == FS_incore)
    return ok;
  if (afm->file.stat == FS_void)
    return fault;
  elif (afm->file.fullname)
  {
    FILE	*fp;
    int		rc;

    if (!(fp = pfreopen (&afm->file, O_rdonly)))
      error ("completeafm(): can't open \"%s\" a second time!\n", afm->file.fullname);
    rc = rescanafm (fp, afm, false);
    pfclose (&afm->file);
    return rc;
  }
  else /* internal font */
  {
    pscompleteafm (afm);
    return ok;
  }
}
