/*
 * program: psfilt
 * file: output.c
 *
 * Copyright  1992 1993 Robert Joop
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: output.c,v $
 * Revision 1.6  1994/07/09  16:43:58  rj
 * a lot of const's removed and added
 *
 * Revision 1.5  1994/07/09  15:34:19  rj
 * stuff for configuration via GNU autoconf added
 *
 * Revision 1.4  1994/03/18  12:07:31  rj
 * support for duplex mode added
 * lots of missing `#ifdef COLORED' inserted
 *
 * Revision 1.3  1994/01/09  23:45:54  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.2  1994/01/04  09:27:23  rj
 * va_start() was called with wrong argument.
 *
 * Revision 1.1.1.1  1993/12/31  20:56:42  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: output.c,v 1.6 1994/07/09 16:43:58 rj Exp $";

#ifndef DEBUG
#  undef LONGDEBUG
#endif

/* AIX requires this to be the first thing in the file.  */
#ifdef __GNUC__
#define alloca __builtin_alloca
#else /* not __GNUC__ */
#if HAVE_ALLOCA_H
#include <alloca.h>
#else /* not HAVE_ALLOCA_H */
#ifdef _AIX
 #pragma alloca
#else /* not _AIX */
char *alloca ();
#endif /* not _AIX */
#endif /* not HAVE_ALLOCA_H */
#endif /* not __GNUC__ */

#include <math.h>	/* floor(), ceil() */
#include <time.h>	/* ctime() */

#include "psfilt.h"
#include "job.h"
#include "mem.h"
#include "verbose.h"
#include "error.h"

int	kernflags = KERN_pairwise;

typedef struct
{
  cstring	username;
  long		printtime;
  t_inputtype	inputtype;
  const t_goutput *output;

  t_rectangle	imageablearea;

  t_rectangle	imagearea;
} t_ospec;

static t_ospec	ospec;

typedef enum
{
  PE_void,
  PE_separator,
  PE_bannerbox,
  PE_chars,	/* list of t_chargroup */
  PE_pageelems,	/* recursion! */
} t_pageelemtype;

typedef struct
{
  t_pageelemtype	type;
  t_vector		offset;
  t_dimension		area;
  t_point		ascender, descender;	/* max and min of all chars, resp., or separator separation */
  union	/* one member for each type: */
  {
    t_separator		separator;
    /* no member for bannerbox */
    t_list		*chargroups;	/* list of t_chargroup (see below) */
    t_list		*pageelems;
  } u;
} t_pageelem;

static void freepageelem (t_pageelem *pageelem)
{
  switch (pageelem->type)
  {
    case PE_chars:
	if (pageelem->u.chargroups)
	  pageelem->u.chargroups = lidestroy (pageelem->u.chargroups);
	break;
    case PE_pageelems:
	if (pageelem->u.pageelems)
	  pageelem->u.pageelems = lidestroy (pageelem->u.pageelems);
	break;
  }
}

/* group of characters sharing same font and fontsize	*/
typedef struct
{
  t_vector	offset;
  t_pfont	*pfont;
  t_point	fontsize;
#ifdef COLORED
  t_color	color;
#endif
  int		kern;		/* see below */
  int		nchars;
  cstring	charnames[0];	/* has to be the last member! */
} t_chargroup;

/* values for t_chargroup.kern: */
#define NOKERN	0x00
#define XKERN	0x01
#define YKERN	0x02

#define KERN(chgp, type)	((type *)((chgp)->charnames+(chgp)->nchars))

static int fillchargroup (cstring *charname, va_list va)
{
  t_chargroup	*chargroup = va_arg (va, t_chargroup *);
  int		*index = va_arg (va, int *);

  if (kernflags & KERN_pairwise)
    if (*index & 1)
      switch (chargroup->kern)
      {
	case NOKERN:
	    break;
	case XKERN:
	    KERN (chargroup, t_point)[*index>>1] = ((t_width *)charname)->x;
	    break;
	default:
	    KERN (chargroup, t_width)[*index>>1] = *(t_width *)charname;
	    break;
      }
    else
      chargroup->charnames[*index>>1] = *charname;
  else
    chargroup->charnames[*index] = *charname;
  ++*index;
  return ok; /* for liamap() */
}

#ifdef COLORED
#define APPENDCHARGROUP( pageelem, offset, pfont, fontsize, color, charnames, kernflag)	\
	appendchargroup (pageelem, offset, pfont, fontsize, color, charnames, kernflag)
#else
#define APPENDCHARGROUP( pageelem, offset, pfont, fontsize, color, charnames, kernflag)	\
	appendchargroup (pageelem, offset, pfont, fontsize,        charnames, kernflag)
#endif

static void APPENDCHARGROUP (t_pageelem *pageelem, const t_vector *offset, t_pfont *pfont, t_point fontsize, const t_color *color, t_list *charnames, int kernflag)
{
  size_t	size;
  int		nchars,
		index;
  t_chargroup	*chargroup;

  assert (pageelem->type == PE_chars);
  if (!pageelem->u.chargroups)
    pageelem->u.chargroups = licreate (/* variable: */ 0, NULL, NULL);	/* list of t_chargroup	*/
  assert (pageelem->u.chargroups);

  size = sizeof (t_chargroup);
  if (kernflags & KERN_pairwise)
  {
    int	n = lilength (charnames),
	nkernings = n>>1;

    nchars = nkernings + 1;
    switch (kernflag)
    {
      case NOKERN:
	  break;
      case XKERN:
	  size += nkernings * sizeof (t_point);
	  break;
      default:
	  size += nkernings * sizeof (t_width);
	  break;
    }
  }
  else
    nchars = lilength (charnames);
  size += nchars * sizeof (cstring);
  chargroup = liappend (pageelem->u.chargroups, NULL, size);
  assert (chargroup);
  chargroup->offset = *offset;
  chargroup->pfont = pfont;
  chargroup->fontsize = fontsize;
#ifdef COLORED
  chargroup->color = *color;
#endif
  chargroup->kern = kernflag;
  chargroup->nchars = nchars;
  index = 0;
  livamap (charnames, fillchargroup, chargroup, &index);
}

typedef enum
{
  HDR_void,
  HDR_SHEET,
  HDR_FILE,
  HDR_PAGE,
  HDR_Banner,
  HDR_UserBanner,
  HDR_UserBanner3,
} t_headertype;

typedef enum
{
  PT_SHEET,	/* a PT_HBOX with additional header */
  PT_HBOX,
  PT_VBOX,
  PT_SIMPLE,
} t_pagetype;

typedef struct t_page
{
  double		rotation;
  t_vector		offset;
  t_dimension		area;		/* einschlielich header */

  t_list		*header;	/* list of t_pageelem; first element determines bounding box */

  t_pagetype		type;
  union
  {
    struct
    {
      int		ncolumn,
			filled;
      struct t_page	*columns;
    } hbox;
    t_list		*vbox;
    t_list		*simple;	/* list of t_pageelem */
  } u;
} t_page;

static t_point headerheight (const t_page *page)
{
  if (page->header && lilength (page->header))
  {
    t_pageelem *header = ((t_pageelem *)lifirst (page->header));
    if (header->type != PE_void)
      return header->area.height;
  }
  return 0.;
}

static void freepage (t_page *page)
{
  page->header = lidestroy (page->header);
  switch (page->type)
  {
    case PT_SHEET:
    case PT_HBOX:
	while (page->u.hbox.filled--)
	  freepage (page->u.hbox.columns+page->u.hbox.filled);
	break;
    case PT_VBOX:
	ligoto (page->u.vbox, 0);
	lidelete (page->u.vbox, lilength (page->u.vbox));
	break;
    case PT_SIMPLE:
	page->u.simple = lidestroy (page->u.simple);
  }
}

typedef enum
{
  I_OK,
  I_FLUSH,	/* ignore characters around some commands */
  I_MORE,
  I_NEXTPAGE,	/* start new page/block	*/
  I_too_low,	/* remaining space too small	*/
  I_too_narrow,	/* remaining space too small	*/
  I_NEWPAGE,	/* start new page (e.g. formfeed)	*/
  I_EOF,	/* end of input file reached	*/
  I_EOJOBS,	/* end of job list	*/
} t_iresult;

typedef struct
{
  t_list	*jobs;
  int		jobno;
  t_job		*job;
  int		ilineno;	/* current job's input line number	*/
  int		pageno;		/* current job's output page number	*/

  int		sheetno;
  bool		firstcolumn;

  t_list	*token;		/* token der eingabezeile	*/
  int		listoff;	/* die position in <token>, an der angefangen werden soll	*/
  t_point	widthoff,	/* die summe der breiten der bisher aus der eingabezeile gemachten ausgabezeilen	*/
		heightoff;	/* summe der bisherigen vertikalen bewegungen in der eingabezeile	*/
} t_iostate;

static int gettoken (t_iostate *iostate)
{
  if (!iostate->token)
  {
    iostate->listoff = 0;
    iostate->widthoff = iostate->heightoff = 0.;
    if (iostate->job)
    {
      if (!(iostate->token = parse_input (NULL, false)))
	ierror ("gettoken(), if");
    }
    else
    {
      if (!(iostate->job = ligoto (iostate->jobs, ++iostate->jobno)))
	return fault;
      say (IMP_debug2, "gettoken(): switching to job #%d of %d\n", iostate->jobno, lilength (iostate->jobs));
      if (!(iostate->token = parse_input (iostate->job, false)))
	ierror ("gettoken(), else");
    }
  }
  return ok;
}

/* <area> ist die fr die ausgabezeile zur verfgung stehende flche */
/* <iostate> wird nur verndert, wenn die funktion nicht mit einem fehler zurckkehrt */
static t_iresult fillpageelem (t_iostate *iostate, const t_dimension *area, double scalefactor, t_pageelem *pageelem)
{
  int		i;
  t_token	*t;

  t_vector	offset = vector (0., iostate->heightoff);	/* offset vom zeilenanfang */
  t_pfont	*pfont = NULL;
  t_point	fontsize = 0.;
#ifdef COLORED
  t_color	color;
#endif
  t_list	*charnames = NULL;	/* collect charnames for a chargroup */
  int		kernflag = NOKERN;

#ifdef COLORED
  clearcolor (&color);
#endif

  assert (pageelem);
  pageelem->type = PE_void;
  pageelem->ascender = pageelem->descender = 0.0;
  pageelem->area.width = pageelem->area.height = 0.0;
  pageelem->offset = vector (0., 0.);	/* the pageelem will be positioned by the caller */

  if (!iostate->token)
    return I_OK;

  for (i=iostate->listoff; t=ligoto (iostate->token, i); i++)
  {
    switch (t->type)
    {
      case TOKEN_GLYPH:
	{
	  const t_glyph		*glyph = &t->u.glyph;
	  t_glyphinfo		glyphinfo;
	  const t_kernpair	*kp = NULL;		/* kernwidth relative to ptsz of 1 */
	  t_width		kpw = { 0., 0. };	/* kernwidth relative to current pointsize */

	  if (pageelem->type != PE_chars)
	  {
	    assert (pageelem->type == PE_void);
	    pageelem->type = PE_chars;
	    pageelem->u.chargroups = NULL;
	  }
	  getglyphinfo (glyph, scalefactor, &glyphinfo);
	  if (kernflags & KERN_pairwise && charnames && glyph->pfont == pfont && glyph->fontsize == fontsize && (kp = getkernpair (glyph->pfont->afm, *(string *)lilast (charnames), glyphinfo.charname)))
	    kpw = scalewidth (kp->width, scalefactor * glyph->fontsize);
	  if (kpw.y + glyphinfo.ascender + offset.y > pageelem->ascender)
	  {
	    pageelem->ascender = kpw.y + glyphinfo.ascender + offset.y;
	    pageelem->area.height = pageelem->ascender - pageelem->descender;
	  }
	  if (kpw.y + glyphinfo.descender + offset.y < pageelem->descender)
	  {
	    pageelem->descender = kpw.y + glyphinfo.descender + offset.y;
	    pageelem->area.height = pageelem->ascender - pageelem->descender;
	  }
	  if (pageelem->area.width + kpw.x + glyphinfo.width.x <= area->width)
	  {
	    if (charnames && (glyph->pfont != pfont || glyph->fontsize != fontsize
#ifdef COLORED
				|| colorpcmp (&glyph->color, &color)
#endif
				))	/* any font change? */
	    {
	      APPENDCHARGROUP (pageelem, &offset, pfont, scalefactor *fontsize, &color, charnames, kernflag);
	      charnames = lidestroy (charnames);
	      kernflag = NOKERN;
	    }
	    if (!charnames)
	    {
	      offset.x = pageelem->area.width;
	      pfont = glyph->pfont;
	      fontsize = glyph->fontsize;
#ifdef COLORED
	      color = glyph->color;
#endif
	      charnames = licreate (0, NULL, NULL);
	    }
	    elif (kernflags & KERN_pairwise)
	    {
	      if (kpw.x != 0.)
		kernflag |= XKERN;
	      elif (kpw.y != 0.)
		kernflag |= YKERN;
	      liappend (charnames, &kpw, sizeof (t_width));
	      pageelem->area.width += kpw.x;
	      offset.y += kpw.y;
	    }
	    liappend (charnames, &glyphinfo.charname, sizeof (cstring));
	    pageelem->area.width += glyphinfo.width.x;
	    offset.y += glyphinfo.width.y;
	  }
	  else	/* current char would make line too long	*/
	  {
	    say (IMP_debug, "at token #%d: line width=%g, glyph width=%g, limit=%g\n", i, pageelem->area.width, glyphinfo.width.x, area->width);
	    if (charnames)
	    {
	      APPENDCHARGROUP (pageelem, &offset, pfont, scalefactor * fontsize, &color, charnames, kernflag);
	      charnames = lidestroy (charnames);
	      kernflag = NOKERN;
	    }
	    if (i == iostate->listoff)
	      return I_too_narrow;
	    if (iostate->job)
	      say (IMP_entertain, "folding line %d of file %s (at %s)\n", iostate->ilineno, jobname (iostate->job), "glyph");
	    iostate->listoff = i; /* retry */
	    iostate->widthoff += area->width;
	    iostate->heightoff = offset.y;
	    return I_MORE;
	  }
	  if (pageelem->area.height > area->height)				/* area is not high enough	*/
	  {
	    if (charnames)
	      charnames = lidestroy (charnames);
	    return I_too_low;
	  }
	}
	break;

      case TOKEN_COMMAND:
	switch (t->u.command.code)	/* all commands terminate a chargroup, but not necessarily their page elem */
	{
	  case COMMAND_START:	/* commands that terminate a page elem of chars	*/
	  case COMMAND_MailMessage:
	  case COMMAND_NewsArticle:
	  case COMMAND_Separator:
	  case COMMAND_Banner:
	      switch (pageelem->type)
	      {
		case PE_void:
		    break;
		case PE_chars:
		    switch (t->u.command.code)
		    {
		      case COMMAND_Separator:
		      case COMMAND_Banner:	/* drop characters */
			  charnames = lidestroy (charnames);
			  pageelem->type = PE_void;
			  break;
		      default:
			  if (charnames)
			  {
			    APPENDCHARGROUP (pageelem, &offset, pfont, scalefactor * fontsize, &color, charnames, kernflag);
			    charnames = lidestroy (charnames);
			  }
			  iostate->listoff = i; /* retry */
			  iostate->widthoff += area->width;
			  iostate->heightoff = offset.y;
			  return I_MORE;
		    }
		    break;
		default:
		    ierror ("fillpageelem(), TOKENCOMMAND, pageelem->type");
	      }
	      break;
	  default:		/* terminate chargroup only */
	      if (pageelem->type == PE_chars && charnames)
	      {
		APPENDCHARGROUP (pageelem, &offset, pfont, scalefactor * fontsize, &color, charnames, kernflag);
		charnames = lidestroy (charnames);
		kernflag = NOKERN;
	      }
	}

	switch (t->u.command.code)
	{
	  case COMMAND_START:
	  case COMMAND_MailMessage:
	  case COMMAND_NewsArticle:
	      break;			/* have already been handled in fillpage() */

	  case COMMAND_SPACE:
	    {
	      t_point	spacing = scalefactor * t->u.command.u.spacewidth;

	      if (pageelem->area.width + spacing <= area->width)
		pageelem->area.width += spacing;
	      else
	      {
		if (i == iostate->listoff)
		  return I_too_narrow;
		if (iostate->job)
		  say (IMP_entertain, "folding line %d of file %s (at %s)\n", iostate->ilineno, jobname (iostate->job), "space");
		iostate->listoff = i; /* retry */
		iostate->widthoff += area->width;
		iostate->heightoff = offset.y;
		return I_MORE;
	      }
	      break;
	    }

	  case COMMAND_TAB:
	    {
	      t_point	tabstop = scalefactor * t->u.command.u.tabstop;

	      if (tabstop < pageelem->area.width + iostate->widthoff)
	      {
		say (IMP_entertain, "file %s, line %d: tab moves left from %g to %g.\n", jobname (iostate->job), iostate->ilineno, pageelem->area.width + iostate->widthoff, tabstop);
		if (tabstop - iostate->widthoff < 0.)
		{
		  say (IMP_warning, "file %s, line %d: tab moves left of beginning of output line. moving to left margin instead.\n", jobname (iostate->job), iostate->ilineno);
		  pageelem->area.width = 0.;
		  break;
		}
	      }
	      if (tabstop - iostate->widthoff > area->width)
	      {
		if (iostate->job)
		  say (IMP_entertain, "folding line %d of file %s (at %s)\n", iostate->ilineno, jobname (iostate->job), "tab");
		iostate->listoff = i; /* retry */
		iostate->widthoff += area->width;
		iostate->heightoff = offset.y;
		return I_MORE;
	      }
	      else
		pageelem->area.width = tabstop - iostate->widthoff;
	      break;
	    }

	  case COMMAND_need_space2:
	  case COMMAND_need_space3:
	      if (pageelem->area.width + scalefactor * t->u.command.u.space2.width > area->width)
	      {
		if (i == iostate->listoff)
		  return I_too_narrow;
		if (iostate->job)
		  say (IMP_entertain, "folding line %d of file %s (at %s)\n", iostate->ilineno, jobname (iostate->job), "need");
		iostate->listoff = i; /* retry */
		iostate->widthoff += area->width;
		iostate->heightoff = offset.y;
		return I_MORE;
	      }
	      if ((t->u.command.code == COMMAND_need_space2
		    ? MAX (pageelem->area.height, scalefactor * t->u.command.u.space2.height + offset.y)
		    : MAX (pageelem->ascender, scalefactor * t->u.command.u.space3.ascender + offset.y) - MIN (pageelem->descender, scalefactor * t->u.command.u.space3.descender + offset.y))
		  > area->height)
		return I_too_low;
	      break;

	  case COMMAND_move_vertical:
	      offset.y += scalefactor * t->u.command.u.motion;
	      break;

	  case COMMAND_Separator:
	      if (area->height < scalefactor * t->u.command.u.separator.separation)
		return I_too_low;
	      pageelem->type = PE_separator;
	      pageelem->area.width = area->width;
	      pageelem->area.height = scalefactor * t->u.command.u.separator.separation;
	      pageelem->u.separator = t->u.command.u.separator;
	      pageelem->ascender = pageelem->descender = scalefactor * .5 * t->u.command.u.separator.separation;
	      iostate->ilineno++;
	      iostate->listoff = i+1;
	      return I_FLUSH; /* drop the rest of the input characters */

	  case COMMAND_Banner:
	    {
	      /*forward:*/static t_iresult fillheader (t_iostate *iostate, const t_dimension *area, t_pageelem *header, t_headertype type, ...);

	      fillheader (iostate, area, pageelem, HDR_UserBanner, t->u.command.u.banner.token, t->u.command.u.banner.horadj);
	      iostate->listoff = i+1;
	      return I_FLUSH; /* drop the rest of the input characters */
	    }
	  case COMMAND_Banner3:
	    {
	      /*forward:*/static t_iresult fillheader (t_iostate *iostate, const t_dimension *area, t_pageelem *header, t_headertype type, ...);

	      fillheader (iostate, area, pageelem, HDR_UserBanner3, t->u.command.u.banner3);
	      iostate->listoff = i+1;
	      return I_FLUSH; /* drop the rest of the input characters */
	    }

	  case COMMAND_NEWLINE:
	      iostate->ilineno++;
	      return I_OK;
	  case COMMAND_NEWPAGE:
	      return I_NEWPAGE;
	  case COMMAND_END:
	      return I_EOF;

	  default:
	      ierror ("fillpageelem(): illegal command\n");
	}
	break;

      default:
	  ierror ("fillpageelem(): illegal token type %d\n", t->type);
    }
  }
  /* list always has to be terminated by COMMAND_NEWLINE, COMMAND_NEWPAGE or COMMAND_EOF	*/
  ierror ("fillpageelem(): unterminated line, i=%d, t=%#x\n", i, t);
  abort();/* standard C doesn't recognize volatile for non-returning functions. but gcc knows that abort() and exit() don't return... */
}

typedef struct
{
/* input: */
  t_list	*token;
  struct
  {
    t_adjustment	horizontal, vertical;
  } adjust;

/* output:	*/
  t_pageelem	line;
} t_filldata;

static t_iresult fillarea (const t_dimension *area, double scalefactor, t_filldata *fd)
{
  t_iostate	iostate;
  t_iresult	rc;

  iostate.token = fd->token;
  iostate.listoff = 0;
  iostate.widthoff = iostate.heightoff = 0.;
iostate.ilineno = -1;
iostate.job = NULL;

  switch (rc = fillpageelem (&iostate, area, scalefactor, &fd->line))
  {
    case I_MORE:
	freepageelem (&fd->line);
	rc = I_too_narrow;
	break;
    case I_too_low:
    case I_too_narrow:
	freepageelem (&fd->line);
	break;

    case I_EOF:
	rc = I_OK;
	/* FALL THRU */
    case I_OK:
#define MARGIN	(.5*area->height)	/* horizontal margin */
	switch (fd->adjust.horizontal)
	{
	  case ADJleft:
	      fd->line.offset.x = MARGIN;
	      break;
	  case ADJright:
	      fd->line.offset.x = area->width - fd->line.area.width - MARGIN;
	      break;
	  case ADJcenter:
	      fd->line.offset.x = (area->width - fd->line.area.width) / 2.;
	      break;
	}
#undef MARGIN
#define MARGIN	(.1*area->height)	/* vertical margin */
	switch (fd->adjust.vertical)
	{
	  case ADJtop:
	      fd->line.offset.y = area->height - fd->line.descender - MARGIN;
	      break;
	  case ADJcenter:
	      fd->line.offset.y = (area->height - fd->line.area.height) / 2. - fd->line.descender;
	      break;
	  case ADJbottom:
	      fd->line.offset.y = MARGIN - fd->line.ascender;
	      break;
	}
#undef MARGIN
	break;

    default:
	ierror ("fillarea(), default");
  }

  return rc;
}

/* a maximum of 3 fields is silently enforced... */
static void afillheaderarea (t_pageelem *header, int nfield, t_filldata	*fields)
{
  int		field;
  double	scalefactor;

  for (field=0; field<nfield; field++)
  {
    fields[field].adjust.vertical = ADJcenter;
    fields[field].line.type = PE_void;
    fields[field].line.area.width = 0;
    switch (fields[field].adjust.horizontal)
    {
      case ADJleft:
	  fields[field].line.offset.x = 0;
	  break;
      case ADJcenter:
	  fields[field].line.offset.x = header->area.width / 2.;
	  break;
      case ADJright:
	  fields[field].line.offset.x = header->area.width;
	  break;
    }
  }

  switch (nfield)
  {
    case 1:	/* nur eine mglichkeit: alles auf einer zeile */
	for (scalefactor=1.0; scalefactor * header->area.height > 1.; scalefactor *= 0.9)
	  if (fillarea (&header->area, scalefactor, fields) == I_OK)
	    break;
	break;
    case 2:	/* zwei mglichkeiten: alles auf eine zeile, oder jedes auf eine zeile fr sich */
	/* wenn schon einer zu breit fr die area ist, oder beide zusammen, auf zwei zeilen setzen	*/
	fields[0].adjust.horizontal = ADJleft;
	fields[1].adjust.horizontal = ADJright;
	for (scalefactor=1.0; scalefactor * header->area.height > 1.; scalefactor *= 0.9)
	{
	  if (fillarea (&header->area, scalefactor, fields) == I_OK && fillarea (&header->area, scalefactor, fields+1) == I_OK)
	    if (fields[0].line.area.width + fields[1].line.area.width < header->area.width-2.*header->area.height)
	      break;	/* both on same line */
	    elif (scalefactor <= .4)
	    {
	      fields[0].line.offset.y += header->area.height / 4.;
	      fields[1].line.offset.y -= header->area.height / 4.;
	      break;	/* each on a separate line */
	    }
	}
	break;
    case 3:	/* vier mglichkeiten: alles auf eine zeile, 1. und 2. auf eine, 3. auf zweite zeile, 1. auf eine, 2. und 3. auf eine zeile oder auf drei zeilen */
	/* wenn schon einer zu breit fr die area ist, oder die summe der drei zusammen:
	     wenn zwei benachbarte auf eine zeile passen: die beiden auf eine zeile,
	       sonst: auf drei zeilen setzen	*/
	fields[0].adjust.horizontal = ADJleft;
	/* adjustment of field 1 varies */
	fields[2].adjust.horizontal = ADJright;
	for (scalefactor=1.0; scalefactor * header->area.height > 1.; scalefactor *= 0.9)
	{
	  fields[1].adjust.horizontal = ADJcenter;
	  if (fillarea (&header->area, scalefactor, fields) == I_OK && fillarea (&header->area, scalefactor, fields+1) == I_OK && fillarea (&header->area, scalefactor, fields+2) == I_OK)
	    if (fields[0].line.offset.x + fields[0].line.area.width < fields[1].line.offset.x - scalefactor * header->area.height
	      && fields[1].line.offset.x + fields[1].line.area.width < fields[2].line.offset.x - scalefactor * header->area.height)
	      break;	/* all on same line */
	    elif (scalefactor <= .3)
	    {
	      fields[0].line.offset.y += header->area.height / 3.;
	      fields[1].line.offset.y -= header->area.height / 3.;
	      break;	/* each field on a separate line */
	    }
	    elif (scalefactor <= .4)
	    {
	      fields[1].adjust.horizontal = ADJright;
	      if (fillarea (&header->area, scalefactor, fields+1) == I_OK && fields[0].line.offset.x + fields[0].line.area.width < fields[1].line.offset.x - scalefactor * header->area.height)
	      {
		fields[0].line.offset.y += header->area.height / 4.;
		fields[1].line.offset.y += header->area.height / 4.;
		fields[2].line.offset.y -= header->area.height / 4.;
		break;	/* fields 0 and 1 on one line, field 2 on a second line */
	      }
	      fields[1].adjust.horizontal = ADJleft;
	      if (fillarea (&header->area, scalefactor, fields+1) == I_OK && fields[1].line.offset.x + fields[1].line.area.width < fields[2].line.offset.x - scalefactor * header->area.height)
	      {
		fields[0].line.offset.y += header->area.height / 4.;
		fields[1].line.offset.y -= header->area.height / 4.;
		fields[2].line.offset.y -= header->area.height / 4.;
		break;	/* field 0 on one line, fields 1 and 2 on a second line */
	      }
	    }
	}
	break;
    default:
	ierror ("afillheaderarea: illegal number of fields (%d)", nfield);
  }
  assert (header->type == PE_pageelems);
  while (nfield--)
    liappend (header->u.pageelems, &fields[nfield].line);
}

static void fillheaderarea (t_pageelem *header, ...)
{
  va_list	va;
  t_filldata	fields[3];	/* a maximum of 3 fields is silently enforced... */
  int		nfield;

  va_start (va, header);
  for (nfield=0; nfield < ARRAYDIM (fields); nfield++)
  {
    string		str;
    t_encoding		encoding;
    t_translation	translation;
    string		fontfamily;
    t_point		fontsize;
    
    if (!(str = va_arg (va, string)))
      break;
    encoding = va_arg (va, t_encoding);
    translation = va_arg (va, t_translation);
    fontfamily = va_arg (va, string);
    fontsize = .8 * header->area.height;

    fields[nfield].token = parse_string (str, encoding, translation, fontfamily, fontsize, false);
  }
  va_end (va);
  if (nfield == 1)
    fields[0].adjust.horizontal = ADJcenter;

  afillheaderarea (header, nfield, fields);

  while (nfield--)
    fields[nfield].token = lidestroy (fields[nfield].token);
}

static t_iresult fillheader (t_iostate *iostate, const t_dimension *area, t_pageelem *header, t_headertype type, ...)
{
  char		Times[] = "Times",
		Helvetica[] = "Helvetica";
  va_list	va;
  t_pageelem	bannerbox;

  header->type = PE_void;

  va_start (va, type);

  bannerbox.type = PE_bannerbox;
  bannerbox.offset.x = 0.;
  bannerbox.offset.y = 0.;
  bannerbox.area.width = area->width;
  switch (type)
  {
    case HDR_SHEET:
	++iostate->sheetno;
	bannerbox.area.height = ospec.output->headerheight.sheet;
	break;
    case HDR_FILE:
	bannerbox.area.height = iostate->job->output.headerheight.file;
	break;
    case HDR_PAGE:
	++iostate->pageno;
	bannerbox.area.height = iostate->job->output.headerheight.page;
	break;
    case HDR_Banner:
	bannerbox.area.height = iostate->job->output.headerheight.banner;
	break;
    case HDR_UserBanner:
    case HDR_UserBanner3:
	bannerbox.area.height = iostate->job->output.headerheight.userbanner;
	break;
    default:
	ierror ("fillheader(): default1");
  }
  if (bannerbox.area.height <= 0.)
    return I_OK;
  if (bannerbox.area.height > area->height)
    return I_too_low;

  header->type = PE_pageelems;
  header->u.pageelems = licreate (sizeof (t_pageelem), NULL, freepageelem);
  liappend (header->u.pageelems, &bannerbox);

  header->offset.x = 0.;
  header->offset.y = area->height - bannerbox.area.height;
  header->area.width = area->width;
  header->area.height = bannerbox.area.height;

  switch (type)
  {
    case HDR_SHEET:
      {
	t_buf		buf;

	sprintf (buf, "%d", iostate->sheetno);
	fillheaderarea (header,
		ospec.username, ENC_ISO646, TRLT_guess, Times,
		buf, ENC_ISO646, TRLT_none, Times,
		ctime (&ospec.printtime), ENC_ISO646, TRLT_none, Times,
		NULL);
	say (IMP_entertain, "starting sheet #%d\n", iostate->sheetno);
      }
      break;

    case HDR_FILE:
      {
	cstring	fn;
	long	clock;

	if (fn = va_arg (va, cstring))
	{
	  clock = va_arg (va, long);
	  fillheaderarea (header,
			    fn, ENC_ISOlatin1, TRLT_none, Helvetica,
			    ctime (&clock), ENC_ISO646, TRLT_none, Helvetica,
			    NULL);
	  say (IMP_entertain, "processing file \"%s\"\n", fn);
	}
	else
	{
	  fillheaderarea (header,
			"\\fI(stdin)", ENC_ISO646, TRLT_TROFF, Helvetica,
			NULL);
	  say (IMP_entertain, "processing stdin\n");
	}
      }
      break;

    case HDR_PAGE:
      {
	t_buf	pageno;

	sprintf (pageno, "%d", iostate->pageno);
	if (iostate->firstcolumn)
	{
	  t_buf		fncontd;

	  if (iostate->job->srctype == JOB_FILE)
	    sprintf (fncontd, "%s \\fI(cont'd)", iostate->job->source.fn);
	  fillheaderarea (header,
			  iostate->job->srctype == JOB_FILE ? fncontd : "\\fI(stdin, cont'd)", ENC_ISOlatin1, TRLT_TROFF, Helvetica,
			  pageno, ENC_ISO646, TRLT_none, Times,
			  NULL);
	}
	else
	  fillheaderarea (header,
			  pageno, ENC_ISO646, TRLT_none, Times,
			  NULL);
      }
      break;

    case HDR_Banner:
      {
	cstring	fmt = va_arg (va, cstring);
	t_buf	banner;

	vsprintf (banner, fmt, va);

	fillheaderarea (header,
			banner, ENC_ISO646, TRLT_none, Times,
			NULL);
      }
      break;

    case HDR_UserBanner:
      {
	t_list		*token = va_arg (va, t_list *);
	t_adjustment	horadj = va_arg (va, t_adjustment);
	t_filldata	field;

#if 0
fprintf (stderr, "userbanner:\n");
liamap (token, printtoken, stderr);
#endif
	field.token = token;
	field.adjust.horizontal = horadj;
	afillheaderarea (header, 1, &field);
      }
      break;

    case HDR_UserBanner3:
      {
	t_list		**token = va_arg (va, t_list **);
	t_filldata	fields[3];
	int		field;

	for (field=0; field<3; field++)
	  fields[field].token = token[field];
	afillheaderarea (header, 3, fields);
      }
      break;

    default:
	ierror ("fillheader(): default2");
  }
  va_end (va);
  return I_OK;
}

static t_iresult fillpage (t_iostate *iostate, t_page *page)
{
  page->header = licreate (sizeof (t_pageelem), NULL, freepageelem);

  switch (page->type)
  {
    case PT_SHEET:
	fillheader (iostate, &page->area, liappend (page->header, NULL), HDR_SHEET);
	/* FALL THROUGH */
    case PT_HBOX:
	for (page->u.hbox.filled=0; page->u.hbox.filled<page->u.hbox.ncolumn; page->u.hbox.filled++)
	{
	  iostate->firstcolumn = !page->u.hbox.filled;
	  switch (fillpage (iostate, page->u.hbox.columns+page->u.hbox.filled))
	  {
	    case I_EOJOBS:
		page->u.hbox.filled++;
		return I_EOJOBS;
	    case I_too_low:
	      {
		t_buf	buf;
		error ("file %s, line %d: area too low at token [%s]", jobname (iostate->job), iostate->ilineno, strtoken (buf, liaddr (iostate->token)));
	      }
	  }
	}
	return I_MORE;

    case PT_VBOX:
      {
	t_dimension	remain = page->area;	/* in der VBOX noch zur Verfgung stehender Platz. Die Gre der zum Teil
							gefllten Blcke wird nicht reduziert, um nicht den Header und alle
							Zeilen des Blocks verschieben zu mssen. Statt dessen werden ggf.
							weitere Seiten ber die leere Flche gelegt */
	for (;;)
	{
	  t_page	simple, *subpage;

	  simple.type = PT_SIMPLE;
	  simple.rotation = 0.;
	  simple.offset = vector (0., 0.);
	  simple.area = remain;
	  simple.u.simple = NULL;

#ifndef LONGDEBUG
	  switch (fillpage (iostate, subpage = liappend (page->u.vbox, &simple)))
#else
	  subpage = liappend (page->u.vbox, &simple);
fprintf(stderr, "0 vbox, subpage #%d: pos=%g,%g; area=%gx%g\n", lilength (page->u.vbox), subpage->offset.x, subpage->offset.y, subpage->area.width, subpage->area.height);
{ int fprc;
	  switch (fprc = fillpage (iostate, subpage))
#endif
	  {
	    case I_EOJOBS:
		return I_EOJOBS;
	    case I_too_low:
		lidelete (page->u.vbox, 1);
		if (!lilength (page->u.vbox))
		  return I_too_low;	/* too low for even one block */
		else
		  return I_MORE;	/* try again on next page */
	    case I_EOF:
	    case I_MORE:		/* Seitenberlauf */
		return I_MORE;
	    case I_NEXTPAGE:
	      {
		t_point	filled;

#ifdef LONGDEBUG
fprintf (stderr, "vbox, nextpage: ");
#endif
		/* Hhe des wirklich ausgenutzten Raums feststellen: */
		if (subpage->u.simple && lilength (subpage->u.simple))
		{
		  t_pageelem	*lastpageelem;

		  lastpageelem = lilast (subpage->u.simple); assert (lastpageelem);
		  filled = subpage->area.height - (lastpageelem->offset.y + lastpageelem->descender);
#ifdef LONGDEBUG
fprintf (stderr, "lastpageelem: y=%g, asc=%g, desc=%g; filled=%g\n", lastpageelem->offset.y, lastpageelem->ascender, lastpageelem->descender, filled);
#endif
		}
		else
		{
		  filled = headerheight (subpage);
#ifdef LONGDEBUG
fprintf (stderr, "no lines: filled=%g\n", filled);
#endif
		}
		remain.height -= filled;
#ifdef LONGDEBUG
fprintf(stderr, "1 vbox, subpage #%d: pos=%g,%g; area=%gx%g\n", lilength (page->u.vbox), subpage->offset.x, subpage->offset.y, subpage->area.width, subpage->area.height);
#endif
		continue;
	      }
	    default:
#ifdef LONGDEBUG
		ierror ("fillpage(), PT_VBOX, default, fillpage-rc=%d", fprc);
#else
		ierror ("fillpage(), PT_VBOX, default");
#endif
	  }
#ifdef LONGDEBUG
}
#endif
	}
	ierror ("fillpage(), PT_VBOX...");
      }

    case PT_SIMPLE:
      {
	t_token		*current;
	t_dimension	remain;	/* auf der Seite (oder dem Zeilenblock) noch verbleibender Platz */

	if (gettoken (iostate))
	  return I_EOJOBS;

	if ((current = ligoto (iostate->token, iostate->listoff))->type == TOKEN_COMMAND)
	  switch (current->u.command.code)
	  {
	    case COMMAND_START:
if (false) say (IMP_debug, "COMMAND_START: \"%s\", mod: %s", current->u.command.u.file.fn ? current->u.command.u.file.fn : "<stdin>", ctime (&current->u.command.u.file.modtime));
		iostate->ilineno = 1;
		iostate->pageno = 1;
		if (fillheader (iostate, &page->area, liappend (page->header, NULL), HDR_FILE, current->u.command.u.file.fn, current->u.command.u.file.modtime) != I_OK)
		  return I_too_low;
		if (!structured)
		  if (current->u.command.u.file.fn)
		    pscomment ("start file \"%s\"", current->u.command.u.file.fn);
		  else
		    pscomment ("start file <stdin>");
		iostate->listoff++;
		break;
	    case COMMAND_MailMessage:
		if (fillheader (iostate, &page->area, liappend (page->header, NULL), HDR_Banner, "Message %d", current->u.command.u.messageno) != I_OK)
		  return I_too_low;
		if (!structured)
		  pscomment ("mail message %d", current->u.command.u.messageno);
		iostate->listoff++;
		break;
	    case COMMAND_NewsArticle:
		if (fillheader (iostate, &page->area, liappend (page->header, NULL), HDR_Banner, "Article %d", current->u.command.u.messageno) != I_OK)
		  return I_too_low;
		if (!structured)
		  pscomment ("news article %d", current->u.command.u.messageno);
		iostate->listoff++;
		break;
	    case COMMAND_Banner:
		if (fillheader (iostate, &page->area, liappend (page->header, NULL), HDR_UserBanner, current->u.command.u.banner.token, current->u.command.u.banner.horadj) != I_OK)
		  return I_too_low;
		if (!structured)
		  pscomment ("user header");
		iostate->listoff++;
		break;
	    case COMMAND_Banner3:
		if (fillheader (iostate, &page->area, liappend (page->header, NULL), HDR_UserBanner3, current->u.command.u.banner3) != I_OK)
		  return I_too_low;
		if (!structured)
		  pscomment ("user header 3");
		iostate->listoff++;
		break;
	    default:
		if (fillheader (iostate, &page->area, liappend (page->header, NULL), HDR_PAGE) != I_OK)
		  return I_too_low;
		if (!structured)
		  pscomment ("page %d", iostate->pageno);
	  }
	  else
	  {
	    if (fillheader (iostate, &page->area, liappend (page->header, NULL), HDR_PAGE) != I_OK)
	      return I_too_low;
	    if (!structured)
	      pscomment ("page %d", iostate->pageno);
	  }

	remain.width = page->area.width;
	remain.height = page->area.height - headerheight (page);

	/* oberste Zeile etwas vom Header absetzen: */
	remain.height -= MAX (.3 * iostate->job->output.font[iostate->job->inout.mode].size, iostate->job->output.linespacing.minheight / 2);

	page->u.simple = licreate (sizeof (t_pageelem), NULL, freepageelem);
	for (;;)
	{
	  t_pageelem	*pageelem;

	  if (gettoken (iostate))
	    return I_EOJOBS;

#if 0
fprintf (stderr, "iostate=%#x\n", iostate);
fprintf (stderr, "iostate->token=%#x\n", iostate->token);
fprintf (stderr, "iostate->listoff=%d\n", iostate->listoff);
fprintf (stderr, "iostate->token[iostate->listoff]=%#x\n", ligoto (iostate->token, iostate->listoff));
#endif
	  if ((current = ligoto (iostate->token, iostate->listoff))->type == TOKEN_COMMAND)
	    switch (current->u.command.code)
	    {
	      case COMMAND_MailMessage:
	      case COMMAND_NewsArticle:
		  return I_NEXTPAGE;
	    }

	  switch (fillpageelem (iostate, &remain, 1.0, pageelem = liappend (page->u.simple, NULL)))
	  {
	    case I_EOF:
		iostate->token = lidestroy (iostate->token);
		pageelem->offset.y = remain.height - pageelem->ascender;
		{
		  t_point	pageheight = page->area.height - headerheight (page),
				usedheight = pageheight - remain.height;

		  say (IMP_entertain, "%s: ", jobname (iostate->job));
		  say (IMP_entertain, iostate->pageno<2 ? "one page," : "%d pages, last page", iostate->pageno);
		  say (IMP_entertain, " %.0f%% filled.\n", 100.*usedheight/pageheight);
		}
		iostate->job = NULL;
		if (gettoken (iostate))
		  return I_EOJOBS;
		return I_EOF;
	    case I_OK:
		iostate->token = lidestroy (iostate->token);
		/* FALL THRU */
	    case I_FLUSH:
		if (iostate->token)
		{
		  t_token	*next;

		  /* don't flush the list iff there's another command following: */
		  if (!(next = ligoto (iostate->token, iostate->listoff)) || next->type != TOKEN_COMMAND)
		    iostate->token = lidestroy (iostate->token);
		}
	    case I_MORE:
#if 0
say (IMP_debug, "fillpageelem returns I_OK/I_MORE; %d chargroups\n", lilength (pageelem->chargroups));
say (IMP_debug, "I_OK/I_MORE: listoff=%d, widthoff=%g\n", iostate->listoff, iostate->widthoff);
#endif
		switch (pageelem->type)
		{
		  case PE_separator:
		      pageelem->offset.y = remain.height - pageelem->ascender;
		      remain.height -= pageelem->area.height;
		      break;
		  case PE_void:
		  case PE_chars:
		    {
		      t_point	linespacing;

		      pageelem->offset.y = remain.height - pageelem->ascender;
		      linespacing = iostate->job->output.linespacing.automatic
			? iostate->job->output.linespacing.relative
			  ? iostate->job->output.linespacing.height.factor * pageelem->area.height
			  : pageelem->area.height + iostate->job->output.linespacing.height.size
			: iostate->job->output.linespacing.relative
			  ? iostate->job->output.linespacing.height.factor * iostate->job->output.font[iostate->job->inout.mode].size
			  : iostate->job->output.linespacing.height.size;
		      remain.height -= MAX (iostate->job->output.linespacing.minheight, linespacing);
		      break;
		    }
		  case PE_pageelems:
		      remain.height -= pageelem->area.height;
		      /* nchste Zeile etwas vom Header absetzen: */
		      remain.height -= MAX (.3 * iostate->job->output.font[iostate->job->inout.mode].size, iostate->job->output.linespacing.minheight / 2);
		      break;
		  default:
		      ierror ("fillpage(), PT_SIMPLE, I_OK/I_MORE, pageelem->type=%d", pageelem->type);
		      ierror ("fillpage(), PT_SIMPLE, I_OK/I_MORE");
		}
		continue;
	    case I_too_narrow:
	      {
		t_buf	buf;
		error ("file %s, line %d: area too narrow at token [%s]", jobname (iostate->job), iostate->ilineno, strtoken (buf, liaddr (iostate->token)));
	      }
	    case I_too_low:	/* insufficient space for this pageelem	*/
		lidelete (page->u.simple, 1);
		if (!lilength (page->u.simple))
		  return I_too_low;	/* too low for even one pageelem */
		else
		  return I_MORE;	/* try again on next page */
	    case I_NEWPAGE:
		pageelem->offset.y = remain.height - pageelem->ascender;
		iostate->token = lidestroy (iostate->token);
		return I_NEXTPAGE;
	    default:
		ierror ("fillpage(), PT_SIMPLE, default");
	  }
	}
	ierror ("fillpage(), PT_SIMPLE...");
      }
    default:
	ierror ("fillpage(), default");
  }
  ierror ("fillpage()...");
  abort();/* standard C doesn't recognize volatile for non-returning functions. but gcc knows that abort() and exit() don't return... */
}

/*\[sep]-------------------------------------------------------------------------------------------------------------------------*/ 
static int printchargroup (t_chargroup *chargroup, const t_vector *offset)
{
  uchar	*buf;
  int	left, right, code;

/*
pscomment ("printchargroup: %d chars", chargroup->nchars);
*/
  psmoveto (vecadd (&chargroup->offset, offset));
  buf = (uchar *)alloca (chargroup->nchars);

#ifdef COLORED
  setcolor (&chargroup->color);
#endif
  findfont (chargroup->pfont, chargroup->fontsize);
  for (left = right = 0; right < chargroup->nchars; right++)
  {
    if ((code = encodefont (chargroup->charnames[right])) == fault)
    {
      assert (right - left);
      setfont();
      switch (chargroup->kern)
      {
	case NOKERN:
	    show (buf, right - left);
	    break;
	case XKERN:
	    kxshow (buf, KERN (chargroup, t_point)+left, right - left);
	    break;
	case YKERN:
	    kshow (buf, KERN (chargroup, t_width)+left, right - left);
      }
      left = right;
      findfont (chargroup->pfont, chargroup->fontsize);
      /* retry: */ --right;
    }
    else
      buf[right-left] = code;
  }
  assert (right - left);
  setfont();
  switch (chargroup->kern)
  {
    case NOKERN:
	show (buf, right - left);
	break;
    case XKERN:
	kxshow (buf, KERN (chargroup, t_point)+left, right - left);
	break;
    case YKERN:
	kshow (buf, KERN (chargroup, t_width)+left, right - left);
  }

  return ok; /* for liamap() */
}

static int printpageelem (const t_pageelem *pageelem, const t_vector *offset)
{
  t_vector	suboff = vecadd (offset, &pageelem->offset);

  switch (pageelem->type)
  {
    case PE_void:
	break;
    case PE_separator:
	psgsave();
	setline (&pageelem->u.separator.line);
	psmovetoalign (suboff);
	psrlineto (vector (pageelem->area.width, 0.));
	psstroke();
	psgrestore();
	break;
    case PE_bannerbox:
	psbannerbox (suboff, pageelem->area);
	break;
    case PE_chars:
	liamap (pageelem->u.chargroups, printchargroup, &suboff);
	break;
    case PE_pageelems:
	liamap (pageelem->u.pageelems, printpageelem, &suboff);
	break;
    default:
	ierror ("printpageelem(), default");
  }
  return ok; /* for liamap() */
}

static int printpage (t_page *page, const t_vector *offset)
{
  t_vector	suboff = vecadd (offset, &page->offset);

  if (page->rotation != 0.)
    psrotate (page->rotation);

  liamap (page->header, printpageelem, &suboff);

  switch (page->type)
  {
    case PT_SHEET:
    case PT_HBOX:
	{
	  int	column;
/*
pscomment ("printpage(): PT_SHEET/PT_HBOX");
pscomment ("printpage(): %d/%d columns", page->u.hbox.filled, page->u.hbox.ncolumn);
*/
	  for (column=0; column<page->u.hbox.filled; column++)
	  {
	    if (structured)
	      psstructure (Pages, "BeginObject: (HBox: column %d of %d)", column, page->u.hbox.ncolumn);
	    printpage (page->u.hbox.columns+column, &suboff);
	    if (structured)
	      psstructure (Pages, "EndObject");
	  }
	}
	break;

    case PT_VBOX:
/*
pscomment ("printpage(): PT_VBOX");
pscomment ("printpage(): %d blocks", lilength (page->u.vbox));
*/
	liamap (page->u.vbox, printpage, &suboff);
	break;

    case PT_SIMPLE:
/*
pscomment ("printpage(): PT_SIMPLE");
pscomment ("printpage(): %d page elems", lilength (page->u.simple));
*/
	if (structured)
	  psstructure (Pages, "BeginObject: (Simple Page)");
	liamap (page->u.simple, printpageelem, &suboff);
	if (structured)
	  psstructure (Pages, "EndObject");
	break;

    default:
	abort();
  }

  if (page->rotation != 0.)
    psrotate (-page->rotation);

  return ok;	/* for liamap() */
}

/*\[sep]-------------------------------------------------------------------------------------------------------------------------*/ 
static t_page	*root;

void process (t_list *jobs)
{
  t_vector	offset = vector (0., 0.);
  t_iostate	iostate;
  t_iresult	rc;

  iostate.jobs = jobs;
  iostate.jobno = -1;
  iostate.job = NULL;
  iostate.sheetno = 0;
  iostate.token = NULL;
  do
  {
    rc = fillpage (&iostate, root);
    if (structured)
      psstructure (Pages, "Page: (Blatt %d) %d", iostate.sheetno, iostate.sheetno);
    printpage (root, &offset);
    psshowpage();
    if (rc == I_EOJOBS && root->u.hbox.ncolumn > 1)
    {
      say (IMP_entertain, "%d of %d columns filled on ", root->u.hbox.filled, root->u.hbox.ncolumn);
      if (iostate.sheetno > 1)
	say (IMP_entertain, "the last of %d sheets.\n", iostate.sheetno);
      else
	say (IMP_entertain, "the sheet.\n");
    }
    freepage (root);
    if (structured)
      resetpfonts();
  }
  while (rc != I_EOJOBS);
  if (structured)
    psstructure (Header, "Pages: %d", iostate.sheetno);
  finish_ps();
}

/*\[sep]-------------------------------------------------------------------------------------------------------------------------*/ 
static t_page *construct (t_inputtype inputtype, const t_goutput *output)
{
  t_page	*sheet, *subpage;
  int		ncolumn, column;
  t_point	columnsep, columnwidth, columnheight;

  sheet = getmem0 (sizeof (t_page));
  sheet->type = PT_SHEET;
  if (output->landscape)
  {
    sheet->rotation = 90.;
    sheet->offset.x = ospec.imagearea.lly;
    sheet->offset.y = -ospec.imagearea.urx;
    sheet->area.width = ospec.imagearea.ury - ospec.imagearea.lly;
    sheet->area.height = ospec.imagearea.urx - ospec.imagearea.llx;
  }
  else
  {
    sheet->rotation = 0.;
    sheet->offset.x = ospec.imagearea.llx;
    sheet->offset.y = ospec.imagearea.lly;
    sheet->area.width = ospec.imagearea.urx - ospec.imagearea.llx;
    sheet->area.height = ospec.imagearea.ury - ospec.imagearea.lly;
  }

  ncolumn = output->columns;
  columnsep = output->columnsep;
  columnwidth = (sheet->area.width - (ncolumn-1) * columnsep) / ncolumn;
  columnheight = sheet->area.height - output->headerheight.sheet;
/*
pscomment ("construct(): width=%g, %d columns, columnsep=%g, columnwidth=%g", sheet->area.width, ncolumn, columnsep, columnwidth);
*/

  sheet->u.hbox.columns = getmem0 ((sheet->u.hbox.ncolumn = ncolumn) * sizeof (t_page));
  for (column=0, subpage=sheet->u.hbox.columns; column<ncolumn; subpage++, column++)
  {
    subpage->rotation = 0.;
    subpage->offset.x = column * (columnwidth + columnsep);
    subpage->offset.y = 0.;
    subpage->area.width = columnwidth;
    subpage->area.height = columnheight;
    switch (inputtype)
    {
      case IT_Mail:
      case IT_News:
	  subpage->type = PT_VBOX;
          subpage->u.vbox = licreate (sizeof (t_page), NULL, freepage);
	  break;
      default:
	  subpage->type = PT_SIMPLE;
	  subpage->u.simple = NULL;
    }
  }

  return sheet;
}

static int title (const t_job *job, int *files)
{
  psstructure (Header, "%s (%s)", (*files)++ ? "+" : "Title:", jobname (job));
  return ok; /* for liamap() */
}

void init_output (t_printerspec *printerspec, FILE *ostream, cstring username, t_inputtype inputtype, const t_goutput *output, t_list *jobs)
{
  init_printer (printerspec, ostream);

  ospec.username = username;
  ospec.printtime = time (NULL);
  ospec.inputtype = inputtype;
  ospec.output = output;

  get_imageablearea (&ospec.imageablearea);
  ospec.imagearea.llx = ospec.imageablearea.llx + output->borderwidth.left;
  ospec.imagearea.lly = ospec.imageablearea.lly + output->borderwidth.bottom;
  ospec.imagearea.urx = ospec.imageablearea.urx - output->borderwidth.right;
  ospec.imagearea.ury = ospec.imageablearea.ury - output->borderwidth.top;
  if (structured)
    psstructure (Header, "BoundingBox: %d %d %d %d", (int)floor (ospec.imageablearea.llx), (int)floor (ospec.imageablearea.lly), (int)ceil (ospec.imageablearea.urx), (int)ceil (ospec.imageablearea.ury));

  root = construct (inputtype, output);

  say (IMP_entertain2, "username: \"%s\"\n", ospec.username);
  say (IMP_entertain2, "image area: %g %g %g %g\n", ospec.imagearea.llx, ospec.imagearea.lly, ospec.imagearea.urx, ospec.imagearea.ury);
  if (structured)
  {
    int	files = 0;

    psstructure (Header, "For: %.240s", ospec.username);
    liamap (jobs, title, &files);
    psstructure (Header, "Orientation: %s", output->landscape ? "Landscape" : "Portrait");
    psstructure (Header, "PageOrder: Ascend");
  }
  else
    pscomment ("output generated for \"%s\"", ospec.username);
}
