%{

/*
 * program: psfilt
 * file: lex-input.l
 *
 * Copyright  1992 1993 Robert Joop
 *
 * lexical parser for all types of input
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: lex-input.l,v $
 * Revision 1.7  1994/08/19  09:36:37  rj
 * IT_TEXTline mode for mail and news
 *
 * Revision 1.6  1994/08/18  12:13:18  rj
 * new input mode for Tcl scripts.
 *
 * Revision 1.5  1994/07/09  17:34:36  rj
 * input token buffers (especially in mail mode) are no longer in danger of overflowing
 *
 * Revision 1.4  1994/07/09  16:43:49  rj
 * a lot of const's removed and added
 *
 * Revision 1.3  1994/01/09  23:45:47  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.2  1994/01/03  13:22:49  rj
 * fix for flex 2.3.7 (unsigned char *yytext) vs. flex 2.4.5 (char *yytext).
 *
 * Revision 1.1.1.1  1993/12/31  20:56:38  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: lex-input.l,v 1.7 1994/08/19 09:36:37 rj Exp $";

#include <math.h>	/* atof() */

#include "psfilt.h"
#include "list.h"
#include "ucs2list.h"
#include "lex-input.h"
#include "str.h"
#include "verbose.h"
#include "error.h"

#undef YY_DECL
#define YY_DECL		int lex_input (t_inputtoken *lvalp)

/* flex normally read(2)s from (fileno (yyin)), but we also want to read from strings... (like sscanf(3))	*/
#undef YY_INPUT
#define YY_INPUT(buf, result, max_size)	\
			result = fread (buf, 1, max_size, yyin)

typedef enum
{
  NO_COMMENT,
  LINE_COMMENT,
  LONG_COMMENT,
} t_commenttype;

typedef struct
{
  cstring		jobname;
  const t_input		*input;
  const t_inout		*inout;
  int			translation;

  YY_BUFFER_STATE	buffer;

  int			*lineno;	/* for useful error messages */
  int			*colno;
  int			state;
#define begin(newstate)	BEGIN (job->state = (newstate))
  int			aroundstring;
  int			psstrnesting;
  t_commenttype		comment;
  int			aroundcomment;
  int			aroundcommand;
  int			header, body;	/* NEWS_* or MAIL_* */
} t_lexjob;

#define begin_comment(type)	\
do {						\
  (void)memcpy (lvalp->text, yytext, yyleng+1);	\
  job->aroundcomment = job->state;		\
  job->comment = type;				\
  begin (job->translation);			\
  return ENTER_TRANSLATION;			\
} while (false)

#define end_comment	\
do {				\
  job->comment = NO_COMMENT;	\
  begin (job->aroundcomment);	\
} while (false)

static t_list	*jobstack;
static t_lexjob	*job;

static void lexinputsay (t_verblevel importance, cstring fmt, ...)
{
  va_list	va;

  say (importance, "input lexer: file %s, line %d: ", job->jobname, *job->lineno);
  va_start (va, fmt);
  vsay (importance, fmt, va);
  va_end (va);
  say (importance, "\n");
}

%}

wsp		[ \t\r\n\f]
sep		[,;/]
digit		[0-9]
float		({digit}+|{digit}*\.{digit}+)
symbol		((\[[^\]\n]+\])|(\{[^}\n]\})|(\<[^>\n]\>))
ident		([A-Za-z_][A-Za-z0-9]+)

float3		({float}{sep}{wsp}*{float}{sep}{wsp}*{float})
floattrip	((:{wsp}*{float3})|(\({float3}\))|(\({float3}\)))

%start OPTIONS_LHS OPTIONS_RHS MAKEFILE SHELL C Cplusplus PERL minusminus PS LISP TCL_START TCL_CMD SHELL_STRING1 C_STRING PS_STRING MAN TROFF MOS MS COMMAND CMD_MNEMO TEX MAIL_HEADER MAIL_BODY NEWS_BODY NEWS_HEADER sendmail_7bit

%%

<OPTIONS_LHS>#.*		; /* comment */
<OPTIONS_LHS>[\t ]*		; /* empty line */
<OPTIONS_LHS>-[^\t]+		{
				  (*job->colno)++;	/* count the words */
				  lvalp->option = strunique (yytext);
				  begin (OPTIONS_RHS);
				  return OPT_NAME;
				}

<OPTIONS_RHS>[\t ]+		; /* white space */
<OPTIONS_RHS>[^\t \r\n\f]+	{
				  (*job->colno)++;	/* count the words */
				  lvalp->option = strunique (yytext);
				  return OPT_ARG;
				}

<OPTIONS_LHS,OPTIONS_RHS>\r?\n	{
				  ++*job->lineno;
				  *job->colno = 0;
				  begin (OPTIONS_LHS);
				  return END_OPT;
				}

^[\t ]+				{
				  uchar	*p;

				  if (job->inout->tabmode != TMsimple)
				    ierror ("lex-input, INDENT");
				  for (lvalp->number=0, p=yytext; *p; p++)
				    switch (*p)
				    {
				      case '\t':
					  lvalp->number = job->input->tabcolumns * (lvalp->number / job->input->tabcolumns + 1);
					  break;
				      case ' ':
					  lvalp->number++;
					  break;
				      default:
					  abort();
				    }
				  return INDENT;
				}

\t				{ return TAB; }

<MAKEFILE,SHELL,PERL>#		{
				  begin_comment (LINE_COMMENT);
				}

<C,Cplusplus>"/*"		{
				  begin_comment (LONG_COMMENT);
				}

"*/"				{
				  (void)memcpy (lvalp->text, yytext, yyleng+1);
				  if (job->comment == LONG_COMMENT)
				  {
				    end_comment;
				    return LEAVE_TRANSLATION;
				  }
				  else
				    return STRING;
				}

<C,Cplusplus>"//"		{
				  begin_comment (LINE_COMMENT);
				}

<PS>"%"				{
				  begin_comment (LINE_COMMENT);
				}

<LISP>";"			{
				  begin_comment (LINE_COMMENT);
				}

<PERL>\$['"#]			{ /* don't start string or comment */
				  (void)memcpy (lvalp->text, yytext, yyleng+1);
				  return STRING;
				}
<PERL>(&|{ident})('{ident})+	{ /* a'b is an identifier, has nothing to do with a string */
				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng);
				  (void)memcpy (lvalp->ltext.buf, yytext, yyleng);
				  return lSTRING;
				}

<TCL_START>#			{
				  begin_comment (LINE_COMMENT);
				}
<TCL_START>[^\t\r\n ]		{
				  begin (TCL_CMD);
				  REJECT;
				}
<TCL_CMD>;			{
				  begin (TCL_START);
				  return *(uchar *)yytext;
				}
<TCL_CMD>\\\r?\n		{
				  lvalp->text[0] = '\\';
				  lvalp->text[1] = '\0';
				  /* tcl state isn't changed! */
				  ++*job->lineno;
				  *job->colno = 0;
				  return escapedNEWLINE;
				}
<TCL_CMD>\r?\n			{
				  ++*job->lineno;
				  *job->colno = 0;
				  begin (TCL_START);
				  return NEWLINE;
				}

<SHELL,PERL>'			{
				  job->aroundstring = job->state;
				  begin (SHELL_STRING1);
				  return *(uchar *)yytext;
				}
<SHELL_STRING1>'		{ /* `echo '\'` results in a single backslash */
				  begin (job->aroundstring);
				  return *(uchar *)yytext;
				}

<MAKEFILE,SHELL,C,Cplusplus,PERL>'((\\([0-7]{1,3}|.))|.)'	{
				  (void)memcpy (lvalp->text, yytext, yyleng+1);
				  return STRING;
				}
<MAKEFILE,SHELL,C,Cplusplus,PERL,COMMAND,C_STRING>\\\"	{
				  (void)memcpy (lvalp->text, yytext, yyleng+1);
				  return STRING;
				}
<MAKEFILE,SHELL,C,Cplusplus,PERL,LISP,TCL_CMD,COMMAND>\"	{
				  job->aroundstring = job->state;
				  begin (C_STRING);
				  return *(uchar *)yytext;
				}
<C_STRING>\"			{
				  begin (job->aroundstring);
				  return *(uchar *)yytext;
				}

<PS,COMMAND>\(			{
				  job->aroundstring = job->state;
				  begin (PS_STRING);
				  job->psstrnesting = 1;
				  return *(uchar *)yytext;
				}
<PS_STRING>\\[()]		{
				  (void)memcpy (lvalp->text, yytext, yyleng+1);
				  return STRING;
				}
<PS_STRING>\(			{
				  job->psstrnesting++;
				}
<PS_STRING>\)			{
				  if (!--job->psstrnesting)
				    begin (job->aroundstring);
				  return *(uchar *)yytext;
				}

<C_STRING,PS_STRING>\r?\n	{
				  if (job->input->type == IT_MAKEFILE)	/* sh -c 'echo "hi!' yields hi! */
				  {
				    lexinputsay (IMP_warning, "missing trailing `\"'.");
				    begin (job->aroundstring);
				  }
				  ++*job->lineno, *job->colno = 0;
				  return NEWLINE;
				}
<C_STRING,PS_STRING>\\\r?\n	{
				  lvalp->text[0] = '\\';
				  lvalp->text[1] = '\0';
				  ++*job->lineno, *job->colno = 0;
				  return escapedNEWLINE;
				}
<C_STRING,PS_STRING>\\[0-7]{1,3}	|
<C_STRING,PS_STRING>\\.		|
<C_STRING,PS_STRING>.		{
				  (void)memcpy (lvalp->text, yytext, yyleng+1);
				  return STRING;
				}

<MAKEFILE,SHELL,PS,C,Cplusplus,PERL,LISP,TCL_CMD>-	{
				  lvalp->ucs2 = UCS2_minus_sign; return GLYPH;
				}
<C,Cplusplus,LISP,PERL>-/-	{ /* --> shall be -- >, not - -> */
				#define around aroundstring
				  job->around = job->state;
				  begin (minusminus);
				  lvalp->ucs2 = UCS2_minus_sign;
				  return GLYPH;
				}
<minusminus>-			{
				  begin (job->around);
				  lvalp->ucs2 = UCS2_minus_sign;
				  return GLYPH;
				#undef around
				}
<C,Cplusplus,PERL,LISP>->	{ lvalp->ucs2 = UCS2_rightwards_arrow; return GLYPH; }

<TROFF,MOS,MS>,,		{ lvalp->ucs2 = UCS2_low_double_comma_quotation_mark; return GLYPH; }
<TROFF,MOS,MS>``		{ lvalp->ucs2 = UCS2_double_turned_comma_quotation_mark; return GLYPH; }
<TROFF,MOS,MS>''		{ lvalp->ucs2 = UCS2_double_comma_quotation_mark; return GLYPH; }

<TROFF,MOS,MS>\\?`		{ lvalp->ucs2 = UCS2_single_turned_comma_quotation_mark; return GLYPH; }
<TROFF,MOS,MS>\\?'		{ lvalp->ucs2 = UCS2_single_comma_quotation_mark; return GLYPH; }

<TROFF,MOS,MS>-			{ lvalp->ucs2 = UCS2_hyphen; return GLYPH; }
<TROFF,MOS,MS>\\-		{ lvalp->ucs2 = UCS2_minus_sign; return GLYPH; }

<TROFF,MOS,MS>\\%		{ /* hyphenation marker */
				  return TROFF_HYPHENATION;
				}
<TROFF,MOS,MS>\\[ 0|^&]		{ /* troff space */
				  (void)memcpy (lvalp->text, yytext+1, yyleng);
				  return TROFF_SPACE;
				}

<TROFF,MOS,MS>\\\(..		{ /* troff non-ASCII character with two letter name */
				  (void)memcpy (lvalp->text, yytext+2, yyleng-1);
				  return TROFF_CHAR2;
				}

<TROFF,MOS,MS>\\f{digit}	{
				  lvalp->number = atoi ((char *)yytext+2);
				  return TROFF_FONTn;
				}

<TROFF,MOS,MS>\\f\(..		{
				  (void)memcpy (lvalp->text, yytext+3, yyleng-2);
				  return TROFF_FONT2;
				}

<TROFF,MOS,MS>\\f.		{
				  (void)memcpy (lvalp->text, yytext+2, yyleng-1);
				  return TROFF_FONT1;
				}

<TROFF,MOS,MS>\\f{symbol}	{
				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng-2);
				  (void)memcpy (lvalp->ltext.buf, chop ((char *)yytext+3), yyleng-2);
				  return TROFF_FONT;
				}

<TROFF,MOS,MS>\\s[-+*/]?{float}	{
				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng-1);
				  (void)memcpy (lvalp->ltext.buf, chop ((char *)yytext+2), yyleng-1);
				  return TROFF_SIZE;
				}

<TROFF,MOS,MS>\\u		{ return TROFF_half_line_up; }
<TROFF,MOS,MS>\\d		{ return TROFF_half_line_down; }

<MOS>\\\*['`^,~:C].		{ /* troff -mos string with one letter name indicating diacritical mark	*/
				  lvalp->text[0] = yytext[2];
				  lvalp->text[1] = yytext[3];
				  lvalp->text[2] = '\0';
				  if (lvalp->text[1] == 'C')
				    lvalp->text[1] = 'v'; /* Hachek */
				  return TROFF_DIACRITIC;
				}

<MS>.\\\*['`^,~:v_./o]		{ /* troff -ms string with one letter name indicating diacritical mark	*/
				  lvalp->text[0] = yytext[3];
				  lvalp->text[1] = yytext[0];
				  lvalp->text[2] = '\0';
				  return TROFF_DIACRITIC;
				}

<TROFF,MOS,MS>\\\*\(..		{ /* troff string with two letter name */
				  (void)memcpy (lvalp->text, yytext+3, yyleng-2);
				  return TROFF_STRING2;
				}

<TROFF,MOS,MS>\\\*.		{ /* troff string with one letter name */
				  (void)memcpy (lvalp->text, yytext+2, yyleng-1);
				  return TROFF_STRING1;
				}

<TROFF,MOS,MS>\\[\\e]		{ lvalp->ucs2 = UCS2_backslash; return GLYPH; }

<TROFF,MOS,MS>\\\[		{
				  job->aroundcommand = job->state;
				  begin (COMMAND);
				  return BEGIN_COMMAND;
				}
<COMMAND,CMD_MNEMO>\]		{
				  begin (job->aroundcommand);
				  return END_COMMAND;
				}

<COMMAND>[+-]			{
				  lvalp->boolean = *yytext == '+';
				  return COMMAND_PLUSMINUS;
				}
<COMMAND>[*/]{float}		{
				  double factor = atof ((char *)yytext+1);
				  lvalp->factor = *yytext == '*' ? factor : 1./factor;
				  return COMMAND_FACTOR;
				}
<COMMAND>{digit}+		{
				  lvalp->number = atoi ((char *)yytext);
				  return COMMAND_NUMBER;
				}
<COMMAND>{float}pt?		{
				  lvalp->size = atof ((char *)yytext);
				  return COMMAND_SIZE;
				}
<COMMAND>{float}\"		{
				  lvalp->size = atof ((char *)yytext) * 72.;
				  return COMMAND_SIZE;
				}
<COMMAND>{float}cm		{
				  lvalp->size = atof ((char *)yytext) / 2.54 * 72.;
				  return COMMAND_SIZE;
				}
<COMMAND>{float}mm		{
				  lvalp->size = atof ((char *)yytext) / 25.4 * 72.;
				  return COMMAND_SIZE;
				}
<COMMAND>rgb{wsp}*{floattrip}	{
#ifdef COLORED
				  if (scancolor (yytext, &lvalp->color))
				    ierror ("lex-input, rgb color");
#endif
				  return COMMAND_COLOR;
				}
<COMMAND>hsb{wsp}*{floattrip}	{
#ifdef COLORED
				  if (scancolor (yytext, &lvalp->color))
				    ierror ("lex-input, hsb color");
#endif
				  return COMMAND_COLOR;
				}
<COMMAND>\/[^ \t/"(;\r\n\f\]]+	{
				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng);
				  (void)memcpy (lvalp->ltext.buf, yytext+1, yyleng);
				  return COMMAND_SYMBOL;
				}
<COMMAND>mn(emos?)?		{ begin (CMD_MNEMO); }
<CMD_MNEMO>;			{ begin (COMMAND); }
<CMD_MNEMO>(\\;|[-!"%'()*+,./0-9:<=>?A-Za-z])+	{ /* semicolon has to be escaped */
				  unsigned char	*in, *out;

				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng+1);
				  for (in=yytext, out=lvalp->ltext.buf; *in; )
				  {
				    if (*in == '\\')
				      if (!*in++)
					break;
				    *out++ = *in++;
				  }
				  *out = '\0';
				  return COMMAND_MNEMO;
				}
<COMMAND>"+-"|[^-+ \t/"(;\r\n\f\]]+	{
				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng+1);
				  (void)memcpy (lvalp->ltext.buf, yytext, yyleng+1);
				  return COMMAND_NAME;
				}
<COMMAND,CMD_MNEMO>{wsp}+	{
				  while (yyleng-->0)
				    if (yytext[yyleng] == '\n')
				    {
				      ++*job->lineno;
				      *job->colno = 0;
				    }
				}

<TEX>\"[AOUaous]		{
				  (void)memcpy (lvalp->text, yytext, yyleng+1);
				  return TEX_DIACRITIC;
				}
<TEX>\\3			{ lvalp->ucs2 = UCS2_latin_small_letter_sharp_s; return GLYPH; }

<TEX>-				{ lvalp->ucs2 = UCS2_hyphen; return GLYPH; }
<TEX>--				{ lvalp->ucs2 = UCS2_en_dash; return GLYPH; }
<TEX>---			{ lvalp->ucs2 = UCS2_em_dash; return GLYPH; }

<MAN>_\b.			{ /* italic (processed manuals)	*/
				  *lvalp->text = yytext[2];
				  return ITALIC_CHAR;
				}

<MAN>.(\b.){1,3}		{ /* bold (processed manuals)	*/
				  int	i;
				  assert (yyleng & 1);
				  for (i=2; i<yyleng; i+=2)
				    if (*yytext != yytext[i])
				      break;
				  if (i < yyleng)
				    yyless (i+1);
				  *lvalp->text = *yytext;
				  return BOLD_CHAR;
				}

<NEWS_BODY>\n((Path|Newsgroups|Article):)\ .*\n	|
<MAIL_BODY,NEWS_BODY>\nFrom\ +[^\t\n ]+\ +[^\t\n ]+.*\n	{
				  *job->lineno += 2;
				  *job->colno = 0;
				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng+1);
				  (void)memcpy (lvalp->ltext.buf, yytext, yyleng+1);
				  begin (job->header);
				  return MailHeaderBeginField;
				}
<MAIL_HEADER,NEWS_HEADER>[ \t].*\n	{
				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng+1);
				  (void)memcpy (lvalp->ltext.buf, yytext, yyleng+1);
				  ++*job->lineno;
				  *job->colno = 0;
				  return MailHeaderContField;
				}
<MAIL_HEADER,NEWS_HEADER>.+\n	{
				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng+1);
				  (void)memcpy (lvalp->ltext.buf, yytext, yyleng+1);
				  ++*job->lineno;
				  *job->colno = 0;
				  return MailHeaderBeginField;
				}
<MAIL_HEADER,NEWS_HEADER>\n	{
				  ++*job->lineno;
				  *job->colno = 0;
				  begin (job->body);
				  return MailEndHeader;
				}
<MAIL_BODY,NEWS_BODY>.*\n	{
				  lvalp->ltext.buf = getmem (lvalp->ltext.len = yyleng+1);
				  (void)memcpy (lvalp->ltext.buf, yytext, yyleng+1);
				  ++*job->lineno;
				  *job->colno = 0;
				  return MailBody;
				}

<sendmail_7bit>\035ss		{ lvalp->ucs2 = UCS2_latin_small_letter_sharp_s; return GLYPH; }

<sendmail_7bit>\035.['`^,~:v_./o]	{
				  lvalp->text[0] = yytext[2];
				  lvalp->text[1] = yytext[1];
				  lvalp->text[2] = '\0';
				  return TROFF_DIACRITIC;	/* kludge! */
				}

<TROFF,MOS,MS>\\c?\r?\n		{
				  ++*job->lineno;
				  *job->colno = 0;
				  if (job->input->type != IT_TEXT)
				  {
				    int i = 0;
				    lvalp->text[i++] = '\\';
				    if (yytext[1] == 'c')
				      lvalp->text[i++] = 'c';
				    lvalp->text[i] = '\0';
				    return escapedNEWLINE;
				  }
				}
\\\r?\n				{
				  lvalp->text[0] = '\\';
				  lvalp->text[1] = '\0';
				  ++*job->lineno;
				  *job->colno = 0;
				  if (job->comment == LINE_COMMENT && job->aroundcomment != TCL_START)
				  {
				    end_comment;
				    return LEAVE_TRANSLATION__NEWLINE;
				  }
				  else
				    return escapedNEWLINE;
				}
\r?\n				{
				  ++*job->lineno;
				  *job->colno = 0;
				  if (job->comment == LINE_COMMENT)
				  {
				    end_comment;
				    *lvalp->text = '\0';
				    return LEAVE_TRANSLATION__NEWLINE;
				  }
				  else
				    return NEWLINE;
				}
\r				{ return NEWLINE; }
\f\n?				{ /* ignore LF following FF */
				  if (yyleng > 1)
				  {
				    ++*job->lineno;
				    *job->colno = 0;
				    if (job->input->ignLFafterFF)
				      lexinputsay (IMP_entertain2, "LF after FF ignored.");
				    else /* don't ignore LF after FF */
				      yyless (1);
				  }
				  return NEWPAGE;
				}

-				{ lvalp->ucs2 = UCS2_en_dash; return GLYPH; }
.				return *(uchar *)yytext;

%%

void pushlexjob (cstring jobname, FILE *fp, const t_input *input, const t_inout *inout, int *lineno, int *colno)
{
  if (!jobstack)
    jobstack = licreate (sizeof (t_lexjob), NULL, NULL);
  assert (jobstack);
  job = lipush (jobstack, NULL);

  job->buffer = yy_create_buffer (fp, YY_BUF_SIZE);
  yy_switch_to_buffer (job->buffer);

  job->jobname = jobname;
  job->input = input;
  job->inout = inout;
  job->lineno = lineno;
  job->colno = colno;
  switch (job->input->translation)
  {
    case TRLT_none:
    case TRLT_guess:
	job->translation = INITIAL;
	break;
    case TRLT_MAN:		/* processed manuals (nroff -man)	*/
	job->translation = MAN;
	break;
    case TRLT_TROFF:		/* troff		*/
	job->translation = TROFF;
	break;
    case TRLT_MOS:		/* troff -mos		*/
	job->translation = MOS;
	break;
    case TRLT_MS:		/* troff -ms		*/
	job->translation = MS;
	break;
    case TRLT_TEX:		/* troff -ms		*/
	job->translation = TEX;
	break;
    case TRLT_sendmail:		/* sendmail(1)'s mapping of diacritical marks	*/
	job->translation = sendmail_7bit;
	break;
    default:
	abort();
  }
  switch (job->input->type)
  {
    case IT_options:
	begin (OPTIONS_LHS);
	break;
    case IT_TEXT:
    case IT_MAN:
    case IT_TEXTline:
	begin (job->translation);
	break;
    case IT_MAKEFILE:
	begin (MAKEFILE);
	break;
    case IT_SHELL:
	begin (SHELL);
	break;
    case IT_C:
	begin (C);
	break;
    case IT_Cplusplus:
	begin (Cplusplus);
	break;
    case IT_PERL:
	begin (PERL);
	break;
    case IT_PS:
	begin (PS);
	break;
    case IT_LISP:
	begin (LISP);
	break;
    case IT_TCL:
	begin (TCL_START);
	break;
    case IT_Mail:
	job->header = MAIL_HEADER;
	job->body = MAIL_BODY;
	begin (job->header);
	break;
    case IT_News:
	job->header = NEWS_HEADER;
	job->body = NEWS_BODY;
	begin (job->header);
	break;
    default:
	ierror ("illegal input type %d", job->input->type);
  }
  job->comment = NO_COMMENT;
}

void poplexjob (void)
{
  if (jobstack && lilength (jobstack))
  {
    yy_delete_buffer (job->buffer);
    lidelete (jobstack, 1);
    if (job = lifirst (jobstack))
    {
      yy_switch_to_buffer (job->buffer);
      BEGIN job->state;
    }
  }
  else
    error ("input lexer: empty job stack");
}
