/*
 * program: psfilt
 * file: files.c
 *
 * Copyright  1992 1993 Robert Joop
 *
 * functions to handle files
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: files.c,v $
 * Revision 1.4  1994/07/09  16:43:37  rj
 * a lot of const's removed and added
 *
 * Revision 1.3  1994/07/09  15:34:11  rj
 * stuff for configuration via GNU autoconf added
 *
 * Revision 1.2  1994/01/09  23:45:41  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.1.1.1  1993/12/31  20:56:46  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: files.c,v 1.4 1994/07/09 16:43:37 rj Exp $";
#ifdef HAVE_UNISTD_H
#include <sys/types.h>
#include <unistd.h>
#endif
          
/* unistd.h defines _POSIX_VERSION on POSIX.1 systems.  */
#if defined(DIRENT) || defined(_POSIX_VERSION)
#include <dirent.h>
#define NLENGTH(dirent) (strlen((dirent)->d_name))
#else /* not (DIRENT or _POSIX_VERSION) */
#define dirent direct
#define NLENGTH(dirent) ((dirent)->d_namlen)
#ifdef SYSNDIR
#include <sys/ndir.h>
#endif /* SYSNDIR */
#ifdef SYSDIR
#include <sys/dir.h>
#endif /* SYSDIR */
#ifdef NDIR
#include <ndir.h>
#endif /* NDIR */
#endif /* not (DIRENT or _POSIX_VERSION) */

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include <errno.h>

#include "psfilt.h"
#include "files.h"
#include "mem.h"
#include "str.h"
#include "verbose.h"
#include "error.h"

extern FILE	*popen (cstring, cstring);
extern int	pclose (FILE *);
extern FILE	*fopen (cstring, cstring);
extern int	fclose (FILE *);

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
t_filter	nofilter[] =
{
  { "", NULL },
  { NULL }
};

static t_stdfilter stdfilters[] =
{
  { "", NULL, NULL },
  { ".Z", "uncompress", "compress" },
  { ".F", "melt", "freeze" },
  { ".z", "gunzip", "gzip" },
  { NULL }
};

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
void clearfileinfo (t_fileinfo *fileinfo)
{
  fileinfo->fullname = NULL;
  fileinfo->flags = 0;
  fileinfo->stat = FS_void;
  fileinfo->dynfilt = false;
  fileinfo->clientfilter = NULL;
  fileinfo->stdfilter = NULL;
  fileinfo->fd = fault;
  fileinfo->fp = NULL;
}

int fileinfopcmpfilename (const t_fileinfo *l, const t_fileinfo *r)
{
  return strcmp (l->fullname, r->fullname);
}

#if 0
int fileinfopcmp (const t_fileinfo *l, const t_fileinfo *r)
{
  int rc;
  if (rc = strcmp (l->fullname, r->fullname))
    return rc;
  if (rc = str00cmp (l->clientfilter->suffix, r->clientfilter->suffix))
    return rc;
  return str00cmp (l->stdfilter->suffix, r->stdfilter->suffix);
}
#endif

t_fileinfo makefileinfo (string fullname, time_t mtime, t_filestat stat, string clientfilter, string stdrdfilter, string stdwrfilter)
{
  t_fileinfo	fileinfo;

  fileinfo.fullname = strsave (fullname);
  fileinfo.flags = 0;
  fileinfo.mtime = mtime;
  fileinfo.stat = stat;
  fileinfo.dynfilt = true;
  fileinfo.clientfilter = getmem (sizeof (t_filter));
  fileinfo.clientfilter->suffix = NULL;
  fileinfo.clientfilter->filter = strsave (clientfilter);
  fileinfo.stdfilter = getmem (sizeof (t_stdfilter));
  fileinfo.stdfilter->suffix = NULL;
  fileinfo.stdfilter->read = strsave (stdrdfilter);
  fileinfo.stdfilter->write = strsave (stdwrfilter);
  fileinfo.fd = fault;
  fileinfo.fp = NULL;
  return fileinfo;
}

int matchfileinfo (string dir, string fnbase, const t_filter *clientfilters, string fullnamebuf, t_fileinfo *fileinfo)
{
  size_t	fnlen = strlen (fnbase),
		suflen = 0;

  fileinfo->fullname = fullnamebuf;
  fileinfo->stat = FS_void;
  fileinfo->dynfilt = false;
  fileinfo->fd = fault;
  fileinfo->fp = NULL;

  for (fileinfo->stdfilter=(t_stdfilter *)stdfilters; fileinfo->stdfilter->suffix; fileinfo->stdfilter++)
    if ((suflen = strlen (fileinfo->stdfilter->suffix)) < fnlen && !memcmp (fnbase+fnlen-suflen, fileinfo->stdfilter->suffix, suflen))
      break;
  assert (fileinfo->stdfilter->suffix);

assert (clientfilters);
  fnlen -= suflen;
  for (fileinfo->clientfilter=(t_filter *)clientfilters; fileinfo->clientfilter->suffix; fileinfo->clientfilter++)
    if ((suflen = strlen (fileinfo->clientfilter->suffix)) < fnlen && !memcmp (fnbase+fnlen-suflen, fileinfo->clientfilter->suffix, suflen))
      break;
  return fileinfo->clientfilter->suffix ? ok : fault;
}

void savefileinfo (t_fileinfo *fileinfo)
{
  fileinfo->fullname = strsave (fileinfo->fullname);
  if (fileinfo->dynfilt)
  {
    fileinfo->clientfilter = memcpy (getmem (sizeof (t_filter)), fileinfo->clientfilter, sizeof (t_filter));
    fileinfo->clientfilter->filter = strsave (fileinfo->clientfilter->filter);
    fileinfo->stdfilter = memcpy (getmem (sizeof (t_stdfilter)), fileinfo->stdfilter, sizeof (t_stdfilter));
    fileinfo->stdfilter->read = strsave (fileinfo->stdfilter->read);
    fileinfo->stdfilter->write = strsave (fileinfo->stdfilter->write);
  }
}

int freefileinfo (t_fileinfo *fileinfo)
{
  int	rc = pfclose (fileinfo);

  fileinfo->fullname = strfree (fileinfo->fullname);
  if (fileinfo->dynfilt)
  {
    strfree (fileinfo->clientfilter->filter);
    free (fileinfo->clientfilter);
    strfree (fileinfo->stdfilter->read);
    strfree (fileinfo->stdfilter->write);
    free (fileinfo->stdfilter);
  }

  return rc;
}

string fi_filename (char *buf, const t_fileinfo *fileinfo)
{
  return strcpy (buf, fileinfo->fullname);
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static FILE *pfopendir (string dir, t_fileinfo *fileinfo, int omode, string basename)
{
assert (!(omode&O_rd) ^ !(omode&O_wr));
assert (!fileinfo->stdfilter->read == !fileinfo->stdfilter->write);

  if (dir)
    sprintf (fileinfo->fullname, "%s/%s", dir, basename);

  if (fileinfo->fp) /* file is already open */
  {
    if (fileinfo->flags & FI_pipe && pclose (fileinfo->fp))
      say (IMP_error, "error during pclose of %s\n", fileinfo->fullname);
    lseek (fileinfo->fd, 0l, SEEK_SET);
    if ((omode&O_rw_wr) == O_rw_wr && fileinfo->flags & FI_nowr)
      return NULL;
  }
  elif (omode & O_rw || omode & O_lock)
  {
    if ((fileinfo->fd = open (fileinfo->fullname, omode&O_rw?O_RDWR:O_RDONLY)) == fault)
    {
      if ((omode&O_rw_wr) == O_rw_wr || (fileinfo->fd = open (fileinfo->fullname, O_RDONLY)) == fault)
	return NULL;
      fileinfo->flags |= FI_nowr;
      if (omode & O_rw)
	say (IMP_entertain, "can't open %s read-write\n", fileinfo->fullname);
    }
  }

  if (omode & O_lock)
    fileinfo->flags |= FI_lock;

  if (fileinfo->flags & FI_lock)
  {
    struct flock	flockbuf;

    flockbuf.l_type = omode & O_wr ? F_WRLCK : F_RDLCK;
    flockbuf.l_whence = SEEK_SET;
    flockbuf.l_start = 0;
    flockbuf.l_len = 0;

assert (fileinfo->fd != fault);
    if (fcntl (fileinfo->fd, F_SETLK, &flockbuf))
      switch (errno)
      {
	case EINVAL: /* return by SunOS' when on tmpfs */
	  say (IMP_warning, "can't get %s lock on %s (locking not supported?)\n", omode&O_wr?"exclusive":"shared", fileinfo->fullname);
	  break;
	case EAGAIN: /* may be returned in future OS'es */
	case EACCES: /* returned when already locked */
	  say (IMP_warning, "can't get %s lock on %s (already locked?)\n", omode&O_wr?"exclusive":"shared", fileinfo->fullname);
	  close (fileinfo->fd);
	  return NULL;
	default:
	  say (IMP_warning, "can't get %s lock on %s (reason unknown)\n", omode&O_wr?"exclusive":"shared", fileinfo->fullname);
      }
    else
      say (IMP_entertain2, "got %s lock on %s\n", omode&O_wr?"exclusive":"shared", fileinfo->fullname);
  }

  if (!fileinfo->clientfilter->filter && !fileinfo->stdfilter->read)
  {
    if (fileinfo->flags & FI_lock)
    {
      if (fileinfo->fp)
	rewind (fileinfo->fp); /* has already been opened read-write or else we would be here */
      elif (!(fileinfo->fp = fdopen (fileinfo->fd, fileinfo->flags & FI_nowr ? "r" : "r+")))
      {
	say (IMP_warning, "can't fdopen %s\n", fileinfo->fullname);
	close (fileinfo->fd);
	return NULL;
      }
    }
    else
    {
      if (!(fileinfo->fp = fopen (fileinfo->fullname, omode & O_rd ? "r" : "w")))
	return NULL;
    }
    return fileinfo->fp;
  }
  else
  {
    fileinfo->flags |= FI_pipe;
    if (omode & O_rd)
    {
      if (!access (fileinfo->fullname, R_OK))
      {
	t_fnbuf	cmdbuf;

	if (fileinfo->flags & FI_lock)
	  sprintf (cmdbuf, "0<&%d ", fileinfo->fd);
	else
	  strcat (strcat (strcpy (cmdbuf, "< "), fileinfo->fullname), " ");
	if (fileinfo->stdfilter->read)
	  strcat (cmdbuf, fileinfo->stdfilter->read);
	if (fileinfo->clientfilter->filter)
	{
	  if (fileinfo->stdfilter->read)
	    strcat (cmdbuf, " | ");
	  strcat (cmdbuf, fileinfo->clientfilter->filter);
	}
	return fileinfo->fp = popen (cmdbuf, "r");
      }
    }
    else
    {
      if (!access (fileinfo->fullname, W_OK))
      {
	t_fnbuf	cmdbuf;

	if (fileinfo->clientfilter->filter)
	{
	  strcpy (cmdbuf, fileinfo->clientfilter->filter);
	  if (fileinfo->stdfilter->write)
	    strcat (cmdbuf, " | ");
	}
	else
	  *cmdbuf = '\0';
	if (fileinfo->stdfilter->write)
	  strcat (cmdbuf, fileinfo->stdfilter->write);
	if (fileinfo->flags & FI_lock)
	{
	  ftruncate (fileinfo->fd, 0);
	  sprintf (cmdbuf+strlen (cmdbuf), " 1>&%d", fileinfo->fd);
	}
	else
	  strcat (strcat (cmdbuf, " > "), fileinfo->fullname);
	return fileinfo->fp = popen (cmdbuf, "w");
      }
    }
    return NULL;
  }
}

#if 0
static int vpfopendir (string *dir, va_list va)
{
  t_fileinfo	*fileinfo = va_arg (va, t_fileinfo *);
  int		omode = va_arg (va, int);
  string	base = va_arg (va, string);
  FILE const	**fpp = va_arg (va, FILE const **);

  return (*fpp = pfopendir (*dir, fileinfo, omode, base)) ? true : false;
}
#endif

bool pfisreadonly (const t_fileinfo *fileinfo)
{
  return !!fileinfo->flags & FI_nowr;
}

FILE *pfreopen (t_fileinfo *fileinfo, int omode)
{
  return pfopendir (NULL, fileinfo, omode, NULL);
}

int pfclose (t_fileinfo *fileinfo)
{
  int	rc = ok;

  if (fileinfo->fp)
  {
    if (fileinfo->flags & FI_pipe)
    {
      if (rc = pclose (fileinfo->fp))
      {
	say (IMP_entertain2, "error %#x during pclose of %s\n", rc, fileinfo->fullname);
	rc = ok;
      }
      if (fileinfo->flags & FI_lock && close (fileinfo->fd))
      {
	perror ("close");
	say (IMP_error, "error during close of %s\n", fileinfo->fullname);
	rc = fault;
      }
      fileinfo->fd = fault;
    }
    else
    {
      if (fclose (fileinfo->fp))
      {
	say (IMP_error, "error during fclose of %s\n", fileinfo->fullname);
	rc = fault;
      }
    }
    fileinfo->fp = NULL;
  }
  return rc;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
/* append suffices and try to open */
FILE *pfopen (string dir, string fnstub, const t_filter *clientfilters, int omode, char *fullnamebuf, t_fileinfo *fileinfo)
{
  int	stublen;
  FILE	*fp;

assert (clientfilters);
  fileinfo->fullname = fullnamebuf;
  fileinfo->flags = 0;
  fileinfo->dynfilt = false;
  fileinfo->fp = NULL;
  if (dir)
#if 0
    stublen = sprintf (fileinfo->fullname, "%s/%s", dir, fnstub);
#else /* some sprintf's are broken: */
  {
    sprintf (fullnamebuf, "%s/%s", dir, fnstub);
    stublen = strlen (fullnamebuf);
  }
#endif
  else
    memcpy (fullnamebuf, fnstub, stublen = strlen (fnstub));

  if (omode & O_wr)
  {
    for (fileinfo->stdfilter=(t_stdfilter *)stdfilters; fileinfo->stdfilter->suffix; fileinfo->stdfilter++)
      for (fileinfo->clientfilter=(t_filter *)clientfilters; fileinfo->clientfilter->suffix; fileinfo->clientfilter++)
      {
	strcat (strcpy (fullnamebuf+stublen, fileinfo->clientfilter->suffix), fileinfo->stdfilter->suffix);
	if (!access (fullnamebuf, W_OK))
	  goto break2;
      }
  break2:
    if (!fileinfo->stdfilter->suffix && !fileinfo->clientfilter->suffix)
    {
      int fd;

      fullnamebuf[stublen] = '\0';
      if ((fd = open (fullnamebuf, O_CREAT, 0666)) == fault)
      {
	say (IMP_warning, "can't create %s\n", fullnamebuf);
	return NULL;
      }
      say (IMP_entertain, "creating %s\n", fullnamebuf);
      if (!(fp = pfopendir (NULL, fileinfo, omode, NULL)))
	ierror ("pfopen, create");
      close (fd);
    }
    return fileinfo->fp;
  }

  for (fileinfo->stdfilter=(t_stdfilter *)stdfilters; fileinfo->stdfilter->suffix; fileinfo->stdfilter++)
    for (fileinfo->clientfilter=(t_filter *)clientfilters; fileinfo->clientfilter->suffix; fileinfo->clientfilter++)
    {
      strcat (strcpy (fullnamebuf+stublen, fileinfo->clientfilter->suffix), fileinfo->stdfilter->suffix);
      if (fp = pfopendir (NULL, fileinfo, omode, NULL))
	return fp;
    }

  return NULL;
}

static int vfopendir (string *dir, va_list va)
{
  string		fnstub = va_arg (va, string);
  const t_filter	*clientfilters = va_arg (va, const t_filter *);
  int			omode = va_arg (va, int);
  char			*fullnamebuf = va_arg (va, char *);
  t_fileinfo		*fileinfo = va_arg (va, t_fileinfo *);
  FILE			**fp = va_arg (va, FILE **);

  return (*fp = pfopen (*dir, fnstub, clientfilters, omode, fullnamebuf, fileinfo)) ? true : false;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
/* find the first matching file */
FILE *pfopenpath (t_list *dirs, string fnstub, const t_filter *clientfilters, int omode, char *fullnamebuf, t_fileinfo *fileinfo)
{
assert (clientfilters);
  if (strchr (fnstub, '/'))
    return pfopen (NULL, fnstub, clientfilters, omode, fullnamebuf, fileinfo);
  else
  {
    FILE	*fp;
    return livamap (dirs, vfopendir, fnstub, clientfilters, omode, fullnamebuf, fileinfo, &fp) ? fp : NULL;
  }
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int _findfile (string name, va_list va)
{
  string		dirfn = va_arg (va, string);
  string		direntry = va_arg (va, string);
  int			(*every)(FILE *, const t_fileinfo *fileinfo, va_list) = va_arg (va, /* int (*)(FILE *, const t_fileinfo *fileinfo, va_list): */ void *);
  va_list		cbva = va_arg (va, va_list);
  va_list		_va = va_arg (va, va_list);
  const t_filter	*clientfilters = va_arg (_va, const t_filter *);
  int			omode = va_arg (_va, int);
  string		fullname = va_arg (_va, string);

  t_fileinfo		fileinfo;
  FILE			*fp;
  int			rc = ok;

  fileinfo.fullname = fullname;
  fileinfo.flags = 0;
  fileinfo.dynfilt = false;
  fileinfo.fp = NULL;
  for (fileinfo.clientfilter=(t_filter *)clientfilters; fileinfo.clientfilter->suffix; fileinfo.clientfilter++)
    for (fileinfo.stdfilter=(t_stdfilter *)stdfilters; fileinfo.stdfilter->suffix; fileinfo.stdfilter++)
    {
      t_fnbuf	pattern;

      sprintf (pattern, "%s%s%s", name, fileinfo.clientfilter->suffix, fileinfo.stdfilter->suffix);
      if (strmatch (pattern, direntry))
	if (fp = pfopendir (dirfn, &fileinfo, omode, direntry))
	{
	  rc = (*every)(fp, &fileinfo, cbva);
	  pfclose (&fileinfo);
	  if (rc)
	    break;
	}
    }
  return rc;
}

static int _findfiles (string *dirfn, va_list va)
{
  t_list	*names = va_arg (va, t_list *);

  DIR		*dir;
  int		rc = ok;

  if (dir = opendir (*dirfn))
  {
    int			(*before)(string dir, va_list) = va_arg (va, void *);
    int			(*every)(FILE *, const t_fileinfo *fileinfo, va_list) = va_arg (va, /* int (*)(FILE *, const t_fileinfo *fileinfo, va_list): */ void *);
    int			(*after)(string dir, va_list) = va_arg (va, void *);
    va_list		cbva = va_arg (va, va_list);
    struct dirent	*entry;

    if (before)
      rc = (*before)(*dirfn, cbva);
    if (every)
      while (!rc && (entry = readdir (dir)))
	rc = livamap (names, _findfile, *dirfn, entry->d_name, every, cbva, va);
    if (!rc && after)
      rc = (*after)(*dirfn, cbva);
    closedir (dir);
  }
  /* else: it is no error for a directory not to be present */
  return rc;
}

int findnamedfiles (t_list *dirs, t_list *names, const t_filter *suffices, int omode, string fullname, int (*before)(string dir, va_list), int (*every)(FILE *, t_fileinfo *fileinfo, va_list), int (*after)(string dir, va_list), ...)
{
  va_list	va;
  int		rc;

assert (fullname);
assert (suffices);
  va_start (va, after);
  rc = livamap (dirs, _findfiles, names, before, every, after, va, suffices, omode, fullname);
  va_end (va);
  return rc;
}

int findfiles (t_list *dirs, const t_filter *suffices, int omode, string fullname, int (*before)(string dir, va_list), int (*every)(FILE *, t_fileinfo *fileinfo, va_list), int (*after)(string dir, va_list), ...)
{
  t_list	*name;
  va_list	va;
  int		rc;

assert (suffices);
assert (fullname);
  name = licreate (sizeof "*", NULL, NULL);
  liappend (name, "*");
  va_start (va, after);
  rc = livamap (dirs, _findfiles, name, before, every, after, va, suffices, omode, fullname);
  va_end (va);
  lidestroy (name);
  return rc;
}
