/*
 * program: psfilt
 * file: color.c
 *
 * Copyright  1992 1993 Robert Joop
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: color.c,v $
 * Revision 1.3  1994/07/09  16:43:31  rj
 * a lot of const's removed and added
 *
 * Revision 1.2  1994/07/09  15:34:08  rj
 * stuff for configuration via GNU autoconf added
 *
 * Revision 1.1.1.1  1993/12/31  20:56:47  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: color.c,v 1.3 1994/07/09 16:43:31 rj Exp $";

#ifdef COLORED

#ifdef HAVE_NDBM
#include <ndbm.h>
#else
#ifdef HAVE_DBM
#include <dbm.h>
#endif
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include "psfilt.h"
#include "color.h"
#include "error.h"

void clearcolor (t_color *color)
{
  color->model = CM_void;
}

bool isdefaultcolor (const t_color *color)
{
  return color->model == CM_void;
}

int colorpcmp (const t_color *l, const t_color *r)
{
  if (l->model != r->model)
    return 1;
  switch (l->model)
  {
    case CM_void:
      return 0;
    case CM_rgb:
      return l->u.rgb.red != r->u.rgb.red || l->u.rgb.green != r->u.rgb.green || l->u.rgb.blue != r->u.rgb.blue;
    case CM_hsb:
      return l->u.hsb.hue != r->u.hsb.hue || l->u.hsb.saturation != r->u.hsb.saturation || l->u.hsb.brightness != r->u.hsb.brightness;
    default:
      ierror ("colorpcmp: illegal color model");
      abort();/* standard C doesn't recognize volatile for non-returning functions. but gcc knows that abort() and exit() don't return... */
  }
}

t_color rgbcolor (double red, double green, double blue)
{
  t_color	rgb;

  rgb.model = CM_rgb;
  rgb.u.rgb.red = red;
  rgb.u.rgb.green = green;
  rgb.u.rgb.blue = blue;
  return rgb;
}

int scancolor (string name, t_color *color)
{
  if (sscanf (name, "rgb%*[^0-9.-]%g%*[,;:/ ]%g%*[,;:/ ]%g", &color->u.rgb.red, &color->u.rgb.green, &color->u.rgb.blue) == 3)
    color->model = CM_rgb;
  elif (sscanf (name, "hsb%*[^0-9.-]%g%*[,;:/ ]%g%*[,;:/ ]%g", &color->u.hsb.hue, &color->u.hsb.saturation, &color->u.hsb.brightness) == 3)
    color->model = CM_hsb;
  else
#ifndef X11COLORDB
    return fault;
#else
  {
#ifdef HAVE_NDBM
    static DBM		*colordb = NULL;
#endif
    datum		colorname, colorvalue;

#ifdef HAVE_NDBM
    if (!colordb && !(colordb = dbm_open (X11COLORDB, O_RDONLY, 0)))
      return fault;
#else
#ifdef HAVE_DBM
    static bool initialized = false;

    if (!initialized)
    {
      dbminit (X11COLORDB);
      initialized = true;
    }
#endif
#endif

    colorname.dptr = name;
    colorname.dsize = strlen (name);

#ifdef HAVE_NDBM
    colorvalue = dbm_fetch (colordb, colorname);
#else
#ifdef HAVE_DBM
    fetch (colorname);
#endif
#endif
    if (!colorvalue.dptr)
      return fault;

    color->model = CM_rgb;
    /* datum.dptr may be odd! */
#define COPY(to, i)				\
{						\
  unsigned short	val;			\
  memcpy (&val, colorvalue.dptr+2*i, 2);	\
  to = (double)val/65535.;			\
}
    COPY (color->u.rgb.red, 0)
    COPY (color->u.rgb.green, 1)
    COPY (color->u.rgb.blue, 2)
#undef COPY
  }
#endif
  return ok;
}

string strcolor (string buf, const t_color *color)
{
  switch (color->model)
  {
    case CM_void:
      strcpy (buf, "<void>");
      break;
    case CM_rgb:
      sprintf (buf, "RGB[%g,%g,%g]", color->u.rgb.red, color->u.rgb.green, color->u.rgb.blue);
      break;
    case CM_hsb:
      sprintf (buf, "HSB[%g,%g,%g]", color->u.hsb.hue, color->u.hsb.saturation, color->u.hsb.brightness);
      break;
    default:
      strcpy (buf, "<illegal>");
  }
  return buf;
}

#endif
