/*
 * program: psfilt
 * file: cache.c
 *
 * Copyright  1992 1993 Robert Joop
 *
 * format of the cache file(s):
 *	magic line
 *	iff the file is global: afm and font paths
 *	afm infos grouped by fontfamilies
 *	font infos
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: cache.c,v $
 * Revision 1.5  1994/07/16  12:35:43  rj
 * provisions to use X11R6 PS type 1 fonts
 *
 * Revision 1.4  1994/07/09  16:43:29  rj
 * a lot of const's removed and added
 *
 * Revision 1.3  1994/01/09  23:45:37  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.2  1994/01/04  11:56:19  rj
 * minor adjustments
 *
 * Revision 1.1.1.1  1993/12/31  20:56:47  rj
 * erster cvs import.
 *
 */

static const char RCSId[] = "$Id: cache.c,v 1.5 1994/07/16 12:35:43 rj Exp $";

/*
 *	begin:
 *		holt die infos ggf. aus dem cache
 *	scan{afm,font}file:
 *		infos aus cache (von begin) berprfen oder ggf. dazufgen
 *	end:
 *		gltige infos in interne daten konvertieren
 *		gltige infos ggf. im cache abgespeichern
 */

#include <sys/types.h>
#include <sys/stat.h>

#include "psfilt.h"
#include "afm.h"
#include "fonts.h"	/* for fontdirs, fonts, fontfamilies	*/
#include "ps.h"		/* for psinitafm(), pscompleteafm() */
#include "cache.h"
#include "iso646.h"
#include "str.h"
#include "mem.h"
#include "verbose.h"
#include "error.h"

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
typedef struct
{
  t_fileinfo	cachefileinfo;

  int		flags;
#define DI_readonly	0x01
#define DI_rewrite	0x02
#define DI_create	0x04

  t_avl		*afmfiles;	/* (t_afmfile) */
  t_avl		*fontfiles;	/* (t_fontfile) */
} t_dirinfos;

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static t_avl	*fonts,			/* (t_font), nach fontname sortiert */
		*fontfamilies;		/* (t_fontfamily), nach familyname sortiert */

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
#define CHECK_afm	0x01
#define CHECK_font	0x02
#define CHECK_all	(CHECK_afm|CHECK_font)

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int checkinode (const struct stat *statbuf, const t_fileinfo *info)
{
  return statbuf->st_mtime == info->mtime ? ok : fault;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static const t_filter	afmsuffices[] =
{
  { ".afm", NULL },
  { NULL }
};

static void freevoidafmfile (t_afmfile *afmfile)
{
  if (afmfile->file.stat == FS_void)
    freeafmfile (afmfile);
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static const t_filter	fontsuffices[] =
{
  { ".ps", NULL },
  { ".gsf", NULL },
  { ".pfa", NULL },
  { ".pfb", "pfbtops" },
  { NULL }
};

static void freevoidfontfile (t_fontfile *fontfile)
{
  if (fontfile->file.stat == FS_void)
    freefontfile (fontfile);
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static cstring magic (char *buf, bool global)
{
  static const char	magicfmt[] = "=> this file contains information about %s. DON'T TOUCH! <=\n";

  sprintf (buf, magicfmt, global ? "all fonts and their metrics" : "the fonts or font metrics in this directory");
  return buf;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int readfileinfo (const char *buf, string dirname, t_fileinfo *fileinfo)
{
  int		sublen, len;
  t_fnbuf	fnbuf, clfbuf, stdrfbuf, stdwfbuf;
  char		*filename, *clf = clfbuf, *stdrf = stdrfbuf, *stdwf = stdwfbuf;
  time_t	mtime;
  
  if (dirname)
  {
    int	dirnmlen = strlen (dirname);
    memcpy (fnbuf, dirname, dirnmlen);
    fnbuf[dirnmlen++] = '/';
    filename = fnbuf + dirnmlen;
  }
  else
    filename = fnbuf;
  if ((len = getstring (buf, &filename)) == fault)
    return fault;
  if ((sublen = getstring (buf+len, &clf)) == fault)
    return fault;
  if ((sublen = getstring (buf+(len+=sublen), &stdrf)) == fault)
    return fault;
  if ((sublen = getstring (buf+(len+=sublen), &stdwf)) == fault)
    return fault;
  if (sscanf (buf+(len+=sublen), "%lu", &mtime) != 1)
    return fault;
  *fileinfo = makefileinfo (fnbuf, mtime, FS_fromcache, clf, stdrf, stdwf);
  return len+10;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int checkmatchold (cstring old, cstring *new)
{
  return streq (old, *new);
}

/* fr jedes aktuelle dir berprfen, ob's schon im cache ist */
static int checkmissnew (cstring *new, va_list va)
{
  if (!liamap (va_arg (va, t_list *), checkmatchold, new))
  {
    say (IMP_warning, "directory \"%s\" is new in the %s directory list\n", *new, va_arg (va, cstring));
    ++*va_arg (va, int *);
  }
  return ok;
}

static int checkmatchnew (cstring *new, cstring old)
{
  return streq (old, *new);
}

/* fr jedes gecachte dir berprfen, ob's noch in der aktuellen liste ist */
static int checkmissold (string old, va_list va)
{
  if (!liamap (va_arg (va, t_list *), checkmatchnew, old))
  {
    say (IMP_warning, "directory \"%s\" has vanished from the %s directory list\n", old, va_arg (va, cstring));
    ++*va_arg (va, int *);
  }
  return ok;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int begin (string dirname, va_list va)
{
  const t_cachespec	*cache = va_arg (va, const t_cachespec *);
  int			checkwhat = va_arg (va, int);
  t_dirinfos		*dirinfos = va_arg (va, t_dirinfos *);

  t_fnbuf		fnbuf;
  FILE			*fp;

  checkwhat = checkwhat;

  dirinfos->flags = 0;
  dirinfos->afmfiles = avlcreate (sizeof (t_afmfile), afmfilepcmpfilename, freevoidafmfile);
  dirinfos->fontfiles = avlcreate (sizeof (t_fontfile), fontfilepcmpfilename, freevoidfontfile);

  if (cache->flags & CACHE_use)
  {
    if (fp = pfopen (dirname, cache->filename, nofilter, O_lock | (cache->flags & CACHE_update ? O_rw_rd : O_rdonly), fnbuf, &dirinfos->cachefileinfo))
    {
      if (pfisreadonly (&dirinfos->cachefileinfo))
	dirinfos->flags |= DI_readonly;

      if (CACHE_rebuild (cache->flags))
      {
	say (IMP_entertain, "rebuilding %s\n", dirinfos->cachefileinfo.fullname);
	dirinfos->flags |= DI_rewrite;
      }
      else
      {
	t_fnbuf	line;
	int	lino = 1;
	cstring	corrupt = NULL;

	/* check the magic line: */
	{
	  t_buf	magicbuf;

	  if (!fgets (line, sizeof line, fp))
	    corrupt = "empty file?";
	  if (!corrupt && strcmp (line, magic (magicbuf, cache->flags & CACHE_global)))
	    corrupt = "wrong magic line";
	}

	/* get the afm and font paths: */
	if (cache->flags & CACHE_global)
	{
	  bool	eodir;
	  t_fnbuf	dirbuf;
	  string	dir = dirbuf;
	  int	dirlen;
	  int	miss;
	  t_list	*afmpath, *fontpath;

	  afmpath = licreate (0, NULL, NULL);
	  eodir = false;
	  while (!eodir && !corrupt && fgets (line, sizeof line, fp))
	  {
	    lino++;
	    switch (*line)
	    {
	      case ISO646_GS:
		eodir = true;
		break;
	      default:
		if ((dirlen = getstring (line, &dir)) == fault)
		  corrupt = "afm path section";
		else
		  liappend (afmpath, dir, dirlen+1);
	    }
	  }
	  if (!corrupt)
	  {
	    miss = 0;
	    livamap (afmpath, checkmissold, afmdirs, "afm", &miss);
	    livamap (afmdirs, checkmissnew, afmpath, "afm", &miss);
	  }
	  lidestroy (afmpath);

	  fontpath = licreate (0, NULL, NULL);
	  eodir = false;
	  while (!eodir && !corrupt && fgets (line, sizeof line, fp))
	  {
	    lino++;
	    switch (*line)
	    {
	      case ISO646_GS:
		eodir = true;
		break;
	      default:
		if ((dirlen = getstring (line, &dir)) == fault)
		  corrupt = "font path section";
		else
		  liappend (fontpath, dir, dirlen+1);
	    }
	  }
	  if (!corrupt)
	  {
	    miss = 0;
	    livamap (fontpath, checkmissold, fontdirs, "font", &miss);
	    livamap (fontdirs, checkmissnew, fontpath, "font", &miss);
	  }
	  lidestroy (fontpath);
	}

	/* get the afm infos sorted by font families: */
	{
	  bool	eoafm = false;
	  string	fontfamily = NULL;

	  while (!eoafm && !corrupt && fgets (line, sizeof line, fp))
	  {
	    lino++;
	    switch (*line)
	    {
	      case ISO646_GS:
		eoafm = true;
		break;
	      case '\t':	/* font */
	      {
		int		len;
		t_afmfile	afmfile;
		t_buf		fontname, fontversion, fontweight;

		clearafmfile (&afmfile);
		if ((len = readfileinfo (line+1, cache->flags & CACHE_global ? NULL : dirname, &afmfile.file)) == fault
		  || sscanf (line+1+len, "%[^(](%[^)])%[^/]/%lg",
					  fontname,
						fontversion,
						      fontweight,
							    &afmfile.italicangle) != 4)
		  corrupt = "afm section";
		else
		{
		  afmfile.file.stat = FS_fromcache;
		  afmfile.familyname = fontfamily;
		  afmfile.fontname = strsave (fontname);
		  afmfile.version = getversion (fontversion);
		  afmfile.weight = fontweightbyname (fontweight);
		  avlinsert (dirinfos->afmfiles, &afmfile);
		}
		break;
	      }
	      default:	/* new fontfamily */
		fontfamily = strsave (chop (line));
	    }
	  }
	}

	/* get the font infos (sorted by fontname): */
	{
	  bool		eofont = false;

	  while (!eofont && !corrupt && fgets (line, sizeof line, fp))
	  {
	    lino++;
	    switch (*line)
	    {
	      case ISO646_GS:
		eofont = true;
		break;
	      default:
	      {
		int		len;
		t_fontfile	fontfile;
		t_buf		fontname, fontversion;

		clearfontfile (&fontfile);
		if ((len = readfileinfo (line, cache->flags & CACHE_global ? NULL : dirname, &fontfile.file)) == fault
		  || sscanf (line+len, "%[^(](%[^)])",
					fontname,
					      fontversion) != 2)
		  corrupt = "font section";
		else
		{
		  fontfile.file.stat = FS_fromcache;
		  fontfile.fontname = strsave (fontname);
		  fontfile.version = getversion (fontversion);
		  avlinsert (dirinfos->fontfiles, &fontfile);
		}
	      }
	    }
	  }
	}

	if (corrupt)
	{
	  say (IMP_warning, "file %s, line %d: corrupt input (%s)\n", dirinfos->cachefileinfo.fullname, lino, corrupt);
	  dirinfos->flags |= DI_rewrite;
	}
      }
    }
    else
    {
      dirinfos->flags |= DI_create;
      if (cache->flags & CACHE_global)
	say (cache->flags & CACHE_update ? IMP_entertain : IMP_warning, "can't open cache file %s\n", cache->filename);
      else
	say (cache->flags & CACHE_update ? IMP_entertain : IMP_warning, "can't open cache file in directory %s\n", dirname);
      if (cache->flags & CACHE_update)
      {
	matchfileinfo (dirname, cache->filename, nofilter, fnbuf, &dirinfos->cachefileinfo);
	dirinfos->flags |= DI_rewrite;
      }
    }
    if (cache->flags & CACHE_update)
      savefileinfo (&dirinfos->cachefileinfo);
  }
  return ok;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static t_font *storefont (string fontname)
{
  t_font	fsearch, *font;
  bool		insert = true;

  fsearch.fontname = fontname;
  font = avlinsrch (fonts, &insert, &fsearch);
  if (insert)
  {
    font->afms = avlcreate (sizeof (t_afmfile), afmfilepcmpversion, freeafmfile);
    font->fonts = avlcreate (sizeof (t_fontfile), fontfilepcmpversion, freefontfile);
  }
  return font;
}

static int storefontfamily (string familyname, t_font *font)
{
  t_fontfamily	ffsearch, *fontfamily;
  bool		insert;

  if (familyname)
  {
    ffsearch.familyname = familyname;
    insert = true;
    fontfamily = avlinsrch (fontfamilies, &insert, &ffsearch);
    if (insert)
    {
      fontfamily->fonts = avlcreate (sizeof (t_font *), fontppcmp, NULL);
    }
    insert = true;
    avlinsrch (fontfamily->fonts, &insert, &font);
    return ok;
  }
  else
    return fault;
}

static int storeafmfile (t_afmfile *_afmfile, va_list va)
{
  t_font	*font;
  bool		insert;
  t_afmfile	*afmfile;

  if (_afmfile->file.fullname)
  {
    int	flags = va_arg (va, int);

    if (_afmfile->file.stat != (!(flags & CACHE_use) || (flags & (CACHE_check|CACHE_update)) ? FS_valid : FS_fromcache))
    {
      if (_afmfile->file.stat != FS_void)
	say (IMP_entertain, "file %s has vanished.\n", _afmfile->file.fullname);
      _afmfile->file.stat = FS_void;
      return ok;
    }
  }

  font = storefont (_afmfile->fontname);
  insert = true;
  afmfile = avlinsrch (font->afms, &insert, _afmfile); assert (afmfile);
  if (!insert)
  {
    t_buf	fontversion;
    t_fnbuf	fn0, fn1;

    say (IMP_warning, "metrics for font %s with version %s appearing again (files: \"%s\" and \"%s\")\n", afmfile->fontname, strversion (fontversion, afmfile->version), fi_filename (fn0, &afmfile->file), fi_filename (fn1, &_afmfile->file));
  }
  if (storefontfamily (afmfile->familyname, font))
  {
    t_fnbuf	fn;
    say (IMP_warning, "file %s: font /%s has no FontFamily!\n", fi_filename (fn, &afmfile->file), afmfile->fontname);
  }

  return ok;
}

static int storefontfile (t_fontfile *_fontfile, va_list va)
{
  int		flags = va_arg (va, int);

  if (_fontfile->file.stat != (!(flags & CACHE_use) || (flags & (CACHE_check|CACHE_update)) ? FS_valid : FS_fromcache))
  {
    if (_fontfile->file.stat != FS_void)
      say (IMP_entertain, "file %s has vanished.\n", _fontfile->file.fullname);
    _fontfile->file.stat = FS_void;
  }
  else
  {
    t_font *font = storefont (_fontfile->fontname);
    bool insert = true;
    t_fontfile *fontfile = avlinsrch (font->fonts, &insert, _fontfile); assert (fontfile);
    if (!insert)
    {
      t_buf	fontversion;
      t_fnbuf	fn0, fn1;

      say (IMP_warning, "metrics for font %s with version %s appearing again (files: \"%s\" and \"%s\")\n", fontfile->fontname, strversion (fontversion, fontfile->version), fi_filename (fn0, &fontfile->file), fi_filename (fn1, &_fontfile->file));
    }

    /* no way to store the font under its fontfamily directly, only thru the afmfile */
    /* doesn't matter as if there's no afm file for this font, it can't be used anyway */
  }
  return ok;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int writedir (cstring *dir, FILE *fp)
{
  return putstring (*dir, fp) || putc ('\n', fp) == EOF;
}

static int writepath (t_list *dirs, FILE *fp)
{
  if (liamap (dirs, writedir, fp))
    return fault;
  if (fprintf (fp, "%c\n", ISO646_GS) == EOF)
    return fault;
  return ok;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int writefileinfo (const t_fileinfo *fileinfo, bool global, FILE *fp)
{
  if (putstring (global ? fileinfo->fullname : basename (fileinfo->fullname), fp))
    return fault;
  if (putstring (fileinfo->clientfilter->filter, fp))
    return fault;
  if (putstring (fileinfo->stdfilter->read, fp))
    return fault;
  if (putstring (fileinfo->stdfilter->write, fp))
    return fault;
#ifdef ultrix
  if (fprintf (fp, "%10lu", fileinfo->mtime) != 0)
#else
  if (fprintf (fp, "%10lu", fileinfo->mtime) != 10)
#endif
    return fault;
  return ok;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int writeafmfile (const t_afmfile *afmfile, va_list va)
{
  if (afmfile->file.stat == va_arg (va, t_filestat))
  {
    bool	global = va_arg (va, bool);
    FILE	*fp = va_arg (va, FILE *);
    string	*familyname = va_arg (va, string *);
    t_buf	fontversion, fontweight;

    if (!*familyname || strcmp (*familyname, afmfile->familyname))
    {
      *familyname = afmfile->familyname;
      if (*familyname && fprintf (fp, "%s\n", *familyname) == EOF)
	return fault;
    }
    if (putc ('\t', fp) == EOF)
      return fault;
    if (writefileinfo (&afmfile->file, global, fp))
      return fault;
    if (fprintf (fp, "%s(%s)%s/%g\n", afmfile->fontname, strversion (fontversion, afmfile->version), strfontweight (fontweight, afmfile->weight), afmfile->italicangle) == EOF)
      return fault;
  }
  return ok;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int writefontfile (const t_fontfile *fontfile, va_list va)
{
  if (fontfile->file.stat == va_arg (va, t_filestat))
  {
    bool	global = va_arg (va, bool);
    FILE	*fp = va_arg (va, FILE *);
    t_buf	fontversion;

    if (writefileinfo (&fontfile->file, global, fp))
      return fault;
    if (fprintf (fp, "%s(%s)\n", fontfile->fontname, strversion (fontversion, fontfile->version)) == EOF)
      return fault;
  }
  return ok;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
/*
 *	infos traversieren und feststellen, ob sie gltig sind.
 *	wenn gecheckt wird, hat der status FS_valid zu sein, sonst FS_fromcache
 */
static int end (string dirname, va_list va)
{
  const t_cachespec	*cache = va_arg (va, const t_cachespec *);
  int			checkwhat = va_arg (va, int);
  t_dirinfos		*dirinfos = va_arg (va, t_dirinfos *);
  int			rc = ok;

  assert (dirinfos->afmfiles);
  assert (dirinfos->fontfiles);

/*
  avlamap (dirinfos->afmfiles, printafmfile, (void *)false);
  avlamap (dirinfos->fontfiles, printfontfile, (void *)false);
*/

  if (checkwhat & CHECK_afm)
    avlvamap (dirinfos->afmfiles, storeafmfile, cache->flags);
  if (checkwhat & CHECK_font)
    avlvamap (dirinfos->fontfiles, storefontfile, cache->flags);

  if (cache->flags & CACHE_use)
  {
    if (cache->flags & CACHE_update)
    {
      if (dirinfos->flags & DI_rewrite)
      {
	if (dirinfos->flags & DI_readonly)
	  say (IMP_warning, "can't update %s: read-only\n", dirinfos->cachefileinfo.fullname);
	else
	{
	  t_fnbuf	fnbuf;
	  FILE		*fp;

	  if (dirinfos->flags & DI_create )/* || CACHE_rebuild (cache->flags))*/ /* cache file hadn't been opened in begin() */
	  {
	    if (!(fp = pfopen (dirname, cache->filename, nofilter, O_lock|O_rw_wr, fnbuf, &dirinfos->cachefileinfo)))
	    {
	      say (IMP_warning, "can't open %s for writing.\n", dirinfos->cachefileinfo.fullname);
	      rc = fault;
	    }
	  }
	  else
	  {
	    if (!(fp = pfreopen (&dirinfos->cachefileinfo, O_rw_wr)))
	      ierror ("can't reopen %s for writing.\n", dirinfos->cachefileinfo.fullname);
	  }

	  if (!rc)
	  {
	    t_buf	magicbuf;
	    bool	corrupt = false;

	    say (IMP_entertain, "rewriting %s.\n", dirinfos->cachefileinfo.fullname);
	    if (fputs (magic (magicbuf, cache->flags & CACHE_global), fp) == EOF)
	      corrupt = true;
	    elif (cache->flags & CACHE_global)
	    {
	      if (writepath (afmdirs, fp) || writepath (fontdirs, fp))
		corrupt = true;
	    }
	    if (!corrupt)
	    {
	      string	familyname = NULL;

	      dirinfos->afmfiles = avlresort (dirinfos->afmfiles, afmfilepdupcmpfamilyname); assert (dirinfos->afmfiles);
	      corrupt = avlvamap (dirinfos->afmfiles, writeafmfile, (cache->flags & (CACHE_check|CACHE_update) && checkwhat & CHECK_afm) ? FS_valid : FS_fromcache, cache->flags & CACHE_global, fp, &familyname)
		|| fprintf (fp, "%c\n", ISO646_GS) == EOF;
	    }
	    if (!corrupt)
	    {
	      dirinfos->fontfiles = avlresort (dirinfos->fontfiles, fontfilepcmp); assert (dirinfos->fontfiles);
	      corrupt = avlvamap (dirinfos->fontfiles, writefontfile, (cache->flags & (CACHE_check|CACHE_update) && checkwhat & CHECK_font) ? FS_valid : FS_fromcache, cache->flags & CACHE_global, fp)
		|| fprintf (fp, "%c\n", ISO646_GS) == EOF;
	    }
	    if (corrupt)
	    {
	      say (IMP_warning, "write error on %s.  ", dirinfos->cachefileinfo.fullname);
	      say (IMP_warning, unlink (dirinfos->cachefileinfo.fullname) ? "can't remove %s.\n" : "removed.\n", dirinfos->cachefileinfo.fullname);
	      rc = fault;
	    }
	  }
	}
      }
      else
	say (IMP_entertain, "cache file \"%s\" is up to date.\n", dirinfos->cachefileinfo.fullname);
      freefileinfo (&dirinfos->cachefileinfo);
    }
  }

  dirinfos->afmfiles = avldestroy (dirinfos->afmfiles);
  dirinfos->fontfiles = avldestroy (dirinfos->fontfiles);
  return rc;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int scanafmfile (FILE *fp, t_fileinfo *fileinfo, va_list va)
{
  const t_cachespec	*cache = va_arg (va, const t_cachespec *);
  int			checkwhat = va_arg (va, int);
  t_dirinfos		*dirinfos = va_arg (va, t_dirinfos *);
  t_fnbuf		filename;
  t_afmfile		afmisearch, *afmfile;
  bool			insert;
  bool			scan;
  struct stat		statbuf;

#if YYDEBUG
  yydebug = true;
#endif

  cache = cache;
  assert (checkwhat == CHECK_afm);

  (void)fi_filename (filename, fileinfo);

  clearafmfile (&afmisearch);
  afmisearch.file = *fileinfo;
  insert = true;
  afmfile = avlinsrch (dirinfos->afmfiles, &insert, &afmisearch); assert (afmfile);
  if (insert)
  {
    if (cache->flags & CACHE_use && !CACHE_rebuild (cache->flags))
      say (IMP_entertain, "info for %s is not in cache.\n", filename);
    afmfile->file.fp = NULL;
    savefileinfo (&afmfile->file);
    scan = true;
  }
  else
  {
assert (cache->flags & CACHE_checkfiles);
    if (cache->flags & CACHE_checkfiles)
    {
      if (stat (filename, &statbuf))
      {
	say (IMP_error, "can't stat file \"%s\".\n", filename);
	scan = false;
	afmfile->file.stat = FS_void;
	dirinfos->flags |= DI_rewrite;
      }
      elif (checkinode (&statbuf, &afmfile->file))
      {
	say (IMP_entertain, "file %s has been touched.\n", filename);
	scan = true;
	dirinfos->flags |= DI_rewrite;
      }
      else
      {
	say (IMP_entertain2, "getting info about /%s from cache. (checked)\n", afmfile->fontname);
	scan = false;
	afmfile->file.stat = FS_valid;
      }
    }
    else
    {
      say (IMP_entertain2, "getting info about /%s from cache. (unchecked)\n", afmfile->fontname);
      scan = false;
    }
  }
  if (scan)
  {
    int		rc;

    {
      int len = strlen (filename);
      say (IMP_entertain, "scanning %s%-50s\r", len > 50 ? "...":"", len > 50 ? filename+len-50 : filename);
    }
    if (cache->flags & CACHE_update && !(!insert && cache->flags & CACHE_checkfiles) && stat (filename, &statbuf))
    {
      say (IMP_warning, "can't stat already open file \"%s\"!", filename);
      rc = fault;
    }
    else
      rc = (insert ? scanafm : rescanafm)(fp, afmfile, true);
    if (!rc && !afmfile->fontname)
    {
      say (IMP_warning, "file %s: missing FontName!\n", filename);
      rc = fault;
    }
    if (rc)
      afmfile->file.stat = FS_void;
    else
    {
      if (!afmfile->familyname)
	say (IMP_entertain, "file %s: missing FamilyName!\n", filename);
      if (afmfile->version == VERSION_unspecified)
      {
	say (IMP_warning, "file %s: missing Version!\n", filename);
	afmfile->version = VERSION_unknown;
      }
      if (afmfile->weight == FONTWEIGHT_unknown)
	say (IMP_entertain, "file %s: unknown Weight!\n", filename);

      if (cache->flags & CACHE_update)
      {
	say (IMP_entertain2, "inserting info for %s into cache.\n", filename);

	afmfile->file.mtime = statbuf.st_mtime;
	dirinfos->flags |= DI_rewrite;
      }
    }
  }
  return ok;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int getfontversion (FILE *fp, string fn, t_fontfile *fontfile)
{
  t_buf		line, fontname;
  int		maj, min;

  if (!fgets (line, sizeof line, fp))
  {
    say (IMP_error, "file %s: can't read first line.\n", fn);
    return fault;
  }
  if (sscanf (line, "%%!PS-AdobeFont-%*g: %s %d.%d", fontname, &maj, &min) == 3
    || sscanf (line, "%%!FontType1-%*g: %s %d.%d", fontname, &maj, &min) == 3
    || sscanf (line, "%%!PS-Adobe-3.0 Resource-Font: %s %d.%d", fontname, &maj, &min) == 3)
    fontfile->version = VERSION (maj, min);
  else
  {
    int	lino = 1;
    bool bingo = false;

    do
      if (bingo = sscanf (line, "/FontName /%s", fontname) == 1)
	break;
    while (lino < 50 && fgets (line, sizeof line, fp));
    if (!bingo)
    {
      say (IMP_error, "file %s: can't find any fontname.\n", fn);
      return fault;
    }
    say (IMP_warning, "file %s: can't identify fontversion. (fontname is expected to be %s).\n", fn, fontname);
    fontfile->version = VERSION_unknown;
  }
  if (!fontfile->fontname)
    fontfile->fontname = strsave (fontname);
  elif (strcmp (fontname, fontfile->fontname))
  {
    say (IMP_error, "file %s: fontname doesn't match (%s vs. %s).\n", fn, fontfile->fontname, fontname);
    return fault;
  }
  return ok; /* assume that the fontname is right */
}

static int scanfontfile (FILE *fp, t_fileinfo *fileinfo, va_list va)
{
  const t_cachespec	*cache = va_arg (va, const t_cachespec *);
  int			checkwhat = va_arg (va, int);
  t_dirinfos		*dirinfos = va_arg (va, t_dirinfos *);
  t_fnbuf		filename;
  t_fontfile		fontisearch, *fontfile;
  bool			insert;
  bool			scan;

  cache = cache;
  assert (checkwhat == CHECK_font);

  (void)fi_filename (filename, fileinfo);

  clearfontfile (&fontisearch);
  fontisearch.file = *fileinfo;
  insert = true;
  fontfile = avlinsrch (dirinfos->fontfiles, &insert, &fontisearch); assert (fontfile);
  if (insert)
  {
    if (cache->flags & CACHE_use && !CACHE_rebuild (cache->flags))
      say (IMP_entertain, "info for %s is not in cache.\n", filename);
    fontfile->file.fp = NULL;
    savefileinfo (&fontfile->file);
    scan = true;
  }
  else
  {
assert (cache->flags & CACHE_checkfiles);
    if (cache->flags & CACHE_checkfiles)
    {
      struct stat	statbuf;

      if (stat (filename, &statbuf))
      {
	say (IMP_error, "can't stat file \"%s\".\n", filename);
	scan = false;
	fontfile->file.stat = FS_void;
	dirinfos->flags |= DI_rewrite;
      }
      elif (checkinode (&statbuf, &fontfile->file))
      {
	say (IMP_entertain, "file %s has been touched.\n", filename);
	scan = true;
	dirinfos->flags |= DI_rewrite;
      }
      else
      {
	say (IMP_entertain2, "getting info about /%s from cache. (checked)\n", fontfile->fontname);
	scan = false;
	fontfile->file.stat = FS_valid;
      }
    }
    else
    {
      say (IMP_entertain2, "getting info about /%s from cache. (unchecked)\n", fontfile->fontname);
      scan = false;
    }
  }
  if (scan)
  {
    {
      int len = strlen (filename);
      say (IMP_entertain, "scanning %s%-50s\r", len > 50 ? "...":"", len > 50 ? filename+len-50 : filename);
    }
    if (getfontversion (fp, filename, fontfile))
      fontfile->file.stat = FS_void;
    elif (cache->flags & CACHE_update)
    {
      struct stat	statbuf;

      say (IMP_entertain2, "inserting info for %s into cache.\n", filename);

      if (stat (filename, &statbuf))
	ierror ("can't stat already open file \"%s\"!", filename);

      fontfile->file.mtime = statbuf.st_mtime;
      fontfile->file.stat = FS_valid;
      dirinfos->flags |= DI_rewrite;
    }
  }
  return ok;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int cv (string dirname, int (*f)(string, va_list), ...)
{
  va_list	va;
  int		rc;

  va_start (va, f);
  rc = (*f)(dirname, va);
  va_end (va);
  return rc;
}

static int scanfiles (t_cachespec *cache)
{
  int		rc;
  t_fnbuf	fnbuf;
  t_dirinfos	dirinfos;

  void		*before, *everyafm, *everyfont, *after;

  {
    t_afmfile	internal;
    va_list	dummy;

    clearafmfile (&internal);
    psinitafm (&internal);
    storeafmfile (&internal, dummy);
  }

#ifdef DEBUG
  say (IMP_debug, "cache flags:");
  if (cache->flags & CACHE_global)
    say (IMP_debug, " global");
  if (cache->flags & CACHE_use)
    say (IMP_debug, " use");
  if (cache->flags & CACHE_checkfiles)
    say (IMP_debug, " ckfiles");
  if (cache->flags & CACHE_checkdir)
    say (IMP_debug, " ckdir");
  if (cache->flags & CACHE_update)
    say (IMP_debug, " update");
  say (IMP_debug, "\ncache filename: \"%s\"\n", cache->filename);
#endif

  if (cache->flags & (CACHE_check|CACHE_update))
    cache->flags |= CACHE_use;
  if (cache->flags & CACHE_global)
  {
    if (*cache->filename != '/')
      say (IMP_warning, "name of global cache file should be absolute! (is \"%s\")\n", cache->filename);
  }
  else
  {
    if (strchr (cache->filename, '/'))
      error ("name of local cache file mustn't contain `/': \"%s\"\n", cache->filename);
  }
  if (cache->flags & CACHE_checkdir && !(cache->flags & CACHE_checkfiles))
  {
    say (IMP_warning, "cache: `checkdir' implies `checkfiles'.\n");
    cache->flags |= CACHE_checkfiles;
  }
  if (cache->flags & CACHE_use)
  {
    before = cache->flags & CACHE_global ? NULL : begin;
    if (cache->flags & CACHE_checkdir || CACHE_rebuild (cache->flags))
    {
      everyafm = scanafmfile;
      everyfont = scanfontfile;
    }
    else
      everyafm = everyfont = NULL;
    after = cache->flags & CACHE_global ? NULL : end;
  }
  else
  {
    before = begin;
    everyafm = scanafmfile;
    everyfont = scanfontfile;
    after = end;
  }
  if (cache->flags & CACHE_global && cache->flags & CACHE_use && (rc = cv (NULL, begin, cache, CHECK_all, &dirinfos)))
    return rc;
  if (rc = findfiles (afmdirs, afmsuffices, O_rd, fnbuf, before, everyafm, after, cache, CHECK_afm, &dirinfos))
    return rc;
  if (rc = findfiles (fontdirs, fontsuffices, O_rd, fnbuf, before, everyfont, after, cache, CHECK_font, &dirinfos))
    return rc;
  if (cache->flags & CACHE_global && cache->flags & CACHE_use)
    rc = cv (NULL, end, cache, CHECK_all, &dirinfos);
  return rc;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
#ifdef DEBUG
static int printfont (const t_font *font, bool verbose)
{
  say (IMP_debug, "FontName: %s\n", font->fontname);
  return avlamap (font->afms, printafmfile, (void *)verbose);
  return avlamap (font->fonts, printfontfile, (void *)verbose);
}

static int printfontp (const t_font **font, bool verbose)
{
  return printfont (*font, verbose);
}

static int printfontfamily (t_fontfamily *fontfamily, bool verbose)
{
  say (IMP_debug, "FontFamily: %s\n", fontfamily->familyname);
  return avlamap (fontfamily->fonts, printfontp, (void *)verbose);
}
#endif

static int fontfamilypcmp (const t_fontfamily *l, const t_fontfamily *r)
{
assert (l->familyname);
assert (r->familyname);
  return strcmp (l->familyname, r->familyname);
}

static void freefontfamily (t_fontfamily *fontfamily)
{
  fontfamily->fonts = avldestroy (fontfamily->fonts);
}

int init_fonts (t_cachespec *cache)
{
  int	rc;

  fonts = avlcreate (sizeof (t_font), fontpcmp, freefont);
  fontfamilies = avlcreate (sizeof (t_fontfamily), fontfamilypcmp, freefontfamily);
  rc = scanfiles (cache);
#ifdef DEBUG
  if (!rc && VERBp (IMP_debug))
  {
    fprintf (stderr, "=== Fonts:\n");
    avlamap (fonts, printfont, (void *)false);
    fprintf (stderr, "=== FontFamilies:\n");
    avlamap (fontfamilies, printfontfamily, (void *)false);
    fprintf (stderr, "===\n");
  }
#endif
  return rc;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
t_font *getfontbyname (cstring fontname)
{
  t_font	fsearch;

  if (!fontname)
    return NULL;
  fsearch.fontname = (string)fontname;
  return avlsearch (fonts, &fsearch);
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int afm_findrendition (const t_afmfile *afmfile, va_list va)
{
  const t_fontrendition	*rendition = va_arg (va, const t_fontrendition *);

  if (!fontweightcmp (afmfile->weight, rendition->weight) && (rendition->italic == -1 || ISITALIC (afmfile->italicangle) == rendition->italic))
  {
    t_cnames	**names = va_arg (va, t_cnames **);

    if (!*names)
    {
      *names = getmem (sizeof (t_cnames));
      (*names)->count = 0;
    }
    else
      *names = regetmem (*names, sizeof (t_cnames)+(*names)->count*sizeof (cstring));

    (*names)->names[(*names)->count++] = afmfile->fontname;
  }
  return ok;
}

static int font_findrendition (const t_font **font, va_list va)
{
  return avlamap ((*font)->afms, afm_findrendition, va);
}

t_cnames *getfontsfromfamily (string familyname, const t_fontrendition *fontrendition)
{
  t_fontfamily		ffsearch, *fontfamily;
  t_cnames		*names = NULL;
  t_fontrendition	fallback;

assert (familyname);
  ffsearch.familyname = familyname;
  if (!(fontfamily = avlsearch (fontfamilies, &ffsearch)))
    return NULL;

  avlvamap (fontfamily->fonts, font_findrendition, fontrendition, &names);
  if (names)
    return names;

  if (fontrendition->weight == FONTWEIGHT_extrabold)
  {
    fallback.weight = FONTWEIGHT_bold;
    fallback.italic = fontrendition->italic;
    avlvamap (fontfamily->fonts, font_findrendition, &fallback, &names);
    if (names)
      return names;
  }

  fallback.weight = fontrendition->weight;
  fallback.italic = -1;
  avlvamap (fontfamily->fonts, font_findrendition, &fallback, &names);
  if (names)
    return names;

  fallback.weight = FONTWEIGHT_unspecified;
  fallback.italic = fontrendition->italic;
  avlvamap (fontfamily->fonts, font_findrendition, &fallback, &names);
  if (names)
    return names;

  fallback.weight = FONTWEIGHT_unspecified;
  fallback.italic = -1;
  avlvamap (fontfamily->fonts, font_findrendition, &fallback, &names);
  if (names)
    return names;

  return NULL;
}

t_cnames *getfontsfromfamily2 (string familyname, t_fontweight weight, bool italic)
{
  t_fontrendition	rendition = { weight, italic };

  return getfontsfromfamily (familyname, &rendition);
}
