/*
 * program: psfilt
 * file: afm.c
 *
 * Copyright  1992 1993 1994 Robert Joop
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: afm.c,v $
 * Revision 1.5  1994/07/09  16:43:27  rj
 * a lot of const's removed and added
 *
 * Revision 1.4  1994/07/09  15:34:07  rj
 * stuff for configuration via GNU autoconf added
 *
 * Revision 1.3  1994/01/09  23:45:33  rj
 * PPD parser fr version 4 PPD files total umgeschrieben.
 * partieller support fr perl.
 *
 * Revision 1.2  1994/01/04  11:56:16  rj
 * minor adjustments
 *
 * Revision 1.1  1993/12/31  20:56:40  rj
 * Initial revision
 *
 */

static const char RCSId[] = "$Id: afm.c,v 1.5 1994/07/09 16:43:27 rj Exp $";

#include "psfilt.h"
#include "afm.h"
#include "str.h"
#include "error.h"
#include "verbose.h"

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
int fontweightcmp (t_fontweight l, t_fontweight r)
{
  if (l == FONTWEIGHT_unspecified || r == FONTWEIGHT_unspecified)
    return 0;
  if (l == FONTWEIGHT_unknown)
    return -1;
  if (r == FONTWEIGHT_unknown)
    return 1;
  return intcmp (l, r);
}

typedef struct
{
  cstring	name;
  t_fontweight	weight;
} t_weightmap;

static const t_weightmap weightmap[] =
{
  "Light", FONTWEIGHT_light,
  "Normal", FONTWEIGHT_normal,
  "Medium", FONTWEIGHT_normal,
  "Roman", FONTWEIGHT_normal,
  "Standard", FONTWEIGHT_normal,
  "Bold", FONTWEIGHT_bold,
  "Extra Bold", FONTWEIGHT_extrabold,
  "Book", FONTWEIGHT_normal,
  "Demi", FONTWEIGHT_bold,
};

t_fontweight fontweightbyname (cstring weightname)
{
  int	i;

  for (i=0; i<ARRAYDIM (weightmap); i++)
    if (!strcasecmp (weightmap[i].name, weightname))
      return weightmap[i].weight;
  return FONTWEIGHT_unknown;
}

string strfontweight (string buf, t_fontweight fontweight)
{
  switch (fontweight)
  {
    case FONTWEIGHT_unspecified:
	strcpy (buf, "<unspecified>");
	break;
    case FONTWEIGHT_unknown:
	strcpy (buf, "<unknown>");
	break;
    case FONTWEIGHT_light:
	strcpy (buf, "Light");
	break;
    case FONTWEIGHT_normal:
	strcpy (buf, "Normal");
	break;
    case FONTWEIGHT_bold:
	strcpy (buf, "Bold");
	break;
    case FONTWEIGHT_extrabold:
	strcpy (buf, "Extra Bold");
	break;
    default:
	ierror ("strfontweight");
  }
  return buf;
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static void clearbbox (t_bbox *bbox)
{
  bbox->llx = bbox->lly = bbox->urx = bbox->ury = 0.0;
}

#ifdef DEBUG
static void printbbox (const t_bbox *bbox)
{
  printf ("B (%g, %g, %g, %g);", bbox->llx, bbox->lly, bbox->urx, bbox->ury);
}
#endif

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
#ifdef DEBUG
static int printligature (const t_ligature *lig)
{
  return printf (" L (%s, %s);", lig->first, lig->second) == EOF ? fault : ok;
}
#endif

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int charmetriccmp (const t_charmetric *ml0, const t_charmetric *ml1)
{
  return strcmp (ml0->name, ml1->name);
}

void clearcharmetric (t_charmetric *cm)
{
  cm->code = -1;
  cm->width.x = -1.0;
  cm->width.y = -1.0;
  cm->name = NULL;
  clearbbox (&cm->bbox);
  cm->ligatures = NULL;
}

static void freecharmetric (t_charmetric *cm)
{
  cm->name = strfree (cm->name);
  cm->ligatures = lidestroy (cm->ligatures);
}

#ifdef DEBUG
/* output of metric line informations in symbolic form on stdout	*/
static void printcharmetric (const t_charmetric *cm)
{
  printf ("Name: %s; ", cm->name);
  printf ("Code: %d; ", cm->code);
  printf ("WidthVector: (%g, %g); ", cm->width.x, cm->width.y);
  printbbox (&cm->bbox);
  limap (cm->ligatures, printligature);
  printf ("\n");
}

static int charmetricprint (const t_charmetric *ml, int *count)
{
  printcharmetric (ml);
  (*count)++;
  return ok;
}
#endif

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
static int kernpaircmp (const t_kernpair *kpl, const t_kernpair *kpr)
{
  int	cmp;

  return (cmp = strcmp (kpl->left, kpr->left)) ? cmp : strcmp (kpl->right, kpr->right);
}

#ifdef DEBUG
static int kernpairprint (const t_kernpair *kp)
{
  return printf ("KP %s %s %g %g\n", kp->left, kp->right, kp->width.x, kp->width.y) == EOF ? fault : ok;
}
#endif

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
void clearafmfile (t_afmfile *afm)
{
  int	code;

  clearfileinfo (&afm->file);
  afm->fontname = NULL;
  afm->fullname = NULL;
  afm->familyname = NULL;
  afm->weight = FONTWEIGHT_unknown;
  afm->italicangle = 0.0;
  afm->isfixedpitch = -1;
  clearbbox (&afm->bbox);
  afm->underlineposition = 0.0;
  afm->underlinethickness = 0.0;
  afm->version = VERSION_unspecified;
  afm->notice = NULL;
  afm->encodingscheme = NULL;
  afm->capheight = 0.0;
  afm->Xheight = 0.0;
  afm->ascender = 0.0;
  afm->descender = 0.0;
  afm->metrics = avlcreate (sizeof (t_charmetric), charmetriccmp, freecharmetric);
  afm->kernpairs = avlcreate (sizeof (t_kernpair), kernpaircmp, NULL);
  for (code=256; code--; )
    afm->encoding[code] = NULL;
}

#ifdef DEBUG
/* output of t_afmfile informations in symbolic form on stdout	*/
int printafmfile (const t_afmfile *afm, bool verbose)
{
  if (verbose)
  {
    t_buf	buf;
    int	nchar = 0, c;

    printf ("FontName: %s; ", afm->fontname ? afm->fontname : "<none>");
    printf ("FullName: %s; ", afm->fullname ? afm->fullname : "<none>");
    printf ("FamilyName: %s\n", afm->familyname ? afm->familyname : "<none>");
    printf ("Weight: %s; ", strfontweight (buf, afm->weight));
    printf ("ItalicAngle: %g\n", afm->italicangle);

    printf ("IsFixedPitch: %s; ", afm->isfixedpitch ? "True" : "False");

    printbbox (&afm->bbox);
    printf ("\n");

    printf ("UnderlinePosition: %g; ", afm->underlineposition);
    printf ("UnderlineThickness: %g\n", afm->underlinethickness);
    printf ("Version: %s\n", strversion (buf, afm->version));
    printf ("Notice: %s\n", afm->notice ? afm->notice : "<none>");
    printf ("EncodingScheme: %s\n", afm->encodingscheme ? afm->encodingscheme : "<none>");
    printf ("CapHeight: %g; ", afm->capheight);
    printf ("XHeight: %g; ", afm->Xheight);
    printf ("Ascender: %g; ", afm->ascender);
    printf ("Descender: %g\n", afm->descender);

    avlamap (afm->metrics, charmetricprint, &nchar);
    avlmap (afm->kernpairs, kernpairprint);
    printf ("total of %d chars\n", nchar);

    printf ("/Encoding: [ ");
    for (c=0; c<256; c++)
      printf ("/%s ", afm->encoding[c] ? afm->encoding[c] : ".notdef");
    printf ("]\n");
    return feof (stdout) ? fault : ok;
  }
  else
  {
    t_buf	fontversion, fontweight;

    say (IMP_debug, "  FontName: %s; FontFamily: %s; Version: %s; Weight: %s; ItalicAngle: %g (italic=%s)\n", afm->fontname, afm->familyname, strversion (fontversion, afm->version), strfontweight (fontweight, afm->weight), afm->italicangle, ISITALIC (afm->italicangle) ? "true" : "false");
    return feof (stderr) ? fault : ok;
  }
}
#endif

int afmfilepcmpversion (t_afmfile *l, t_afmfile *r)
{
  return versioncmp (l->version, r->version);
}

int afmfilepdupcmpfamilyname (const t_afmfile *l, const t_afmfile *r)
{
  int rc;
  if (rc = str0cmp (l->familyname, r->familyname))
    return rc;
  if (rc = strcmp (l->fontname, r->fontname))
    return rc;
  return -1; /* allow dups */
}

#if 0
int afmfilepcmpfamilyname (const t_afmfile *l, const t_afmfile *r)
{
  return str0cmp (l->familyname, r->familyname);
}
#endif

int afmfilepcmpfilename (const t_afmfile *l, const t_afmfile *r)
{
  return fileinfopcmpfilename (&l->file, &r->file);
}

void freeafmfile (t_afmfile *afm)
{
#define STRFREE(s)	(s) = strfree (s);
  int	code;

  freefileinfo (&afm->file);
  STRFREE (afm->fontname);
  STRFREE (afm->fullname);
  STRFREE (afm->familyname);
  STRFREE (afm->notice);
  STRFREE (afm->encodingscheme);
  afm->metrics = avldestroy (afm->metrics);
  afm->kernpairs = avldestroy (afm->kernpairs);
  for (code=256; code--; )
    afm->encoding[code] = NULL; /* names were shared */
#undef STRFREE
}

/*\[sep]--------------------------------------------------------------------------------------------------------------------------*/
/* get the metric informations for specified glyph */
const t_charmetric *getcharmetric (const t_afmfile *afm, cstring charname)
{
  t_charmetric	srch;

  srch.name = (string)charname;
  return avlsearch (afm->metrics, &srch);
}

const t_kernpair *getkernpair (const t_afmfile *afm, string left, string right)
{
  t_kernpair	srch;

  srch.left = left;
  srch.right = right;
  return avlsearch (afm->kernpairs, &srch);
}
