/* Ld2.c
 * NS32000 linker
 *
 * Code for second pass.
 */
#include <fcntl.h>
#ifdef MSDOS
#include <sys\types.h>
#include <sys\stat.h>
#endif
#include "ld.h"

int out;                       /* output file */
static long text_pos,          /* bytes written so far to text */
            data_pos,          /* bytes written so far to data */
            tr_pos,            /* bytes written so far to text reloc */
            dr_pos;            /* bytes written so far to data reloc */

/* Makes a second pass over input files, writing the output file.
 */
pass2()
{
# ifdef DEBUG
  if (debug) {
    print_symtab();
    print_segs();
  }
# endif
  open_outfile();
  write_head();
  files_loop();
  write_sym();
  write_str();
  close (out);
}

/* Open the output file.
 */
open_outfile()
{
  unlink (outname);
#ifdef MSDOS
  if (-1 == (out = open (outname, O_WRONLY | O_CREAT | O_BINARY,
    S_IREAD | S_IWRITE)))
#else
  if (-1 == (out = open (outname, O_WRONLY | O_CREAT, reloc? 0666: 0777)))
#endif
    error ("!could not open %s", outname);
}

/* Write the a.out file header.
 */
write_head()
{
  unsigned long t;

  CL2M (head.a_magic, t);
  CL2M (head.a_text, t);
  CL2M (head.a_data, t);
  CL2M (head.a_bss, t);
  CL2M (head.a_trsize, t);
  CL2M (head.a_drsize, t);
  CL2M (head.a_sym, t);
  CL2M (head.a_str, t);
  CL2M (head.a_entry, t);
  CL2M (head.a_tstart, t);
  CL2M (head.a_dstart, t);
  mywrite ((char *)&head, (unsigned) sizeof head);
  CM2L (head.a_magic);
  CM2L (head.a_text);
  CM2L (head.a_data);
  CM2L (head.a_bss);
  CM2L (head.a_trsize);
  CM2L (head.a_drsize);
  CM2L (head.a_sym);
  CM2L (head.a_str);
  CM2L (head.a_entry);
}

/* This is the major portion of pass 2.  Writes the text, data and relocation
 * tables.
 */
files_loop()
{
  fileptr fp;
  char *namep;

  text_pos = 0;
  data_pos = 0;
  tr_pos = 0;
  dr_pos = 0;
  namep = NULL;
  for (fp = filehead; fp != NULL; fp = fp->link) {
    if (fp->name != namep) {
      namep = fp->name;
      fclose (infile);
      myfopen (namep);
    }
    write_file (fp);
  }
}

/* Writes text, data, and relocation tables for a file.
 */
write_file (fp)
fileptr fp;
{
  struct r_info *rtab;
  char *code;

  code = getseg ((int)fp->h.e.a_text,            /* read text segment */
    TEXTPOS(fp->h.e) + fp->offset);
  rtab = (struct r_info *) getseg                /* read text reloc table */
    ((int)fp->h.e.a_trsize, TRPOS(fp->h.e) + fp->offset);
  reloc_seg (code, rtab, R_TEXT, fp);
  mylseek ((long)(TEXTPOS (head) + text_pos));   /* write text segment */
  mywrite (code, (unsigned)fp->h.e.a_text);
  text_pos += fp->h.e.a_text;
  if (reloc) {                                   /* write text reloc table */
    mylseek ((long)(TRPOS (head) + tr_pos));
    mywrite ((char *)rtab, (unsigned)(fp->h.e.a_trsize));
    tr_pos += fp->h.e.a_trsize;
  }
  free ((char *)rtab);
  free (code);

  code = getseg ((int)fp->h.e.a_data,            /* read data segment */
    DATAPOS(fp->h.e) + fp->offset);
  rtab = (struct r_info *) getseg                /* read data reloc table */
    ((int)fp->h.e.a_drsize, DRPOS(fp->h.e) + fp->offset);
  reloc_seg (code, rtab, R_DATA, fp);
  mylseek ((long)(DATAPOS (head) + data_pos));   /* write data segment */
  mywrite (code, (unsigned)(fp->h.e.a_data));
  data_pos += fp->h.e.a_data;
  if (reloc) {                                   /* write data reloc table */
    mylseek ((long)(DRPOS (head) + dr_pos));
    mywrite ((char *)rtab, (unsigned)(fp->h.e.a_drsize));
    dr_pos += fp->h.e.a_drsize;
  }
  free ((char *)rtab);
  free (code);
}

/* This is the heart of the second pass.  Relocates a segment.  Updates
 * the relocation table if producing another relocatable (reloc=TRUE).
 */
reloc_seg (code, rtab, seg, fp)
char *code;             /* code to modify */
struct r_info *rtab;    /* relocation info */
int seg;                /* this segment: R_TEXT or R_DATA */
fileptr fp;             /* data for this file */
{
  int r_sym, numrec;
  struct r_info *rp;
  unsigned long val;
  unsigned char *cp;

  numrec = ((seg == R_TEXT)? fp->h.e.a_trsize: fp->h.e.a_drsize) /
    sizeof (struct r_info);
  for (rp = rtab; rp < rtab + numrec; ++rp) {
    r_sym = WM2S (rp->r_sym);
    cp = (unsigned char *)               /* get pointer to location */
      (code + WM2L (rp->r_adr));
    val = (r_sym & (R_NSCDISP | R_GENIMM))?
      ((long)(*cp)<<24 |                 /* read disp or imm (msb first) */
      (long)(*(cp+1))<<16 |
      *(cp+2)<<8 | *(cp+3)):
      WM2L (*cp);                        /* read normal lsb first */
    if (r_sym & R_NSCDISP)               /* remove disp size field */
      val &= 0x3fffffff;
    val += seg_reloc (seg, fp, r_sym);   /* correct for segment reloc */
    if (R_SPECIAL != (r_sym & R_SPECIAL))   /* link to symbol */
      val += link_sym (fp, &r_sym);
    if (r_sym & (R_NSCDISP | R_GENIMM)) {  /* write back value */
      *cp++ =                            /* disp and imm cases: msb first */
        (r_sym & R_NSCDISP? 0xc0: 0) |   /* or in disp size */
        (val >> 24);
      *cp++ = (val >> 16) & 0xff;
      *cp++ = (val >> 8) & 0xff;
      *cp = val & 0xff;                  /* lsb */
    } else WL2M (*cp, val);
    if (reloc) {                         /* fixup reloc table */
      WS2M (rp->r_sym, r_sym);
      val = WM2L (rp->r_adr) +
        ((seg == R_TEXT)? fp->new_text:
        fp->new_data - head.a_text);
      WL2M (rp->r_adr, val);
    }
  }
}

/* Compute correction for segments having moved.
 */
long
seg_reloc (seg, fp, r_sym)
int seg, r_sym;
fileptr fp;
{
  int resolved, rel;

  rel = (R_PCREL == (r_sym & R_PCREL));
  resolved = (R_SPECIAL == (r_sym & R_SPECIAL));
  if (!rel) 
    if (!resolved) return 0;
    else switch (r_sym & R_SEG) {	/* adjust using target's segment */
      case R_TEXT: return fp->new_text;
      case R_DATA: return fp->new_data - fp->h.e.a_text;
      case R_BSS:  return fp->new_bss - fp->h.e.a_data - fp->h.e.a_text;
    }
  /* must be pc relative */
  else if (!resolved) return (seg == R_TEXT)? -fp->new_text:
    -fp->new_data + fp->h.e.a_text;
  /* must be resolved and pc relative */
  else switch (seg << 2 | (r_sym & R_SEG)) {
    case R_TEXT << 2 | R_TEXT:
      return 0;
    case R_TEXT << 2 | R_DATA:
      return fp->new_data - fp->new_text - fp->h.e.a_text;
    case R_TEXT << 2 | R_BSS:
      return fp->new_bss - fp->new_text - fp->h.e.a_text - fp->h.e.a_data;
    case R_DATA << 2 | R_TEXT:
      return -fp->new_data + fp->new_text + fp->h.e.a_text;
    case R_DATA << 2 | R_DATA:
      return 0;
    case R_DATA << 2 | R_BSS:
      return fp->new_bss - fp->new_data - fp->h.e.a_data;
  }
}

/* Links a reference to the referenced symbol.  Updates the reloc table
 * r_sym field.
 */
long
link_sym (fp, r_sym)
fileptr fp;
int *r_sym;
{
  hashptr hp;
  nlptr np;

  np = fp->nlp + (*r_sym & R_SYM);
  if (np->n_type == T_UNDF) {
    hp = (hashptr)np->n_value;
    np = hp->nlp;
  }
  if (np->n_type == T_UNDF) {
    /* change r_sym to index in symbol new table */
    *r_sym = (*r_sym & R_FLAGS) | hp->index;
    return 0;
  } else {
    if (reloc) {
      /* change r_sym to special symbol to show it is resolved */
      if (np->n_type == T_TEXT)
        *r_sym = (*r_sym & R_FLAGS) | R_SPECIAL | R_TEXT;
      else if (np->n_type == T_DATA) 
        *r_sym = (*r_sym & R_FLAGS) | R_SPECIAL | R_DATA;
      else *r_sym = (*r_sym & R_FLAGS) | R_SPECIAL | R_BSS;
    }
    return np->n_value;
  }
}

/* Write out the symbol table.
 */
write_sym()
{
  mylseek ((long)SYMPOS (head));
  hashnav (write_nlist);
}

/* Passed a hashptr, write the symbol table entry to the output file.
 */
write_nlist (p)
hashptr p;
{
  unsigned long t;

  CL2M (p->nlp->n_value, t);
  CL2M (p->nlp->n_stroff, t);
  CS2M (p->nlp->n_type, t);
  mywrite ((char *)(p->nlp), (unsigned) sizeof (struct nlist));
}

/* Write the string table to the output file.
 */
write_str()
{
  struct strnode *p;

  for (p = strhead; p != NULL; p = p->link)
    mywrite (p->sbuf, (unsigned)(p->size));
}
  
/* Do a write, check for errors.
 */
mywrite (buf, size)
char *buf;
unsigned size;
{
  if (size == 0) return;
  if (size != write (out, buf, size))
    error ("!write to %s failed", outname);
}

/* Do an lseek, check for errors.
 */
mylseek (pos)
long pos;
{
  if (-1 == lseek (out, pos, 0))
    error ("!lseek to %s failed", outname);
}
    
#ifdef DEBUG 
/* Print an entry of the symbol table.
 */
print_sym (p)
hashptr p;
{
  char *typ;

  if (p->nlp->n_type == T_TEXT) typ = "TEXT";
  else if (p->nlp->n_type == T_DATA) typ = "DATA";
  else if (p->nlp->n_type == T_BSS) typ = "BSS";
  else if (p->nlp->n_type == T_UNDF) typ = "UNDF";

  if (p->nlp->n_type == T_UNDF)
    printf ("\t%s\t%s\n", typ, p->id);
  else printf ("%7lx\t%s\t%s\n", p->nlp->n_value, typ, p->id);
  print_refs (p);
}

print_refs(p)
hashptr p;
{
  nlptr nlp;
  fileptr fp;

  for (fp = filehead; fp != NULL; fp = fp->link)
    for (nlp = fp->nlp; nlp < fp->nlp + fp->h.e.a_sym/sizeof (struct nlist);
    ++nlp)
      if (nlp->n_type == T_UNDF && (hashptr)(nlp->n_value) == p)
        printf ("\t\treferenced by %s\n", fp->name);
      else if (p->nlp == nlp)
	printf ("\t\tdefined by %s\n", fp->name);
}

print_symtab()
{
  printf ("Symbols:\n");
  hashnav (print_sym);
}

print_segs()
{
  fileptr fp;

  printf ("\nNew segments:\n");
  for (fp = filehead; fp != NULL; fp = fp->link)
    printf ("  text=0x%-7lx data=0x%-7lx bss=0x%-7lx %s(0x%lx)\n",
    fp->new_text, fp->new_data, fp->new_bss, fp->name, fp->offset);
}
#endif
