/* conv.h
 *
 * Macros needed for portability between machines which place the
 * most significant byte of an integer in the lowest address and
 * machines which place the same byte in the highest address.
 * When running on the native machine, NATIVE may be defined,
 * in which case these macros emit minimal code.
 */
#define M0(x)           (*((unsigned char *)&(x)))
#define M1(x)           (*((unsigned char *)&(x)+1))
#define M2(x)           (*((unsigned char *)&(x)+2))
#define M3(x)           (*((unsigned char *)&(x)+3))
#define B0(x)           ((x) & 0xff)
#define B1(x)           ((x)>>8 & 0xff)
#define B2(x)           ((x)>>16 & 0xff)
#define B3(x)           ((x)>>24 & 0xff)

/* Explanation of the mneumonics:
 *   W = write: used to assign to a different memory location or to generate
 *       an rvalue (WM2x).
 *   C = convert: used to convert one memory location from one format to
 *       another.
 *   L = long
 *   S = short
 *   M = memory
 *   d = destination
 *   s = source
 *   t = temporary
 *
 * These macros assume x, d, and s are defined to be shorts or longs.
 */
#ifndef NATIVE
#  define WM2L(x)       (M0(x) | M1(x)<<8 | (long)M2(x)<<16 | (long)M3(x)<<24)
#  define WM2S(x)       (M0(x) | M1(x)<<8)
#  define CM2L(x)       ((x) = WM2L(x))
#  define CM2S(x)       ((x) = WM2S(x))
#  define WL2M(d,s)     {M0(d)=B0(s);M1(d)=B1(s);M2(d)=B2(s);M3(d)=B3(s);}
#  define WS2M(d,s)     {M0(d)=B0(s);M1(d)=B1(s);}
#  define CL2M(x,t)     {t=x;WL2M(x,t)}
#  define CS2M(x,t)     {t=x;WS2M(x,t)}
#else
#  define WM2L(x)       (x)
#  define WM2S(x)       (x)
#  define CM2L(x)
#  define CM2S(x)
#  define WL2M(d,s)     ((d)=(s))
#  define WS2M(d,s)     ((d)=(s))
#  define CL2M(x,t)     
#  define CS2M(x,t)     
#endif
