/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Supply global procedures for ORITOPS and ORIDRAW program sources
     First version dd.19940818   (Pascal)
     Converted to C   19950220
*/

#include "orilcmnd.h"
/*GenMake off */
#include "orilxxxx.h"
/*GenMake on */

/*extracth Local */

#define last_revision "19990911"
#define this_unit "OriLglbl"
#define prog_version "4.10"
#define mytracelevel 4
#define orifilelevel "3"
static boolean initialized=false;


/*extracth Functions */

FUNCTION void InitializeOriLGlbl(int *argc,char *argv[])
{
   if (initialized) return;
   initialized=true;
   InitString(ori_program_version,prog_version,20);
   InitializeOriLCmnd(argc,argv);
   InitializeOriLXxxx(argc,argv);
   CheckOriUpdate(last_revision);
   TRACE CheckMgSystem(this_unit,"InitializeOriLGlbl",true,mytracelevel,"Compiled at %s %s",__DATE__,__TIME__);
   DeleteFile(dumpfilename);
   penwidthdefault=0.03;
   penwidthfactor=1.0;
   penwidth=penwidthdefault;
   lastdiagram=0;
   again=false;
   editfile=false;
   finitobasta=false;
   fastshow=false;
   textlanguage='E';
   diagramtoskipto= -1;
   editstatus.linenumber=0;
   pidiv180=atan(1.0)/45.0;
   rmirror=0;
   imirror2=0;
   imirror2ps=0;
   imirror20ps=0;
   xyfactor=1;
   objectx=0;
   objecty=0;
   cosalfa=1;
   sinalfa=0;
   deltax=0;
   deltay=0;
   xyfactor=1;
   marginx=0;
   marginy=0;
   xscale=1;
   yscale=1;
}                                      /* InitializeOriLGlbl */


/*OriDoc Intro
ORIDRAW, ORITOPS and ORITOAIL are programs designed to make Origami diagrams.
.
They are not WYSIWYG (What You See Is What You Get) programs: both programs
read commands from a file and converts them to diagrams.
.
The drawing commands are especially devised for Origami, but you may use
them for other drawings too.
.
This program was originally devised especially for the IBM compatible PC
(graphics) and written in Turbo Pascal (Borland). Later on the PostScript
generator is extracted conforming as much as possible to standard Pascal:
ORITOPS. And still later ORITOAIL is added to create Adobe Illustrator files.


You start ORIXXXX with the command:

   ORIXXXX  filename
.
or:
   ORIXXXX  filename  option(s)

The filename is obligatory and the command file <filename.ORI> will be
converted.
*/

/*OriDoc CommandOptions
.PAGE Options on the ORIXXXX command
You may add one of the following options:

.T -DEFINE=key  or -DEF=key
Defines one or more symbols for use in IF directives.
.T -LANGUAGE=x or -LANG=x
ORIXXXX takes only text lines marked with an x (see Text commands;
default is E).
.
Symbol LANGISx is defined automatically.
.SEL TOPS-
.T -FAST
The diagrams will be drawn without pausing.
After that you are asked for a closing action.
.SEL TOPS+ DRAW-
.T -A4
Layout on the paper is in A4 format (default).
.T -A5
Layout on the paper is in A5 format.
.T -COLOR
Tells ORITOPS and ORITOAIL to generate
.T -COLOR=saturation
PostScript code for a color printer. Generate brighter colors
(default is Black/White; -COLOR defaults to -COLOR=1.25).
.SEL DRAW+
.T -DUMPATEND
After the last diagram is drawn the status of ORIXXXX is written to the
file ORIXXXX_.DMP (for debugging only).
*/


FUNCTION void ScanProgramParameters(int argc,char *argv[])
{
   maxstring key,par;
   int npar,i,code;
   char *p;

   CHECKINIT CheckMgSystem(this_unit,"ScanProgramParameters",true,mytracelevel,NULL);
   InitString(par,"",-1);
   InitString(key,"",-1);
   npar=0;
   for (i=1; i<argc; i++) {
      if ((argv[i]!=NULL) && (*argv[i]!='\0')) {
         StringCopy(par,-1,argv[i],0);
         if (par[0]=='-') {
            DelChars(par,0,1);
            UpString(par);
            SplitString(key,-1,par,-1,"=-");
            code=-1;
            if (Equal(key,"DUMPATEND"))  dumpatend=true;                         else
            if (Equal(key,"FAST"))       fastshow=true;                          else
            if (Equal(key,"LANGUAGE"))   textlanguage=par[0];                    else
            if (Equal(key,"LANG"))       textlanguage=par[0];                    else
            if (Equal(key,"DEF"))        IfDefAddKeys(par);                      else
            if (Equal(key,"DEFINE"))     IfDefAddKeys(par);                      else
            if (Equal(key,"A4"))         paper=a4;                               else
            if (Equal(key,"A5"))         paper=a5;                               else
            if (Equal(key,"COLOR"))      saturation=DoubleFromString(par,&code); else
            {
               /* ignore */
            }
            if (code>=0) saturation=1.25;
         } else {
            npar++;
            if (npar==1) StringCopy(globaldisk.filename,-1,par,-1);
            if (npar>1) {
               printf("Only one file parameter allowed (<%s> ignored)\n",par);
               ShowError("");
            }
         }
      }
   }
   sprintf(par,"LANGIS%c",textlanguage);
   IfDefAddKeys(par);
   p=IfDefKeyString();
   StringCopy(startifkey,-1,p,0);
   if (globaldisk.filename[0]==0) AbortBecause("You should start %s with:   %s  filename",ori_program_name,ori_program_name);
   GetRootFrom(dataroot,globaldisk.filename);
   TRACE CheckMgSystem(this_unit,"ScanProgramParameters",true,mytracelevel,"end IfDefKeyString: %s",p);
}                                      /* ScanProgramParameters */


/*OriDoc FileStructure
.PAGE Structure of a command file
Command files converted by ORIDRAW, ORITOPS and ORITOAIL should have the
following structure:

   Header
   Commands

The headers contains special information in lines beginning with the
following keywords:

   OriFile
   Level=3
   Model=l name
   Author=name
   Date=date of copyright
   EndOfHeader

Model directive(s)
.MAR 3
.RIGHT 72
.
Specifies the name of the model. You may give alternative
names in other languages. The first letter of each alternative is examined.
The alternative marked with the /LANG option letter (or an asterisk (*))
is selected used in drawing and printing.
.
.MAR 0
Author directive(s)
.MAR 3
.
Specifies the name of the author(s). If you created the model with more
than one author repeat the names in subsequent Author directives.
.
.MAR 0
Date directive
.MAR 3
.
Specifies the date of copyright (or last revision).
.
.MAR 0
EndOfHeader directive
.MAR 3
.
Marks the end of the header. Following the header you may specify commands.
.MAR 0
.RIGHT

For examples see file DEMO.ORI.
*/


FUNCTION void CreateNewFile(void)
{
   maxstring now;

   CHECKINIT CheckMgSystem(this_unit,"CreateNewFile",true,mytracelevel,NULL);
   InitString(now,"",-1);
   globaldisk.disk=fopen(globaldisk.filename,"w");
   fprintf(globaldisk.disk,"OriFile\nLevel=3\nModel=* Name\nAuthor=your name\n");
   DateTime(now);
   now[8]=0;
   fprintf(globaldisk.disk,"Date=%s\nEndOfHeader\n",now);
   fprintf(globaldisk.disk,"n 1\nr 0 0 5 5\n");
   fclose(globaldisk.disk);
}                                    /* CreateNewFile */


static FUNCTION boolean CheckFileLevel(char *line)
{
   TRACE CheckMgSystem(this_unit,"CheckFileLevel",true,mytracelevel,NULL);
   if (Equal(line,orifilelevel)) return true;
   return false;
}                                    /* CheckFileLevel */


static FUNCTION void CheckModelName(char *line)
{
   TRACE CheckMgSystem(this_unit,"CheckModelName",true,mytracelevel,NULL);
   if ((line[0]=='*') || ((line[0]==textlanguage) && (line[1]==' '))) {
      SubString(modelname,-1,line,1,maxstringlength,-1);
      StripBlanks(modelname);
   }
}                                    /* CheckModelName */


static int monlen[13]={0,31,28,31,30,31,30,31,31,30,31,30,31};


static FUNCTION boolean CheckModelDate(char *line)
{
   int i,mm,yy;
   string10 yys,mms,dds;
   boolean ok;

   TRACE CheckMgSystem(this_unit,"CheckModelDate",true,mytracelevel,NULL);
   InitString(mms,"",10);
   InitString(dds,"",10);
   InitString(yys,"",10);
   ok=true;
   modeldate[0]=0;
   if ((int)strlen(line)==6) StandardizeDate(line,false);
   if ((int)strlen(line)==8) {
      SubString(dds,10,line,6,2,-1);
      SubString(mms,10,line,4,2,-1);
      SubString(yys,10,line,0,4,-1);
   } else {
      ok=false;
   }
   if (ok) {
      mm=(int)LongFromString(mms,NULL);
      if ((mm>0) && (mm<=12)) {
         MonthName(mms,mm,10);
         i=(int)LongFromString(dds,NULL);
         yy=(int)LongFromString(yys,NULL);
         if ((yy % 4)==0) monlen[2]=29; else monlen[2]=28;
         if ((i<1) || (i>monlen[mm])) ok=false;
      } else {
         ok=false;
      }
   }
   if (ok) sprintf(modeldate,"%s %s",yys,mms);
   return ok;
}                                    /* CheckModelDate */


static FUNCTION void AddAuthorName(char *line)
{
   TRACE CheckMgSystem(this_unit,"AddAuthorName",true,mytracelevel,NULL);
   if (author[0]!=0) StringCat(author,-1,"/",0);
   StringCat(author,-1,line,-1);
}                                    /* AddAuthorName */


FUNCTION boolean StartGlobalDataFileOk(void)
{
   maxstring key,line;
   boolean inheader,levelok,headerok;

   CHECKINIT CheckMgSystem(this_unit,"StartGlobalDataFileOk",true,mytracelevel,NULL);
   InitString(key,"",-1);
   InitString(line,"",-1);
   modelname[0]=0;
   modeldate[0]=0;
   author[0]=0;
   levelok=false;
   ResetDataStatus(&globaldisk,globaldisk.filename,true,true);
   inheader=true;
   headerok=true;
   ReadLine(&globaldisk,line);
   StripBlanks(line);
   UpString(line);
   if (! Equal(line,"ORIFILE")) inheader=false;
   while (headerok && inheader && ! EofData(&globaldisk)) {
      ReadLine(&globaldisk,line);
      StripBlanks(line);
      SplitString(key,-1,line,-1," =");
      UpString(key);
      if (Equal(key,"LEVEL"))         inheader=levelok=CheckFileLevel(line);   else
      if (Equal(key,"MODEL"))         CheckModelName(line);                    else
      if (Equal(key,"DATE"))          inheader=CheckModelDate(line);           else
      if (Equal(key,"AUTHOR"))        AddAuthorName(line);                     else
      if (Equal(key,"ENDOFHEADER"))   inheader=false;                          else
      if (key[0]=='*')                /* ignore */;                            else
      {
         PrintString("Header line in error: %s=%s",key,line);
         inheader=false;
         headerok=false;
      }
   }
   if (! levelok) {
      Beep();
      PrintString("File %s is not a correct level %s .ORI file.",globaldisk.filename,orifilelevel);
      PrintString("You should upgrade it to level %s with the program UpGrade or edit it",orifilelevel);
      PrintString("yourself when you made a mistake while typing the header of the .ORI file.");
      headerok=false;
   } else {
      if (modeldate[0]==0) {
         Beep();
         PrintString("Date is incorrect/missing");
         headerok=false;
      }
      if (modelname[0]==0) {
         Beep();
         PrintString("Model name is incorrect/missing");
         headerok=false;
      }
      if (author[0]==0) {
         Beep();
         PrintString("Author is missing");
         headerok=false;
      }
   }
   if (! headerok) {
      Beep();
      PrintString("Error detected on line:");
      if (key[0]!=0) PrintString("%s %s",key,line); else PrintString("%s",line);
   }
   firstdiagram=true;
   return headerok;
}                                      /* StartGlobalDataFileOk */


FUNCTION void CloseDiagram(void)
{
   int wide,high,dotsperline,i;
   double maxtextlen;

   CHECKINIT CheckMgSystem(this_unit,"CloseDiagram",true,mytracelevel,NULL);
   xdrawmax=xdrawmax+1;
   xdrawmin=xdrawmin-1;
   if (xdrawmax<xdrawmin+10) xdrawmax=xdrawmin+10;
   if (xdrawmin<0) xdrawmin=0;
   ydrawmax=ydrawmax+1;
   ydrawmin=ydrawmin-1;
   if (ydrawmax<ydrawmin+10) ydrawmax=ydrawmin+10;
   if (ydrawmin<0) ydrawmin=0;
   dotsperline=52;
   xdrawmax=xdrawmax+0.9999;
   ydrawmax=ydrawmax+0.9999;
   high=Round((ydrawmax-ydrawmin)*paperfactor);
   wide=Round((xdrawmax-xdrawmin)*paperfactor);
   if (withframe) DrawFrame();
   if (diagramheigth==0) diagramheigth=ydrawmax-ydrawmin;
   if ((paperx+wide>paper_max_x) && autopos) {
      paperx=0;
      papery=papery+200+Round(diagramheigth*paperfactor)+dotsperline*paperplus;
      paperplus=0;
      if (papery>paper_max_y) {
         papery=Round(diagramheigth*paperfactor);
         FormFeed();
      }
   }
   if (paperx<0) paperx=0;
   if (papery<high) papery=high;
   lastdiagram=lastdiagram+1;
   (diagram0+lastdiagram)->page=printpage;
   (diagram0+lastdiagram)->x=paperx;
   (diagram0+lastdiagram)->y=papery;
   (diagram0+lastdiagram)->width=wide;
   (diagram0+lastdiagram)->heigth=high;
   (diagram0+lastdiagram)->lines=lastline;
   sprintf((diagram0+lastdiagram)->numbertext,"%3d%c",number,numberabc);
   (diagram0+lastdiagram)->xdmin=xdrawmin;
   (diagram0+lastdiagram)->xdmax=xdrawmax;
   (diagram0+lastdiagram)->ydmin=ydrawmin;
   (diagram0+lastdiagram)->ydmax=ydrawmax;
   maxtextlen=0;
   for (i=1; i<=lastline; i++) {
      (diagram0+lastdiagram)->textlen[i]=MeasureTextLength(&maxtextlen,(char *)(textline0+i));
   }
   (diagram0+lastdiagram)->maxtextlen=maxtextlen;
   (diagram0+lastdiagram)->pfactor=paperfactor;
   CloseDiagramSpecialToProgram(dotsperline,wide,high);
   paperx=paperx+wide+200;
   autopos=true;
   if ((number>0) && (lastline>paperplus)) paperplus=lastline;
}                                      /* CloseDiagram */


FUNCTION void StartDiagram(textfile *disk)
{
   int i;
   maxstring txt;

   CHECKINIT CheckMgSystem(this_unit,"StartDiagram",true,mytracelevel,NULL);
   ResetDefaults(false);
   InitString(txt,"",-1);
   ReadItem(disk,txt);
   numberabc=' ';
   i=(int)strlen(txt)-1;
   if (! isdigit(txt[i])) {
      numberabc=txt[i];
      txt[i]=0;
   }
   number=(int)LongFromString(txt,NULL);
   if (numbers[abs(number)]) doubledef[abs(number)]=true;
   numbers[abs(number)]=true;
   if (number==0) ShowError("Diagram 0 %s",illegal);
   ShowNumberInDiagram();
}                                      /* StartDiagram */


/*OriDoc Next
.PAGE Next command
The command:

.C N n

.SEL TOPS-
clears the screen and appends a number to a diagram; is the number
positive then it is drawn in the diagram, a negative number is not shown.
.SEL TOPS+ DRAW- HLP-
appends a number to a diagram; is the number positive
then it is drawn in the diagram, a negative number is not shown.
The diagram is closed and the information is appended to the PostScript
file.
.SEL ALL

Positioning of the diagrams on the print page is done automatically. But you
may direct ORIDRAW, ORITOPS and ORITOAIL to position a diagram elsewhere on a
page with one of the Paper layout commands.
.
ORIDRAW can show you the layout of the diagrams on the printerpages as it
will be generated by ORITOPS or ORITOAIL.
.SEL TOPS-

ORIDRAW pauses only if something is drawn after the last N command.
.
See under "Interaction" for the possible actions to invoke.
.SEL ALL
*/


FUNCTION void NextDiagram(textfile *disk,boolean restart)
{
   CHECKINIT CheckMgSystem(this_unit,"NextDiagram",true,mytracelevel,NULL);
   ClosePoly("NextDiagram");
   if (! firstdiagram) {
      CloseDiagram();
      if (abs(number)>=diagramtoskipto) Interaction(false);
   }
   firstdiagram=false;
   if (restart) StartDiagram(disk);
}                                      /* NextDiagram */


static FUNCTION void FilenameOfMacro(char *name,char kar)
{
   TRACE CheckMgSystem(this_unit,"FilenameOfMacro",true,mytracelevel,NULL);
   sprintf(name,"ORIMAC_%c.tmp",kar);
}                                      /* FilenameOfMacro */


/* */ FUNCTION void InterpretCommands(textfile *disk);


static FUNCTION void MacroDefine(textfile *disk)
{
   FILE *macfile;
   maxstring linex,fname,upline;
   char karm;

   TRACE CheckMgSystem(this_unit,"MacroDefine",true,mytracelevel,NULL);
   InitString(linex,"",-1);
   InitString(fname,"",-1);
   InitString(upline,"",-1);
   karm=ReadChar(disk,false);
   if (strchr(macroactive,karm)!=NULL) {
      ShowError("Macro %c already active",karm);
   } else {
      do {
         ReadLine(disk,linex);
      } while (linex[0]==0);
      UpString(linex);
      StripTrailing(linex);
      FilenameOfMacro(fname,karm);
      macfile=fopen(fname,"w");
      if (macfile==NULL) ShowError("Can't open macrofile %c for write",karm);
      while (! Equal(upline,"M=") && ! EofData(disk)) {
         if ((macfile!=NULL) && (linex[0]!=0)) fprintf(macfile,"%s\n",linex);
         ReadLine(disk,linex);
         StripTrailing(linex);
         StringCopy(upline,-1,linex,-1);
         UpString(upline);
      }
      if (! Equal(upline,"M=")) ShowError("Macro definition of macrofile %c not closed",karm);
      if (macfile!=NULL) fclose(macfile);
   }
}                                      /* MacroDefine */


/*OriDoc Include
.PAGE Including files
To insert a subdiagram from another file you may use the include command:

.C > filename sub

From the file 'filename'.ORI the diagram sub is inserted.
.
Subdiagrams should start with a line with the name of the diagram on it, as
follows:

.C < name

Mind that the character < is in the first position of the line.
.
After that line follow the drawing commands. The subdiagram is closed by the
start of a following subdiagram or the end of the file.
*/


static FUNCTION void IncludeFile(textfile *disk)
{
   textfile *incfile,*aid;
   int i;
   maxstring fname,linex,iname,ifcopy;
   boolean ok;
   char *p;

   TRACE CheckMgSystem(this_unit,"IncludeFile",true,mytracelevel,"level %d",includelevel+1);
   InitString(fname,"",-1);
   InitString(linex,"",-1);
   InitString(iname,"",-1);
   InitString(ifcopy,"",-1);
   incfile=NULL;
   includelevel++;
   ReadLine(disk,linex);
   for (i=(int)strlen(linex)-1; i>=0; i--) {
      if (linex[i]==',') linex[i]=' ';
   }
   StripBlanks(linex);
   if (linex[0]=='=') {
      StringCopy(fname,-1,disk->filename,-1);
      DelChars(linex,0,1);
   } else {
      SplitString(fname,-1,linex,-1," ");
      StringCat(fname,-1,".ori",0);
   }
   SplitString(iname,-1,linex,-1," ");
   InsertString(dataroot,fname,0,-1);
   FormatSlashes(fname,slash_char);
#if defined(__VMS__)
   MakeVaxPath(fname);
#endif
   ok=true;
   if (ok) {
      aid=disk;
      while (aid!=NULL) {
         if (Equal(fname,aid->filename)) {
            ok=false;
            ShowError("Can't include file in itself: %s",fname);
         }
         aid=aid->previous;
      }
   }
   if (! PathExist(fname)) {
      status.kar2=' ';
      ShowError("File <%s> not found",fname);
      ok=false;
   }
   if (ok) {
      ok=false;
      incfile=AllocateTextFile();
      if (incfile!=NULL) {
         incfile->disk=fopen(fname,"r");
         ok=(incfile->disk!=NULL);
      }
      if (! ok) ShowError("Not enough memory to include file %s at level %d",fname,includelevel);
   }
   if (ok) {
      p=IfDefKeyString();
      TRACE CheckMgSystem(this_unit,"IncludeFile",true,mytracelevel,"Keys_1: %s",p);
      StringCopy(ifcopy,-1,p,0);
      incfile->previous=disk;
      incfile->linenumber=0;
      SkipToDiagram(incfile,iname);
      ResetDataStatus(incfile,fname,true,false);
      InterpretCommands(incfile);
      fclose(incfile->disk);
      ResetDataStatus(disk,disk->filename,false,false);
      TRACE CheckMgSystem(this_unit,"IncludeFile",true,mytracelevel,"Keys_2: %s",p);
      ok=IfDefReset(false);
      IfDefAddKeys(ifcopy);
      TRACE CheckMgSystem(this_unit,"IncludeFile",true,mytracelevel,"Keys_3: %s",p);
   }
   if (incfile!=NULL) {
      if (incfile->disk!=NULL) fclose(incfile->disk);
      FreeTextFile(incfile);
   }
   includelevel--;
}                                      /* IncludeFile */


static FUNCTION void GetMacroParams(textfile *disk,char *macropars)
{
   int m;
   char kar,kar2;

   CHECKINIT CheckMgSystem(this_unit,"GetMacroParams",true,mytracelevel,NULL);
   SkipChar(disk);
   m=0;
   while ((NextChar(disk)!=')') && ! EndOfLine(disk)) {
      kar=(char)toupper(ReadChar(disk,false));
      if (isalpha(kar)) {
         if ((defpointA+kar-'A')->defined) {
            macropars[m]=kar;
            m++;
         }
      } else {
         if ((kar=='-') && (m>0)) {
            kar=macropars[m-1];
            kar2=(char)toupper(ReadChar(disk,false));
            if (! isalpha(kar2)) kar2=' ';
            while (kar<kar2) {
               kar++;
               if ((defpointA+kar-'A')->defined) {
                  macropars[m]=kar;
                  m++;
               }
            }
         }
      }
   }
   if (m==0) {
      macropars[0]='@';
      m=1;
   }
   macropars[m]=0;
   if (NextChar(disk)==')') {
      (void)ReadChar(disk,false);
   } else {
      ShowError("Macro call: parameter list not closed with a )");
   }
}                                      /* GetMacroParams */


static FUNCTION void MacroExecute(textfile *disk,char karm)
{
   textfile *macfile;
   int mtimes,nmpars,p,ma;
   maxstring fname,macropars,ifcopy;
   char *idks;

   TRACE CheckMgSystem(this_unit,"MacroExecute",true,mytracelevel,NULL);
   InitString(fname,"",-1);
   InitString(macropars,"@",-1);
   InitString(ifcopy,"",-1);
   mtimes=1;
   while (NextChar(disk)==' ') SkipChar(disk);
   if (isdigit(NextChar(disk))) mtimes=ReadInteger(disk);           else
   if (NextChar(disk)=='#')     mtimes=Round(ReadDouble(disk));     else
   if (NextChar(disk)=='(')     GetMacroParams(disk,macropars);
   if (strchr(macroactive,karm)!=NULL) {
      ShowError("Macro %c already active",karm);
   } else {
      nmpars=(int)strlen(macropars);
      ma=(int)strlen(macroactive);
      macroactive[ma]=karm;
      macroactive[ma+1]=0;
      FilenameOfMacro(fname,karm);
      if (PathExist(fname)) {
         macfile=AllocateTextFile();
         idks=IfDefKeyString();
         TRACE CheckMgSystem(this_unit,"MacroExecute",true,mytracelevel,"Keys_1: %s",idks);
         StringCopy(ifcopy,-1,idks,0);
         for (p=0; p<nmpars; p++) {
            macroparam=macropars[p];
            while (mtimes>0) {
               macfile->disk=fopen(fname,"r");
               ResetDataStatus(macfile,fname,true,true);
               InterpretCommands(macfile);
               fclose(macfile->disk);
               mtimes--;
               TRACE CheckMgSystem(this_unit,"IncludeFile",true,mytracelevel,"Keys_2: %s",idks);
               IfDefReset(false);
               IfDefAddKeys(ifcopy);
               TRACE CheckMgSystem(this_unit,"IncludeFile",true,mytracelevel,"Keys_3: %s",idks);
            }
            mtimes=1;
         }
         free(macfile);
         ResetDataStatus(disk,disk->filename,false,false);
      } else {
         ShowError("Macro %c not defined",karm);
      }
      macroactive[ma]=0;
   }
}                                      /* MacroExecute */


/*OriDoc Macros
.PAGE Macros
You may use macros. When you need a group of commands several times you may
declare them as a macro. ORIXXXX can remember 26 macros (each has a letter
appended to it). Later on in the commands you can have ORIXXXX repeat a
macro.

A macro is defined by:

.C M+l
      macro contents
.C M=

and executed with:

.C Ml count

When you omit the count the macro is executed only once.
.
A macro in combination with point commands and executed several times is
very powerfull.

Example (draw lines from 8,8 in all directions every 30 degrees):

   M+a
   vs .R .M
   Drd .R 12 .M
   M=
   D M 8 8     D R 16 8      Ma 30
*/


static FUNCTION void Macro(textfile *disk)
{
   char kar2;

   TRACE CheckMgSystem(this_unit,"Macro",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   if (isalpha(kar2))    MacroExecute(disk,kar2);         else
   if (kar2=='+')        MacroDefine(disk);               else
   {
      ShowError("Macro command (M%c) %s",kar2,illegal);
   }
}                                      /* Macro */


FUNCTION void DeleteMacroFiles(void)
{
   char c;
   maxstring macname;

   CHECKINIT CheckMgSystem(this_unit,"DeleteMacroFiles",true,mytracelevel,NULL);
   InitString(macname,"",-1);
   for (c='A'; c<='Z'; c++) {
      FilenameOfMacro(macname,c);
      DeleteFile(macname);
   }
}                                      /* DeleteMacroFiles */


/*OriDoc Commands
.PAGE Commands
A lot of commands are implemented. Some of the global commands we mention
here:

.C [
Comment (close with ])
.C ]
Close comment
.C <
Stop converting this file


Blanks are used as separator between commands and parameters.
Most commands need options to specify the actual action.
These options should follow the command immediately without intervening
blanks.
.
You may type more than one command on one line.
And commands may also span more lines.


For more information see:

.T Next command
Drawing commands
.T Arc/Arrow commands
Fill commands
.T Coordinate system
Define points
.T Using points
Symbol commands
.T Text commands
Paper layout
.T IF directives
Macros
.T Including files
Defaults
.SEL TOPS-
.T Interaction
Closing action
.T Editing the .ORI file
.SEL ALL
*/


FUNCTION void InterpretCommands(textfile *disk)
{
   char kar;
   boolean ready;
   maxstring dummy;
   int ok;

   CHECKINIT CheckMgSystem(this_unit,"InterpretCommands",true,mytracelevel,NULL);
   InitString(dummy,"",-1);
   ready=false;
   while (! ready && ! EofData(disk)) {
      if (! IfDefSaysYes()) SkipLine(disk);
      kar=ReadChar(disk,false);
      status.kar=kar;
      status.kar2=' ';
      if (IfDefSaysYes()) {
         switch (kar) {
            case ' ': ;                                              break;
            case 'A': ArcArrowCommand(disk);                         break;
            case 'B': BezierCurveCommand(disk);                      break;
            case 'C': CoordinateSystemCommand(disk);                 break;
            case 'D': PointCommand(disk);                            break;
            case 'E': EllipsCommand(disk);                           break;
            case 'F': FillCommand(disk);                             break;
            case 'L': PolyLine(disk);                                break;
            case 'M': Macro(disk);                                   break;
            case 'N': NextDiagram(disk,true);                        break;
            case 'P': PaperLayoutCommand(disk);                      break;
            case 'R': RectangleCommand(disk);                        break;
            case 'S': SymbolCommand(disk);                           break;
            case 'T': TextCommand(disk);                             break;
            case 'U': UnitCommand(disk);                             break;
            case '?': IfDirectiveCheck(disk);                        break;
            case '>': if (abs(number)>=diagramtoskipto) {
                         IncludeFile(disk);
                      } else {
                         SkipLine(disk);
                      }                                              break;
            case '<': ready=true;                                    break;
            case '[': SkipComment(disk);                             break;
            case ';': TestCommand(disk);                             break;
             default: {
                         ShowError("%c is not a correct command",kar);
                         while (! (isalpha(NextChar(disk)) || (strchr("[<>?",NextChar(disk))!=NULL)) && ! EofData(disk)) ReadItem(disk,dummy);
                      }
         }
      } else {
         if (kar=='?') IfDirectiveCheck(disk);
      }
      if (! makepostscript) {
         if (KeyPressed()) Interaction(true);
      }
      if (again || finitobasta) ready=true;
   }
   if (! ready) {
      ok=IfDefReset(true);
      if (ok!=IFDEF_OK) ShowIfError();
   }
}                                      /* InterpretCommands */

