/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Supply input procedures for ORITOPS and ORIDRAW program sources
     First version dd.19940815   (Pascal)
     Converted to C   19950216
*/

/*GenMake off */
#include "orilxxxx.h"
/*GenMake on */
#include "../clib/mgifdef.h"


/*extracth Local */

#define last_revision "19990701"
#define this_unit "OriLfile"
#define prog_version "unit"
#define mytracelevel 4
static boolean initialized=false;


/*extracth Vars */

char macroparam;

/*extracth Functions */

FUNCTION void InitializeOriLFile(int *argc,char *argv[])
{
   if (initialized) return;
   initialized=true;
   InitializeOriLXxxx(argc,argv);
   CheckOriUpdate(last_revision);
   InitializeMgIfDef();
   TRACE CheckMgSystem(this_unit,"InitializeOriLFile",true,mytracelevel,"Compiled at %s %s",__DATE__,__TIME__);
   macroparam='@';
}                                      /* InitializeOriLFile */


FUNCTION boolean EofData(textfile *filerecord)
{
   CHECKINIT CheckMgSystem(this_unit,"EofData",true,mytracelevel,filerecord->filename);
   return filerecord->finished;
}                                      /* EofData */


FUNCTION boolean EndOfLine(textfile *filerecord)
{
   CHECKINIT CheckMgSystem(this_unit,"EndOfLine",true,mytracelevel,filerecord->filename);
   return (filerecord->position>(int)strlen(filerecord->line));
}                                      /* EndOfLine */


FUNCTION void SkipChar(textfile *filerecord)
{
   maxstring temp;
   int i;

   CHECKINIT CheckMgSystem(this_unit,"SkipChar",true,mytracelevel,filerecord->filename);
   InitString(temp,"",-1);
   filerecord->position++;
   if (filerecord->position>=(int)strlen(filerecord->line)) {
      StringCopy(filerecord->line,-1,"===",0);
      if (feof(filerecord->disk)) {
         filerecord->finished=true;
      } else {
         ReadLn(filerecord->disk,temp,-1);
         StripBlanks(temp);
         for (i=(int)strlen(temp)-1; i>=0; i--) {
            if (temp[i]=='@') {
               if ((i==0) || (temp[i-1]!='\\')) temp[i]=macroparam;
            }
         }
         if ((int)strlen(temp)>240) ShowError("Next line counts more than 240 characters");
         StringCopy(filerecord->line,-1,temp,-1);
      }
      filerecord->linenumber++;
      filerecord->position=0;
      StringCat(filerecord->line,-1,"  \0\0\0",0);
      StringCopy(status.line,-1,filerecord->line,-1);
      status.linenumber=filerecord->linenumber;
   }
   status.position=filerecord->position;
}                                      /* SkipChar */


FUNCTION char ReadChar(textfile *filerecord,boolean iskar2)
{
   char kar;

   CHECKINIT CheckMgSystem(this_unit,"ReadChar",true,mytracelevel,filerecord->filename);
   kar=(char)toupper(filerecord->line[filerecord->position]);
   SkipChar(filerecord);
   if (iskar2) status.kar2=kar;
   return kar;
}                                      /* ReadChar */


FUNCTION void SkipBlanks(textfile *filerecord)
{
   CHECKINIT CheckMgSystem(this_unit,"SkipBlanks",true,mytracelevel,filerecord->filename);
   while ((filerecord->line[filerecord->position]==' ') && ! EndOfLine(filerecord)) SkipChar(filerecord);
}                                      /* SkipBlanks */


FUNCTION void ReadItem(textfile *filerecord,char *txt)
{
   int i;

   CHECKINIT CheckMgSystem(this_unit,"ReadItem",true,mytracelevel,filerecord->filename);
   while ((strchr(blankcomma,filerecord->line[filerecord->position])!=NULL) && ! filerecord->finished) SkipChar(filerecord);
   i=0;
   while ((strchr(blankcomma,filerecord->line[filerecord->position])==NULL) && ! filerecord->finished) {
      txt[i]=ReadChar(filerecord,false);
      i++;
   }
   txt[i]=0;
   SkipBlanks(filerecord);
}                                      /* ReadItem */


FUNCTION double ReadDouble(textfile *filerecord)
{
   int code;
   maxstring txt;
   char kar,kar2;
   double nr;

   CHECKINIT CheckMgSystem(this_unit,"ReadDouble",true,mytracelevel,filerecord->filename);
   InitString(txt,"",-1);
   nr=0;
   while ((strchr(blankcomma,filerecord->line[filerecord->position])!=NULL) && ! filerecord->finished) SkipChar(filerecord);
   if (filerecord->line[filerecord->position]=='#') {
      nr=1.0;
      SkipChar(filerecord);
      if (strchr(blankcomma,filerecord->line[filerecord->position])!=NULL) kar=' ';  else kar=ReadChar(filerecord,false);
      if (strchr(blankcomma,filerecord->line[filerecord->position])!=NULL) kar2=' '; else kar2=ReadChar(filerecord,false);
      if (isalpha(kar) && (strchr("XY",kar2)!=NULL)) {
         if ((defpointA+kar-'A')->defined) {
            if (kar2=='X') nr=(defpointA+kar-'A')->x;
            if (kar2=='Y') nr=(defpointA+kar-'A')->y;
         } else {
            ShowError("Point %c not (yet) defined",kar);
         }
      } else {
         if (isalpha(kar)) {
            ShowError("X or Y missing in number selection (=.%c)",kar);
         } else {
            ShowError("Number selection should be: .<letter><x|y>");
         }
      }
      code=0;
   } else if (filerecord->line[filerecord->position]=='>') {
      code=1;
   } else {
      ReadItem(filerecord,txt);
      nr=DoubleFromString(txt,&code);
   }
   if (code!=0) {
      nr=1.0;
      ShowError("Real number expected");
   }
   return nr;
}                                      /* ReadDouble */


FUNCTION void CheckMaxCoordinate(char which,double *c)
{
   CHECKINIT CheckMgSystem(this_unit,"CheckMaxCoordinate",true,mytracelevel,"%c %1.5f",which,*c);
   if (*c>3300) {
      ShowError("%c coordinate>3300",which);
      *c=3300;
   }
   if (*c<-3300) {
      ShowError("%c coordinate<-3300",which);
      *c= -3300;
   }
}                                      /* CheckMaxCoordinate */


FUNCTION void ReadPoint(textfile *filerecord,double *px,double *py)
{
   char kar;
   double px1,pz;

   CHECKINIT CheckMgSystem(this_unit,"ReadPoint",true,mytracelevel,filerecord->filename);
   while ((strchr(" ,",filerecord->line[filerecord->position])!=NULL) && ! filerecord->finished) SkipChar(filerecord);
   if (filerecord->line[filerecord->position]=='.') {
      SkipChar(filerecord);
      kar=ReadChar(filerecord,false);
      *px=1;
      *py=1;
      if (isalpha(kar)) {
         if ((defpointA+kar-'A')->defined) {
            *px=(defpointA+kar-'A')->x;
            *py=(defpointA+kar-'A')->y;
         } else {
            ShowError("Point %c not (yet) defined",kar);
         }
      } else if (kar=='@') {
         *px=lastpoint_x;
         *py=lastpoint_y;
      } else {
         ShowError("Point selector (=%c) should be a letter",kar);
      }
   } else {
      *px=ReadDouble(filerecord);
      *py=ReadDouble(filerecord);
      if (threedimensional) {
         pz=ReadDouble(filerecord);
         px1=(*px)*projection.xx+(*py)*projection.yx+pz*projection.zx;
         *py=(*px)*projection.xy+(*py)*projection.yy+pz*projection.zy;
         *px=px1;
      }
      CheckMaxCoordinate('X',px);
      CheckMaxCoordinate('Y',py);
   }
   lastpoint_x= *px;
   lastpoint_y= *py;
}                                      /* ReadPoint */


FUNCTION void ReadDouble3(textfile *filerecord,double *r1,double *r2,double *r3)
{
   CHECKINIT CheckMgSystem(this_unit,"ReadDouble3",true,mytracelevel,NULL);
   ReadPoint(filerecord,r1,r2);
   *r3=ReadDouble(filerecord);
}                                      /* ReadDouble3 */


FUNCTION void ReadDouble4(textfile *filerecord,double *r1,double *r2,double *r3,double *r4)
{
   CHECKINIT CheckMgSystem(this_unit,"ReadDouble4",true,mytracelevel,NULL);
   ReadPoint(filerecord,r1,r2);
   ReadPoint(filerecord,r3,r4);
}                                      /* ReadDouble4 */


FUNCTION void ReadDouble5(textfile *filerecord,double *r1,double *r2,double *r3,double *r4,double *r5)
{
   CHECKINIT CheckMgSystem(this_unit,"ReadDouble5",true,mytracelevel,NULL);
   ReadDouble4(filerecord,r1,r2,r3,r4);
   *r5=ReadDouble(filerecord);
}                                      /* ReadDouble5 */


FUNCTION void ReadDouble6(textfile *filerecord,double *r1,double *r2,double *r3,double *r4,double *r5,double *r6)
{
   CHECKINIT CheckMgSystem(this_unit,"ReadDouble6",true,mytracelevel,NULL);
   ReadDouble4(filerecord,r1,r2,r3,r4);
   ReadPoint(filerecord,r5,r6);
}                                      /* ReadDouble6 */


FUNCTION void ReadDouble8(textfile *filerecord,double *r1,double *r2,double *r3,double *r4,double *r5,double *r6,double *r7,double *r8)
{
   CHECKINIT CheckMgSystem(this_unit,"ReadDouble8",true,mytracelevel,NULL);
   ReadDouble4(filerecord,r1,r2,r3,r4);
   ReadDouble4(filerecord,r5,r6,r7,r8);
}                                      /* ReadDouble8 */


FUNCTION int ReadInteger(textfile *filerecord)
{
   int nr,code;
   maxstring txt;

   CHECKINIT CheckMgSystem(this_unit,"ReadInteger",true,mytracelevel,filerecord->filename);
   InitString(txt,"",-1);
   nr=0;
   if (filerecord->line[filerecord->position]=='>') {
      code=1;
   } else {
      ReadItem(filerecord,txt);
      nr=(int)LongFromString(txt,&code);
   }
   if (code!=0) {
      nr=0;
      ShowError("Integer number expected");
   }
   return nr;
}                                      /* ReadInteger */


FUNCTION char NextChar(textfile *filerecord)
{
   CHECKINIT CheckMgSystem(this_unit,"NextChar",true,mytracelevel,filerecord->filename);
   return filerecord->line[filerecord->position];
}                                      /* NextChar */


FUNCTION void SkipLine(textfile *filerecord)
{
   CHECKINIT CheckMgSystem(this_unit,"SkipLine",true,mytracelevel,filerecord->filename);
   filerecord->position=(int)strlen(filerecord->line);
   SkipChar(filerecord);
}                                      /* SkipLine */


FUNCTION void ReadLine(textfile *filerecord,char *txt)
{
   CHECKINIT CheckMgSystem(this_unit,"ReadLine",true,mytracelevel,filerecord->filename);
   if (filerecord->position>=(int)strlen(filerecord->line)) SkipChar(filerecord);
   SubString(txt,-1,filerecord->line,filerecord->position,maxstringlength,-1);
   SkipLine(filerecord);
}                                      /* ReadLine */


/*OriDoc If
.PAGE IF directives
You may direct the generation of diagrams in a flexible way with IF
directives. To do so the following directives are available:

.C ?DEFINE keyword
.C ?DEF keyword
.C ?UNDEF keyword
.C ?NOT keyword
.C ?COPY keyfrom keyto
.C ?IF keyword
.C ?IFDEF keyword
.C ?IFNOT keyword
.C ?IFNDEF keyword
.C ?ELSE
.C ?IFEND
.C ?ENDIF

With DEFINE or DEF you define a new keyword;
with UNDEF you remove it (again).
.
With NOT 'you switch the presency'.
.
COPY copies the presency of the first keyword to the second.
.
With IF or IFDEF ORIXXXX checks if the keyword is defined. If so, the
following lines are converted up to an ELSE or ENDIF. If not, then that
lines are skipped up to an ELSE and then the conversion is continued.
The conversion is always continued at an ENDIF.
The action of IFNOT or IFNDEF is just inverse: if the keyword is present
then the lines are skipped up to the ELSE and vice versa.
The ENDIF and IFEND are synonyms.

The maximum length of a keyword is 20 characters.
*/


FUNCTION void ShowIfError(void)
{
   char *p;

   CHECKINIT CheckMgSystem(this_unit,"ShowIfError",initialized,mytracelevel,NULL);
   p=IfDefErrorMessage();
   ShowError("%s",p);
}                                    /* ShowIfError */


FUNCTION void IfDirectiveCheck(textfile *filerecord)
{
   maxstring line;
   int ok;

   CHECKINIT CheckMgSystem(this_unit,"IfDirectiveCheck",true,mytracelevel,NULL);
   InitString(line,"",-1);
   ReadLine(filerecord,line);
   StripBlanks(line);
   ok=IfDefDirectiveCheck(line);
   if (ok!=IFDEF_OK) ShowIfError();
}                                      /* IfDirectiveCheck */


FUNCTION void ResetDataStatus(textfile *filerecord,char *fname,boolean filealso,boolean linenumberalso)
{
   CHECKINIT CheckMgSystem(this_unit,"ResetDataStatus",true,mytracelevel,NULL);
   if (filealso) {
      if (filerecord->filename!=fname) StringCopy(filerecord->filename,-1,fname,-1);
      filerecord->line[0]=0;
      if (linenumberalso) filerecord->linenumber=0;
      filerecord->position=0;
      filerecord->finished=false;
      SkipChar(filerecord);
   } else {
      StringCopy(status.line,-1,filerecord->line,-1);
      status.linenumber=filerecord->linenumber;
      status.position=filerecord->position;
   }
   StringCopy(status.filename,-1,fname,-1);
}                                      /* ResetDataStatus */


