/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Supply Adobe Illustrator procedures for ORITOAI
     First version dd.19940815   (Pascal)
     Converted to C   19950220
*/

#include "orilproc.h"

/*extracth Local */

#define last_revision "19990701"
#define this_unit "OriLadil"
#define prog_version "unit"
#define mytracelevel 5
#define maxlinesinbuffer 4000
#define maxkarsinbufferline 60
#define psbufferfile "temptemp.ai"
static boolean initialized=false;
static FILE *psfile;


typedef char *pslinepointer;

typedef struct{
   maxstring now,fname,line;
   pslinepointer buffer[maxlinesinbuffer];
   int pslinenumber,maxbuffer,bufline,pages,filecounter;
} postscriptstatusrecord;

typedef struct{
   double xmin,xmax,ymin,ymax;
} boundingboxrecord;

static postscriptstatusrecord psstatus;
static double actualpen,a5factor,a5xpos,a5ypos;
static int actualcolor;
static boundingboxrecord boundingbox;


/*extracth Functions */


FUNCTION void InitializeOriLAdil(int *argc,char *argv[])
{
   if (initialized) return;
   initialized=true;
   InitializeOriLProc(argc,argv);
   CheckOriUpdate(last_revision);
   TRACE CheckMgSystem(this_unit,"InitializeOriLAdil",true,mytracelevel,"Compiled at %s %s",__DATE__,__TIME__);
   psstatus.maxbuffer=1;
   psstatus.pslinenumber=0;
   psstatus.bufline=0;
   psstatus.pages=0;
   psstatus.filecounter=0;
   InitString(psstatus.line,"",-1);
   InitString(psstatus.fname,"",-1);
   InitString(psstatus.now,"",-1);
   psfile=NULL;
   actualpen=25.0;
   actualcolor=0;
}                                      /* InitializeOriLAdil */


static FUNCTION void Ps_WriteBuffer(char *txt)
{
   TRACE CheckMgSystem(this_unit,"Ps_WriteBuffer",true,mytracelevel,NULL);
   StringCat(psstatus.line,-1,txt,0);
}                                      /* Ps_WriteBuffer */


static FUNCTION void Ps_WriteBufferDouble(double r,int fraction)
{
   maxstring txt;

   TRACE CheckMgSystem(this_unit,"Ps_WriteBufferDouble",true,mytracelevel,NULL);
   InitString(txt,"",-1);
   if ((r<=0.0) && (r>-0.5)) r=0.0;
   DoubleToString(txt,r,1,fraction);
   StringCat(psstatus.line,-1,txt,-1);
   StringCat(psstatus.line,-1," ",0);
}                                      /* Ps_WriteBufferDouble */


static FUNCTION void Ps_WriteLnBuffer(char *txt)
{
   FILE *buffile;
   int i;

   TRACE CheckMgSystem(this_unit,"Ps_WriteLnBuffer",true,mytracelevel,"%d",psstatus.pslinenumber+1);
   StringCat(psstatus.line,-1,txt,0);
   if (psstatus.bufline>=psstatus.maxbuffer) {
      printf("\r%70s\r%5d%c%6d   Buffer to file ...","",number,numberabc,psstatus.pslinenumber);
      if (psstatus.bufline==psstatus.pslinenumber) {
         buffile=fopen(psbufferfile,"w");
      } else {
         buffile=fopen(psbufferfile,"a");
      }
      for (i=0; i<psstatus.bufline; i++) fprintf(buffile,"%s\n",psstatus.buffer[i]);
      fclose(buffile);
      psstatus.bufline=0;
   }
   psstatus.pslinenumber++;
   InitString(psstatus.buffer[psstatus.bufline],psstatus.line,maxkarsinbufferline);
   psstatus.bufline++;
   if (toppslinestrlength<(int)strlen(psstatus.line)) toppslinestrlength=(int)strlen(psstatus.line);
   if (psstatus.pslinenumber % 32==0) {
      printf("\r%70s\r%5d%c%6d","",number,numberabc,psstatus.pslinenumber);
      if (testmode) printf("  (%d)",toppslinestrlength);
   }
   psstatus.line[0]=0;
}                                      /* Ps_WriteLnBuffer */


static FUNCTION void Ps_TransformCommand(char *line)
{
   maxstring item;
   double x,y;
   int i,r,code;
   char command;

   TRACE CheckMgSystem(this_unit,"Ps_TransformCommand",true,mytracelevel,NULL);
   InitString(item,"",-1);
   StripBlanks(line);
   if (line[0]==0) return;
   command=line[0];
   switch (toupper(command)) {
      case 'C':   r=3;    break;
      case 'L':   r=1;    break;
      case 'M':   r=1;    break;
       default:   r=0;    break;
   }
   line[0]=' ';
   StripBlanks(line);
   for (i=0; i<r; i++) {
      SplitString(item,-1,line,-1," ");
      x=DoubleFromString(item,&code);
      if (code!=0) x=0.0;
      SplitString(item,-1,line,-1," ");
      y=DoubleFromString(item,&code);
      if (code!=0) y=0.0;
      x=a5xpos+a5factor*(paperx+paperfactor*(x-xdrawmin));
      y=a5ypos-a5factor*(papery+paperfactor*(y-ydrawmax));
      fprintf(psfile,"%7.3f %7.3f ",x,y);
      if (boundingbox.xmin>x) boundingbox.xmin=x;
      if (boundingbox.xmax<x) boundingbox.xmax=x;
      if (boundingbox.ymin>y) boundingbox.ymin=y;
      if (boundingbox.ymax<y) boundingbox.ymax=y;
   }
   if (line[0]!=0) fprintf(psfile,"%s ",line);
   fprintf(psfile,"%c\n",command);
}                                    /* Ps_TransformCommand */


static FUNCTION void Ps_CopyBuffer(void)
{
   FILE *buffile;
   maxstring line;
   int i;

   TRACE CheckMgSystem(this_unit,"Ps_CopyBuffer",true,mytracelevel,NULL);
   InitString(line,"",-1);
   printf("\r%5d%c%6d",number,numberabc,psstatus.pslinenumber);
   if (testmode) printf("  (%d)",toppslinestrlength);
   printf("   Write to file...");
   paper=a4;
   if (paper==a5) {
      a5factor=0.24*sqrt(0.5);
      a5xpos=100;
      a5ypos=680;
   } else {
      a5factor=0.24;
      a5xpos=22;
      a5ypos=815;
   }
   if (toppslines<psstatus.pslinenumber) toppslines=psstatus.pslinenumber;
   fprintf(psfile,"u\n");
   if (psstatus.bufline!=psstatus.pslinenumber) {
      buffile=fopen(psbufferfile,"r");
      while (ReadLn(buffile,line,-1)!=EOF) Ps_TransformCommand(line);
      fclose(buffile);
      DeleteFile(psbufferfile);
   }
   for (i=0; i<psstatus.bufline; i++) {
      StringCopy(line,-1,psstatus.buffer[i],maxkarsinbufferline);
      Ps_TransformCommand(line);
   }
   fprintf(psfile,"U\n\n");
   psstatus.line[0]=0;
   psstatus.pslinenumber=0;
   psstatus.bufline=0;
}                                      /* Ps_CopyBuffer */


FUNCTION void PsSetPenWidth(void)
{
   CHECKINIT CheckMgSystem(this_unit,"PsSetPenWidth",true,mytracelevel,NULL);
   actualpen=penwidth*6.7*penwidthfactor;
}                                      /* PsSetPenWidth */


static FUNCTION void Ps_StartAction(void)
{
   CHECKINIT CheckMgSystem(this_unit,"Ps_StartAction",true,mytracelevel,NULL);
   Ps_WriteLnBuffer("K 0.000 0.000 0.000 1.000");
   Ps_WriteLnBuffer("j 1");
   Ps_WriteLnBuffer("J 0");
   Ps_WriteBuffer("w ");
   Ps_WriteBufferDouble(actualpen,3);
   Ps_WriteLnBuffer("");
}                                      /* Ps_StartAction */


FUNCTION void PsStartStopUnit(boolean start)
{
   CHECKINIT CheckMgSystem(this_unit,"PsStartStopUnit",initialized,mytracelevel,NULL);
   if (start) Ps_WriteLnBuffer("u"); else Ps_WriteLnBuffer("U");
}                                    /* PsStartStopUnit */


FUNCTION void PsStartDiagramGraphics(void)
{
   CHECKINIT CheckMgSystem(this_unit,"PsStartDiagramGraphics",true,mytracelevel,NULL);
   psstatus.line[0]=0;
}                                      /* PsStartDiagramGraphics */


static FUNCTION void Ps_WriteFillColor(int fillindex)
{
   int r,g,b;
   double grayfactor;

   CHECKINIT CheckMgSystem(this_unit,"Ps_WriteFillColor",true,mytracelevel,NULL);
   if (saturation<0.5) actualcolor=0;
   switch (actualcolor) {
       case 0: r=0X3F;  g=0X3F;  b=0X3F;           /* Black */   break;
       case 1: r=0X2A;  g=0X15;  b=0X00;           /* Brown */   break;
       case 2: r=0X00;  g=0X00;  b=0X3F;           /* Blue  */   break;
       case 3: r=0X3F;  g=0X00;  b=0X3F;           /* Magenta */ break;
       case 4: r=0X3F;  g=0X00;  b=0X00;           /* Red   */   break;
       case 5: r=0X3F;  g=0X15;  b=0X00;           /* Orange */  break;
       case 6: r=0X3F;  g=0X3F;  b=0X00;           /* Yellow */  break;
       case 7: r=0X00;  g=0X3F;  b=0X00;           /* Green */   break;
       case 8: r=0X3F;  g=0X3F;  b=0X3F;           /* White */   break;
      default: r=0X00;  g=0X00;  b=0X3F;           /* Blue  */
   }
   switch (fillindex) {
       case 0: grayfactor=1.0;     break;
       case 1: grayfactor=0.0;     break;
       case 2: grayfactor=0.25;    break;
       case 3: grayfactor=0.5;     break;
       case 4: grayfactor=0.75;    break;
       case 5: grayfactor=0.875;   break;
      default: grayfactor=0.0;
   }
   if (grayfactor<0.05) {
      r=g=b=0;
      grayfactor=1.0;
   } else if (grayfactor>0.95) {
      r=g=b=0X3F;
      grayfactor=1.0;
   } else {
      grayfactor+=0.125;
   }
   Ps_WriteBuffer("k ");
   Ps_WriteBufferDouble(1.0-(grayfactor*r)/0X3F,4);
   Ps_WriteBufferDouble(1.0-(grayfactor*g)/0X3F,4);
   Ps_WriteBufferDouble(1.0-(grayfactor*b)/0X3F,4);
   Ps_WriteBufferDouble(0.0,3);
   Ps_WriteLnBuffer("");
}                                      /* Ps_WriteFillColor */


FUNCTION void PsWriteLine(double x1,double y1,double x2,double y2)
{
   CHECKINIT CheckMgSystem(this_unit,"PsWriteLine",true,mytracelevel,NULL);
   Ps_StartAction();
   Ps_WriteBuffer("m ");
   Ps_WriteBufferDouble(x1,1);
   Ps_WriteBufferDouble(y1,1);
   Ps_WriteLnBuffer("");
   Ps_WriteBuffer("L ");
   Ps_WriteBufferDouble(x2,1);
   Ps_WriteBufferDouble(y2,1);
   Ps_WriteLnBuffer("");
   Ps_WriteLnBuffer("S");
}                                      /* PsWriteLine */


FUNCTION void PsWritePolyPoints(int fillindex,boolean draw,boolean fill)
{
   int i,j,io;

   CHECKINIT CheckMgSystem(this_unit,"PsWritePolyPoints",true,mytracelevel,NULL);
   PsStartStopUnit(true);
   Ps_StartAction();
   Ps_WriteFillColor(fillindex);
   Ps_WriteBuffer("m ");
   Ps_WriteBufferDouble((polypoint0+1)->x/10.0,1);
   Ps_WriteBufferDouble((polypoint0+1)->y/10.0,1);
   Ps_WriteLnBuffer("");
   i=2;
   io=1;
   while (i<=polypoints) {
      if (polypointw[i]=='B') {
         Ps_WriteBuffer("C ");
         for (j=0; j<3; j++) {
            Ps_WriteBufferDouble((polypoint0+i+j)->x/10.0,1);
            Ps_WriteBufferDouble((polypoint0+i+j)->y/10.0,1);
         }
         Ps_WriteLnBuffer("");
         i+=3;
         io+=3;
      } else {
         Ps_WriteBuffer("L ");
         Ps_WriteBufferDouble((polypoint0+i)->x/10.0,1);
         Ps_WriteBufferDouble((polypoint0+i)->y/10.0,1);
         Ps_WriteLnBuffer("");
         io++;
         i++;
      }
   }
   if (fill) {
      if (draw) Ps_WriteLnBuffer("b"); else Ps_WriteLnBuffer("f");
   } else {
      Ps_WriteLnBuffer("S");
   }
   PsStartStopUnit(false);
}                                      /* PsWritePolyPoints */


FUNCTION void PsWriteBezier(double x0,double y0,double x1,double y1,double x2,double y2,double x3,double y3)
{
   CHECKINIT CheckMgSystem(this_unit,"PsWriteBezier",true,mytracelevel,"%f %f   %f %f   %f %f   %f %f",x0,y0,x1,y1,x2,y2,x3,y3);
   Ps_StartAction();
   Ps_WriteBuffer("m ");
   Ps_WriteBufferDouble(x0,1);
   Ps_WriteBufferDouble(y0,1);
   Ps_WriteLnBuffer("");
   Ps_WriteBuffer("C ");
   Ps_WriteBufferDouble(x1,1);
   Ps_WriteBufferDouble(y1,1);
   Ps_WriteBufferDouble(x2,1);
   Ps_WriteBufferDouble(y2,1);
   Ps_WriteBufferDouble(x3,1);
   Ps_WriteBufferDouble(y3,1);
   Ps_WriteLnBuffer("");
   Ps_WriteLnBuffer("S");
}                                      /* PsWriteBezier */


FUNCTION void PsSetFillColor(int color)
{
   actualcolor=color;
}                                      /* PsSetFillColor */


static FUNCTION void Ps_IncrementChar(char *kar)
{
   TRACE CheckMgSystem(this_unit,"Ps_IncrementChar",true,mytracelevel,NULL);
   if ((*kar)=='9') (*kar)='A'; else (*kar)++;
}                                    /* Ps_IncrementChar */


static FUNCTION void Ps_StartFile(void)
{
   TRACE CheckMgSystem(this_unit,"Ps_StartFile",true,mytracelevel,NULL);
   Ps_IncrementChar(&psstatus.fname[psstatus.filecounter]);
   psfile=fopen(psstatus.fname,"w");
   fprintf(psfile,"%%!PS-Adobe-3.0\n");
   fprintf(psfile,"%%%%Creator: program ORITOPS (V.%s %s)\n",ori_program_version,ori_program_date);
   fprintf(psfile,"%%%% Program made by M.J.van Gelder, KVI / RUG, Groningen, Nederland\n");
   fprintf(psfile,"%%%% E-mail address: M.J.van.Gelder@kvi.nl\n");
   fprintf(psfile,"%%%%Title: %s (page %d)\n",modelname,psstatus.pages);
   fprintf(psfile,"%%%%CreationDate:  %s\n",psstatus.now);
   fprintf(psfile,"%%%%BoundingBox: 000 000 999 999\n");
   fprintf(psfile,"%%%%DocumentProcessColors: Cyan Magenta Yellow Black\n");
   fprintf(psfile,"%%%%DocumentNeededResources: procset Adobe_packedarray 2.0 0\n");
   fprintf(psfile,"%%%%+ procset Adobe_cmykcolor 1.1 0\n");
   fprintf(psfile,"%%%%+ procset Adobe_cshow 1.1 0\n");
   fprintf(psfile,"%%%%+ procset Adobe_customcolor 1.0 0\n");
   fprintf(psfile,"%%%%+ procset Adobe_typography_AI3 1.0 1\n");
   fprintf(psfile,"%%%%+ procset Adobe_IllustratorA_AI3 1.0 1\n");
   fprintf(psfile,"%%AI3_ColorUsage: Color\n");
   fprintf(psfile,"%%AI3_TemplateBox:0 0 612 792\n");
   fprintf(psfile,"%%AI3_DocumentPreview: None\n");
   fprintf(psfile,"%%%%EndComments\n");
   fprintf(psfile,"%%%%BeginProlog\n");
   fprintf(psfile,"%%%%IncludeResource: procset Adobe_packedarray 2.0 0\n");
   fprintf(psfile,"Adobe_packedarray /initialize get exec\n");
   fprintf(psfile,"%%%%IncludeResource: procset Adobe_cmykcolor 1.1 0\n");
   fprintf(psfile,"%%%%IncludeResource: procset Adobe_cshow 1.1 0\n");
   fprintf(psfile,"%%%%IncludeResource: procset Adobe_customcolor 1.0 0\n");
   fprintf(psfile,"%%%%IncludeResource: procset Adobe_typography_AI3 1.0 1\n");
   fprintf(psfile,"%%%%IncludeResource: procset Adobe_IllustratorA_AI3 1.0 1\n");
   fprintf(psfile,"%%%%EndProlog\n");
   fprintf(psfile,"%%%%BeginSetup\n");
   fprintf(psfile,"%%%%IncludeFont:\n");
   fprintf(psfile,"Adobe_cmykcolor /initialize get exec\n");
   fprintf(psfile,"Adobe_cshow /initialize get exec\n");
   fprintf(psfile,"Adobe_customcolor /initialize get exec\n");
   fprintf(psfile,"Adobe_typography_AI3 /initialize get exec\n");
   fprintf(psfile,"Adobe_IllustratorA_AI3 /initialize get exec\n");
   fprintf(psfile,"[\n");
   fprintf(psfile,"39/quotesingle 96/grave 128/Adieresis/Aring/Ccedilla/Eacute/Ntilde/Odieresis\n");
   fprintf(psfile,"/Udieresis/aacute/agrave/acircumflex/adieresis/atilde/aring/ccedilla/eacute\n");
   fprintf(psfile,"/egrave/ecircumflex/edieresis/iacute/igrave/icircumflex/idieresis/ntilde\n");
   fprintf(psfile,"/oacute/ograve/ocircumflex/odieresis/otilde/uacute/ugrave/ucircumflex\n");
   fprintf(psfile,"/udieresis/dagger/degree/cent/sterling/section/bullet/paragraph/germandbls\n");
   fprintf(psfile,"/registered/copyright/trademark/acute/dieresis/.notdef/AE/Oslash\n");
   fprintf(psfile,"/.notdef/plusminus/.notdef/.notdef/yen/mu/.notdef/.notdef\n");
   fprintf(psfile,"/.notdef/.notdef/.notdef/ordfeminine/ordmasculine/.notdef/ae/oslash\n");
   fprintf(psfile,"/questiondown/exclamdown/logicalnot/.notdef/florin/.notdef/.notdef\n");
   fprintf(psfile,"/guillemotleft/guillemotright/ellipsis/.notdef/Agrave/Atilde/Otilde/OE/oe\n");
   fprintf(psfile,"/endash/emdash/quotedblleft/quotedblright/quoteleft/quoteright/divide\n");
   fprintf(psfile,"/.notdef/ydieresis/Ydieresis/fraction/currency/guilsinglleft/guilsinglright\n");
   fprintf(psfile,"/fi/fl/daggerdbl/periodcentered/quotesinglbase/quotedblbase/perthousand\n");
   fprintf(psfile,"/Acircumflex/Ecircumflex/Aacute/Edieresis/Egrave/Iacute/Icircumflex\n");
   fprintf(psfile,"/Idieresis/Igrave/Oacute/Ocircumflex/.notdef/Ograve/Uacute/Ucircumflex\n");
   fprintf(psfile,"/Ugrave/dotlessi/circumflex/tilde/macron/breve/dotaccent/ring/cedilla\n");
   fprintf(psfile,"/hungarumlaut/ogonek/caron\n");
   fprintf(psfile,"TE\n");
   fprintf(psfile,"%%AI3_BeginEncoding: _Courier Courier\n");
   fprintf(psfile,"[/_Courier/Courier 0 0 1 TZ\n");
   fprintf(psfile,"%%AI3_EndEncoding\n");
   fprintf(psfile,"%%%%EndSetup\n");
   boundingbox.xmin=1999;
   boundingbox.xmax=-1;
   boundingbox.ymin=1999;
   boundingbox.ymax=-1;
}                                      /* Ps_StartFile */


FUNCTION void Ps_CloseFile(boolean tell_lower)
{
   char buffer[1000],*p;
   int xl,xr,yt,yb;

   CHECKINIT CheckMgSystem(this_unit,"Ps_CloseFile",true,mytracelevel,NULL);
   if (psfile!=NULL) {
      fprintf(psfile,"%%%%PageTrailer\n");
      fprintf(psfile,"gsave annotatepage grestore showpage\n");
      fprintf(psfile,"%%%%Trailer\n");
      fprintf(psfile,"Adobe_IllustratorA_AI3 /terminate get exec\n");
      fprintf(psfile,"Adobe_typography_AI3 /terminate get exec\n");
      fprintf(psfile,"Adobe_customcolor /terminate get exec\n");
      fprintf(psfile,"Adobe_cshow /terminate get exec\n");
      fprintf(psfile,"Adobe_cmykcolor /terminate get exec\n");
      fprintf(psfile,"Adobe_packedarray /terminate get exec\n");
      fprintf(psfile,"%%%%EOF\n");
      fclose(psfile);
      psfile=fopen(psstatus.fname,"r+");
      fread(&buffer,1000,1,psfile);
      p=strstr(buffer,"000 000 999 999");
      if (p!=NULL) {
         xl=(int)floor(boundingbox.xmin);
         xr=(int)ceil(boundingbox.xmax);
         yb=(int)floor(boundingbox.ymin);
         yt=(int)ceil(boundingbox.ymax);
         sprintf(p,"%3d %3d %3d %3d",xl,yb,xr,yt);
         *(p+15)='\n';
         fseek(psfile,0,SEEK_SET);
         fwrite(&buffer,1000,1,psfile);
      }
      fclose(psfile);
      printf("\rWritten: %s (page %2d)",psstatus.fname,psstatus.pages);
      if (tell_lower && (psstatus.pages>1)) printf(" and lower numbers");
      printf("\n");
   }
   psfile=NULL;
}                                      /* Ps_CloseFile */


FUNCTION void PsWriteInitPage(int pagenr)
{
   int i;

   CHECKINIT CheckMgSystem(this_unit,"PsWriteInitPage",initialized,mytracelevel,NULL);
   Ps_CloseFile(false);
   psstatus.pages=pagenr;
   Ps_StartFile();
   if (pagenr==1) {
      papery= 40;
      for (i=0; author[i]!=0; i++) {
         if (author[i]=='/') papery+=50;
      }
   }
}                                    /* PsWriteInitPage */


FUNCTION void PsOpenFile(char *orifilename)
{
   maxstring inpname;
   int i;

   CHECKINIT CheckMgSystem(this_unit,"PsOpenFile",true,mytracelevel,NULL);
   InitString(inpname,orifilename,-1);
   inpname[(int)strlen(inpname)-4]=0;
   i=strlen(inpname)-1;
   while ((i>=0) && (strchr("/\\:",inpname[i])==NULL)) i--;
   StringCat(inpname,-1,"-------",0);
   inpname[i+7]='-';
   inpname[i+8]='0';
   inpname[i+9]=0;
   psstatus.filecounter=i+8;
   InitString(psstatus.fname,"",-1);
   sprintf(psstatus.fname,"%s.ai",inpname);
   DateTime(psstatus.now);
   PsWriteInitPage(1);
   for (i=0; i<maxlinesinbuffer; i++) {
      psstatus.buffer[i]=NULL;
      psstatus.buffer[i]=AllocString(maxkarsinbufferline);
      if (psstatus.buffer[i]==NULL) {
         psstatus.maxbuffer-=100;
         i--;
         while (i>psstatus.maxbuffer) {
            free(psstatus.buffer[i]);
            i--;
         }
         break;
      }
      psstatus.maxbuffer=i;
   }
}                                      /* PsOpenFile */


FUNCTION void PsCloseDiagram(int dotsperline,int wide,int high)
{
   CHECKINIT CheckMgSystem(this_unit,"PsCloseDiagram",true,mytracelevel,"%d %d %d",dotsperline,wide,high);
   if ((xdrawmin<0.5) && (xdrawmin>-0.5)) xdrawmin= -0.1;
   if ((ydrawmax<0.5) && (ydrawmax>-0.5)) ydrawmax= -0.1;
   Ps_CopyBuffer();
   if (number!=0) {
      if (lastline>paperplus) paperplus=lastline;
   }
}                                      /* PsCloseDiagram */


FUNCTION void PsCloseFile(void)
{
   CHECKINIT CheckMgSystem(this_unit,"PsCloseFile",true,mytracelevel,NULL);
   Ps_CloseFile(true);
   psstatus.fname[0]=0;
}                                      /* PsCloseFile */


FUNCTION void PsRemoveFile(void)
{
   CHECKINIT CheckMgSystem(this_unit,"PsRemoveFile",true,mytracelevel,NULL);
   if (psfile!=NULL) fclose(psfile);
   psfile=NULL;
   DeleteFile(psstatus.fname);
   psstatus.fname[0]=0;
}                                      /* PsRemoveFile */


