/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Provide Graphical Functions to OriDraw and OriToPs
     First version dd.19850515   (Pascal)
     Converted to C   19950216
*/

/*extracth Local */

#define last_revision "19990701"
#define this_unit "OriLxint"
#define prog_version "unit"
#define mytracelevel 5

#if defined(__MSDOS__)
#define kar_width   8
#define kar_height 16
#include <graphics.h>
#include <conio.h>
#include <dos.h>
#include <alloc.h>
#else
#define minimum_width  640
#define minimum_heigth 480
#define button_width    90
#define button_height   28
#define maxkar         300
#define normal_font_name "*courier-medium-r-normal-*-140-*-m*"
#define bold_font_name   "*courier-bold-r-normal-*-140-*-m*"
#include "../clib/mgprogio.h"
#include <signal.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>
#endif


/*extracth Global */

#include "orilproc.h"


/*extracth Vars */


/*extracth Local */

static boolean initialized=false,ingraphics=false;
static int ydisplay;
static char fillstipple[32]=
   {0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,    /*  1 Full/Extra   100 % */
    0xAA,0xFF,0xAA,0xFF,0xAA,0xFF,0xAA,0xFF,    /*  2 Dark          25 % */
    0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,    /*  3 Middle        50 % */
    0xAA,0x00,0xAA,0x00,0xAA,0x00,0xAA,0x00,    /*  4 Light         75 % */
   };

#if defined(__MSDOS__)
static int max_mode;
#else
static Display *display;
static Window root,win;
static char *display_name=NULL;
static int screen,lastkar,lastcommand;
static GC gc0,gcbutton,gcfill;
static char commands[30],karbuffer[maxkar+1];
static Colormap defcolormap;
static unsigned long colorpixel[9];
static Pixmap stipple[4];
static Font normalfont,boldfont;
static int kar_width=10;
static int kar_height=20;
#endif


/*extracth Functions */


#if ! defined(__MSDOS__)

static FUNCTION void XoriAllocColor(int index,char *name)
{
   XColor scrdef,exactdef;

   TRACE CheckMgSystem(this_unit,"XoriAllocColor",true,mytracelevel,NULL);
   XAllocNamedColor(display,defcolormap,name,&scrdef,&exactdef);
   colorpixel[index]=scrdef.pixel;
}                                    /* XoriAllocColor */


static FUNCTION void XoriCreatePixmap(int index)
{
   TRACE CheckMgSystem(this_unit,"XoriCreatePixmap",true,mytracelevel,NULL);
   stipple[index]=XCreateBitmapFromData(display,win,&fillstipple[8*index],8,8);
}                                    /* XoriCreatePixmap */

#endif


static FUNCTION void SetColorTable(void)
{
#if ! defined(__MSDOS__)
   int i;
#endif

   TRACE CheckMgSystem(this_unit,"SetColorTable",true,mytracelevel,NULL);
#if defined(__MSDOS__)
   setrgbpalette( 0,0x3F,0x3F,0x3F);       /* Black -> White */
   setrgbpalette( 1,0x2A,0x15,0x00);       /* Brown */
   setrgbpalette( 2,0x00,0x00,0x3F);       /* Blue */
   setrgbpalette( 3,0x3F,0x00,0x3F);       /* Magenta */
   setrgbpalette( 4,0x3F,0x00,0x00);       /* Red */
   setrgbpalette( 5,0x3F,0x1D,0x00);       /* Orange */
   setrgbpalette( 6,0x3F,0x3F,0x00);       /* Yellow */
   setrgbpalette(20,0x3F,0x3F,0x00);       /* Yellow */
   setrgbpalette( 7,0x00,0x00,0x00);       /* Gray -> Black */
   setrgbpalette( 8,0x00,0x00,0x00);       /* Gray -> Black */
   setrgbpalette(56,0x00,0x00,0x00);       /* Gray -> Black */
   setrgbpalette( 9,0x00,0x3F,0x00);       /* Green */
   setrgbpalette(57,0x00,0x3F,0x00);       /* Green */
   setrgbpalette(15,0x00,0x00,0x00);       /* White -> Black */
   setrgbpalette(63,0x00,0x00,0x00);       /* White -> Black */
#else
   defcolormap=DefaultColormap(display,screen);
   XoriAllocColor(0,"Black");
   for (i=1; i<9; i++) colorpixel[i]=colorpixel[0];
   XoriAllocColor(1,"Brown");
   XoriAllocColor(2,"Blue");
   XoriAllocColor(3,"Magenta");
   XoriAllocColor(4,"Red");
   XoriAllocColor(5,"Orange");
   XoriAllocColor(6,"Yellow");
   XoriAllocColor(7,"Green");
   XoriAllocColor(8,"White");
   for (i=0; i<4; i++) XoriCreatePixmap(i);
#endif
}                                      /* SetColorTable */


FUNCTION void SetPenWidth(void)
{
   int p;
#if ! defined(__MSDOS__)
   XGCValues gc_values;
#endif

   CHECKINIT CheckMgSystem(this_unit,"SetPenWidth",true,mytracelevel,NULL);
   p=(int)Round(penwidth*90.0*penwidthfactor-1.0);
   if (p>3) p=3;
   if (p<1) p=1;
#if defined(__MSDOS__)
   setlinestyle(0,0,p);
#else
   gc_values.line_width=p;
   XChangeGC(display,gc0,GCLineWidth,&gc_values);
#endif
}                                      /* SetPenWidth */


FUNCTION void ClearWindow(void)
{
   CHECKINIT CheckMgSystem(this_unit,"ClearWindow",true,mytracelevel,NULL);
#if defined(__MSDOS__)
   clearviewport();
   gotoxy(1,1);
#else
   XClearWindow(display,win);
#endif
   ydisplay=kar_height;
}                                      /* ClearWindow */


#if ! defined(__MSDOS__)

static FUNCTION void XoriInitWindow(void)
{
   unsigned long border_pixel,background_pixel;
   XGCValues gc_values;
   XClassHint class_hints;
   int dir_return,ascent,descent;
   XCharStruct overall;

   TRACE CheckMgSystem(this_unit,"XoriInitWindow",true,mytracelevel,NULL);
   root=RootWindow(display,screen);
   border_pixel=BlackPixel(display,screen);
   background_pixel=WhitePixel(display,screen);
   win=XCreateSimpleWindow(display,root,0,0,maxscreenx+1,maxscreeny+31,2,border_pixel,background_pixel);
   normalfont=XLoadFont(display,normal_font_name);
   boldfont=XLoadFont(display,bold_font_name);
   gc_values.foreground=BlackPixel(display,screen);
   gc_values.background=WhitePixel(display,screen);
   gc_values.font=normalfont;
   gc0=XCreateGC(display,win,GCForeground|GCBackground|GCFont,&gc_values);
   gcfill=XCreateGC(display,win,GCForeground|GCBackground,&gc_values);
   SetColorTable();
   gc_values.foreground=colorpixel[4];
   gc_values.background=colorpixel[4];
   gcbutton=XCreateGC(display,win,GCForeground|GCBackground,&gc_values);
   XStoreName(display,win,ori_program_name);
   class_hints.res_name=ori_program_name;
   class_hints.res_class=ori_program_name;
   XSetClassHint(display,win,&class_hints);
   XSetIconName(display,win,ori_program_name);
   XSelectInput(display,win,ExposureMask|ButtonPressMask|ButtonReleaseMask|KeyPressMask|KeyReleaseMask);
   XMapWindow(display,win);
   XFlush(display);
   sleep(1);
   XQueryTextExtents(display,normalfont,"X",1,&dir_return,&ascent,&descent,&overall);
   kar_height=ascent+descent;
   kar_width=overall.width;
}                                      /* XoriInitWindow */

#endif


FUNCTION void DrawButtonsIfInX(int color_index,char *labelpar)
{
#if defined(__MSDOS__)
   if ((color_index==0) || (labelpar[0]==0)) color_index=0; /* So Turbo C wont complain */
#else
   int x,xt,kl;
   XGCValues gc_values;
   maxstring key,labels;

   CHECKINIT CheckMgSystem(this_unit,"DrawButtonsIfInX",true,mytracelevel,NULL);
   gc_values.foreground=colorpixel[color_index];
   gc_values.background=colorpixel[color_index];
   gc_values.font=boldfont;
   XChangeGC(display,gcbutton,GCForeground|GCBackground|GCFont,&gc_values);
   lastcommand=0;
   x=0;
   InitString(key,"",-1);
   InitString(labels,labelpar,-1);
   while (labels[0]!=0) {
      SplitString(key,-1,labels,-1,",");
      if (Equal(key,"Help")) x=maxscreenx-button_width;
      XFillRectangle(display,win,gcbutton,x+2,maxscreeny+3,button_width-4,button_height-4);
      kl=strlen(key);
      xt=(button_width-kl*kar_width)/2;
      if (xt<0) xt=0;
      XDrawString(display,win,gc0,x+xt,maxscreeny+button_height-7,key,kl);
      if (key[0]!='H') {
         commands[lastcommand]=(char)toupper(key[0]);
         lastcommand++;
      }
      x=x+button_width;
   }
   commands[lastcommand]=0;
#endif
}                                      /* DrawButtonsIfInX */


static FUNCTION void DrawStandardButtonsIfInX(int color)
{
   TRACE CheckMgSystem(this_unit,"DrawStandardButtonsIfInX",true,mytracelevel,NULL);
   DrawButtonsIfInX(color,"Quit,Redraw,Fast,Next,Skip,Edit,Mark,Dump,Help");
}                                    /* DrawStandardButtonsIfInX */


FUNCTION void StartDiagramGraphics(void)
{
#if ! defined(__MSDOS__)
   Window root;
   int x,y;
   unsigned int width,height,border_width,depth;
#endif

   CHECKINIT CheckMgSystem(this_unit,"StartDiagramGraphics",true,mytracelevel,NULL);
   if (! ingraphics) {
#if defined(__MSDOS__)
      setgraphmode(max_mode);
      maxscreenx=getmaxx();
      maxscreeny=getmaxy();
      if (maxscreeny<400) extrafactor=400.0/480.0;
      if (maxscreenx>=639) xscale=1.0;                    else
      if (maxscreenx>=319) xscale=0.5;                    else
      {
         xscale=0.25;
      }
      if (maxscreeny>=399) yscale=1.0;                    else
      if (maxscreeny>=199) yscale=0.5;                    else
      {
         yscale=0.25;
      }
      directvideo=false;
      setviewport(0,0,maxscreenx,maxscreeny,true);
      setlinestyle(0,0,1);
      SetColorTable();
      setcolor(7);
#else
      maxscreenx=1200;
      maxscreeny=840;
      XoriInitWindow();
#endif
   }
#if ! defined(__MSDOS__)
   XGetGeometry(display,win,&root,&x,&y,&width,&height,&border_width,&depth);
   if ((width<minimum_width) || (height<minimum_heigth)) {
      if (width<minimum_width) width=minimum_width;
      if (height<minimum_heigth) height=minimum_heigth;
      XResizeWindow(display,win,width,height);
   }
#endif
   ClearWindow();
#if ! defined(__MSDOS__)
   maxscreenx=width-1;
   maxscreeny=height-button_height;
   DrawStandardButtonsIfInX(4);
   if ((maxscreenx>=900) && (maxscreeny>=700)) xscale=2.0; else xscale=1.0;
   yscale=xscale;
#endif
   ingraphics=true;
   polypoints=0;
   addtopoly=false;
}                                      /* StartDiagramGraphics */


#if ! defined(__MSDOS__)

static FUNCTION void XoriBufferKar(char kar)
{
   if (lastkar<maxkar) {
      lastkar++;
      karbuffer[lastkar]=toupper(kar);
   }
}                                    /* XoriBufferKar */


static FUNCTION void XoriCheckKey(XEvent report)
{
   char buffer[10];
   int bufsize=10,count;
   KeySym keysym;
   XComposeStatus compose;

   count=XLookupString((void *) &report,buffer,bufsize,&keysym,&compose);
   if (((keysym>=XK_space) && (keysym<=XK_asciitilde)) || (strchr("\n\003\010\033",keysym)!=NULL)) {
      XoriBufferKar((char)keysym);
   } else {
      switch (keysym) {
         case XK_Escape:       XoriBufferKar(27);         break;
         case XK_KP_Enter:     XoriBufferKar('\n');       break;
         case XK_KP_Space:     XoriBufferKar(' ');        break;
         case XK_Return:       XoriBufferKar('\n');       break;
         case XK_BackSpace:    XoriBufferKar(8);          break;
         case XK_Delete:       XoriBufferKar(8);          break;
         default:              XoriBufferKar(1);          break;
      }
   }
}                                    /* XoriCheckKey */


static FUNCTION void XoriCheckButton(XEvent report)
{
   int xb;

   xb=report.xbutton.x;
   if ((report.xbutton.y>=maxscreeny) && (xb>=0) && (xb<=maxscreenx) && (lastcommand>0)) {
      if (xb>=maxscreenx-button_width) {
          XoriBufferKar('h');
      } else {
         if (((xb+1) % button_width)>3) {
            if (xb/button_width<lastcommand) XoriBufferKar(commands[xb/button_width]);
         }
      }
   } else {
      XoriBufferKar(' ');
   }
}                                    /* XoriCheckButton */

#endif


FUNCTION boolean KeyPressed(void)
{
#if ! defined(__MSDOS__)
   XEvent report;
#endif

   CHECKINIT CheckMgSystem(this_unit,"KeyPressed",true,mytracelevel,NULL);
#if defined(__MSDOS__)
   return kbhit();
#else
   if (XPending(display)!=0) {
      XNextEvent(display,&report);
      switch (report.type) {
         case Expose:         /* ignore */;                 break;
         case KeyRelease:     XoriCheckKey(report);         break;
         case ButtonRelease:  XoriCheckButton(report);      break;
         case DestroyNotify:  exit(1);                      break;
      }
   }
   return (lastkar>=0);
#endif
}                                      /* KeyPressed */


FUNCTION char GetKey(void)
{
   char kar;
#if ! defined(__MSDOS__)
   int i;
#endif

   CHECKINIT CheckMgSystem(this_unit,"GetKey",true,mytracelevel,NULL);
#if defined(__MSDOS__)
   kar=(char)toupper((char)getch());
   if (kar==0) {
      (void)getch();
      kar=' ';
   }
#else
   while (! KeyPressed()) /* wait for an event */ ;
   kar=karbuffer[0];
   for (i=0; i<lastkar; i++) karbuffer[i]=karbuffer[i+1];
   lastkar--;
#endif
   if (kar=='Q') QuitSeen=true;
   return kar;
}                                      /* GetKey */


FUNCTION void ResetTextModeAndColors(void)
{
   CHECKINIT CheckMgSystem(this_unit,"ResetTextModeAndColors",true,mytracelevel,NULL);
#if defined(__MSDOS__)
   if (ingraphics) {
      restorecrtmode();
      textmode(3);
   }
   ingraphics=false;
#else
   XCloseDisplay(display);
#endif
}                                      /* ResetTextModeAndColors */


FUNCTION void Beep(void)
{
   CHECKINIT CheckMgSystem(this_unit,"Beep",true,mytracelevel,NULL);
#if defined(__MSDOS__)
   if (fastshow) {
      sound(420);
      delay(15);
   } else {
      sound(280);
      delay(50);
   }
   nosound();
#else
   XBell(display,25);
#endif
}                                      /* Beep */


static FUNCTION void ClearRectangle(int x1,int y1,int x2,int y2)
{
   int dd;

   TRACE CheckMgSystem(this_unit,"ClearRectangle",true,mytracelevel,NULL);
   if (x2<x1) { dd=x1;   x1=x2;   x2=dd; }
   if (y2<y1) { dd=y1;   y1=y2;   y2=dd; }
#if defined(__MSDOS__)
   if (x1<0) x1=0;
   if (y1<0) y1=0;
   if (x2>maxscreenx) x2=maxscreenx;
   if (y2>maxscreeny) y2=maxscreeny;
   setviewport(x1,y1,x2,y2,1);
   clearviewport();
   setviewport(0,0,maxscreenx,maxscreeny,1);
#else
   XClearArea(display,win,x1,y1,x2-x1,y2-y1,false);
#endif
}                                      /* ClearRectangle */


static FUNCTION void DrawTextXY(int x,int y,char *text,boolean standard)
{
#if ! defined(__MSDOS__)
   XGCValues gc_values;
#endif

   TRACE CheckMgSystem(this_unit,"DrawTextXY",true,mytracelevel,NULL);
#if defined(__MSDOS__)
   if (standard) {
      settextjustify(LEFT_TEXT,BOTTOM_TEXT);
      settextstyle(0,0,1);
   }
   outtextxy(x,y,text);
#else
   if (standard) {
      gc_values.font=boldfont;
      XChangeGC(display,gcbutton,GCFont,&gc_values);
   }
   XDrawString(display,win,gc0,x,y,text,strlen(text));
   gc_values.font=normalfont;
   XChangeGC(display,gcbutton,GCFont,&gc_values);
#endif
}                                      /* DrawTextXY */


FUNCTION void PrintString(char *fmt,...)
{
   va_list args;
   maxstring text;

/*lint -save -e737 */
   va_start(args,fmt);
/*lint -restore */
   (void)vsprintf(text,fmt,args);
   if (ingraphics) {
      DrawTextXY(0,ydisplay,text,true);
      ydisplay+=kar_height;
   } else {
      printf("%s\n",text);
   }
   va_end(args);
}                                      /* PrintString */


static FUNCTION void AskForContinue(char *prompt,boolean quitalso)
{
   if (quitalso) DrawButtonsIfInX(7,"Continue,Quit"); else DrawButtonsIfInX(7,"Continue");
   ydisplay=maxscreeny;
   PrintString(prompt);
   Beep();
   GetKey();
}                                    /* AskForContinue */


#include "orilhelp.inc"


FUNCTION void InitializeOriLXxxx(int *argc,char *argv[])
{
#if defined(__MSDOS__)
   int driver,mode,error;
   char *error_msg;
   maxstring path,part;
#else
   int i;
   maxstring argi;
#endif

   if (initialized) return;
   initialized=true;
   InitializeOriLProc(argc,argv);
   InitializeOriLHelp();
#if defined(__MSDOS__)
   InitString(path,"",-1);
   InitString(part,"",-1);
#else
   display_name=AllocString(-1);
   InitString(argi,"",-1);
   for (i=1; i<=(*argc); i++)
   {
      if ((argv[i]!=NULL) && (*argv[i]!='\0')) {
         StringCopy(argi,-1,argv[i],0);
         UpString(argi);
         if (Equal(argi,"-DISPLAY")) {
            argv[i]=NULL;
            i++;
            StringCopy(display_name,-1,argv[i],0);
            argv[i]=NULL;
         }
      }
   }
#endif
   CheckOriUpdate(last_revision);
   TRACE CheckMgSystem(this_unit,"InitializeOriLXxxx",true,mytracelevel,"Compiled at %s %s",__DATE__,__TIME__);
   maxscreenx=639;
   maxscreeny=479;
   extrafactor=1.0;
   xscale=0.25;
   yscale=0.25;
#if defined(__MSDOS__)
   sound(420);
   delay(1);
   nosound();
   error_msg=getenv("PATH");
   if (error_msg!=NULL) StringCopy(path,-1,error_msg,0); else path[0]=0;
   InsertString(";D:\\BC;C:\\BC;",path,0,-1);
   StringCopy(part,-1,argv[0],0);
   error_msg=strrchr(part,'\\');
   if (error_msg!=NULL) {
      *error_msg=0;
      InsertString(part,path,0,-1);
   }
   if (testmode) {
      printf("PATH=<%s>\n",path);
      PressEnter();
   }
   error=1;
   while ((path[0]!=0) && (error!=0)) {
      SplitString(part,-1,path,-1,";");
      driver=0;
      initgraph(&driver,&mode,part);
      error=graphresult();
      if ((error!=0) && (part[0]!=0)) {
         StringCat(part,-1,"\\BGI",0);
         driver=0;
         initgraph(&driver,&mode,part);
         error=graphresult();
      }
   }
   if (error==0) {
      settextstyle(2,0,3);
      error=graphresult();
   }
   max_mode=getmaxmode();
   error_msg=grapherrormsg(error);
   if (testmode || (error<0)) {
      printf("Driver=%d   Mode=%d      %s\n",driver,max_mode,error_msg);
      PressEnter();
   }
   if (error<0) exit(1);
   restorecrtmode();
   textmode(3);
   (void)atexit(ResetTextModeAndColors);
#else
   for (i=0; i<30; i++) commands[i]=0;
   lastkar= -1;
   display=XOpenDisplay(display_name);
   if (display==NULL) {
      fprintf(stderr,"Can't connect to server %s\n",XDisplayName(display_name));
      exit(1);
   }
   screen=DefaultScreen(display);
#endif
   fillcolor=2;
}                                      /* InitializeOriLXxxx */


FUNCTION void DumpStatus(void)
{
   FILE *dumpfile;

   CHECKINIT CheckMgSystem(this_unit,"DumpStatus",true,mytracelevel,NULL);
   dumpfile=DumpFileStart();
   fprintf(dumpfile,"   Number of points in polygon: %d\n",toppolypoints);
   fprintf(dumpfile,"\n");
   fclose(dumpfile);
}                                      /* DumpStatus */


static FUNCTION void ClearErrorMessageArea(boolean full)
{
   TRACE CheckMgSystem(this_unit,"ClearErrorMessageArea",true,mytracelevel,NULL);
   if (full) {
      ClearRectangle(0,0,maxscreenx,kar_height*5);
   } else {
      ClearRectangle(kar_width*35,0,kar_width*42,kar_height+2);
   }
}                                      /* ClearErrorMessageArea */


static FUNCTION void AskNewNumber(void)
{
   string10 nr;
   char kar;
   int pos;

   TRACE CheckMgSystem(this_unit,"AskNewNumber",true,mytracelevel,NULL);
   InitString(nr,"0",10);
   ClearErrorMessageArea(true);
   DrawTextXY(0,kar_height,"Type number of diagram to skip to: ",true);
   pos=1;
   do {
      nr[pos]='_';
      nr[pos+1]=0;
      ClearErrorMessageArea(false);
      DrawTextXY(35*kar_width,kar_height,nr,true);
      kar=GetKey();
      if (isdigit(kar) && (pos<6)) {
         if ((pos!=1) || (nr[0]!='0')) pos++;
         nr[pos-1]=kar;
      } else if ((kar==8) && (pos>0)) {
         pos--;
      } else if (kar==27){
         pos=0;
      } else {
         Beep();
      }
   } while (strchr("qQ\003\n\r",kar)==NULL);
   if (strchr("qQ\003",kar)!=NULL) exit(1);
   nr[pos]=0;
   if ((pos==0) || Equal(nr,"0")) diagramtoskipto=number+1; else diagramtoskipto=atoi(nr);
   if ((diagramtoskipto<=labs(number)) && (diagramtoskipto>-1)) again=true;
}                                      /* AskNewNumber */


FUNCTION void ShowHelp(void)
{
   CHECKINIT CheckMgSystem(this_unit,"ShowHelp",true,mytracelevel,NULL);
   ShowOriDrawHelp(ori_program_name,"oridraw.hlp");
}                                      /* ShowHelp */


static FUNCTION void ShowHelpNow(void)
{
   TRACE CheckMgSystem(this_unit,"ShowHelpNow",true,mytracelevel,NULL);
   ShowHelp();
   ClearWindow();
   ydisplay=maxscreeny-3*kar_height;
   PrintString("Press R to redraw the last diagram");
   PrintString("Press E to edit the .ORI file");
   DrawStandardButtonsIfInX(4);
}                                      /* ShowHelpNow */


/*OriDoc Interaction
.PAGE Interaction
When ORIDRAW encounters an N command or an error the program will pause.
.
You will hear a short sound.
.
You then may press one of the following keys:

.T S
ORIDRAW asks for the number of the diagram to skip to; if you only
press <Enter> drawing is continued with the next diagram.
.T D
The status of ORIDRAW is written to the file ORIDRAW_.TMP (for
debugging only).
.SYS DOS
.T E
You may edit the .ORI file.
After exiting the editor you will return in ORIDRAW
and this diagram is drawn again.
.SYS XWIN
.T E
You may edit the .ORI file.
The buttons will turn green to show you can safely edit and save the file.
After saving press the Restart button to redraw the diagram just shown.
.SYS ALL
.T F
Continue drawing without pauses.
.SYS DOS
.T H, ?
Shows this information screen by screen.
.SYS XWIN
.T H, ?
Tells you how to browse the HELP information.
.SYS ALL
.T N
Pause again at the end of each diagrams (stop F action).
.T M
Mark all points defined with their letter.
.T Q
ORIDRAW quits.
.T R
ORIDRAW redraws the last diagram.
.T <Enter>
ORIDRAW draws the next diagram.

You may press these keys also during the drawing process.
Drawing is postponed and the command is executed at once.
*/


FUNCTION void Interaction(boolean read)
{
   char kar;
   boolean doagain,wassymmetrical;

   CHECKINIT CheckMgSystem(this_unit,"Interaction",true,mytracelevel,NULL);
   diagramtoskipto= -1;
   wassymmetrical=symmetrical;
   symmetrical=false;
   doagain=true;
   kar=' ';
   if (macroactive[0]==0) CopyStatus(&editstatus,&status); else CopyStatus(&editstatus,&globalstatus);
   while (doagain) {
      Beep();
      if (KeyPressed() || ! fastshow || read) kar=GetKey();
      if (kar==27) kar='Q';
      switch (kar) {
         case 'D': DumpStatus();               break;
         case 'T': testmode=! testmode;        break;
         case 'M': (*markprocedure)();         break;
         case 'H': case '?': ShowHelpNow();    break;
          default: doagain=false;
      }
   }
   switch (kar) {
      case 'F': fastshow=true;           break;
      case 'N': fastshow=false;          break;
      case 'S': AskNewNumber();          break;
      case 'E': editfile=true;           break;
      case 'R': doagain=true;            break;
      case 'Q': finitobasta=true;        break;
       default: ;
   }
   if (finitobasta) {
      again=true;
      fastshow=false;
   }
   if (editfile || doagain) {
      again=true;
      if (fastshow) number=1;
      diagramtoskipto=abs(number);
   } else {
      if (! finitobasta) lasterror[0]=0;
   }
   symmetrical=wassymmetrical;
}                                      /* Interaction */


FUNCTION void ShowNumberInDiagram(void)
{
   char line[10];

   CHECKINIT CheckMgSystem(this_unit,"ShowNumberInDiagram",true,mytracelevel,NULL);
   if ((abs(number)<diagramtoskipto)) {
      sprintf(line,"%2d < %2d",number,diagramtoskipto);
      DrawTextXY(maxscreenx-8*kar_width,kar_height,line,true);
   } else {
      if (number>0) {
         sprintf(line,"%2d%c",number,numberabc);
         DrawTextXY(maxscreenx-4*kar_width,kar_height,line,true);
      }
   }
}                                      /* ShowNumberInDiagram */


static FUNCTION void ShowOriErrorMessage(char *txt)
{
   maxstring copy,temp;
   int posnr;

   TRACE CheckMgSystem(this_unit,"ShowOriErrorMessage",true,mytracelevel,txt);
   InitString(temp,"",-1);
   sprintf(temp,"==>> Error in %s on line %d,  command: %c %c",status.filename,status.linenumber,status.kar,status.kar2);
   InitString(copy,status.line,-1);
   posnr=status.position;
   while (posnr>60) {
      DelChars(copy,0,20);
      posnr-=20;
   }
   if (strlen(copy)>76) copy[76]=0;
   if (ingraphics) {
      ClearErrorMessageArea(true);
      DrawTextXY(0,ydisplay,temp,true);
      ydisplay+=kar_height;
      sprintf(temp,"<%s>",copy);
      DrawTextXY(0,ydisplay,temp,true);
      ydisplay+=kar_height;
      sprintf(temp,"^(%d)",status.position);
      DrawTextXY(posnr*kar_width,ydisplay,temp,true);
      ydisplay+=kar_height;
      DrawTextXY(0,ydisplay,"  >> ",true);
      DrawTextXY(5*kar_width,ydisplay,txt,true);
   }
}                                      /* ShowOriErrorMessage */


static FUNCTION void ShowLastError(void)
{
   CHECKINIT CheckMgSystem(this_unit,"ShowLastError",true,mytracelevel,NULL);
   if (lasterror[0]!=0) ShowOriErrorMessage(lasterror);
}                                      /* ShowLastError */


FUNCTION void ShowError(char *fmt,...)
{
   va_list args;

/*lint -save -e737 */
   va_start(args,fmt);
/*lint -restore */
   (void)vsprintf(lasterror,fmt,args);
   ydisplay=kar_height;
   ShowLastError();
   Interaction(true);
   if (ingraphics) ClearErrorMessageArea(true);
   va_end(args);
}                                      /* ShowError */


FUNCTION void SetFillColor(int color)
{
   CHECKINIT CheckMgSystem(this_unit,"SetFillColor",true,mytracelevel,NULL);
   if ((color<0) || (color>7)) {
      ShowError("FillColor must be in range 0..7");
      color=2;
   }
#if defined(__MSDOS__)
   if (color>=7) color+=2;
   if (color==0) color=15;
#endif
   fillcolor=color;
}                                      /* SetFillColor */


FUNCTION void StartStopUnit(boolean start)
{
   CHECKINIT CheckMgSystem(this_unit,"StartStopUnit",initialized,mytracelevel,NULL);
   if (start==0) start=0; /* So Turbo C wont complain */
}                                    /* StartStopUnit */


static FUNCTION void LineOnScreen(int x1,int y1,int x2,int y2)
{
#if defined(__MSDOS__)
   line(x1,y1,x2,y2);
#else
   XDrawLine(display,win,gc0,x1,y1,x2,y2);
#endif
}                                      /* LineOnScreen */


static FUNCTION void DrawPolyLine(boolean closed)
{
   int plus;

   TRACE CheckMgSystem(this_unit,"DrawPolyLine",true,mytracelevel,NULL);
   if (polypoints<2) ShowError("PolyLine with length %d",polypoints);
   plus=1;
   if (closed) plus=0;
#if defined(__MSDOS__)
   drawpoly(polypoints+1-plus,(int *)(polypoint0+plus));
#else
   XDrawLines(display,win,gc0,(XPoint *)(polypoint0+plus),polypoints+1-plus,CoordModeOrigin);
#endif
}                                      /* DrawPolyLine */


#if ! defined(__MSDOS__)

static FUNCTION void XoriSetFillStipple(Pixmap stipple,int cindex)
{
   XGCValues gc_values;

   TRACE CheckMgSystem(this_unit,"XoriSetFillStipple",true,mytracelevel,NULL);
   gc_values.fill_style=FillOpaqueStippled;
   gc_values.stipple=stipple;
   gc_values.foreground=colorpixel[cindex];
   gc_values.background=colorpixel[8];
   XChangeGC(display,gcfill,GCBackground|GCForeground|GCFillStyle|GCStipple,&gc_values);
}                                    /* XoriSetFillStipple */

#endif


FUNCTION void DrawPolyPoints(int fillindex,boolean draw,boolean fill)
{
   int i;

   CHECKINIT CheckMgSystem(this_unit,"DrawPolyPoints",true,mytracelevel,NULL);
   if (labs(number)>=diagramtoskipto) {
      polypoint0->x=(polypoint0+polypoints)->x;
      polypoint0->y=(polypoint0+polypoints)->y;
      if (polypoints<3) fill=false;
      if (fill) {
         if (fillindex>5) fillindex=5;
         if (fillindex<0) fillindex=0;
#if defined(__MSDOS__)
         if (! draw) setlinestyle(4,0,1);
         if (fillindex==1) setfillpattern(&fillstipple[0],15);      else
         if (fillindex==0) setfillpattern(&fillstipple[0],0);       else
         {
            setfillpattern(&fillstipple[(fillindex-2)*8],fillcolor);
         }
#else
         if (fillindex==1) XoriSetFillStipple(stipple[0],0);               else
         if (fillindex==0) XoriSetFillStipple(stipple[0],8);               else
         {
            XoriSetFillStipple(stipple[fillindex-2],fillcolor);
         }
#endif
#if defined(__MSDOS__)
         fillpoly(polypoints,(int *)(void *)polypoint0);
#else
         XFillPolygon(display,win,gcfill,(XPoint *)polypoint0,polypoints,Complex,CoordModeOrigin);
#endif
      }
      if (draw) DrawPolyLine(fill);
      if (symmetrical) {
         for (i=0; i<=polypoints; i++) (polypoint0+i)->x=imirror2-(polypoint0+i)->x;
#if defined(__MSDOS__)
         if (fill) fillpoly(polypoints,(int *)(void *)polypoint0);
#else
         if (fill) XFillPolygon(display,win,gcfill,(XPoint *)polypoint0,polypoints,Complex,CoordModeOrigin);
#endif
         if (draw) DrawPolyLine(fill);
      }
#if defined(__MSDOS__)
      if (! draw) setlinestyle(0,0,1);
#endif
   }
   addtopoly=false;
   if (polypoints>toppolypoints) toppolypoints=polypoints;
   polypoints=0;
}                                      /* DrawPolyPoints */


FUNCTION void InitPoly(char *what)
{
   CHECKINIT CheckMgSystem(this_unit,"InitPoly",true,mytracelevel,NULL);
   if (addtopoly && (polypoints>0)) {
      ShowError("PolyLine not closed before drawing %s",what);
      DrawPolyPoints(0,true,false);
   }
   addtopoly=true;
}                                      /* InitPoly */


FUNCTION void ClosePoly(char *what)
{
   CHECKINIT CheckMgSystem(this_unit,"ClosePoly",true,mytracelevel,NULL);
   InitPoly(what);
   addtopoly=false;
}                                      /* ClosePoly */


FUNCTION void PointToPoly(double x,double y,char what)
{
   int xi,yi;
   double xp,yp;
   boolean add;

   CHECKINIT CheckMgSystem(this_unit,"PointToPoly",true,mytracelevel,NULL);
   ConvertXY(x,y,&xi,&yi,&xp,&yp);
   CheckXyDraw(xp,yp);
   addtopoly=true;
   if (polypoints==0) {
      add=true;
   } else {
      add=false;
      if (xi!=(polypoint0+polypoints)->x) add=true;
      if (yi!=(polypoint0+polypoints)->y) add=true;
      if (polypoints>=max_poly) add=false;
   }
   if (add) {
      polypoints++;
      (polypoint0+polypoints)->x=xi;
      (polypoint0+polypoints)->y=yi;
      polypointw[polypoints]=what;
   }
}                                      /* PointToPoly */


FUNCTION void DrawLine(double x1,double y1,double x2,double y2)
{
   int xp1,xp2,yp1,yp2;
   double xps1,yps1,xps2,yps2;

   CHECKINIT CheckMgSystem(this_unit,"DrawLine",true,mytracelevel,NULL);
   if (labs(number)>=diagramtoskipto) {
      if (addtopoly) {
         PointToPoly(x1,y1,' ');
         PointToPoly(x2,y2,' ');
      } else {
         ConvertXY(x2,y2,&xp2,&yp2,&xps2,&yps2);
         ConvertXY(x1,y1,&xp1,&yp1,&xps1,&yps1);
         CheckXyDraw(xps2,yps2);
         CheckXyDraw(xps1,yps1);
         LineOnScreen(xp1,yp1,xp2,yp2);
         if (symmetrical) LineOnScreen(imirror2-xp1,yp1,imirror2-xp2,yp2);
      }
   }
}                                      /* DrawLine */


FUNCTION void DrawString(double x1,double y1,char *txt,boolean bold)
{
   int x,y,fsize,dx,dy;
   double xps,yps,tw,th,tm;
#if ! defined(__MSDOS__)
   XGCValues gc_values;
#endif

   CHECKINIT CheckMgSystem(this_unit,"DrawString",true,mytracelevel,NULL);
   if ((labs(number)>=diagramtoskipto) && (txt[0]!=0)) {
      fsize=Round(14/paperfactor);
      if (fsize<2) fsize=2;
      if (strlen(txt)>1) {
         tw=0;
         th=0;
#if defined(__MSDOS__)
         settextjustify(LEFT_TEXT,BOTTOM_TEXT);
#else
         dx=0;
         dy= -5;
#endif
      } else {
         tw=29/paperfactor;
         th=40/paperfactor;
#if defined(__MSDOS__)
         settextjustify(CENTER_TEXT,CENTER_TEXT);
#else
         dx= -5;
         dy=5;
#endif
      }
#if defined(__MSDOS__)
      if (bold) settextstyle(0,0,2); else settextstyle(2,0,fsize+1);
      setcolor(7);
      dx=0;
      dy=0;
#else
      if (bold) {
         gc_values.font=boldfont;
         XChangeGC(display,gc0,GCFont,&gc_values);
      }
#endif
      ConvertXY(x1,y1,&x,&y,&xps,&yps);
      xps=xps-tw*0.5;
      yps=yps+th*0.5;
      DrawTextXY(x+dx,y+dy,txt,false);
      if (symmetrical) DrawTextXY(imirror2-x+dx,y+dy,txt,false);
#if ! defined(__MSDOS__)
      if (bold) {
         gc_values.font=normalfont;
         XChangeGC(display,gc0,GCFont,&gc_values);
      }
#endif
      CheckXyDraw(xps,yps);
      tm=0;
      tw=MeasureTextLength(&tm,txt);
      CheckXyDraw(xps+tw*39/paperfactor,yps-40/paperfactor);
   }
}                                      /* DrawString */


FUNCTION void NextHeaderLine(char *txt)
{
   CHECKINIT CheckMgSystem(this_unit,"NextHeaderLine",true,mytracelevel,NULL);
   papery=papery+50;
   diagram0->lines++;
   diagram0->textlen[diagram0->lines]=MeasureTextLength(&(diagram0->maxtextlen),txt);
}                                      /* NextHeaderLine */


static FUNCTION void DrawPoint(double x1,double y1,int rx,int ry)
{
   int xp1,yp1;
   double xps,yps;

   TRACE CheckMgSystem(this_unit,"DrawPoint",true,mytracelevel,NULL);
   if (labs(number)>=diagramtoskipto) {
      ConvertXY(x1,y1,&xp1,&yp1,&xps,&yps);
      CheckXyDraw(xps,yps);
      LineOnScreen(xp1-rx,yp1-ry,xp1+rx,yp1+ry);
      if (symmetrical) LineOnScreen(imirror2-(xp1-rx),yp1-ry,imirror2-(xp1+rx),yp1+ry);
   }
}                                      /* DrawPoint */


FUNCTION void DrawBezierCurve(double x0,double y0,double x1,double y1,double x2,double y2,double x3,double y3,double dltd,boolean dotted)
{
   boolean filling;
   double ax,bx,cx,ay,by,cy,t,t2,t3,r,dt,dist,xp,yp;

   TRACE CheckMgSystem(this_unit,"DrawBezierCurve",true,mytracelevel,"0: %f %f  1: %f %f  2: %f %f  3: %f %f  dltd=%f dotted=%d",x0,y0,x1,y1,x2,y2,x3,y3,dltd,dotted);
   r=BezierCurveLength(x0,y0,x1,y1,x2,y2,x3,y3);
   dltd=0.6*dltd;
   dt=0.1*dltd/r;
   if (dt>0.1) dt=0.1;
   filling=addtopoly;
   if (! filling && ! dotted) InitPoly("BezierCurve");
   if (dotted) DrawPoint(x0,y0,0,0); else PointToPoly(x0,y0,' ');
   cx=3.0*(x1-x0);
   cy=3.0*(y1-y0);
   bx=3.0*(x2-2.0*x1+x0);
   by=3.0*(y2-2.0*y1+y0);
   ax=x3-x0-3.0*(x2-x1);
   ay=y3-y0-3.0*(y2-y1);
   t=dt;
   dist=0.0;
   xp=x0;
   yp=y0;
   while (t<1.0) {
      t2=t*t;
      t3=t2*t;
      x1=ax*t3+bx*t2+cx*t+x0;
      y1=ay*t3+by*t2+cy*t+y0;
      dist=dist+Distance(x1,y1,xp,yp);
      xp=x1;
      yp=y1;
      if (dist>dltd) {
         if (dotted) DrawPoint(x1,y1,0,0); else PointToPoly(x1,y1,' ');
         dist=0.0;
      }
      t+=dt;
   }
   if (dotted) DrawPoint(x3,y3,0,0); else PointToPoly(x3,y3,' ');
   if (! filling && ! dotted) DrawPolyPoints(0,true,false);
   addtopoly=filling;
}                                    /* DrawBezierCurve */


static FUNCTION void RectangleOnScreen(int x1,int y1,int x2,int y2,boolean fill)
{
   int dd;

   TRACE CheckMgSystem(this_unit,"RectangleOnScreen",true,mytracelevel,NULL);
   if (x2<x1) { dd=x1;   x1=x2;   x2=dd; }
   if (y2<y1) { dd=y1;   y1=y2;   y2=dd; }
#if defined(__MSDOS__)
   if (fill) bar(x1,y1,x2,y2); else rectangle(x1,y1,x2,y2);
#else
   if (fill) {
      XFillRectangle(display,win,gcfill,x1,y1,x2-x1,y2-y1);
   } else {
      XDrawRectangle(display,win,gc0,x1,y1,x2-x1,y2-y1);
   }
#endif
}                                      /* RectangleOnScreen */


FUNCTION void DrawFrame(void)
{
   CHECKINIT CheckMgSystem(this_unit,"DrawFrame",true,mytracelevel,NULL);
   RectangleOnScreen(xdrawmin,ydrawmin,xdrawmax,ydrawmax,false);
}                                      /* DrawFrame */


FUNCTION void FormFeed(void)
{
   CHECKINIT CheckMgSystem(this_unit,"FormFeed",true,mytracelevel,NULL);
   printpage++;
}                                      /* FormFeed */


FUNCTION void CloseDiagramSpecialToProgram(int dotsperline,int wide,int high)
{
   CHECKINIT CheckMgSystem(this_unit,"CloseDiagramSpecialToProgram",true,mytracelevel,NULL);
   if ((dotsperline==0) || (wide==0) || (high==0)) wide=0; /* So Turbo C wont complain */
}                                      /* CloseDiagramSpecialToProgram */


FUNCTION void DrawValleyFold(double x1,double y1,double x2,double y2,double dltd,double r)
{
   double dx,dy,ninvers;
   int i,n;

   CHECKINIT CheckMgSystem(this_unit,"DrawValleyFold",true,mytracelevel,NULL);
   n=2*Round((r-dltd-0.001)/(2*dltd))+1;
   if (n<3) n=3;
   ninvers=1.0/n;
   dx=(x2-x1)*ninvers;
   dy=(y2-y1)*ninvers;
   for (i=1; i<=n/2; i++) {
      DrawLine(x1,y1,x1+dx,y1+dy);
      x1=x1+2*dx;
      y1=y1+2*dy;
   }
   DrawLine(x1,y1,x2,y2);
}                                      /* DrawValleyFold */


FUNCTION void DrawDottedLine(double x1,double y1,double x2,double y2,double dltd,double r)
{
   double dx,dy,ninvers;
   int i,n;

   CHECKINIT CheckMgSystem(this_unit,"DrawDottedLine",true,mytracelevel,NULL);
   n=Round(0.75*(r-0.041667)/dltd);
   if (n<3) n=3;
   ninvers=1.0/n;
   dx=(x2-x1)*ninvers;
   dy=(y2-y1)*ninvers;
   for (i=1; i<=n; i++) {
      DrawPoint(x1,y1,0,0);
      x1=x1+dx;
      y1=y1+dy;
   }
   DrawPoint(x2,y2,0,0);
}                                      /* DrawDottedLine */


FUNCTION void DrawMountainFold(double x1,double y1,double x2,double y2,double dltd,double r)
{
   double dx,dy,ninvers;
   int i,n,rx,ry;

   CHECKINIT CheckMgSystem(this_unit,"DrawMountainFold",true,mytracelevel,NULL);
   dx=fabs(x2-x1);
   dy=fabs(y2-y1);
   rx=1;
   ry=1;
   if (dx>dy*2)                 rx=0;            else
   if (dx*2<dy)                 ry=0;            else
   if ((x2-x1)*(y2-y1)<0)    rx= -1;
   n=2*Round((r-dltd-0.001)/(2*dltd))+1;
   if (n<3) n=3;
   ninvers=1.0/n;
   dx=(x2-x1)*ninvers;
   dy=(y2-y1)*ninvers;
   for (i=1; i<=n/2; i++) {
      DrawLine(x1,y1,x1+dx,y1+dy);
      DrawPoint(x1+dx*1.5,y1+dy*1.5,rx,ry);
      x1=x1+2*dx;
      y1=y1+2*dy;
   }
   DrawLine(x1,y1,x2,y2);
}                                      /* DrawMountainFold */


static FUNCTION void TextBar(int x,int y,double tl)
{
   TRACE CheckMgSystem(this_unit,"TextBar",true,mytracelevel,NULL);
   RectangleOnScreen(x / 8,y / 8,x / 8+Round(tl*5),(y+39) / 8,true);
}                                      /* TextBar */


FUNCTION void ShowPages(void)
{
   int p,d,r,xtxt,ytxt;
   double maxtextlen,textlen;
   maxstring auth,auth1,temp;

   CHECKINIT CheckMgSystem(this_unit,"ShowPages",true,mytracelevel,NULL);
   ClearWindow();
   xtxt=paper_max_x/8+16;
#if defined(__MSDOS__)
   setlinestyle(0,0,1);
   setfillstyle(1,9);
#else
   XoriSetFillStipple(stipple[0],7);
#endif
   maxtextlen=0;
   textlen=MeasureTextLength(&maxtextlen,modelname);
   TextBar(0,40,textlen);
   textlen=MeasureTextLength(&maxtextlen,modeldate);
   TextBar(paper_max_x,40,-textlen);
   InitString(auth,author,-1);
   InitString(temp,"",-1);
   InitString(auth1,"",-1);
   d= -10;
   do {
      SplitString(auth1,-1,auth,-1,"/");
      textlen=MeasureTextLength(&maxtextlen,auth1);
      d=d+50;
      TextBar(paper_max_x / 2+paper_max_x / 26,d,textlen);
   } while (!(auth[0]==0));
   for (r=3; r<=diagram0->lines+2; r++) TextBar(0,r*55+paperyatstart,diagram0->textlen[r-2]);
   QuitSeen=false;
   for (p=1; p<=printpage; p++) {
      ytxt=15;
      sprintf(temp,"Page %d of %d",p,printpage);
      DrawTextXY(xtxt,ytxt,temp,true);
      ytxt+=16;
      for (d=1; d<=lastdiagram; d++) {
         if ((diagram0+d)->page==p) {
            sprintf(temp,"%3d%3d %4d %4d %4d %4d %3d %6.1f",d,(diagram0+d)->page,(diagram0+d)->x,(diagram0+d)->y,
               (diagram0+d)->width,(diagram0+d)->heigth,(diagram0+d)->lines,maxtextlen);
            DrawTextXY(xtxt,ytxt,temp,true);
            ytxt+=16;
            RectangleOnScreen((diagram0+d)->x / 8,(diagram0+d)->y / 8,((diagram0+d)->x+(diagram0+d)->width)/ 8,((diagram0+d)->y-(diagram0+d)->heigth)/ 8,false);
            for (r=1; r<=(diagram0+d)->lines; r++) TextBar((diagram0+d)->x-25,(diagram0+d)->y+r*55,(diagram0+d)->textlen[r]);
         }
      }
      RectangleOnScreen(0,0,paper_max_x / 8,paper_max_y / 8,false);
      AskForContinue("Continue ...",true);
      if (QuitSeen) {
         p=printpage+1;
      } else {
         ClearWindow();
#if defined(__MSDOS__)
         setfillstyle(1,9);
#endif
      }
   }
   ClearWindow();
}                                      /* ShowPages */


#if defined(__MSDOS__)

static FUNCTION boolean SpawnOk(char *cmd)
{
   double stime;
   int error;

   TRACE CheckMgSystem(this_unit,"SpawnOk",true,mytracelevel,NULL);
   FreeVariables();
   printf(" %ld core left\n",coreleft());
   printf("Command:<%s>\n",cmd);
   stime=HhMmSs();
   error=0;
   error=(int)system(cmd);
   if ((error!=0) || testmode) printf("Error=%d\n",error);
   stime=HhMmSs()-stime;
   if (testmode) printf("%5.2f seconds\n",stime);
   AllocateVariables();
   if ((error!=0) || (stime<2) || testmode) return false;
   return true;
}                                      /* SpawnOk */

#endif


/*OriDoc Edit
.PAGE Editing the .ORI file
ORIDRAW doesn't have a builtin editor. So you need your own editor. To start
it ORIDRAW issues the next DOS command:

   ORIEDIT.BAT  filename.ORI  linenumber

So you should make your editor available via the ORIEDIT command.
To do so make a file ORIEDIT.BAT with some simple commands:

   @ECHO off
   Copy %1 EditFile.Bak
   Emacs %1 -G%2

This example starts by copying the .ORI file to a backup.

The editor Emacs (Micro Emacs) can be directed to start editing on e specific
text line (the -G option). If your editor doesn't have an option like that you
should do the line positioning inside the editor yourself.
*/


FUNCTION void EditOriFile(void)
{
#if defined(__MSDOS__)
   maxstring ecmd,batfile;
#endif

   CHECKINIT CheckMgSystem(this_unit,"EditOriFile",true,mytracelevel,NULL);
   ClearWindow();
#if defined(__MSDOS__)
   ResetTextModeAndColors();
#endif
   ShowLastError();
#if defined(__MSDOS__)
   InitString(ecmd,"",-1);
   InitString(batfile,"",-1);
   sprintf(batfile,"%soriedit.bat",programroot);
   if (! PathExist(batfile)) StringCopy(batfile,-1,"oriedit.bat",0);
   sprintf(ecmd,"%s %s %d",batfile,editstatus.filename,editstatus.linenumber);
   if (! SpawnOk(ecmd)) PressEnter();
#else
   AskForContinue("Press a key after saving the 'new' file ...",false);
#endif
   StartDiagramGraphics();
}                                      /* EditOriFile */


