/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Provide Graphical Functions to OriDraw and OriToPs
     First version dd.19850515   (Pascal)
     Converted to C   19950216
*/

/*extracth Local */

#define last_revision "19990701"
#define this_unit "OriLxail"
#define prog_version "unit"
#define mytracelevel 5


/*extracth Global */

#include "orilproc.h"

/*extracth Local */

#include "oriladil.h"


/*extracth Vars */


/*extracth Local */

static boolean initialized=false,errorcount=0;


/*extracth Functions */


FUNCTION void SetPenWidth(void)
{
   PsSetPenWidth();
}                                      /* SetPenWidth */


FUNCTION void ClearWindow(void)
{
}                                      /* ClearWindow */


FUNCTION void DrawButtonsIfInX(int color_index,char *labelpar)
{
   if ((color_index==0) || (labelpar[0]==0)) color_index=0; /* So Turbo C wont complain */
}                                      /* DrawButtonsIfInX */


FUNCTION void StartDiagramGraphics(void)
{
   CHECKINIT CheckMgSystem(this_unit,"StartDiagramGraphics",true,mytracelevel,NULL);
   PsStartDiagramGraphics();
}                                      /* StartDiagramGraphics */


FUNCTION boolean KeyPressed(void)
{
   return false;
}                                      /* KeyPressed */


FUNCTION char GetKey(void)
{
   return ' ';
}                                      /* GetKey */


FUNCTION void ResetTextModeAndColors(void)
{
}                                      /* ResetTextModeAndColors */


FUNCTION void Beep(void)
{
   printf("\007");
}                                      /* Beep */


FUNCTION void PrintString(char *fmt,...)
{
   va_list args;

/*lint -save -e737 */
   va_start(args,fmt);
/*lint -restore */
   (void)vprintf(fmt,args);
   printf("\n");
   va_end(args);
}                                      /* PrintString */


FUNCTION void InitializeOriLXxxx(int *argc,char *argv[])
{
   if (initialized) return;
   initialized=true;
   InitializeOriLProc(argc,argv);
   makepostscript=true;
   InitializeOriLAdil(argc,argv);
   CheckOriUpdate(last_revision);
   TRACE CheckMgSystem(this_unit,"InitializeOriLXxxx",true,mytracelevel,"Compiled at %s %s",__DATE__,__TIME__);
   fillcolor=2;
}                                      /* InitializeOriLXxxx */


FUNCTION void DumpStatus(void)
{
   FILE *dumpfile;

   CHECKINIT CheckMgSystem(this_unit,"DumpStatus",true,mytracelevel,NULL);
   dumpfile=DumpFileStart();
   fprintf(dumpfile,"   PostScript lines in a diagram: %d\n",toppslines);
   fprintf(dumpfile,"\n");
   fprintf(dumpfile,"\n");
   fclose(dumpfile);
}                                      /* DumpStatus */


FUNCTION void ShowHelp(void)
{
}                                      /* ShowHelp */


FUNCTION void Interaction(boolean read)
{
   CHECKINIT CheckMgSystem(this_unit,"Interaction",true,mytracelevel,NULL);
   if (read) read=0;   /* So Turbo C wont complain */
}                                      /* Interaction */


FUNCTION void ShowNumberInDiagram(void)
{
   CHECKINIT CheckMgSystem(this_unit,"ShowNumberInDiagram",true,mytracelevel,NULL);
   printf("\r%5d%c\r",number,numberabc);
}                                      /* ShowNumberInDiagram */


static FUNCTION void ShowOriErrorMessage(char *txt)
{
   maxstring copy,temp;
   int posnr;

   TRACE CheckMgSystem(this_unit,"ShowOriErrorMessage",true,mytracelevel,txt);
   InitString(temp,"",-1);
   printf("\r%70s\r","\a");
   sprintf(temp,"==>> Error on line %d, char %d, in %s",status.linenumber,status.position,status.filename);
   InitString(copy,status.line,-1);
   posnr=status.position;
   while (posnr>60) {
      DelChars(copy,0,20);
      posnr-=20;
   }
   if (strlen(copy)>76) copy[76]=0;
   printf("%s\n<%s>\n",temp,copy);
   printf("   Command %c %c >> %s\n",status.kar,status.kar2,txt);
   errorcount++;
   if (errorcount>5) {
      printf("More than 5 errors found; conversion aborted\n");
      PsRemoveFile();
      exit(1);
   }
}                                      /* ShowOriErrorMessage */


static FUNCTION void ShowLastError(void)
{
   if (lasterror[0]!=0) ShowOriErrorMessage(lasterror);
}                                      /* ShowLastError */


FUNCTION void ShowError(char *fmt,...)
{
   va_list args;

/*lint -save -e737 */
   va_start(args,fmt);
/*lint -restore */
   (void)vsprintf(lasterror,fmt,args);
   ShowLastError();
   va_end(args);
}                                      /* ShowError */


FUNCTION void SetFillColor(int color)
{
   CHECKINIT CheckMgSystem(this_unit,"SetFillColor",true,mytracelevel,NULL);
   if ((color<0) || (color>7)) {
      ShowError("FillColor must be in range 0..7");
      color=2;
   }
   PsSetFillColor(color);
}                                      /* SetFillColor */


FUNCTION void StartStopUnit(boolean start)
{
   CHECKINIT CheckMgSystem(this_unit,"StartStopUnit",initialized,mytracelevel,NULL);
   PsStartStopUnit(start);
}                                    /* StartStopUnit */


static FUNCTION void DrawPolyLine(void)
{
   TRACE CheckMgSystem(this_unit,"DrawPolyLine",true,mytracelevel,NULL);
   if (polypoints<2) ShowError("PolyLine with length %d",polypoints);
   PsWritePolyPoints(1,true,false);
}                                      /* DrawPolyLine */


FUNCTION void DrawPolyPoints(int fillindex,boolean draw,boolean fill)
{
   int i;

   CHECKINIT CheckMgSystem(this_unit,"DrawPolyPoints",true,mytracelevel,NULL);
   if (polypoints<3) fill=false;
   if (fill) {
      if (fillindex>5) fillindex=5;
      if (fillindex<0) fillindex=0;
      PsWritePolyPoints(fillindex,draw,fill);
   } else {
      DrawPolyLine();
   }
   if (symmetrical) {
      for (i=1; i<=polypoints; i++) (polypoint0+i)->x=imirror20ps-(polypoint0+i)->x;
      if (fill) PsWritePolyPoints(fillindex,draw,fill); else DrawPolyLine();
   }
   addtopoly=false;
   if (polypoints>toppolypoints) toppolypoints=polypoints;
   polypoints=0;
}                                      /* DrawPolyPoints */


FUNCTION void InitPoly(char *what)
{
   CHECKINIT CheckMgSystem(this_unit,"InitPoly",true,mytracelevel,NULL);
   if (addtopoly && (polypoints>0)) {
      ShowError("PolyLine not closed before drawing %s",what);
      DrawPolyPoints(0,true,false);
   }
   addtopoly=true;
}                                      /* InitPoly */


FUNCTION void ClosePoly(char *what)
{
   CHECKINIT CheckMgSystem(this_unit,"ClosePoly",true,mytracelevel,NULL);
   InitPoly(what);
   addtopoly=false;
}                                      /* ClosePoly */


FUNCTION void PointToPoly(double x,double y,char what)
{
   int xi,yi;
   double xp,yp;
   boolean add;

   CHECKINIT CheckMgSystem(this_unit,"PointToPoly",true,mytracelevel,NULL);
   if (what==' ') {
      ConvertXY(x,y,&xi,&yi,&xp,&yp);
      CheckXyDraw(xp,yp);
   } else {
      ConvertBZ(x,y,&xp,&yp);
   }
   addtopoly=true;
   xi=Round(10.0*xp);
   yi=Round(10.0*yp);
   if ((polypoints==0) || (what!=' ')) {
      add=true;
   } else {
      add=false;
      if (xi!=(polypoint0+polypoints)->x) add=true;
      if (yi!=(polypoint0+polypoints)->y) add=true;
   }
   if (polypoints>=max_poly) add=false;
   if (add) {
      polypoints++;
      (polypoint0+polypoints)->x=xi;
      (polypoint0+polypoints)->y=yi;
      polypointw[polypoints]=what;
      CHECKINIT CheckMgSystem(this_unit,"PointToPoly.Add",true,mytracelevel,"pp=%d %d,%d what=%c diag=%d",polypoints,xi,yi,what,number);
   }
}                                      /* PointToPoly */


FUNCTION void DrawLine(double x1,double y1,double x2,double y2)
{
   int xp1,xp2,yp1,yp2;
   double xps1,yps1,xps2,yps2;

   CHECKINIT CheckMgSystem(this_unit,"DrawLine",true,mytracelevel,NULL);
   if (addtopoly) {
      PointToPoly(x1,y1,' ');
      PointToPoly(x2,y2,' ');
   } else {
      ConvertXY(x2,y2,&xp2,&yp2,&xps2,&yps2);
      ConvertXY(x1,y1,&xp1,&yp1,&xps1,&yps1);
      CheckXyDraw(xps2,yps2);
      CheckXyDraw(xps1,yps1);
      PsWriteLine(xps1,yps1,xps2,yps2);
      if (symmetrical) PsWriteLine(imirror2ps-xps1,yps1,imirror2ps-xps2,yps2);
   }
}                                      /* DrawLine */


FUNCTION void DrawString(double x1,double y1,char *txt,boolean bold)
{
   int x,y,fsize;
   double xps,yps,tw,th,tm;

   CHECKINIT CheckMgSystem(this_unit,"DrawString",true,mytracelevel,"%d",bold);
   if (txt[0]!=0) {
      fsize=Round(14/paperfactor);
      if (fsize<2) fsize=2;
      if (strlen(txt)>1) {
         tw=0;
         th=0;
      } else {
         tw=29/paperfactor;
         th=40/paperfactor;
      }
      ConvertXY(x1,y1,&x,&y,&xps,&yps);
      xps=xps-tw*0.5;
      yps=yps+th*0.5;
      CheckXyDraw(xps,yps);
      tm=0;
      tw=MeasureTextLength(&tm,txt);
      CheckXyDraw(xps+tw*39/paperfactor,yps-40/paperfactor);
   }
}                                      /* DrawString */


FUNCTION void NextHeaderLine(char *txt)
{
   CHECKINIT CheckMgSystem(this_unit,"NextHeaderLine",true,mytracelevel,NULL);
   papery=papery+50;
   diagram0->lines++;
   diagram0->textlen[diagram0->lines]=MeasureTextLength(&(diagram0->maxtextlen),txt);
}                                      /* NextHeaderLine */


static FUNCTION void CheckPoint(double x1,double y1)
{
   int xp1,yp1;
   double xps1,yps1;

   CHECKINIT CheckMgSystem(this_unit,"CheckPoint",true,mytracelevel,NULL);
   ConvertXY(x1,y1,&xp1,&yp1,&xps1,&yps1);
   CheckXyDraw(xps1,yps1);
}                                      /* CheckPoint */


static FUNCTION void DrawPoint(double x1,double y1)
{
   TRACE CheckMgSystem(this_unit,"DrawPoint",true,mytracelevel,NULL);
   DrawLine(x1,y1,x1+0.03,y1+0.03);
}                                      /* DrawPoint */


static FUNCTION void DrawBezierCurveDotted(double x0,double y0,double x1,double y1,double x2,double y2,double x3,double y3,double dltd)
{
   double ax,bx,cx,ay,by,cy,t,t2,t3,r,dt,dist,xp,yp;

   TRACE CheckMgSystem(this_unit,"DrawBezierCurveDotted",true,mytracelevel,"0: %f %f  1: %f %f  2: %f %f  3: %f %f  dltd=%f",x0,y0,x1,y1,x2,y2,x3,y3,dltd);
   r=BezierCurveLength(x0,y0,x1,y1,x2,y2,x3,y3);
   dltd=0.6*dltd;
   dt=0.1*dltd/r;
   if (dt>0.1) dt=0.1;
   DrawPoint(x0,y0);
   cx=3.0*(x1-x0);
   cy=3.0*(y1-y0);
   bx=3.0*(x2-2.0*x1+x0);
   by=3.0*(y2-2.0*y1+y0);
   ax=x3-x0-3.0*(x2-x1);
   ay=y3-y0-3.0*(y2-y1);
   t=dt;
   dist=0.0;
   xp=x0;
   yp=y0;
   while (t<1.0) {
      t2=t*t;
      t3=t2*t;
      x1=ax*t3+bx*t2+cx*t+x0;
      y1=ay*t3+by*t2+cy*t+y0;
      dist=dist+Distance(x1,y1,xp,yp);
      xp=x1;
      yp=y1;
      if (dist>dltd) {
         DrawPoint(x1,y1);
         dist=0.0;
      }
      t+=dt;
   }
   DrawPoint(x3,y3);
}                                    /* DrawBezierCurveDotted */


FUNCTION void DrawBezierCurve(double x0,double y0,double x1,double y1,double x2,double y2,double x3,double y3,double dltd,boolean dotted)
{
   double ax,bx,cx,ay,by,cy,t,t2,t3,len,rdot;
   double xps0,yps0,xps1,yps1,xps2,yps2,xps3,yps3;
   int n;

   TRACE CheckMgSystem(this_unit,"DrawBezierCurve",true,mytracelevel,"%f %f  %f %f  %f %f  %f %f  dltd=%f dotted=%d",x0,y0,x1,y1,x2,y2,x3,y3,dltd,dotted);
   if (dotted) {
      DrawBezierCurveDotted(x0,y0,x1,y1,x2,y2,x3,y3,dltd);
      return;
   }
   len=BezierCurveLength(x0,y0,x1,y1,x2,y2,x3,y3);
   n=Round(1.5*(len-0.041667)/dltd);
   if (n<12) n=12;
   rdot=(len*24-1)*xyfactor/n;
   TRACE CheckMgSystem(this_unit,"DrawBezierCurve.dot",true,mytracelevel,"len=%f  rdot=%f  n=%d  xyfactor=%f",len,rdot,n,xyfactor);
   if (! addtopoly) {
      ConvertBZ(x0,y0,&xps0,&yps0);
      ConvertBZ(x1,y1,&xps1,&yps1);
      ConvertBZ(x2,y2,&xps2,&yps2);
      ConvertBZ(x3,y3,&xps3,&yps3);
      if (dotted) {
         PsWriteBezier(xps0,yps0,xps1,yps1,xps2,yps2,xps3,yps3);
         if (symmetrical) PsWriteBezier(imirror2ps-xps0,yps0,imirror2ps-xps1,yps1,imirror2ps-xps2,yps2,imirror2ps-xps3,yps3);
      } else {
         PsWriteBezier(xps0,yps0,xps1,yps1,xps2,yps2,xps3,yps3);
         if (symmetrical) PsWriteBezier(imirror2ps-xps0,yps0,imirror2ps-xps1,yps1,imirror2ps-xps2,yps2,imirror2ps-xps3,yps3);
      }
   } else {
      PointToPoly(x0,y0,' ');
      PointToPoly(x1,y1,'B');
      PointToPoly(x2,y2,'B');
      PointToPoly(x3,y3,' ');
   }
   cx=3.0*(x1-x0);
   cy=3.0*(y1-y0);
   bx=3.0*(x2-2.0*x1+x0);
   by=3.0*(y2-2.0*y1+y0);
   ax=x3-x0-3.0*(x2-x1);
   ay=y3-y0-3.0*(y2-y1);
   rdot=1.0/n;
   t=rdot;
   CheckPoint(x0,y0);
   while (t<1.0) {
      t2=t*t;
      t3=t2*t;
      x1=ax*t3+bx*t2+cx*t+x0;
      y1=ay*t3+by*t2+cy*t+y0;
      CheckPoint(x1,y1);
      t+=rdot;
   }
   CheckPoint(x3,y3);
}                                    /* DrawBezierCurve */


FUNCTION void DrawFrame(void)
{
}                                      /* DrawFrame */


FUNCTION void FormFeed(void)
{
   CHECKINIT CheckMgSystem(this_unit,"FormFeed",true,mytracelevel,NULL);
   printpage++;
   PsWriteInitPage(printpage);
}                                      /* FormFeed */


FUNCTION void CloseDiagramSpecialToProgram(int dotsperline,int wide,int high)
{
   CHECKINIT CheckMgSystem(this_unit,"CloseDiagramSpecialToProgram",true,mytracelevel,NULL);
   PsCloseDiagram(dotsperline,wide,high);
   if (testmode) printf("\n");
   printf("\r%70s\r%6s  Page %2d   Position: %4d, %4d\n","",(diagram0+lastdiagram)->numbertext,
      (diagram0+lastdiagram)->page,(diagram0+lastdiagram)->x,(diagram0+lastdiagram)->y);
}                                      /* CloseDiagramSpecialToProgram */


FUNCTION void DrawValleyFold(double x1,double y1,double x2,double y2,double dltd,double r)
{
   double dx,dy,ninvers;
   int i,n;

   CHECKINIT CheckMgSystem(this_unit,"DrawValleyFold",true,mytracelevel,NULL);
   n=2*Round((r-dltd-0.001)/(2*dltd))+1;
   if (n<3) n=3;
   ninvers=1.0/n;
   dx=(x2-x1)*ninvers;
   dy=(y2-y1)*ninvers;
   for (i=1; i<=n/2; i++) {
      DrawLine(x1,y1,x1+dx,y1+dy);
      x1=x1+2*dx;
      y1=y1+2*dy;
   }
   DrawLine(x1,y1,x2,y2);
}                                      /* DrawValleyFold */


FUNCTION void DrawDottedLine(double x1,double y1,double x2,double y2,double dltd,double r)
{
   double dx,dy,ninvers;
   int i,n;

   CHECKINIT CheckMgSystem(this_unit,"DrawDottedLine",true,mytracelevel,NULL);
   n=Round(0.75*(r-0.041667)/dltd);
   if (n<3) n=3;
   ninvers=1.0/n;
   dx=(x2-x1)*ninvers;
   dy=(y2-y1)*ninvers;
   for (i=1; i<=n; i++) {
      DrawPoint(x1,y1);
      x1=x1+dx;
      y1=y1+dy;
   }
   DrawPoint(x2,y2);
}                                      /* DrawDottedLine */


FUNCTION void DrawMountainFold(double x1,double y1,double x2,double y2,double dltd,double r)
{
   double dx,dy,ninvers;
   int i,n;

   CHECKINIT CheckMgSystem(this_unit,"DrawMountainFold",true,mytracelevel,NULL);
   dx=fabs(x2-x1);
   dy=fabs(y2-y1);
   n=2*Round((r-dltd-0.001)/(2*dltd))+1;
   if (n<3) n=3;
   ninvers=1.0/n;
   dx=(x2-x1)*ninvers;
   dy=(y2-y1)*ninvers;
   for (i=1; i<=n/2; i++) {
      DrawLine(x1,y1,x1+dx,y1+dy);
      DrawPoint(x1+dx*1.5,y1+dy*1.5);
      x1=x1+2*dx;
      y1=y1+2*dy;
   }
   DrawLine(x1,y1,x2,y2);
}                                      /* DrawMountainFold */


FUNCTION void ShowPages(void)
{
}                                      /* ShowPages */


FUNCTION void EditOriFile(void)
{
}                                      /* EditOriFile */


