/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Supply various procedures for ORITOPS and ORIDRAW program sources
     First version dd.19940815   (Pascal)
     Converted to C   19950215
*/

/*extracth Local */

#define last_revision "19990701"
#define this_unit "OriLproc"
#define prog_version "unit"
#define mytracelevel 5

#if defined(__MSDOS__)
#include <alloc.h>
#endif

/*extracth Global */


#include "../clib/mgprogio.h"
#include "../clib/mgifdef.h"


/*extracth Local */

static boolean initialized=false;

/*extracth Global */


#define blankcomma " ,"
#define curvature_minimum  0.005
#define digits "0123456789-"
#define illegal " not permitted"
#define max_diagram       100
#define max_poly         2000
#define max_text           30
#define max_unit           10
#define nearly_two          1.99999
#define paper_max_x      2230
#define paper_max_y      3200


typedef struct tf_record *textfile_pointer;

typedef struct tf_record{
   maxstring filename,line;
   FILE * disk;
   int position,linenumber;
   boolean finished;
   textfile_pointer previous;
} textfile;

typedef struct{
   double x,y;
   boolean defined;
} defpointrecord;

typedef struct{
   maxstring filename,line;
   char kar,kar2;
   int position,linenumber;
} statusrecord;

typedef struct{
   short x,y;
} polypointrecord;

typedef char polypointtype[max_poly+1];

typedef struct{
   double xx,yx,zx,xy,yy,zy;
} projectionrecord;

typedef double textlenbuffer[max_text+1];

typedef struct{
   int page,x,y,width,heigth,lines;
   double xdmin,xdmax,ydmin,ydmax,pfactor,maxtextlen;
   string10 numbertext;
   textlenbuffer textlen;
} diagramrecord;

typedef struct us_record *unitstatus_pointer;

typedef struct us_record{
   boolean defined,symmetrical;
   double marginx,marginy,deltax,deltay,objectx,objecty,rmirror;
   double extrafactor,xyfactor,globalalfa,cosalfa,sinalfa;
   int fillcolor,imirror2,imirror2ps,imirror20ps;
   defpointrecord defpointA[27];
   projectionrecord projection;
} unitstatusrecord;

typedef double karlengthbuffer[129];

typedef boolean byteset[max_diagram+1];

typedef void procempty(void);

enum papertype {a4,a5};


/*extracth Vars */

boolean again,editfile,finitobasta,fastshow,makepostscript;
boolean symmetrical,addtopoly,threedimensional;
boolean firstdiagram,dumpatend,autopos,withframe;
byteset doubledef,numbers;
char numberabc,textlanguage;
double deltad,marginx,marginy,deltax,deltay,objectx,objecty,rmirror;
double penwidth,penwidthdefault,penwidthfactor,paperfactor,diagramheigth,xdrawmin,xdrawmax,ydrawmin,ydrawmax;
double saturation,extrafactor,lastpoint_x,lastpoint_y;
double xscale,yscale,xyfactor,globalalfa,cosalfa,sinalfa,fixpos,pidiv180,pi,pitimes2,pidiv2;
enum papertype paper;
int fillcolor,paperyatstart,imirror2,maxscreenx,maxscreeny,toppslines,toppslinestrlength;
int imirror2ps,imirror20ps,toptextlinestrlength,includelevel,diagramtoskipto,dumpnumber;
int lastdiagram,lastline,number,polypoints,toppolypoints,printpage,paperx,papery,paperplus;
karlengthbuffer karlength;
string20 dumpfilename,ori_program_version,ori_program_date,ori_program_name;
maxstring *textline0,copyright,author,modelname,modeldate,lasterror,macroactive,programroot,dataroot;
procempty *markprocedure;
defpointrecord *defpointA;
diagramrecord *diagram0;
maxstring startifkey;
projectionrecord projection;
statusrecord editstatus,globalstatus,status;
textfile globaldisk;
polypointrecord *polypoint0;
polypointtype polypointw;
unitstatus_pointer unitstatus[max_unit+1];


/*extracth Functions */

static FUNCTION void AddKarLengths(double l,char *kars)
{
   int i;

   TRACE CheckMgSystem(this_unit,"AddKarLengths",true,mytracelevel,NULL);
   for (i=0; i<(int)strlen(kars); i++) karlength[(int)kars[i]]=(double)l;
}                                      /* AddKarLengths */


static FUNCTION void InitKarLengths(void)
{
   char c;

   TRACE CheckMgSystem(this_unit,"InitKarLengths",true,mytracelevel,NULL);
   for (c=0; c<127; c++) karlength[(int)c]=0;
   AddKarLengths( 4.9 ,"\'`ijl");
   AddKarLengths( 5.75,"|");
   AddKarLengths( 6.15," !,./:;I[\\]ft");
   AddKarLengths( 7.3 ,"()-r(**)");
   AddKarLengths( 7.8 ,"\"");
   AddKarLengths( 8.55,"*");
   AddKarLengths(10.3 ,"^");
   AddKarLengths(11.0 ,"Jcksvxyz");
   AddKarLengths(12.2 ,"#$0123456789?L_abdeghnopqu");
   AddKarLengths(12.8 ,"+<=>~");
   AddKarLengths(13.4 ,"FTZ");
   AddKarLengths(14.6 ,"&ABEKPSVXY");
   AddKarLengths(15.8 ,"CDHNRUw");
   AddKarLengths(17.0 ,"GOQ");
   AddKarLengths(18.2 ,"Mm");
   AddKarLengths(19.5 ,"%");
   AddKarLengths(20.7 ,"W");
   AddKarLengths(22.2 ,"@");
}                                      /* InitKarLengths */


static FUNCTION void InitGlobalStrings(void)
{
   TRACE CheckMgSystem(this_unit,"InitGlobalStrings",initialized,mytracelevel,NULL);
   InitString(dumpfilename,"",20);
   InitString(copyright,"",-1);
   InitString(author,"",-1);
   InitString(modelname,"???",-1);
   InitString(modeldate,"",-1);
   InitString(lasterror,"",-1);
   InitString(macroactive,"",-1);
   InitString(programroot,"",-1);
   InitString(dataroot,"",-1);
   InitString(editstatus.filename,"",-1);
   InitString(editstatus.line,"",-1);
   InitString(globalstatus.filename,"",-1);
   InitString(globalstatus.line,"",-1);
   InitString(status.filename,"",-1);
   InitString(status.line,"",-1);
   InitString(globaldisk.filename,"",-1);
   InitString(globaldisk.line,"",-1);
}                                    /* InitGlobalStrings */


FUNCTION void AllocateVariables(void)
{
   int i;

   CHECKINIT CheckMgSystem(this_unit,"AllocateVariables",true,mytracelevel,NULL);
   polypoint0=malloc(sizeof(polypointrecord)*(max_poly+2));
   textline0=malloc(sizeof(maxstring)*(max_text+1));
   diagram0=malloc(sizeof(diagramrecord)*(max_diagram+2));
   defpointA=malloc(sizeof(defpointrecord)*27);
   for (i=0; i<=max_unit; i++) unitstatus[i]=malloc(sizeof(unitstatusrecord));
   if ((polypoint0==NULL) || (textline0==NULL) || (diagram0==NULL) || (defpointA==NULL) || (unitstatus[max_unit]==NULL)) {
      AbortBecause("Not enough memory for buffers");
   }
   for (i=0; i<=max_text; i++) InitString((char *)(textline0+i),"",-1);
   for (i=0; i<=max_diagram+1; i++) InitString((diagram0+i)->numbertext,"",10);
   for (i=0; i<=max_unit; i++) unitstatus[i]->defined=false;
}                                      /* AllocateVariables */


FUNCTION void FreeVariables(void)
{
   int i;

   CHECKINIT CheckMgSystem(this_unit,"FreeVariables",true,mytracelevel,NULL);
   free(polypoint0);
   free(textline0);
   free(diagram0);
   free(defpointA);
   for (i=0; i<=max_unit; i++) free(unitstatus[i]);
}                                      /* FreeVariables */


FUNCTION textfile *AllocateTextFile(void)
{
   textfile *ptr;

   CHECKINIT CheckMgSystem(this_unit,"*AllocateTextFile",true,mytracelevel,NULL);
   ptr=malloc(sizeof(textfile));
   if (ptr==NULL) AbortBecause("Not enough memory for extra file");
   InitString(ptr->line,"",-1);
   InitString(ptr->filename,"",-1);
   ptr->previous=NULL;
   return ptr;
}                                      /* AllocateTextFile */


FUNCTION void FreeTextFile(textfile *it)
{
   CHECKINIT CheckMgSystem(this_unit,"FreeTextFile",true,mytracelevel,NULL);
   free(it);
}                                      /* FreeTextFile */


FUNCTION void CopyStatus(statusrecord *dest,statusrecord *from)
{
   CHECKINIT CheckMgSystem(this_unit,"CopyStatus",true,mytracelevel,NULL);
   memcpy(dest,from,sizeof(statusrecord));
}                                      /* CopyStatus */


FUNCTION void GetRootFrom(char *root,char *from)
{
   int i;

   CHECKINIT CheckMgSystem(this_unit,"GetRootFrom",true,mytracelevel,from);
   StringCopy(root,-1,from,0);
#if defined(__VMS__)
   VaxPathToUnix(root);
#endif
   FormatSlashes(root,slash_char);
   i=strlen(root)-1;
   while ((i>=0) && (root[i]!=slash_char)) i--;
   root[i+1]=0;
}                                    /* GetRootFrom */


FUNCTION void DeleteFile(char *fname)
{
#if defined(__VMS__)
   maxstring cmd;
#endif

   TRACE CheckMgSystem(this_unit,"DeleteFile",true,mytracelevel,NULL);
   if (! PathExist(fname)) return;
#if defined(__VMS__)
   InitString(cmd,"",-1);
   sprintf(cmd,"Delete %s;*",fname);
   system(cmd);
#else
   remove(fname);
#endif
}                                    /* DeleteFile */


FUNCTION void CheckOriUpdate(char *last_rev)
{
   CHECKINIT CheckMgSystem(this_unit,"CheckOriUpdate",true,mytracelevel,last_rev);
   if (strcmp(ori_program_date,last_rev)<0) StringCopy(ori_program_date,20,last_rev,0);
   CheckUpdate(ori_program_date);
}                                    /* CheckOriUpdate */


FUNCTION void InitializeOriLProc(int *argc,char *argv[])
{
   if (initialized) return;
   initialized=true;
   InitializeMgProgIo(ori_program_name,ori_program_date,ori_program_version,argc,argv);
   InitializeMgIfDef();
   CheckOriUpdate(last_revision);
   TRACE CheckMgSystem(this_unit,"InitializeOriLProc",true,mytracelevel,"Compiled at %s %s",__DATE__,__TIME__);
   InitString(startifkey,"",-1);
   textlanguage='E';
   addtopoly=false;
   dumpatend=false;
   makepostscript=false;
   deltad=0.35;
   firstdiagram=true;
   dumpnumber=0;
   pi=4*atan(1.0);
   pidiv180=pi/180;
   pitimes2=pi*2.0;
   pidiv2=pi*0.5;
   rmirror=0;
   imirror2ps=0;
   imirror20ps=0;
   toptextlinestrlength=0;
   toppolypoints=0;
   lastdiagram=0;
   includelevel=0;
   saturation=0.0;
   xyfactor=1;
   objectx=0;
   objecty=0;
   cosalfa=1;
   sinalfa=0;
   deltax=0;
   deltay=0;
   xyfactor=1;
   marginx=0;
   marginy=0;
   xscale=1.0;
   yscale=1.0;
   extrafactor=1.0;
   lastpoint_x=0.0;
   lastpoint_y=0.0;
   penwidthdefault=0.03;
   penwidthfactor=1.0;
   penwidth=penwidthdefault;
   paper=a4;
   InitKarLengths();
   toppslines=0;
   toppslinestrlength=0;
   AllocateVariables();
   InitGlobalStrings();
   GetRootFrom(programroot,argv[0]);
}                                      /* InitializeOriLProc */


FUNCTION void NormalProjection(void)
{
   CHECKINIT CheckMgSystem(this_unit,"NormalProjection",true,mytracelevel,NULL);
   projection.xx=1;
   projection.yx=0;
   projection.zx=0;
   projection.xy=0;
   projection.yy=1;
   projection.zy=0;
}                                      /* NormalProjection */


FUNCTION double MeasureTextLength(double *maxtextlen,char *txt)
{
   int i;
   double textlen;

   CHECKINIT CheckMgSystem(this_unit,"MeasureTextLength",true,mytracelevel,NULL);
   textlen=0;
   if (txt[0]=='\\') {
      for (i=1; txt[i]!=0; i++) {
         if ((textlen>1) || (txt[i]!=' ')) textlen=textlen+13;
      }
   } else {
      for (i=0; txt[i]!=0; i++) textlen=textlen+karlength[(int)txt[i]];
   }
   textlen=textlen*0.062;
   if (textlen>*maxtextlen) *maxtextlen=textlen;
   return textlen;
}                                      /* MeasureTextLength */


FUNCTION double Distance(double x1,double y1,double x2,double y2)
{
   if ((fabs(x1-x2)<30000.0) || (fabs(y1-y2)<30000.0)) return sqrt((x1-x2)*(x1-x2)+(y1-y2)*(y1-y2));
   return 30000.0;
}                                      /* Distance */


FUNCTION double BezierCurveLength(double x0,double y0,double x1,double y1,double x2,double y2,double x3,double y3)
{
   double ax,bx,cx,ay,by,cy,t,t2,t3,len,delta;

   TRACE CheckMgSystem(this_unit,"BezierCurveLength",true,mytracelevel,"0: %f %f  1: %f %f  2: %f %f  3: %f %f",x0,y0,x1,y1,x2,y2,x3,y3);
   cx=3.0*(x1-x0);
   cy=3.0*(y1-y0);
   bx=3.0*(x2-2.0*x1+x0);
   by=3.0*(y2-2.0*y1+y0);
   ax=x3-x0-3.0*(x2-x1);
   ay=y3-y0-3.0*(y2-y1);
   delta=0.01;
   t=delta;
   len=0.0;
   x1=x0;
   y1=y0;
   while (t<1.0) {
      t2=t*t;
      t3=t2*t;
      x2=ax*t3+bx*t2+cx*t+x0;
      y2=ay*t3+by*t2+cy*t+y0;
      len=len+Distance(x2,y2,x1,y1);
      x1=x2;
      y1=y2;
      t+=delta;
   }
   len=len+Distance(x3,y3,x1,y1);
   return len;
}                                    /* BezierCurveLength */


FUNCTION void CheckXyDraw(double x,double y)
{
   double xs;

   CHECKINIT CheckMgSystem(this_unit,"CheckXyDraw",true,mytracelevel,NULL);
   if (x<xdrawmin) xdrawmin=x;
   if (x>xdrawmax) xdrawmax=x;
   if (y<ydrawmin) ydrawmin=y;
   if (y>ydrawmax) ydrawmax=y;
   if (symmetrical) {
      xs=imirror2ps-x;
      if (xs<xdrawmin) xdrawmin=xs;
      if (xs>xdrawmax) xdrawmax=xs;
   }
}                                      /* CheckXyDraw */


FUNCTION void ConvertXY(double x,double y,int *xp,int *yp,double *xps,double *yps)
{
   CHECKINIT CheckMgSystem(this_unit,"ConvertXY",true,mytracelevel,NULL);
   x=x+objectx;
   y=y+objecty;
   *xps=((x*cosalfa-y*sinalfa+deltax)*xyfactor+marginx)*24.0;
   *yps=((x*sinalfa+y*cosalfa+deltay)*xyfactor+marginy)*24.0;
   if (*xps>2400.0) *xps=2400.0; else if (*xps<-1600.0) *xps= -1600.0;
   if (*yps>1500.0) *yps=1500.0; else if (*yps<-1000.0) *yps= -1000.0;
   *xp=Round(*xps*xscale);
   *yp=Round(maxscreeny-*yps*yscale*extrafactor);
   *yps=479.0-*yps;
}                                      /* ConvertXY */


FUNCTION void ConvertBZ(double x,double y,double *xps,double *yps)
{
   CHECKINIT CheckMgSystem(this_unit,"ConvertBZ",true,mytracelevel,NULL);
   x=x+objectx;
   y=y+objecty;
   *xps=((x*cosalfa-y*sinalfa+deltax)*xyfactor+marginx)*24.0;
   *yps=((x*sinalfa+y*cosalfa+deltay)*xyfactor+marginy)*24.0;
   *yps=479.0-*yps;
}                                      /* ConvertBZ */


FUNCTION void ConvertRM(double r,double x,double y,double *rps,double *xps,double *yps)
{
   CHECKINIT CheckMgSystem(this_unit,"ConvertRM",true,mytracelevel,NULL);
   *rps=r*xyfactor*24.0;
   x=x+objectx;
   y=y+objecty;
   *xps=((x*cosalfa-y*sinalfa+deltax)*xyfactor+marginx)*24.0;
   *yps=((x*sinalfa+y*cosalfa+deltay)*xyfactor+marginy)*24.0;
   *yps=479.0-*yps;
}                                      /* ConvertRM */


FUNCTION void CheckBorderPoint(double x,double y)
{
   int xi,yi;
   double xp,yp;

   CHECKINIT CheckMgSystem(this_unit,"CheckBorderPoint",true,mytracelevel,NULL);
   ConvertXY(x,y,&xi,&yi,&xp,&yp);
   CheckXyDraw(xp,yp);
}                                      /* CheckBorderPoint */


FUNCTION void SetFactor(double factor)
{
   double xps,yps;
   int yp;

   CHECKINIT CheckMgSystem(this_unit,"SetFactor",true,mytracelevel,NULL);
   xyfactor=factor;
   ConvertXY(rmirror,0.0,&imirror2,&yp,&xps,&yps);
   imirror2=imirror2*2;
   imirror2ps=Round(xps*2+0.4999);
   imirror20ps=10*imirror2ps;
}                                      /* SetFactor */


FUNCTION void SetAngle(double angle)
{
   CHECKINIT CheckMgSystem(this_unit,"SetAngle",true,mytracelevel,NULL);
   globalalfa=angle;
   cosalfa=cos(angle*pidiv180);
   sinalfa=sin(angle*pidiv180);
}                                      /* SetAngle */


FUNCTION void UnitStatusSave(int u,boolean defpar)
{
   int i;

   TRACE CheckMgSystem(this_unit,"UnitStatusSave",true,mytracelevel,"%d defpar=%d",u,defpar);
   if ((u<0) || (u>max_unit)) return;
   unitstatus[u]->defined=defpar;
   unitstatus[u]->marginx=marginx;
   unitstatus[u]->marginy=marginy;
   unitstatus[u]->deltax=deltax;
   unitstatus[u]->deltay=deltay;
   unitstatus[u]->objectx=objectx;
   unitstatus[u]->objecty=objecty;
   unitstatus[u]->symmetrical=symmetrical;
   unitstatus[u]->rmirror=rmirror;
   unitstatus[u]->imirror2=imirror2;
   unitstatus[u]->imirror20ps=imirror20ps;
   unitstatus[u]->globalalfa=globalalfa;
   unitstatus[u]->cosalfa=cosalfa;
   unitstatus[u]->sinalfa=sinalfa;
   unitstatus[u]->extrafactor=extrafactor;
   unitstatus[u]->xyfactor=xyfactor;
   unitstatus[u]->fillcolor=fillcolor;
   for (i=0; i<=26; i++) {
      memcpy(&(unitstatus[u]->defpointA[i]),defpointA+i,sizeof(defpointrecord));
   }
   memcpy(&(unitstatus[u]->projection),&projection,sizeof(projectionrecord));
}                                    /* UnitStatusSave */


FUNCTION void UnitStatusRestore(int u)
{
   int i;

   TRACE CheckMgSystem(this_unit,"UnitStatusRestore",true,mytracelevel,"%d",u);
   if ((u<0) || (u>max_unit)) return;
   if (! (unitstatus[u]->defined)) return;
   symmetrical=unitstatus[u]->symmetrical;
   marginx=unitstatus[u]->marginx;
   marginy=unitstatus[u]->marginy;
   deltax=unitstatus[u]->deltax;
   deltay=unitstatus[u]->deltay;
   objectx=unitstatus[u]->objectx;
   objecty=unitstatus[u]->objecty;
   rmirror=unitstatus[u]->rmirror;
   extrafactor=unitstatus[u]->extrafactor;
   xyfactor=unitstatus[u]->xyfactor;
   globalalfa=unitstatus[u]->globalalfa;
   cosalfa=unitstatus[u]->cosalfa;
   sinalfa=unitstatus[u]->sinalfa;
   fillcolor=unitstatus[u]->fillcolor;
   imirror2=unitstatus[u]->imirror2;
   imirror20ps=unitstatus[u]->imirror20ps;
   for (i=0; i<=26; i++) {
      memcpy(defpointA+i,&(unitstatus[u]->defpointA[i]),sizeof(defpointrecord));
   }
   memcpy(&projection,&(unitstatus[u]->projection),sizeof(projectionrecord));
}                                    /* UnitStatusRestore */


static FUNCTION void DumpEnvironmentStatus(FILE * dumpfile,int u)
{
   int c;

   TRACE CheckMgSystem(this_unit,"DumpEnvironmentStatus",true,mytracelevel,"%d",u);
   fprintf(dumpfile,"Environment of unit %3d -------------------  ",u);
   if (!(unitstatus[u]->defined)) fprintf(dumpfile,"NOT ");
   fprintf(dumpfile,"defined\n");
   if (!(unitstatus[u]->defined || testmode)) return;
   fprintf(dumpfile,"   angle=%1.5f   factor=%1.5f   mirror=%1.5f   fillcolor=%d\n",
      unitstatus[u]->globalalfa,unitstatus[u]->xyfactor,unitstatus[u]->rmirror,unitstatus[u]->fillcolor);
   fprintf(dumpfile,"     %15s %15s %15s\n","margin","object","delta");
   fprintf(dumpfile,"   x %15.5f %15.5f %15.5f\n",unitstatus[u]->marginx,unitstatus[u]->objectx,unitstatus[u]->deltax);
   fprintf(dumpfile,"   y %15.5f %15.5f %15.5f\n",unitstatus[u]->marginy,unitstatus[u]->objecty,unitstatus[u]->deltay);
   fprintf(dumpfile,"Projection matrix:\n");
   fprintf(dumpfile,"%15.5f%15.5f\n",unitstatus[u]->projection.xx,unitstatus[u]->projection.xy);
   fprintf(dumpfile,"%15.5f%15.5f\n",unitstatus[u]->projection.yx,unitstatus[u]->projection.yy);
   fprintf(dumpfile,"%15.5f%15.5f\n",unitstatus[u]->projection.zx,unitstatus[u]->projection.zy);
   for (c='A'; c<='Z'; c++) {
      if (unitstatus[u]->defpointA[c-'A'].defined) {
         fprintf(dumpfile,"   Point[%c]= (%8.3f,%8.3f)\n",c,unitstatus[u]->defpointA[c-'A'].x,unitstatus[u]->defpointA[c-'A'].y);
      }
   }
   fprintf(dumpfile,"\n");
}                                    /* DumpEnvironmentStatus */


static FUNCTION void DumpDiagramStatus(FILE * dumpfile)
{
   int i;

   TRACE CheckMgSystem(this_unit,"DumpDiagramStatus",true,mytracelevel,NULL);
   fprintf(dumpfile,"\nDiagrams:\n");
   fprintf(dumpfile,".  %6c%6s%7c%5c%7s%5s%14s%14s%7s%4s\n",'d',"page",'x','y',"wdt","hgt","xmin/max","ymin/max","lines","len");
   fprintf(dumpfile,".  %6d%65d%4.0f\n",0,diagram0->lines,diagram0->maxtextlen);
   for (i=1; i<=lastdiagram; i++) {
      fprintf(dumpfile,".  %6d%6d%7d%5d%7d%5d%7.2f%7.2f%7.2f%7.2f %6d%4.0f\n",
         i,(diagram0+i)->page,(diagram0+i)->x,(diagram0+i)->y,(diagram0+i)->width,(diagram0+i)->heigth,
         (diagram0+i)->xdmin,(diagram0+i)->xdmax,(diagram0+i)->ydmin,(diagram0+i)->ydmax,(diagram0+i)->lines,(diagram0+i)->maxtextlen);
   }
}                                      /* DumpDiagramStatus */


FUNCTION FILE *DumpFileStart(void)
{
   int i;
   FILE *dumpfile;
   char *p;

   CHECKINIT CheckMgSystem(this_unit,"DumpFileStart",true,mytracelevel,NULL);
   dumpnumber++;
   if (dumpnumber==1) {
      dumpfile=fopen(dumpfilename,"w");
      fprintf(dumpfile,"Program: %s   %s   %s\n",ori_program_name,ori_program_date,ori_program_version);
      fprintf(dumpfile,"File: %s\n",globaldisk.filename);
   } else {
      dumpfile=fopen(dumpfilename,"a");
   }
   fprintf(dumpfile,"\n====== Dump %d ==========================\n",dumpnumber);
   fprintf(dumpfile,"Position in file:    line %d   position %d\n",globaldisk.linenumber,globaldisk.position);
   fprintf(dumpfile,"Actual line: %s\n",globaldisk.line);
   fprintf(dumpfile,"Diagram %d     Page %d\n",number,printpage);
   fprintf(dumpfile,"PaperX %d     PaperY %d    PaperFactor: %1.3f\n",paperx,papery,paperfactor);
   fprintf(dumpfile,"\n");
   UnitStatusSave(0,true);
   for (i=0; i<=max_unit; i++) DumpEnvironmentStatus(dumpfile,i);
   fprintf(dumpfile,"\nKeywords for IF:\n");
   fprintf(dumpfile,"\n");
   p=IfDefKeyString();
   if ((*p)==0) {
      fprintf(dumpfile,"   no keywords defined\n");
   } else {
      fprintf(dumpfile,"   %s\n",p);
   }
   DumpDiagramStatus(dumpfile);
   fprintf(dumpfile,"\nMaximum used:\n");
   fprintf(dumpfile,"   StrLength of a text line: %d\n",toptextlinestrlength);
   return dumpfile;
}                                      /* DumpFileStart */


