/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Supply PostScript procedures for ORITOPS
     First version dd.19940815   (Pascal)
     Converted to C   19950220
*/

#include "orilproc.h"

/*extracth Local */

#define last_revision "19990701"
#define this_unit "OriLpost"
#define prog_version "unit"
#define mytracelevel 5
#define maxlinesinbuffer 2000
#define psbufferfile "temptemp.ps"
static boolean initialized=false;
static FILE *psfile;


typedef char *pslinepointer;

typedef struct{
   maxstring now,fname,line;
   pslinepointer buffer[maxlinesinbuffer];
   int pslinenumber,maxbuffer,bufline,pages;
} postscriptstatusrecord;

static postscriptstatusrecord psstatus;
static double actualpen,previouspen;

/*extracth Functions */


FUNCTION void InitializeOriLPost(int *argc,char *argv[])
{
   if (initialized) return;
   initialized=true;
   InitializeOriLProc(argc,argv);
   CheckOriUpdate(last_revision);
   TRACE CheckMgSystem(this_unit,"InitializeOriLPost",true,mytracelevel,"Compiled at %s %s",__DATE__,__TIME__);
   psstatus.maxbuffer=1;
   psstatus.pslinenumber=0;
   psstatus.bufline=0;
   psstatus.pages=0;
   InitString(psstatus.line,"",-1);
   InitString(psstatus.fname,"",-1);
   InitString(psstatus.now,"",-1);
   psfile=NULL;
   actualpen=25.0;
   previouspen=-1.0;
}                                      /* InitializeOriLPost */


static FUNCTION void Ps_WriteBuffer(char *txt)
{
   TRACE CheckMgSystem(this_unit,"Ps_WriteBuffer",true,mytracelevel,NULL);
   StringCat(psstatus.line,-1,txt,0);
}                                      /* Ps_WriteBuffer */


static FUNCTION void Ps_WriteBufferDouble(double r,int fraction)
{
   maxstring txt;

   TRACE CheckMgSystem(this_unit,"Ps_WriteBufferDouble",true,mytracelevel,NULL);
   InitString(txt,"",-1);
   if ((r<=0.0) && (r>-0.5)) r=0.0;
   DoubleToString(txt,r,1,fraction);
   StringCat(psstatus.line,-1,txt,-1);
   StringCat(psstatus.line,-1," ",0);
}                                      /* Ps_WriteBufferDouble */


static FUNCTION void Ps_WriteLnBuffer(char *txt)
{
   FILE *buffile;
   int i;

   TRACE CheckMgSystem(this_unit,"Ps_WriteLnBuffer",true,mytracelevel,"%d",psstatus.pslinenumber+1);
   StringCat(psstatus.line,-1,txt,0);
   if (psstatus.bufline>=psstatus.maxbuffer) {
      printf("\r%70s\r%5d%c%6d   Buffer to file ...","",number,numberabc,psstatus.pslinenumber);
      if (psstatus.bufline==psstatus.pslinenumber) {
         buffile=fopen(psbufferfile,"w");
      } else {
         buffile=fopen(psbufferfile,"a");
      }
      for (i=0; i<psstatus.bufline; i++) fprintf(buffile,"%s\n",psstatus.buffer[i]);
      fclose(buffile);
      psstatus.bufline=0;
   }
   psstatus.pslinenumber++;
   InitString(psstatus.buffer[psstatus.bufline],psstatus.line,-1);
   psstatus.bufline++;
   if (toppslinestrlength<(int)strlen(psstatus.line)) toppslinestrlength=(int)strlen(psstatus.line);
   if (psstatus.pslinenumber % 32==0) {
      printf("\r%70s\r%5d%c%6d","",number,numberabc,psstatus.pslinenumber);
      if (testmode) printf("  (%d)",toppslinestrlength);
   }
   psstatus.line[0]=0;
}                                      /* Ps_WriteLnBuffer */


static FUNCTION void Ps_CopyBuffer(void)
{
   boolean lastwasblank;
   FILE *buffile;
   maxstring line;
   int i;

   TRACE CheckMgSystem(this_unit,"Ps_CopyBuffer",true,mytracelevel,NULL);
   InitString(line,"",-1);
   printf("\r%5d%c%6d",number,numberabc,psstatus.pslinenumber);
   if (testmode) printf("  (%d)",toppslinestrlength);
   printf("   Write to file...");
   if (toppslines<psstatus.pslinenumber) toppslines=psstatus.pslinenumber;
   lastwasblank=false;
   if (psstatus.bufline!=psstatus.pslinenumber) {
      buffile=fopen(psbufferfile,"r");
      while (ReadLn(buffile,line,-1)!=EOF) {
         StripBlanks(line);
         if (line[0]!=0) lastwasblank=false;
         if (! lastwasblank) fprintf(psfile,"%s\n",line);
         if (line[0]==0) lastwasblank=true;
      }
      fclose(buffile);
      DeleteFile(psbufferfile);
   }
   for (i=0; i<psstatus.bufline; i++) {
      StringCopy(line,-1,psstatus.buffer[i],-1);
      StripBlanks(line);
      if (line[0]!=0) lastwasblank=false;
      if (! lastwasblank) fprintf(psfile,"%s\n",line);
      if (line[0]==0) lastwasblank=true;
   }
   fprintf(psfile,"\n");
   psstatus.line[0]=0;
   psstatus.pslinenumber=0;
   psstatus.bufline=0;
}                                      /* Ps_CopyBuffer */


static FUNCTION void Ps_XyStrToString(char *result,double x,double y,char *txt,boolean strip)
{
   int i;

   TRACE CheckMgSystem(this_unit,"Ps_XyStrToString",true,mytracelevel,NULL);
   if (strip) StripBlanks(txt); else StripTrailing(txt);
   for (i=(int)strlen(txt)-1; i>=0; i--) {
      if (! isprint(txt[i]))            DelChars(txt,i,1);              else
      if (strchr("()\\",txt[i])!=NULL)  InsertString("\\",txt,i,-1);
   }
   if ((x<0.5) && (x>-0.5)) x=0.1;
   if ((y<0.5) && (y>-0.5)) y=0.1;
   if (txt[0]!=0) sprintf(result,"%1.0f %1.0f moveto (%s) SH",x,y,txt); else result[0]=0;
}                                      /* Ps_XyStrToString */


FUNCTION void PsWriteString(double x,double y,char *txt_value,boolean bold)
{
   maxstring result,txt;

   CHECKINIT CheckMgSystem(this_unit,"PsWriteString",true,mytracelevel,NULL);
   InitString(txt,txt_value,-1);
   InitString(result,"",-1);
   if (bold) Ps_WriteBuffer("SFB "); else Ps_WriteBuffer("SF ");
   Ps_XyStrToString(result,x,y,txt,true);
   Ps_WriteLnBuffer(result);
}                                      /* PsWriteString */


FUNCTION void PsWriteHeader(double x,double y,char *txt_value)
{
   maxstring result,txt;

   CHECKINIT CheckMgSystem(this_unit,"PsWriteHeader",true,mytracelevel,NULL);
   InitString(txt,txt_value,-1);
   if (txt[0]=='\\') {
      fprintf(psfile,"SFC ");
      DelChars(txt,0,1);
   } else {
      fprintf(psfile,"SF ");
   }
   Ps_XyStrToString(result,x,y,txt,false);
   fprintf(psfile,"%s\n",result);
}                                      /* PsWriteHeader */


FUNCTION void PsSetPenWidth(void)
{
   CHECKINIT CheckMgSystem(this_unit,"PsSetPenWidth",true,mytracelevel,NULL);
   actualpen=penwidth*25.0*penwidthfactor;
}                                      /* PsSetPenWidth */


static FUNCTION void Ps_CheckPenWidth(void)
{
   CHECKINIT CheckMgSystem(this_unit,"Ps_CheckPenWidth",true,mytracelevel,NULL);
   if (actualpen!=previouspen) {
      Ps_WriteBufferDouble(actualpen,2);
      Ps_WriteLnBuffer("SLW");
      previouspen=actualpen;
   }
}                                      /* Ps_CheckPenWidth */


FUNCTION void PsStartDiagramGraphics(void)
{
   CHECKINIT CheckMgSystem(this_unit,"PsStartDiagramGraphics",true,mytracelevel,NULL);
   psstatus.line[0]=0;
   if (psfile!=NULL) fprintf(psfile,"%% ====== Next diagram ======\n");
   previouspen=actualpen*2.0;
}                                      /* PsStartDiagramGraphics */


FUNCTION void PsWriteLine(double x1,double y1,double x2,double y2)
{
   CHECKINIT CheckMgSystem(this_unit,"PsWriteLine",true,mytracelevel,NULL);
   Ps_CheckPenWidth();
   Ps_WriteBufferDouble(x1,1);
   Ps_WriteBufferDouble(y1,1);
   Ps_WriteBufferDouble(x2,1);
   Ps_WriteBufferDouble(y2,1);
   Ps_WriteLnBuffer("LS");
}                                      /* PsWriteLine */


FUNCTION void PsWriteDashLine(double x1,double y1,double x2,double y2,double d1,double d2,double d3,double d4)
{
   CHECKINIT CheckMgSystem(this_unit,"PsWriteDashLine",true,mytracelevel,NULL);
   Ps_CheckPenWidth();
   Ps_WriteBufferDouble(x1,1);
   Ps_WriteBufferDouble(y1,1);
   Ps_WriteBufferDouble(x2,1);
   Ps_WriteBufferDouble(y2,1);
   Ps_WriteBuffer("[");
   Ps_WriteBufferDouble(d1,2);
   if (d2>=0) Ps_WriteBufferDouble(d2,2);
   if (d3>=0) Ps_WriteBufferDouble(d3,2);
   if (d4>=0) Ps_WriteBufferDouble(d4,2);
   Ps_WriteLnBuffer("] DSL");
}                                      /* PsWriteDashLine */


FUNCTION void PsWritePolyPoints(int fillindex,boolean draw,boolean fill)
{
   int i,j,io;
   double grayfactor;

   CHECKINIT CheckMgSystem(this_unit,"PsWritePolyPoints",true,mytracelevel,NULL);
   Ps_CheckPenWidth();
   Ps_WriteBufferDouble((polypoint0+1)->x/10.0,1);
   Ps_WriteBufferDouble((polypoint0+1)->y/10.0,1);
   Ps_WriteBuffer("NPMT ");
   i=2;
   io=1;
   while (i<=polypoints) {
      if (polypointw[i]=='B') {
         if (io>2) {
            Ps_WriteLnBuffer("");
            io=0;
         }
         for (j=0; j<3; j++) {
            Ps_WriteBufferDouble((polypoint0+i+j)->x/10.0,1);
            Ps_WriteBufferDouble((polypoint0+i+j)->y/10.0,1);
         }
         Ps_WriteBuffer("C");
         i+=3;
         io+=3;
      } else {
         Ps_WriteBufferDouble((polypoint0+i)->x/10.0,1);
         Ps_WriteBufferDouble((polypoint0+i)->y/10.0,1);
         Ps_WriteBuffer("LT");
         io++;
         i++;
      }
      if (io>4) {
         Ps_WriteLnBuffer("");
         io=0;
      } else {
         Ps_WriteBuffer(" ");
      }
   }
   if (fill) {
      Ps_WriteBuffer("closepath ");
      switch (fillindex) {
          case 0: grayfactor=1.0;     break;
          case 1: grayfactor=0.0;     break;
          case 2: grayfactor=0.25;    break;
          case 3: grayfactor=0.5;     break;
          case 4: grayfactor=0.75;    break;
          case 5: grayfactor=0.875;   break;
         default: grayfactor=0.0;
      }
      Ps_WriteBufferDouble(grayfactor,3);
      Ps_WriteBuffer("GRAY");
      if (draw) Ps_WriteBuffer(" FD"); else Ps_WriteBuffer(" F");
   } else {
      Ps_WriteBuffer(" stroke");
   }
   Ps_WriteLnBuffer("");
}                                      /* PsWritePolyPoints */


FUNCTION void PsWriteBezier(double x0,double y0,double x1,double y1,double x2,double y2,double x3,double y3,double d1,double d2)
{
   Ps_CheckPenWidth();
   CHECKINIT CheckMgSystem(this_unit,"PsWriteBezier",true,mytracelevel,"%f %f   %f %f   %f %f   %f %f   d1=%f  d2=%f",x0,y0,x1,y1,x2,y2,x3,y3,d1,d2);
   Ps_WriteBufferDouble(x0,1);
   Ps_WriteBufferDouble(y0,1);
   Ps_WriteBuffer("NPMT ");
   Ps_WriteBufferDouble(x1,1);
   Ps_WriteBufferDouble(y1,1);
   Ps_WriteBufferDouble(x2,1);
   Ps_WriteBufferDouble(y2,1);
   Ps_WriteBufferDouble(x3,1);
   Ps_WriteBufferDouble(y3,1);
   Ps_WriteBuffer("[");
   if (d1>0) {
      Ps_WriteBufferDouble(d1,2);
      Ps_WriteBufferDouble(d2,2);
   }
   Ps_WriteLnBuffer("] BS");
}                                      /* PsWriteBezier */


static FUNCTION void Ps_SetFont(double scale,boolean bold)
{
   CHECKINIT CheckMgSystem(this_unit,"Ps_SetFont",true,mytracelevel,NULL);
   if (scale>0) fprintf(psfile,"/FontSize %1.2f def ",scale);
   if (bold) fprintf(psfile,"SFB\n"); else fprintf(psfile,"SF\n");
}                                      /* Ps_SetFont */


FUNCTION void PsSetFillColor(int color)
{
   int r,g,b;

   CHECKINIT CheckMgSystem(this_unit,"PsSetFillColor",true,mytracelevel,NULL);
   switch (color) {
       case 0: r=0X3F;  g=0X3F;  b=0X3F;           /* Black */   break;
       case 1: r=0X2A;  g=0X15;  b=0X00;           /* Brown */   break;
       case 2: r=0X00;  g=0X00;  b=0X3F;           /* Blue  */   break;
       case 3: r=0X3F;  g=0X00;  b=0X3F;           /* Magenta */ break;
       case 4: r=0X3F;  g=0X00;  b=0X00;           /* Red   */   break;
       case 5: r=0X3F;  g=0X15;  b=0X00;           /* Orange */  break;
       case 6: r=0X3F;  g=0X3F;  b=0X00;           /* Yellow */  break;
       case 7: r=0X00;  g=0X3F;  b=0X00;           /* Green */   break;
       case 8: r=0X3F;  g=0X3F;  b=0X3F;           /* White */   break;
      default: r=0X00;  g=0X00;  b=0X3F;           /* Blue  */
   }
   Ps_WriteBufferDouble(1.0*r/0X3F,4);
   Ps_WriteBufferDouble(1.0*g/0X3F,4);
   Ps_WriteBufferDouble(1.0*b/0X3F,4);
   Ps_WriteLnBuffer("COLOR");
}                                      /* PsSetFillColor */


FUNCTION void PsWriteInitPage(int pagenr)
{
   CHECKINIT CheckMgSystem(this_unit,"PsWriteInitPage",initialized,mytracelevel,NULL);
   psstatus.pages=pagenr;
   if (pagenr>1) fprintf(psfile,"showpage\n\n");
   fprintf(psfile,"%%%%Page: %d %d\n",psstatus.pages,psstatus.pages);
   fprintf(psfile,"InitPage\n");
}                                    /* PsWriteInitPage */


static FUNCTION void Ps_StartFile(void)
{
   maxstring au,auth;

   TRACE CheckMgSystem(this_unit,"Ps_StartFile",true,mytracelevel,NULL);
   psfile=fopen(psstatus.fname,"w");
   fprintf(psfile,"%%!PS-Adobe-1.0\n");
   fprintf(psfile,"%%%%Creator: program ORITOPS (V.%s %s)\n",ori_program_version,ori_program_date);
   fprintf(psfile,"%% Program made by M.J.van Gelder, KVI / RUG, Groningen, Nederland\n");
   fprintf(psfile,"%% E-mail address: M.J.van.Gelder@kvi.nl\n");
   fprintf(psfile,"%%%%Title: %s\n",modelname);
   fprintf(psfile,"%%%%CreationDate:  %s\n",psstatus.now);
   fprintf(psfile,"%%%%BoundingBox: 0 0 594 842\n");
   fprintf(psfile,"%%%%Pages: 0000 \n");
   fprintf(psfile,"%%%%DocumentFonts: Courier\n");
   fprintf(psfile,"%%%%+ Helvetica\n");
   fprintf(psfile,"%%%%+ Helvetica-Bold\n");
   fprintf(psfile,"%%%%+ Symbol\n");
   fprintf(psfile,"%%%%DocumentNeededFonts: Courier\n");
   fprintf(psfile,"%%%%+ Helvetica\n");
   fprintf(psfile,"%%%%+ Helvetica-Bold\n");
   fprintf(psfile,"%%%%+ Symbol\n");
   fprintf(psfile,"%%%%EndComments\n");
   fprintf(psfile,"%%StartDef\n");
   fprintf(psfile,"/SH {1 -1 scale show 1 -1 scale} def\n");
   fprintf(psfile,"/S {stroke [] 0 setdash} def\n");
   fprintf(psfile,"/M {moveto} def\n");
   fprintf(psfile,"/LT {lineto} def\n");
   fprintf(psfile,"/LS {newpath 4 2 roll moveto LT S} def\n");
   fprintf(psfile,"/NPMT {newpath moveto} def\n");
   fprintf(psfile,"/DSL {0 setdash LS} def\n");
   fprintf(psfile,"/BS {0 setdash curveto S} def\n");
   fprintf(psfile,"/C {curveto} def\n");
   fprintf(psfile,"/F {fill 0 GRAY} def\n");
   fprintf(psfile,"/FD {gsave fill grestore 0 GRAY stroke} def\n");
   fprintf(psfile,"/SF {/Helvetica findfont FontSize FontFactor mul scalefont setfont} def\n");
   fprintf(psfile,"/SFB {/Helvetica-Bold findfont FontSize FontFactor mul scalefont setfont} def\n");
   fprintf(psfile,"/SFC {/Courier findfont FontSize FontFactor mul scalefont setfont} def\n");
   fprintf(psfile,"/InColor %3.2f def\n",saturation);
   fprintf(psfile,"%%  In the above line you may redefine the use of colors in your printer. \n");
   fprintf(psfile,"%%  A value of 0 means that your printer only prints black and white.\n");
   fprintf(psfile,"%%  If it supports colors you may change the value to a value larger than 0.5\n");
   fprintf(psfile,"%%  telling to use them.\n");
   fprintf(psfile,"%%  You may experiment with this value. A value of 1.25 gave nice colors on\n");
   fprintf(psfile,"%%  an HP ink jet printer. A larger value makes all colors brighter\n");
   fprintf(psfile,"/GRAY {\n");
   fprintf(psfile,"  /Gray exch def Gray 0.99 gt InColor 0.5 lt or {\n");
   fprintf(psfile,"    Gray setgray\n");
   fprintf(psfile,"  } {\n");
   fprintf(psfile,"    Gray InColor mul /Gray exch def \n");
   fprintf(psfile,"    Red Gray mul Green Gray mul Blue Gray mul setrgbcolor\n");
   fprintf(psfile,"  } ifelse\n");
   fprintf(psfile,"} def\n");
   fprintf(psfile,"/COLOR {/Blue exch def /Green exch def /Red exch def} def\n");
   fprintf(psfile,"0 0 1 COLOR  %% set color to Blue\n");
   fprintf(psfile,"/FontSize 9 def\n");
   fprintf(psfile,"/SLW {PenFactor mul setlinewidth} def\n");
   fprintf(psfile,"/PenFactor 1 def\n");
   fprintf(psfile,"%%  In the above line you may redefine the width of all lines in all\n");
   fprintf(psfile,"%%  diagrams via the variable PenFactor. The default value is 1.\n");
   fprintf(psfile,"%%  To double the width of all lines change the line to:\n");
   fprintf(psfile,"%%    /PenFactor 2 def\n");
   fprintf(psfile,"%%  The value may not exceed 10 and may not be lower than 0.1.\n");
   fprintf(psfile,"PenFactor 10 gt {/PenFactor 10 def} if\n");
   fprintf(psfile,"PenFactor 0.1 lt {/PenFactor 0.1 def} if\n");
   fprintf(psfile,"/Sqrt2 %1.5f def\n",sqrt(2.0));
   fprintf(psfile,"/PrintPage 0 def\n");
   fprintf(psfile,"/FontFactor Sqrt2 0.24 div def\n");
   fprintf(psfile,"/PaperType ");
   switch (paper) {
      case a4:    fprintf(psfile,"(A4)");   break;
      case a5:    fprintf(psfile,"(A5)");   break;
   }
   fprintf(psfile," def\n");
   fprintf(psfile,"%%  In the above line you may redefine the format of printing via the\n");
   fprintf(psfile,"%%  variable PaperType. You may use one of the following formats to print:\n");
   fprintf(psfile,"%%    (A4) A4 format\n");
   fprintf(psfile,"%%    (A5) A5 format\n");
   fprintf(psfile,"/InitPage {\n");
   fprintf(psfile,"  /PrintPage PrintPage 1 add def\n");
   fprintf(psfile,"  [] 0 setdash 0.5 setlinewidth 0 GRAY 1 setlinecap 1 setlinejoin\n");
   fprintf(psfile,"  PaperType  (A4) eq {22 815 translate} if\n");
   fprintf(psfile,"  PaperType  (A5) eq {22 815 translate} if\n");
   fprintf(psfile,"  1 FontFactor div\n");
   fprintf(psfile,"  PaperType (A4) eq {Sqrt2 mul} if\n");
   fprintf(psfile,"  dup neg scale\n");
   fprintf(psfile,"  0.975 dup scale /FontSize 9 def SF\n");
   fprintf(psfile,"} def\n");
   fprintf(psfile,"%%EndDef\n\n");
   PsWriteInitPage(1);
   Ps_SetFont(14.0,true);
   PsWriteHeader(0.0,40.0,modelname);
   Ps_SetFont(9.0,false);
   InitString(auth,author,-1);
   InitString(au,"",-1);
   papery= -10;
   do {
      papery+=50;
      SplitString(au,-1,auth,-1,"/");
      PsWriteHeader(paper_max_x/2.0+paper_max_x/26.0-0.5,papery*1.0,au);
   } while (auth[0]!=0);
   Ps_SetFont(9.0,false);
   fprintf(psfile,"%d (   %s) stringwidth pop sub 40 moveto\n",paper_max_x,modeldate);
   fprintf(psfile,"/Symbol findfont 37 scalefont setfont (\\343) SH\n");
   Ps_SetFont(9.0,false);
   fprintf(psfile," ( %s) SH\n",modeldate);
   Ps_SetFont(9.0,false);
   fprintf(psfile,"\n");
}                                      /* Ps_StartFile */


FUNCTION void PsOpenFile(char *orifilename)
{
   maxstring inpname;
   int i;

   CHECKINIT CheckMgSystem(this_unit,"PsOpenFile",true,mytracelevel,NULL);
   InitString(inpname,orifilename,-1);
   inpname[(int)strlen(inpname)-4]=0;
   InitString(psstatus.fname,"",-1);
   sprintf(psstatus.fname,"%s.ps",inpname);
   DateTime(psstatus.now);
   Ps_StartFile();
   for (i=0; i<maxlinesinbuffer; i++) {
      psstatus.buffer[i]=NULL;
      psstatus.buffer[i]=AllocString(-1);
      if (psstatus.buffer[i]==NULL) {
         psstatus.maxbuffer-=100;
         i--;
         while (i>psstatus.maxbuffer) {
            free(psstatus.buffer[i]);
            i--;
         }
         break;
      }
      psstatus.maxbuffer=i;
   }
}                                      /* PsOpenFile */


FUNCTION void PsCloseDiagram(int dotsperline,int wide,int high)
{
   int i;
   char numbertxt[20];

   CHECKINIT CheckMgSystem(this_unit,"PsCloseDiagram",true,mytracelevel,NULL);
   fprintf(psfile,"gsave %d %d translate\n",paperx,papery);
   if (withframe) {
      fprintf(psfile,"0 0 NPMT %d 0 LT %d %d LT 0 %d LT 0 0 LT stroke\n",wide,wide,-high,-high);
   }
   if ((xdrawmin<0.5) && (xdrawmin>-0.5)) xdrawmin= -0.1;
   if ((ydrawmax<0.5) && (ydrawmax>-0.5)) ydrawmax= -0.1;
   fprintf(psfile,"%1.2f %1.2f scale\n",paperfactor,paperfactor);
   fprintf(psfile,"%1.0f %1.0f translate\n",-xdrawmin,-ydrawmax);
   fprintf(psfile,"-2 -2 NPMT %1.0f -2 LT %1.0f ",xdrawmax*2+200,xdrawmax*2+200);
   fprintf(psfile,"%1.0f LT -2 %1.0f LT closepath clip\n",ydrawmax*2+200,ydrawmax*2+200);
   Ps_SetFont(9.0/paperfactor,false);
   Ps_CopyBuffer();
   fprintf(psfile,"grestore\n");
   Ps_SetFont(9.0,false);
   if (number!=0) {
      if (number>0) {
         sprintf(numbertxt,"%3d%c",number,numberabc);
         PsWriteHeader(paperx-25.0,papery+dotsperline*2.0,numbertxt);
      }
      for (i=1; i<=lastline; i++) PsWriteHeader(paperx+75.0,papery+dotsperline*(i+1.0),(char *)(textline0+i));
      if (lastline>paperplus) paperplus=lastline;
   }
   fprintf(psfile,"\n");
}                                      /* PsCloseDiagram */


FUNCTION void PsCloseFile(void)
{
   char buffer[500],*p;

   CHECKINIT CheckMgSystem(this_unit,"PsCloseFile",true,mytracelevel,NULL);
   if (psfile!=NULL) {
      fprintf(psfile,"showpage\n\n%%%%EndDocument: _\n");
      fclose(psfile);
      psfile=fopen(psstatus.fname,"r+");
      fread(&buffer,500,1,psfile);
      p=strstr(buffer,"0000");
      if (p!=NULL) {
         sprintf(p,"%4d",psstatus.pages);
         *(p+4)=' ';
         fseek(psfile,0,SEEK_SET);
         fwrite(&buffer,500,1,psfile);
      }
      fclose(psfile);
   }
   psfile=NULL;
   psstatus.fname[0]=0;
}                                      /* PsCloseFile */


FUNCTION void PsRemoveFile(void)
{
   CHECKINIT CheckMgSystem(this_unit,"PsRemoveFile",true,mytracelevel,NULL);
   if (psfile!=NULL) fclose(psfile);
   psfile=NULL;
   DeleteFile(psstatus.fname);
   psstatus.fname[0]=0;
}                                      /* PsRemoveFile */


