/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Supply graphical procedures for ORITOPS and ORIDRAW program sources
     First version dd.19940815   (Pascal)
     Converted to C   19950216
*/

/*GenMake off */
#include "orilxxxx.h"
/*GenMake on */

/*extracth Local */

#define last_revision "19990701"
#define this_unit "OriLdraw"
#define prog_version "unit"
#define mytracelevel 4
#define bezier_minimum_default 0.7854
static boolean initialized=false;
static double bezier_minimum=bezier_minimum_default;


/*extracth Functions */

FUNCTION void InitializeOriLDraw(int *argc,char *argv[])
{
   if (initialized) return;
   initialized=true;
   InitializeOriLXxxx(argc,argv);
   CheckOriUpdate(last_revision);
   TRACE CheckMgSystem(this_unit,"InitializeOriLDraw",true,mytracelevel,"Compiled at %s %s",__DATE__,__TIME__);
}                                      /* InitializeOriLDraw */


FUNCTION void PointCrossLines(double *x,double *y,double x2,double y2,double x3,double y3,double x4,double y4)
{
   double x1,y1,y21,y43,x21,x43,x43y21,x21y43,divisor;

   TRACE CheckMgSystem(this_unit,"PointCrossLines",true,mytracelevel,NULL);
   x1= *x;
   y1= *y;
   y21=y2-y1;
   y43=y4-y3;
   x21=x2-x1;
   x43=x4-x3;
   x43y21=x43*y21;
   x21y43=x21*y43;
   divisor=x43y21-x21y43;
   if (fabs(divisor)<=0.0001) {
      ShowError("PointCrossLines: Lines are parallel");
      *x=(x1+x2+x3+x4)*0.25;
      *y=(y1+y2+y3+y4)*0.25;
   } else {
      divisor=1/divisor;
      *y=((x1-x3)*y21*y43-y1*x21y43+y3*x43y21)*divisor;
      *x=((y3-y1)*x21*x43-x3*x21y43+x1*x43y21)*divisor;
   }
}                                      /* PointCrossLines */


FUNCTION void PointBisectrice(double *x,double *y,double x1,double y1,double x2,double y2)
{
   double a,b;

   TRACE CheckMgSystem(this_unit,"PointBisectrice",true,mytracelevel,NULL);
   a=Distance(*x,*y,x1,y1);
   b=Distance(x1,y1,x2,y2);
   a=a/(a+b);
   *x=(x2-(*x))*a+(*x);
   *y=(y2-(*y))*a+(*y);
}                                      /* PointBisectrice */


FUNCTION void PointMirrorInLine(double *x,double *y,double x1,double y1,double x2,double y2)
{
   double x3,y3;

   TRACE CheckMgSystem(this_unit,"PointMirrorInLine",true,mytracelevel,NULL);
   x3= *x+y2-y1;
   y3= *y+x1-x2;
   PointCrossLines(&x1,&y1,x2,y2,x3,y3,*x,*y);
   *x=2*x1-*x;
   *y=2*y1-*y;
}                                      /* PointMirrorInLine */


FUNCTION void PointTurn(double *x,double *y,double angle)
{
   double ca,sa,z;

   TRACE CheckMgSystem(this_unit,"PointTurn",true,mytracelevel,NULL);
   ca=cos(angle*pidiv180);
   sa=sin(angle*pidiv180);
   z=(*x)*ca-(*y)*sa;
   *y=(*x)*sa+(*y)*ca;
   *x=z;
}                                      /* PointTurn */


FUNCTION void PointPlus(double *x,double *y,double dx,double dy)
{
   TRACE CheckMgSystem(this_unit,"PointPlus",true,mytracelevel,NULL);
   *x+=dx;
   *y+=dy;
}                                      /* PointPlus */


FUNCTION void PointTimes(double *x,double *y,double factor)
{
   TRACE CheckMgSystem(this_unit,"PointTimes",true,mytracelevel,NULL);
   (*x)*=factor;
   (*y)*=factor;
}                                      /* PointTimes */


static FUNCTION void DrawToPoint2(double x1,double y1,double x2,double y2,boolean draw)
{
   CHECKINIT CheckMgSystem(this_unit,"DrawToPoint2",true,mytracelevel,NULL);
   PointToPoly(x1,y1,' ');
   PointToPoly(x2,y2,' ');
   if (draw) DrawPolyPoints(0,true,false);
}                                      /* DrawToPoint2 */


static FUNCTION void DrawToPoint3(double x1,double y1,double x2,double y2,double x3,double y3,boolean draw)
{
   CHECKINIT CheckMgSystem(this_unit,"DrawToPoint3",true,mytracelevel,NULL);
   PointToPoly(x1,y1,' ');
   DrawToPoint2(x2,y2,x3,y3,draw);
}                                      /* DrawToPoint3 */


static FUNCTION void DrawToPoint4(double x1,double y1,double x2,double y2,double x3,double y3,double x4,double y4,boolean draw)
{
   CHECKINIT CheckMgSystem(this_unit,"DrawToPoint4",true,mytracelevel,NULL);
   DrawToPoint2(x1,y1,x2,y2,false);
   DrawToPoint2(x3,y3,x4,y4,draw);
}                                      /* DrawToPoint4 */


FUNCTION void DrawBezierCurveInStyle(double x0,double y0,double x1,double y1,double x2,double y2,double x3,double y3,char kar2)
{
   double xps0,yps0,xps3,yps3,penfactorwas;
   int xp0,xp3,yp0,yp3;
   boolean dotted;

   CHECKINIT CheckMgSystem(this_unit,"DrawBezierCurveInStyle",true,mytracelevel,NULL);
   penfactorwas=penwidthfactor;
   ConvertXY(x3,y3,&xp3,&yp3,&xps3,&yps3);
   ConvertXY(x0,y0,&xp0,&yp0,&xps0,&yps0);
   CheckXyDraw(xps3,yps3);
   CheckXyDraw(xps0,yps0);
   if ((kar2=='B') && (penwidthfactor<1.5)) {
      penwidthfactor=2.5;
      SetPenWidth();
   }
   dotted=(kar2=='D');
   if (addtopoly) dotted=false; else StartStopUnit(true);
   if (strchr("DB ",kar2)!=NULL) {
      DrawBezierCurve(x0,y0,x1,y1,x2,y2,x3,y3,deltad,dotted);
   } else {
      ShowError("BezierCurve style (%c) %s",kar2,illegal);
   }
   if (! addtopoly) StartStopUnit(false);
   if (penwidthfactor!=penfactorwas) {
      penwidthfactor=penfactorwas;
      SetPenWidth();
   }
}                                      /* DrawBezierCurveInStyle */


static FUNCTION void FindArrowOrigin(double xp,double yp,double xe,double ye,double alfa,double da,double *xq,double *yq,double k,boolean verybended)
{
   double dx,dy,d,sd,cd;

   TRACE CheckMgSystem(this_unit,"FindArrowOrigin",true,mytracelevel,"p: %f %f  e: %f %f  alfa=%f  da=%f",xp,yp,xe,ye,alfa,da);
   d=da/Distance(xp,yp,xe,ye);
   dx=(xe-xp);
   dy=(ye-yp);
   sd=sin(d);
   cd=cos(d);
   if (k<0) sd=-sd;
   if (verybended) sd=-sd;
   xe=xp-dy*sd+dx*cd;
   ye=yp+dx*sd+dy*cd;
   d=Distance(xp,yp,xe,ye);
   dx=(xe-xp);
   dy=(ye-yp);
   if (alfa<0) {
      dx=-dx;
      dy=-dy;
   }
   (*xq)=xe-dy;
   (*yq)=ye+dx;
}                                    /* FindArrowOrigin */


FUNCTION void DrawArrowHead(double xa,double ya,double xe,double ye,double alfa,double dltd,char arrowstyle)
{
   double dx,dy,xb,yb,xc,yc,xd,yd,wfactor,f;
   string80 debug;
   int fillindex;
   string20 ahtxt;

   CHECKINIT CheckMgSystem(this_unit,"DrawArrowHead",true,mytracelevel,"a: %f %f  e: %f %f  alfa=%f  dltd=%f  style=%c",xa,ya,xe,ye,alfa,dltd,arrowstyle);
   InitString(debug,"",80);
   InitString(ahtxt,"ArrowHead x",20);
   ahtxt[10]=arrowstyle;
   f=2.5*dltd/Distance(xa,ya,xe,ye);
   dx=(xe-xa)*f;
   dy=(ye-ya)*f;
   xa=xe-dx;
   ya=ye-dy;
   switch (arrowstyle) {
      case 'M':  wfactor=0.3;   break;
      case 'O':  wfactor=0.25;  break;
      case 'V':  wfactor=0.2;   break;
      default: wfactor=0.25;
   }
   xb=xa-dy*wfactor;
   yb=ya+dx*wfactor;
   xc=xa+dy*wfactor;
   yc=ya-dx*wfactor;
   xd=xe-dx*0.8;
   yd=ye-dy*0.8;
   fillindex=-1;
   InitPoly(ahtxt);
   if (arrowstyle=='O') {
      DrawToPoint3(xb,yb,xe,ye,xc,yc,false);
      fillindex=0;
   } else if (arrowstyle=='M') {
      if (alfa>0) {
         DrawToPoint3(xd,yd,xb,yb,xe,ye,false);
      } else {
         DrawToPoint3(xd,yd,xc,yc,xe,ye,false);
      }
      fillindex=0;
   } else if (arrowstyle=='V') {
      DrawToPoint3(xb,yb,xe,ye,xc,yc,false);
      fillindex=1;
   } else {
      ShowError("Arrow style (%c) %s",arrowstyle,illegal);
   }
   if (fillindex>=0) DrawPolyPoints(fillindex,true,fillindex==1);
}                                      /* DrawArrowHead */


static FUNCTION void DrawArcViaBezier(double xp,double yp,double x0,double y0,double alfa2,double dltd,boolean dotted)
{
   double x1,y1,x2,y2,x3,y3,delta,dalfa,dalfagrad;
   int n,right;

   CHECKINIT CheckMgSystem(this_unit,"DrawArcViaBezier",initialized,mytracelevel,"0: %f %f  alfa2=%f  dltd=%f dotted=%d",xp,yp,alfa2,dltd,dotted);
   x0-=xp;
   y0-=yp;
   right=-1;
   if (alfa2<0) right=-right;
   dalfa=bezier_minimum;
   if (dotted) n=1; else n=(int)(fabs(alfa2/dalfa))+1;
   dalfa=-alfa2/n;
   dalfagrad=dalfa/pidiv180;
   delta=fabs(4.0*(1-cos(dalfa*0.5))/(3.0*sin(dalfa*0.5)));
   CHECKINIT CheckMgSystem(this_unit,"DrawArcViaBezier",initialized,mytracelevel,"dalfa=%f dalfagrad=%f right=%f  delta=%f n=%d",dalfa,dalfagrad,right,delta,n);
   CHECKINIT CheckMgSystem(this_unit,"DrawArcViaBezier",initialized,mytracelevel,"cos-dalfa=%f sin-dalfa=%f",cos(dalfa),sin(dalfa));
   x3=x0;
   y3=y0;
   PointTurn(&x3,&y3,dalfagrad);
   x1=-y0*delta*right+x0;
   y1=x0*delta*right+y0;
   x2=y3*delta*right+x3;
   y2=-x3*delta*right+y3;
   if (! addtopoly) StartStopUnit(true);
   while (n>0) {
      DrawBezierCurve(x0+xp,y0+yp,x1+xp,y1+yp,x2+xp,y2+yp,x3+xp,y3+yp,dltd,dotted);
      PointTurn(&x0,&y0,dalfagrad);
      PointTurn(&x1,&y1,dalfagrad);
      PointTurn(&x2,&y2,dalfagrad);
      PointTurn(&x3,&y3,dalfagrad);
      n--;
   }
   if (! addtopoly) StartStopUnit(false);
}                                    /* DrawArcViaBezier */


/*OriDoc ArcArrow
.PAGE Arc commands
In the A command for drawing an arc the characters just following the A tell
if and how arrow heads should be drawn and how the arc should be drawn.

The command should look like:

.SEL DRAW- TOPS- HLP-
.C Asn x1 y1 x2 y2 curv
Style s: one of OMV and/or one of LD or empty; # of arrow heads n: one of 012 or empty
.SEL ALL
   Asn x1 y1 x2 y2 curv        with
     n ---> # of arrowheads:  1, 2 or 0
    s ----> style, a combination of one of OMV and/or one of LD
   The style options fall in two sets:
      arrow head: one of O (open), M (mountain) or V (valley)
      linestyle: one of L (line), D (dotted)
   The default when no options are mentioned is:
      ALV1 . . . . .   [ bold line and 1 closed arrowhead ]

Examples of possible combinations:

   ALO1 x1 y1 x2 y2 curvature    ---->   Line arc with 1 arrowhead
   A0 x1 y1 x2 y2 curvature      -----   Line arc without arrowheads
   A  x1 y1 x2 y2 curvature      ---->   As ALV1 ...
   AM x1 y1 x2 y2 curvature      ___\>   As A1 with mountain fold arrow
   AD0 x1 y1 x2 y2 curvature     .....   Dotted arc without arrowheads
   AD1 x1 y1 x2 y2 curvature     ....>   Dotted arc with 1 arrowhead
   ADV x1 y1 x2 y2 curvature     ...|>   As AD1 with valley fold arrow


If the arc has one or two arrowheads the arc is drawn bold.
If the arc has no arrowheads the line is drawn in the actual line style:
bold after a P.B command; normal after P.N command.
.
The curvature should be between -4 and +4.
.
A value of 0 renders a (nearly) straight line.
.
A curvature of 2 or -2 renders half a circle.
.
The bigger the curvature, the bigger the arc will be. When you draw two arcs
with 4 as difference between the curvatures (p.e. 3 and -1) the two arcs will
make a circle together. Example:

   a0 5 5 5 10 3   a0 5 5 5 10 -1
*/


FUNCTION void DrawArcArrowInStyle(double x1,double y1,double x2,double y2,double k,double dltd,char *options)
{
   double r,xp,yp,xq,yq,d,alfa,penfactorwas;
   int i,heads;
   char arrowstyle;
   boolean verybended,dotted,thick_arrow;

   CHECKINIT CheckMgSystem(this_unit,"DrawArcArrowInStyle",true,mytracelevel,"1: %f %f  2: %f %f  k=%f  dltd=%f  options=%s",x1,y1,x2,y2,k,dltd,options);
   dotted=false;
   thick_arrow=true;
   heads=1;
   arrowstyle=' ';
   penfactorwas=penwidthfactor;
   for (i=0; options[i]!=0; i++) {
      if (strchr("012",options[i])!=NULL)   heads=options[i]-'0';      else
      if (strchr("LD",options[i])!=NULL)    dotted=options[i]=='D';    else
      if (strchr("OMV",options[i])!=NULL)   arrowstyle=options[i];     else
      if (strchr("S",options[i])!=NULL)     thick_arrow=false;         else
      {
         ShowError("Attribute for curved line (%c) %s",options[i],illegal);
      }
   }
   if ((heads!=0) && (arrowstyle==' ')) arrowstyle='V';
   if ((heads==0) && (arrowstyle!=' ')) heads=1;
   if (addtopoly && (heads!=0)) ClosePoly("Arrow with head(s)");
   if (dotted) thick_arrow=false;
   if ((heads!=0) && (penwidthfactor<1.5) && thick_arrow) {
      penwidthfactor=2.5;
      SetPenWidth();
   }
   verybended=false;
   if (fabs(k)>4) {
      ShowError("Curvature should be between -4 and +4");
      k=0;
   }
   if (fabs(k)>2) {
      if (k>=(2+nearly_two))    k= -curvature_minimum;    else
      if (k<=(-2-nearly_two))   k=curvature_minimum;      else
      if (k<0)                  k=k+4;                    else k=k-4;
      verybended=true;
   }
   if (k<-nearly_two) k= -nearly_two;
   if (k>nearly_two) k=nearly_two;
   d=Distance(x1,y1,x2,y2);
   if (! addtopoly) StartStopUnit(true);
   if ((d<0.001)  || ((heads>0) && (d*xyfactor<0.2))) {
      ShowError("Distance between points (%f %f) and (%f %f) is too small",x1,y1,x2,y2);
      DrawLine(x1,y1,x2,y2);
      return;
   }
   if (fabs(k)<curvature_minimum) k=curvature_minimum*0.99;
   r=d/k;
   alfa=sqrt(1/(k*k)-0.25);
   if (k<0) alfa= -alfa;
   xp=(x1+x2)*0.5-alfa*(y1-y2);
   yp=(y1+y2)*0.5-alfa*(x2-x1);
   alfa=2*atan(0.5/alfa);
   if (fabs(k)<=curvature_minimum) {
      if (dotted) DrawDottedLine(x1,y1,x2,y2,dltd*0.5,d); else DrawLine(x1,y1,x2,y2);
      if (heads>0) {
         d=d*0.15/xyfactor;
         if (d<dltd) dltd=d;
         DrawArrowHead(x1,y1,x2,y2,alfa,dltd,arrowstyle);
         if (heads>1) DrawArrowHead(x2,y2,x1,y1,-alfa,dltd,arrowstyle);
      }
   } else {
      if (verybended) {
         if (k>0) alfa=alfa-pitimes2; else alfa=alfa+pitimes2;
      }
      DrawArcViaBezier(xp,yp,x1,y1,alfa,dltd,dotted);
      if (heads>0) {
         d=fabs(r*alfa)*0.125;
         CHECKINIT CheckMgSystem(this_unit,"DrawArcArrowInStyle",true,mytracelevel,"d=%f  dltd=%f  r=%f  alfa=%f  xyfactor=%f",d,dltd,r,alfa,xyfactor);
         if (d<dltd) dltd=d;
         d=dltd*4/r;
         if (alfa<0) d=-d;
         CHECKINIT CheckMgSystem(this_unit,"DrawArcArrowInStyle",true,mytracelevel,"d=%f  dltd=%f  r=%f  alfa=%f  xyfactor=%f",d,dltd,r,alfa,xyfactor);
         FindArrowOrigin(xp,yp,x2,y2,alfa,dltd,&xq,&yq,k,verybended);
         DrawArrowHead(xq,yq,x2,y2,alfa,dltd,arrowstyle);
         if (heads>1) {
            FindArrowOrigin(xp,yp,x1,y1,-alfa,-dltd,&xq,&yq,k,verybended);
            DrawArrowHead(xq,yq,x1,y1,-alfa,dltd,arrowstyle);
         }
      }
   }
   if (! addtopoly) StartStopUnit(false);
   if (penwidthfactor!=penfactorwas) {
      penwidthfactor=penfactorwas;
      SetPenWidth();
   }
}                                      /* DrawArcArrowInStyle */


FUNCTION void DrawEllips(double x0,double y0,double xr,double yr,double angle)
{
   double x[4],y[4];
   int i;

   CHECKINIT CheckMgSystem(this_unit,"DrawEllips",true,mytracelevel,NULL);
   xr=xr*4.0/3.0;
   x[0]=0.0;
   y[0]=yr;
   x[1]=xr;
   y[1]=yr;
   x[2]=xr;
   y[2]=-yr;
   x[3]=0.0;
   y[3]=-yr;
   for (i=0; i<4; i++) {
      PointTurn(&x[i],&y[i],angle);
      x[i]+=x0;
      y[i]+=y0;
   }
   if (! addtopoly) StartStopUnit(true);
   DrawBezierCurve(x[0],y[0],x[1],y[1],x[2],y[2],x[3],y[3],deltad,false);
   x[1]=-xr;
   y[1]=yr;
   x[2]=-xr;
   y[2]=-yr;
   for (i=1; i<3; i++) {
      PointTurn(&x[i],&y[i],angle);
      x[i]+=x0;
      y[i]+=y0;
   }
   DrawBezierCurve(x[3],y[3],x[2],y[2],x[1],y[1],x[0],y[0],deltad,false);
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawEllips */


FUNCTION void DrawRepeatArrow(double x1,double y1,double xp,double yp,char kar2,char *txt)
{
   double d,dx,dy;
   int i,rep;

   CHECKINIT CheckMgSystem(this_unit,"DrawRepeatArrow",true,mytracelevel,NULL);
   rep=1;
   if (strchr("23456789",kar2)!=NULL)   rep=kar2-'0';         else
   if ((kar2>='A') && (kar2<='I'))      rep=kar2-'A'+1;
   d=Distance(x1,y1,xp,yp)*xyfactor*0.5;
   if (d<0.001) {
      ShowError("Distance between points (%f %f) and (%f %f) is too small",x1,y1,xp,yp);
      return;
   }
   ClosePoly("RepeatArrow (lines)");
   d=0.8/d;
   x1=xp+(x1-xp)*d;
   y1=yp+(y1-yp)*d;
   dx= (y1-yp)*0.2;
   dy= -(x1-xp)*0.2;
   if (! addtopoly) StartStopUnit(true);
   for (i=1; i<=rep; i++) {
      DrawLine(x1-dx,y1-dy,x1+dx,y1+dy);
      x1=x1-dy*1.25;
      y1=y1+dx*1.25;
   }
   DrawArcArrowInStyle(x1,y1,xp,yp,0,0.5*deltad,"OS");
   if ((kar2>='A') && (kar2<='I')) DrawString(x1-dy-dx,y1+dx-dy,txt,false);
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawRepeatArrow */


FUNCTION void DrawZigZagFold(double x1,double y1,double x2,double y2,char kar2)
{
   double d,dx,dy;

   CHECKINIT CheckMgSystem(this_unit,"DrawZigZagFold",true,mytracelevel,NULL);
   d=Distance(x1,y1,x2,y2)*xyfactor;
   if (d<0.001) {
      ShowError("Distance between points (%f %f) and (%f %f) is too small",x1,y1,x2,y2);
      return;
   }
   d=2.0/d;
   x1=x2+(x1-x2)*d;
   y1=y2+(y1-y2)*d;
   dx= (y1-y2)*0.1;
   dy= -(x1-x2)*0.1;
   if (kar2=='/') {
      dx= -dx;
      dy= -dy;
   }
   if (! addtopoly) StartStopUnit(true);
   DrawToPoint3(x1,y1,x1*0.4+x2*0.6-dx,y1*0.4+y2*0.6-dy,x1*0.6+x2*0.4+dx,y1*0.6+y2*0.4+dy,true);
   DrawArcArrowInStyle(x1*0.6+x2*0.4+dx,y1*0.6+y2*0.4+dy,x2,y2,0,0.5*deltad,"OS");
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawZigZagFold */


FUNCTION void DrawRollOverFold(double x1,double y1,double x2,double y2)
{
   double d,dx,dy,dxr,dyr;

   CHECKINIT CheckMgSystem(this_unit,"DrawRollOverFold",true,mytracelevel,NULL);
   d=Distance(x1,y1,x2,y2)*xyfactor;
   if (d<0.001) {
      ShowError("Distance between points (%f %f) and (%f %f) is too small",x1,y1,x2,y2);
      return;
   }
   d=1/d;
   dx=(x2-x1)*d;
   dy=(y2-y1)*d;
   dxr=dx*0.15;
   dyr=dy*0.15;
   ClosePoly("RollOver");
   bezier_minimum=pi;
   if (! addtopoly) StartStopUnit(true);
   DrawArcArrowInStyle(x2-3*dx,y2-3*dy,x2-2*dx-dyr,y2-2*dy+dxr,0.8,0.1,"0");
   DrawArcArrowInStyle(x2-2*dx-dyr,y2-2*dy+dxr,x2-2*dx+dyr,y2-2*dy-dxr,2.0,0.5,"0");
   DrawArcArrowInStyle(x2-2*dx+dyr,y2-2*dy-dxr,x2-2*dx-dyr,y2-2*dy+dxr,2.0,0.5,"0");
   DrawArcArrowInStyle(x2-2*dx-dyr,y2-2*dy+dxr,x2-dx-dyr,y2-dy+dxr,0.7,0.1,"0");
   DrawArcArrowInStyle(x2-dx-dyr,y2-dy+dxr,x2-dx+dyr,y2-dy-dxr,2.0,0.5,"0");
   DrawArcArrowInStyle(x2-dx+dyr,y2-dy-dxr,x2-dx-dyr,y2-dy+dxr,2.0,0.5,"0");
   DrawArcArrowInStyle(x2-dx-dyr,y2-dy+dxr,x2,y2,1.0,0.1,"0");
   bezier_minimum=bezier_minimum_default;
   DrawToPoint3(x2-2*(dxr+dyr),y2-2*(dyr-dxr),x2,y2,x2-2.6*dxr-0.5*dyr,y2-2.6*dyr+0.5*dxr,true);
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawRollOverFold */


FUNCTION void DrawAnEye(double x1,double y1,double x2,double y2)
{
   double d,dx,dy;

   CHECKINIT CheckMgSystem(this_unit,"DrawAnEye",true,mytracelevel,NULL);
   d=Distance(x1,y1,x2,y2)*xyfactor;
   if (d<0.001) {
      ShowError("Distance between points (%f %f) and (%f %f) is too small",x1,y1,x2,y2);
      return;
   }
   d=2.0/d;
   x1=x2+(x1-x2)*d;
   y1=y2+(y1-y2)*d;
   dx= (y1-y2)*0.3;
   dy= -(x1-x2)*0.3;
   if (! addtopoly) StartStopUnit(true);
   DrawToPoint3(x2+dx,y2+dy,x1,y1,x2-dx,y2-dy,true);
   InitPoly("");
   x2=x1+(x2-x1)*0.8;
   y2=y1+(y2-y1)*0.8;
   dx= (y1-y2)*0.3;
   dy= -(x1-x2)*0.3;
   bezier_minimum=pi;
   DrawArcArrowInStyle(x2+dx,y2+dy,x2-dx,y2-dy,1.0,0.1,"0");
   DrawArcArrowInStyle(x2-dx,y2-dy,x2+dx,y2+dy,1.0,0.1,"0");
   DrawPolyPoints(1,true,true);
   bezier_minimum=bezier_minimum_default;
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawAnEye */


static FUNCTION void DrawDouble(double x,double y,double k)
{
   maxstring txt;

   TRACE CheckMgSystem(this_unit,"DrawDouble",true,mytracelevel,NULL);
   DoubleToString(txt,k,1,0);
   DrawString(x,y,txt,true);
}                                      /* DrawDouble */


FUNCTION void DrawTurnOver(void)
{
   CHECKINIT CheckMgSystem(this_unit,"DrawTurnOver",true,mytracelevel,NULL);
   SetFactor(1.0);
   ClosePoly("TurnOver");
   if (! addtopoly) StartStopUnit(true);
   DrawBezierCurve(0.2,1,0.25,-0.25,1.5,-0.5,1.8,-0.1,deltad,false);
   DrawBezierCurve(1.8,-0.1,3.7,1.5,0.1,1.5,1.8,-0.1,deltad,false);
   DrawBezierCurve(1.8,-0.1,2.4,-0.5,3.1,-0.35,3.3,0.2,deltad,false);
   DrawToPoint2(3.3,0.2,3.5,0.75,true);
   DrawToPoint3(3.1,0.2,3.5,0.75,3.45,0.0,true);
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawTurnOver */


FUNCTION void DrawEnlarge(void)
{
   CHECKINIT CheckMgSystem(this_unit,"DrawEnlarge",true,mytracelevel,NULL);
   if (! addtopoly) StartStopUnit(true);
   DrawToPoint4(1.0,0.5,0.0,0.5,0.0,1.25,1.0,1.25,false);
   DrawToPoint3(1.0,1.25,1.0,1.75,1.875,0.875,false);
   DrawToPoint2(1.0,0.0,1.0,0.5,true);
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawEnlarge */


FUNCTION void DrawUnfold(double k)
{
   CHECKINIT CheckMgSystem(this_unit,"DrawUnfold",true,mytracelevel,NULL);
   if (! addtopoly) StartStopUnit(true);
   DrawToPoint3(0.1,0.0,0.5,0.3,0.0,0.45,true);
   bezier_minimum=pi;
   DrawArcArrowInStyle(0.0,0.45,1.0,0.45,0.6,0.1,"0");
   DrawArcArrowInStyle(1.0,0.45,1.6,0.45,-0.9,0.1,"0");
   DrawToPoint3(1.6,0.45,1.5,0.7,2.1,0.5,false);
   DrawToPoint2(1.55,0.0,1.6,0.25,true);
   DrawArcArrowInStyle(1.6,0.25,0.9,0.1,0.6,0.1,"0");
   DrawArcArrowInStyle(0.9,0.1,0.1,0.0,-0.9,0.1,"0");
   if (k>0) DrawDouble(2.5,0.4,k);
   bezier_minimum=bezier_minimum_default;
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawUnfold */


FUNCTION void DrawFoldInsideOut(void)
{
   CHECKINIT CheckMgSystem(this_unit,"DrawFoldInsideOut",true,mytracelevel,NULL);
   marginy=marginy+0.6;
   ClosePoly("xxx");
   if (! addtopoly) StartStopUnit(true);
   DrawLine(0.7,0.0,1.3,0.0);
   DrawLine(0.7,0.04,1.3,0.04);
   bezier_minimum=pi;
   DrawArcArrowInStyle(1.0,0.0,2.0,-0.3,2.0,0.15,"0");
   DrawArcArrowInStyle(2.0,-0.3,1.0,-0.5,2.0,0.1,"0");
   DrawArcArrowInStyle(1.0,0.0,0.0,-0.3,-2.0,0.15,"0");
   DrawArcArrowInStyle(0.0,-0.3,1.0,-0.5,-2.0,0.1,"0");
   DrawToPoint3(1.8,0.0,2.0,-0.3,2.2,0.0,true);
   DrawToPoint3(0.2,0.0,0.0,-0.3,-0.2,0.0,true);
   DrawLine(1.0,-0.5,1.0,-0.15);
   DrawToPoint3(0.8,-0.6,1.0,-0.15,1.2,-0.6,true);
   bezier_minimum=bezier_minimum_default;
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawFoldInsideOut */


FUNCTION void DrawRotate(double k)
{
   CHECKINIT CheckMgSystem(this_unit,"DrawRotate",true,mytracelevel,NULL);
   if (! addtopoly) StartStopUnit(true);
   DrawToPoint3(0.0,1.0,0.0,-0.5,1.5,-0.5,true);
   bezier_minimum=pi;
   DrawArcArrowInStyle(0.0,0.5,1.0,-0.5,1.4,0.05,"0");
   DrawToPoint3(0.75,-0.2,1.0,-0.5,1.15,-0.2,true);
   DrawDouble(1.6,0.0,k);
   bezier_minimum=bezier_minimum_default;
   if (! addtopoly) StartStopUnit(false);
}                                      /* DrawRotate */


FUNCTION void DrawLineInStyle(double x1,double y1,double x2,double y2,char kar2)
{
   double r,xps1,yps1,xps2,yps2,penfactorwas;
   int xp1,xp2,yp1,yp2;

   CHECKINIT CheckMgSystem(this_unit,"DrawLineInStyle",true,mytracelevel,NULL);
   penfactorwas=penwidthfactor;
   ConvertXY(x2,y2,&xp2,&yp2,&xps2,&yps2);
   ConvertXY(x1,y1,&xp1,&yp1,&xps1,&yps1);
   CheckXyDraw(xps2,yps2);
   CheckXyDraw(xps1,yps1);
   r=Distance(x1,y1,x2,y2);
   if ((strchr("BVM",kar2)!=NULL) && (penwidthfactor<1.5)) {
      penwidthfactor=2.5;
      SetPenWidth();
   }
   if (kar2!=' ') ClosePoly("DrawLineInStyle_not_normal");
   if (! addtopoly) StartStopUnit(true);
   switch (kar2) {
      case 'V': DrawValleyFold(x1,y1,x2,y2,deltad,r);          break;
      case 'M': DrawMountainFold(x1,y1,x2,y2,deltad*1.5,r);    break;
      case 'D': DrawDottedLine(x1,y1,x2,y2,deltad*0.5,r);      break;
      case 'B': case ' ': DrawLine(x1,y1,x2,y2);               break;
       default: ShowError("Line style (%c) %s",kar2,illegal);
   }
   if (! addtopoly) StartStopUnit(false);
   if (penwidthfactor!=penfactorwas) {
      penwidthfactor=penfactorwas;
      SetPenWidth();
   }
}                                      /* DrawLineInStyle */


static FUNCTION void ShowInSet(byteset xset,boolean yes,char *txt)
{
   int i;
   maxstring stxt;
   string10 itxt;

   TRACE CheckMgSystem(this_unit,"ShowInSet",true,mytracelevel,NULL);
   InitString(stxt,"",-1);
   InitString(itxt,"",10);
   for (i=1; i<=abs(number); i++) {
      if (xset[i]==yes) {
         if (strlen(stxt)>=60) {
            PrintString("   %s",stxt);
            stxt[0]=0;
         }
         sprintf(itxt,"%4d",i);
         StringCat(stxt,-1,itxt,10);
      }
   }
   PrintString("   %s %s",stxt,txt);
}                                      /* ShowInSet */


FUNCTION void ShowStatistics(void)
{
   int i;
   boolean missing,doubles;

   CHECKINIT CheckMgSystem(this_unit,"ShowStatistics",true,mytracelevel,NULL);
   PrintString("File:  %s",globaldisk.filename);
   PrintString("Model: %s",modelname);
   PrintString("");
   PrintString("%10d is last diagram",abs(number));
   doubles=false;
   for (i=1; i<=abs(number); i++) {
      if (doubledef[i]) doubles=true;
   }
   if (doubles) ShowInSet(doubledef,true,"double drawn");
   missing=false;
   for (i=1; i<=abs(number); i++) {
      if (! numbers[i]) missing=true;
   }
   if (missing) ShowInSet(numbers,false,"not used");
   PrintString("%10d pages with diagrams",printpage);
   PrintString("%11sTextlanguage selector is %c"," ",textlanguage);
}                                      /* ShowStatistics */


