/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Supply command procedures for ORITOPS and ORIDRAW program sources
     First version dd.19940815   (Pascal)
     Converted to C   19950216
*/

#include "orildraw.h"
#include "orilfile.h"

/*extracth Local */

#define last_revision "19990701"
#define this_unit "OriLcmnd"
#define prog_version "unit"
#define mytracelevel 4
static boolean initialized=false;


/*extracth Functions */

FUNCTION void MarkAllPoints(void)
{
   char c;
   maxstring txt;

   CHECKINIT CheckMgSystem(this_unit,"MarkAllPoints",true,mytracelevel,NULL);
   InitString(txt," ",-1);
   for (c='A'; c<='Z'; c++) {
      txt[0]=c;
      if ((defpointA+c-'A')->defined) DrawString((defpointA+c-'A')->x,(defpointA+c-'A')->y,txt,false);
   }
}                                      /* MarkAllPoints */


FUNCTION void InitializeOriLCmnd(int *argc,char *argv[])
{
   if (initialized) return;
   initialized=true;
   InitializeOriLDraw(argc,argv);
   InitializeOriLFile(argc,argv);
   CheckOriUpdate(last_revision);
   TRACE CheckMgSystem(this_unit,"InitializeOriLCmnd",true,mytracelevel,"Compiled at %s %s",__DATE__,__TIME__);
   markprocedure=MarkAllPoints;
}                                      /* InitializeOriLCmnd */


/*OriDoc Fill
.PAGE Fill commands
To shade parts of a diagram you may use the following commands:

.C FS
Start registering a polygon
.C Ff
Fill the registered polygon with gray factor f
.C FfD
Fill and draw the registered polygon with gray factor f
.C FCn
Set fill color to color n

For the gray factor f and the fill color n you may use the following values:

.T f  gray factor
n  color
.T W  100%   White
0  Black/Gray/White
.T L   87%   Light
1  Brown
.T M   75%   Middle
2  Blue        (default)
.T D   50%   Dark
3  Magenta
.T E   25%   Extra dark
4  Red
.T B    0%   Black
5  Orange
.T
6  Yellow
.T
7  Green

Gray factor B and W always fill the polygon with Black or White
independent of the fill color set.
*/


FUNCTION void FillCommand(textfile *disk)
{
   int grayscale;
   char kar2,kar3;

   CHECKINIT CheckMgSystem(this_unit,"FillCommand",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   if (kar2=='C') {
      SetFillColor(ReadInteger(disk));
   } else if (kar2=='S') {
      InitPoly("Polygon");
   } else {
      grayscale=4;
      switch (kar2) {
         case 'L': grayscale=5;                                  break;
         case 'M': grayscale=4;                                  break;
         case 'D': grayscale=3;                                  break;
         case 'E': grayscale=2;                                  break;
         case 'B': grayscale=1;                                  break;
         case 'W': grayscale=0;                                  break;
          default: ShowError("Fill code (%c)%s",kar2,illegal);
      }
      kar3=ReadChar(disk,true);
      DrawPolyPoints(grayscale,kar3=='D',true);
      ClosePoly("Polygon");
   }
}                                      /* FillCommand */


static FUNCTION void SymbolSinkArrow(textfile *disk)
{
   double x1,y1,xp,yp,d;

   TRACE CheckMgSystem(this_unit,"SymbolSinkArrow",true,mytracelevel,NULL);
   ReadDouble4(disk,&x1,&y1,&xp,&yp);
   d=Distance(x1,y1,xp,yp);
   if (d<0.001) {
      ShowError("Distance between points (%f %f) and (%f %f) is too small",x1,y1,xp,yp);
      return;
   }
   d=deltad/d;
   x1=(x1-xp)*d+xp;
   y1=(y1-yp)*d+yp;
   DrawArrowHead(x1,y1,xp,yp,0,deltad,'V');
}                                      /* SymbolSinkArrow */


static FUNCTION void SymbolRepeatArrow(textfile *disk)
{
   double x1,y1,xp,yp;
   char kar2;
   maxstring txt;
   boolean wassym;

   TRACE CheckMgSystem(this_unit,"SymbolRepeatArrow",true,mytracelevel,NULL);
   InitString(txt,"",-1);
   kar2=ReadChar(disk,true);
   ReadDouble4(disk,&x1,&y1,&xp,&yp);
   wassym=symmetrical;
   if (strchr("123456789",kar2)!=NULL) {
      /* ignore */
   } else if (strchr("ABCDEFGHI",kar2)!=NULL) {
      ReadLine(disk,txt);
      while (((int)strlen(txt)>0) && (txt[0]==' ')) DelChars(txt,0,1);
      symmetrical=false;
   } else {
      if (kar2!=' ') ShowError("SymbolRepeatArrow symbol: %c %s",kar2,illegal);
   }
   DrawRepeatArrow(x1,y1,xp,yp,kar2,txt);
   symmetrical=wassym;
}                                      /* SymbolRepeatArrow */


static FUNCTION void SymbolZigZagFold(textfile *disk)
{
   double x1,y1,x2,y2;
   char kar2;

   TRACE CheckMgSystem(this_unit,"SymbolZigZagFold",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   ReadDouble4(disk,&x1,&y1,&x2,&y2);
   if (strchr("/ ",kar2)==NULL) ShowError("ZigZag fold: %c %s",kar2,illegal);
   DrawZigZagFold(x1,y1,x2,y2,kar2);
}                                      /* SymbolZigZagFold */


static FUNCTION void SymbolSimple(textfile *disk,char kar2)
{
   double x1,y1,x2,y2;

   TRACE CheckMgSystem(this_unit,"SymbolSimple",true,mytracelevel,NULL);
   ReadDouble4(disk,&x1,&y1,&x2,&y2);
   switch (kar2) {
      case 'O': DrawRollOverFold(x1,y1,x2,y2);              break;
      case 'E': DrawAnEye(x1,y1,x2,y2);                     break;
       default: ;
   }
}                                      /* SymbolSimple */


/*OriDoc Symbol
.PAGE Symbol commands
Some Origami symbols are known:

.C SE x1 y1 x2 y2
Draws 'eye' symbol
.C SO x1 y1 x2 y2
Draws 'roll-over' symbol
.C SR  x1 y1 x2 y2
Draws 'repeat' symbol
.C SRn x1 y1 x2 y2
Draws 'repeat-n' symbol (repetition: 2..9)
.C SRl x1 y1 x2 y2 string
Draws 'repeat-n' symbol with text (rest of line) (l is letter: A=1 .. I=9)
.C SS x1 y1 x2 y2
Draws the 'sink' symbol
.C SZ  x1 y1 x2 y2
Draws the 'zigzag-fold' symbol (/ or blank
.C SZ/ x1 y1 x2 y2
sets the orientation: left or right)

De following commands draw their symbol in the right margin or (if you add
x and y) on the given coordinates:

.C SF       [ x y ]
Draws the symbol 'Fold-inside-out'
.C ST       [ x y ]
Draws the symbol 'Turn-over'
.C SA angel [ x y ]
Draws the symbol 'rotate over this Angel'
.C SU nr    [ x y ]
Draws the symbol 'Unfold to diagram'
.C SL       [ x y ]
Draws the symbol 'enLarged'
*/


FUNCTION void SymbolCommand(textfile *disk)
{
   char kar2;
   double xf,angle,x,y,x0margin,y0margin,k,wasdeltad;
   boolean wassym;

   CHECKINIT CheckMgSystem(this_unit,"SymbolCommand",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   InitPoly("Symbol");
   k=0.0;
   if (strchr("EO",kar2)!=NULL)   SymbolSimple(disk,kar2);        else
   if (kar2=='R')                 SymbolRepeatArrow(disk);        else
   if (kar2=='Z')                 SymbolZigZagFold(disk);         else
   if (kar2=='S')                 SymbolSinkArrow(disk);          else
   {
      wasdeltad=deltad;
      deltad=0.35;
      if (strchr("UA",kar2)!=NULL) k=ReadDouble(disk);
      xf=xyfactor;
      angle=globalalfa;
      x0margin=marginx;
      y0margin=marginy;
      wassym=symmetrical;
      symmetrical=false;
      while (NextChar(disk)==' ') SkipChar(disk);
      if (strchr("0123456789-#",NextChar(disk))!=NULL) {
         ReadPoint(disk,&x,&y);
         x=x+objectx;
         y=y+objecty;
         marginx=(x*cosalfa-y*sinalfa+deltax)*xyfactor+marginx;
         marginy=(x*sinalfa+y*cosalfa+deltay)*xyfactor+marginy;
      } else {
         marginx=19.3;
         marginy=fixpos;
      }
      SetFactor(1.5);
      SetAngle(0.0);
      switch (kar2) {
         case 'F': DrawFoldInsideOut();                            break;
         case 'T': DrawTurnOver();                                 break;
         case 'A': DrawRotate(k);                                  break;
         case 'U': DrawUnfold(k);                                  break;
         case 'L': DrawEnlarge();                                  break;
          default: ShowError("%c is unknown symbol command",kar2);
      }
      marginx=x0margin;
      marginy=y0margin;
      SetFactor(xf);
      SetAngle(angle);
      symmetrical=wassym;
      fixpos=fixpos+2;
      deltad=wasdeltad;
   }
   if (addtopoly) ClosePoly("Symbol");
}                                      /* SymbolCommand */


static FUNCTION void CoordinateSystemFactor(textfile *disk)
{
   CHECKINIT CheckMgSystem(this_unit,"CoordinateSystemFactor",true,mytracelevel,NULL);
   if (NextChar(disk)=='*') {
      SkipChar(disk);
      SetFactor(xyfactor*ReadDouble(disk));
   } else {
      SetFactor(ReadDouble(disk));
   }
}                                      /* CoordinateSystemFactor */


static FUNCTION void CoordinateSystemAngle(textfile *disk)
{
   CHECKINIT CheckMgSystem(this_unit,"CoordinateSystemAngle",true,mytracelevel,NULL);
   SetAngle(ReadDouble(disk));
}                                      /* CoordinateSystemAngle */


static FUNCTION void CoordinateSystemMirror(textfile *disk)
{
   char kar3;

   CHECKINIT CheckMgSystem(this_unit,"CoordinateSystemMirror",true,mytracelevel,NULL);
   while (NextChar(disk)==' ') SkipChar(disk);
   if (strchr("0123456789-#",NextChar(disk))!=NULL) {
      rmirror=ReadDouble(disk);
      symmetrical=true;
      SetFactor(xyfactor);
   } else {
      kar3=ReadChar(disk,true);
      if (kar3=='N') symmetrical=false; else ShowError("Character %c in Coordinate system command %s",kar3,illegal);
   }
}                                      /* CoordinateSystemMirror */


static FUNCTION void CoordinateSystemXY(textfile *disk,double *margin,boolean isx)
{
   CHECKINIT CheckMgSystem(this_unit,"CoordinateSystemXY",true,mytracelevel,NULL);
   *margin=ReadDouble(disk);
   if (isx) SetFactor(xyfactor);
}                                      /* CoordinateSystemXY */


static FUNCTION void CoordinateSystemPlus(textfile *disk,double *valx,double *valy,boolean tozero)
{
   double x,y;

   CHECKINIT CheckMgSystem(this_unit,"CoordinateSystemPlus",true,mytracelevel,NULL);
   ReadPoint(disk,&x,&y);
   if (tozero) {
      *valx=0;
      *valy=0;
   }
   *valx+=x;
   *valy+=y;
}                                      /* CoordinateSystemPlus */


static FUNCTION void ComputeProjection(double xp,double yp,double zp,double *xx,double *yx,double *zx,double *xy,double *yy,double *zy)
{
   double alfa,beta;
   int quadrant;

   CHECKINIT CheckMgSystem(this_unit,"ComputeProjection",true,mytracelevel,NULL);
   if ((fabs(xp)<0.01) && (fabs(zp)<0.01)) {
      *yy=0.0;
      *zy= -1.0;
   } else {
      if (xp<0) {
         if (zp<0) quadrant=3; else quadrant=4;
      } else {
         if (zp<0) quadrant=2; else quadrant=1;
      }
      if (fabs(zp)<fabs(xp)) alfa=pidiv2-atan(fabs(zp/xp)); else alfa=atan(fabs(xp/zp));
      switch (quadrant) {
          case 1: ;                           break;
          case 2: alfa=pi-alfa;               break;
          case 3: alfa=pi+alfa;               break;
          case 4: alfa= -alfa;                break;
         default: ;
      }
      beta=sqrt(xp*xp+zp*zp);
      if (fabs(yp)<fabs(beta)) beta=atan(yp/beta); else beta=pidiv2-atan(beta/yp);
      *xx=cos(alfa);
      *yx=0.0;
      *zx= -sin(alfa);
      *xy= -sin(alfa)*sin(beta);
      *yy=cos(beta);
      *zy= -cos(alfa)*sin(beta);
   }
   threedimensional=true;
}                                      /* ComputeProjection */


static FUNCTION void CoordinateSystemProjection(textfile *disk)
{
   char kar2;
   double xp,yp,zp;

   TRACE CheckMgSystem(this_unit,"CoordinateSystemProjection",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   while (NextChar(disk)==' ') SkipChar(disk);
   if (strchr("XYZ",kar2)!=NULL) {
      ReadDouble3(disk,&xp,&yp,&zp);
      if ((fabs(xp)<0.001) && (fabs(yp)<0.001) && (fabs(zp)<0.001)) {
         ShowError("Projection command (=PZ 0 0 0) %s",illegal);
         kar2='N';
      }
   }
   threedimensional=false;
   NormalProjection();
/*lint -save -e644 */
   switch (kar2) {
      case 'N': ;                                                                                                                        break;
      case 'X': ComputeProjection(yp,zp,xp,&projection.yx,&projection.zx,&projection.xx,&projection.yy,&projection.zy,&projection.xy);   break;
      case 'Y': ComputeProjection(zp,xp,yp,&projection.zx,&projection.xx,&projection.yx,&projection.zy,&projection.xy,&projection.yy);   break;
      case 'Z': ComputeProjection(xp,yp,zp,&projection.xx,&projection.yx,&projection.zx,&projection.xy,&projection.yy,&projection.zy);   break;
       default: ShowError("Projection command (=%c) %s",kar2,illegal);
   }
/*lint -restore */
}                                      /* CoordinateSystemProjection */


/*OriDoc CoordinateSystem
.PAGE Coordinate system commands
.C CF factor
Sets the scaling factor to the new value
.C CF* factor
Multiplies the scaling factor with the extra factor
.C CA angle
Sets the drawing angle to the new value
.C CD xd yd
New values for deltax and deltay
.C CE xplus yplus
deltax:=deltax+xplus  and  deltay:=deltay+yplus
.C CO xobject yobject
New values for xobject and yobject
.C CM x
Puts a mirror on the vertical line through xp,0; all subsequent drawing is
done also "in the mirror"; xp is computed as stated below
.C CMN
The mirror is withdrawn
.C CX xmargin
New value for xmargin
.C CY ymargin
New value for ymargin
.C C+ xplus yplus
xobject:=xobject+xplus  and  yobject:=yobject+yplus

The drawing coordinates in the diagram for eacht point are computed as
follows:

   xo:=x+xobject;      yo:=y+yobject;
   xp:=xmargin+(xo*cos(angle)-yo*sin(angle)+deltax)*factor
   yp:=ymargin+(xo*sin(angle)+yo*cos(angle)+deltay)*factor

The point (0,0) is in the lower left corner of the screen/diagram.


By default for each point two coordinates (x and y) are read.
.
It is also possible to work in 3D. Each point then has 3 coordinates (x y z).
For each point the parallel projection on the surface perpendicular to the
line through the given projection point (xp,yp,zp) and (0,0,0) is computed.
This also works for the other Coordinate system commands etc.

The possible projection commands are:

                  Turn around ..-axis
                   first    then

.C CPX xp yp zp
x       y
.C CPY xp yp zp
y       z
.C CPZ xp yp zp
z       x

.C CPN
Back to normal (2D) drawing (x and y coordinates)

You may find an example of the use of projection together with macros in
CATERPIL.ORI.
*/


FUNCTION void CoordinateSystemCommand(textfile *disk)
{
   char kar2;

   CHECKINIT CheckMgSystem(this_unit,"CoordinateSystemCommand",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   switch (kar2) {
      case 'F': CoordinateSystemFactor(disk);                            break;
      case 'A': CoordinateSystemAngle(disk);                             break;
      case 'D': CoordinateSystemPlus(disk,&deltax,&deltay,false);        break;
      case 'E': CoordinateSystemPlus(disk,&deltax,&deltay,true);         break;
      case 'O': CoordinateSystemPlus(disk,&objectx,&objecty,true);       break;
      case 'M': CoordinateSystemMirror(disk);                            break;
      case 'P': CoordinateSystemProjection(disk);                        break;
      case 'X': CoordinateSystemXY(disk,&marginx,true);                  break;
      case 'Y': CoordinateSystemXY(disk,&marginy,false);                 break;
      case '+': CoordinateSystemPlus(disk,&objectx,&objecty,false);      break;
       default: ShowError("Coordinate system command (%c) %s",kar2,illegal);
   }
}                                      /* CoordinateSystemCommand */


FUNCTION void DoPointCommand(char kar,char kar2,boolean relative,double x,double y,double x2,double y2,double x3,double y3,double x4,double y4,double z,double dx,double dy)
{
   TRACE CheckMgSystem(this_unit,"DoPointCommand",true,mytracelevel,NULL);
   if (relative) PointPlus(&x,&y,-x2,-y2);
   switch (kar2) {
      case '*': PointTimes(&x,&y,z);                               break;
      case '/': PointTimes(&x,&y,1/z);                             break;
      case 'T': PointTurn(&x,&y,z);                                break;
      case '+': PointPlus(&x,&y,dx,dy);                            break;
      case '-': PointPlus(&x,&y,-dx,-dy);                          break;
      case 'X': PointCrossLines(&x,&y,x2,y2,x3,y3,x4,y4);          break;
      case 'B': PointBisectrice(&x,&y,x2,y2,x3,y3);                break;
      case 'M': PointMirrorInLine(&x,&y,x2,y2,x3,y3);              break;
      case 'W':
      case ' ':
      case 'r': ;                                                  break;
       default: ShowError("PointCommand: <%c> internal error",kar2);
   }
   if (relative) PointPlus(&x,&y,x2,y2);
   CheckMaxCoordinate('X',&x);
   CheckMaxCoordinate('Y',&y);
   (defpointA+kar-'A')->x=x;
   (defpointA+kar-'A')->y=y;
   (defpointA+kar-'A')->defined=true;
}                                    /* DoPointCommand */


/*OriDoc Define
.PAGE Point commands
The program can remember the coordinates of 26 points. Each point has a
letter connected to it. You define a (new) point with one of the following
commands (put the letter wanted on the position of the l):

.C D  l x y
Definition of point l
.C D+ l x y dx dy
Plus:     x:=x+dx;       y:=y+dy
.C D+ * dx dy
Plus:        operating on all points
.C D- l x y dx dy
Minus:    x:=x-dx;       y:=y-dy
.C D- * dx dy
Minus:       operating on all points
.C D* l x y factor
Times:    x:=x*factor;   y:=y*factor
.C D* * factor
Times:       operating on all points
.C D/ l x y divisor
Divide:   x:=x/divisor;  y:=y/divisor
.C D/ * divisor
Divide:      operating on all points
.C DM l x y x1 y1 x2 y2
Mirror x,y in line x1,y1_x2,y2
.C DX l x1 y1 x2 y2 x3 y3 x4 y4
Crosspoint of x1,y1_x2,y2 with x3,y3_x4,y4
.C DT l x y angle
Turn x,y over angle around 0,0
.
Positive is counter-clock-wise
.C DT * angle
Turn:        operating on all points
.C DB l x1 y1 x2 y2 x3 y3
Bisectrice of angle in x2,y2 between lines to x1,y1 and x3,y3.
Point is on x1,y1_x3,y3.

.C DRa l x y x0 y0 ...
Operator a on x,y relative to x0,y0
(operator: T, * or /)
.C DRa * x0 y0 ...
Operator:    operating on all points

.C DW lll
Wipe point(s) lll (or all points if DW *)
.C DW*
Wipe all points
.C DW- lll
Wipe all points except lll

See below for the usage of the defined points.
.PAGE Using points
You use a point by putting on the place of a coordinate pair the notation:

   .l

You may use one point that you can't define. That is denoted by the at sign:

   .@

That gives always the coordinates of the last point read/used.


Examples:

   DR* M .B .A 0.5   [ middle M of a line A_B ]
   DRT A .A .M 90    [ turn A 90 degrees counterclockwise around M ]
   A 0 0 10 10 0.5   A .@ 0 0 0.5   [ .@ gives as coordinates: 10 10 ]


If you only want to use the x or y coordinate of a point as a single number
then use one of the following notations:

   #lx       or   #ly

As an example we have a rectangle drawn in A4 format via:

   D R 16 16   DT R .R -45
   L 0 0  .R  #Rx 16  0 16  0 0
*/


FUNCTION void PointCommand(textfile *disk)
{
   char kar,kar2,pos;
   double x,y,dx,dy,z,x2,y2,x3,y3,x4,y4;
   boolean relative;
   string40 karw;

   CHECKINIT CheckMgSystem(this_unit,"PointCommand",true,mytracelevel,NULL);
   InitString(karw,"*",40);
   kar2=ReadChar(disk,true);
   if (kar2=='W') {
      kar2=ReadChar(disk,true);
      if (kar2=='-') {
         while (NextChar(disk)==' ') SkipChar(disk);
      }
      if (kar2=='*') {
         kar=' ';
         pos=1;
      } else {
         kar=NextChar(disk);
         pos=0;
      }
      while (kar!=' ') {
         if (isalpha(kar)) {
            if (strchr(karw,kar)==NULL) {
               karw[(int)pos]=kar;
               pos++;
               karw[(int)pos]=0;
            }
         } else if (kar=='*') {
            kar2='*';
         } else {
            ShowError("Point selector for Wipe (=%c) should be a letter or an asterisk (*)",kar);
         }
         kar=ReadChar(disk,true);
      }
      karw[(int)pos]=0;
      if (kar2=='-') {
         for (kar='A'; kar<='Z'; kar++) {
            if (strchr(karw,kar)==NULL) (defpointA+kar-'A')->defined=false;
         }
      } else if (kar2=='*') {
         for (kar='A'; kar<='Z'; kar++) (defpointA+kar-'A')->defined=false;
      } else {
         for (pos=0; karw[(int)pos]!=0; pos++) {
            kar=karw[(int)pos];
            (defpointA+kar-'A')->defined=false;
         }
      }
      return;
   }
   relative=(kar2=='R');
   if (relative) {
      kar2=ReadChar(disk,true);
      if (strchr("*/T",kar2)==NULL) {
         ShowError("PointCommand: R%c %s",kar2,illegal);
         kar2='r';
      }
   }
   while (NextChar(disk)==' ') SkipChar(disk);
   kar=ReadChar(disk,false);
   if (isalpha(kar) || ((kar=='*') && (strchr("*/T+-",kar2)!=NULL))) {
      x=0;
      y=0;
      if (kar!='*') ReadPoint(disk,&x,&y);
      switch (kar2) {
         case ' ':                                              break;
         case '*':
         case '/':
         case 'T': if (relative) ReadDouble3(disk,&x2,&y2,&z);
                      else z=ReadDouble(disk);
                                                                break;
         case '-':
         case '+': ReadPoint(disk,&dx,&dy);                     break;
         case 'B': ReadDouble4(disk,&x2,&y2,&x3,&y3);           break;
         case 'M': ReadDouble4(disk,&x2,&y2,&x3,&y3);           break;
         case 'X': ReadDouble6(disk,&x2,&y2,&x3,&y3,&x4,&y4);   break;
         case 'r': ;                                            break;
          default: ShowError("PointCommand: %c is unknown operator",kar2);
      }
      if (kar=='*') {
         for (kar='A'; kar<='Z'; kar++) {
            if ((defpointA+kar-'A')->defined) {
               x=(defpointA+kar-'A')->x;
               y=(defpointA+kar-'A')->y;
               DoPointCommand(kar,kar2,relative,x,y,x2,y2,x3,y3,x4,y4,z,dx,dy);
            }
         }
      } else {
         DoPointCommand(kar,kar2,relative,x,y,x2,y2,x3,y3,x4,y4,z,dx,dy);
      }
   } else {
      ShowError("Point selector (=%c) should be a letter",kar);
   }
}                                      /* PointCommand */


FUNCTION void ArcArrowCommand(textfile *disk)
{
   double x1,y1,x2,y2,k;
   string40 options;

   CHECKINIT CheckMgSystem(this_unit,"ArcArrowCommand",true,mytracelevel,NULL);
   InitString(options,"",40);
   if (NextChar(disk)!=' ') ReadItem(disk,options);
   ReadDouble5(disk,&x1,&y1,&x2,&y2,&k);
   DrawArcArrowInStyle(x1,y1,x2,y2,k,deltad,options);
}                                      /* ArcArrowCommand */


/*OriDoc Lines
.PAGE Drawing commands
.T A.. x1 y1 x2 y2 curvature
Draws an arc (eventually with arrowheads); see "Arc commands" for the
curvature and the n
.C Bs x0 y0 x1 y1 x2 y2 x3 y3
Draws a Bezier curve from x0,y0 to x3,y3 with attractors x1,y1 and x2,y2
.C E x0 y0 rx ry [ angle ]
Draws an ellips (centre x0,y0, axes rx,ry, eventually rotated over angle)
.C Ls x1 y1 x2 y2 x3 y3 ...
Draws a line/polygon (whole pair of numbers); the row of coordinate pairs
may be closed by an asterisk (*), but also by the following command
.C Rs x1 y1 x2 y2
Draws a rectangle

In the drawing commands B, L and R a second character (option) may denote
the linestyle. The possibilities are:

For Line and Rectangle:

       M: point dash line (mountain fold),         -.-.-.-.-
       V: dash line (valley fold),                 - - - - -

For BezierCurve, Line and Rectangle:

       D: dotted line,                             .........
       B: bold line,                               _________
   blank: normal line,                             _________
    else: error message; normal line.

The valley and mountain fold are drawn bold automatically.
*/


FUNCTION void EllipsCommand(textfile *disk)
{
   double x0,y0,xr,yr,angle;

   CHECKINIT CheckMgSystem(this_unit,"EllipsCommand",true,mytracelevel,NULL);
   ReadDouble4(disk,&x0,&y0,&xr,&yr);
   angle=0.0;
   while (strchr(blankcomma,NextChar(disk))!=NULL) SkipChar(disk);
   if (strchr(".0123456789-+",NextChar(disk))!=NULL) angle=ReadDouble(disk);
   DrawEllips(x0,y0,xr,yr,angle);
}                                      /* EllipsCommand */


FUNCTION void BezierCurveCommand(textfile *disk)
{
   double x0,y0,x1,y1,x2,y2,x3,y3;
   char kar2;

   CHECKINIT CheckMgSystem(this_unit,"BezierCurveCommand",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   ReadDouble4(disk,&x0,&y0,&x1,&y1);
   ReadDouble4(disk,&x2,&y2,&x3,&y3);
   DrawBezierCurveInStyle(x0,y0,x1,y1,x2,y2,x3,y3,kar2);
}                                      /* BezierCurveCommand */


FUNCTION void RectangleCommand(textfile *disk)
{
   double x1,y1,x2,y2;
   char kar2;

   CHECKINIT CheckMgSystem(this_unit,"RectangleCommand",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   ReadDouble4(disk,&x1,&y1,&x2,&y2);
   DrawLineInStyle(x1,y1,x1,y2,kar2);
   DrawLineInStyle(x1,y2,x2,y2,kar2);
   DrawLineInStyle(x2,y2,x2,y1,kar2);
   DrawLineInStyle(x2,y1,x1,y1,kar2);
}                                      /* RectangleCommand */


FUNCTION void PolyLine(textfile *disk)
{
   double x1,y1,x2,y2;
   char kar2;

   CHECKINIT CheckMgSystem(this_unit,"PolyLine",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   ReadPoint(disk,&x1,&y1);
   while (strchr(blankcomma,NextChar(disk))!=NULL) SkipChar(disk);
   if (strchr("#.0123456789-+",NextChar(disk))!=NULL) {
      ReadPoint(disk,&x2,&y2);
      while (strchr(blankcomma,NextChar(disk))!=NULL) SkipChar(disk);
   } else {
      x2=x1;
      y2=y1;
   }
   while (strchr("#.0123456789-+",NextChar(disk))!=NULL) {
      DrawLineInStyle(x1,y1,x2,y2,kar2);
      x1=x2;
      y1=y2;
      ReadPoint(disk,&x2,&y2);
      while (strchr(blankcomma,NextChar(disk))!=NULL) SkipChar(disk);
      if (KeyPressed()) Interaction(true);
   }
   if (NextChar(disk)=='*') SkipChar(disk);
   DrawLineInStyle(x1,y1,x2,y2,kar2);
}                                      /* PolyLine */


FUNCTION void SetWithFrame(textfile *disk)
{
   char kar3;

   CHECKINIT CheckMgSystem(this_unit,"SetWithFrame",true,mytracelevel,NULL);
   if (NextChar(disk)!=' ') kar3=ReadChar(disk,true); else kar3=' ';
   withframe=(strchr("+1YJTyjt",kar3)!=NULL);
}                                      /* SetWithFrame */


static FUNCTION void PenWidthCommand(textfile *disk,char kar2)
{
   char kar3;

   TRACE CheckMgSystem(this_unit,"PenWidthCommand",true,mytracelevel,NULL);
   if (kar2=='-') penwidth=penwidthdefault;
   if (strchr("0123456789-#",NextChar(disk))!=NULL) {
      penwidth=ReadDouble(disk);
   } else {
      kar3=ReadChar(disk,true);
      switch (kar3) {
         case 'B':   penwidthfactor=2.0;                           break;
         case 'N':   penwidthfactor=1.0;                           break;
         case '*':   penwidth=penwidth*ReadDouble(disk);           break;
          default:   ShowError("Penwidth command (=%c) %s",kar2,illegal);
      }
   }
   if (penwidth<0.01) penwidth=0.01;
   if (penwidth>2) penwidth=2;
   if (kar2=='-') penwidthdefault=penwidth;
   SetPenWidth();
}                                      /* PenWidthCommand */


/*OriDoc Layout
.PAGE Paper layout
By default the diagrams are printed by ORITOPS with 150 dpi. The position
of each diagram is computed automatically.
.
In this positioning both the x and the y count (in 300 dpi) from (0,0) in the
upper left corner of the paper up to (2230,3200) in the down right corner.

For gouverning the layout you may use the following commands:

.C Pn
Print with 300/n dpi (n: digit ranging from 1 to 9)
.C PF factor
Print with 300/factor dpi
.C PFn
Print with 300/n dpi (n: digit ranging from 1 to 9)
.C PH heigth
Dictates the heigth of the diagram in dots
Default is  PH 0 (heigth of the diagram itself)
.C PA x y
Set the position of the next diagram to x,y (absolute)
.C PR dx dy
Change the position of the next diagram with dx,dy (relative)
.C PD deltad
Set the length of the dashes for the mountain and valley folds
.C PM
Mark all points defined with their letter
.C P. width
Set the pen/line width
.C P.* factor
Multiplies the pen/line width with factor
.C P.B
Sets pen/line width to bold (sets extra width factor to 2)
.C P.N
Sets pen/line width to normal (sets extra width factor to 1)
.C P- width
Set the default pen/line width
.C P|+   P|-
Draw frames around each diagram (+) or not (-). Default is L|-
.C PB
Mark point in diagram to force border wider than points drawn
*/


FUNCTION void PaperLayoutCommand(textfile *disk)
{
   char kar2;
   double incrx,incry,x,y;

   CHECKINIT CheckMgSystem(this_unit,"PaperLayoutCommand",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   while (NextChar(disk)==' ') SkipChar(disk);
   incrx=0;
   incry=0;
   switch (kar2) {
      case 'B': ReadPoint(disk,&x,&y);                              break;
      case 'D': deltad=ReadDouble(disk);                            break;
      case 'F': paperfactor=ReadDouble(disk);                       break;
      case 'H': diagramheigth=ReadDouble(disk);                     break;
      case 'M': MarkAllPoints();                                    break;
      case 'X': incrx=ReadDouble(disk);                             break;
      case 'Y': incry=ReadDouble(disk);                             break;
      case 'A':
      case 'R': incrx=ReadDouble(disk); incry=ReadDouble(disk);     break;
      case '-':
      case '.': PenWidthCommand(disk,kar2);                         break;
      case '|': SetWithFrame(disk);                                 break;
       default: if (isdigit(kar2)) {
                   paperfactor=kar2-'0';
                } else {
                   ShowError("Paper layout command (=%c) %s",kar2,illegal);
                }
   }
   if (fabs(incrx)<30) incrx=incrx*118.1;
   if (fabs(incry)<30) incry=incry*118.1;
   if (kar2=='A') {
      paperx=Round(incrx);
      papery=Round(incry);
      autopos=false;
   } else if (kar2=='B') {
/*lint -save -e644 */
      CheckBorderPoint(x,y);
/*lint -restore */
   } else if (kar2=='R') {
      paperx=Round(paperx+incrx);
      papery=Round(papery+incry);
   }
   if (deltad<0.1) {
      ShowError("DeltaD < 0.1 %s",illegal);
      deltad=0.1;
   }
   if (deltad>1) {
      ShowError("DeltaD > 1 %s",illegal);
      deltad=1;
   }
}                                      /* PaperLayoutCommand */


FUNCTION void SkipComment(textfile *disk)
{
   char kar;

   CHECKINIT CheckMgSystem(this_unit,"SkipComment",true,mytracelevel,NULL);
   kar=ReadChar(disk,false);
   while ((kar!=']') && ! EofData(disk)) {
      if (kar=='[') SkipComment(disk);
      if (! EofData(disk)) kar=ReadChar(disk,false);
   }
}                                      /* SkipComment */


/*OriDoc Text
.PAGE Text commands
You may add text to diagrams in two ways.
.
To put text at the bottom of a diagram (or at the top of the first page)
give one of:

.C TB
The rest of the line is put as text under the diagram
.C TBx
The same, but text is only used when the option /LANG=x was given
.MAR 6
.RIGHT 75
.
These lines are normally written in Helvetica (proportional). If a line
starts with a backslash (\) this is omitted and the remaining line is
written in Courrier (fixed character width).
.MAR
.RIGHT

Do you specify these commands before the first N (Next) command, then the
text is put at the top of the first page.

To put text (string or character) in a diagram, use one of:

.C TS  x y string
The rest of the line is drawn in the diagram starting at position x,y
.C TSB x y string
The rest of the line is drawn BOLD in the diagram starting at position x,y
.C TC  x y char
The first non-blank character after the y coordinate is drawn
at position x,y
.C TCB x y char
The first non-blank character after the y coordinate is drawn
BOLD at position x,y
*/


FUNCTION void TextCommand(textfile *disk)
{
   double x1,y1;
   char kar,kar2;
   maxstring txt;
   boolean show,bold;

   CHECKINIT CheckMgSystem(this_unit,"TextCommand",true,mytracelevel,NULL);
   InitString(txt,"",-1);
   kar2=ReadChar(disk,true);
   if (kar2=='B') {
      kar2=ReadChar(disk,true);
      if (! EndOfLine(disk)) {
         ReadLine(disk,txt);
         StripBlanks(txt);
         if (toptextlinestrlength<(int)strlen(txt)) toptextlinestrlength=(int)strlen(txt);
         show=((kar2==' ') || (kar2==textlanguage));
         if (show) {
            if (number!=0) {
               if (lastline<max_text) lastline+=1;
               StringCopy(*(textline0+lastline),-1,txt,-1);
            } else {
               NextHeaderLine(txt);
            }
         }
      }
   } else {
      bold=false;
      if (toupper(NextChar(disk))=='B') {
         bold=true;
         SkipChar(disk);
      }
      ReadPoint(disk,&x1,&y1);
      while ((NextChar(disk)==' ') && ! EndOfLine(disk)) SkipChar(disk);
      if (kar2=='S') {
         if (EndOfLine(disk)) StringCopy(txt,-1,"*",0); else ReadLine(disk,txt);
         DrawString(x1,y1,txt,bold);
      } else if (kar2=='C') {
         if (EndOfLine(disk)) kar=(char)219; else kar=ReadChar(disk,false);
         sprintf(txt,"%c",kar);
         DrawString(x1,y1,txt,bold);
      } else {
         ShowError("Text style (%c) %s",kar2,illegal);
      }
   }
}                                      /* TextCommand */


/*OriDoc Units
.PAGE Unit commands
You may draw more units in one diagram using separate environments per unit.
Each environment contains:

 - defined points
 - fill color
 - xmargin, ymargin, xobject, yobject, deltax, deltay
 - angle, factor, mirror, projection system

To save or restore the environment for a unit give one of:

.C US n
Save the environment of unit n
.C UR n
Restore the environment of unit n

MIND:
.MAR 3
.
When a new diagram is started several settings in the environment are reset
to their default value. See for this: 'Defaults'.
.
When you restore the environment of a unit the total environment of that unit
is restored. So the defaulted settings are overruled!
.MAR 0
*/


FUNCTION void UnitCommand(textfile *disk)
{
   char kar2;
   int i;

   CHECKINIT CheckMgSystem(this_unit,"UnitCommand",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   if (kar2=='S') {
      i=ReadInteger(disk);
      if ((i>0) && (i<=max_unit)) {
         UnitStatusSave(i,true);
      } else {
         ShowError("Unit number (%d)%s",i,illegal);
      }
   } else if (kar2=='R') {
      i=ReadInteger(disk);
      if ((i>0) && (i<=max_unit) && unitstatus[i]->defined) {
         UnitStatusRestore(i);
      } else {
         ShowError("Unit %d is never saved",i);
      }
   } else {
      ShowError("UnitCommand (%c)%s",kar2,illegal);
   }
}                                    /* UnitCommand */


FUNCTION void SkipToDiagram(textfile *ifile,char *iname)
{
   maxstring xname;
   boolean found;

   CHECKINIT CheckMgSystem(this_unit,"SkipToDiagram",true,mytracelevel,NULL);
   InitString(xname,"",-1);
   if (iname[0]==0) {
      while (! Equal(xname,"ENDOFHEADER")) {
         ReadLn(ifile->disk,xname,-1);
         UpString(xname);
         ifile->linenumber++;
      }
   } else {
      StringCopy(ifile->line,-1," ",0);
      UpString(iname);
      found=false;
      while (! found && (ReadLn(ifile->disk,ifile->line,-1)!=EOF)) {
         ifile->linenumber++;
         StringCat(ifile->line,-1," ",0);
         if (ifile->line[0]=='<') {
            ifile->line[0]=' ';
            StripBlanks(ifile->line);
            SplitString(xname,-1,ifile->line,-1," ");
            UpString(xname);
            if (Equal(iname,xname)) found=true;
            ifile->line[0]=0;
         }
      }
      if (! found) ShowError("Subdiagram %s not found on %s",iname,ifile->filename);
   }
}                                      /* SkipToDiagram */


/*OriDoc Defaults
.PAGE Defaults
ORIXXXX starts with the next defaults:

At the beginning of each file:

.T CF  1
Factor
.T PD  0.35
DeltaD
.T CPN
Normal 2D (only x and y coordinates)
.T PF  2
Paperfactor
.T P-  0.03
Default pen/line width
.T P|-
Don't draw frames around each diagram
.T DW*
Wipe all points
.T FC2
Set fill color to 2 (Blue)

At the start of each diagram:

.T CO   0  0
XObject  YObject
.T CD   0  0
DeltaX   DeltaY
.T P2
150 DPI
.T CX   1
XMargin
.T CY   0.5
YMargin
.T CA   0
Angle
.T CMN
No mirror
.T PH   0
Diagramheigth from diagram itself
.T P.  default
Pen/line width to default pen/line width
*/


FUNCTION void ResetDefaults(boolean all)
{
   char c;
   int i;

   CHECKINIT CheckMgSystem(this_unit,"ResetDefaults",true,mytracelevel,NULL);
   if (all) {
      for (c='A'; c<='Z'; c++) (defpointA+c-'A')->defined=false;
      deltad=0.35;
      diagram0->lines=0;
      diagram0->maxtextlen=0;
      for (i=0; i<=max_diagram; i++) doubledef[i]=false;
      SetFactor(1.0);
      i=IfDefReset(false);
      lastdiagram=0;
      marginx=1;
      marginy=0.5;
      number=0;
      numberabc=' ';
      for (i=0; i<=max_diagram; i++) numbers[i]=false;
      paperx=9999;
      papery= -10;
      paperplus=0;
      paperfactor=2;
      printpage=1;
      NormalProjection();
      IfDefReset(false);
      IfDefAddKeys(startifkey);
      threedimensional=false;
      withframe=false;
   }
   StartDiagramGraphics();
   SetAngle(0.0);
   autopos=true;
   deltax=0;
   deltay=0;
   diagramheigth=0;
   fixpos=1.5;
   imirror2=0;
   imirror2ps=0;
   imirror20ps=0;
   lastline=0;
   marginx=1;
   marginy=0.5;
   objectx=0;
   objecty=0;
   penwidth=penwidthdefault;
   penwidthfactor=1.0;
   if (! all) SetPenWidth();
   rmirror=0;
   symmetrical=false;
   xdrawmax=0;
   xdrawmin=1500;
   ydrawmax=0;
   ydrawmin=1000;
   if (all) {
      for (i=0; i<=max_unit; i++) UnitStatusSave(i,false);
   }
}                                      /* ResetDefaults */


static FUNCTION void TestSetBoolean(textfile *disk,boolean *bool)
{
   char kar3;

   TRACE CheckMgSystem(this_unit,"TestSetBoolean",true,mytracelevel,NULL);
   kar3=ReadChar(disk,false);
   if (kar3=='+') (*bool)=true;
   if (kar3=='-') (*bool)=false;
}                                    /* TestSetBoolean */


static FUNCTION void TestSetTraceLevel(textfile *disk)
{
   CheckMgSystem(this_unit,"TestSetTraceLevel",true,1,"old=%d",tracelevel);
   tracelevel=ReadInteger(disk);
   if (tracelevel<0) tracelevel=0;
   if (tracelevel>999) tracelevel=999;
   CheckMgSystem(this_unit,"TestSetTraceLevel",true,1,"new=%d",tracelevel);
}                                    /* TestSetTraceLevel */


/*OriDoc Debugging
.PAGE Debugging the ORIDRAW programs
Some commands are available to debug ORIXXXX and the other ORIDRAW programs.
These are:

.C ;D
Generate a dump
.C ;Tx
Set test mode on (;T+) or off (;T-)
.C ;L n
Set trace level to n (0..9)

Dumps are appended to the file ORIXXXX_.DMP and numbered.
When ORIXXXX generates the first dump the dumpfile is rewritten
with the program name and the .ORI file name.

Test mode may also be set with the command option -TESTMODE.

And also you may set the trace level with the command option -TRACEx where x
denotes the trace level. Legal values are -TRACE0 (no trace) up to -TRACE9
(most elaborate trace).
.
The trace is generated on file TRACE.TMP. This may become a very long file
when the trace level is set to 9. Also the program slows down very much on
this trace level.
.
So be carefull when setting these options.

When set test mode and/or trace level are also in effect before the first
command is read from the .ORI file.
*/


FUNCTION void TestCommand(textfile *disk)
{
   char kar2;

   CHECKINIT CheckMgSystem(this_unit,"TestCommand",true,mytracelevel,NULL);
   kar2=ReadChar(disk,true);
   switch (kar2) {
      case 'D': DumpStatus();                            break;
      case 'T': TestSetBoolean(disk,&testmode);          break;
      case 'L': TestSetTraceLevel(disk);                 break;
       default: /* ignore */;                            break;
   }
}                                    /* TestCommand */


