/*
    Author: Maarten van Gelder, Groningen, Nederland
    E-mail: M.J.van.Gelder@kvi.nl
   Purpose: Format DOC files for OriDraw and OriToPs from source files
     First version dd.19940723   (Pascal)
     Converted to C   19950220
*/

#define last_revision "19990331"
#define this_unit "OriDoc"
#define prog_version "2.01"
#define mytracelevel 0


#include "orilproc.h"


#define maxbuffer   50
#define maxindex   140
#define maxitem     40
#define maxfile     20
#define maxdocspec  50
#define maxselkey   10
#define configname    "oridoc.dat"
#define indexname     "oridraw.inx"
#define indexdump     "index.tmp"


typedef struct l_record *line_pointer;
typedef struct l_record{
   line_pointer next;
   string100 text;
} line_record;

typedef struct it_record{
   line_pointer next;
   string40 name;
   int indentmargin;
} item_record;

typedef struct it_record *item_record_pointer;

typedef struct{
   maxstring filename;
   char action;
   FILE *f;
} disk_record;

typedef struct{
   string10 key;
   boolean on;
} selkey_record;


maxstring textline,margin,blanks;
disk_record disk;
string10 docversion,docname,doclastrevision,systemname;
long linesio,showtxtline=5;
int lastbuffer,lastindex=0,lastitem=0,lastincfile=0,lastdocfile=0,lastdocspec=0,lastselkey=0;
int left,right,indentmargin,lastprevious= -1;
boolean testbuffer=false,ok=true,writeison,systemcorrect,addtoitem,indented,addtoindex;
string100 linebuffer[maxbuffer+1];
item_record item[maxitem+1];
item_record_pointer indx[maxindex+1];
string40 incfile[maxfile+1],docfile[maxfile+1],docspec[maxdocspec+1];
selkey_record selkey[maxselkey+1];


FUNCTION void InitializeOriDoc(int *argc,char *argv[])
{
   maxstring key,par;
   int i;

   TRACE CheckMgSystem(this_unit,"InitializeOriDoc",true,mytracelevel,NULL);
   InitializeMgProgIo(this_unit,last_revision,prog_version,argc,argv);
   ProgramHeader(stdout);
   InitString(key,"",-1);
   InitString(par,"",-1);
   InitString(docversion,"000000",10);
   InitString(doclastrevision,"00000000",10);
   InitString(docname,"",10);
   InitString(textline,"",-1);
   InitString(margin,"",-1);
   InitString(blanks,"",-1);
#if defined(__MSDOS__)
   InitString(systemname,"DOS",10);
#else
   InitString(systemname,"XWIN",10);
#endif
   sprintf(blanks,"%80s","");
   InitString(disk.filename,"",-1);
   disk.action='?';
   InitString(linebuffer[1],"",100);
   for (i=1; i<(*argc); i++) {
      StringCopy(par,-1,argv[i],0);
      if (strchr("-/",par[0])!=NULL) {
         DelChars(par,0,1);
         SplitString(key,-1,par,-1,"=");
         UpString(key);
         if (Equal(key,"TESTBUFFER"))   testbuffer=true;
      }
   }
   if (testmode) printf("   TESTMODE\n");
}                                      /* InitializeOriDoc */


FUNCTION void ShowMessage(long linenr,char *txt1,char *txt2,boolean error)
{
   TRACE CheckMgSystem(this_unit,"ShowMessage",true,mytracelevel,NULL);
   showtxtline++;
   if (showtxtline>11) {
      PressEnter();
      if (QuitSeen) exit(0);
      showtxtline=1;
   }
   printf("\r%5ld %s %s\n",linenr,txt1,txt2);
   if (error) ok=false;
}                                      /* ShowMessage */


FUNCTION void OpenDiskFile(char *filename,char *action)
{
   TRACE CheckMgSystem(this_unit,"*OpenDiskFile",true,mytracelevel,filename);
   StringCopy(disk.filename,-1,filename,0);
   disk.f=fopen(filename,action);
   if (disk.f==NULL) AbortBecause("Could not open <%s> for <%s>",filename,action);
   disk.action=action[0];
   linesio=0;
}                                      /* OpenDiskFile */


FUNCTION void CloseFile(void)
{
   TRACE CheckMgSystem(this_unit,"CloseFile",true,mytracelevel,disk.filename);
   fclose(disk.f);
   printf("%70s\r%10ld lines","",linesio);
   if (disk.action=='r') {
      if (lastprevious>=0) printf(" and %d items",lastitem-lastprevious);
      lastprevious=lastitem;
      printf(" read from");
   } else {
      printf(" written to");
   }
   printf(" %s\n",disk.filename);
}                                      /* CloseFile */


FUNCTION int ReadLine(void)
{
   int i;

   if (4<=tracelevel) CheckMgSystem(this_unit,"ReadLine",true,mytracelevel,NULL);
   i=ReadLn(disk.f,textline,-1);
   if (i>=0) linesio++;
   if (linesio % 64==0) printf("\r%10ld\r",linesio);
   return i;
}                                    /* ReadLine */


FUNCTION void AddIncFileName(char *text)
{
   TRACE CheckMgSystem(this_unit,"AddIncFileName",true,mytracelevel,text);
   lastincfile++;
   if (lastincfile>maxfile) AbortBecause("Const maxfile is too small");
   InitString(incfile[lastincfile],text,40);
}                                    /* AddIncFileName */


FUNCTION void AddDocFileName(char *text)
{
   TRACE CheckMgSystem(this_unit,"AddDocFileName",true,mytracelevel,text);
   lastdocfile++;
   if (lastdocfile>maxfile) AbortBecause("Const maxfile is too small");
   InitString(docfile[lastdocfile],text,40);
}                                    /* AddDocFileName */


FUNCTION void ReadConfigFile(void)
{
   boolean inheader;
   maxstring key,copy;

   TRACE CheckMgSystem(this_unit,"ReadConfigFile",true,mytracelevel,NULL);
   InitString(key,"",-1);
   InitString(copy,"",-1);
   OpenDiskFile(configname,"r");
   inheader=true;
   printf("%10s Read config file\r","");
   while (inheader && (ReadLine()!=EOF)) {
      if (textline[0]=='*') continue;
      StringCopy(copy,-1,textline,-1);
      SplitString(key,-1,textline,-1," =:");
      UpString(key);
      if (Equal(key,"FROM"))   AddIncFileName(textline);           else
      if (Equal(key,"TO"))     AddDocFileName(textline);           else
      if (Equal(key,"PAGES"))  inheader=false;                     else
      {
         ShowMessage(linesio,"Config error:",copy,true);
      }
   }
   while (ReadLine()!=EOF) {
      if (textline[0]=='*') continue;
      lastdocspec++;
      if (lastdocspec>maxdocspec) AbortBecause("Const maxdocspec is too small");
      InitString(docspec[lastdocspec],textline,40);
   }
   CloseFile();
   printf("%10d FROM files to read\n",lastincfile);
   printf("%10d TO files to write\n",lastdocfile);
   printf("%10d item specs for file creation\n",lastdocspec);
   if (testmode) PressEnter();
}                                    /* ReadConfigFile */


FUNCTION void GetVersion(void)
{
   TRACE CheckMgSystem(this_unit,"GetVersion",true,mytracelevel,NULL);
   DelChars(textline,0,1);
   textline[strlen(textline)-1]=0;
   if (isdigit(textline[0])) StringCopy(docversion,10,textline,-1);
}                                      /* GetVersion */


FUNCTION void GetLastRevision(void)
{
   TRACE CheckMgSystem(this_unit,"GetLastRevision",true,mytracelevel,NULL);
   DelChars(textline,0,1);
   textline[strlen(textline)-1]=0;
   if (strcmp(doclastrevision,textline)<0) StringCopy(doclastrevision,10,textline,-1);
}                                      /* GetLastRevision */


FUNCTION void AddLineTo(char *line,item_record *item)
{
   line_pointer aid,aid2;

   TRACE CheckMgSystem(this_unit,"AddLineTo",true,mytracelevel,NULL);
   aid=item->next;
   if (aid==NULL) {
      aid=malloc(sizeof(line_record));
      if (aid==NULL) AbortBecause("Not enough memory");
      item->next=aid;
   } else {
      while (aid!=NULL) {
         aid2=aid;
         aid=aid->next;
      }
      aid=malloc(sizeof(line_record));
      if (aid==NULL) AbortBecause("Not enough memory");
      aid2->next=aid;
   }
   aid->next=NULL;
   InitString(aid->text,line,100);
}                                    /* AddLineTo */


FUNCTION void ReadSourceFile(char *incname)
{
   maxstring key,copy;

   TRACE CheckMgSystem(this_unit,"ReadSourceFile",true,mytracelevel,NULL);
   InitString(key,"",-1);
   InitString(copy,"",-1);
   OpenDiskFile(incname,"r");
   addtoitem=false;
   printf("%10s Read file %s\r","",incname);
   while (ReadLine()!=EOF) {
      if (4<=tracelevel) CheckMgSystem(this_unit,"ReadSourceFile",true,mytracelevel,"line=<%s>",textline);
      StringCopy(copy,-1,textline,-1);
      SplitString(key,-1,textline,-1," ");
      if (Equal(key,"#define")) {
         SplitString(key,-1,textline,-1," ");
         if (HeadOf(key,"prog_version"))        GetVersion();          else
         if (HeadOf(key,"last_revision"))       GetLastRevision();
      } else if (Equal(key,"/*OriDoc")) {
         lastitem++;
         if (lastitem>=maxitem) AbortBecause("Const maxitem is too small");
         InitString(item[lastitem].name,textline,40);
         item[lastitem].next=NULL;
         addtoitem=true;
      } else if (Equal(key,"*/")) {
         addtoitem=false;
      } else if (addtoitem) {
         AddLineTo(copy,&item[lastitem]);
      }
   }
   CloseFile();
}                                    /* ReadSourceFile */


FUNCTION void ReadSourceFiles(void)
{
   int i;

   TRACE CheckMgSystem(this_unit,"ReadSourceFiles",true,mytracelevel,NULL);
   for (i=1; i<=lastincfile; i++) ReadSourceFile(incfile[i]);
   if (testmode) PressEnter();
}                                    /* ReadSourceFiles */


FUNCTION void NewItemToIndex(char *line)
{
   TRACE CheckMgSystem(this_unit,"NewItemToIndex",true,mytracelevel,line);
   lastindex++;
   if (lastindex>maxindex) AbortBecause("Const maxindex is too small");
   indx[lastindex]=malloc(sizeof(item_record));
   if (indx[lastindex]==NULL) AbortBecause("Not enough memory for index");
   InitString(indx[lastindex]->name,line,40);
   indx[lastindex]->next=NULL;
   if (indentmargin<strlen(line)) indentmargin=strlen(line);
   addtoindex=true;
}                                    /* NewItemToIndex */


FUNCTION void CheckItemSpecs(void)
{
   int i,s,found,format,mar,l;
   string40 spec;
   line_pointer aid;

   TRACE CheckMgSystem(this_unit,"CheckItemSpecs",true,mytracelevel,NULL);
   InitString(spec,"",40);
   format=0;
   for (s=1; s<=lastdocspec; s++) {
      found=0;
      StringCopy(spec,40,docspec[s],40);
      if (strncmp(spec,".ITEM=",6)==0) {
         DelChars(spec,0,6);
         for (i=1; i<=lastitem; i++) {
            if (Equal(spec,item[i].name)) found++;
         }
         if (found!=1) printf("%10d times found docspec %s in items\n",found,spec);
      } else {
         format++;
      }
   }
   printf("%10d format items found\n",format);
   for (i=1; i<=lastitem; i++) {
      s=0;
      addtoindex=false;
      mar=0;
      aid=item[i].next;
      while (aid!=NULL) {
         s++;
         if (aid->text[0]=='.') {
            if (strchr("TC",aid->text[1])!=NULL) {
               l=strlen(aid->text);
               if (mar<l) mar=l;
            } else {
               addtoindex=false;
            }
            if (aid->text[1]=='C') NewItemToIndex(aid->text);
         } else if ((aid->text[0]==' ') || (aid->text[0]==0)) {
            addtoindex=false;
         } else if (addtoindex) {
            AddLineTo(aid->text,indx[lastindex]);
         }
         aid=aid->next;
      }
      if (mar>=20) mar+=3; else mar+=6;
      item[i].indentmargin=mar;
      printf("%10d lines in item %s\n",s,item[i].name);
   }
   indentmargin+=3;
}                                    /* CheckItemSpecs */


FUNCTION void DumpIndex(void)
{
   int i;
   line_pointer aid;

   TRACE CheckMgSystem(this_unit,"DumpIndex",true,mytracelevel,NULL);
   OpenDiskFile(indexdump,"w");
   for (i=1; i<=lastindex; i++) {
      fprintf(disk.f,"%s\n",indx[i]->name);
      linesio++;
      aid=indx[i]->next;
      while (aid!=NULL) {
         fprintf(disk.f,"%s\n",aid->text);
         linesio++;
         aid=aid->next;
      }
      fprintf(disk.f,"\n");
      linesio++;
   }
   CloseFile();
}                                      /* DumpIndex */


FUNCTION void SortIndex(item_record_pointer indx[],int stop)
{
   int step,start,last,search;
   item_record_pointer temp;
   boolean exchange;

   TRACE CheckMgSystem(this_unit,"SortIndex",true,mytracelevel,NULL);
   step=stop / 3;
   if (step<1) step=1;
   while (step>0) {
      for (start=1; start<=step; start++) {
         last=start+step;
         while (last<=stop) {
            search=last-step;
            temp=indx[last];
            exchange=true;
            while ((search>0) && exchange) {
               if (strcmp(temp->name,indx[search]->name)<0) {
                  indx[search+step]=indx[search];
                  search=search-step;
               } else {
                  exchange=false;
               }
            }
            indx[search+step]=temp;
            last=last+step;
         }
      }
      step=(step+1) / 3;
   }
   printf("%10d commands in index\n",lastindex);
}                                      /* SortIndex */


FUNCTION void WriteBufferToFile(void)
{
   int i,cp;
   char *p;
   maxstring bufline;

   TRACE CheckMgSystem(this_unit,"WriteBufferTo",true,mytracelevel,NULL);
   InitString(bufline,"",-1);
   for (i=1; i<=lastbuffer; i++) {
      StringCopy(bufline,-1,linebuffer[i],100);
      p=strstr(bufline,"ORIXXXX");
      while (p!=NULL) {
         cp=(int)(p-bufline);
         DelChars(bufline,cp,7);
         InsertString(docname,bufline,cp,-1);
         p=strstr(bufline,"ORIXXXX");
      }
      fprintf(disk.f,"%s\n",bufline);
      linesio++;
      if (linesio % 64==0) printf("\r%10ld\r",linesio);
   }
}                                      /* WriteBufferTo */


FUNCTION void WriteBuffer(void)
{
   TRACE CheckMgSystem(this_unit,"WriteBuffer",true,mytracelevel,NULL);
   if (writeison) WriteBufferToFile();
   lastbuffer=0;
}                                      /* WriteBuffer */


FUNCTION void WriteLine(char *txt)
{
   TRACE CheckMgSystem(this_unit,"WriteLine",true,mytracelevel,NULL);
   lastbuffer=1;
   StringCopy(linebuffer[1],100,txt,0);
   WriteBuffer();
}                                      /* WriteLine */


FUNCTION void BufferLine(void)
{
   int p;
   maxstring copy,bufline;

   TRACE CheckMgSystem(this_unit,"BufferLine",true,mytracelevel,textline);
   InitString(copy,"",-1);
   InitString(bufline,"",-1);
   if (lastbuffer==0) {
      sprintf(copy,"%s%s",margin,textline);
   } else {
      p=strlen(linebuffer[lastbuffer])-1;
      if (linebuffer[lastbuffer][p]==' ') {
         sprintf(copy,"%s%s",linebuffer[lastbuffer],textline);
      } else {
         sprintf(copy,"%s %s",linebuffer[lastbuffer],textline);
      }
      StripTrailing(copy);
      lastbuffer--;
   }
   do {
      lastbuffer++;
      if (lastbuffer>maxbuffer) AbortBecause("Const maxbuffer is too small");
      p=(int)strlen(copy);
      if (p>right) {
         p=right;
         while ((p>=50) && (copy[p]!=' ')) p--;
         if (copy[p]!=' ') {
            p=right-1;
            InsertString(" ",copy,p,-1);
         }
      }
      StringCopy(bufline,-1,copy,-1);
      bufline[p]=0;
      DelChars(copy,0,p);
      StripBlanks(copy);
      InsertString(margin,copy,0,-1);
      if (testbuffer) {
         CheckMgSystem(this_unit,"BufferLine",true,mytracelevel,"B %s",bufline);
         CheckMgSystem(this_unit,"BufferLine",true,mytracelevel,"C %s",copy);
      }
      InitString(linebuffer[lastbuffer],bufline,100);
      StripTrailing(copy);
   } while (strlen(copy)>left);
}                                      /* BufferLine */


FUNCTION void NewPage(void)
{
   int i;
   maxstring extraline;

   TRACE CheckMgSystem(this_unit,"NewPage",true,mytracelevel,textline);
   InitString(extraline,"",-1);
   WriteBuffer();
   if (linesio>0) {
      for (i=1; i<=2; i++) WriteLine("");
      sprintf(extraline,"*** %s",textline);
   } else {
      sprintf(extraline,"%s",textline);
   }
   WriteLine(extraline);
   WriteLine("");
   left=0;
   right=78;
   margin[0]=0;
   indented=false;
}                                      /* NewPage */


FUNCTION void IndentLine(void)
{
   TRACE CheckMgSystem(this_unit,"IndentLine",true,mytracelevel,textline);
   WriteBuffer();
   left=3;
   textline[60]=0;
   InitString(linebuffer[1],"",100);
   sprintf(linebuffer[1],"   %s %50s",textline,"");
   left=indentmargin;
   linebuffer[1][left]=0;
   lastbuffer=1;
   StringCopy(margin,-1,blanks,-1);
   margin[left]=0;
   indented=true;
}                                    /* IndentLine */


FUNCTION void SetWriteOn(void)
{
   int s;

   TRACE CheckMgSystem(this_unit,"SetWriteOn",true,mytracelevel,NULL);
   writeison=systemcorrect;
   for (s=1; s<=lastselkey; s++) {
      if (! selkey[s].on) writeison=false;
   }
}                                    /* SetWriteOn */


FUNCTION void CheckSelectionKeys(void)
{
   int s;
   boolean on;
   char onoff;
   string10 key;

   TRACE CheckMgSystem(this_unit,"CheckSelectionKeys",true,mytracelevel,NULL);
   InitString(key,"",10);
   WriteBuffer();
   while (textline[0]!=0) {
      SplitString(key,10,textline,-1," ");
      onoff=key[strlen(key)-1];
      if (strchr("-+",onoff)!=NULL) {
         key[strlen(key)-1]=0;
      } else {
         onoff='+';
      }
      if (Equal(key,"ALL")) onoff='+';
      on=(onoff=='+');
      for (s=1; s<=lastselkey; s++) {
         if (Equal(selkey[s].key,key) || Equal(key,"ALL")) selkey[s].on=on;
      }
   }
   SetWriteOn();
}                                    /* CheckSelectionKeys */


FUNCTION void CheckSystemName(void)
{
   TRACE CheckMgSystem(this_unit,"CheckSystemName",true,mytracelevel,NULL);
   WriteBuffer();
   UpString(textline);
   systemcorrect=(Equal(textline,"ALL") || Equal(textline,systemname));
   SetWriteOn();
   if (testmode) printf("System: %s sc=%d wio=%d\n",textline,systemcorrect,writeison);
}                                    /* CheckSystemName */


FUNCTION void CheckCommand(void)
{
   maxstring keyword,copy;
   int code;

   TRACE CheckMgSystem(this_unit,"CheckCommand",true,mytracelevel,textline);
   InitString(keyword,"",-1);
   InitString(copy,textline,-1);
   SplitString(keyword,-1,textline,-1," ");
   DelChars(keyword,0,1);
   UpString(keyword);
   if (Equal(keyword,""))       WriteBuffer();                             else
   if (Equal(keyword,"MAR"))    left=(int)LongFromString(textline,&code);  else
   if (Equal(keyword,"LEFT"))   left=(int)LongFromString(textline,&code);  else
   if (Equal(keyword,"RIGHT"))  right=(int)LongFromString(textline,&code); else
   if (Equal(keyword,"C"))      IndentLine();                              else
   if (Equal(keyword,"T"))      IndentLine();                              else
   if (Equal(keyword,"PAGE"))   NewPage();                                 else
   if (Equal(keyword,"SEL"))    CheckSelectionKeys();                      else
   if (Equal(keyword,"SYS"))    CheckSystemName();                         else
   {
      printf("Error in command: %s %s\n",keyword,textline);
      PressEnter();
   }
   if (right==0) right=78;
   if (right>78) right=78;
   if (right<60) right=60;
   if (left<0)  left=0;
   if (left>40) left=40;
   StringCopy(margin,-1,blanks,-1);
   margin[left]=0;
}                                      /* CheckCommand */


FUNCTION void WriteIndex(void)
{
   int i;
   line_pointer aid;

   TRACE CheckMgSystem(this_unit,"WriteIndex",true,mytracelevel,NULL);
   writeison=true;
   systemcorrect=true;
   lastbuffer=0;
   OpenDiskFile(indexname,"w");
   sprintf(textline,"Index of commands for ORIDRAW and ORITOPS version %s",docversion);
   fprintf(disk.f,"%s\n\n",textline);
   linesio+=2;
   left=0;
   right=78;
   indented=false;
   margin[0]=0;
   for (i=1; i<=lastindex; i++) {
      StringCopy(textline,-1,indx[i]->name,40);
      CheckCommand();
      aid=indx[i]->next;
      while (aid!=NULL) {
         StringCopy(textline,-1,aid->text,100);
         BufferLine();
         aid=aid->next;
      }
   }
   WriteBuffer();
   CloseFile();
}                                      /* WriteIndex */


FUNCTION void OutputItem(item_record item)
{
   line_pointer aid;

   TRACE CheckMgSystem(this_unit,"OutputItem",true,mytracelevel,NULL);
   aid=item.next;
   indentmargin=item.indentmargin;
   while (aid!=NULL) {
      StringCopy(textline,-1,aid->text,100);
      if (textline[0]==0) {
         WriteBuffer();
         if (indented) {
            indented=false;
            left=0;
            margin[0]=0;
         }
         WriteLine("");
      } else if (textline[0]=='.') {
         CheckCommand();
      } else {
         if (textline[0]==' ') {
            WriteBuffer();
            if (indented) {
               indented=false;
               left=0;
               margin[0]=0;
            }
         }
         BufferLine();
      }
      aid=aid->next;
   }
}                                    /* OutputItem */


FUNCTION void GenerateFile(char *fspec)
{
   int i,s;
   maxstring spec,fname;
   string40 key;
   char onoff;

   TRACE CheckMgSystem(this_unit,"GenerateFile",true,mytracelevel,fspec);
   InitString(spec,"",-1);
   InitString(fname,"",-1);
   InitString(key,"",40);
   writeison=true;
   systemcorrect=true;
   printf("%10s Write %s\r","",fspec);
   SplitString(fname,-1,fspec,40," ");
   SplitString(docname,10,fspec,40," ");
   lastselkey=0;
   while (fspec[0]!=0) {
      SplitString(key,40,fspec,40," ");
      onoff=key[strlen(key)-1];
      if (strchr("-+",onoff)!=NULL) {
         key[strlen(key)-1]=0;
      } else {
         onoff='+';
      }
      lastselkey++;
      if (lastselkey>maxselkey) AbortBecause("Const maxselkey is too small");
      InitString(selkey[lastselkey].key,key,10);
      selkey[lastselkey].on=(onoff=='+');
      if (! selkey[lastselkey].on) writeison=false;
   }
   OpenDiskFile(fname,"w");
   left=0;
   right=78;
   indented=false;
   indentmargin=0;
   margin[0]=0;
   lastbuffer=0;
   sprintf(textline,"ORIXXXX %s                               Last revision dd.%s",docversion,doclastrevision);
   WriteLine(textline);
   WriteLine("                                  Author: M.J.van Gelder");
   WriteLine("                                          Groningen,   Holland");
   WriteLine("                                   Email: M.J.van.Gelder@kvi.nl");
   WriteLine("");
   for (s=1; s<=lastdocspec; s++) {
      StringCopy(spec,-1,docspec[s],40);
      if (strncmp(spec,".ITEM=",6)==0) {
         DelChars(spec,0,6);
         for (i=1; i<=lastitem; i++) {
            if (Equal(spec,item[i].name)) OutputItem(item[i]);
         }
      } else {
         StringCopy(textline,-1,spec,-1);
         CheckCommand();
      }
   }
   WriteBuffer();
   CloseFile();
}                                    /* GenerateFile */


FUNCTION void GenerateFiles(void)
{
   int i;

   TRACE CheckMgSystem(this_unit,"GenerateFiles",true,mytracelevel,NULL);
   for (i=1; i<=lastdocfile; i++) GenerateFile(docfile[i]);
}                                    /* GenerateFiles */


FUNCTION int /* OriDoc */ main (int argc,char *argv[])
{
   InitializeOriDoc(&argc,argv);
   ReadConfigFile();
   ReadSourceFiles();
   CheckItemSpecs();
   printf("%-10s is version\n",docversion);
   printf("%-10s is date of last revision\n",doclastrevision);
   printf("%10d doc items found\n",lastitem);
   if (testmode) DumpIndex();
   SortIndex(indx,lastindex);
   WriteIndex();
   GenerateFiles();
   ProgramTrailer(stdout);
   return(0);
}
