/*
      File: mgprogio.c
    Author: Maarten van Gelder,  KVI / RuG,  Groningen,  Nederland
   Purpose: Provide basic functions for my C programs
     First version dd.19900913
*/

/*extracth Local */

/*lint -library */

#define last_revision "19990824"
#define this_unit "MgProgIo"
#define prog_version "unit"
#define mytracelevel 9

/*extracth Global */

#if defined(__VMS__)
#pragma nostandard
#endif

#include <ctype.h>
#include <stdio.h>
#include <stdarg.h>
#include <assert.h>
#include <errno.h>
#include <limits.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <math.h>
#include <float.h>

#if defined(__MSDOS__)
#include <dos.h>
/*extracth Local */
extern unsigned _stklen=0XF000;
/*extracth Global */
#endif

#if defined(__VMS__)
#include <starlet.h>
#include <stsdef.h>
#include <jpidef.h>
#include <clidef.h>
#include <lib$routines.h>
#include <ssdef.h>
#include <lnmdef.h>
#include <prvdef.h>
#include <descrip.h>
#define DSC_DESCR_S struct dsc$descriptor_s
#endif


#define FUNCTION /* */
#define CHECKINIT      if (( mytracelevel <= tracelevel ) || ! initialized )
#define TRACE          if (  mytracelevel <= tracelevel )


#define maxstringlength 255
#define extra_string_bytes 5
#define extra_string "\0MVG\0"
typedef char maxstring[maxstringlength+extra_string_bytes];

typedef char string10[10+extra_string_bytes];
typedef char string12[12+extra_string_bytes];
typedef char string20[20+extra_string_bytes];
typedef char string30[30+extra_string_bytes];
typedef char string40[40+extra_string_bytes];
typedef char string50[50+extra_string_bytes];
typedef char string60[60+extra_string_bytes];
typedef char string70[70+extra_string_bytes];
typedef char string80[80+extra_string_bytes];
typedef char string90[90+extra_string_bytes];
typedef char string100[100+extra_string_bytes];
typedef char string110[110+extra_string_bytes];
typedef char string120[120+extra_string_bytes];
typedef char string128[128+extra_string_bytes];

#define boolean int
#define false 0
#define true  1

#if defined(__MSDOS__)
#define slash_char    '\\'
#define slash_str     "\\"
#define dereference   "..\\"
#else
#define slash_char    '/'
#define slash_str     "/"
#define dereference   "../"
#endif

#define Ctrl_A           1
#define Ctrl_B           2
#define Ctrl_C           3
#define Ctrl_D           4
#define Ctrl_E           5
#define Ctrl_F           6
#define Ctrl_G           7
#define Ctrl_H           8
#define Ctrl_I           9
#define Ctrl_J          10
#define Ctrl_K          11
#define Ctrl_L          12
#define Ctrl_M          13
#define Ctrl_N          14
#define Ctrl_O          15
#define Ctrl_P          16
#define Ctrl_Q          17
#define Ctrl_R          18
#define Ctrl_S          19
#define Ctrl_T          20
#define Ctrl_U          21
#define Ctrl_V          22
#define Ctrl_W          23
#define Ctrl_X          24
#define Ctrl_Y          25
#define Ctrl_Z          26
#define Esc_Key         27

#define Page_Up        -71
#define Page_Down      -72
#define Up_Arrow       -73
#define Down_Arrow     -74
#define Left_Arrow     -75
#define Right_Arrow    -76
#define Home_Key       -77
#define End_Key        -78
#define Del_Key        -79
#define Ins_Key        -80
#define Del_Back       -81

#define GFI_ABSENT      -1
#define GFI_DIR         -2

#if defined(__VMS__)
#define MG_RETURN_CODE_OK   1
#else
#define MG_RETURN_CODE_OK   0
#endif

#define longnamefile "00long.nam"


/*extracth Vars */

string20 global_program_name,global_program_date,global_program_version,Hexal;
boolean QuitSeen,testmode,checkstrings,mgprogio_test,in_batch,dontchecknonascii;
int tracelevel;
char bell_char;

/*extracth Local */

static boolean initialized=false;
static boolean tracefileopen=false,tracefileok=true;
static FILE *tracefile=NULL;


/*extracth Functions */


FUNCTION void ShowErrorMVaList(char *fmt,va_list args)
{
   fflush(stdout);
   vfprintf(stdout,fmt,args);
   fflush(stdout);
   if (in_batch) {
      vfprintf(stderr,fmt,args);
      fflush(stderr);
   }
}                                      /* ShowErrorMVaList */


FUNCTION void ShowErrorMessage(char *fmt,...)
{
   va_list args;

   va_start(args,fmt);
   ShowErrorMVaList(fmt,args);
   va_end(args);
}                                      /* ShowErrorMessage */


FUNCTION void AbortBecause(char *fmt,...)
{
   va_list args;

/*lint -save -e737 */
   va_start(args,fmt);
/*lint -restore */
   ShowErrorMessage("\n >>> Program \"%s\" stopped because of:\n",global_program_name);
   ShowErrorMVaList(fmt,args);
   ShowErrorMessage("\n");
   va_end(args);
   exit(255);
}                                      /* AbortBecause */


FUNCTION void CheckMgSystem(char *libname,char *procname,boolean initialized,int requestlevel,char *fmt,...)
{
   va_list args;
   maxstring proc,fname;
   int i;
   char *envptr;

/*lint -save -e737 */
   va_start(args,fmt);
/*lint -restore */
   if (! initialized) {
      for (i=0; (procname[i]!=0) && (procname[i]!=' '); i++) proc[i]=procname[i];
      proc[i]=0;
      AbortBecause("%s.%s not Initialized",libname,proc);
   }
   if (requestlevel<1) requestlevel=1;
   if (tracefileok && (requestlevel<=tracelevel)) {
      if (! tracefileopen) {
         envptr=getenv("TRACEFILE");
         if (envptr==NULL) strcpy(fname,"trace.tmp"); else strcpy(fname,envptr);
         tracefile=fopen(fname,"w");
         if (tracefile==NULL) {
            tracefileok=false;
         } else {
            tracefileopen=true;
            fprintf(tracefile,"Trace level is %d\n",tracelevel);
         }
      }
      if (tracefileok) {
         fprintf(tracefile,"%2d %s.%s",requestlevel,libname,procname);
         if (fmt!=NULL) {
            fprintf(tracefile," ");
            (void)vfprintf(tracefile,fmt,args);
         }
         fprintf(tracefile,"\n");
         fflush(tracefile);
      }
   }
   va_end(args);
}                                    /* CheckMgSystem */


FUNCTION void CheckNonAscii(boolean doit)
{
   CHECKINIT CheckMgSystem(this_unit,"CheckNonAscii",initialized,mytracelevel,NULL);
   dontchecknonascii=! doit;
}                                    /* CheckNonAscii */


FUNCTION void StringCheck(char *str,char *where,int maxchars)
{
   int i,chk;
   boolean ok,nonascii;
   maxstring msg;

   strcpy(msg,"");
   ok=true;
   if ((maxchars>0) && ((strcmp(str+maxchars+1,extra_string+1)!=0) || (str[maxchars]!=0))) {
      chk=1;
      while ((strcmp(str+chk+1,extra_string+1)!=0) && (chk<2048)) chk++;
      CheckMgSystem(this_unit,"StrCheck",true,0,"%s: ILL check str=<%s> extra(%d)=<%c%c%c> posfound=%d",where,str,maxchars,str[maxchars+1],str[maxchars+2],str[maxchars+3],chk);
      ok=false;
      strcat(msg," Length_error");
   }
   nonascii=false;
   if (! dontchecknonascii) {
      for (i=0; str[i]!=0; i++) {
         if (! isprint(str[i]) && (str[i]!='\t')) nonascii=true;
      }
      if (nonascii) {
         CheckMgSystem(this_unit,"StrCheck",true,0,"%s: ILL char(s) <%s>",where,str);
         if (! ok) strcat(msg," and");
         ok=false;
         strcat(msg," Contents_NON_ASCII");
      }
   }
   if (! ok) {
      if (! in_batch && (tracelevel<8)) tracelevel=8;
      ShowErrorMessage(">> StringCheck ERROR in %s:%s\n",where,msg);
      ShowErrorMessage(" > String: ..<%s>..\n",str);
      if (strstr(msg,"Leng")!=NULL) {
         ShowErrorMessage(" > ILL check extra(%d)=<%c%c%c> posfound=%d\n",maxchars,str[maxchars+1],str[maxchars+2],str[maxchars+3],chk);
      }
      if (checkstrings) AbortBecause("String error: %s",msg);
   }
}                                      /* StringCheck */


FUNCTION void StringCopy(char *dest,int maxdest,char *from,int maxfrom)
{
#if defined(__MSDOS__)
   char local_var;
#endif

   if (maxdest<=0) maxdest=maxstringlength;
   if (maxfrom<0) maxfrom=maxstringlength;
   if (checkstrings) {
#if defined(__MSDOS__)
      TRACE CheckMgSystem(this_unit,"StringCopy",true,mytracelevel,"... %d <%s> %d  Stack: &local_var=%p",maxdest,from,maxfrom,&local_var);
#else
      TRACE CheckMgSystem(this_unit,"StringCopy",true,mytracelevel,"... %d <%s> %d",maxdest,from,maxfrom);
#endif
      if ((maxfrom>0) && (from[0]!=0)) StringCheck(from,"StringCopy.From",maxfrom);
      dest[0]=0;
      StringCheck(dest,"StringCopy.Dest",maxdest);
   }
   strncpy(dest,from,(unsigned int)maxdest);
   dest[maxdest]=0;
   if ((strcmp(dest+maxdest+1,extra_string+1)!=0) || checkstrings) StringCheck(dest,"StringCopy.New",maxdest);
}                                      /* StringCopy */


FUNCTION void InitString(char *txt,char *info,int max)
{
   int i;

   if (max<=0) max=maxstringlength;
   for (i=0; i<extra_string_bytes; i++) txt[max+i]=extra_string[i];
   TRACE CheckMgSystem(this_unit,"InitString",initialized,mytracelevel,"%d <%c%c%c> <%s>",max,txt[max+1],txt[max+2],txt[max+3],info);
   StringCopy(txt,max,info,0);
}                                    /* InitString */


FUNCTION void StringCat(char *dest,int maxdest,char *from,int maxfrom)
{
   int dl;

   if (maxdest<=0) maxdest=maxstringlength;
   if (maxfrom<0) maxfrom=maxstringlength;
   dl=(int)strlen(dest);
   if (checkstrings) {
      if ((maxfrom>0) && (from[0]!=0)) StringCheck(from,"StringCat.From",maxfrom);
      if (dest[0]!=0) StringCheck(dest,"StringCat.To",maxdest);
   }
   if (maxdest-dl>0) strncat(dest,from,(unsigned int)(maxdest-dl));
   dest[maxdest]=0;
   if ((strcmp(dest+maxdest+1,extra_string+1)!=0) || checkstrings) StringCheck(dest,"StringCat.New",maxdest);
}                                      /* StringCat */


/*extracth Extra */

#if ! defined(_lint)
#if defined(strcpy)
#undef strcpy
#endif
#if defined(strncpy)
#undef strncpy
#endif
#if defined(strcat)
#undef strcat
#endif
#if defined(strncat)
#undef strncat
#endif
#define strcpy StringCopy
#define strncpy StringCopy
#define strcat StringCat
#define strncat StringCat
#endif
#define Equal(str1,str2)   (strcmp(str1,str2)==0)
#define HeadOf(str1,str2)   (strncmp(str1,str2,strlen(str2))==0)


FUNCTION char *AllocString(int max)
{
   char *str;

   CHECKINIT CheckMgSystem(this_unit,"AllocString",initialized,mytracelevel,NULL);
   if (max<=0) max=maxstringlength;
   str=(char *)malloc(max+extra_string_bytes);
   if (str!=NULL) InitString(str,"",max);
   return str;
}                                    /* AllocString */


FUNCTION void UpString(char *str)
{
   CHECKINIT CheckMgSystem(this_unit,"UpString",initialized,mytracelevel,NULL);
   while (*str!=0) {
      *str=(char)toupper(*str);
      str++;
   }
}                                      /* UpString */


FUNCTION void LowString(char *str)
{
   CHECKINIT CheckMgSystem(this_unit,"LowString",initialized,mytracelevel,NULL);
   while (*str!=0) {
      *str=(char)tolower(*str);
      str++;
   }
}                                      /* LowString */


FUNCTION void GetSymbol(char *answer,char *name)
{
   maxstring line;
   char *ans;

   CHECKINIT CheckMgSystem(this_unit,"GetSymbol",initialized,mytracelevel,name);
   InitString(line,name,-1);
#if ! defined(__UNIX__)
   UpString(line);
#endif
   ans=getenv(line);
   if (ans==NULL) answer[0]=0; else sprintf(answer,"%s",ans);
   if (mgprogio_test) printf("... Symbol: <%s>   Answer: <%s>\n",line,answer);
}                                    /* GetSymbol */


FUNCTION void InitializeMgProgIo(char *p_name,char *p_date,char *p_version,int *argc,char *argv[])
{
   int i;
   maxstring argi;

   if (initialized) return;
   initialized=true;
   InitString(argi,"",-1);
   bell_char='\007';
   in_batch=false;
   GetSymbol(argi,"BATCH");
   UpString(argi);
   if (Equal(argi,"YES")) in_batch=true;
   testmode=false;
   mgprogio_test=false;
   checkstrings=false;
   dontchecknonascii=false;
   tracelevel= -1;
   for (i=1; i<(*argc); i++) {
      if ((argv[i]!=NULL) && (*argv[i]!='\0')) {
         StringCopy(argi,-1,argv[i],0);
         LowString(argi);
         if (Equal(argi,"-trace"))         tracelevel=1;                else
         if (Equal(argi,"-trace2"))        tracelevel=2;                else
         if (Equal(argi,"-trace3"))        tracelevel=3;                else
         if (Equal(argi,"-trace4"))        tracelevel=4;                else
         if (Equal(argi,"-trace5"))        tracelevel=5;                else
         if (Equal(argi,"-trace6"))        tracelevel=6;                else
         if (Equal(argi,"-trace7"))        tracelevel=7;                else
         if (Equal(argi,"-trace8"))        tracelevel=8;                else
         if (Equal(argi,"-trace9"))        tracelevel=9;                else
         if (Equal(argi,"-testmode"))      testmode=true;               else
         if (Equal(argi,"-batch"))         in_batch=true;               else
         if (Equal(argi,"-checkstrings"))  checkstrings=true;           else
         if (Equal(argi,"-checkstring"))   checkstrings=true;           else
         {
            /* ignore */
         }
      }
   }
   if (in_batch) bell_char=' ';
   if (tracefileok && (tracelevel>0)) {
      CheckMgSystem(this_unit,"InitializeMgProgIo",true,0,"Compiled at %s %s",__DATE__,__TIME__);
      for (i=0; i<(*argc); i++) {
         if (argv[i]!=NULL) fprintf(tracefile,"   ARGV %d : %s\n",i,argv[i]);
      }
   }
   InitString(global_program_name,p_name,20);
   InitString(global_program_version,p_version,20);
   InitString(global_program_date,last_revision,20);
   if (strcmp(global_program_date,p_date)<0) StringCopy(global_program_date,20,p_date,0);
   InitString(Hexal,"0123456789ABCDEF",20);
   QuitSeen=false;
}                                      /* InitializeMgProgIo */


FUNCTION void CheckUpdate(char *p_date)
{
   CHECKINIT CheckMgSystem(this_unit,"CheckUpdate",initialized,mytracelevel,NULL);
   if (strcmp(global_program_date,p_date)<0) StringCopy(global_program_date,20,p_date,0);
}                                      /* CheckUpdate */


FUNCTION void DelChars(char *str,int start,int count)
{
   int from,to;

   CHECKINIT CheckMgSystem(this_unit,"DelChars",initialized,mytracelevel,NULL);
   if ((count<=0) || (start<0) || (start>=(int)strlen(str))) return;
   to=start;
   from=to+count;
   if (start+count<=(int)strlen(str)) {
      while (str[from]!=0) {
         str[to]=str[from];
         from++;
         to++;
      }
   }
   str[to]=0;
}                                      /* DelChars */


FUNCTION void InsertString(char *extra,char *dest,int start,int maxdest)
{
   int i,to,elen,slen;

   CHECKINIT CheckMgSystem(this_unit,"InsertString",initialized,mytracelevel,NULL);
   if (maxdest<=0) maxdest=maxstringlength;
   if (extra[0]==0) return;
   if (start<0) start=0;
   slen=(int)strlen(dest);
   if (start>=slen) start=slen;
   elen=(int)strlen(extra);
   if ((elen+start)>maxdest) elen=maxdest-start;
   if (checkstrings) {
      if (extra[0]!=0) StringCheck(extra,"InsertString.From",0);
      if (dest[0]!=0) StringCheck(dest,"InsertString.To",0);
   }
   for (i=maxdest-1; i>start; i--) dest[i]=dest[i-elen];
   dest[maxdest]=0;
   to=start;
   for (i=0; i<elen; i++) {
      dest[to]=extra[i];
      to++;
   }
   if ((strcmp(dest+maxdest+1,extra_string+1)!=0) || checkstrings) StringCheck(dest,"InsertString.New",maxdest);
}                                      /* InsertString */


FUNCTION void StripTrailing(char *str)
{
   int pos;

   CHECKINIT CheckMgSystem(this_unit,"StripTrailing",initialized,mytracelevel,NULL);
   pos=(int)strlen(str)-1;
   while ((pos>0) && (str[pos]==' ')) pos--;
   if ((pos==0) && (str[0]==' ')) pos= -1;
   str[pos+1]=0;
}                                      /* StripTrailing */


FUNCTION void StripBlanks(char *str)
{
   int pos,i;

   CHECKINIT CheckMgSystem(this_unit,"StripBlanks",initialized,mytracelevel,NULL);
   StripTrailing(str);
   pos=0;
   while (str[pos]==' ') pos++;
   i=0;
   while (str[pos]!=0) {
      str[i]=str[pos];
      i++;
      pos++;
   }
   str[i]=0;
}                                      /* StripBlanks */


FUNCTION void StripMultipleBlanks(char *str)
{
   int i,j;
   boolean prevblank;

   CHECKINIT CheckMgSystem(this_unit,"StripMultipleBlanks",initialized,mytracelevel,NULL);
   i=0;
   j=0;
   prevblank=true;
   while (str[i]!=0) {
      if ((str[i]!=' ') || ! prevblank) {
         str[j]=str[i];
         j++;
      }
      prevblank=(str[i]==' ');
      i++;
   }
   str[j]=0;
}                                      /* StripMultipleBlanks */


FUNCTION void SubString(char *dest,int maxdest,char *from,int start,int count,int maxfrom)
{
   int i,ifr,ids;

   CHECKINIT CheckMgSystem(this_unit,"SubString",initialized,mytracelevel,NULL);
   if (maxdest<=0) maxdest=maxstringlength;
   if (maxfrom<0) maxfrom=maxstringlength;
   ids=0;
   if (checkstrings) StringCheck(from,"SubString.From",maxfrom);
   if ((start>=0) && (start<(int)strlen(from))) {
      ifr=start;
      if ((maxfrom>0) && (count>maxfrom-start)) count=maxfrom-start;
      if (count>maxdest) count=maxdest;
      for (i=1; i<=count; i++) {
         if (from[ifr]==0) break;
         dest[ids]=from[ifr];
         ifr++;
         ids++;
      }
   }
   dest[ids]=0;
   if ((strcmp(dest+maxdest+1,extra_string+1)!=0) || checkstrings) StringCheck(dest,"SubString.New",maxdest);
}                                      /* SubString */


FUNCTION void SplitString(char *head,int maxhead,char *tail,int maxtail,char *split_on)
{
   char *sptr;

   CHECKINIT CheckMgSystem(this_unit,"SplitString",initialized,mytracelevel,NULL);
   if (maxhead<=0) maxhead=maxstringlength;
   if (maxtail<=0) maxtail=maxstringlength;
   sptr=strpbrk(tail,split_on);
   if (sptr!=NULL) {
      SubString(head,maxhead,tail,0,(int)(sptr-tail),maxtail);
      DelChars(tail,0,(int)(sptr-tail+1));
   } else {
      StringCopy(head,maxhead,tail,maxtail);
      tail[0]=0;
   }
   StripBlanks(head);
   StripBlanks(tail);
}                                      /* SplitString */


FUNCTION void SplitStringRight(char *head,int maxhead,char *tail,int maxtail,char *split_on)
{
   int i;

   CHECKINIT CheckMgSystem(this_unit,"SplitStringRight",initialized,mytracelevel,NULL);
   if (maxhead<=0) maxhead=maxstringlength;
   if (maxtail<=0) maxtail=maxstringlength;
   i=(int)strlen(tail)-1;
   while ((strchr(split_on,tail[i])==NULL) && (i>0)) i--;
   if (strchr(split_on,tail[i])!=NULL) {
      SubString(head,maxhead,tail,0,i,maxtail);
      DelChars(tail,0,i+1);
   } else {
      StringCopy(head,maxhead,tail,maxtail);
      tail[0]=0;
   }
   StripBlanks(head);
   StripBlanks(tail);
}                                      /* SplitStringRight */


FUNCTION void StandardizeDate(char *date,boolean punctuated)
{
   int i,j;
   char previous;

   j=0;
   previous=' ';
   for (i=0; date[i]!=0; i++) {
      if (isdigit(date[i]) || ((date[i]==' ') && (previous!=' ') && (j>=6))) {
         previous=date[i];
         if (j!=i) date[j]=previous;
         j++;
      }
   }
   if ((j>0) && (date[j-1]==' ')) j--;
   date[j]=0;
   if (date[0]==0) return;
   if ((date[6]==' ') || (strlen(date)<8)) {
      if (date[0]<'6') InsertString("20",date,0,-1); else InsertString("19",date,0,-1);   /* for dates shorter than 8 */
      j+=2;
   }
   if ((j<=8) && (j%2==1)) {
      j--;
      date[j]=0;
   }
   if (strchr(date,' ')==NULL) {
      if (j>8) {
         InsertString(" ",date, 8,-1);
         j++;
      }
   }
   if ((j>8) && (j<=15) && (j%2==0)) {
      j--;
      date[j]=0;
   }
   date[18]=0;
   if (j>18) j=18;
   if (date[j-1]==' ') {
      j--;
      date[j]=0;
   }
   if (punctuated) {
      if (j>15) InsertString(".",date,15,-1);
      if (j>13) InsertString(":",date,13,-1);
      if (j>11) InsertString(":",date,11,-1);
      if (j> 6) InsertString("-",date, 6,-1);
      if (j> 4) InsertString("-",date,4,-1);
   }
}                                    /* StandardizeDate */


FUNCTION void FormatDateTime(char *datime,time_t tv)
{
   struct tm *t1;

   CHECKINIT CheckMgSystem(this_unit,"FormatDateTime",initialized,mytracelevel,NULL);
   tzset();
   t1=localtime(&tv);
   sprintf(datime,"%04d%02d%02d %02d%02d%02d",t1->tm_year+1900,t1->tm_mon+1,
      t1->tm_mday,t1->tm_hour,t1->tm_min,t1->tm_sec);
}                                    /* FormatDateTime */


FUNCTION void DateTime(char *datime)
{
   time_t tv;

   CHECKINIT CheckMgSystem(this_unit,"DateTime",initialized,mytracelevel,NULL);
   tv=time(NULL);
   FormatDateTime(datime,tv);
}                                    /* DateTime */


FUNCTION double HhMmSs(void)
{
#if defined(__MSDOS__)
   struct REGPACK regs;
   double tim;
#else
   time_t tv;
   struct tm *t1;
#endif

   CHECKINIT CheckMgSystem(this_unit,"HhMmSs",initialized,mytracelevel,NULL);
#if defined(__MSDOS__)
   regs.r_ax=0x2C00;
   intr(0x21,&regs);
   tim=(((regs.r_dx >> 8) + 60.0 * (regs.r_cx & 0xFF)) + 3600.0 * (regs.r_cx >> 8)) + (regs.r_dx & 0xFF) * 0.01;
   return tim;
#else
   tv=time(NULL);
   t1=localtime(&tv);
   return ((t1->tm_hour*60.0)+t1->tm_min)*60.0+t1->tm_sec;
#endif
}                                      /* HhMmSs */


FUNCTION void MonthName(char *name,int month,int max)
{
   CHECKINIT CheckMgSystem(this_unit,"MonthName",initialized,mytracelevel,NULL);
   switch (month) {
      case  1: StringCopy(name,max,"Jan",0);   break;
      case  2: StringCopy(name,max,"Feb",0);   break;
      case  3: StringCopy(name,max,"Mar",0);   break;
      case  4: StringCopy(name,max,"Apr",0);   break;
      case  5: StringCopy(name,max,"May",0);   break;
      case  6: StringCopy(name,max,"Jun",0);   break;
      case  7: StringCopy(name,max,"Jul",0);   break;
      case  8: StringCopy(name,max,"Aug",0);   break;
      case  9: StringCopy(name,max,"Sep",0);   break;
      case 10: StringCopy(name,max,"Oct",0);   break;
      case 11: StringCopy(name,max,"Nov",0);   break;
      case 12: StringCopy(name,max,"Dec",0);   break;
      default: StringCopy(name,max,"???",0);
   }
}                                      /* MonthName */


FUNCTION int MonthNumber(char *monthname)
{
   string10 dup;
   int month;

   TRACE CheckMgSystem(this_unit,"MonthNumber",true,mytracelevel,monthname);
   InitString(dup,monthname,10);
   UpString(dup);
   dup[3]=0;
   if (Equal(dup,"JAN"))   month= 1;       else
   if (Equal(dup,"FEB"))   month= 2;       else
   if (Equal(dup,"MAA"))   month= 3;       else
   if (Equal(dup,"MAR"))   month= 3;       else
   if (Equal(dup,"APR"))   month= 4;       else
   if (Equal(dup,"MEI"))   month= 5;       else
   if (Equal(dup,"MAY"))   month= 5;       else
   if (Equal(dup,"JUN"))   month= 6;       else
   if (Equal(dup,"JUL"))   month= 7;       else
   if (Equal(dup,"AUG"))   month= 8;       else
   if (Equal(dup,"SEP"))   month= 9;       else
   if (Equal(dup,"OCT"))   month=10;       else
   if (Equal(dup,"OKT"))   month=10;       else
   if (Equal(dup,"NOV"))   month=11;       else
   if (Equal(dup,"DEC"))   month=12;       else
   {
      month=0;
   }
   return month;
}                                    /* MonthNumber */


FUNCTION void LongToString(char *txt,long number,int width)
{
   string20 format;

   CHECKINIT CheckMgSystem(this_unit,"LongToString",initialized,mytracelevel,NULL);
   InitString(format,"",20);
   if (width<1) width=1;
   if (width>40) width=40;
   sprintf(format,"%%%dld",width);
   sprintf(txt,format,number);
}                                      /* LongToString */


FUNCTION void DoubleToString(char *txt,double number,int width,int fraction)
{
   string20 format;

   CHECKINIT CheckMgSystem(this_unit,"DoubleToString",initialized,mytracelevel,NULL);
   InitString(format,"",20);
   if (width<1) width=1;
   if (width>40) width=40;
   if (fraction<0) fraction=0;
   if (fraction>40) fraction=40;
   sprintf(format,"%%%d.%df",width,fraction);
   sprintf(txt,format,number);
}                                      /* DoubleToString */


FUNCTION double DoubleFromString(char *txt,int *code)
{
   char *e;
   double r;
   int lcode;

   CHECKINIT CheckMgSystem(this_unit,"DoubleFromString",initialized,mytracelevel,NULL);
   lcode=0;
   if (strchr("-+0123456789",*txt)==NULL) lcode=1; else lcode=0;
   r=strtod(txt,&e);
   if ((void *)*e!=NULL) lcode=1;
   if (code!=NULL) *code=lcode;
   return r;
}                                      /* DoubleFromString */


FUNCTION long LongFromString(char *txt,int *code)
{
   long l;
   char *e;
   int lcode;

   CHECKINIT CheckMgSystem(this_unit,"LongFromString",initialized,mytracelevel,NULL);
   lcode=0;
   l=strtol(txt,&e,10);
   if ((void *)*e!=NULL) lcode=1;
   if (code!=NULL) *code=lcode;
   return l;
}                                      /* LongFromString */


FUNCTION int IntFromPar(char *txt,int deflt)
{
   int l;
   char *e;

   CHECKINIT CheckMgSystem(this_unit,"IntFromPar",initialized,mytracelevel,NULL);
   l=(int)strtol(txt,&e,10);
   if ((void *)*e!=NULL) l=deflt;
   return l;
}                                      /* IntFromPar */


FUNCTION int Round(double real)
{
   CHECKINIT CheckMgSystem(this_unit,"Round",initialized,mytracelevel,NULL);
   return (int)ceil(real-0.5);
}                                      /* Round */


FUNCTION int ReadLn(FILE *inputfile,char *line,int max)
{
   int c,len;

   if (((mytracelevel-1)<=tracelevel) || ! initialized) CheckMgSystem(this_unit,"ReadLn",initialized,mytracelevel-1,"%p %p max=%d",inputfile,line,max);
   if (max<=0) max=maxstringlength;
   len=0;
   c=getc(inputfile);
   while (c!=EOF) {
      line[len]=(char)c;
      len++;
      if (len>=max) break;
      if (c=='\n') {
         len--;
         break;
      }
      if (c=='\r') len--;
      c=getc(inputfile);
   }
   line[len]=0;
   if (strcmp(line+max+1,extra_string+1)!=0) StringCheck(line,"ReadLn",max);
   if ((c==EOF) && (len==0)) return EOF;
   return len;
}                                      /* ReadLn */


FUNCTION char *ProgramHeaderString(void)
{
   int i;
   static maxstring header;

   CHECKINIT CheckMgSystem(this_unit,"ProgramHeaderString",initialized,mytracelevel,NULL);
   InitString(header,"",-1);
   if (strchr(global_program_date,'-')==NULL) {
      InsertString("-",global_program_date,6,20);
      InsertString("-",global_program_date,4,20);
   }
   sprintf(header,"%s   (%s  %s) ",global_program_name,global_program_version,global_program_date);
   for (i=0; i<8; i++) StringCat(header,-1,"__________",0);
   header[79]=0;
   return &header[0];
}                                      /* ProgramHeaderString */


FUNCTION void ProgramHeader(FILE *fp)
{
   CHECKINIT CheckMgSystem(this_unit,"ProgramHeader",initialized,mytracelevel,NULL);
   fprintf(fp,"\n%s\n\n",ProgramHeaderString());
}                                      /* ProgramHeader */


FUNCTION void ProgramTrailer(FILE *fp)
{
   int i,len;

   CHECKINIT CheckMgSystem(this_unit,"ProgramTrailer",initialized,mytracelevel,NULL);
   len=39-(int)strlen(global_program_name);
   fprintf(fp,"\n%s ",global_program_name);
   for (i=1; i<len; i++) fprintf(fp,"_");
   fprintf(fp," Ready _________________________________\n");
}                                      /* ProgramTrailer */


FUNCTION boolean AskIf(char *txt,int deflt)
{
   int answer;

   CHECKINIT CheckMgSystem(this_unit,"AskIf",initialized,mytracelevel,NULL);
   answer='?';
   while (answer=='?') {
      answer='N';
      if (deflt) answer='Y';
      printf("%s (Y/N/?) [%c]? ",txt,answer);
      answer=getchar();
      if (answer=='\n') return deflt;
      if (answer==EOF)  return 0;
/*lint -save -e722 */
      while (getchar()!='\n');
/*lint -restore */
      if (answer=='?') {
         printf("  > Answer with Y or N followed by the <ENTER> key.\n");
         printf("  > Answering with only <ENTER> key results in the value given between the [].\n");
         printf("  > Answering with end_of_file results in NO.\n");
      }
   }
   if (strchr("YyJj",answer)) return true;
   return false;
}                                      /* AskIf */


FUNCTION int AskInt(char *prompt,int dflt,int min,int max)
{
   int number,code;
   maxstring ptxt,answer;
   boolean ok;

   CHECKINIT CheckMgSystem(this_unit,"AskInt",initialized,mytracelevel,NULL);
   InitString(ptxt,"",-1);
   InitString(answer,"",-1);
   sprintf(ptxt,"%s (%d..%d)[%d]? ",prompt,min,max,dflt);
   sprintf(answer,"%d",dflt);
   ok=false;
   number=dflt;
   while (! ok) {
      number=dflt;
      printf("\r%79s\r%s"," ",ptxt);
      (void)ReadLn(stdin,answer,-1);
      code=0;
      if (mgprogio_test) printf(" answer(%d/%d): <%s>   default: %d  >> %d\n",(int)strlen(answer),answer[0],answer,dflt,number);
      if (answer[0]!=0) number=(int)LongFromString(answer,&code);
      ok=(number>=min) && (number<=max) && (code==0);
      if (! ok) fprintf(stderr,"%cValue in error: %s",bell_char,answer);
   }
   return number;
}                                      /* AskInt */


FUNCTION void PressEnter(void)
{
   char kar;

   CHECKINIT CheckMgSystem(this_unit,"PressEnter",initialized,mytracelevel,NULL);
   printf("\rPress <ENTER> to continue ... ");
   do {
      kar=(char)getchar();
      if (strchr("qQxX",kar)!=NULL) QuitSeen=true;
   } while (kar!='\n');
}                                      /* PressEnter */


FUNCTION void GetUserName(char *user_name)
{
   CHECKINIT CheckMgSystem(this_unit,"GetUserName",initialized,mytracelevel,NULL);
   InitString(user_name,"??? not yet implemented ???",-1);
#if defined(__MSDOS__)
   GetSymbol(user_name,"UNAAM");
   if (user_name[0]==0) GetSymbol(user_name,"USERNAME");
   if (user_name[0]==0) GetSymbol(user_name,"NAME");
#else
   cuserid(user_name);
#endif
}                                    /* GetUserName */


FUNCTION void FormatSlashes(char *text,char slash)
{
   int i,len;

   CHECKINIT CheckMgSystem(this_unit,"FormatSlashes",initialized,mytracelevel,text);
   len=(int)strlen(text);
   for (i=0; i<len; i++) {
      if (strchr("/\\",text[i])) text[i]=slash;
   }
   i=(int)strlen(text)-2;
   while (i>=0) {
      if ((text[i]==slash) && (text[i+1]==slash)) DelChars(text,i,1);
      i--;
   }
   if ((text[0]=='.') && (text[1]==slash) && (text[3]!=0)) DelChars(text,0,2);
   TRACE CheckMgSystem(this_unit,"FormatSlashes",initialized,mytracelevel,text);
}                                    /* FormatSlashes */


FUNCTION void GetDirOption(char *root,char *keyword_par,char *option)
{
   maxstring key,line,keyword;
   FILE *inp;

   TRACE CheckMgSystem(this_unit,"GetDirOption",true,mytracelevel,NULL);
   InitString(line,"",-1);
   InitString(key,"",-1);
   InitString(keyword,keyword_par,-1);
   UpString(keyword);
   option[0]=0;
   sprintf(line,"%s\\000dir.opt",root);
   FormatSlashes(line,slash_char);
   inp=fopen(line,"r");
   if (inp==NULL) return;
   while (ReadLn(inp,line,-1)!=EOF) {
      SplitString(key,-1,line,-1," ");
      UpString(key);
      if (Equal(key,keyword)) StringCopy(option,-1,line,-1);
   }
   fclose(inp);
   return;
}                                    /* GetDirOption */


FUNCTION void LongNameToDos(char *longname_par,char *dosname)
{
   maxstring longname,part,line;
   string12 kort;
   FILE *inp;
   int l;

   TRACE CheckMgSystem(this_unit,"LongNameToDos",true,mytracelevel,"LN=<%s>",longname_par);
   InitString(longname,longname_par,-1);
   InitString(part,"",-1);
   InitString(line,"",-1);
   InitString(kort,"",12);
   dosname[0]=0;
   FormatSlashes(longname,'\\');
   LowString(longname);
   while (longname[0]!=0) {
      SplitString(part,-1,longname,-1,"\\");
      sprintf(line,"%s%s",dosname,longnamefile);
      inp=fopen(line,"r");
      if (inp!=NULL) {
         while (ReadLn(inp,line,-1)!=EOF) {
            SplitString(kort,12,line,-1," ");
            LowString(line);
            if (Equal(part,line)) {
               StringCopy(part,-1,kort,12);
               break;
            }
         }
         fclose(inp);
      }
      StringCat(dosname,-1,part,-1);
      StringCat(dosname,-1,"\\",0);
   }
   l=strlen(dosname);
   dosname[l-1]=0;
   TRACE CheckMgSystem(this_unit,"LongNameToDos",true,mytracelevel,"DN=<%s>",dosname);
}                                    /* LongNameToDos */


FUNCTION void DosNameToLong(char *dosname_par,char *longname)
{
   string12 kort;
   maxstring dosname,part,line,dospath;
   FILE *inp;
   int l;

   TRACE CheckMgSystem(this_unit,"DosNameToLong",true,mytracelevel,"DN=<%s>",dosname_par);
   InitString(dosname,dosname_par,-1);
   InitString(dospath,"",-1);
   InitString(part,"",-1);
   InitString(line,"",-1);
   InitString(kort,"",12);
   longname[0]=0;
   FormatSlashes(dosname,'\\');
   LowString(dosname);
   while (dosname[0]!=0) {
      SplitString(part,-1,dosname,-1,"\\");
      sprintf(line,"%s%s",dospath,longnamefile);
      StringCat(dospath,-1,part,-1);
      StringCat(dospath,-1,"\\",0);
      inp=fopen(line,"r");
      if (inp!=NULL) {
         while (ReadLn(inp,line,-1)!=EOF) {
            SplitString(kort,12,line,-1," ");
            if (Equal(part,kort)) {
               StringCopy(part,-1,line,-1);
               break;
            }
         }
         fclose(inp);
      }
      StringCat(longname,-1,part,-1);
      StringCat(longname,-1,"\\",0);
   }
   l=strlen(longname);
   longname[l-1]=0;
   TRACE CheckMgSystem(this_unit,"DosNameToLong",true,mytracelevel,"LN=<%s>",longname);
}                                    /* DosNameToLong */


FUNCTION void DosFileNameToLong(char *dospath,char *dosname_par,char *longname)
{
   string12 kort;
   maxstring dosname,line;
   FILE *inp;

   TRACE CheckMgSystem(this_unit,"DosFileNameToLong",true,mytracelevel,"DP=<%s>",dospath);
   TRACE CheckMgSystem(this_unit,"DosFileNameToLong",true,mytracelevel,"DN=<%s>",dosname_par);
   InitString(dosname,dosname_par,-1);
   InitString(line,"",-1);
   InitString(kort,"",12);
   LowString(dosname);
   StringCopy(longname,-1,dosname,0);
   sprintf(line,"%s/%s",dospath,longnamefile);
   FormatSlashes(line,slash_char);
   inp=fopen(line,"r");
   if (inp!=NULL) {
      while (ReadLn(inp,line,-1)!=EOF) {
         TRACE CheckMgSystem(this_unit,"DosFileNameToLong",true,mytracelevel,"?? %s",line);
         SplitString(kort,12,line,-1," ");
         if (Equal(dosname,kort)) StringCopy(longname,-1,line,-1);
      }
      fclose(inp);
   }
   TRACE CheckMgSystem(this_unit,"DosFileNameToLong",true,mytracelevel,"LN=<%s>",longname);
}                                    /* DosFileNameToLong */


FUNCTION void MakeLongNameFiles(char *data_root,char *par,boolean show_cmd)
{
   maxstring option,line;

   TRACE CheckMgSystem(this_unit,"MakeLongNameFiles",true,mytracelevel,NULL);
   InitString(line,"",-1);
   InitString(option,"",-1);
#if defined(__MSDOS__)
   GetDirOption(data_root,"longnames",line);
   UpString(line);
   while (line[0]!=0) {
      SplitString(option,-1,line,-1," ");
      if (Equal(option,"NO")) return;
   }
   sprintf(line,"Start.EXE /w LongName %s %s",data_root,par);
   if (testmode) StringCat(line,-1," -testmode",0);
   if (show_cmd) printf("%s\n",line);
   system(line);
#endif
}                                    /* MakeLongNameFiles */


FUNCTION int PathExist(char *name)
{
   FILE *tmp;
   boolean found;
   maxstring dosname;

   CHECKINIT CheckMgSystem(this_unit,"PathExist",initialized,mytracelevel,name);
   InitString(dosname,name,-1);
   tmp = fopen(dosname,"r");
   found = (tmp != NULL);
   if (found) {
      fclose(tmp);
#if defined(__MSDOS__)
   } else {
      LongNameToDos(dosname,dosname);
      tmp = fopen(dosname,"r");
      found = (tmp != NULL);
      if (found) fclose(tmp);
#endif
   }
   return found;
}                                      /* PathExist */


FUNCTION void DereferencePath(char *path)
{
   int from,to,stop;

   CHECKINIT CheckMgSystem(this_unit,"DereferencePath",initialized,mytracelevel,path);
   FormatSlashes(path,slash_char);
   to= -1;
   stop=(int)strlen(path);
   from=0;
   while (from<stop) {
      if ((path[from]=='.') && (path[from+1]=='.')) {
         if (path[to]==slash_char) to--;
         while ((to>=0) && (path[to]!=':') && (path[to]!=slash_char)) to--;
         while ((from<=stop) && (path[from]!=slash_char)) from++;
         if ((from<=stop) && (path[from]==slash_char)) from++;
      } else {
         to++;
         path[to]=path[from];
         from++;
      }
   }
   path[to+1]='\0';
   TRACE CheckMgSystem(this_unit,"DereferencePath",initialized,mytracelevel,path);
}                                    /* DereferencePath */


FUNCTION void FSearch(char *my_path,char *fname,char *path_found)
{
   char *envpath;
   maxstring root,path,delimiter;
   int found,i;

   CHECKINIT CheckMgSystem(this_unit,"FSearch",initialized,mytracelevel,NULL);
   InitString(path,"",-1);
#if defined(__MSDOS__)
   InitString(delimiter,";",-1);
#else
#if defined(__VMS__)
   InitString(delimiter,"#",-1);
#else
   InitString(delimiter,":",-1);
#endif
#endif
   envpath=getenv("PATH");
   InitString(root,"",-1);
   if (my_path!=NULL) InitString(root,my_path,-1);
   i=strlen(root);
   if ((i>1) && (root[i-1]==slash_char)) root[i-1]=0;
   sprintf(path,"%s%s.%s",root,delimiter,delimiter);
   if (envpath && ((*envpath)!=0)) StringCat(path,-1,envpath,0);
   TRACE CheckMgSystem(this_unit,"FSearch",initialized,mytracelevel,path);
   found=0;
   while ((path[0]!=0) && ! found) {
      SplitString(root,-1,path,-1,delimiter);
      StringCopy(path_found,-1,fname,0);
      if (root[0]!=0) sprintf(path_found,"%s%s%s",root,slash_str,fname);
      FormatSlashes(path_found,slash_char);
      found=PathExist(path_found);
   }
   if (! found) path_found[0]=0;
}                                    /* FSearch */


FUNCTION void MakeVaxPath(char *path)
{
   int i,slashseen,head;
   char *pos;

   CHECKINIT CheckMgSystem(this_unit,"MakeVaxPath.ini",initialized,mytracelevel,path);
   head=0;
   slashseen=0;
   pos=strchr(path,':');
   if (pos!=NULL) head=(int)(pos-path)+1;
   if (path[0]==0) return;
   pos=strrchr(path,slash_char);
   if (pos!=NULL) {
      i=(int)(pos-path);
      path[i]=']';
      slashseen=1;
   }
   TRACE CheckMgSystem(this_unit,"MakeVaxPath.rgt",true,mytracelevel,path);
   for (i=(int)strlen(path)-2; i>head; i--) {
      if ((path[i]=='.') && (path[i-1]=='.')) {
         DelChars(path,i-1,1);
         path[i-1]='-';
      } else if (path[i]==slash_char) {
         path[i]='.';
         slashseen++;
      }
   }
   TRACE CheckMgSystem(this_unit,"MakeVaxPath.min",true,mytracelevel,path);
   if (slashseen>0) {
      if ((path[head]==slash_char) && (slashseen>0)) {
         path[head]='[';
      } else {
         if (head>0) InsertString("[000000",path,head,-1); else InsertString("[.",path,head,-1);
      }
   }
   TRACE CheckMgSystem(this_unit,"MakeVaxPath.end",true,mytracelevel,path);
}                                    /* MakeVaxPath */


FUNCTION void VaxPathToUnix(char *path)
{
   int i,head=0,last;
   char *pos;

   CHECKINIT CheckMgSystem(this_unit,"VaxPathToUnix",initialized,mytracelevel,path);
   pos=strchr(path,':');
   if (pos!=NULL) head=(int)(pos-path)+1;
   if (path[head]==slash_char) return;
   pos=strchr(path,']');
   if (pos==NULL) return;
   if (path[0]==0) return;
   if (path[head]=='[') {
      if (path[head+1]=='.') DelChars(path,head,2);    else
      if (path[head+1]=='-') DelChars(path,head,1);
   }
   last=(int)strlen(path)-1;
   pos=strchr(path,']');
   if (pos!=NULL) last=(int)(pos-path);
   for (i=last; i>=head; i--) {
      if (path[i]=='-') {
         path[i]='.';
         InsertString(".",path,i,-1);
      } else if (strchr("[].",path[i])!=NULL) {
         path[i]='/';
      }
   }
   TRACE CheckMgSystem(this_unit,"VaxPathToUnix",true,mytracelevel,path);
}                                    /* VaxPathToUnix */


FUNCTION FILE *OpenFile(char *name,char *how,boolean abortonerror)
{
   FILE *out;
   maxstring fname;

   CHECKINIT CheckMgSystem(this_unit,"OpenFile",true,mytracelevel,"<%s> %s aoe=%d",name,how,abortonerror);
   InitString(fname,name,-1);
#if defined(__VMS__)
   MakeVaxPath(fname);
#endif
   out=fopen(fname,how);
#if defined(__MSDOS__)
   if (out==NULL) {
      LongNameToDos(fname,fname);
      out=fopen(fname,how);
   }
#endif
   if (abortonerror && (out==NULL)) {
      AbortBecause("Can't open file %s for %s",fname,how);
   }
   return out;
}                                    /* OpenFile */


FUNCTION void VaxErrorMessage(char *txt,int error)
{
#if defined(__VMS__)
   maxstring cmd;

   InitString(cmd,"",-1);
   sprintf(cmd,"write sys$output \"%s \",f$message(%d)",txt,error);
   system(cmd);
#else
   fprintf(stderr,"--->>> Error %4X (%d) in %s\n",error,error,txt);
#endif
}                                    /* VaxErrorMessage */


FUNCTION long GetFileInfo(char *fname_par,char *modtime)
{
   FILE *inp;
   struct stat status;
   maxstring fname;

   CHECKINIT CheckMgSystem(this_unit,"GetFileInfo",initialized,mytracelevel,fname);
   InitString(fname,fname_par,-1);
   sprintf(modtime,"-ABSENT-");
   inp=fopen(fname,"r");
#if defined(__MSDOS__)
   if (inp==NULL) {
      LongNameToDos(fname,fname);
      inp=fopen(fname,"r");
   }
#endif
   stat(fname,&status);
   if (status.st_mode & S_IFDIR) {
      sprintf(modtime,"<DIR>");
      return GFI_DIR;
   }
   if (inp==NULL) return GFI_ABSENT;
   fclose(inp);
   FormatDateTime(modtime,status.st_mtime);
   return status.st_size;
}                                    /* GetFileInfo */


FUNCTION char *CdUpString(int level)
{
   int i;
   static char cdup[100];

   if (level<0) level=0;
   if (level>33) level=33;
   for (i=0; i<level*3; i++) {
      if ((i % 3)==2) cdup[i]='/'; else cdup[i]='.';
   }
   cdup[level*3]=0;
   CHECKINIT CheckMgSystem(this_unit,"CdUpString",initialized,mytracelevel,"%d [%s]",level,cdup);
   return (char *)(&cdup);
}                                    /* CdUpString */


FUNCTION void ConvertItem(char *line,char *old_txt,char *new_txt)
{
   char *p;

   CHECKINIT CheckMgSystem(this_unit,"ConvertItem",initialized,mytracelevel,"<%s> <%s>:=<%s>",line,old_txt,new_txt);
   p=line;
   while (p!=NULL) {
      p=strstr(line,old_txt);
      if (p!=NULL) {
         DelChars(p,0,strlen(old_txt));
         InsertString(new_txt,line,(int)(p-line),-1);
      }
   }
}                                    /* ConvertItem */


