#ifndef lint
static char *sccsid = "@(#)%M%  %I%  Teemu Torma %H%";
#endif

/* Send file(s) unsing Xmodem/TeLink/MODEM7 Batch protocols.
   
   @(#)Copyright (c) 1987 by Teemu Torma
   
   Permission is given to distribute this program and alter this code as
   needed to adapt it to forign systems provided that this header is
   included and that the original author's name is preserved. */

/* LINTLIBRARY */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <time.h>
#include "hsu.h"
#include "fnet.h"
#include "fio.h"
#include "crc.h"

extern time_t time();
extern int readline();
extern void sendline();

#define BytesSend       (block * BlockSize)

/* General states for finite state machines. These are used to break
   loops mostly. */

#define Error           (-1)    /* error state */
#define Done            (-2)    /* state to break loops */

/* XMODEM/TeLink send states. */

#define WaitTeLnk       (0)
#define WaitStart       (1)
#define SendBlock       (2)
#define WaitACK         (3)
#define WaitEnd         (4)

/* BATCH File sender states. */

#define MoreFiles       (0)
#define CheckFNm        (1)
#define CheckFile       (2)
#define EndSend         (3)

/* States for MODEM7 filename sender. */

#define WaitNak         (0)
#define WaitChAck       (1)
#define WaitCksm        (2)

/* Return filename for MS-DOS. Because MS-DOS has many restrictions in
   filenames (that silly dot in between), we must use something equivalent
   while sending TeLink. */

char *
conver_to_msdos_name(filename)
     char *filename;
{
  static char msdos_name[16];
  register int pos;
  register char *cp;
  
  /* strip off pathname */
  cp = basename(filename);
  
  /* ignore leading dot */
  if (*cp == '.')
    cp++;
  
  /* create first 8 characters of ms-dos name */
  for (pos = 0; *cp && *cp != '.' && pos < 8; pos++, cp++)
    msdos_name[pos] = *cp;
  
  /* add dot for ms-dos */
  msdos_name[pos] = '.';
  
  /* add 3 character type */
  for (pos = 9; *cp && *cp != '.' && pos < 12; pos++, cp++)
    msdos_name[pos] = *cp;
  
  /* null terminate ms-dos name */
  msdos_name[pos] = 0;
  
  debug(2, "File %s changed to %s in MS-DOS", filename, msdos_name);
  return msdos_name;
}

/* Create special TeLink block. This block will be send as block 0 before
   all other data will be sent. Return false if it couldn't be created,
   otherwise true. */

bool
maketelnk(filename, buffer)
     char *filename, *buffer;
{
  struct stat stbuf;
  register int cnt;
  register char *cp;
  
  /* get status of file */
  if (stat(filename, &stbuf) == -1)
    {
      log("$Cannot stat %s", filename);
      return False;
    }
  
  /* save file length */
  for (cnt = 0; cnt < 4; cnt++)
    buffer[cnt] = (stbuf.st_size >> (cnt * 8)) & 0377;
  
  /* creation time and date will be all zeroes */
  for (cnt = 4; cnt < 8; cnt++)
    buffer[cnt] = 0;
  
  /* save file name */
  for (cnt = 8, cp = /*msdosname(*/ filename /*)*/ ; *cp; cp++, cnt++)
    buffer[cnt] = *cp;
  
  /* if name was shorter that 16 chars, fill rest of it with blanks */
  while (cnt < 24)
    buffer[cnt++] = ' ';
  
  /* don't know why here's zero */
  buffer[cnt] = 0;
  
  /* name of sending program... let's ignore */
  for (cnt = 25; cnt < 41; cnt++)
    buffer[cnt] = 0;
  (void) strcpy(&buffer[25], PROGRAMNAME);

  /* crc mode */

  buffer[41] = 1;
  
  /* rest of buffer will be full of zeroes */
  for (cnt = 42; cnt < 128; cnt++)
    buffer[cnt] = 0;
  return True;
}

/* Send block to line. Note that intial SOH or SYN is not sent by
   this routine. */

void
xtsendblk(sxbuf, blocknum, crcmode)
     char *sxbuf;
     int blocknum;
     bool crcmode;
{
  register unsigned short crc;
  register int checksum, cnt;
  
  debug(1, "Send block %d", blocknum);
  
  /* send block number */
  sendline(blocknum & 0377);
  sendline(~(blocknum & 0377));
  
  /* send the block itself */
  debug(1, "Send %d bytes", BlockSize);
  (void) write(line, sxbuf, BlockSize);
  
  /* count crc and checksum */
  for (crc = 0, checksum = 0, cnt = 0; cnt < BlockSize; cnt++)
    {
      crc = updcrc(sxbuf[cnt], crc);
      checksum += sxbuf[cnt];
    }
  
  if (crcmode)
    {
      /* send crc */
      /* crc = updcrc(0, updcrc(0, crc)); /* What is this for? */
      debug(1, "Send crc %d", crc);
      sendline((int) (crc >> 8));
      sendline((int) crc);
    }
  else
    {
      /* send checksum */
      debug(1, "send checksum %d", checksum & 0377);
      sendline(checksum & 0377);
    }
}

/* Send file using XMODEM/TeLink protocol. Return True is everything went
   fine, otherwise false. */

bool
xtsend(filename, telink)
     char *filename;
     bool telink;
{
  int state = telink ? WaitTeLnk : WaitStart;
  struct stat st;
  time_t stime;
  int tries, c;
  char sxbuf[BlockSize];
  FILE *fp;
  bool resend = False, lastblock = False;
  int block = 1, cnt;
  bool crcmode = True;
  
  if ((fp = fopen(filename, "r")) == NULL)
    log("$Can not open %s, sending empty file", filename);
  else
    log("Sending %s", filename);
  
  (void) stat(filename, &st);
  
  while (state >= WaitTeLnk && state <= WaitEnd)
    switch (state)
      {
      case WaitTeLnk:
        if (!maketelnk(filename, sxbuf))
          {
            log("Unable to send file %s", filename);
            state = Error;
          }
        else
          {
            SetStart();
            for (tries = 0; state == WaitTeLnk; tries++)
              {
                if ((c = readline(40)) == NAK || c == 'C')
                  {
                    crcmode = c == 'C';
                    debug(2, "Got %02x, sending TeLink block", c);
                    sendline(SYN);
		    /* telink block always has checksum */
                    xtsendblk(sxbuf, 0, False);
                  }
                else if (c == ACK)
                  {
                    state = WaitStart;
                    debug(2, "Got ACK, TeLink block sent ok");
                  }
		else if (c == NAK)
		  {
		    if (tries > 2)
		      {
			log("Too many tries on telink block");
			state = WaitStart;
		      }
		    /* If other stuff, not ack/nak received, just ignore */
		  }
                else if (Timeout(40))
                  {
                    log("Timeout on telink block");
                    state = WaitStart;
                  }
              } 
          }
        break;
      case WaitStart:
        SetStart();
        for (tries = 0; state == WaitStart; tries++)
          if ((c = readline(60)) == NAK || c == 'C')
            {
	      debug(1, "Got NAK, sendblock start");
              crcmode = True; /* c == 'C'; */
              state = SendBlock;
            }
	  else if (Timeout(60))
            {
              log("Timeout on xmodem start");
              state = Error;
            }
	  else if (tries > 20)
	    {
	      debug(1, "Too many retries on xmodem start");
	      state = Error;
	    }
        break;
      case SendBlock:
        if (!resend)
          if (lastblock || !fp)
            {
              sendline(EOT);
              state = WaitEnd;
            }
          else {
            for (cnt = 0; cnt < BlockSize; cnt++)
              {
                if ((c = getc(fp)) == EOF) {
                  c = CTRLZ;
                  lastblock = True;
                }
                sxbuf[cnt] = c;
              }
            sendline(SOH);
            xtsendblk(sxbuf, block, crcmode);
            state = WaitACK;
          }
        else {
          sendline(SOH);
          xtsendblk(sxbuf, block, crcmode);
          state = WaitACK;
          resend = False;
        }
        break;
      case WaitACK:
        SetStart();
        for (tries = 0; state == WaitACK; tries++)
          if ((c = readline(60)) == NAK)
            {
              resend = True;
              state = SendBlock;
	      debug(1, "Got NAK, resend current block");
	      if (tries >= 10)
		{
		  log("Too many tries on xmodem send");
		  state = Error;
		}
            }
          else if (c == ACK)
            {
              debug(1, "Got ACK, send next block");
              block++;
              debug(2, "%ld bytes send (%d %%)", BytesSend,
                    (BytesSend / st.st_size) * 100);
              state = SendBlock;
            }
          else if (Timeout(60))
            {
              log("Xodem send timeout");
              state = Error;
            }
        break;
      case WaitEnd:
        for (tries = 0; state == WaitEnd; tries++)
          if ((c = readline(60)) == NAK)
            {
              sendline(EOT);
              debug(2, "Send EOT");
            }
          else if (c == ACK)
            {
              log("Xmodem/TeLink send successful");
              state = Done;
            }
          else if (Timeout(60))
            {
              log("Timeout on xmodem/telink end");
              state = Error;
            }
          else if (tries >= 10)
            {
              log("Too many retries on xmodem end");
              state = Error;
            }
        break;
      }
  
  return state == Error ? False : True;
}

/* Send MODEM7 filename. */

bool
sendmdmfn(filename)
     char *filename;
{
  int state = WaitNak;
  time_t stime;
  int tries = 0;
  int c, checksum, cnt;
  
  SetStart();
  while (state >= WaitNak && state <= WaitCksm)
    switch (state)
      {
      case WaitNak:
        for (tries = 0; state == WaitNak; )
          if ((c = readline(60)) == NAK)
            {
              debug(2, "Got NAK for filename %s", filename);
              sendline(ACK);
              sendline(*filename);
              cnt = 1;
              state = WaitChAck;
            }
          else if (Timeout(60))
            {
              debug(1, "Timeout on filename");
              state = Error;
            }
          else if (tries >= 20)
            {
              debug(1, "Too many retries on filename");
              state = Error;
            }
        break;
      case WaitChAck:
        if (readline(2) == ACK)
          {
            if (filename[cnt])
              sendline(filename[cnt++]);
            else
              {
                sendline(SUB);
                state = WaitCksm;
              }
          }
        else
          {
            sendline('u');
            tries++;
            state = WaitNak;
          }
        break;
      case WaitCksm:
        if ((c = readline(2)) == TIMEOUT)
          {
            sendline('u');
            tries++;
            state = WaitNak;
          }
        else
          {
            for (cnt = 0, checksum = SUB; filename[cnt]; cnt++)
              checksum += filename[cnt];
            if (c != (checksum & 0377))
              {
                debug(1, "Checksum error in filename");
                sendline('u');
                state = WaitNak;
                tries++;
              }
            else
              {
                sendline(ACK);
                debug(2, "Filename sent ok");
                state = Done;
              }
          }
        break;
      }
  
  return state != Error;
}

/* Batch file sender. If filename is NULL, no more files to send. */

bool
batchsend(filename)
     char *filename;
{
  int state = MoreFiles;
  int c;
  time_t stime;
  bool ok;
  
  while (state >= MoreFiles && state <= EndSend)
    switch (state)
      {
      case MoreFiles:
        if (filename)
          {
            debug(2, "Sending filename for %s", filename);
            ok = sendmdmfn(filename);
            state = CheckFNm;
          }
        else
          state = EndSend;
        break;
      case CheckFNm:
        if (ok)
          {
            debug(1, "Sending file %s", filename);
            ok = xtsend(filename, True);
            state = CheckFile;
          }
        else
          state = Error;
        break;
      case CheckFile:
        if (ok)
          {
            debug(1, "File send ok");
            state = Done;
          }
        else
          {
            log("TeLink file send failed");
            state = Error;
          }
        break;
      case EndSend:
        SetStart();
        while (!Timeout(10))
          if ((c = readline(10)) == NAK || c == 'C')
            {
              sendline(EOT);
              log("Batch file send ok");
              state = Done;
              break;
            }
        if (state != Done)
          {
            log("Batch file send failed, no NAK");
            sendline(EOT);
            state = Error;
          }
        break;
      }
  
  return state != Error;
}
