#ifndef lint
static char *sccsid = "@(#)%M%  %I%  Teemu Torma %H%";
#endif lint

/* Routines to get and translate information in nodelist.
   
   @(#)Copyright (c) 1987 by Teemu Torma
   
   Permission is given to distribute this program and alter this code as
   needed to adapt it to forign systems provided that this header is
   included and that the original author's name is preserved. */

/* LINTLIBRARY */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "config.h"
#include "fnet.h"
#include "nodelist.h"

extern long atol();

/* Make nodelist's index-file. That index-file will be used to search
   desired net/region from nodelist.
   
   Format of index-file is following:
   First line contains same information than nodelist's first line.
   All following lines contains two number's separated by tab:
   first of them is region/net number, second offset in nodelist.
   
   Return NULL if everything was file, otherwise error-message string. */

char *
update_index()
{
  struct stat nlstat, inlstat;
  FILE *nl = NULL, *nlidx = NULL;
  char nodelist[BUFLEN], inodelist[BUFLEN];
  char inlid[BUFLEN], nlid[BUFLEN];
  char buffer[BUFSIZ];
  long offset;
  char *cp, *error;
  
  /* generate nodelist and index-file names */
  (void) sprintf(nodelist, "%s/%s", LIBDIR, NODELIST);
  (void) sprintf(inodelist, "%s/%s", LIBDIR, INODELIST);
  
  /* get statuses of nodelist and index */
  if (stat(nodelist, &nlstat) == -1)
    return "$Error in getting nodelist status";
  errno = 0;
  if (stat(inodelist, &inlstat) == -1 && errno != ENOENT)
    return "$Error in getting status of existing nodelist-index";
  
  /* If index-file does exists then check modification times and
     first lines. If nodelist is older and first lines are the same,
     no update is needed. */
  
  if (errno == 0 && nlstat.st_mtime <= inlstat.st_mtime)
    {
      if ((nl = fopen(nodelist, "r")) == NULL)
        return "$Unable to open nodelist";
      if (nlidx = fopen(inodelist, "r"))
        {
          if (fgets(nlid, BUFLEN, nl) == NULL)
            {
              error = "Missing id string in nodelist";
              goto done;
            }
          if (fgets(inlid, BUFLEN, nlidx) && !strcmp(nlid, inlid))
            {
              error = NULL;
              goto done;
            }
          log("Update nodelist-index: different id string");
        }
      log("$Update nodelist-index: no index-file");
    }
  else
    log("Update nodelist-index: nodelist is newer than index");
  
  /* open index-file for writing */
  if (nlidx != NULL)
    (void) fclose(nlidx);
  if ((nlidx = fopen(inodelist, "w")) == NULL)
    {
      error = "$Unable to open nodelist-index for writing";
      goto done;
    }
  
  /* save id string of nodelist */
  (void) rewind(nl);
  if (fgets(buffer, BUFSIZ, nl) == NULL)
    {
      error = "Missing id string in nodelist";
      goto done;
    }
  (void) fputs(buffer, nlidx);
  
  /* save host/region offsets */
  for (offset = ftell(nl); fgets(buffer, BUFSIZ, nl); offset = ftell(nl))
    if (!strncmp(buffer, "Host,", 5) || !strncmp(buffer, "Region,", 7))
      {
        cp = strchr(buffer, ',') + 1;
        (void) fprintf(nlidx, "%d\t%ld\n", atoi(cp), offset);
      }
  error = NULL;
  
  /* done, close files and return error message */
 done:
  if (nl)
    (void) fclose(nl);
  if (nlidx)
    (void) fclose(nlidx);
  return error;
}

/* Convert underscores in string to spaces. In nodelist there is always
   underscore insted of space (flags is special case). */

void
convert_space(s)
     register char *s;
{
  while (*s)
    {
      if (*s == '_')
        *s = ' ';
      s++;
    }
}

/* Get one comma-terminated field from buffer, retrun pointer to right
   after terminating comma. Convert underscores to spaces in string. */

char *
parse_field(buffer, entry, size)
     char *buffer, *entry;
     int size;
{
  char *np = entry;
  
  /* copy string */
  while (--size >= 0 && *buffer && *buffer != ',')
    *entry++ = *buffer++;
  *entry = 0;
  
  switch (*buffer)
    {
    case 0:
      /* end of buffer */
      log("No closing comma in field");
      return (char *) 0;
    case ',':
      /* succesful copy */
      convert_space(np);
      return buffer + 1;
    default:
      /* buffer too long, find comma */
      while (*buffer && *buffer != ',')
        buffer++;
      if (*buffer)
        {
          convert_space(np);
          return buffer + 1;
        }
      else
        {
          log("Missing comma in field");
          return (char *) 0;
        }
    }
  /* NOTREACHED */
}

/* Parse one line of nodelist to node structure. Return NULL is there
   was error's in that line or missing fields. */

Node *
parse_entry(entry, buffer, net)
     Node *entry;
     char *buffer;
     int net;
{
  char *cp;
  int n;
  
  /* strip newline if exists */
  if (cp = strchr(buffer, '\n'))
    *cp = 0;
  
  /* get type of entry */
  if (!strncmp("Region,", buffer, 7))
    entry->type = REGION;
  else if (!strncmp("Host,", buffer, 5))
    entry->type = HOST;
  else if (!strncmp("Hub,", buffer, 4))
    entry->type = HUB;
  else if (!strncmp("Pvt,", buffer, 4))
    entry->type = PVT;
  else if (!strncmp("Hold,", buffer, 5))
    entry->type = HOLD;
  else if (!strncmp("Down,", buffer, 5))
    entry->type = DOWN;
  else if (!strncmp("Kenl,", buffer, 5))
    entry->type = KENL;
  else if (*buffer == ',')
    entry->type = NORMAL;
  else
    {
      log("Unknown type in line '%s'", buffer);
      return (Node *) 0;
    }
  
  /* get net/region/node number */
  if ((cp = strchr(buffer, ',')) == NULL)
    {
      log("Missing net/node/region in line '%s'", buffer);
      return (Node *) 0;
    }
  if ((n = atoi(++cp)) == 0)
    {
      log("Value of net/node/region is 0 in line '%s'", buffer);
      return (Node *) 0;
    }
  if (entry->type == REGION || entry->type == HOST)
    {
      entry->net = n;
      entry->node = 0;
    }
  else
    {
      entry->net = net;
      entry->node = n;
    }
  while (*cp && *cp++ != ',');
  
  /* get node/net/region name */
  if ((cp = parse_field(cp, entry->name, sizeof(entry->name))) == NULL)
    {
      log("Invalid name in line '%s'", buffer);
      return (Node *) 0;
    }
  
  /* get city */
  if ((cp = parse_field(cp, entry->city, sizeof(entry->city))) == NULL)
    {
      log("Invalid city in line '%s'", buffer);
      return (Node *) 0;
    }
  
  /* get sysop */
  if ((cp = parse_field(cp, entry->sysop, sizeof(entry->sysop))) == NULL)
    {
      log("Invalid sysop in line '%s'", buffer);
      return (Node *) 0;
    }
  
  /* get phone number */
  if ((cp = parse_field(cp, entry->phone, sizeof(entry->phone))) == NULL)
    {
      log("Invalid phone-number in line '%s'", buffer);
      return (Node *) 0;
    }
  
  /* get maximum baud rate */
  if ((n = atoi(cp)) == 0)
    {
      log("Baud rate is 0 in line '%s'", buffer);
      return (Node *) 0;
    }
  entry->speed = n;
  while (*cp && *cp++ != ',');
  
  /* get flags */
  (void) strncpy(entry->flags, cp, sizeof(entry->flags));
  entry->flags[sizeof(entry->flags) - 1] = 0;
  
  /* all done */
  return entry;
}

/* Get entry for one node from nodelist. Return NULL, if there is no
   entry for that node. */

Node *
node_entry(net, node)
     int net, node;
{
  static Node entry;
  FILE *fp;
  char buffer[BUFSIZ], *cp;
  long offset;
  
  /* open index-file */
  (void) sprintf(buffer, "%s/%s", LIBDIR, INODELIST);
  if ((fp = fopen(buffer, "r")) == NULL)
    {
      log("$Unable to open %s", buffer);
      return (Node *) 0;
    }
  
  /* skip id string */
  (void) fgets(buffer, BUFSIZ, fp);
  
  /* try to find offset of node */
  while (fgets(buffer, BUFSIZ, fp))
    /* found offset */
    if (atoi(buffer) == net)
      {
        (void) fclose(fp);
        
        for (cp = buffer; *cp && *cp != '\t'; cp++)
          /* find offset in buffer */;
        
        /* if offset is zero, there must be something wrong with index */
        if ((offset = atol(cp + 1)) == 0l)
          {
            log("Got offset 0 for net %d", net);
            return (Node *) 0;
          }
        
        /* open nodelist */
        (void) sprintf(buffer, "%s/%s", LIBDIR, NODELIST);
        if ((fp = fopen(buffer, "r")) == NULL)
          {
            log("$Unable to open %s", buffer);
            return (Node *) 0;
          }
        
        /* seek to net/region in nodelist */
        if (fseek(fp, offset, 0))
          {
            log("$Seek error on nodelist");
            (void) fclose(fp);
            return (Node *) 0;
          }
        
        /* try to find node in this net/region */
        while (fgets(buffer, BUFSIZ, fp))
          {
            
            /* is this net/region host entry? */
            if (!strncmp("Host,", buffer, 5) ||
                !strncmp("Region,", buffer, 7))
              {
                /* yes, check if this is our net */
                if (atoi(strchr(buffer, ',') + 1) == net)
                  {
                    /* yeps, check if mail node is host */
                    if (node == 0)
                      {
                        /* this is host for our net/region */
                        (void) fclose(fp);
                        return parse_entry(&entry, buffer, net);
                      } 
                  }
                else
                  {
                    /* this is next net... */
                    (void) fclose(fp);
                    return (Node *) 0;
                  }
              }
            else
              /* is this node we wanted? */
              if ((cp = strchr(buffer, ',')) && atoi(cp + 1) == node)
                {
                  /* yap, return it */
                  (void) fclose(fp);
                  return parse_entry(&entry, buffer, net);
                }
          }
        
        /* we didn't find node */
        (void) fclose(fp);
        return (Node *) 0;
      }
  
  /* we didn't find net/region */
  (void) fclose(fp);
  return (Node *) 0;
}
