/****************************************************************
 *	Copyright 1986, Gene H. Olson, Quest Research, Burnsville   *
 *	Minnesota.   Permission to copy and distribute this         *
 *	program, all associated code in source and binary form,     *
 *	for any purpose, so long as this notice is preserved.       *
 ****************************************************************/


/***************************************************************
 *        Procedures to read "info" files in the style         *
 *        of "terminfo" or "dialinfo".                         *
 ***************************************************************/


#include <stdio.h>
#include <ctype.h>

#include "readinfo.h"

#define loop for(;;)

extern char *strcpy() ;
extern char *getenv() ;

/******
 *	findinfo - Find entry in info file.
 *
 *	Returns: 0=found, -1=not found.
 */

static int
findinfo(file, entry)
register FILE *file ;				/* Open file to search in */
register char *entry ;				/* Entry to search for */
{
	register short ch ;
	register char *cp ;

	/*
	 *	Always start from the beginning
	 *	of the file.
	 */

	(void) fseek(file, 0L, 0) ;

	/*
	 *	Loop through lines in the file.
	 */

	loop {

		/*
		 *	If the first character of the line is
		 *	alphanumeric,  scan the line for strings
		 *	separated by "|" characters which exactly
		 *	match the entry name.
		 */

		ch = getc(file) ;
		if (isalnum(ch) || ch == '/') {
			loop {

				/*
				 *	Scan for a match terminated by '|' or ','.
				 *	If found, succeed with the file pointer
				 *	positioned to the first field entry.
				 */

				cp = entry ;
				while (ch == *cp) {
					ch = getc(file) ;
					cp++ ;
					}

    			if (*cp == 0 && (ch == '|' || ch == ',')) {
					loop {
						if (ch == ',') return(0) ;
						if (ch == '\n') break ;
						if (ch == EOF) return(-1) ;
						ch = getc(file) ;
						}
					}

				/*
				 *	If no match found, position past any "|"
				 *	character found and try again.  Otherwise
				 *	the entry is not on this line.
				 */

				loop {
					if (ch == '|' || ch == ',' || ch == '\n') break ;
					ch = getc(file) ;
					if (ch == EOF) return(-1) ;
					}

				if (ch != '|') break ;

				ch = getc(file) ;
				}
			}

		/*
		 *	Skip to end-of-line.
		 */

		loop {
			if (ch == EOF) return(-1) ;
			if (ch == '\n') break ;
			ch = getc(file) ;
			}
		}
	}



/******
 *	getnext - Get next character from descriptor file, or
 *	          embedded environment variable.
 */

int
getnext(file, cpp)
register FILE *file ;		/* File to read char from */
register char **cpp ;		/* Macro string/state pointer */
{
	register int ch ;
	register int i ;
	char name[21] ;
	static char defaddr ;

	/*
	 *	Loop until we find a character to return.
	 */

	loop {

		/*
		 *	If reading a default string from the file,
		 *	read until a unescaped "}" character is seen.
		 */

		if (*cpp == &defaddr) {
			ch = getc(file) ;
			if (ch == '}') *cpp = 0 ;
			else if (ch == '\\') {
				ch = getc(file) ;
				if (ch == '}') return(ch) ;
				(void) ungetc(ch, file) ;
				return('\\') ;
				}
			else return(ch) ;
			}

		/*
		 *	If in a macro string, get the next character.
		 */

		else if (*cpp) {
			if (**cpp == 0) *cpp = 0 ;
			else {
				ch = *(*cpp)++ ;
				return(ch) ;
				}
			}

		/*
		 *	Otherwise, just get the next character from
		 *	the file.   Unless of course it is an unescaped
		 *	"${" sequence which begins an environment variable
		 *	specification.
		 */

		else {
			ch = getc(file) ;

			if (ch == '\\') {
				ch = getc(file) ;
				if (ch != '$') {
					(void) ungetc(ch, file) ;
					return('\\') ;
					}
				return('$') ;
				}

			if (ch != '$') return(ch) ;

			ch = getc(file) ;
			if (ch != '{') {
				(void) ungetc(ch, file) ;
				return('$') ;
				}

			/*
			 *	We saw an unescaped "${".  Get the environment
			 *	variable name and look it up.
			 *
			 *	If the variable is found, set *cpp to return
			 *	its value, and skip over any default string seen.
			 *
			 *	If not found, and a default string is given, set
			 *	*cpp so the default string will be read from the
			 *	file.
			 */

			i = 0 ;
			loop {
				ch = getc(file) ;
				if (ch == EOF) return(EOF) ;
				if (ch == '-' || ch == '}') break ;
				if (i < sizeof(name)-1) name[i++] = ch ;
				}
			name[i] = 0 ;

			*cpp = getenv(name) ;

			if (ch == '-') {
				if (*cpp) {
					loop {
						ch = getc(file) ;
						if (ch == EOF) return(EOF) ;
						if (ch == '}') break ;
						if (ch == '\\') {
							ch = getc(file) ;
							if (ch == EOF) return(EOF) ;
							}
						}
					}
				else *cpp = &defaddr ;
				}
			}
		}
	}



/******
 *	getinfo - Get next key=text entry in info description.
 *
 *	Returns:	1=found, 0=end of entry, -1=error.
 */

static int
getinfo(file, cpp, key, text)
FILE *file ;					/* Info file pointer */
char **cpp ;					/* Macro char pointer */
char *key ;						/* Returned key string */
char *text ;					/* Returned text string */
{
	register short ch ;
	register char *cp ;
	register short count ;

	/*
	 *	Scan for the next keyword.
	 */

	ch = getnext(file, cpp) ;
	loop {
		if (ch == EOF) return(0) ;

		if (isalpha(ch)) break ;

		if (ch == '\n') {
			ch = getnext(file, cpp) ;
			if (isalpha(ch)) return(0) ;
			}

		else if (ch == '#') {
			loop {
				ch = getnext(file, cpp) ;
				if (ch == EOF) return(-1) ;
				if (ch == '\n') break ;
				}
			}

		else if (ch == ' ' || ch == '\t' || ch == ',')
			ch = getnext(file, cpp) ;

		else return(-1) ;
		}

	/*
	 *	Pick up keyword.
	 */

	count = 0 ;
	cp = key ;

	while (isalnum(ch)) {
		if (++count > IKEYSIZE) return(-1) ;
		*cp++ = ch ;
		ch = getnext(file, cpp) ;
		}

	*cp = 0 ;

	if (ch != '=') return(-1) ;

	/*
	 *	Pick up associated string text.
	 */

	count = 0 ;
	cp = text ;

	loop {
		ch = getnext(file, cpp) ;
		if (ch == ',') break ;

		if (ch == EOF) return(-1) ;

		if (ch == '\\') {
			ch = getnext(file, cpp) ;
			if (ch == EOF) return(-1) ;
			if (ch != ',') {
				*cp++ = '\\' ;
				count++ ;
				}
			}

		if (++count > ITEXTSIZE) return(-1) ;

		*cp++ = ch ;
		}

	*cp = 0 ;

	return(1) ;
	}



/******
 *	readinfo - Read info file for data.
 */

int
readinfo(fname, entry, build, info)
char *fname ;					/* Info File name to read */
char *entry ;					/* Info entry name to return */
int (*build)() ;				/* Build entry procedure */
char *info ;					/* Info pointer */
{
	register FILE *file ;
	register int i ;
	register int usecount ;

	char *cp ;
	char key[IKEYSIZE+1] ;
	char text[ITEXTSIZE+1] ;
	char use[ITEXTSIZE+1] ;

	/*
	 *	Open info file.
	 */

	file = fopen(fname,"r") ;
	if (file == 0) {
		(void) fprintf(stderr, "Cannot open: %s\n", fname) ;
		return(-1) ;
		}
	
	/*
	 *	Loop to read all info entries in "use" list.
	 */

	usecount = 20 ;
	loop {

		/*
		 *	Find entry name in file.
		 */

		if (findinfo(file, entry) < 0) {
			(void) fprintf(stderr, "File %s, no entry found: %s\n",
				fname, entry) ;
			(void) fclose(file) ;
			return(-1) ;
			}

#ifdef DEBUG
		(void) fprintf(stderr,"Found entry: %s\n",entry) ;
#endif

		/*
		 *	Unpack and process all entries.
		 */

		cp = 0 ;
		loop {
			i = getinfo(file,&cp,key,text) ;
			if (i == 0) {
				(void) fclose(file) ;
				return(0) ;
				}

			if (i < 0) {
				(void) fprintf(stderr, "File %s, corrupted entry: %s\n",
					fname, entry) ;
				(void) fclose(file) ;
				return(-1) ;
				}

#ifdef DEBUG
			(void) fprintf(stderr,"Got %s=%s\n",key,text) ;
#endif

			/*
			 *	Follow "use" list chain to next entry.
			 */

			if (strcmp(key, "use") == 0) {
				(void) strcpy(use, text) ;
				entry = use ;
				if (--usecount > 0) break ;
				(void) fclose(file) ;
				return(-1) ;
				}

			/*
			 *	Add data to info entry.
			 */

			if ((*build)(info, key, text) < 0) {
				(void) fprintf(stderr,
					"Error in file %s, entry %s, %s=%s\n",
					fname, entry, key, text) ;
				(void) fclose(file) ;
				return(-1) ;
				}
			}
		}
	}
