#ifdef COMMENT

   new cp command

	-a ask on all files
	-b backup - only if st_ctime of source > dest
	-d ask only on directories and special files
#ifdef HOLES
	-h "write" holes with (lseek) when possible
#endif
	-i interactive - ask if would overwrite
	-l link if possible
	-m move ** not fully implemented yet
	-n only if file doesn't already exist
	-o owner - preserve uid and gid
	-p full path name to destination
	-r recursive
	-s do special files specially
	-t times - preserve atime and mtime
	-T times - -t plus retain original source access time
	-u unlink dest file before copying
	-x only if file already exists
	-N No copy or link or mkdir, just do -t and -o
	-v verbose, partial
	-V verbose, full
	-S allow args from stdin with "=" arg

    From copy.c written by Keith Davis Feb 13, 1976
    Modified by William G. Peterson  April 21, 1977
    Rewritten by Dave Yost, Rand, May-June, 1981


    Room for more work:

	The -m option is coded but not tested, and it
	should invoke mv(1) if you try to recursively move
	a directory, whereis now it will try to move everything
	one file at a time.

	Could run a little faster if we used chdir to
	move around in the source trees.

	In the mkdir code, on 4.1bsd signals should be deferred, not ignored.

#endif  COMMENT

#include <sys/types.h>
#include <sys/dir.h>
#include <signal.h>
#include <stdio.h>
#include <ctype.h>
#include <args.h>

#include <localenv.h>
#include <c_env.h>

/* if you don't have c_env.h, use this: */
#ifdef  NOC_ENV_H
typedef int void;       /* if old C compiler */

/* For optimum register usage, define 3 or 6 (or whatever) of these */
/* to register, and the rest to nothing */
/* i.e. 6 for vax, 3 for most others */
#define Reg1 register
#define Reg2 register
#define Reg3 register
#define Reg4 register
#define Reg5 register
#define Reg6 register
#define Reg7
#endif  NOCSTD

#include <sys/stat.h>
#define S_IFNOT -1  /* doesn't exist */

#define YES 1
#define NO  0
#define Block
typedef int Bool;

#define COPYBUFSIZE (20 * BUFSIZE)
#define MAXNAME 1500    /* big enough for largest filename */
#ifdef  SYMLINKS
#define MAXSYMLINK 1500 /* big enough for largest filename */
#endif  SYMLINKS

#if defined(SYSFCHOWN) || defined(SYSFCHMOD)
#define FCHOWN_MOD
#endif

struct cpfile {
    int type;           /* st_mode & S_IFMT */
    char *endstr;       /* last char '\0' in name */
    struct stat stbuf;
#ifdef  FCHOWN_MOD
    int fd;             /* if we have it open */
#endif  FCHOWN_MOD
    char name[MAXNAME]; /* buffer for pathname */
 };
struct cpfile file1;
struct cpfile file2;

/*  At each level of recursion we need another file descriptor
/*  for the open on the new directory.  If we run out of file
/*  descriptors, we close the fd for the last level and reopen
/*  it when we return to that level.  In order to reserve two
/*  file descriptors for the source and destination on file copy
/*  operations, instead of the normal sequence of opening a file
/*  and then closing it when done, we open it and leave it open
/*  until we close it when we need the file descriptor again.
/*  These are the two variables for these file descriptors.
/**/
int fdsource = -1;
int fddest = -1;

int nsources;           /* 1: 1 src, no dest; 1: 1 src; 2: 2 or more src'es */

/* The flag options: */
Bool    askflag;        /* -a  ask on all files */
Bool    backflag;       /* -b  backup - only if st_ctime of source > dest */
Bool    askdirflag;     /* -d  ask only on directories and special files */
#ifdef HOLES
Bool	holeflag;	/* -h  lseek(2) over zero-filled blocks */
#endif
Bool    interflag;      /* -i  interactive - ask if would overwrite */
Bool    linkflag;       /* -l  link if possible */
Bool    moveflag;       /* -m  move ** not fully implemented yet */
Bool    newonlyflag;    /* -n  only if file doesn't already exist */
Bool    ownerflag;      /* -o  owner - preserve uid and gid */
Bool    fullpathflag;   /* -p  full path name to destination */
Bool    recursiveflag;  /* -r  recursive */
Bool    specflag;       /* -s  do special files specially */
int     timesflag;      /* -t 1 times - preserve atime and mtime */
			/* -T 2 times - -t + retain orig source access time */
Bool    unlinkflag;     /* -u  unlink dest file before copying */
Bool    oldonlyflag;    /* -x  only if file already exists */
int     verboseflag;    /* -vV 1 or 2 for some or lots of information */
Bool    stdinflag;      /* -S  allow args from stdin with "=" arg */
Bool    nocopyflag;     /* -N  No copy or link or mkdir, just do -t and -o */

int uid;                /* our effective uid */
int gid;                /* our effective gid */
Bool    suflag;         /* YES if super user otherwise NO */
Bool    trouble;        /* YES if we should exit(2) for trouble */
int oldumask;           /* umask setting */
int puntdirmode;        /* mode for intermediate mkdirs in -p mode */

Bool freemem = YES;     /* we still have some free memory */

extern char *malloc ();
extern char *strcpy ();
extern char *strncpy ();
extern char *strcat ();
extern char *index ();
extern char *rindex ();
extern char *sprintf ();

extern char *extend ();
extern char *f1shorten ();
extern char *f2shorten ();
extern char *xstrcpy ();
extern char *getfilepart ();

extern void getdirstat ();
#ifdef  SYMLINKS
extern void lngetstat ();
#else   SYMLINKS
#define lngetstat getstat
#endif  SYMLINKS
extern void getstat ();
extern void gettimes ();
extern void deslash ();
extern void debug1 ();
extern void vprint ();
extern void utime_and_chown ();
extern void couldnot ();
void help();

main (argc, argv)
int argc;
char **argv;
{
    Reg2 char *dest; /* destination file name */
    Reg3 char *arg;

    if(argc == 2 && !strcmp(argv[1], "-help"))
      help();
    fdsource = dup (1);
    fddest   = dup (1);
    uid = geteuid ();
    gid = getegid ();
    suflag = uid == 0;
    oldumask = umask (0);
    puntdirmode = 0777 & ~oldumask;
    args_stdinstr = "=";
    Block {
	Reg1 int firstfile;
	if ((firstfile = getoptions (argc, argv)) < 0)
	    goto usage;
	(void) initargs (ARGS_SCAN, stdinflag,
		  &argv[firstfile], argc - firstfile);
    }
    nsources = chkmulti (argc, argv);
    if (nsources <= 0)
	goto usage;

    arg = curarg ();
    if (!arg) {
usage:
	fprintf (stderr, "Usage: %s\n   or: %s\n   or: %s\n",
			 "cp [ -abdilnNoprsStTuvVx ] [ - ] file1 file2",
			 "cp [ -abdilnNoprsStTuvVx ] [ - ] file ... directory",
		 	 "cp -help");
	exit (2);
    }
    dest = argv[argc - 1];
    argv[argc - 1] = NULL;

    /* get the destination */
    file2.endstr = xstrcpy (file2.name, dest);
    deslash (&file2);    /* remove trailing slash from file2.name if any */
    lngetstat (&file2);
    if (   nsources > 1
	|| recursiveflag
	|| fullpathflag
       ) {
	if (file2.type == S_IFNOT)
	    fprintf (stderr, "cp: destination does not exist.\n");
	if (file2.type != S_IFDIR) {
	    if (nsources <= 1)
		fprintf (stderr,
		     "cp: with -f or -r, destination must be a directory.\n");
	    goto usage;
	}
    }

    Block {
	Reg1 Bool donestat;   /* all copied or if moveflag, all moved */
	extern int doit ();
	if ((donestat = doit (arg, dest)) < 0)
	    goto usage;
	exit (trouble ? 2 : !donestat);
    }
}

/* Print help and exit
 * (by sja@hupu)
 */

void
help()
{
    printf("cp: the Better Copying Program; The Flags are:\n");
    printf("\t-a ask on all files\n");
    printf("\t-b backup - only if st_ctime of source > dest\n");
    printf("\t-d ask only on directories and special files\n");
#ifdef HOLES
    printf("\t-h copy holes with lseek() when possible\n");
#endif
    printf("\t-i interactive - ask if would overwrite\n");
    printf("\t-l link if possible\n");
    printf("\t-m move ** not fully implemented yet\n");
    printf("\t-n only if file doesn't already exist\n");
    printf("\t-o owner - preserve uid and gid\n");
    printf("\t-p full path name to destination\n");
    printf("\t-r recursive\n");
    printf("\t-s do special files specially\n");
    printf("\t-t times - preserve atime and mtime\n");
    printf("\t-T times - -t plus retain original source access time\n");
    printf("\t-u unlink dest file before copying\n");
    printf("\t-x only if file already exists\n");
    printf("\t-N No copy or link or mkdir, just do -t and -o\n");
    printf("\t-v verbose, partial\n");
    printf("\t-V verbose, full\n");
    printf("\t-S allow args from stdin with \"=\" arg\n");
    exit(0);
}


/* decide whether we have multi-source case or not.
 * Returns:
 *  0 - one source, no dest
 *  1 - one source, one dest
 *  2 - two or more sources, or sources redirected from stdin.
 * -1 - error
 */
int
chkmulti (argc, argv)
Reg2 int argc;
Reg1 char **argv;
{
    if (argc < 2)
	return -1;
    if (argv[argc - 1][0] == '-')
	return -1;
    if (   stdinflag
	&& !strcmp (argv[argc - 1], args_stdinstr)
       ) {
	fprintf (stderr, "cp: Can't use \"%s\" as last arg.\n", args_stdinstr);
	return -1;
    }

    if (argc < 3)
	return 0;
    if (argv[argc - 2][0] == '-')
	return 0;
    if (   stdinflag
	&& !strcmp (argv[argc - 2], args_stdinstr)
       )
	return 2;

    if (argc < 4)
	return 1;
    if (argv[argc - 3][0] == '-')
	return 1;
    return 2;
}

int
getoptions (argc, argv)
int argc;
char **argv;
{
    Reg3 int curarg;
    Reg2 char *arg;

    for ( curarg = 1, arg = argv[curarg]
	; curarg < argc
	; arg = argv[++curarg]
	) {
	if (arg[0] != '-')
	    break;
	if (arg[1] == '\0') {
	    /*
	     *  all files following a '-' are considered file names
	     */
	    (void) nextarg ();
	    break;
	}
	else Block {
	    Reg1 char *cp;
	    for (cp = &arg[1]; *cp; cp++)
		switch (*cp) {
		case 'a':
		    askflag = askdirflag = YES;
		    break;
		case 'b':
		    backflag = YES;
		    break;
		case 'd':
		    askdirflag = YES;
		    break;
#ifdef HOLES
		case 'h':
		    holeflag = YES;
		    break;
#endif
		case 'i':
		    interflag = YES;
		    break;
		case 'l':
		    linkflag = YES;
		    break;
	    /*  case 'm':
		    moveflag = YES;
		    break;
	     */ case 'n':
		    newonlyflag = YES;
		    break;
		case 'N':
		    nocopyflag = YES;
		    break;
		case 'o':
		    if (!suflag) {
			fprintf (stderr, "cp: only superuser may set -o flag\n");
			return -1;
		    }
		    ownerflag = YES;
		    break;
		case 'p':
		    fullpathflag = YES;
		    break;
		case 'r':
		    recursiveflag = YES;
		    break;
		case 's':
		    specflag = YES;
		    break;
		case 'S':
		    stdinflag = YES;
		    break;
		case 't':
		    timesflag = 1;
		    break;
		case 'T':
		    timesflag = 2;
		    break;
		case 'u':
		    unlinkflag = YES;
		    break;
		case 'v':
		    verboseflag = 1;
		    break;
		case 'V':
		    verboseflag = 2;
		    break;
		case 'x':
		    oldonlyflag = YES;
		    break;
		default:
		    fprintf (stderr, "cp: unknown option \"-%c\".\n", *cp);
		    return -1;
		}
	}
    }
    if (nocopyflag) {
	if (   linkflag
	    || newonlyflag
	    || unlinkflag
	   ) {
	    fprintf (stderr, "cp: \"-%c\" and \"-N\" incompatible.\n",
		     linkflag
		     ? 'l'
		     : newonlyflag
		       ? 'n'
		       : 'u');
	    return -1;
	}
    }
    if (newonlyflag && oldonlyflag) {
	fprintf (stderr, "cp: \"-n\" and \"-x\" incompatible.\n");
	return -1;
    }
    return curarg;
}

int
doit (arg, dest)
Reg2 char *arg;
Reg3 char *dest; /* destination file name */
{
    Reg1 Bool donestat;   /* all copied or if moveflag, all moved */

    donestat = YES;
    for (; arg; arg = nextarg ()) {
	/* get a source */
	file1.endstr = xstrcpy (file1.name, arg);
	deslash (&file1);/* remove trailing slash from file1.name if any */
	getstat (&file1);
	file2.endstr = xstrcpy (file2.name, dest);
	deslash (&file2);/* remove trailing slash from file2.name if any */
	lngetstat (&file2);
	if (file1.type == S_IFDIR) {
	    if (!recursiveflag) {
		printf("%s is a directory, skipped\n", file1.name);
		(void) fflush (stdout);
		donestat = NO;
	    }
	    else {
		Reg1 char *cp;
		if (file2.type != S_IFDIR) {
		    printf ("--destination is not a directory, %s not copied\n",
			    file1.name);
		    (void) fflush (stdout);
		    return -1;
		}

		/* NOTE: if -r, should check here to see */
		/* that dest is not subdir of source */

		if (   !fullpathflag
		    || dofullpath ()
		   ) {
		    /* pad file2.name with last part of file2.name */
		    if (!dotname (cp =  getfilepart (file1.name))) {
			(void) extend (&file2, cp);
			lngetstat (&file2);
		    }
#ifdef  NDIR
		    /* do directory copy */
		    donestat &= dircopy ((DIR *) 0);
#else   NDIR
		    donestat &= dircopy (-1); /* do directory copy */
#endif  NDIR
		}
		else
		    donestat = NO;
	    }
	}
	else if (file1.type == S_IFNOT) {
	    fprintf (stderr, "cp: %s does not exist\n", file1.name);
	    (void) fflush (stderr);
	    trouble = YES;
	    donestat = NO;
	}
	else {
	    /* do a simple file copy */
	    if (   !fullpathflag
		|| dofullpath ()
	       )
		/* use the file part of the pathname */
		donestat &= filecopy (); /* do the cp command */
	    else
		donestat = NO;
	}
    }
    return donestat;
}

Bool
dircopy (adirfd)    /* copy a directory */
#ifdef  NDIR
Reg2 DIR *adirfd;   /* source-directory file descriptor */
#else   NDIR
Reg2 int adirfd;    /* source-directory file descriptor */
#endif  NDIR
{
#ifdef  NDIR
    Reg2 DIR *dirfd;    /* source-directory file descriptor */
#else   NDIR
    Reg2 int dirfd;     /* source-directory file descriptor */
#endif  NDIR
    Reg3 int modeflags;
    Reg4 int svfile2type;
    Reg5 Bool retval;       /* all went ok in this directory */
    time_t times[2];
    struct direct *direntry;/* pointer to current directory entry */

    if (!askprint (askdirflag, "--examine directory", file1.name, (char *) 0))
	return NO;
    svfile2type = file2.type;
    modeflags = file1.stbuf.st_mode; /* save for later use */
    if (file2.type == S_IFNOT) {  /* does not exist so create the directory */
	if (nocopyflag)
	    goto ret;
	if (oldonlyflag) {
	    if (verboseflag >= 1) {
		printf ("--destination directory does not exist, %s not made\n",
			file2.name);
		(void) fflush (stdout);
	    }
	    goto ret;
	}
	(void) mkdir (file2.name, modeflags);
	lngetstat (&file2);
    }
    if (file2.type != S_IFDIR) {
	if (file2.type == S_IFNOT)
	    couldnot ("mkdir", file2.name);
	else {
	    fprintf (stderr, "cp: %s is not a directory\n", file2.name);
	    (void) fflush (stderr);
	}
	trouble = YES;
	retval = NO;
	goto ret;
    }
    retval = YES;        /* we presume ok until something fails */

    if (timesflag)
	gettimes (&file1.stbuf, times);
#ifdef  NDIR
    if (adirfd)
#else   NDIR
    if (adirfd >= 0)
#endif  NDIR
	dirfd = adirfd;
    else {
#ifdef  NDIR
	if ((dirfd = opendir (file1.name) ) == NULL) {
#else   NDIR
	if ((dirfd = open (file1.name, 0) ) < 0) {
#endif  NDIR
	    couldnot ("open directory", file1.name);
	    trouble = YES;
	    retval = NO;
	    goto ret;
	}
    }

    while (getentry (dirfd, &direntry)) {
	if (dotname (direntry->d_name))
	    /* skip "." and ".." */
	    continue;
	/* append entry names onto file1 and file2 and get new stats */
	(void) extend (&file1, direntry->d_name);
	getstat (&file1);
	(void) extend (&file2, direntry->d_name);
	lngetstat (&file2);

	switch (file1.type) {
	case S_IFREG:           /* just a file */
	case S_IFCHR:           /* character-type special file */
	case S_IFBLK:           /* block-type special file */
	default:                /* other special file */
	    retval &= filecopy ();
	    break;
	case S_IFDIR:           /* directory */
	    if (recursiveflag) Block { /* ignore if not */
#ifdef  NDIR
		Reg7 DIR *ndirfd;

		if (ndirfd = opendir (file1.name)) {
		    retval &= dircopy (ndirfd); /* copy the directory */
		    (void) closedir (ndirfd);
#else   NDIR
		Reg7 int ndirfd;

		if ((ndirfd = open (file1.name, 0)) >= 0) {
		    retval &= dircopy (ndirfd); /* copy the directory */
		    (void) close (ndirfd);
#endif  NDIR
		} else Block {
		    /* Oh shit.  Gotta close it and later reopen it */
		    Reg1 char *savend;
#ifdef  NDIR
		    long dirseekpos;
		    dirseekpos = telldir (dirfd);
		    (void) closedir (dirfd);
		    /* copy the directory */
		    retval &= dircopy ((DIR *) 0);
		    savend = f1shorten ();
		    if ((dirfd = opendir (file1.name)) == NULL) {
			couldnot ("reopen directory", file1.name);
			(void) f2shorten ();
			trouble = YES;
			retval = NO;
			goto ret;
		    }
		    fprintf (stderr,
			    "cp: warning: closed and reopened directory %s\n",
			    file1.name);
		    *file1.endstr = '/';
		    file1.endstr = savend;
		    (void) seekdir (dirfd, dirseekpos);
#else   NDIR
		    long dirseekpos;
		    extern long lseek ();
		    dirseekpos = lseek (dirfd, (long)0, 1);
		    (void) close (dirfd);
		    retval &= dircopy (-1); /* copy the directory */
		    savend = f1shorten ();
		    if ((dirfd = open (file1.name, 0)) < 0) {
			couldnot ("reopen directory", file1.name);
			(void) f2shorten ();
			trouble = YES;
			retval = NO;
			goto ret;
		    }
		    *file1.endstr = '/';
		    file1.endstr = savend;
		    (void) lseek (dirfd, dirseekpos, 0);
#endif  NDIR
		}
	    }
	    break;
	case S_IFNOT:
	    fprintf (stderr, "cp: %s does not exist\n", file1.name);
	    (void) fflush (stderr);
	    trouble = YES;
	    break;
	}
	(void) f1shorten ();
	(void) f2shorten ();
    }
#ifdef  NDIR
    if (adirfd == NULL)
	(void) closedir (dirfd); /* close the file source-directory */
#else   NDIR
    if (adirfd < 0)
	(void) close (dirfd); /* close the file source-directory */
#endif  NDIR
    if (   nocopyflag
	|| svfile2type == S_IFNOT   /* if directory was created then change */
       ) {
#ifndef SYSMKDIR
	(void) chmod (file2.name, modeflags);
#endif  SYSMKDIR
	utime_and_chown (times);
    }
 ret:
    if (retval && moveflag)
	return rmdir (file1.name) >= 0 ? YES : NO;
    return retval;
}

/* read a directory entry */
Bool
#ifdef  NDIR
getentry (df, direntry)
Reg1 DIR *df;
Reg2 struct direct **direntry;
{
    return (*direntry = readdir (df)) != NULL;
}
#else   NDIR
getentry (df, direntry)
Reg1 int df;    /* the directory file descriptor */
Reg2 struct direct **direntry;
{
    static struct direct dirbuf;

    do {
	if (read (df, (char *) &dirbuf, sizeof dirbuf) != sizeof dirbuf)
	    return NO;
    } while (dirbuf.d_ino == 0);
    *direntry = &dirbuf;
    return YES;
}
#endif  NDIR

/* copy file1.name to file2.name */
Bool
filecopy ()
{
    Reg4 Bool retval;
    Reg3 char *extname;
    Reg2 Bool special_copy;

    /* is target a directory? if so then extend file2.name with source name */
    if (file2.type == S_IFDIR) {
	extname = extend (&file2, getfilepart (file1.name));
	lngetstat (&file2);
    } else
	extname = NULL;

    /* If special_copy == YES, -s must be set or file will be skiped. */
    special_copy =   (file1.type == S_IFCHR || file1.type == S_IFBLK)
		  && (nsources > 1 || recursiveflag || specflag);
    /*
     *  Ask for confirmation if -a.
     *  Give up early if undoable.
     */
    if (special_copy) {
	if (!specflag) {
	    printf ("special file; %s not copied to %s\n",
		    file1.name, file2.name);
	    (void) fflush (stdout);
	    retval = NO;
	    goto ret;
	}
	if (   askflag
	    && !askprint (askdirflag,
		     linkflag ? "link (special)" : "copy (special)",
		     file1.name, file2.name)
	   ) {
	    retval = NO;
	    goto ret;
	}
    } else {
	if (   file1.type != S_IFREG
#ifdef  SYMLINKS
	    && file1.type != S_IFLNK
#endif  SYMLINKS
	    && file2.type == S_IFDIR
	   ) {
	    fprintf (stderr,
		     "cp: can't copy a special file to a directory\n");
	    (void) fflush (stderr);
	    trouble = YES;
	    retval = NO;
	    goto ret;
	}
	if (   askflag
#ifdef  SYMLINKS
	    && !askprint (askflag,
			  file1.type == S_IFLNK ? "copy (symlink)" : "copy",
			  file1.name, file2.name)
#else   SYMLINKS
	    && !askprint (askflag, "copy", file1.name, file2.name)
#endif  SYMLINKS
	   ) {
	    retval = NO;
	    goto ret;
	}
    }

    /*
     *  See if the operation meets -n, -x, -i, etc. conditions
     *  If already linked, skip the hard part.
     */
    if (file2.type == S_IFNOT) {
	/* destination does not exist */
	if (oldonlyflag) {       /* cannot process if flag is on and old */
	    if (verboseflag >= 1) {
		printf ("destination does not exist, %s not copied to %s\n",
			file1.name, file2.name);
		(void) fflush (stdout);
	    }
	    retval = NO;
	    goto ret;
	}
	/* if we are linking file1 to file2, we need dev number of dest dir */
	if (linkflag && nsources == 1)
	    getdirstat (&file2);
    } else {
	/* destination exists */
	if (   interflag
	    && !askprint (interflag, "destroy", file2.name, (char *) 0)
	   ) {
	    retval = NO;
	    goto ret;
	}
	if (newonlyflag) {       /* cannot process if flag is on and old */
	    if (verboseflag >= 1) {
		printf ("destination exists, %s not copied to %s\n",
			file1.name, file2.name);
		(void) fflush (stdout);
	    }
	    retval = NO;
	    goto ret;
	}
	else
	    if (   backflag         /* backup  */
		&& file1.stbuf.st_ctime <= file2.stbuf.st_ctime
	       ) {
		if (verboseflag >= 1) {
		    printf ("source not newer, %s not copied to %s\n",
			    file1.name, file2.name);
		    (void) fflush (stdout);
		}
		retval = NO;
		goto ret;
	    }
	if (file1.stbuf.st_dev == file2.stbuf.st_dev) {
	    /* source and dest are on same file system */
	    if (   file1.stbuf.st_ino == file2.stbuf.st_ino
		&& (linkflag || !unlinkflag)
	       ) {
		if (verboseflag >= 1) {
		    printf ("already linked, %s to %s\n",
			    file1.name, file2.name);
		    (void) fflush (stdout);
		}
		retval = YES;
		goto ret;
	    }
	}
    }

    /*
     *  Now we get to the actual linking or copying.
     */
    if (nocopyflag && file2.type == S_IFNOT) {
	retval = YES;
	goto ret;
    }
    if (   linkflag
	&& !nocopyflag
	&& file1.stbuf.st_dev == file2.stbuf.st_dev
#ifdef  SYMLINKS
	&& file1.type != S_IFLNK
#endif  SYMLINKS
	&& (file2.type == S_IFNOT || unlink (file2.name) >= 0)
	&& ((file2.type = S_IFNOT), link (file1.name, file2.name) >= 0)
       ) {
	/* we did the link successfully */
	if (!askflag)
	    vprint ("link", file1.name, file2.name);
    }
    else Block {
	time_t times[2];
	Reg1 char *cp;
	extern char *notelink ();
	if (  !nocopyflag
	    && (cp = notelink ())
#ifdef  SYMLINKS
	    && file1.type != S_IFLNK
#endif  SYMLINKS
	   ) {
	    if (!askflag && verboseflag >= 2) {
		printf ("copy %s to %s linked to %s\n",
			file1.name, file2.name, cp);
		(void) fflush (stdout);
	    }
	    (void) unlink (file2.name);
	    if (link (cp, file2.name) >= 0) {
		retval = YES;
		goto ret;
	    }
	    couldnot ("preserve link to", file2.name);
	    trouble = YES;
	}
	/* copy file1 to file2 the hard way */
	if (!askflag) {
#ifdef  SYMLINKS
	    vprint (file1.type == S_IFLNK ? "copy (symlink)" : "copy",
		    file1.name, file2.name);
#else   SYMLINKS
	    vprint ("copy", file1.name, file2.name);
#endif  SYMLINKS
	}
	if (timesflag)
	    gettimes (&file1.stbuf, times);
	if (!nocopyflag && !docopy (special_copy)) {
	    retval = NO;
	    goto ret;
	}
	utime_and_chown (times);
#ifndef CLR_SUID_ON_WRITE
	if (   !ownerflag
	    && file2.type == S_IFNOT
#else   CLR_SUID_ON_WRITE
	if (   file2.type == S_IFNOT
#endif  CLR_SUID_ON_WRITE
	    && (file1.stbuf.st_mode & (S_ISUID | S_ISGID | S_ISVTX))
	   ) Block {
	    Reg1 int newmode;
	    newmode = file1.stbuf.st_mode;
#ifdef  CLR_SUID_ON_WRITE
	    if (!ownerflag)
#endif  CLR_SUID_ON_WRITE
	    {
		if (file1.stbuf.st_uid != uid)
		    newmode &= ~S_ISUID;
		if (file1.stbuf.st_gid != gid)
		    newmode &= ~S_ISGID;
	    }
	    if (   newmode != file1.stbuf.st_mode
#ifdef  CLR_SUID_ON_WRITE
		|| (newmode & (S_ISUID | S_ISGID))
#endif  CLR_SUID_ON_WRITE
	       )
#ifdef  SYSFCHMOD
		if (file2.fd >= 0)
		    (void) fchmod (file2.fd, newmode);
		else
		    (void) chmod (file2.name, newmode);
#else   SYSFCHMOD
		(void) chmod (file2.name, newmode);
#endif  SYSFCHMOD
	}
    }
    retval = YES;
 ret:
    if (extname)
	*extname = '\0';
    if (retval && moveflag)
	return unlink (file1.name) >= 0;
    return retval;
}

Bool
askprint (option, operation, file, file2)
Bool option;
char *operation;
char *file;
char *file2;
{
    if (option) {        /* if option true then ask question */
	Reg1 int chr;
	Reg2 Bool flg;
	fprintf (stderr, "%s %s", operation, file);
	if (file2)
	    fprintf (stderr, " to %s", file2);
	fprintf (stderr, "? ");
	(void) fflush (stderr);
	chr = getchar();
	flg = chr == 'y' || tolower(chr) == 'y';
	while (chr != '\n' && chr != EOF)
		chr = getchar();
	return flg;
    }
    vprint (operation, file, file2);
    return YES;
}

/* lengthen a pathname by one element
 *
 * name is a char array no larger than a directory entry.
 * If it is full size, it likely won't not be null-terminated.
 * return original value of endstr.
 */
char *
extend (file, name)
Reg4 struct cpfile *file;
Reg3 char *name;
{
    Reg1 char *dest;
    Reg5 char *oldendstr;

    oldendstr = dest = file->endstr;
#ifdef  NDIR
    if (&file->name[MAXNAME] - dest <= strlen(name) + 1) {
#else   NDIR
    if (&file->name[MAXNAME] - dest <= DIRSIZ + 2) {
#endif  NDIR
	fprintf (stderr, "cp: file name too long %s\n", file->name);
	(void) fflush (stderr);
	exit (2);
    }

    *dest++ = '/';
#ifdef  NDIR
    file->endstr = xstrcpy(dest, name);
#else   NDIR
    {
	Reg2 int num;

	num = DIRSIZ;
	do {
	    if (*name == '\0')
		break;
	    *dest++ = *name++;
	} while (--num);
    }
    *dest = '\0';
    file->endstr = dest;
#endif  NDIR

#ifdef  FCHOWN_MOD
    file->fd = -1;
#endif  FCHOWN_MOD
    return oldendstr;
}

/* trim the last component from a pathname */
char *
f1shorten ()
{
    Reg1 char *cp;
    Reg2 char *oldendstr;

    oldendstr = file1.endstr;
    /* shorten file name to directory */
    if (cp = rindex (file1.name, '/'))
	*(file1.endstr = cp) = '\0';
    getstat (&file1);
#ifdef  FCHOWN_MOD
    file1.fd = -1;
#endif  FCHOWN_MOD
    return oldendstr;
}

/* trim the last component from a pathname */
char *
f2shorten ()
{
    Reg1 char *cp;
    Reg2 char *oldendstr;

    oldendstr = file2.endstr;
    /* shorten file name to directory */
    if (cp = rindex (file2.name, '/'))
	*(file2.endstr = cp) = '\0';
    lngetstat (&file2);
#ifdef  FCHOWN_MOD
    file2.fd = -1;
#endif  FCHOWN_MOD
    return oldendstr;
}

/*
 *  Do a stat of the directory part of file->name into file->stbuf
 */
void
getdirstat (file)
Reg2 struct cpfile *file;
{
    Reg1 char *cp;

    if (cp = rindex(file->name, '/')) {
	*cp = '\0';
#ifdef  SYMLINKS
	(void) lstat (file->name, &file->stbuf);
#else   SYMLINKS
	(void) stat (file->name, &file->stbuf);
#endif  SYMLINKS
	*cp = '/';
    } else
	(void) stat (".", &file->stbuf);
    return;
}

void
getstat (file)
Reg1 struct cpfile *file;
{
#ifdef  SYMLINKS
    if (lstat (file->name, &file->stbuf) < 0)
#else   SYMLINKS
    if (stat (file->name, &file->stbuf) < 0)
#endif  SYMLINKS
	file->type = S_IFNOT;
    else
	file->type = file->stbuf.st_mode & S_IFMT;
#ifdef  DEBUG
    debug1 (file == &file1, file->name, file->type);
#endif  DEBUG
    return;
}

#ifdef  SYMLINKS
void
lngetstat (file)
Reg1 struct cpfile *file;
{
    if (stat (file->name, &file->stbuf) < 0)
	file->type = S_IFNOT;
    else
	file->type = file->stbuf.st_mode & S_IFMT;
#ifdef  DEBUG
    debug1 (file == &file1, file->name, file->type);
#endif  DEBUG
    return;
}
#endif  SYMLINKS

/* build the time_t array required by utime() */
void
gettimes (stbuf, times)
struct stat *stbuf;
time_t times[2];
{
    times[0] = stbuf->st_atime;
    times[1] = stbuf->st_mtime;
    return;
}

/*
 * Copy string source to dest.  dest must be large enough.
 * Like strcpy except returns pointer to null at end of resulting string.
 */
char *
xstrcpy (dest, source)
Reg1 char *dest;
Reg2 char *source;
{
    while (*dest++ = *source++)
	continue;
    return &dest[-1];
}

/* trim off any slashes at end of file.name */
void
deslash (file)
Reg2 struct cpfile *file;
{
    Reg1 char *cp;

    for (;;) {
	if ((cp = rindex (file->name, '/')) == NULL)
	    break;
	if (cp == file->name)
	    break;
	if (cp[1])
	    break;
	*cp = '\0';
	file->endstr--;
    }
    return;
}

#ifdef  DEBUG
void
debug1 (which, nam, typ)
int which;
char *nam;
unsigned short typ;
{
/*  print the type and name of the file */

	printf ("Getstat %s %s :: ", which ? "1" : "2", nam);
	switch (typ) {
		case S_IFDIR:
				printf (" directory\n");
				break;
		case S_IFCHR:
				printf (" character special\n");
				break;
		case S_IFBLK:
				printf (" block special\n");
				break;
		case S_IFREG:
				printf (" plain\n");
				break;
		default:
				printf (" ?\n");
				break;
	}
	(void) fflush (stdout);
	return;
}
#endif  DEBUG

/*  trim off trailing slashes and
 *  return pointer to last component of name
 */
char *
getfilepart (name)
Reg2 char *name;
{
    Reg1 char *cp;

    for (;;) {
	if (   (cp = rindex (name, '/'))
	    && cp != name
	   ) {
	    if (*++cp)
		break;
	    *--cp = '\0';
	}
	else {
	    cp = name;
	    break;
	}
    }
    return cp;
}

/* return YES if "." or ".." */
Bool
dotname (cp)
Reg1 char *cp;
{
    if (*cp++ != '.')
	return NO;
    if (*cp == '\0')
	return YES;
    if (*cp++ != '.')
	return NO;
    if (*cp == '\0')
	return YES;
    return NO;
}

struct linkbuf {
    struct  linkbuf *nextp;
    ino_t   inum;
    dev_t   devnum;
    short   count;
    char    pathname[1];    /* actually alloced larger */
 } *ihead;

/*  note if this file is linked
 *  If it is, and its name isn't already on the link list add it.
 *  If the name is on the list, return a pointer to the name to link it to.
 */
char *
notelink ()
{
    Reg1 struct linkbuf *lp;
    Reg2 struct stat *stbuf;
    static struct linkbuf *freeptr = (struct linkbuf *) (char *) 1;

    stbuf = &file1.stbuf;
    if (stbuf->st_nlink <= 1)
	return NULL;
    if (ihead == freeptr) {
	ihead = freeptr->nextp;
	free ((char *) freeptr);
	freeptr = (struct linkbuf *) (char *) 1;
    }
    for (lp = ihead; ; lp = lp->nextp) {
	if (lp == NULL) {
	    lp = (struct linkbuf *)
		 malloc ((unsigned) (strlen (file2.name) + sizeof *lp));
	    if (lp == NULL) {
		if (freemem) {
		    fprintf (stderr, "cp: Out of memory. Link information lost\n");
		    (void) fflush (stderr);
		    trouble = YES;
		    freemem = NO;
		}
	    }
	    else {
		lp->nextp = ihead;
		ihead = lp;
		lp->inum = stbuf->st_ino;
		lp->devnum = stbuf->st_dev;
		lp->count = stbuf->st_nlink - 1;
		(void) xstrcpy(lp->pathname, file2.name);
	    }
	    break;
	}
	if (lp->nextp == freeptr) {
	    lp->nextp = freeptr->nextp;
	    free ((char *) freeptr);
	    freeptr = (struct linkbuf *) (char *) 1;
	}
	if (   lp->inum == stbuf->st_ino
	    && lp->devnum == stbuf->st_dev
	   ) {
	    if (--lp->count <= 0)
		freeptr = lp;
	    return lp->pathname;
	}
    }
    return NULL;
}

#ifdef HOLES
Bool
write_with_holes(buf, num)
char *buf;
int num;
{
    char *data_start;
    char *data_end;
    char *here;

    data_end = data_start = here = buf;

    while(here <= num) {
	if(!memcmp(here, zerobuf, BUFSIZE)) {
	    while(!memcmp(here, zerobuf, BUFSIZE))
	      here += BUFSIZE;
	    if(data_end - data_start)
	      if(!write_block(data_start, data_end - data_start))
		return(NO);
	    if(lseek(fddest, (long) , ) == -1)
	      if(!write_block(data_end, here - data_end))
		return(NO);
	    data_start = here;
	}
    }
}

void
write_block(data, num)
char *data;
int num; {
    if(write(fddest, data, num) != num) {
	fprintf (stderr, "cp: write error on %s\n", file2.name);
	perror("cp: write");
	(void) fflush (stderr);
	if (file2.type == S_IFNOT)
	  (void) unlink (file2.name);
	trouble = YES;
	return NO;
    }
}
#endif
#endif HOLES

/* copy file1 to file2.  You know, with reads and writes */
/* Yeah, but you can write with lseek, like, too, you know.  -- sja@hupu */
Bool
docopy (special_copy)
Bool special_copy;
{
    if (file2.type != S_IFNOT) {
	if (unlinkflag) {
	    if (unlink (file2.name) < 0) {
		fprintf (stderr, "cp: cannot unlink %s\n", file2.name);
		(void) fflush (stderr);
		trouble = YES;
		return NO;
	    }
	    file2.type = S_IFNOT;
	}
	else if (special_copy)
	    {}
	else if (   file2.type != S_IFREG
#ifdef  SYMLINKS
		 && file1.type != S_IFLNK
#endif  SYMLINKS
		 && (nsources > 1 || recursiveflag)
		) {
	    fprintf (stderr, "cp: will not copy to special file %s\n", file2.name);
	    (void) fflush (stderr);
	    trouble = YES;
	    return NO;
	}
    }

    if (special_copy) {
	(void) mknod (file2.name, (int)file1.stbuf.st_mode,
		      file1.stbuf.st_rdev);
	lngetstat (&file2);
	if (   file1.type != file2.type
	    || file1.stbuf.st_mode != file2.stbuf.st_mode
	    || file1.stbuf.st_rdev != file2.stbuf.st_rdev
	   ) {
	    couldnot ("create special file", file2.name);
	    trouble = YES;
	    return NO;
	}
	return YES;
    }

#ifdef  SYMLINKS
    if (file1.type == S_IFLNK) {
	char lname[MAXSYMLINK];
	int  lnamelen;

	if (   (lnamelen = readlink (file1.name, lname, MAXSYMLINK)) < 0
	    || lnamelen == MAXSYMLINK
	   ) {
	    couldnot ("read symbolic link", file1.name);
	    trouble = YES;
	    return NO;
	}
	lname[lnamelen] = '\0';
	if (   file2.type != S_IFNOT
	    && unlink (file2.name) < 0
	   ) {
	    couldnot ("create symbolic link", file2.name);
	    trouble = YES;
	    return NO;
	}
	if (symlink (lname, file2.name) < 0) {
	    couldnot ("create symbolic link", file2.name);
	    trouble = YES;
	    return NO;
	}
	return YES;
    }
#endif  SYMLINKS

    if (fdsource >= 0) {
	(void) close (fdsource);
#ifdef  FCHOWN_MOD
	file1.fd = -1;
#endif  FCHOWN_MOD
    }
    if ((fdsource = open (file1.name, 0)) < 0) {
	fprintf (stderr, "cp: cannot open %s\n", file1.name);
	(void) fflush (stderr);
	fdsource = dup (1);
	trouble = YES;
	return NO;
    }

    if (fddest >= 0) {
	(void) close (fddest);
#ifdef  FCHOWN_MOD
	file2.fd = -1;
#endif  FCHOWN_MOD
    }
    Block {
	Reg1 int mode;
	if (file1.type == S_IFREG)
	    mode = file1.stbuf.st_mode;
	else
	    mode = 0666 & ~oldumask;
	if ((fddest = creat (file2.name, mode)) < 0) {
	    fprintf (stderr, "cp: cannot create %s\n", file2.name);
	    (void) fflush (stderr);
	    fddest = dup (1);
	    trouble = YES;
	    return NO;
	}
    }

#ifdef  FCHOWN_MOD
    file1.fd = fdsource;
    file2.fd = fddest;
#endif  FCHOWN_MOD

    Block {
	Reg1 int num;
	char buf[COPYBUFSIZE];      /* NOTE: this requires a lot of stack space */
	while (num = read (fdsource, buf, COPYBUFSIZE)) {
	    if (num < 0) {
		fprintf (stderr, "cp: read error on %s\n", file1.name);
		perror("cp: read");
		(void) fflush (stderr);
		trouble = YES;
		return NO;
	    }
#ifdef HOLES
	    if (holeflag) {
		if(!write_with_holes(buf, num))
		  return(NO);
	    } else {
#endif
	    if (write (fddest, buf, num) != num) {
		fprintf (stderr, "cp: write error on %s\n", file2.name);
		perror("cp: write");
		(void) fflush (stderr);
		if (file2.type == S_IFNOT)
		    (void) unlink (file2.name);
		trouble = YES;
		return NO;
	    }
#ifdef HOLES
	}
#endif
	}
    }
    return YES;
}

void
utime_and_chown (times)
time_t times[2];
{
#ifdef  SYMLINKS
    if (timesflag && file1.type != S_IFLNK) {
#else   SYMLINKS
    if (timesflag) {
#endif  SYMLINKS
	if (utime (file2.name, times) < 0)
	    couldnot ("set times on", file2.name);
	if (   timesflag == 2
	    && utime (file1.name, times) < 0
	   )
	    couldnot ("set times on", file1.name);
    }
    if (ownerflag) {
#ifdef  SYSFCHOWN
	Bool ok;

	if (file2.fd >= 0)
	    ok = fchown (file2.fd, file1.stbuf.st_uid, file1.stbuf.st_gid);
	else
	    ok = chown (file2.name, file1.stbuf.st_uid, file1.stbuf.st_gid);
	if (ok < 0)
#else   SYSFCHOWN
	if (chown (file2.name, file1.stbuf.st_uid, file1.stbuf.st_gid) < 0)
#endif  SYSFCHOWN
	    couldnot ("change owner on", file2.name);
    }
    return;
}

/*
 * Tack file1.name onto file2.name and try to make all directories
 * along the path that don't already exist.
 */
Bool
dofullpath ()
{
    Reg1 char *cur;     /* current path element in file2.name */
    Reg3 char *last;    /* last path element */
    Reg2 Bool atend;

    if ((last = getfilepart (file1.name)) == file1.name)
	return YES;
    cur = file2.endstr;
    *cur++ = '/';
    *--last = '\0';
    (void) xstrcpy (cur, file1.name);
    *last = '/';
    file2.endstr = &cur[strlen (cur)];
    atend = NO;
    for (; ; cur++) {
	if (*cur == '\0' || *cur == '/') {
	    if (*cur == '\0')
		atend = YES;
	    else
		*cur = '\0';
	    lngetstat (&file2);
	    if (file2.type == S_IFNOT) {
		(void) mkdir (file2.name, puntdirmode);
		lngetstat (&file2);
	    }
	    if (file2.type != S_IFDIR) {
		couldnot ("make destination directory", file2.name);
		return NO;
	    }
	    if (atend)
		break;
	    *cur = '/';
	}
    }
    return YES;
}

#ifndef SYSMKDIR
/* 0 means mkdir went ok, -1 means error */
int
mkdir (dir, mode)
char *dir;
{
    vprint ("--make directory", dir, (char *) 0);
    if (!suflag) Block {
	int retval;
	static char mk[] = "/bin/mkdir \"%s\"";
	char cmd[sizeof mk + MAXNAME];
	(void) sprintf (cmd, mk, dir);
	retval = !system (cmd) ? 0 : -1;
	(void) chmod (dir, mode);
	return (retval);
    }
    else Block {
	Reg1 Bool retval;
	int (*hupsignal) ();
	int (*intsignal) ();
	int (*quitsignal) ();
	int (*pipesignal) ();
	int (*termsignal) ();

	hupsignal  = signal (SIGHUP,  SIG_IGN);
	intsignal  = signal (SIGINT,  SIG_IGN);
	quitsignal = signal (SIGQUIT, SIG_IGN);
	pipesignal = signal (SIGPIPE, SIG_IGN);
	termsignal = signal (SIGTERM, SIG_IGN);

	retval = domkdir (dir);

	(void) signal (SIGHUP,  hupsignal);
	(void) signal (SIGINT,  intsignal);
	(void) signal (SIGQUIT, quitsignal);
	(void) signal (SIGPIPE, pipesignal);
	(void) signal (SIGTERM, termsignal);

	return retval ? 0 : -1;
    }
}

/* YES means mkdir went ok */
Bool
domkdir(d)
char *d;
{
	register Bool Errors;
	char pname[128], dname[128];
	register i, slash = 0;

	Errors = NO;
	pname[0] = '\0';
	for(i = 0; d[i]; ++i)
		if(d[i] == '/')
			slash = i + 1;
	if(slash)
		(void) strncpy(pname, d, slash);
	(void) xstrcpy(pname+slash, ".");
	if (access(pname, 02)) {
		fprintf(stderr,"cp: cannot access %s\n", pname);
		(void) fflush (stderr);
		return NO;
	}
	if ((mknod(d, 040777, 0)) < 0) {
		fprintf(stderr,"cp: cannot make directory %s\n", d);
		(void) fflush (stderr);
		return NO;
	}
	(void) chown(d, getuid(), getgid());
	(void) xstrcpy(dname, d);
	(void) strcat(dname, "/.");
	if((link(d, dname)) < 0) {
		fprintf(stderr, "cp: cannot link %s\n", dname);
		(void) fflush (stderr);
		(void) unlink(d);
		return NO;
	}
	(void) strcat(dname, ".");
	if((link(pname, dname)) < 0) {
		fprintf(stderr, "cp: cannot link %s\n",dname);
		(void) fflush (stderr);
		dname[strlen(dname)] = '\0';
		(void) unlink(dname);
		(void) unlink(d);
		Errors = YES;
	}
	return !Errors;
}
#endif  SYSMKDIR

void
vprint (operation, file, file2)
char *operation;
char *file;
char *file2;
{
    if (verboseflag >= 2) {
	printf ("%s %s", operation, file);
	if (file2)
	    printf (" to %s", file2);
	printf ("\n");
	(void) fflush (stdout);
    }
    return;
}

#ifndef SYSRMDIR
int
rmdir (dir)
char *dir;
{
    static char rm[] = "/bin/rmdir %s";
    char cmd[sizeof rm + MAXNAME];

    vprint ("--remove directory", dir, (char *) 0);
    (void) sprintf (cmd, rm, dir);
    return !system (cmd) ? 0 : -1;
}
#endif  SYSRMDIR

void
couldnot (str1, str2)
char *str1;
char *str2;
{
    fprintf (stderr, "cp: could not %s %s\n", str1, str2);
    (void) fflush (stderr);
    return;
}
