/* rasttopbm.c - read a Sun raster file and produce a portable bitmap
**
** Copyright (C) 1988 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

#include <stdio.h>
#include "pbm.h"

/* Because of the following include, this program compiles only on Suns. */
#include <pixrect/pixrect_hs.h>

main( argc, argv )
int argc;
char *argv[];
    {
    FILE *ifd;
    bit **bits;
    int rows, cols, row, col, shortcount, mask;
    int linesize;
    short *data;

    if ( argc > 2 )
	{
	fprintf( stderr, "usage: %s [rastfile]\n", argv[0] );
	exit( 1 );
	}

    if ( argc == 2 )
	{
	ifd = fopen( argv[1], "r" );
	if ( ifd == NULL )
	    {
	    fprintf( stderr, "%s: can't open.\n", argv[1] );
	    exit( 1 );
	    }
	}
    else
	ifd = stdin;

    if ( ReadRasterFile( ifd, &cols, &rows, &linesize, &data ) < 0 )
	{
	fprintf( stderr, "%s: can't load.\n", argv[1] );
	exit( 1 );
	}

    if ( ifd != stdin )
	fclose( ifd );

    bits = pbm_allocarray( cols, rows );

    for ( row = 0; row < rows; row++ )
	{
	shortcount = 0;
	mask = 0x8000;
	for ( col = 0; col < cols; col++ )
	    {
	    if ( mask == 0 )
		{
		shortcount++;
		mask = 0x8000;
		}
	    bits[row][col] = ( *(data + shortcount) & mask ) ? 1 : 0;
	    mask = mask >> 1;
	    }
	data += linesize / sizeof(short);
	}

    pbm_writepbm( stdout, bits, cols, rows );

    exit( 0 );
    }


int
ReadRasterFile( file, width, height, linebytes, data )
FILE *file;
int *width, *height;
int *linebytes;
short **data;
    {
    struct rasterfile header;
    struct pixrect *pr, *pr_load_image();
    int status, firsttime, value, i, data_length;

    if ( file == NULL )
	return ( -1 );

    *width = *height = -1;


    /* Get the raster file's header. */
    if ( pr_load_header( file, &header ) != 0 )
	{
	fprintf( stderr, "Unable to read in raster file header.\n");
	return ( -1 );
	}

    /* PBM can only handle monochrome bitmaps. */
    if ( header.ras_depth != 1 )
	{
	fprintf( stderr, "Invalid depth.\n" );
	return ( -1 );
	}

    *width = header.ras_width;
    *height = header.ras_height;
    if ( *width <= 0 )
	{
	fprintf( stderr, "Invalid width: %d.\n", *width );
	return ( -1 );
	}
    
    if ( *height <= 0 )
	{
	fprintf( stderr, "Invalid height: %d.\n", *height );
	return ( -1 );
	}

    /* If there is a color map, skip over it. */
    if ( header.ras_maptype != RMT_NONE && header.ras_maplength != 0 )
	{
	if (pr_load_colormap(file, &header, NULL) != 0)
	    {
	    fprintf( stderr, "Unable to skip colormap data.\n");
	    return ( -1 );
	    }
	}

    /* Now load the data.  The pixrect returned is a memory pixrect. */
    if ( (pr = pr_load_image(file, &header, NULL)) == NULL )
	{
	fprintf(
	    stderr, "Unable to read in the image from the raster file.\n");
	return ( -1 );
	}

    *linebytes = ((struct mpr_data *)pr->pr_data)->md_linebytes;
    *data = ((struct mpr_data *)pr->pr_data)->md_image;

    return ( 0 );
    }
