/* term_curses.c 

   ncurses terminal i/o TUI for AdvSys 1.2 / TAiL 1.3
   ---Valerie Winter
*/

#pragma lint -1
#pragma noroot

#if STACKDEBUG
#pragma debug 24
#endif

#if OPTIMIZE
#pragma optimize 9
#endif

#include "interpreter.h"
#include "../shared/dbaccess.h"

#ifdef FALSE
#undef FALSE
#undef TRUE
#endif

#define INT_COMMANDS "\
help for internal commands:\n\
  \\q         quick-quit, exit game and interpreter\n\
  \\reverse   toggle display mode\n\
  \\version   interpreter version information\n"


#include <ncurses.h>
#include <string.h>


char isstatus=FALSE;  /* the flag "is this for the status line?" */
char statbuffer[80]="\0"; /* currently unused */

WINDOW *textwin, *statuswin;
int screenwidth, screenheight;
char f_reverse=FALSE;

/* useful definitions */
#define EOS     '\0'
#define LINEMAX 200
#define WORDMAX 100

/* global variables */
char line[LINEMAX+1];
extern int moreprompts;

/* local variables */
static int col,maxcol,row,maxrow;
static int scnt,wcnt;
static char theWord[WORDMAX+1],*wptr;
static FILE *logfp = NULL;



void waitch(void)
{
  wgetch(textwin);
}

void putch(int ch,FILE *fp)
{
  /*
    if (ch=='\r')*/   /* TUU */
  /*  ch='\n'; */
  putc(ch,fp);
}


void curses_statuswin(void)
{
  int x;

  for(x=0;x<=screenwidth;x++)
    waddch(statuswin,' ');

  waddch(statuswin,'\r');

  /*
  if(!strcmp(storyname,"\0"))
    {
      wprintw(statuswin,"\r %s",IDSTRING);
    }
  else
    {
  */
      for(x=0;x<=screenwidth-strlen(IDSTRING)-2;x++)
	waddch(statuswin,' ');
      wprintw(statuswin,"%s",IDSTRING);

      storyname[0]=toupper(storyname[0]);
      for(x=0;x<=strlen(storyname);x++)
	if(storyname[x]=='_')
	  {
	    storyname[x]=' ';
	    storyname[x+1]=toupper(storyname[x+1]);
	  }

      wprintw(statuswin,"\r %s, release %d", storyname, storyversion);

      /*
    }
      */

  wrefresh(statuswin);
}


/* update_statuswin(c)
   a small hack to display the floating status line from the modified 
   defaults.adi in the status window instead of in the text window.
   introduced the character \b into the lexical scanner to have begin
   and end marks (\b and \n).
 */
 void update_statuswin(char* c) 
 { 
   int ctr; 

   if(c=='\n') 
     { 
       wrefresh(statuswin); 
       isstatus=FALSE; 
       ctr=0; 
     } 
   else if(c=='\b') 
     { 
       waddch(statuswin,'\r'); 
       ctr=0; 
       while(ctr<=screenwidth) 
 	{ 
 	  waddch(statuswin,' '); 
 	  ctr++; 
 	} 
       waddch(statuswin,'\r'); 
       ctr=1; 
     } 
   else 
     { 
       waddch(statuswin,c); 
       ctr++; 
     } 
 }


void set_term_attributes(void)
{
  if(f_reverse==TRUE)
    {
      /*
      wbkgdset(textwin, A_REVERSE);
      wbkgdset(statuswin, A_NORMAL);
      */
      wattrset(statuswin,A_NORMAL);
      wattrset(textwin,A_REVERSE);
      wbkgd(statuswin,' ' | A_NORMAL);
      wbkgd(textwin,' ' | A_REVERSE);
    }
  if(f_reverse==FALSE)
    {
      /*
      wbkgdset(statuswin, A_REVERSE);
      wbkgdset(textwin, A_NORMAL);
      */
      wattrset(textwin,A_NORMAL);
      wattrset(statuswin,A_REVERSE);
      wbkgd(textwin,' ' | A_NORMAL);
      wbkgd(statuswin,' ' | A_REVERSE);
    }
  wrefresh(textwin);
  wrefresh(statuswin);
}

/* trm_init - initialize the terminal module */
void trm_init(int rows,int cols,char *name)
{
  initscr();
  cbreak();
  nonl();
  echo();
  intrflush(stdscr,FALSE);
  keypad(textwin,TRUE);
  newwin(0,0,0,0);
  screenwidth=COLS;
  screenheight=LINES;

  statuswin=subwin(stdscr,1,screenwidth,0,0);
  textwin=subwin(stdscr,screenheight-1,screenwidth,1,0);
  scrollok(textwin,TRUE);

  curses_statuswin();

  waddch(textwin,'\n');

  set_term_attributes();

  wrefresh(statuswin);
  wrefresh(textwin);

    /* initialize the terminal i/o variables */
    maxcol = screenwidth-1; 
    col = 0;
    maxrow = screenheight-2; 
    row = 1;
    wptr = theWord; 
    wcnt = 0;
    scnt = 0;

    /* open the log file */
    if (name && (logfp = fopen(name,"w")) == NULL)
        error("can't open log file");
}

/* trm_done - finish terminal i/o */
void trm_done(void)
{
  if(wcnt) 
    trm_word();

  if(logfp) 
    fclose(logfp);
  
  wprintw(textwin,"\n[Press any key to exit.]");
  wgetch(textwin);
  
  endwin();
  exit(0);
}


/* trm_get - get a line */
char *trm_get(char *line)
{
    if(wcnt) 
      trm_word();

    while(scnt--) 
      putchr(' ');

    row = col = scnt = 0;

    /*    curses_statuswin();
	  update_statuswin('\n'); */
    wrefresh(statuswin);
    wrefresh(textwin);

    return(trm_line(line));
}

/* trm_str - output a string */
void trm_str(char *str)
{
    while(*str)
        trm_chr(*str++);
}

/* trm_xstr - output a string without logging or word wrap */
void trm_xstr(char *str)
{
/*
    while (*str)
        putch(*str++,stdout);
*/
  while (*str)
    waddch(textwin,*str++);
  wrefresh(textwin);
}

/* trm_chr - output a character */
void trm_chr(int ch)
{
    switch(ch) {
    case ' ':
	    if(isstatus==TRUE)
	      update_statuswin(ch);
            if(wcnt)
               trm_word();
            scnt++;
            break;
    case '\t':
            if(wcnt)
                trm_word();
            scnt = (col + 8) & ~7;
            break;
    case '\n': /* I use \n as end toggle for the status window */
	    if(isstatus==TRUE)
	      update_statuswin(ch);
	    if(wcnt)
	      trm_word();
	    trm_eol();
	    scnt = 0;
            break;
    case '\b': /* I use \b as begin toggle for the status window */
      isstatus=TRUE; /* the flag "is this for the status line?" */
	    update_statuswin(ch);
	    break;
    default:
      if(isstatus==TRUE)
	    update_statuswin(ch);
      else
	{
            if(wcnt < WORDMAX)
	      {
                *wptr++ = ch;
                wcnt++;
	      }
	}
      break;
    }
}

/* trm_word - output the current word */
void trm_word(void)
{
    if(col + scnt + wcnt > maxcol)
        trm_eol();
    else
        while(scnt--)
	  { 
	    putchr(' '); 
	    col++; 
	  }
    for(wptr = theWord; wcnt--; col++)
        putchr(*wptr++);

    wptr = theWord;
    wcnt = 0;
    scnt = 0;
}

/* trm_eol - end the current line */
void trm_eol(void)
{
   putchr('\n');
    if(++row >= maxrow)
      { 
	trm_wait(); 
	row = 0; 
      }
    col = 0;
}

/* trm_wait - wait for the user to type return 
   now with a decent [MORE] prompt --- val
*/
void trm_wait(void)
{
  if(moreprompts)
    {
      wprintw(textwin,"[MORE]\r");
      wrefresh(textwin);
      waitch();
      wprintw(textwin,"      \r");
    }
  wrefresh(textwin);
}

/* trm_line - get an input line */
char *trm_line(char *line)
{
  char cmdbuf[100];
  char *p;
  int i=0, ch;

    p = line;
    nocbreak();
    while((ch = wgetch(textwin)) != '\n')
        switch(ch) 
	  {
	  case '\177':
	  case '\010':
	    if(p != line) 
	      {
		if(ch != '\010') 
		  putchr('\010');
		putchr(' ');
		putchr('\010');
		p--;
	      }
	    break;
	  case '\\': /* internal interpreter commands */
	    i=0;
	    while((ch=wgetch(textwin))!='\n' && i < sizeof(cmdbuf))
	      cmdbuf[i++]=ch;
	    cmdbuf[i]=0;
	    if(!strcmp(cmdbuf,"reverse"))
	      {
		if(f_reverse==TRUE)
		  f_reverse=FALSE;
		else if(f_reverse==FALSE)
		  f_reverse=TRUE;
		set_term_attributes();
	      }
	    else if(!strcmp(cmdbuf,"q"))
	      {
		wprintw(textwin,"[quick-quit game --- sure (y/n)?] ");
		cbreak();
		ch=wgetch(textwin);
		nocbreak();
		if(ch=='y' || ch=='Y')
		    trm_done();
		waddch(textwin,'\n');
	      }
	    else if(!strcmp(cmdbuf,"help"))
	      wprintw(textwin,INT_COMMANDS);
	    else if(!strcmp(cmdbuf,"version"))
	      wprintw(textwin,MSG_VERSION);
	    else
	      wprintw(textwin,"[unknown command, try \\help]\n");
	    wprintw(textwin,"\n>");
	    ch='\0';
	    break;
	  default:
	    if((p - line) < LINEMAX)
	      *p++ = ch;
	    break;
	  }
    *p = 0;
    cbreak();
    /*    waddch(textwin,'\n');*/
    return(ch == EOF ? NULL : line);
}

/* getchr - input a single character */
int getchr(void)
{
    int ch;

    ch = wgetch(textwin);
    if(logfp)
        putch(ch,logfp);
    return(ch);
}

/* putchr - output a single character */
void putchr(int ch)
{
 if (logfp) 
   putch(ch,logfp);

 waddch(textwin,ch);
}
